/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.storage;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.apache.ratis.conf.RaftProperties;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.impl.RaftServerImpl;
import org.apache.ratis.server.impl.ServerProtoUtils;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.storage.LogSegment;
import org.apache.ratis.server.storage.RaftLog;
import org.apache.ratis.server.storage.RaftLogCache;
import org.apache.ratis.server.storage.RaftLogIOException;
import org.apache.ratis.server.storage.RaftLogWorker;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.server.storage.RaftStorageDirectory;
import org.apache.ratis.shaded.proto.RaftProtos;
import org.apache.ratis.util.AutoCloseableLock;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.Preconditions;
import org.apache.ratis.util.ProtoUtils;

public class SegmentedRaftLog
extends RaftLog {
    static final String HEADER_STR = "RAFTLOG1";
    static final byte[] HEADER_BYTES = "RAFTLOG1".getBytes(StandardCharsets.UTF_8);
    private final RaftServerImpl server;
    private final RaftStorage storage;
    private final RaftLogCache cache;
    private final RaftLogWorker fileLogWorker;
    private final long segmentMaxSize;

    public SegmentedRaftLog(RaftPeerId selfId, RaftServerImpl server, RaftStorage storage, long lastIndexInSnapshot, RaftProperties properties) throws IOException {
        super(selfId, RaftServerConfigKeys.Log.Appender.bufferCapacity(properties).getSizeInt());
        this.server = server;
        this.storage = storage;
        this.segmentMaxSize = RaftServerConfigKeys.Log.segmentSizeMax(properties).getSize();
        this.cache = new RaftLogCache(selfId, storage, properties);
        this.fileLogWorker = new RaftLogWorker(selfId, server, storage, properties);
        this.lastCommitted.set(lastIndexInSnapshot);
    }

    @Override
    public void open(long lastIndexInSnapshot, Consumer<RaftProtos.LogEntryProto> consumer) throws IOException {
        this.loadLogSegments(lastIndexInSnapshot, consumer);
        File openSegmentFile = null;
        LogSegment openSegment = this.cache.getOpenSegment();
        if (openSegment != null) {
            openSegmentFile = this.storage.getStorageDir().getOpenLogFile(openSegment.getStartIndex());
        }
        this.fileLogWorker.start(Math.max(this.cache.getEndIndex(), lastIndexInSnapshot), openSegmentFile);
        super.open(lastIndexInSnapshot, consumer);
    }

    @Override
    public long getStartIndex() {
        return this.cache.getStartIndex();
    }

    private void loadLogSegments(long lastIndexInSnapshot, Consumer<RaftProtos.LogEntryProto> logConsumer) throws IOException {
        try (AutoCloseableLock writeLock = this.writeLock();){
            List<RaftStorageDirectory.LogPathAndIndex> paths = this.storage.getStorageDir().getLogSegmentFiles();
            int i = 0;
            for (RaftStorageDirectory.LogPathAndIndex pi : paths) {
                boolean isOpen = pi.endIndex == -1L;
                boolean keepEntryInCache = paths.size() - i++ <= this.cache.getMaxCachedSegments();
                this.cache.loadSegment(pi, isOpen, keepEntryInCache, logConsumer);
            }
            if (!this.cache.isEmpty() && this.cache.getEndIndex() < lastIndexInSnapshot) {
                LOG.warn("End log index {} is smaller than last index in snapshot {}", (Object)this.cache.getEndIndex(), (Object)lastIndexInSnapshot);
                this.cache.clear();
            }
        }
    }

    @Override
    public RaftProtos.LogEntryProto get(long index) throws RaftLogIOException {
        LogSegment.LogRecordWithEntry recordAndEntry;
        LogSegment segment;
        this.checkLogState();
        try (AutoCloseableLock readLock = this.readLock();){
            segment = this.cache.getSegment(index);
            if (segment == null) {
                RaftProtos.LogEntryProto logEntryProto = null;
                return logEntryProto;
            }
            recordAndEntry = segment.getEntryWithoutLoading(index);
            if (recordAndEntry == null) {
                RaftProtos.LogEntryProto logEntryProto = null;
                return logEntryProto;
            }
            if (recordAndEntry.hasEntry()) {
                RaftProtos.LogEntryProto logEntryProto = recordAndEntry.getEntry();
                return logEntryProto;
            }
        }
        this.checkAndEvictCache();
        return segment.loadCache(recordAndEntry.getRecord());
    }

    @Override
    public RaftLog.EntryWithData getEntryWithData(long index) throws RaftLogIOException {
        RaftProtos.LogEntryProto entry = this.get(index);
        if (!ProtoUtils.shouldReadStateMachineData((RaftProtos.LogEntryProto)entry)) {
            return new RaftLog.EntryWithData(entry, null);
        }
        try {
            return new RaftLog.EntryWithData(entry, this.server.getStateMachine().readStateMachineData(entry));
        }
        catch (Throwable e) {
            String err = this.server.getId() + ": Failed readStateMachineData for " + ServerProtoUtils.toLogEntryString(entry);
            LOG.error(err, e);
            throw new RaftLogIOException(err, JavaUtils.unwrapCompletionException((Throwable)e));
        }
    }

    private void checkAndEvictCache() {
        if (this.server != null && this.cache.shouldEvict()) {
            this.cache.evictCache(this.server.getFollowerNextIndices(), this.fileLogWorker.getFlushedIndex(), this.server.getState().getLastAppliedIndex());
        }
    }

    @Override
    public TermIndex getTermIndex(long index) {
        this.checkLogState();
        try (AutoCloseableLock readLock = this.readLock();){
            LogSegment.LogRecord record = this.cache.getLogRecord(index);
            TermIndex termIndex = record != null ? record.getTermIndex() : null;
            return termIndex;
        }
    }

    @Override
    public TermIndex[] getEntries(long startIndex, long endIndex) {
        this.checkLogState();
        try (AutoCloseableLock readLock = this.readLock();){
            TermIndex[] termIndexArray = this.cache.getTermIndices(startIndex, endIndex);
            return termIndexArray;
        }
    }

    @Override
    public TermIndex getLastEntryTermIndex() {
        this.checkLogState();
        try (AutoCloseableLock readLock = this.readLock();){
            TermIndex termIndex = this.cache.getLastTermIndex();
            return termIndex;
        }
    }

    @Override
    CompletableFuture<Long> truncate(long index) {
        this.checkLogState();
        try (AutoCloseableLock writeLock = this.writeLock();){
            RaftLogCache.TruncationSegments ts = this.cache.truncate(index);
            if (ts != null) {
                Task task = this.fileLogWorker.truncate(ts);
                CompletableFuture<Long> completableFuture = task.getFuture();
                return completableFuture;
            }
        }
        return CompletableFuture.completedFuture(index);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    CompletableFuture<Long> appendEntry(RaftProtos.LogEntryProto entry) {
        this.checkLogState();
        if (LOG.isTraceEnabled()) {
            LOG.trace("{}: appendEntry {}", (Object)this.server.getId(), (Object)ServerProtoUtils.toLogEntryString(entry));
        }
        try (AutoCloseableLock writeLock = this.writeLock();){
            LogSegment currentOpenSegment = this.cache.getOpenSegment();
            if (currentOpenSegment == null) {
                this.cache.addOpenSegment(entry.getIndex());
                this.fileLogWorker.startLogSegment(entry.getIndex());
            } else if (this.isSegmentFull(currentOpenSegment, entry)) {
                this.cache.rollOpenSegment(true);
                this.fileLogWorker.rollLogSegment(currentOpenSegment);
                this.checkAndEvictCache();
            } else if (currentOpenSegment.numOfEntries() > 0 && currentOpenSegment.getLastTermIndex().getTerm() != entry.getTerm()) {
                long currentTerm = currentOpenSegment.getLastTermIndex().getTerm();
                Preconditions.assertTrue((currentTerm < entry.getTerm() ? 1 : 0) != 0, (String)"open segment's term %s is larger than the new entry's term %s", (Object[])new Object[]{currentTerm, entry.getTerm()});
                this.cache.rollOpenSegment(true);
                this.fileLogWorker.rollLogSegment(currentOpenSegment);
                this.checkAndEvictCache();
            }
            CompletableFuture<Long> writeFuture = this.fileLogWorker.writeLogEntry(entry).getFuture();
            this.cache.appendEntry(entry);
            CompletableFuture<Long> completableFuture = writeFuture;
            return completableFuture;
        }
        catch (Throwable throwable) {
            LOG.error(this.getSelfId() + "exception while appending entry with index:" + entry.getIndex(), throwable);
            throw throwable;
        }
    }

    private boolean isSegmentFull(LogSegment segment, RaftProtos.LogEntryProto entry) {
        if (segment.getTotalSize() >= this.segmentMaxSize) {
            return true;
        }
        long entrySize = LogSegment.getEntrySize(entry);
        return entrySize <= this.segmentMaxSize && segment.getTotalSize() + entrySize > this.segmentMaxSize;
    }

    @Override
    public List<CompletableFuture<Long>> append(RaftProtos.LogEntryProto ... entries) {
        this.checkLogState();
        if (entries == null || entries.length == 0) {
            return Collections.emptyList();
        }
        try (AutoCloseableLock writeLock = this.writeLock();){
            ArrayList<CompletableFuture<Long>> futures;
            int index;
            Iterator<TermIndex> iter = this.cache.iterator(entries[0].getIndex());
            long truncateIndex = -1L;
            block11: for (index = 0; iter.hasNext() && index < entries.length; ++index) {
                TermIndex storedEntry = iter.next();
                Preconditions.assertTrue((storedEntry.getIndex() == entries[index].getIndex() ? 1 : 0) != 0, (String)"The stored entry's index %s is not consistent with the received entries[%s]'s index %s", (Object[])new Object[]{storedEntry.getIndex(), index, entries[index].getIndex()});
                if (storedEntry.getTerm() == entries[index].getTerm()) continue;
                truncateIndex = storedEntry.getIndex();
                if (LOG.isTraceEnabled()) {
                    LOG.trace("{}: truncate to {}, index={}, ti={}, storedEntry={}, entries={}", new Object[]{this.server.getId(), truncateIndex, index, ServerProtoUtils.toTermIndex(entries[index]), storedEntry, ServerProtoUtils.toString(entries)});
                }
                while (true) {
                    try {
                        RaftProtos.LogEntryProto entry = this.get(storedEntry.getIndex());
                        this.server.failClientRequest(entry);
                    }
                    catch (RaftLogIOException e) {
                        LOG.error("Failed to read log " + storedEntry, (Throwable)((Object)e));
                    }
                    if (!iter.hasNext()) break block11;
                    storedEntry = iter.next();
                }
            }
            if (truncateIndex != -1L) {
                futures = new ArrayList(entries.length - index + 1);
                futures.add(this.truncate(truncateIndex));
            } else {
                futures = new ArrayList<CompletableFuture<Long>>(entries.length - index);
            }
            for (int i = index; i < entries.length; ++i) {
                futures.add(this.appendEntry(entries[i]));
            }
            ArrayList<CompletableFuture<Long>> arrayList = futures;
            return arrayList;
        }
    }

    @Override
    public long getLatestFlushedIndex() {
        return this.fileLogWorker.getFlushedIndex();
    }

    @Override
    public void writeMetadata(long term, RaftPeerId votedFor) throws IOException {
        this.storage.getMetaFile().set(term, votedFor != null ? votedFor.toString() : null);
    }

    @Override
    public RaftLog.Metadata loadMetadata() throws IOException {
        return new RaftLog.Metadata(RaftPeerId.getRaftPeerId((String)this.storage.getMetaFile().getVotedFor()), this.storage.getMetaFile().getTerm());
    }

    @Override
    public void syncWithSnapshot(long lastSnapshotIndex) {
        this.fileLogWorker.syncWithSnapshot(lastSnapshotIndex);
    }

    @Override
    public boolean isConfigEntry(TermIndex ti) {
        return this.cache.isConfigEntry(ti);
    }

    @Override
    public void close() throws IOException {
        try (AutoCloseableLock writeLock = this.writeLock();){
            super.close();
            this.cache.clear();
        }
        this.fileLogWorker.close();
        this.storage.close();
    }

    RaftLogCache getRaftLogCache() {
        return this.cache;
    }

    static abstract class Task {
        private final CompletableFuture<Long> future = new CompletableFuture();

        Task() {
        }

        CompletableFuture<Long> getFuture() {
            return this.future;
        }

        void done() {
            this.future.complete(this.getEndIndex());
        }

        abstract void execute() throws IOException;

        abstract long getEndIndex();

        public String toString() {
            return this.getClass().getSimpleName() + ":" + this.getEndIndex();
        }
    }
}

