/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.storage;

import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Timer;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import org.apache.ratis.conf.RaftProperties;
import org.apache.ratis.metrics.RatisMetricsRegistry;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.impl.RaftServerImpl;
import org.apache.ratis.server.impl.ServerProtoUtils;
import org.apache.ratis.server.storage.LogOutputStream;
import org.apache.ratis.server.storage.LogSegment;
import org.apache.ratis.server.storage.RaftLogCache;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.server.storage.SegmentedRaftLog;
import org.apache.ratis.shaded.proto.RaftProtos;
import org.apache.ratis.statemachine.StateMachine;
import org.apache.ratis.util.ExitUtils;
import org.apache.ratis.util.FileUtils;
import org.apache.ratis.util.IOUtils;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.Preconditions;
import org.apache.ratis.util.ProtoUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class RaftLogWorker
implements Runnable {
    static final Logger LOG = LoggerFactory.getLogger(RaftLogWorker.class);
    private final String name;
    private final BlockingQueue<SegmentedRaftLog.Task> queue = new ArrayBlockingQueue<SegmentedRaftLog.Task>(4096);
    private volatile boolean running = true;
    private final Thread workerThread;
    private final RaftStorage storage;
    private volatile LogOutputStream out;
    private final RaftServerImpl raftServer;
    private final StateMachine stateMachine;
    private final Supplier<Timer> logFlushTimer;
    private int pendingFlushNum = 0;
    private long lastWrittenIndex;
    private volatile long flushedIndex;
    private final int forceSyncNum;
    private final long segmentMaxSize;
    private final long preallocatedSize;
    private final int bufferSize;

    RaftLogWorker(RaftPeerId selfId, RaftServerImpl raftServer, RaftStorage storage, RaftProperties properties) {
        this.name = selfId + "-" + this.getClass().getSimpleName();
        LOG.info("new {} for {}", (Object)this.name, (Object)storage);
        this.raftServer = raftServer;
        this.stateMachine = raftServer != null ? raftServer.getStateMachine() : null;
        this.storage = storage;
        this.segmentMaxSize = RaftServerConfigKeys.Log.segmentSizeMax(properties).getSize();
        this.preallocatedSize = RaftServerConfigKeys.Log.preallocatedSize(properties).getSize();
        this.bufferSize = RaftServerConfigKeys.Log.writeBufferSize(properties).getSizeInt();
        this.forceSyncNum = RaftServerConfigKeys.Log.forceSyncNum(properties);
        this.workerThread = new Thread((Runnable)this, this.name);
        Supplier<String> serverId = () -> raftServer == null || raftServer.getId() == null ? "null" : raftServer.getId().toString();
        this.logFlushTimer = JavaUtils.memoize(() -> RatisMetricsRegistry.getRegistry().timer(MetricRegistry.name(RaftLogWorker.class, (String[])new String[]{(String)serverId.get(), "flush-time"})));
    }

    void start(long latestIndex, File openSegmentFile) throws IOException {
        LOG.trace("{} start(latestIndex={}, openSegmentFile={})", new Object[]{this.name, latestIndex, openSegmentFile});
        this.lastWrittenIndex = latestIndex;
        this.flushedIndex = latestIndex;
        if (openSegmentFile != null) {
            Preconditions.assertTrue((boolean)openSegmentFile.exists());
            this.out = new LogOutputStream(openSegmentFile, true, this.segmentMaxSize, this.preallocatedSize, this.bufferSize);
        }
        this.workerThread.start();
    }

    void close() {
        this.running = false;
        this.workerThread.interrupt();
        try {
            this.workerThread.join(3000L);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        IOUtils.cleanup((Logger)LOG, (Closeable[])new Closeable[]{this.out});
        LOG.info("{} close()", (Object)this.name);
    }

    void syncWithSnapshot(long lastSnapshotIndex) {
        this.queue.clear();
        this.lastWrittenIndex = lastSnapshotIndex;
        this.flushedIndex = lastSnapshotIndex;
        this.pendingFlushNum = 0;
    }

    public String toString() {
        return this.name;
    }

    private SegmentedRaftLog.Task addIOTask(SegmentedRaftLog.Task task) {
        LOG.debug("{} adds IO task {}", (Object)this.name, (Object)task);
        try {
            if (!this.queue.offer(task, 1L, TimeUnit.SECONDS)) {
                Preconditions.assertTrue((boolean)this.isAlive(), (Object)"the worker thread is not alive");
                this.queue.put(task);
            }
        }
        catch (Throwable t) {
            if (t instanceof InterruptedException && !this.running) {
                LOG.info("Got InterruptedException when adding task " + task + ". The RaftLogWorker already stopped.");
            }
            ExitUtils.terminate((int)2, (String)("Failed to add IO task " + task), (Throwable)t, (Logger)LOG);
        }
        return task;
    }

    boolean isAlive() {
        return this.running && this.workerThread.isAlive();
    }

    @Override
    public void run() {
        while (this.running) {
            try {
                SegmentedRaftLog.Task task = this.queue.poll(1L, TimeUnit.SECONDS);
                if (task == null) continue;
                try {
                    task.execute();
                }
                catch (IOException e) {
                    if (task.getEndIndex() < this.lastWrittenIndex) {
                        LOG.info("Ignore IOException when handling task " + task + " which is smaller than the lastWrittenIndex. There should be a snapshot installed.", (Throwable)e);
                    }
                    throw e;
                }
                task.done();
            }
            catch (InterruptedException e) {
                if (this.running) {
                    LOG.warn("{} got interrupted while still running", (Object)Thread.currentThread().getName());
                }
                LOG.info(Thread.currentThread().getName() + " was interrupted, exiting. There are " + this.queue.size() + " tasks remaining in the queue.");
                Thread.currentThread().interrupt();
                return;
            }
            catch (Throwable t) {
                if (!this.running) {
                    LOG.info("{} got closed and hit exception", (Object)Thread.currentThread().getName(), (Object)t);
                    continue;
                }
                ExitUtils.terminate((int)1, (String)(Thread.currentThread().getName() + " failed."), (Throwable)t, (Logger)LOG);
            }
        }
    }

    private boolean shouldFlush() {
        return this.pendingFlushNum >= this.forceSyncNum || this.pendingFlushNum > 0 && this.queue.isEmpty();
    }

    private void flushWrites() throws IOException {
        if (this.out != null) {
            LOG.debug("flush data to " + this.out + ", reset pending_sync_number to 0");
            Timer.Context timerContext = this.logFlushTimer.get().time();
            try {
                CompletableFuture<Object> f = this.stateMachine != null ? this.stateMachine.flushStateMachineData(this.lastWrittenIndex) : CompletableFuture.completedFuture(null);
                this.out.flush();
                f.get();
            }
            catch (InterruptedException | ExecutionException e) {
                throw IOUtils.asIOException((Throwable)e);
            }
            finally {
                timerContext.stop();
            }
            this.updateFlushedIndex();
        }
    }

    private void updateFlushedIndex() {
        this.flushedIndex = this.lastWrittenIndex;
        this.pendingFlushNum = 0;
        if (this.raftServer != null) {
            this.raftServer.submitLocalSyncEvent();
        }
    }

    void startLogSegment(long startIndex) {
        this.addIOTask(new StartLogSegment(startIndex));
    }

    void rollLogSegment(LogSegment segmentToClose) {
        LOG.info("Rolling segment:{} index to:{}", (Object)this.name, (Object)(segmentToClose.getEndIndex() + 1L));
        this.addIOTask(new FinalizeLogSegment(segmentToClose));
        this.addIOTask(new StartLogSegment(segmentToClose.getEndIndex() + 1L));
    }

    SegmentedRaftLog.Task writeLogEntry(RaftProtos.LogEntryProto entry) {
        return this.addIOTask(new WriteLog(entry));
    }

    SegmentedRaftLog.Task truncate(RaftLogCache.TruncationSegments ts) {
        return this.addIOTask(new TruncateLog(ts));
    }

    long getFlushedIndex() {
        return this.flushedIndex;
    }

    private class TruncateLog
    extends SegmentedRaftLog.Task {
        private final RaftLogCache.TruncationSegments segments;

        TruncateLog(RaftLogCache.TruncationSegments ts) {
            this.segments = ts;
        }

        @Override
        void execute() throws IOException {
            IOUtils.cleanup(null, (Closeable[])new Closeable[]{RaftLogWorker.this.out});
            RaftLogWorker.this.out = null;
            if (this.segments.toTruncate != null) {
                File fileToTruncate = this.segments.toTruncate.isOpen ? RaftLogWorker.this.storage.getStorageDir().getOpenLogFile(this.segments.toTruncate.startIndex) : RaftLogWorker.this.storage.getStorageDir().getClosedLogFile(this.segments.toTruncate.startIndex, this.segments.toTruncate.endIndex);
                FileUtils.truncateFile((File)fileToTruncate, (long)this.segments.toTruncate.targetLength);
                File dstFile = RaftLogWorker.this.storage.getStorageDir().getClosedLogFile(this.segments.toTruncate.startIndex, this.segments.toTruncate.newEndIndex);
                Preconditions.assertTrue((!dstFile.exists() ? 1 : 0) != 0);
                FileUtils.move((File)fileToTruncate, (File)dstFile);
                RaftLogWorker.this.lastWrittenIndex = this.segments.toTruncate.newEndIndex;
            }
            if (this.segments.toDelete != null && this.segments.toDelete.length > 0) {
                long minStart = this.segments.toDelete[0].startIndex;
                for (RaftLogCache.SegmentFileInfo del : this.segments.toDelete) {
                    File delFile = del.isOpen ? RaftLogWorker.this.storage.getStorageDir().getOpenLogFile(del.startIndex) : RaftLogWorker.this.storage.getStorageDir().getClosedLogFile(del.startIndex, del.endIndex);
                    FileUtils.deleteFile((File)delFile);
                    minStart = Math.min(minStart, del.startIndex);
                }
                if (this.segments.toTruncate == null) {
                    RaftLogWorker.this.lastWrittenIndex = minStart - 1L;
                }
            }
            RaftLogWorker.this.updateFlushedIndex();
        }

        @Override
        long getEndIndex() {
            if (this.segments.toTruncate != null) {
                return this.segments.toTruncate.newEndIndex;
            }
            if (this.segments.toDelete.length > 0) {
                return this.segments.toDelete[this.segments.toDelete.length - 1].endIndex;
            }
            return -1L;
        }

        @Override
        public String toString() {
            return super.toString() + ": " + this.segments;
        }
    }

    private class StartLogSegment
    extends SegmentedRaftLog.Task {
        private final long newStartIndex;

        StartLogSegment(long newStartIndex) {
            this.newStartIndex = newStartIndex;
        }

        @Override
        void execute() throws IOException {
            File openFile = RaftLogWorker.this.storage.getStorageDir().getOpenLogFile(this.newStartIndex);
            LOG.debug("{} creating new log segment {}", (Object)RaftLogWorker.this.name, (Object)openFile);
            Preconditions.assertTrue((!openFile.exists() ? 1 : 0) != 0, (String)"open file %s exists for %s", (Object[])new Object[]{openFile, RaftLogWorker.this.name});
            Preconditions.assertTrue((RaftLogWorker.this.out == null && RaftLogWorker.this.pendingFlushNum == 0 ? 1 : 0) != 0);
            RaftLogWorker.this.out = new LogOutputStream(openFile, false, RaftLogWorker.this.segmentMaxSize, RaftLogWorker.this.preallocatedSize, RaftLogWorker.this.bufferSize);
            Preconditions.assertTrue((boolean)openFile.exists(), (String)"Failed to create file %s for %s", (Object[])new Object[]{openFile.getAbsolutePath(), RaftLogWorker.this.name});
        }

        @Override
        long getEndIndex() {
            return this.newStartIndex;
        }
    }

    private class FinalizeLogSegment
    extends SegmentedRaftLog.Task {
        private final LogSegment segmentToClose;

        FinalizeLogSegment(LogSegment segmentToClose) {
            this.segmentToClose = segmentToClose;
        }

        @Override
        public void execute() throws IOException {
            IOUtils.cleanup((Logger)LOG, (Closeable[])new Closeable[]{RaftLogWorker.this.out});
            RaftLogWorker.this.out = null;
            Preconditions.assertTrue((this.segmentToClose != null ? 1 : 0) != 0);
            File openFile = RaftLogWorker.this.storage.getStorageDir().getOpenLogFile(this.segmentToClose.getStartIndex());
            LOG.debug("{} finalizing log segment {}", (Object)RaftLogWorker.this.name, (Object)openFile);
            Preconditions.assertTrue((boolean)openFile.exists(), () -> RaftLogWorker.this.name + ": File " + openFile + " does not exist, segmentToClose=" + this.segmentToClose.toDebugString());
            if (this.segmentToClose.numOfEntries() > 0) {
                File dstFile = RaftLogWorker.this.storage.getStorageDir().getClosedLogFile(this.segmentToClose.getStartIndex(), this.segmentToClose.getEndIndex());
                Preconditions.assertTrue((!dstFile.exists() ? 1 : 0) != 0);
                FileUtils.move((File)openFile, (File)dstFile);
            } else {
                FileUtils.deleteFile((File)openFile);
            }
            RaftLogWorker.this.updateFlushedIndex();
        }

        @Override
        long getEndIndex() {
            return this.segmentToClose.getEndIndex();
        }

        @Override
        public String toString() {
            return super.toString() + ": " + this.segmentToClose.toDebugString();
        }
    }

    private class WriteLog
    extends SegmentedRaftLog.Task {
        private final RaftProtos.LogEntryProto entry;
        private final CompletableFuture<?> stateMachineFuture;
        private final CompletableFuture<Long> combined;

        WriteLog(RaftProtos.LogEntryProto entry) {
            this.entry = ProtoUtils.removeStateMachineData((RaftProtos.LogEntryProto)entry);
            if (this.entry == entry || RaftLogWorker.this.stateMachine == null) {
                this.stateMachineFuture = null;
            } else {
                try {
                    this.stateMachineFuture = RaftLogWorker.this.stateMachine.writeStateMachineData(entry);
                }
                catch (Throwable e) {
                    LOG.error("{}: writeStateMachineData failed for index:{} proto:{}", new Object[]{RaftLogWorker.this.raftServer.getId(), entry.getIndex(), ServerProtoUtils.toString(entry), e.getMessage()});
                    throw e;
                }
            }
            this.combined = this.stateMachineFuture == null ? super.getFuture() : super.getFuture().thenCombine(this.stateMachineFuture, (index, stateMachineResult) -> index);
        }

        @Override
        CompletableFuture<Long> getFuture() {
            return this.combined;
        }

        @Override
        public void execute() throws IOException {
            Preconditions.assertTrue((RaftLogWorker.this.out != null ? 1 : 0) != 0);
            Preconditions.assertTrue((RaftLogWorker.this.lastWrittenIndex + 1L == this.entry.getIndex() ? 1 : 0) != 0, (String)"lastWrittenIndex == %s, entry == %s", (Object[])new Object[]{RaftLogWorker.this.lastWrittenIndex, this.entry});
            RaftLogWorker.this.out.write(this.entry);
            RaftLogWorker.this.lastWrittenIndex = this.entry.getIndex();
            RaftLogWorker.this.pendingFlushNum++;
            if (RaftLogWorker.this.shouldFlush()) {
                RaftLogWorker.this.flushWrites();
            }
        }

        @Override
        long getEndIndex() {
            return this.entry.getIndex();
        }

        @Override
        public String toString() {
            return super.toString() + ": " + ServerProtoUtils.toLogEntryString(this.entry);
        }
    }
}

