/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.impl;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.management.ObjectName;
import org.apache.ratis.conf.RaftProperties;
import org.apache.ratis.protocol.ClientId;
import org.apache.ratis.protocol.GroupMismatchException;
import org.apache.ratis.protocol.LeaderNotReadyException;
import org.apache.ratis.protocol.Message;
import org.apache.ratis.protocol.NotLeaderException;
import org.apache.ratis.protocol.RaftClientAsynchronousProtocol;
import org.apache.ratis.protocol.RaftClientProtocol;
import org.apache.ratis.protocol.RaftClientReply;
import org.apache.ratis.protocol.RaftClientRequest;
import org.apache.ratis.protocol.RaftException;
import org.apache.ratis.protocol.RaftGroup;
import org.apache.ratis.protocol.RaftGroupId;
import org.apache.ratis.protocol.RaftPeer;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.protocol.ReconfigurationInProgressException;
import org.apache.ratis.protocol.ServerInformationReply;
import org.apache.ratis.protocol.ServerInformationRequest;
import org.apache.ratis.protocol.ServerNotReadyException;
import org.apache.ratis.protocol.SetConfigurationRequest;
import org.apache.ratis.protocol.StaleReadException;
import org.apache.ratis.protocol.StateMachineException;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.RaftServerMXBean;
import org.apache.ratis.server.RaftServerRpc;
import org.apache.ratis.server.impl.CommitInfoCache;
import org.apache.ratis.server.impl.FollowerInfo;
import org.apache.ratis.server.impl.FollowerState;
import org.apache.ratis.server.impl.LeaderElection;
import org.apache.ratis.server.impl.LeaderState;
import org.apache.ratis.server.impl.LogAppender;
import org.apache.ratis.server.impl.PendingRequest;
import org.apache.ratis.server.impl.RaftConfiguration;
import org.apache.ratis.server.impl.RaftServerProxy;
import org.apache.ratis.server.impl.RetryCache;
import org.apache.ratis.server.impl.RoleInfo;
import org.apache.ratis.server.impl.ServerProtoUtils;
import org.apache.ratis.server.impl.ServerState;
import org.apache.ratis.server.protocol.RaftServerAsynchronousProtocol;
import org.apache.ratis.server.protocol.RaftServerProtocol;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.storage.FileInfo;
import org.apache.ratis.server.storage.RaftLog;
import org.apache.ratis.server.storage.RaftStorageDirectory;
import org.apache.ratis.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.ratis.shaded.com.google.protobuf.ByteString;
import org.apache.ratis.shaded.proto.RaftProtos;
import org.apache.ratis.statemachine.SnapshotInfo;
import org.apache.ratis.statemachine.StateMachine;
import org.apache.ratis.statemachine.TransactionContext;
import org.apache.ratis.statemachine.impl.TransactionContextImpl;
import org.apache.ratis.util.CodeInjectionForTesting;
import org.apache.ratis.util.FileUtils;
import org.apache.ratis.util.IOUtils;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.JmxRegister;
import org.apache.ratis.util.LifeCycle;
import org.apache.ratis.util.Preconditions;
import org.apache.ratis.util.ProtoUtils;
import org.apache.ratis.util.TimeDuration;
import org.apache.ratis.util.Timestamp;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RaftServerImpl
implements RaftServerProtocol,
RaftServerAsynchronousProtocol,
RaftClientProtocol,
RaftClientAsynchronousProtocol {
    public static final Logger LOG = LoggerFactory.getLogger(RaftServerImpl.class);
    private static final String CLASS_NAME = RaftServerImpl.class.getSimpleName();
    static final String REQUEST_VOTE = CLASS_NAME + ".requestVote";
    static final String APPEND_ENTRIES = CLASS_NAME + ".appendEntries";
    static final String INSTALL_SNAPSHOT = CLASS_NAME + ".installSnapshot";
    private final RaftServerProxy proxy;
    private final StateMachine stateMachine;
    private final int minTimeoutMs;
    private final int maxTimeoutMs;
    private final int rpcSlownessTimeoutMs;
    private final LifeCycle lifeCycle;
    private final ServerState state;
    private final RaftGroupId groupId;
    private final Supplier<RaftPeer> peerSupplier = JavaUtils.memoize(() -> new RaftPeer(this.getId(), this.getServerRpc().getInetSocketAddress()));
    private final RoleInfo role;
    private volatile FollowerState heartbeatMonitor;
    private volatile LeaderElection electionDaemon;
    private volatile LeaderState leaderState;
    private final RetryCache retryCache;
    private final CommitInfoCache commitInfoCache = new CommitInfoCache();
    private final RaftServerJmxAdapter jmxAdapter;

    RaftServerImpl(RaftGroup group, StateMachine stateMachine, RaftServerProxy proxy) throws IOException {
        RaftPeerId id = proxy.getId();
        LOG.debug("{}: new RaftServerImpl for {}", (Object)id, (Object)group);
        this.groupId = group.getGroupId();
        this.lifeCycle = new LifeCycle((Object)id);
        this.stateMachine = stateMachine;
        this.role = new RoleInfo();
        RaftProperties properties = proxy.getProperties();
        this.minTimeoutMs = RaftServerConfigKeys.Rpc.timeoutMin(properties).toInt(TimeUnit.MILLISECONDS);
        this.maxTimeoutMs = RaftServerConfigKeys.Rpc.timeoutMax(properties).toInt(TimeUnit.MILLISECONDS);
        this.rpcSlownessTimeoutMs = RaftServerConfigKeys.Rpc.slownessTimeout(properties).toInt(TimeUnit.MILLISECONDS);
        Preconditions.assertTrue((this.maxTimeoutMs > this.minTimeoutMs ? 1 : 0) != 0, (String)"max timeout: %s, min timeout: %s", (Object[])new Object[]{this.maxTimeoutMs, this.minTimeoutMs});
        this.proxy = proxy;
        this.state = new ServerState(id, group, properties, this, stateMachine);
        this.retryCache = this.initRetryCache(properties);
        this.jmxAdapter = new RaftServerJmxAdapter();
    }

    private RetryCache initRetryCache(RaftProperties prop) {
        int capacity = RaftServerConfigKeys.RetryCache.capacity(prop);
        TimeDuration expireTime = RaftServerConfigKeys.RetryCache.expiryTime(prop);
        return new RetryCache(capacity, expireTime);
    }

    LogAppender newLogAppender(LeaderState state, RaftPeer peer, Timestamp lastRpcTime, long nextIndex, boolean attendVote) {
        FollowerInfo f = new FollowerInfo(peer, lastRpcTime, nextIndex, attendVote, this.rpcSlownessTimeoutMs);
        return this.getProxy().getFactory().newLogAppender(this, state, f);
    }

    RaftPeer getPeer() {
        return this.peerSupplier.get();
    }

    int getMinTimeoutMs() {
        return this.minTimeoutMs;
    }

    int getMaxTimeoutMs() {
        return this.maxTimeoutMs;
    }

    int getRandomTimeoutMs() {
        return this.minTimeoutMs + ThreadLocalRandom.current().nextInt(this.maxTimeoutMs - this.minTimeoutMs + 1);
    }

    public RaftGroupId getGroupId() {
        return this.groupId;
    }

    public StateMachine getStateMachine() {
        return this.stateMachine;
    }

    @VisibleForTesting
    public RetryCache getRetryCache() {
        return this.retryCache;
    }

    public RaftServerProxy getProxy() {
        return this.proxy;
    }

    public RaftServerRpc getServerRpc() {
        return this.proxy.getServerRpc();
    }

    private void setRole(RaftProtos.RaftPeerRole newRole, String op) {
        LOG.info("{} changes role from {} to {} at term {} for {}", new Object[]{this.getId(), this.role, newRole, this.state.getCurrentTerm(), op});
        this.role.transitionRole(newRole);
    }

    boolean start() {
        if (!this.lifeCycle.compareAndTransition(LifeCycle.State.NEW, LifeCycle.State.STARTING)) {
            return false;
        }
        LOG.info("{}: start {}", (Object)this.getId(), (Object)this.groupId);
        this.state.start();
        RaftConfiguration conf = this.getRaftConf();
        if (conf != null && conf.contains(this.getId())) {
            LOG.debug("{} starts as a follower, conf={}", (Object)this.getId(), (Object)conf);
            this.startAsFollower();
        } else {
            LOG.debug("{} starts with initializing state, conf={}", (Object)this.getId(), (Object)conf);
            this.startInitializing();
        }
        RaftServerImpl.registerMBean(this.getId(), this.getGroupId(), this.jmxAdapter, this.jmxAdapter);
        return true;
    }

    static boolean registerMBean(RaftPeerId id, RaftGroupId groupdId, RaftServerMXBean mBean, JmxRegister jmx) {
        String prefix = "Ratis:service=RaftServer,group=" + groupdId + ",id=";
        String registered = jmx.register((Object)mBean, Arrays.asList(() -> prefix + id, () -> prefix + ObjectName.quote(id.toString())));
        return registered != null;
    }

    private void startAsFollower() {
        this.setRole(RaftProtos.RaftPeerRole.FOLLOWER, "startAsFollower");
        this.startHeartbeatMonitor();
        this.lifeCycle.transition(LifeCycle.State.RUNNING);
    }

    private void startInitializing() {
        this.setRole(RaftProtos.RaftPeerRole.FOLLOWER, "startInitializing");
    }

    public ServerState getState() {
        return this.state;
    }

    LeaderState getLeaderState() {
        return this.leaderState;
    }

    public RaftPeerId getId() {
        return this.getState().getSelfId();
    }

    RaftProtos.RaftPeerRole getRole() {
        return this.role.getCurrentRole();
    }

    RaftConfiguration getRaftConf() {
        return this.getState().getRaftConf();
    }

    RaftGroup getGroup() {
        return new RaftGroup(this.groupId, this.getRaftConf().getPeers());
    }

    void shutdown(boolean deleteDirectory) {
        this.lifeCycle.checkStateAndClose(() -> {
            LOG.info("{}: shutdown {}", (Object)this.getId(), (Object)this.groupId);
            try {
                this.jmxAdapter.unregister();
            }
            catch (Exception ignored) {
                LOG.warn("Failed to un-register RaftServer JMX bean for " + this.getId(), (Throwable)ignored);
            }
            try {
                this.shutdownHeartbeatMonitor();
            }
            catch (Exception ignored) {
                LOG.warn("Failed to shutdown heartbeat monitor for " + this.getId(), (Throwable)ignored);
            }
            try {
                this.shutdownElectionDaemon();
            }
            catch (Exception ignored) {
                LOG.warn("Failed to shutdown election daemon for " + this.getId(), (Throwable)ignored);
            }
            try {
                this.shutdownLeaderState(true);
            }
            catch (Exception ignored) {
                LOG.warn("Failed to shutdown leader state monitor for " + this.getId(), (Throwable)ignored);
            }
            try {
                this.state.close();
            }
            catch (Exception ignored) {
                LOG.warn("Failed to close state for " + this.getId(), (Throwable)ignored);
            }
            if (deleteDirectory) {
                RaftStorageDirectory dir = this.state.getStorage().getStorageDir();
                try {
                    FileUtils.deleteFully((File)dir.getRoot());
                }
                catch (Exception ignored) {
                    LOG.warn(this.getId() + ": Failed to remove RaftStorageDirectory " + dir, (Throwable)ignored);
                }
            }
        });
    }

    public boolean isAlive() {
        return !this.lifeCycle.getCurrentState().isOneOf(new LifeCycle.State[]{LifeCycle.State.CLOSING, LifeCycle.State.CLOSED});
    }

    public boolean isFollower() {
        return this.role.isFollower();
    }

    public boolean isCandidate() {
        return this.role.isCandidate();
    }

    public boolean isLeader() {
        return this.role.isLeader();
    }

    synchronized boolean changeToFollower(long newTerm, boolean sync) throws IOException {
        RaftProtos.RaftPeerRole old = this.role.getCurrentRole();
        boolean metadataUpdated = this.state.updateCurrentTerm(newTerm);
        if (old != RaftProtos.RaftPeerRole.FOLLOWER) {
            this.setRole(RaftProtos.RaftPeerRole.FOLLOWER, "changeToFollower");
            if (old == RaftProtos.RaftPeerRole.LEADER) {
                this.shutdownLeaderState(false);
            } else if (old == RaftProtos.RaftPeerRole.CANDIDATE) {
                this.shutdownElectionDaemon();
            }
            this.startHeartbeatMonitor();
        }
        if (metadataUpdated && sync) {
            this.state.persistMetadata();
        }
        return metadataUpdated;
    }

    private synchronized void shutdownLeaderState(boolean allowNull) {
        if (this.leaderState == null) {
            if (!allowNull) {
                throw new NullPointerException("leaderState == null");
            }
        } else {
            this.leaderState.stop();
            this.leaderState = null;
        }
    }

    private void shutdownElectionDaemon() {
        LeaderElection election = this.electionDaemon;
        if (election != null) {
            election.stopRunning();
        }
        this.electionDaemon = null;
    }

    synchronized void changeToLeader() {
        Preconditions.assertTrue((boolean)this.isCandidate());
        this.shutdownElectionDaemon();
        this.setRole(RaftProtos.RaftPeerRole.LEADER, "changeToLeader");
        this.state.becomeLeader();
        this.leaderState = new LeaderState(this, this.getProxy().getProperties());
        RaftProtos.LogEntryProto e = this.leaderState.start();
        this.getState().setRaftConf(e.getIndex(), ServerProtoUtils.toRaftConfiguration(e));
    }

    private void startHeartbeatMonitor() {
        Preconditions.assertTrue((this.heartbeatMonitor == null ? 1 : 0) != 0, (Object)"heartbeatMonitor != null");
        LOG.debug("{} starts heartbeatMonitor", (Object)this.getId());
        this.heartbeatMonitor = new FollowerState(this);
        this.heartbeatMonitor.start();
    }

    private void shutdownHeartbeatMonitor() {
        FollowerState hm = this.heartbeatMonitor;
        if (hm != null) {
            hm.stopRunning();
            hm.interrupt();
        }
        this.heartbeatMonitor = null;
    }

    Collection<RaftProtos.CommitInfoProto> getCommitInfos() {
        ArrayList<RaftProtos.CommitInfoProto> infos = new ArrayList<RaftProtos.CommitInfoProto>();
        infos.add(this.commitInfoCache.update(this.getPeer(), this.state.getLog().getLastCommittedIndex()));
        if (this.isLeader()) {
            Optional.ofNullable(this.leaderState).ifPresent(leader -> leader.updateFollowerCommitInfos(this.commitInfoCache, infos));
        } else {
            this.getRaftConf().getPeers().stream().filter(p -> !p.getId().equals((Object)this.state.getSelfId())).map(RaftPeer::getId).map(this.commitInfoCache::get).filter(i -> i != null).forEach(infos::add);
        }
        return infos;
    }

    ServerInformationReply getServerInformation(ServerInformationRequest request) {
        return new ServerInformationReply((RaftClientRequest)request, this.getRoleInfoProto(), this.state.getStorage().getStorageDir().hasMetaFile(), this.getCommitInfos(), this.getGroup());
    }

    public RaftProtos.RoleInfoProto getRoleInfoProto() {
        RaftProtos.RaftPeerRole currentRole = this.role.getCurrentRole();
        RaftProtos.RoleInfoProto.Builder roleInfo = RaftProtos.RoleInfoProto.newBuilder().setSelf(ProtoUtils.toRaftPeerProto((RaftPeer)this.getPeer())).setRole(currentRole).setRoleElapsedTimeMs(this.role.getRoleElapsedTimeMs());
        switch (currentRole) {
            case CANDIDATE: {
                RaftProtos.CandidateInfoProto.Builder candidate = RaftProtos.CandidateInfoProto.newBuilder().setLastLeaderElapsedTimeMs(this.state.getLastLeaderElapsedTimeMs());
                roleInfo.setCandidateInfo(candidate);
                break;
            }
            case FOLLOWER: {
                RaftProtos.FollowerInfoProto.Builder follower = RaftProtos.FollowerInfoProto.newBuilder().setLeaderInfo(this.getServerRpcProto(this.getRaftConf().getPeer(this.state.getLeaderId()), this.heartbeatMonitor.getLastRpcTime().elapsedTimeMs())).setInLogSync(this.heartbeatMonitor.isInLogSync());
                roleInfo.setFollowerInfo(follower);
                break;
            }
            case LEADER: {
                RaftProtos.LeaderInfoProto.Builder leader = RaftProtos.LeaderInfoProto.newBuilder();
                Stream<LogAppender> stream = this.getLeaderState().getLogAppenders();
                stream.forEach(appender -> leader.addFollowerInfo(this.getServerRpcProto(appender.getFollower().getPeer(), appender.getFollower().getLastRpcResponseTime().elapsedTimeMs())));
                roleInfo.setLeaderInfo(leader);
                break;
            }
            default: {
                throw new IllegalStateException("incorrect role of server " + currentRole);
            }
        }
        return roleInfo.build();
    }

    private RaftProtos.ServerRpcProto getServerRpcProto(RaftPeer peer, long delay) {
        if (peer == null) {
            return RaftProtos.ServerRpcProto.getDefaultInstance();
        }
        return RaftProtos.ServerRpcProto.newBuilder().setId(ProtoUtils.toRaftPeerProto((RaftPeer)peer)).setLastRpcElapsedTimeMs(delay).build();
    }

    synchronized void changeToCandidate() {
        Preconditions.assertTrue((boolean)this.isFollower());
        this.shutdownHeartbeatMonitor();
        this.setRole(RaftProtos.RaftPeerRole.CANDIDATE, "changeToCandidate");
        if (this.state.checkForExtendedNoLeader()) {
            this.stateMachine.notifyExtendedNoLeader(this.getGroup(), this.getRoleInfoProto());
        }
        this.electionDaemon = new LeaderElection(this);
        this.electionDaemon.start();
    }

    public String toString() {
        return String.format("%8s ", this.role) + this.groupId + " " + this.state + " " + this.lifeCycle.getCurrentState();
    }

    private CompletableFuture<RaftClientReply> checkLeaderState(RaftClientRequest request, RetryCache.CacheEntry entry) {
        try {
            this.assertGroup(request.getRequestorId(), request.getRaftGroupId());
        }
        catch (GroupMismatchException e) {
            return RetryCache.failWithException(e, entry);
        }
        if (!this.isLeader()) {
            NotLeaderException exception = this.generateNotLeaderException();
            RaftClientReply reply = new RaftClientReply(request, (RaftException)exception, this.getCommitInfos());
            return RetryCache.failWithReply(reply, entry);
        }
        if (this.leaderState == null || !this.leaderState.isReady()) {
            RetryCache.CacheEntry cacheEntry = this.retryCache.get(request.getClientId(), request.getCallId());
            if (cacheEntry != null && cacheEntry.isCompletedNormally()) {
                return cacheEntry.getReplyFuture();
            }
            return RetryCache.failWithException((Throwable)new LeaderNotReadyException(this.getId()), entry);
        }
        return null;
    }

    NotLeaderException generateNotLeaderException() {
        if (this.lifeCycle.getCurrentState() != LifeCycle.State.RUNNING) {
            return new NotLeaderException(this.getId(), null, null);
        }
        RaftPeerId leaderId = this.state.getLeaderId();
        if (leaderId == null || leaderId.equals((Object)this.state.getSelfId())) {
            RaftPeer suggestedLeader = this.getRaftConf().getRandomPeer(this.state.getSelfId());
            leaderId = suggestedLeader == null ? null : suggestedLeader.getId();
        }
        RaftConfiguration conf = this.getRaftConf();
        Collection<RaftPeer> peers = conf.getPeers();
        return new NotLeaderException(this.getId(), conf.getPeer(leaderId), peers.toArray(new RaftPeer[peers.size()]));
    }

    private LifeCycle.State assertLifeCycleState(LifeCycle.State ... expected) throws ServerNotReadyException {
        return this.lifeCycle.assertCurrentState((n, c) -> new ServerNotReadyException("Server " + n + " is not " + Arrays.toString(expected) + ": current state is " + c), expected);
    }

    void assertGroup(Object requestorId, RaftGroupId requestorGroupId) throws GroupMismatchException {
        if (!this.groupId.equals((Object)requestorGroupId)) {
            throw new GroupMismatchException(this.getId() + ": The group (" + requestorGroupId + ") of " + requestorId + " does not match the group (" + this.groupId + ") of the server " + this.getId());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CompletableFuture<RaftClientReply> appendTransaction(RaftClientRequest request, TransactionContext context, RetryCache.CacheEntry cacheEntry) throws IOException {
        PendingRequest pending;
        this.assertLifeCycleState(LifeCycle.State.RUNNING);
        RaftServerImpl raftServerImpl = this;
        synchronized (raftServerImpl) {
            long entryIndex;
            CompletableFuture<RaftClientReply> reply = this.checkLeaderState(request, cacheEntry);
            if (reply != null) {
                return reply;
            }
            try {
                entryIndex = this.state.applyLog(context, request.getClientId(), request.getCallId());
            }
            catch (StateMachineException e) {
                RaftClientReply exceptionReply = new RaftClientReply(request, (RaftException)e, this.getCommitInfos());
                cacheEntry.failWithReply(exceptionReply);
                if (this.isLeader() && this.leaderState != null) {
                    this.leaderState.submitUpdateStateEvent(new LeaderState.StateUpdateEvent(LeaderState.StateUpdateEventType.STEPDOWN, this.leaderState.getCurrentTerm()));
                }
                return CompletableFuture.completedFuture(exceptionReply);
            }
            pending = this.leaderState.addPendingRequest(entryIndex, request, context);
            this.leaderState.notifySenders();
        }
        return pending.getFuture();
    }

    public CompletableFuture<RaftClientReply> submitClientRequestAsync(RaftClientRequest request) throws IOException {
        this.assertLifeCycleState(LifeCycle.State.RUNNING);
        LOG.debug("{}: receive client request({})", (Object)this.getId(), (Object)request);
        if (request.is(RaftProtos.RaftClientRequestProto.TypeCase.STALEREAD)) {
            return this.staleReadAsync(request);
        }
        CompletableFuture<RaftClientReply> reply = this.checkLeaderState(request, null);
        if (reply != null) {
            return reply;
        }
        StateMachine stateMachine = this.getStateMachine();
        if (request.is(RaftProtos.RaftClientRequestProto.TypeCase.READ)) {
            return this.processQueryFuture(stateMachine.query(request.getMessage()), request);
        }
        RetryCache.CacheQueryResult previousResult = this.retryCache.queryCache(request.getClientId(), request.getCallId());
        if (previousResult.isRetry()) {
            return previousResult.getEntry().getReplyFuture();
        }
        RetryCache.CacheEntry cacheEntry = previousResult.getEntry();
        TransactionContext context = stateMachine.startTransaction(request);
        if (context.getException() != null) {
            RaftClientReply exceptionReply = new RaftClientReply(request, (RaftException)new StateMachineException(this.getId(), (Throwable)context.getException()), this.getCommitInfos());
            cacheEntry.failWithReply(exceptionReply);
            return CompletableFuture.completedFuture(exceptionReply);
        }
        return this.appendTransaction(request, context, cacheEntry);
    }

    private CompletableFuture<RaftClientReply> staleReadAsync(RaftClientRequest request) {
        long minIndex = request.getType().getStaleRead().getMinIndex();
        long commitIndex = this.state.getLog().getLastCommittedIndex();
        LOG.debug("{}: minIndex={}, commitIndex={}", new Object[]{this.getId(), minIndex, commitIndex});
        if (commitIndex < minIndex) {
            StaleReadException e = new StaleReadException("Unable to serve stale-read due to server commit index = " + commitIndex + " < min = " + minIndex);
            return CompletableFuture.completedFuture(new RaftClientReply(request, (RaftException)new StateMachineException(this.getId(), (Throwable)e), this.getCommitInfos()));
        }
        return this.processQueryFuture(this.getStateMachine().queryStale(request.getMessage(), minIndex), request);
    }

    CompletableFuture<RaftClientReply> processQueryFuture(CompletableFuture<Message> queryFuture, RaftClientRequest request) {
        return ((CompletableFuture)queryFuture.thenApply(r -> new RaftClientReply(request, r, this.getCommitInfos()))).exceptionally(e -> {
            if ((e = JavaUtils.unwrapCompletionException((Throwable)e)) instanceof StateMachineException) {
                return new RaftClientReply(request, (RaftException)((StateMachineException)((Object)e)), this.getCommitInfos());
            }
            throw new CompletionException((Throwable)e);
        });
    }

    public RaftClientReply submitClientRequest(RaftClientRequest request) throws IOException {
        return this.waitForReply(this.getId(), request, this.submitClientRequestAsync(request));
    }

    RaftClientReply waitForReply(RaftPeerId id, RaftClientRequest request, CompletableFuture<RaftClientReply> future) throws IOException {
        return RaftServerImpl.waitForReply(id, request, future, e -> new RaftClientReply(request, e, this.getCommitInfos()));
    }

    static <REPLY extends RaftClientReply> REPLY waitForReply(RaftPeerId id, RaftClientRequest request, CompletableFuture<REPLY> future, Function<RaftException, REPLY> exceptionReply) throws IOException {
        try {
            return (REPLY)((RaftClientReply)future.get());
        }
        catch (InterruptedException e) {
            String s = id + ": Interrupted when waiting for reply, request=" + request;
            LOG.info(s, (Throwable)e);
            throw IOUtils.toInterruptedIOException((String)s, (InterruptedException)e);
        }
        catch (ExecutionException e) {
            RaftClientReply reply;
            Throwable cause = e.getCause();
            if (cause == null) {
                throw new IOException(e);
            }
            if ((cause instanceof NotLeaderException || cause instanceof StateMachineException) && (reply = (RaftClientReply)exceptionReply.apply((RaftException)cause)) != null) {
                return (REPLY)reply;
            }
            throw IOUtils.asIOException((Throwable)cause);
        }
    }

    public RaftClientReply setConfiguration(SetConfigurationRequest request) throws IOException {
        return this.waitForReply(this.getId(), (RaftClientRequest)request, this.setConfigurationAsync(request));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CompletableFuture<RaftClientReply> setConfigurationAsync(SetConfigurationRequest request) throws IOException {
        PendingRequest pending;
        LOG.debug("{}: receive setConfiguration({})", (Object)this.getId(), (Object)request);
        this.assertLifeCycleState(LifeCycle.State.RUNNING);
        this.assertGroup(request.getRequestorId(), request.getRaftGroupId());
        CompletableFuture<RaftClientReply> reply = this.checkLeaderState((RaftClientRequest)request, null);
        if (reply != null) {
            return reply;
        }
        RaftPeer[] peersInNewConf = request.getPeersInNewConf();
        RaftServerImpl raftServerImpl = this;
        synchronized (raftServerImpl) {
            reply = this.checkLeaderState((RaftClientRequest)request, null);
            if (reply != null) {
                return reply;
            }
            RaftConfiguration current = this.getRaftConf();
            if (!current.isStable() || this.leaderState.inStagingState() || !this.state.isConfCommitted()) {
                throw new ReconfigurationInProgressException("Reconfiguration is already in progress: " + current);
            }
            if (current.hasNoChange(peersInNewConf)) {
                PendingRequest pending2 = new PendingRequest(request);
                pending2.setReply(new RaftClientReply((RaftClientRequest)request, this.getCommitInfos()));
                return pending2.getFuture();
            }
            this.getServerRpc().addPeers(Arrays.asList(peersInNewConf));
            pending = this.leaderState.startSetConfiguration(request);
        }
        return pending.getFuture();
    }

    private boolean shouldWithholdVotes(long candidateTerm) {
        if (this.state.getCurrentTerm() < candidateTerm) {
            return false;
        }
        if (this.isLeader()) {
            return true;
        }
        return this.isFollower() && this.state.hasLeader() && this.heartbeatMonitor.shouldWithholdVotes();
    }

    private boolean shouldSendShutdown(RaftPeerId candidateId, TermIndex candidateLastEntry) {
        return this.isLeader() && this.getRaftConf().isStable() && this.getState().isConfCommitted() && !this.getRaftConf().containsInConf(candidateId) && candidateLastEntry.getIndex() < this.getRaftConf().getLogEntryIndex() && !this.leaderState.isBootStrappingPeer(candidateId);
    }

    @Override
    public RaftProtos.RequestVoteReplyProto requestVote(RaftProtos.RequestVoteRequestProto r) throws IOException {
        RaftProtos.RaftRpcRequestProto request = r.getServerRequest();
        return this.requestVote(RaftPeerId.valueOf((ByteString)request.getRequestorId()), ProtoUtils.toRaftGroupId((RaftProtos.RaftGroupIdProto)request.getRaftGroupId()), r.getCandidateTerm(), ServerProtoUtils.toTermIndex(r.getCandidateLastEntry()));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private RaftProtos.RequestVoteReplyProto requestVote(RaftPeerId candidateId, RaftGroupId candidateGroupId, long candidateTerm, TermIndex candidateLastEntry) throws IOException {
        RaftProtos.RequestVoteReplyProto reply;
        CodeInjectionForTesting.execute((String)REQUEST_VOTE, (Object)this.getId(), (Object)candidateId, (Object[])new Object[]{candidateTerm, candidateLastEntry});
        LOG.debug("{}: receive requestVote({}, {}, {}, {})", new Object[]{this.getId(), candidateId, candidateGroupId, candidateTerm, candidateLastEntry});
        this.assertLifeCycleState(LifeCycle.State.RUNNING);
        this.assertGroup(candidateId, candidateGroupId);
        boolean voteGranted = false;
        boolean shouldShutdown = false;
        RaftServerImpl raftServerImpl = this;
        synchronized (raftServerImpl) {
            if (this.shouldWithholdVotes(candidateTerm)) {
                LOG.info("{}-{}: Withhold vote from candidate {} with term {}. State: leader={}, term={}, lastRpcElapsed={}", new Object[]{this.getId(), this.role, candidateId, candidateTerm, this.state.getLeaderId(), this.state.getCurrentTerm(), this.isFollower() ? this.heartbeatMonitor.getLastRpcTime().elapsedTimeMs() + "ms" : null});
            } else if (this.state.recognizeCandidate(candidateId, candidateTerm)) {
                boolean termUpdated = this.changeToFollower(candidateTerm, false);
                if (this.state.isLogUpToDate(candidateLastEntry)) {
                    this.heartbeatMonitor.updateLastRpcTime(false);
                    this.state.grantVote(candidateId);
                    voteGranted = true;
                }
                if (termUpdated || voteGranted) {
                    this.state.persistMetadata();
                }
            }
            if (!voteGranted && this.shouldSendShutdown(candidateId, candidateLastEntry)) {
                shouldShutdown = true;
            }
            reply = ServerProtoUtils.toRequestVoteReplyProto(candidateId, this.getId(), this.groupId, voteGranted, this.state.getCurrentTerm(), shouldShutdown);
            if (LOG.isDebugEnabled()) {
                LOG.debug("{} replies to vote request: {}. Peer's state: {}", new Object[]{this.getId(), ProtoUtils.toString((RaftProtos.RequestVoteReplyProto)reply), this.state});
            }
        }
        return reply;
    }

    private void validateEntries(long expectedTerm, TermIndex previous, RaftProtos.LogEntryProto ... entries) {
        if (entries != null && entries.length > 0) {
            long index0 = entries[0].getIndex();
            if (previous == null || previous.getTerm() == 0L) {
                Preconditions.assertTrue((index0 == 0L ? 1 : 0) != 0, (String)"Unexpected Index: previous is null but entries[%s].getIndex()=%s", (Object[])new Object[]{0, index0});
            } else {
                Preconditions.assertTrue((previous.getIndex() == index0 - 1L ? 1 : 0) != 0, (String)"Unexpected Index: previous is %s but entries[%s].getIndex()=%s", (Object[])new Object[]{previous, 0, index0});
            }
            for (int i = 0; i < entries.length; ++i) {
                long t = entries[i].getTerm();
                Preconditions.assertTrue((expectedTerm >= t ? 1 : 0) != 0, (String)"Unexpected Term: entries[%s].getTerm()=%s but expectedTerm=%s", (Object[])new Object[]{i, t, expectedTerm});
                long indexi = entries[i].getIndex();
                Preconditions.assertTrue((indexi == index0 + (long)i ? 1 : 0) != 0, (String)"Unexpected Index: entries[%s].getIndex()=%s but entries[0].getIndex()=%s", (Object[])new Object[]{i, indexi, index0});
            }
        }
    }

    @Override
    public RaftProtos.AppendEntriesReplyProto appendEntries(RaftProtos.AppendEntriesRequestProto r) throws IOException {
        try {
            return this.appendEntriesAsync(r).join();
        }
        catch (CompletionException e) {
            throw IOUtils.asIOException((Throwable)JavaUtils.unwrapCompletionException((Throwable)e));
        }
    }

    @Override
    public CompletableFuture<RaftProtos.AppendEntriesReplyProto> appendEntriesAsync(RaftProtos.AppendEntriesRequestProto r) throws IOException {
        RaftProtos.RaftRpcRequestProto request = r.getServerRequest();
        RaftProtos.LogEntryProto[] entries = r.getEntriesList().toArray(new RaftProtos.LogEntryProto[r.getEntriesCount()]);
        TermIndex previous = r.hasPreviousLog() ? ServerProtoUtils.toTermIndex(r.getPreviousLog()) : null;
        return this.appendEntriesAsync(RaftPeerId.valueOf((ByteString)request.getRequestorId()), ProtoUtils.toRaftGroupId((RaftProtos.RaftGroupIdProto)request.getRaftGroupId()), r.getLeaderTerm(), previous, r.getLeaderCommit(), request.getCallId(), r.getInitializing(), r.getCommitInfosList(), entries);
    }

    static void logAppendEntries(boolean isHeartbeat, Supplier<String> message) {
        if (isHeartbeat) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("HEARTBEAT: " + message.get());
            }
        } else if (LOG.isDebugEnabled()) {
            LOG.debug(message.get());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CompletableFuture<RaftProtos.AppendEntriesReplyProto> appendEntriesAsync(RaftPeerId leaderId, RaftGroupId leaderGroupId, long leaderTerm, TermIndex previous, long leaderCommit, long callId, boolean initializing, List<RaftProtos.CommitInfoProto> commitInfos, RaftProtos.LogEntryProto ... entries) throws IOException {
        List<CompletableFuture<Long>> futures;
        long currentTerm;
        CodeInjectionForTesting.execute((String)APPEND_ENTRIES, (Object)this.getId(), (Object)leaderId, (Object[])new Object[]{leaderTerm, previous, leaderCommit, initializing, entries});
        boolean isHeartbeat = entries.length == 0;
        RaftServerImpl.logAppendEntries(isHeartbeat, () -> this.getId() + ": receive appendEntries(" + leaderId + ", " + leaderGroupId + ", " + leaderTerm + ", " + previous + ", " + leaderCommit + ", " + initializing + ", commits" + ProtoUtils.toString((Collection)commitInfos) + ", entries: " + ServerProtoUtils.toString(entries));
        LifeCycle.State currentState = this.assertLifeCycleState(LifeCycle.State.STARTING, LifeCycle.State.RUNNING);
        if (currentState == LifeCycle.State.STARTING && this.role.getCurrentRole() == null) {
            throw new ServerNotReadyException("The role of Server " + this.getId() + " is not yet initialized.");
        }
        this.assertGroup(leaderId, leaderGroupId);
        try {
            this.validateEntries(leaderTerm, previous, entries);
        }
        catch (IllegalArgumentException e) {
            throw new IOException(e);
        }
        long nextIndex = this.state.getLog().getNextIndex();
        RaftServerImpl raftServerImpl = this;
        synchronized (raftServerImpl) {
            boolean recognized = this.state.recognizeLeader(leaderId, leaderTerm);
            currentTerm = this.state.getCurrentTerm();
            if (!recognized) {
                RaftProtos.AppendEntriesReplyProto reply = ServerProtoUtils.toAppendEntriesReplyProto(leaderId, this.getId(), this.groupId, currentTerm, nextIndex, RaftProtos.AppendEntriesReplyProto.AppendResult.NOT_LEADER, callId);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("{}: Not recognize {} (term={}) as leader, state: {} reply: {}", new Object[]{this.getId(), leaderId, leaderTerm, this.state, ProtoUtils.toString((RaftProtos.AppendEntriesReplyProto)reply)});
                }
                return CompletableFuture.completedFuture(reply);
            }
            this.changeToFollower(leaderTerm, true);
            this.state.setLeader(leaderId, "appendEntries");
            if (!initializing && this.lifeCycle.compareAndTransition(LifeCycle.State.STARTING, LifeCycle.State.RUNNING)) {
                this.startHeartbeatMonitor();
            }
            if (this.lifeCycle.getCurrentState() == LifeCycle.State.RUNNING) {
                this.heartbeatMonitor.updateLastRpcTime(true);
            }
            if (previous != null && !this.containPrevious(previous)) {
                RaftProtos.AppendEntriesReplyProto reply = ServerProtoUtils.toAppendEntriesReplyProto(leaderId, this.getId(), this.groupId, currentTerm, Math.min(nextIndex, previous.getIndex()), RaftProtos.AppendEntriesReplyProto.AppendResult.INCONSISTENCY, callId);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("{}: inconsistency entries. Leader previous:{}, Reply:{}", new Object[]{this.getId(), previous, ServerProtoUtils.toString(reply)});
                }
                return CompletableFuture.completedFuture(reply);
            }
            futures = this.state.getLog().append(entries);
            this.state.updateConfiguration(entries);
            this.state.updateStatemachine(leaderCommit, currentTerm);
            commitInfos.stream().forEach(c -> this.commitInfoCache.update((RaftProtos.CommitInfoProto)c));
        }
        if (entries.length > 0) {
            CodeInjectionForTesting.execute((String)RaftLog.LOG_SYNC, (Object)this.getId(), null, (Object[])new Object[0]);
            nextIndex = entries[entries.length - 1].getIndex() + 1L;
        }
        RaftProtos.AppendEntriesReplyProto reply = ServerProtoUtils.toAppendEntriesReplyProto(leaderId, this.getId(), this.groupId, currentTerm, nextIndex, RaftProtos.AppendEntriesReplyProto.AppendResult.SUCCESS, callId);
        RaftServerImpl.logAppendEntries(isHeartbeat, () -> this.getId() + ": succeeded to handle AppendEntries. Reply: " + ServerProtoUtils.toString(reply));
        return JavaUtils.allOf(futures).thenApply(v -> {
            RaftServerImpl raftServerImpl = this;
            synchronized (raftServerImpl) {
                if (this.lifeCycle.getCurrentState() == LifeCycle.State.RUNNING && this.isFollower() && this.getState().getCurrentTerm() == currentTerm) {
                    this.heartbeatMonitor.updateLastRpcTime(false);
                }
            }
            return reply;
        });
    }

    private boolean containPrevious(TermIndex previous) {
        if (LOG.isTraceEnabled()) {
            LOG.trace("{}: prev:{}, latestSnapshot:{}, latestInstalledSnapshot:{}", new Object[]{this.getId(), previous, this.state.getLatestSnapshot(), this.state.getLatestInstalledSnapshot()});
        }
        return this.state.getLog().contains(previous) || this.state.getLatestSnapshot() != null && this.state.getLatestSnapshot().getTermIndex().equals(previous) || this.state.getLatestInstalledSnapshot() != null && this.state.getLatestInstalledSnapshot().equals(previous);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public RaftProtos.InstallSnapshotReplyProto installSnapshot(RaftProtos.InstallSnapshotRequestProto request) throws IOException {
        long currentTerm;
        RaftProtos.RaftRpcRequestProto r = request.getServerRequest();
        RaftPeerId leaderId = RaftPeerId.valueOf((ByteString)r.getRequestorId());
        RaftGroupId leaderGroupId = ProtoUtils.toRaftGroupId((RaftProtos.RaftGroupIdProto)r.getRaftGroupId());
        CodeInjectionForTesting.execute((String)INSTALL_SNAPSHOT, (Object)this.getId(), (Object)leaderId, (Object[])new Object[]{request});
        LOG.debug("{}: receive installSnapshot({})", (Object)this.getId(), (Object)request);
        this.assertLifeCycleState(LifeCycle.State.STARTING, LifeCycle.State.RUNNING);
        this.assertGroup(leaderId, leaderGroupId);
        long leaderTerm = request.getLeaderTerm();
        TermIndex lastTermIndex = ServerProtoUtils.toTermIndex(request.getTermIndex());
        long lastIncludedIndex = lastTermIndex.getIndex();
        RaftServerImpl raftServerImpl = this;
        synchronized (raftServerImpl) {
            boolean recognized = this.state.recognizeLeader(leaderId, leaderTerm);
            currentTerm = this.state.getCurrentTerm();
            if (!recognized) {
                RaftProtos.InstallSnapshotReplyProto reply = ServerProtoUtils.toInstallSnapshotReplyProto(leaderId, this.getId(), this.groupId, currentTerm, request.getRequestIndex(), RaftProtos.InstallSnapshotResult.NOT_LEADER);
                LOG.debug("{}: do not recognize leader for installing snapshot. Reply: {}", (Object)this.getId(), (Object)reply);
                return reply;
            }
            this.changeToFollower(leaderTerm, true);
            this.state.setLeader(leaderId, "installSnapshot");
            if (this.lifeCycle.getCurrentState() == LifeCycle.State.RUNNING) {
                this.heartbeatMonitor.updateLastRpcTime(true);
            }
            Preconditions.assertTrue((this.state.getLog().getNextIndex() <= lastIncludedIndex ? 1 : 0) != 0, (String)"%s log's next id is %s, last included index in snapshot is %s", (Object[])new Object[]{this.getId(), this.state.getLog().getNextIndex(), lastIncludedIndex});
            this.state.installSnapshot(request);
            if (request.getDone()) {
                this.state.reloadStateMachine(lastIncludedIndex, leaderTerm);
            }
            if (this.lifeCycle.getCurrentState() == LifeCycle.State.RUNNING) {
                this.heartbeatMonitor.updateLastRpcTime(false);
            }
        }
        if (request.getDone()) {
            LOG.info("{}: successfully install the whole snapshot-{}", (Object)this.getId(), (Object)lastIncludedIndex);
        }
        return ServerProtoUtils.toInstallSnapshotReplyProto(leaderId, this.getId(), this.groupId, currentTerm, request.getRequestIndex(), RaftProtos.InstallSnapshotResult.SUCCESS);
    }

    synchronized RaftProtos.InstallSnapshotRequestProto createInstallSnapshotRequest(RaftPeerId targetId, String requestId, int requestIndex, SnapshotInfo snapshot, List<RaftProtos.FileChunkProto> chunks, boolean done) {
        OptionalLong totalSize = snapshot.getFiles().stream().mapToLong(FileInfo::getFileSize).reduce(Long::sum);
        assert (totalSize.isPresent());
        return ServerProtoUtils.toInstallSnapshotRequestProto(this.getId(), targetId, this.groupId, requestId, requestIndex, this.state.getCurrentTerm(), snapshot.getTermIndex(), chunks, totalSize.getAsLong(), done);
    }

    synchronized RaftProtos.RequestVoteRequestProto createRequestVoteRequest(RaftPeerId targetId, long term, TermIndex lastEntry) {
        return ServerProtoUtils.toRequestVoteRequestProto(this.getId(), targetId, this.groupId, term, lastEntry);
    }

    public synchronized void submitLocalSyncEvent() {
        if (this.isLeader() && this.leaderState != null) {
            this.leaderState.submitUpdateStateEvent(LeaderState.UPDATE_COMMIT_EVENT);
        }
    }

    private CompletableFuture<Message> replyPendingRequest(RaftProtos.LogEntryProto logEntry, CompletableFuture<Message> stateMachineFuture) {
        ClientId clientId = ClientId.valueOf((ByteString)logEntry.getClientId());
        long callId = logEntry.getCallId();
        RaftPeerId serverId = this.getId();
        RetryCache.CacheEntry cacheEntry = this.retryCache.getOrCreateEntry(clientId, logEntry.getCallId());
        if (cacheEntry.isFailed()) {
            this.retryCache.refreshEntry(new RetryCache.CacheEntry(cacheEntry.getKey()));
        }
        return stateMachineFuture.whenComplete((reply, exception) -> {
            RaftClientReply r;
            if (exception == null) {
                r = new RaftClientReply(clientId, serverId, this.groupId, callId, true, reply, null, this.getCommitInfos());
            } else {
                StateMachineException e = new StateMachineException(this.getId(), exception);
                r = new RaftClientReply(clientId, serverId, this.groupId, callId, false, null, (RaftException)e, this.getCommitInfos());
            }
            boolean updateCache = true;
            RaftServerImpl raftServerImpl = this;
            synchronized (raftServerImpl) {
                if (this.isLeader() && this.leaderState != null) {
                    updateCache = this.leaderState.replyPendingRequest(logEntry.getIndex(), r, cacheEntry);
                }
            }
            if (updateCache) {
                cacheEntry.updateResult(r);
            }
        });
    }

    private TransactionContext getTransactionContext(long index) {
        if (this.leaderState != null) {
            return this.leaderState.getTransactionContext(index);
        }
        return null;
    }

    public long[] getFollowerNextIndices() {
        LeaderState s = this.leaderState;
        if (s == null || !this.isLeader()) {
            return null;
        }
        return s.getFollowerNextIndices();
    }

    CompletableFuture<Message> applyLogToStateMachine(RaftProtos.LogEntryProto next) {
        StateMachine stateMachine = this.getStateMachine();
        if (next.getLogEntryBodyCase() == RaftProtos.LogEntryProto.LogEntryBodyCase.CONFIGURATIONENTRY) {
            stateMachine.setRaftConfiguration(ServerProtoUtils.toRaftConfiguration(next));
        } else if (next.getLogEntryBodyCase() == RaftProtos.LogEntryProto.LogEntryBodyCase.SMLOGENTRY) {
            TransactionContext trx = this.getTransactionContext(next.getIndex());
            if (trx == null) {
                trx = new TransactionContextImpl(this.getRole(), stateMachine, next);
            }
            trx = stateMachine.applyTransactionSerial(trx);
            try {
                CompletableFuture<Message> stateMachineFuture = stateMachine.applyTransaction(trx);
                return this.replyPendingRequest(next, stateMachineFuture);
            }
            catch (Throwable e) {
                LOG.error("{}: applyTransaction failed for index:{} proto:{}", new Object[]{this.getId(), next.getIndex(), ServerProtoUtils.toString(next), e.getMessage()});
                throw e;
            }
        }
        return null;
    }

    public void failClientRequest(RaftProtos.LogEntryProto logEntry) {
        if (logEntry.getLogEntryBodyCase() == RaftProtos.LogEntryProto.LogEntryBodyCase.SMLOGENTRY) {
            ClientId clientId = ClientId.valueOf((ByteString)logEntry.getClientId());
            RetryCache.CacheEntry cacheEntry = this.getRetryCache().get(clientId, logEntry.getCallId());
            if (cacheEntry != null) {
                RaftClientReply reply = new RaftClientReply(clientId, this.getId(), this.getGroupId(), logEntry.getCallId(), false, null, (RaftException)this.generateNotLeaderException(), this.getCommitInfos());
                cacheEntry.failWithReply(reply);
            }
        }
    }

    private class RaftServerJmxAdapter
    extends JmxRegister
    implements RaftServerMXBean {
        private RaftServerJmxAdapter() {
        }

        @Override
        public String getId() {
            return RaftServerImpl.this.getState().getSelfId().toString();
        }

        @Override
        public String getLeaderId() {
            return RaftServerImpl.this.getState().getLeaderId().toString();
        }

        @Override
        public long getCurrentTerm() {
            return RaftServerImpl.this.getState().getCurrentTerm();
        }

        @Override
        public String getGroupId() {
            return RaftServerImpl.this.getGroupId().toString();
        }

        @Override
        public String getRole() {
            return RaftServerImpl.this.role.toString();
        }

        @Override
        public List<String> getFollowers() {
            return Optional.ofNullable(RaftServerImpl.this.leaderState).map(leader -> leader.getFollowers().stream().map(RaftPeer::toString).collect(Collectors.toList())).orElse(Collections.emptyList());
        }
    }
}

