/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.node;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.net.InetAddress;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.management.ObjectName;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.VersionInfo;
import org.apache.hadoop.hdds.scm.container.placement.metrics.SCMNodeMetric;
import org.apache.hadoop.hdds.scm.container.placement.metrics.SCMNodeStat;
import org.apache.hadoop.hdds.scm.node.CommandQueue;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.node.NodeStateManager;
import org.apache.hadoop.hdds.scm.node.states.NodeAlreadyExistsException;
import org.apache.hadoop.hdds.scm.node.states.NodeNotFoundException;
import org.apache.hadoop.hdds.scm.server.StorageContainerManager;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.apache.hadoop.ipc.Server;
import org.apache.hadoop.metrics2.util.MBeans;
import org.apache.hadoop.ozone.protocol.StorageContainerNodeProtocol;
import org.apache.hadoop.ozone.protocol.VersionResponse;
import org.apache.hadoop.ozone.protocol.commands.CommandForDatanode;
import org.apache.hadoop.ozone.protocol.commands.RegisteredCommand;
import org.apache.hadoop.ozone.protocol.commands.ReregisterCommand;
import org.apache.hadoop.ozone.protocol.commands.SCMCommand;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SCMNodeManager
implements NodeManager,
StorageContainerNodeProtocol {
    @VisibleForTesting
    static final Logger LOG = LoggerFactory.getLogger(SCMNodeManager.class);
    private final NodeStateManager nodeStateManager;
    private final ConcurrentHashMap<UUID, SCMNodeStat> nodeStats;
    private SCMNodeStat scmStat;
    private int chillModeNodeCount;
    private final String clusterID;
    private final VersionInfo version;
    private AtomicBoolean inStartupChillMode;
    private AtomicBoolean inManualChillMode;
    private final CommandQueue commandQueue;
    private ObjectName nmInfoBean;
    private final StorageContainerManager scmManager;

    public SCMNodeManager(OzoneConfiguration conf, String clusterID, StorageContainerManager scmManager, EventPublisher eventPublisher) throws IOException {
        this.nodeStateManager = new NodeStateManager((Configuration)conf, eventPublisher);
        this.nodeStats = new ConcurrentHashMap();
        this.scmStat = new SCMNodeStat();
        this.clusterID = clusterID;
        this.version = VersionInfo.getLatestVersion();
        this.commandQueue = new CommandQueue();
        this.chillModeNodeCount = 1;
        this.inStartupChillMode = new AtomicBoolean(true);
        this.inManualChillMode = new AtomicBoolean(false);
        this.scmManager = scmManager;
        LOG.info("Entering startup chill mode.");
        this.registerMXBean();
    }

    private void registerMXBean() {
        this.nmInfoBean = MBeans.register((String)"SCMNodeManager", (String)"SCMNodeManagerInfo", (Object)this);
    }

    private void unregisterMXBean() {
        if (this.nmInfoBean != null) {
            MBeans.unregister((ObjectName)this.nmInfoBean);
            this.nmInfoBean = null;
        }
    }

    @Override
    public void removeNode(DatanodeDetails node) throws NodeNotFoundException {
        this.nodeStateManager.removeNode(node);
    }

    @Override
    public List<DatanodeDetails> getNodes(HddsProtos.NodeState nodestate) {
        return this.nodeStateManager.getNodes(nodestate);
    }

    @Override
    public List<DatanodeDetails> getAllNodes() {
        return this.nodeStateManager.getAllNodes();
    }

    @Override
    public int getMinimumChillModeNodes() {
        return this.chillModeNodeCount;
    }

    @VisibleForTesting
    public void setMinimumChillModeNodes(int count) {
        this.chillModeNodeCount = count;
    }

    @Override
    public String getChillModeStatus() {
        if (this.inStartupChillMode.get()) {
            return "Still in chill mode, waiting on nodes to report in." + String.format(" %d nodes reported, minimal %d nodes required.", this.nodeStateManager.getTotalNodeCount(), this.getMinimumChillModeNodes());
        }
        if (this.inManualChillMode.get()) {
            return "Out of startup chill mode, but in manual chill mode." + String.format(" %d nodes have reported in.", this.nodeStateManager.getTotalNodeCount());
        }
        return "Out of chill mode." + String.format(" %d nodes have reported in.", this.nodeStateManager.getTotalNodeCount());
    }

    @Override
    public void forceExitChillMode() {
        if (this.inStartupChillMode.get()) {
            LOG.info("Leaving startup chill mode.");
            this.inStartupChillMode.set(false);
        }
        if (this.inManualChillMode.get()) {
            LOG.info("Leaving manual chill mode.");
            this.inManualChillMode.set(false);
        }
    }

    @Override
    public void enterChillMode() {
        LOG.info("Entering manual chill mode.");
        this.inManualChillMode.set(true);
    }

    @Override
    public void exitChillMode() {
        LOG.info("Leaving manual chill mode.");
        this.inManualChillMode.set(false);
    }

    @Override
    public boolean isOutOfChillMode() {
        return !this.inStartupChillMode.get() && !this.inManualChillMode.get();
    }

    @Override
    public int getNodeCount(HddsProtos.NodeState nodestate) {
        return this.nodeStateManager.getNodeCount(nodestate);
    }

    @Override
    public HddsProtos.NodeState getNodeState(DatanodeDetails datanodeDetails) {
        try {
            return this.nodeStateManager.getNodeState(datanodeDetails);
        }
        catch (NodeNotFoundException e) {
            return null;
        }
    }

    private void updateNodeStat(UUID dnId, StorageContainerDatanodeProtocolProtos.NodeReportProto nodeReport) {
        SCMNodeStat stat = this.nodeStats.get(dnId);
        if (stat == null) {
            LOG.debug("SCM updateNodeStat based on heartbeat from previousdead datanode {}", (Object)dnId);
            stat = new SCMNodeStat();
        }
        if (nodeReport != null && nodeReport.getStorageReportCount() > 0) {
            long totalCapacity = 0L;
            long totalRemaining = 0L;
            long totalScmUsed = 0L;
            List storageReports = nodeReport.getStorageReportList();
            for (StorageContainerDatanodeProtocolProtos.StorageReportProto report : storageReports) {
                totalCapacity += report.getCapacity();
                totalRemaining += report.getRemaining();
                totalScmUsed += report.getScmUsed();
            }
            this.scmStat.subtract(stat);
            stat.set(totalCapacity, totalScmUsed, totalRemaining);
            this.nodeStats.put(dnId, stat);
            this.scmStat.add(stat);
        }
    }

    @Override
    public void close() throws IOException {
        this.unregisterMXBean();
    }

    public VersionResponse getVersion(StorageContainerDatanodeProtocolProtos.SCMVersionRequestProto versionRequest) {
        return VersionResponse.newBuilder().setVersion(this.version.getVersion()).addValue("scmUuid", this.scmManager.getScmStorage().getScmId()).addValue("clusterID", this.scmManager.getScmStorage().getClusterID()).build();
    }

    public RegisteredCommand register(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.NodeReportProto nodeReport, StorageContainerDatanodeProtocolProtos.PipelineReportsProto pipelineReportsProto) {
        InetAddress dnAddress = Server.getRemoteIp();
        if (dnAddress != null) {
            datanodeDetails.setHostName(dnAddress.getHostName());
            datanodeDetails.setIpAddress(dnAddress.getHostAddress());
        }
        UUID dnId = datanodeDetails.getUuid();
        try {
            this.nodeStateManager.addNode(datanodeDetails);
            this.nodeStats.put(dnId, new SCMNodeStat());
            if (this.inStartupChillMode.get() && this.nodeStateManager.getTotalNodeCount() >= this.getMinimumChillModeNodes()) {
                this.inStartupChillMode.getAndSet(false);
                LOG.info("Leaving startup chill mode.");
            }
            this.updateNodeStat(datanodeDetails.getUuid(), nodeReport);
            LOG.info("Data node with ID: {} Registered.", (Object)datanodeDetails.getUuid());
        }
        catch (NodeAlreadyExistsException e) {
            LOG.trace("Datanode is already registered. Datanode: {}", (Object)datanodeDetails.toString());
        }
        return RegisteredCommand.newBuilder().setErrorCode(StorageContainerDatanodeProtocolProtos.SCMRegisteredResponseProto.ErrorCode.success).setDatanodeUUID(datanodeDetails.getUuidString()).setClusterID(this.clusterID).setHostname(datanodeDetails.getHostName()).setIpAddress(datanodeDetails.getIpAddress()).build();
    }

    public List<SCMCommand> processHeartbeat(DatanodeDetails datanodeDetails) {
        Preconditions.checkNotNull((Object)datanodeDetails, (Object)"Heartbeat is missing DatanodeDetails.");
        try {
            this.nodeStateManager.updateLastHeartbeatTime(datanodeDetails);
        }
        catch (NodeNotFoundException e) {
            LOG.warn("SCM receive heartbeat from unregistered datanode {}", (Object)datanodeDetails);
            this.commandQueue.addCommand(datanodeDetails.getUuid(), (SCMCommand)new ReregisterCommand());
        }
        return this.commandQueue.getCommand(datanodeDetails.getUuid());
    }

    @Override
    public void processNodeReport(UUID dnUuid, StorageContainerDatanodeProtocolProtos.NodeReportProto nodeReport) {
        this.updateNodeStat(dnUuid, nodeReport);
    }

    @Override
    public SCMNodeStat getStats() {
        return new SCMNodeStat(this.scmStat);
    }

    @Override
    public Map<UUID, SCMNodeStat> getNodeStats() {
        return Collections.unmodifiableMap(this.nodeStats);
    }

    @Override
    public SCMNodeMetric getNodeStat(DatanodeDetails datanodeDetails) {
        return new SCMNodeMetric(this.nodeStats.get(datanodeDetails.getUuid()));
    }

    @Override
    public Map<String, Integer> getNodeCount() {
        HashMap<String, Integer> nodeCountMap = new HashMap<String, Integer>();
        for (HddsProtos.NodeState state : HddsProtos.NodeState.values()) {
            nodeCountMap.put(state.toString(), this.getNodeCount(state));
        }
        return nodeCountMap;
    }

    @Override
    public void addDatanodeCommand(UUID dnId, SCMCommand command) {
        this.commandQueue.addCommand(dnId, command);
    }

    public void onMessage(CommandForDatanode commandForDatanode, EventPublisher ignored) {
        this.addDatanodeCommand(commandForDatanode.getDatanodeId(), commandForDatanode.getCommand());
    }
}

