/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.node;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.Closeable;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.HddsServerUtil;
import org.apache.hadoop.hdds.scm.events.SCMEvents;
import org.apache.hadoop.hdds.scm.node.DatanodeInfo;
import org.apache.hadoop.hdds.scm.node.states.NodeAlreadyExistsException;
import org.apache.hadoop.hdds.scm.node.states.NodeNotFoundException;
import org.apache.hadoop.hdds.scm.node.states.NodeStateMap;
import org.apache.hadoop.hdds.server.events.Event;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.apache.hadoop.ozone.common.statemachine.InvalidStateTransitionException;
import org.apache.hadoop.ozone.common.statemachine.StateMachine;
import org.apache.hadoop.util.Time;
import org.apache.hadoop.util.concurrent.HadoopExecutors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NodeStateManager
implements Runnable,
Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(NodeStateManager.class);
    private final StateMachine<HddsProtos.NodeState, NodeLifeCycleEvent> stateMachine;
    private final NodeStateMap nodeStateMap = new NodeStateMap();
    private final EventPublisher eventPublisher;
    private final Map<HddsProtos.NodeState, Event<DatanodeDetails>> state2EventMap;
    private final ScheduledExecutorService executorService;
    private final long heartbeatCheckerIntervalMs;
    private final long staleNodeIntervalMs;
    private final long deadNodeIntervalMs;

    public NodeStateManager(Configuration conf, EventPublisher eventPublisher) {
        this.eventPublisher = eventPublisher;
        this.state2EventMap = new HashMap<HddsProtos.NodeState, Event<DatanodeDetails>>();
        this.initialiseState2EventMap();
        HashSet<HddsProtos.NodeState> finalStates = new HashSet<HddsProtos.NodeState>();
        finalStates.add(HddsProtos.NodeState.DECOMMISSIONED);
        this.stateMachine = new StateMachine((Enum)HddsProtos.NodeState.HEALTHY, finalStates);
        this.initializeStateMachine();
        this.heartbeatCheckerIntervalMs = HddsServerUtil.getScmheartbeatCheckerInterval((Configuration)conf);
        this.staleNodeIntervalMs = HddsServerUtil.getStaleNodeInterval((Configuration)conf);
        this.deadNodeIntervalMs = HddsServerUtil.getDeadNodeInterval((Configuration)conf);
        Preconditions.checkState((this.heartbeatCheckerIntervalMs > 0L ? 1 : 0) != 0, (Object)"ozone.scm.heartbeat.thread.interval should be greater than 0.");
        Preconditions.checkState((this.staleNodeIntervalMs < this.deadNodeIntervalMs ? 1 : 0) != 0, (Object)"ozone.scm.stale.node.interval should be less thanozone.scm.dead.node.interval");
        this.executorService = HadoopExecutors.newScheduledThreadPool((int)1, (ThreadFactory)new ThreadFactoryBuilder().setDaemon(true).setNameFormat("SCM Heartbeat Processing Thread - %d").build());
        this.executorService.schedule(this, this.heartbeatCheckerIntervalMs, TimeUnit.MILLISECONDS);
    }

    private void initialiseState2EventMap() {
        this.state2EventMap.put(HddsProtos.NodeState.STALE, (Event<DatanodeDetails>)SCMEvents.STALE_NODE);
        this.state2EventMap.put(HddsProtos.NodeState.DEAD, (Event<DatanodeDetails>)SCMEvents.DEAD_NODE);
    }

    private void initializeStateMachine() {
        this.stateMachine.addTransition((Enum)HddsProtos.NodeState.HEALTHY, (Enum)HddsProtos.NodeState.STALE, (Enum)NodeLifeCycleEvent.TIMEOUT);
        this.stateMachine.addTransition((Enum)HddsProtos.NodeState.STALE, (Enum)HddsProtos.NodeState.DEAD, (Enum)NodeLifeCycleEvent.TIMEOUT);
        this.stateMachine.addTransition((Enum)HddsProtos.NodeState.STALE, (Enum)HddsProtos.NodeState.HEALTHY, (Enum)NodeLifeCycleEvent.RESTORE);
        this.stateMachine.addTransition((Enum)HddsProtos.NodeState.DEAD, (Enum)HddsProtos.NodeState.HEALTHY, (Enum)NodeLifeCycleEvent.RESURRECT);
        this.stateMachine.addTransition((Enum)HddsProtos.NodeState.HEALTHY, (Enum)HddsProtos.NodeState.DECOMMISSIONING, (Enum)NodeLifeCycleEvent.DECOMMISSION);
        this.stateMachine.addTransition((Enum)HddsProtos.NodeState.STALE, (Enum)HddsProtos.NodeState.DECOMMISSIONING, (Enum)NodeLifeCycleEvent.DECOMMISSION);
        this.stateMachine.addTransition((Enum)HddsProtos.NodeState.DEAD, (Enum)HddsProtos.NodeState.DECOMMISSIONING, (Enum)NodeLifeCycleEvent.DECOMMISSION);
        this.stateMachine.addTransition((Enum)HddsProtos.NodeState.DECOMMISSIONING, (Enum)HddsProtos.NodeState.DECOMMISSIONED, (Enum)NodeLifeCycleEvent.DECOMMISSIONED);
    }

    public void addNode(DatanodeDetails datanodeDetails) throws NodeAlreadyExistsException {
        this.nodeStateMap.addNode(datanodeDetails, (HddsProtos.NodeState)this.stateMachine.getInitialState());
        this.eventPublisher.fireEvent(SCMEvents.NEW_NODE, (Object)datanodeDetails);
    }

    public DatanodeInfo getNode(DatanodeDetails datanodeDetails) throws NodeNotFoundException {
        return this.nodeStateMap.getNodeInfo(datanodeDetails.getUuid());
    }

    public void updateLastHeartbeatTime(DatanodeDetails datanodeDetails) throws NodeNotFoundException {
        this.nodeStateMap.getNodeInfo(datanodeDetails.getUuid()).updateLastHeartbeatTime();
    }

    public HddsProtos.NodeState getNodeState(DatanodeDetails datanodeDetails) throws NodeNotFoundException {
        return this.nodeStateMap.getNodeState(datanodeDetails.getUuid());
    }

    public List<DatanodeDetails> getHealthyNodes() {
        return this.getNodes(HddsProtos.NodeState.HEALTHY);
    }

    public List<DatanodeDetails> getStaleNodes() {
        return this.getNodes(HddsProtos.NodeState.STALE);
    }

    public List<DatanodeDetails> getDeadNodes() {
        return this.getNodes(HddsProtos.NodeState.DEAD);
    }

    public List<DatanodeDetails> getNodes(HddsProtos.NodeState state) {
        LinkedList<DatanodeDetails> nodes = new LinkedList<DatanodeDetails>();
        this.nodeStateMap.getNodes(state).forEach(uuid -> {
            try {
                nodes.add(this.nodeStateMap.getNodeDetails((UUID)uuid));
            }
            catch (NodeNotFoundException e) {
                LOG.error("Inconsistent NodeStateMap! " + this.nodeStateMap);
            }
        });
        return nodes;
    }

    public List<DatanodeDetails> getAllNodes() {
        LinkedList<DatanodeDetails> nodes = new LinkedList<DatanodeDetails>();
        this.nodeStateMap.getAllNodes().forEach(uuid -> {
            try {
                nodes.add(this.nodeStateMap.getNodeDetails((UUID)uuid));
            }
            catch (NodeNotFoundException e) {
                LOG.error("Inconsistent NodeStateMap! " + this.nodeStateMap);
            }
        });
        return nodes;
    }

    public int getHealthyNodeCount() {
        return this.getNodeCount(HddsProtos.NodeState.HEALTHY);
    }

    public int getStaleNodeCount() {
        return this.getNodeCount(HddsProtos.NodeState.STALE);
    }

    public int getDeadNodeCount() {
        return this.getNodeCount(HddsProtos.NodeState.DEAD);
    }

    public int getNodeCount(HddsProtos.NodeState state) {
        return this.nodeStateMap.getNodeCount(state);
    }

    public int getTotalNodeCount() {
        return this.nodeStateMap.getTotalNodeCount();
    }

    public void removeNode(DatanodeDetails datanodeDetails) throws NodeNotFoundException {
        this.nodeStateMap.removeNode(datanodeDetails.getUuid());
    }

    @Override
    public void run() {
        long processingStartTime = Time.monotonicNow();
        long healthyNodeDeadline = processingStartTime - this.staleNodeIntervalMs;
        long staleNodeDeadline = processingStartTime - this.deadNodeIntervalMs;
        Predicate<Long> healthyNodeCondition = lastHbTime -> lastHbTime >= healthyNodeDeadline;
        Predicate<Long> staleNodeCondition = lastHbTime -> lastHbTime < healthyNodeDeadline;
        Predicate<Long> deadNodeCondition = lastHbTime -> lastHbTime < staleNodeDeadline;
        try {
            for (HddsProtos.NodeState state : HddsProtos.NodeState.values()) {
                List<UUID> nodes = this.nodeStateMap.getNodes(state);
                for (UUID id : nodes) {
                    DatanodeInfo node = this.nodeStateMap.getNodeInfo(id);
                    switch (state) {
                        case HEALTHY: {
                            this.updateNodeState(node, staleNodeCondition, state, NodeLifeCycleEvent.TIMEOUT);
                            break;
                        }
                        case STALE: {
                            this.updateNodeState(node, deadNodeCondition, state, NodeLifeCycleEvent.TIMEOUT);
                            this.updateNodeState(node, healthyNodeCondition, state, NodeLifeCycleEvent.RESTORE);
                            break;
                        }
                        case DEAD: {
                            this.updateNodeState(node, healthyNodeCondition, state, NodeLifeCycleEvent.RESURRECT);
                            break;
                        }
                    }
                }
            }
        }
        catch (NodeNotFoundException e) {
            LOG.error("Inconsistent NodeStateMap! " + this.nodeStateMap);
        }
        long processingEndTime = Time.monotonicNow();
        if (processingEndTime - processingStartTime > this.heartbeatCheckerIntervalMs) {
            LOG.error("Total time spend processing datanode HB's is greater than configured values for datanode heartbeats. Please adjust the heartbeat configs. Time Spend on HB processing: {} seconds Datanode heartbeat Interval: {} seconds.", (Object)TimeUnit.MILLISECONDS.toSeconds(processingEndTime - processingStartTime), (Object)this.heartbeatCheckerIntervalMs);
        }
        if (!Thread.currentThread().isInterrupted() && !this.executorService.isShutdown()) {
            this.executorService.schedule(this, this.heartbeatCheckerIntervalMs, TimeUnit.MILLISECONDS);
        } else {
            LOG.info("Current Thread is interrupted, shutting down HB processing thread for Node Manager.");
        }
    }

    private void updateNodeState(DatanodeInfo node, Predicate<Long> condition, HddsProtos.NodeState state, NodeLifeCycleEvent lifeCycleEvent) throws NodeNotFoundException {
        try {
            if (condition.test(node.getLastHeartbeatTime())) {
                HddsProtos.NodeState newState = (HddsProtos.NodeState)this.stateMachine.getNextState((Enum)state, (Enum)lifeCycleEvent);
                this.nodeStateMap.updateNodeState(node.getUuid(), state, newState);
                if (this.state2EventMap.containsKey(newState)) {
                    this.eventPublisher.fireEvent(this.state2EventMap.get(newState), (Object)node);
                }
            }
        }
        catch (InvalidStateTransitionException e) {
            LOG.warn("Invalid state transition of node {}. Current state: {}, life cycle event: {}", new Object[]{node, state, lifeCycleEvent});
        }
    }

    @Override
    public void close() {
        this.executorService.shutdown();
        try {
            if (!this.executorService.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.executorService.shutdownNow();
            }
            if (!this.executorService.awaitTermination(5L, TimeUnit.SECONDS)) {
                LOG.error("Unable to shutdown NodeStateManager properly.");
            }
        }
        catch (InterruptedException e) {
            this.executorService.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    private static enum NodeLifeCycleEvent {
        TIMEOUT,
        RESTORE,
        RESURRECT,
        DECOMMISSION,
        DECOMMISSIONED;

    }
}

