/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.keyvalue.helpers;

import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.AsynchronousChannel;
import java.nio.channels.AsynchronousFileChannel;
import java.nio.channels.FileLock;
import java.nio.file.StandardOpenOption;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.concurrent.ExecutionException;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.ozone.container.common.helpers.ChunkInfo;
import org.apache.hadoop.ozone.container.common.helpers.ContainerUtils;
import org.apache.hadoop.ozone.container.common.volume.VolumeIOStats;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerData;
import org.apache.hadoop.ozone.container.keyvalue.impl.ChunkManagerImpl;
import org.apache.hadoop.util.Time;
import org.apache.ratis.shaded.com.google.protobuf.ByteString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ChunkUtils {
    private ChunkUtils() {
    }

    public static void writeData(File chunkFile, ChunkInfo chunkInfo, byte[] data, VolumeIOStats volumeIOStats) throws StorageContainerException, ExecutionException, InterruptedException, NoSuchAlgorithmException {
        Logger log = LoggerFactory.getLogger(ChunkManagerImpl.class);
        if ((long)data.length != chunkInfo.getLen()) {
            String err = String.format("data array does not match the length specified. DataLen: %d Byte Array: %d", chunkInfo.getLen(), data.length);
            log.error(err);
            throw new StorageContainerException(err, ContainerProtos.Result.INVALID_WRITE_SIZE);
        }
        AsynchronousChannel file = null;
        FileLock lock = null;
        try {
            if (chunkInfo.getChecksum() != null && !chunkInfo.getChecksum().isEmpty()) {
                ChunkUtils.verifyChecksum(chunkInfo, data, log);
            }
            long writeTimeStart = Time.monotonicNow();
            file = AsynchronousFileChannel.open(chunkFile.toPath(), StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.SPARSE, StandardOpenOption.SYNC);
            lock = ((AsynchronousFileChannel)file).lock().get();
            int size = ((AsynchronousFileChannel)file).write(ByteBuffer.wrap(data), chunkInfo.getOffset()).get();
            volumeIOStats.incWriteTime(Time.monotonicNow() - writeTimeStart);
            volumeIOStats.incWriteOpCount();
            volumeIOStats.incWriteBytes(size);
            if (size != data.length) {
                log.error("Invalid write size found. Size:{}  Expected: {} ", (Object)size, (Object)data.length);
                throw new StorageContainerException("Invalid write size found. Size: " + size + " Expected: " + data.length, ContainerProtos.Result.INVALID_WRITE_SIZE);
            }
        }
        catch (StorageContainerException ex) {
            throw ex;
        }
        catch (IOException e) {
            throw new StorageContainerException((Throwable)e, ContainerProtos.Result.IO_EXCEPTION);
        }
        finally {
            if (lock != null) {
                try {
                    lock.release();
                }
                catch (IOException e) {
                    log.error("Unable to release lock ??, Fatal Error.");
                    throw new StorageContainerException((Throwable)e, ContainerProtos.Result.CONTAINER_INTERNAL_ERROR);
                }
            }
            if (file != null) {
                try {
                    file.close();
                }
                catch (IOException e) {
                    throw new StorageContainerException("Error closing chunk file", (Throwable)e, ContainerProtos.Result.CONTAINER_INTERNAL_ERROR);
                }
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static ByteBuffer readData(File chunkFile, ChunkInfo data, VolumeIOStats volumeIOStats) throws StorageContainerException, ExecutionException, InterruptedException, NoSuchAlgorithmException {
        ByteBuffer byteBuffer;
        AsynchronousFileChannel file;
        block13: {
            Logger log = LoggerFactory.getLogger(ChunkManagerImpl.class);
            if (!chunkFile.exists()) {
                log.error("Unable to find the chunk file. chunk info : {}", (Object)data.toString());
                throw new StorageContainerException("Unable to find the chunk file. chunk info " + data.toString(), ContainerProtos.Result.UNABLE_TO_FIND_CHUNK);
            }
            file = null;
            FileLock lock = null;
            try {
                long readStartTime = Time.monotonicNow();
                file = AsynchronousFileChannel.open(chunkFile.toPath(), StandardOpenOption.READ);
                lock = file.lock(data.getOffset(), data.getLen(), true).get();
                ByteBuffer buf = ByteBuffer.allocate((int)data.getLen());
                file.read(buf, data.getOffset()).get();
                volumeIOStats.incReadTime(Time.monotonicNow() - readStartTime);
                volumeIOStats.incReadOpCount();
                volumeIOStats.incReadBytes(data.getLen());
                if (data.getChecksum() != null && !data.getChecksum().isEmpty()) {
                    ChunkUtils.verifyChecksum(data, buf.array(), log);
                }
                byteBuffer = buf;
                if (lock == null) break block13;
            }
            catch (IOException e) {
                try {
                    throw new StorageContainerException((Throwable)e, ContainerProtos.Result.IO_EXCEPTION);
                }
                catch (Throwable throwable) {
                    if (lock != null) {
                        try {
                            lock.release();
                        }
                        catch (IOException e2) {
                            log.error("I/O error is lock release.");
                        }
                    }
                    if (file != null) {
                        IOUtils.closeStream(file);
                    }
                    throw throwable;
                }
            }
            try {
                lock.release();
            }
            catch (IOException e) {
                log.error("I/O error is lock release.");
            }
        }
        if (file != null) {
            IOUtils.closeStream((Closeable)file);
        }
        return byteBuffer;
    }

    private static void verifyChecksum(ChunkInfo chunkInfo, byte[] data, Logger log) throws NoSuchAlgorithmException, StorageContainerException {
        MessageDigest sha = MessageDigest.getInstance("SHA-256");
        sha.update(data);
        if (!Hex.encodeHexString((byte[])sha.digest()).equals(chunkInfo.getChecksum())) {
            log.error("Checksum mismatch. Provided: {} , computed: {}", (Object)chunkInfo.getChecksum(), (Object)DigestUtils.sha256Hex((byte[])sha.digest()));
            throw new StorageContainerException("Checksum mismatch. Provided: " + chunkInfo.getChecksum() + " , computed: " + DigestUtils.sha256Hex((byte[])sha.digest()), ContainerProtos.Result.CHECKSUM_MISMATCH);
        }
    }

    public static boolean validateChunkForOverwrite(File chunkFile, ChunkInfo info) throws StorageContainerException {
        Logger log = LoggerFactory.getLogger(ChunkManagerImpl.class);
        if (ChunkUtils.isOverWriteRequested(chunkFile, info)) {
            if (!ChunkUtils.isOverWritePermitted(info)) {
                log.error("Rejecting write chunk request. Chunk overwrite without explicit request. {}", (Object)info.toString());
                throw new StorageContainerException("Rejecting write chunk request. OverWrite flag required." + info.toString(), ContainerProtos.Result.OVERWRITE_FLAG_REQUIRED);
            }
            return true;
        }
        return false;
    }

    public static File getChunkFile(KeyValueContainerData containerData, ChunkInfo info) throws StorageContainerException {
        Preconditions.checkNotNull((Object)containerData, (Object)"Container data can't be null");
        Logger log = LoggerFactory.getLogger(ChunkManagerImpl.class);
        String chunksPath = containerData.getChunksPath();
        if (chunksPath == null) {
            log.error("Chunks path is null in the container data");
            throw new StorageContainerException("Unable to get Chunks directory.", ContainerProtos.Result.UNABLE_TO_FIND_DATA_DIR);
        }
        File chunksLoc = new File(chunksPath);
        if (!chunksLoc.exists()) {
            log.error("Chunks path does not exist");
            throw new StorageContainerException("Unable to get Chunks directory.", ContainerProtos.Result.UNABLE_TO_FIND_DATA_DIR);
        }
        return chunksLoc.toPath().resolve(info.getChunkName()).toFile();
    }

    public static boolean isOverWriteRequested(File chunkFile, ChunkInfo chunkInfo) {
        if (!chunkFile.exists()) {
            return false;
        }
        long offset = chunkInfo.getOffset();
        return offset < chunkFile.length();
    }

    public static boolean isOverWritePermitted(ChunkInfo chunkInfo) {
        String overWrite = (String)chunkInfo.getMetadata().get("OverWriteRequested");
        return overWrite != null && !overWrite.isEmpty() && Boolean.valueOf(overWrite) != false;
    }

    public static ContainerProtos.ContainerCommandResponseProto getChunkResponseSuccess(ContainerProtos.ContainerCommandRequestProto msg) {
        return ContainerUtils.getSuccessResponse(msg);
    }

    public static ContainerProtos.ContainerCommandResponseProto getReadChunkResponse(ContainerProtos.ContainerCommandRequestProto msg, byte[] data, ChunkInfo info) {
        Preconditions.checkNotNull((Object)msg);
        Preconditions.checkNotNull((Object)data, (Object)"Chunk data is null");
        Preconditions.checkNotNull((Object)info, (Object)"Chunk Info is null");
        ContainerProtos.ReadChunkResponseProto.Builder response = ContainerProtos.ReadChunkResponseProto.newBuilder();
        response.setChunkData(info.getProtoBufMessage());
        response.setData(ByteString.copyFrom((byte[])data));
        response.setBlockID(msg.getReadChunk().getBlockID());
        ContainerProtos.ContainerCommandResponseProto.Builder builder = ContainerUtils.getSuccessResponseBuilder(msg);
        builder.setReadChunk(response);
        return builder.build();
    }
}

