/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.common.volume;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdfs.server.datanode.StorageLocation;
import org.apache.hadoop.ozone.common.InconsistentStorageStateException;
import org.apache.hadoop.ozone.container.common.impl.StorageLocationReport;
import org.apache.hadoop.ozone.container.common.interfaces.VolumeChoosingPolicy;
import org.apache.hadoop.ozone.container.common.utils.HddsVolumeUtil;
import org.apache.hadoop.ozone.container.common.volume.HddsVolume;
import org.apache.hadoop.ozone.container.common.volume.VolumeInfo;
import org.apache.hadoop.util.AutoCloseableLock;
import org.apache.hadoop.util.DiskChecker;
import org.apache.hadoop.util.InstrumentedLock;
import org.apache.hadoop.util.ShutdownHookManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VolumeSet {
    private static final Logger LOG = LoggerFactory.getLogger(VolumeSet.class);
    private Configuration conf;
    private Map<String, HddsVolume> volumeMap;
    private Map<String, HddsVolume> failedVolumeMap;
    private EnumMap<StorageType, List<HddsVolume>> volumeStateMap;
    private final AutoCloseableLock volumeSetLock;
    private final String datanodeUuid;
    private String clusterID;
    private Runnable shutdownHook;

    public VolumeSet(String dnUuid, Configuration conf) throws IOException {
        this(dnUuid, null, conf);
    }

    public VolumeSet(String dnUuid, String clusterID, Configuration conf) throws IOException {
        this.datanodeUuid = dnUuid;
        this.clusterID = clusterID;
        this.conf = conf;
        this.volumeSetLock = new AutoCloseableLock((Lock)new InstrumentedLock(this.getClass().getName(), LOG, (Lock)new ReentrantLock(true), conf.getTimeDuration("hdds.write.lock.reporting.threshold.ms", 5000L, TimeUnit.MILLISECONDS), conf.getTimeDuration("hdds.lock.suppress.warning.interval.ms", 10000L, TimeUnit.MILLISECONDS)));
        this.initializeVolumeSet();
    }

    private void initializeVolumeSet() throws IOException {
        this.volumeMap = new ConcurrentHashMap<String, HddsVolume>();
        this.failedVolumeMap = new ConcurrentHashMap<String, HddsVolume>();
        this.volumeStateMap = new EnumMap(StorageType.class);
        Collection rawLocations = this.conf.getTrimmedStringCollection("hdds.datanode.dir");
        if (rawLocations.isEmpty()) {
            rawLocations = this.conf.getTrimmedStringCollection("dfs.datanode.data.dir");
        }
        if (rawLocations.isEmpty()) {
            throw new IllegalArgumentException("No location configured in either hdds.datanode.dir or dfs.datanode.data.dir");
        }
        for (StorageType storageType : StorageType.values()) {
            this.volumeStateMap.put(storageType, new ArrayList());
        }
        for (String locationString : rawLocations) {
            try {
                StorageLocation location = StorageLocation.parse((String)locationString);
                HddsVolume hddsVolume = this.createVolume(location.getUri().getPath(), location.getStorageType());
                this.checkAndSetClusterID(hddsVolume.getClusterID());
                this.volumeMap.put(hddsVolume.getHddsRootDir().getPath(), hddsVolume);
                this.volumeStateMap.get(hddsVolume.getStorageType()).add(hddsVolume);
                LOG.info("Added Volume : {} to VolumeSet", (Object)hddsVolume.getHddsRootDir().getPath());
            }
            catch (IOException e) {
                HddsVolume volume = new HddsVolume.Builder(locationString).failedVolume(true).build();
                this.failedVolumeMap.put(locationString, volume);
                LOG.error("Failed to parse the storage location: " + locationString, (Throwable)e);
            }
        }
        if (this.volumeMap.size() == 0) {
            throw new DiskChecker.DiskOutOfSpaceException("No storage location configured");
        }
        this.shutdownHook = () -> this.saveVolumeSetUsed();
        ShutdownHookManager.get().addShutdownHook(this.shutdownHook, 10);
    }

    private void checkAndSetClusterID(String idFromVersionFile) throws InconsistentStorageStateException {
        if (this.clusterID == null) {
            this.clusterID = idFromVersionFile;
            return;
        }
        if (!idFromVersionFile.equals(this.clusterID)) {
            throw new InconsistentStorageStateException("Mismatched ClusterIDs. VolumeSet has: " + this.clusterID + ", and version file has: " + idFromVersionFile);
        }
    }

    public void acquireLock() {
        this.volumeSetLock.acquire();
    }

    public void releaseLock() {
        this.volumeSetLock.release();
    }

    private HddsVolume createVolume(String locationString, StorageType storageType) throws IOException {
        HddsVolume.Builder volumeBuilder = new HddsVolume.Builder(locationString).conf(this.conf).datanodeUuid(this.datanodeUuid).clusterID(this.clusterID).storageType(storageType);
        return volumeBuilder.build();
    }

    public boolean addVolume(String dataDir) {
        return this.addVolume(dataDir, StorageType.DEFAULT);
    }

    public boolean addVolume(String volumeRoot, StorageType storageType) {
        boolean success;
        String hddsRoot = HddsVolumeUtil.getHddsRoot(volumeRoot);
        try (AutoCloseableLock lock = this.volumeSetLock.acquire();){
            if (this.volumeMap.containsKey(hddsRoot)) {
                LOG.warn("Volume : {} already exists in VolumeMap", (Object)hddsRoot);
                success = false;
            } else {
                if (this.failedVolumeMap.containsKey(hddsRoot)) {
                    this.failedVolumeMap.remove(hddsRoot);
                }
                HddsVolume hddsVolume = this.createVolume(volumeRoot, storageType);
                this.volumeMap.put(hddsVolume.getHddsRootDir().getPath(), hddsVolume);
                this.volumeStateMap.get(hddsVolume.getStorageType()).add(hddsVolume);
                LOG.info("Added Volume : {} to VolumeSet", (Object)hddsVolume.getHddsRootDir().getPath());
                success = true;
            }
        }
        catch (IOException ex) {
            LOG.error("Failed to add volume " + volumeRoot + " to VolumeSet", (Throwable)ex);
            success = false;
        }
        return success;
    }

    public void failVolume(String dataDir) {
        String hddsRoot = HddsVolumeUtil.getHddsRoot(dataDir);
        try (AutoCloseableLock lock = this.volumeSetLock.acquire();){
            if (this.volumeMap.containsKey(hddsRoot)) {
                HddsVolume hddsVolume = this.volumeMap.get(hddsRoot);
                hddsVolume.failVolume();
                this.volumeMap.remove(hddsRoot);
                this.volumeStateMap.get(hddsVolume.getStorageType()).remove(hddsVolume);
                this.failedVolumeMap.put(hddsRoot, hddsVolume);
                LOG.info("Moving Volume : {} to failed Volumes", (Object)hddsRoot);
            } else if (this.failedVolumeMap.containsKey(hddsRoot)) {
                LOG.info("Volume : {} is not active", (Object)hddsRoot);
            } else {
                LOG.warn("Volume : {} does not exist in VolumeSet", (Object)hddsRoot);
            }
        }
    }

    public void removeVolume(String dataDir) throws IOException {
        String hddsRoot = HddsVolumeUtil.getHddsRoot(dataDir);
        try (AutoCloseableLock lock = this.volumeSetLock.acquire();){
            HddsVolume hddsVolume;
            if (this.volumeMap.containsKey(hddsRoot)) {
                hddsVolume = this.volumeMap.get(hddsRoot);
                hddsVolume.shutdown();
                this.volumeMap.remove(hddsRoot);
                this.volumeStateMap.get(hddsVolume.getStorageType()).remove(hddsVolume);
                LOG.info("Removed Volume : {} from VolumeSet", (Object)hddsRoot);
            } else if (this.failedVolumeMap.containsKey(hddsRoot)) {
                hddsVolume = this.failedVolumeMap.get(hddsRoot);
                hddsVolume.setState(HddsVolume.VolumeState.NON_EXISTENT);
                this.failedVolumeMap.remove(hddsRoot);
                LOG.info("Removed Volume : {} from failed VolumeSet", (Object)hddsRoot);
            } else {
                LOG.warn("Volume : {} does not exist in VolumeSet", (Object)hddsRoot);
            }
        }
    }

    public HddsVolume chooseVolume(long containerSize, VolumeChoosingPolicy choosingPolicy) throws IOException {
        return choosingPolicy.chooseVolume(this.getVolumesList(), containerSize);
    }

    private void saveVolumeSetUsed() {
        for (HddsVolume hddsVolume : this.volumeMap.values()) {
            try {
                hddsVolume.shutdown();
            }
            catch (Exception ex) {
                LOG.error("Failed to shutdown volume : " + hddsVolume.getHddsRootDir(), (Throwable)ex);
            }
        }
    }

    public void shutdown() {
        this.saveVolumeSetUsed();
        if (this.shutdownHook != null) {
            ShutdownHookManager.get().removeShutdownHook(this.shutdownHook);
        }
    }

    @VisibleForTesting
    public List<HddsVolume> getVolumesList() {
        return ImmutableList.copyOf(this.volumeMap.values());
    }

    @VisibleForTesting
    public List<HddsVolume> getFailedVolumesList() {
        return ImmutableList.copyOf(this.failedVolumeMap.values());
    }

    @VisibleForTesting
    public Map<String, HddsVolume> getVolumeMap() {
        return ImmutableMap.copyOf(this.volumeMap);
    }

    @VisibleForTesting
    public Map<StorageType, List<HddsVolume>> getVolumeStateMap() {
        return ImmutableMap.copyOf(this.volumeStateMap);
    }

    public StorageContainerDatanodeProtocolProtos.NodeReportProto getNodeReport() throws IOException {
        HddsVolume hddsVolume;
        StorageLocationReport[] reports = new StorageLocationReport[this.volumeMap.size() + this.failedVolumeMap.size()];
        int counter = 0;
        for (Map.Entry<String, HddsVolume> entry : this.volumeMap.entrySet()) {
            hddsVolume = entry.getValue();
            VolumeInfo volumeInfo = hddsVolume.getVolumeInfo();
            long scmUsed = 0L;
            long remaining = 0L;
            boolean failed = false;
            try {
                scmUsed = volumeInfo.getScmUsed();
                remaining = volumeInfo.getAvailable();
            }
            catch (IOException ex) {
                LOG.warn("Failed to get scmUsed and remaining for container storage location {}", (Object)volumeInfo.getRootDir());
                scmUsed = 0L;
                remaining = 0L;
                failed = true;
            }
            StorageLocationReport.Builder builder = StorageLocationReport.newBuilder();
            builder.setStorageLocation(volumeInfo.getRootDir()).setId(hddsVolume.getStorageID()).setFailed(failed).setCapacity(hddsVolume.getCapacity()).setRemaining(remaining).setScmUsed(scmUsed).setStorageType(hddsVolume.getStorageType());
            StorageLocationReport r = builder.build();
            reports[counter++] = r;
        }
        for (Map.Entry<String, HddsVolume> entry : this.failedVolumeMap.entrySet()) {
            hddsVolume = entry.getValue();
            StorageLocationReport.Builder builder = StorageLocationReport.newBuilder();
            builder.setStorageLocation(hddsVolume.getHddsRootDir().getAbsolutePath()).setId(hddsVolume.getStorageID()).setFailed(true).setCapacity(0L).setRemaining(0L).setScmUsed(0L).setStorageType(hddsVolume.getStorageType());
            StorageLocationReport r = builder.build();
            reports[counter++] = r;
        }
        StorageContainerDatanodeProtocolProtos.NodeReportProto.Builder nrb = StorageContainerDatanodeProtocolProtos.NodeReportProto.newBuilder();
        for (int i = 0; i < reports.length; ++i) {
            nrb.addStorageReport(reports[i].getProtoBufMessage());
        }
        return nrb.build();
    }
}

