/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.common.statemachine;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.Closeable;
import java.io.IOException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.ozone.container.common.report.ReportManager;
import org.apache.hadoop.ozone.container.common.statemachine.SCMConnectionManager;
import org.apache.hadoop.ozone.container.common.statemachine.StateContext;
import org.apache.hadoop.ozone.container.common.statemachine.commandhandler.CloseContainerCommandHandler;
import org.apache.hadoop.ozone.container.common.statemachine.commandhandler.CommandDispatcher;
import org.apache.hadoop.ozone.container.common.statemachine.commandhandler.DeleteBlocksCommandHandler;
import org.apache.hadoop.ozone.container.common.statemachine.commandhandler.ReplicateContainerCommandHandler;
import org.apache.hadoop.ozone.container.keyvalue.TarContainerPacker;
import org.apache.hadoop.ozone.container.ozoneimpl.OzoneContainer;
import org.apache.hadoop.ozone.container.replication.DownloadAndImportReplicator;
import org.apache.hadoop.ozone.container.replication.ReplicationSupervisor;
import org.apache.hadoop.ozone.container.replication.SimpleContainerDownloader;
import org.apache.hadoop.ozone.protocol.commands.SCMCommand;
import org.apache.hadoop.util.Time;
import org.apache.hadoop.util.concurrent.HadoopExecutors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DatanodeStateMachine
implements Closeable {
    @VisibleForTesting
    static final Logger LOG = LoggerFactory.getLogger(DatanodeStateMachine.class);
    private final ExecutorService executorService;
    private final Configuration conf;
    private final SCMConnectionManager connectionManager;
    private StateContext context;
    private final OzoneContainer container;
    private DatanodeDetails datanodeDetails;
    private final CommandDispatcher commandDispatcher;
    private final ReportManager reportManager;
    private long commandsHandled;
    private AtomicLong nextHB;
    private Thread stateMachineThread = null;
    private Thread cmdProcessThread = null;
    private final ReplicationSupervisor supervisor;

    public DatanodeStateMachine(DatanodeDetails datanodeDetails, Configuration conf) throws IOException {
        this.conf = conf;
        this.datanodeDetails = datanodeDetails;
        this.executorService = HadoopExecutors.newCachedThreadPool((ThreadFactory)new ThreadFactoryBuilder().setDaemon(true).setNameFormat("Datanode State Machine Thread - %d").build());
        this.connectionManager = new SCMConnectionManager(conf);
        this.context = new StateContext(this.conf, DatanodeStates.getInitState(), this);
        this.container = new OzoneContainer(this.datanodeDetails, new OzoneConfiguration(conf), this.context);
        this.nextHB = new AtomicLong(Time.monotonicNow());
        DownloadAndImportReplicator replicator = new DownloadAndImportReplicator(this.container.getContainerSet(), this.container.getDispatcher(), new SimpleContainerDownloader(conf), new TarContainerPacker());
        this.supervisor = new ReplicationSupervisor(this.container.getContainerSet(), replicator, 10);
        this.commandDispatcher = CommandDispatcher.newBuilder().addHandler(new CloseContainerCommandHandler()).addHandler(new DeleteBlocksCommandHandler(this.container.getContainerSet(), conf)).addHandler(new ReplicateContainerCommandHandler(conf, this.supervisor)).setConnectionManager(this.connectionManager).setContainer(this.container).setContext(this.context).build();
        this.reportManager = ReportManager.newBuilder(conf).setStateContext(this.context).addPublisherFor(StorageContainerDatanodeProtocolProtos.NodeReportProto.class).addPublisherFor(StorageContainerDatanodeProtocolProtos.ContainerReportsProto.class).addPublisherFor(StorageContainerDatanodeProtocolProtos.CommandStatusReportsProto.class).build();
    }

    public DatanodeDetails getDatanodeDetails() {
        return this.datanodeDetails;
    }

    public SCMConnectionManager getConnectionManager() {
        return this.connectionManager;
    }

    public OzoneContainer getContainer() {
        return this.container;
    }

    private void start() throws IOException {
        long now = 0L;
        this.container.start();
        this.reportManager.init();
        this.initCommandHandlerThread(this.conf);
        while (this.context.getState() != DatanodeStates.SHUTDOWN) {
            try {
                LOG.debug("Executing cycle Number : {}", (Object)this.context.getExecutionCount());
                long heartbeatFrequency = this.context.getHeartbeatFrequency();
                this.nextHB.set(Time.monotonicNow() + heartbeatFrequency);
                this.context.execute(this.executorService, heartbeatFrequency, TimeUnit.MILLISECONDS);
                now = Time.monotonicNow();
                if (now >= this.nextHB.get()) continue;
                Thread.sleep(this.nextHB.get() - now);
            }
            catch (InterruptedException heartbeatFrequency) {
            }
            catch (Exception e) {
                LOG.error("Unable to finish the execution.", (Throwable)e);
            }
        }
    }

    public StateContext getContext() {
        return this.context;
    }

    public void setContext(StateContext context) {
        this.context = context;
    }

    @Override
    public void close() throws IOException {
        if (this.stateMachineThread != null) {
            this.stateMachineThread.interrupt();
        }
        if (this.cmdProcessThread != null) {
            this.cmdProcessThread.interrupt();
        }
        this.context.setState(DatanodeStates.getLastState());
        this.executorService.shutdown();
        try {
            if (!this.executorService.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.executorService.shutdownNow();
            }
            if (!this.executorService.awaitTermination(5L, TimeUnit.SECONDS)) {
                LOG.error("Unable to shutdown state machine properly.");
            }
        }
        catch (InterruptedException e) {
            LOG.error("Error attempting to shutdown.", (Throwable)e);
            this.executorService.shutdownNow();
            Thread.currentThread().interrupt();
        }
        if (this.connectionManager != null) {
            this.connectionManager.close();
        }
        if (this.container != null) {
            this.container.stop();
        }
    }

    public void startDaemon() {
        Runnable startStateMachineTask = () -> {
            try {
                this.supervisor.start();
                this.start();
                LOG.info("Ozone container server started.");
            }
            catch (Exception ex) {
                LOG.error("Unable to start the DatanodeState Machine", (Throwable)ex);
            }
        };
        this.stateMachineThread = new ThreadFactoryBuilder().setDaemon(true).setNameFormat("Datanode State Machine Thread - %d").build().newThread(startStateMachineTask);
        this.stateMachineThread.start();
    }

    public void join() throws InterruptedException {
        this.stateMachineThread.join();
        this.cmdProcessThread.join();
    }

    public synchronized void stopDaemon() {
        try {
            this.supervisor.stop();
            this.context.setState(DatanodeStates.SHUTDOWN);
            this.reportManager.shutdown();
            this.close();
            LOG.info("Ozone container server stopped.");
        }
        catch (IOException e) {
            LOG.error("Stop ozone container server failed.", (Throwable)e);
        }
    }

    @VisibleForTesting
    public boolean isDaemonStopped() {
        return this.executorService.isShutdown() && this.getContext().getExecutionCount() == 0L && this.getContext().getState() == DatanodeStates.SHUTDOWN;
    }

    private void initCommandHandlerThread(Configuration config) {
        Runnable processCommandQueue = () -> {
            while (this.getContext().getState() != DatanodeStates.SHUTDOWN) {
                SCMCommand command = this.getContext().getNextCommand();
                if (command != null) {
                    this.commandDispatcher.handle(command);
                    ++this.commandsHandled;
                    continue;
                }
                try {
                    long now = Time.monotonicNow();
                    if (this.nextHB.get() <= now) continue;
                    Thread.sleep(this.nextHB.get() - now + 1000L);
                }
                catch (InterruptedException interruptedException) {}
            }
        };
        this.cmdProcessThread = this.getCommandHandlerThread(processCommandQueue);
        this.cmdProcessThread.start();
    }

    private Thread getCommandHandlerThread(Runnable processCommandQueue) {
        Thread handlerThread = new Thread(processCommandQueue);
        handlerThread.setDaemon(true);
        handlerThread.setName("Command processor thread");
        handlerThread.setUncaughtExceptionHandler((t, e) -> {
            LOG.error("Critical Error : Command processor thread encountered an error. Thread: {}", (Object)t.toString(), (Object)e);
            this.getCommandHandlerThread(processCommandQueue).start();
        });
        return handlerThread;
    }

    @VisibleForTesting
    public long getCommandHandled() {
        return this.commandsHandled;
    }

    @VisibleForTesting
    public CommandDispatcher getCommandDispatcher() {
        return this.commandDispatcher;
    }

    public static enum DatanodeStates {
        INIT(1),
        RUNNING(2),
        SHUTDOWN(3);

        private final int value;

        private DatanodeStates(int value) {
            this.value = value;
        }

        public static DatanodeStates getInitState() {
            return INIT;
        }

        public static DatanodeStates getLastState() {
            return SHUTDOWN;
        }

        public int getValue() {
            return this.value;
        }

        public DatanodeStates getNextState() {
            if (this.value < DatanodeStates.getLastState().getValue()) {
                int stateValue = this.getValue() + 1;
                for (DatanodeStates iter : DatanodeStates.values()) {
                    if (stateValue != iter.getValue()) continue;
                    return iter;
                }
            }
            return DatanodeStates.getLastState();
        }
    }
}

