/*
 * Licensed to the Apache Software Foundation (ASF) under one or more contributor license
 * agreements. See the NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The ASF licenses this file to You under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License. You may obtain a
 * copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */
package org.apache.geode.internal.cache.tier.sockets;

import java.io.IOException;
import java.net.Socket;

import org.apache.geode.internal.cache.InternalCache;
import org.apache.geode.internal.cache.tier.Acceptor;
import org.apache.geode.internal.cache.tier.CachedRegionHelper;
import org.apache.geode.internal.security.SecurityService;

/**
 * Handles everything but the new client protocol.
 */
class OriginalServerConnection extends ServerConnection {

  /**
   * Set to false once handshake has been done
   */
  private boolean doHandshake = true;

  /**
   * Creates a new {@code ServerConnection} that processes messages received from an edge
   * client over a given {@code Socket}.
   */
  OriginalServerConnection(final Socket socket, final InternalCache internalCache,
      final CachedRegionHelper cachedRegionHelper, final CacheServerStats stats,
      final int hsTimeout, final int socketBufferSize, final String communicationModeStr,
      final byte communicationMode, final Acceptor acceptor,
      final SecurityService securityService) {
    super(socket, internalCache, cachedRegionHelper, stats, hsTimeout, socketBufferSize,
        communicationModeStr, communicationMode, acceptor, securityService);

    initStreams(socket, socketBufferSize, stats);
  }

  @Override
  protected boolean doHandShake(byte endpointType, int queueSize) {
    try {
      handshake.handshakeWithClient(theSocket.getOutputStream(), theSocket.getInputStream(),
          endpointType, queueSize, communicationMode, principal);
    } catch (IOException ioe) {
      if (!crHelper.isShutdown() && !isTerminated()) {
        logger.warn("{}: Handshake accept failed on socket {}: {}", name, theSocket, ioe);
      }
      cleanup();
      return false;
    }
    return true;
  }

  @Override
  protected void doOneMessage() {
    if (doHandshake) {
      doHandshake();
      doHandshake = false;
    } else {
      resetTransientData();
      doNormalMessage();
    }
  }
}
