/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flume.sink.hbase2;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;
import org.apache.flume.Channel;
import org.apache.flume.Context;
import org.apache.flume.Event;
import org.apache.flume.EventDeliveryException;
import org.apache.flume.FlumeException;
import org.apache.flume.Sink;
import org.apache.flume.Transaction;
import org.apache.flume.annotations.InterfaceAudience;
import org.apache.flume.auth.FlumeAuthenticationUtil;
import org.apache.flume.auth.PrivilegedExecutor;
import org.apache.flume.conf.BatchSizeSupported;
import org.apache.flume.conf.Configurable;
import org.apache.flume.conf.ConfigurationException;
import org.apache.flume.instrumentation.SinkCounter;
import org.apache.flume.sink.AbstractSink;
import org.apache.flume.sink.hbase2.BatchAware;
import org.apache.flume.sink.hbase2.HBase2EventSerializer;
import org.apache.flume.sink.hbase2.HBase2SinkConfigurationConstants;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.BufferedMutator;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.client.ConnectionFactory;
import org.apache.hadoop.hbase.client.Durability;
import org.apache.hadoop.hbase.client.Increment;
import org.apache.hadoop.hbase.client.Mutation;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Row;
import org.apache.hadoop.hbase.client.Table;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.VersionInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HBase2Sink
extends AbstractSink
implements Configurable,
BatchSizeSupported {
    private String tableName;
    private byte[] columnFamily;
    private Connection conn;
    private BufferedMutator table;
    private long batchSize;
    private final Configuration config;
    private static final Logger logger = LoggerFactory.getLogger(HBase2Sink.class);
    private HBase2EventSerializer serializer;
    private String kerberosPrincipal;
    private String kerberosKeytab;
    private boolean enableWal = true;
    private boolean batchIncrements = false;
    private SinkCounter sinkCounter;
    private PrivilegedExecutor privilegedExecutor;
    private DebugIncrementsCallback debugIncrCallback = null;

    public HBase2Sink() {
        this(HBaseConfiguration.create());
    }

    public HBase2Sink(Configuration conf) {
        this.config = conf;
    }

    @VisibleForTesting
    @InterfaceAudience.Private
    HBase2Sink(Configuration conf, DebugIncrementsCallback cb) {
        this(conf);
        this.debugIncrCallback = cb;
    }

    public void start() {
        Preconditions.checkArgument((this.table == null ? 1 : 0) != 0, (Object)"Please call stop before calling start on an old instance.");
        try {
            this.privilegedExecutor = FlumeAuthenticationUtil.getAuthenticator((String)this.kerberosPrincipal, (String)this.kerberosKeytab);
        }
        catch (Exception ex) {
            this.sinkCounter.incrementConnectionFailedCount();
            throw new FlumeException("Failed to login to HBase using provided credentials.", (Throwable)ex);
        }
        try {
            this.conn = (Connection)this.privilegedExecutor.execute(() -> {
                this.conn = ConnectionFactory.createConnection((Configuration)this.config);
                return this.conn;
            });
            this.table = this.conn.getBufferedMutator(TableName.valueOf((String)this.tableName));
        }
        catch (Exception e) {
            this.sinkCounter.incrementConnectionFailedCount();
            logger.error("Could not load table, " + this.tableName + " from HBase", (Throwable)e);
            throw new FlumeException("Could not load table, " + this.tableName + " from HBase", (Throwable)e);
        }
        try {
            if (!((Boolean)this.privilegedExecutor.execute(() -> {
                try (Table t = null;){
                    t = this.conn.getTable(TableName.valueOf((String)this.tableName));
                    Boolean bl = t.getTableDescriptor().hasFamily(this.columnFamily);
                    return bl;
                }
            })).booleanValue()) {
                throw new IOException("Table " + this.tableName + " has no such column family " + Bytes.toString((byte[])this.columnFamily));
            }
        }
        catch (Exception e) {
            this.sinkCounter.incrementConnectionFailedCount();
            throw new FlumeException("Error getting column family from HBase.Please verify that the table " + this.tableName + " and Column Family, " + Bytes.toString((byte[])this.columnFamily) + " exists in HBase, and the current user has permissions to access that table.", (Throwable)e);
        }
        super.start();
        this.sinkCounter.incrementConnectionCreatedCount();
        this.sinkCounter.start();
    }

    public void stop() {
        try {
            if (this.table != null) {
                this.table.close();
            }
            this.table = null;
        }
        catch (IOException e) {
            throw new FlumeException("Error closing table.", (Throwable)e);
        }
        try {
            if (this.conn != null) {
                this.conn.close();
            }
            this.conn = null;
        }
        catch (IOException e) {
            throw new FlumeException("Error closing connection.", (Throwable)e);
        }
        this.sinkCounter.incrementConnectionClosedCount();
        this.sinkCounter.stop();
    }

    public void configure(Context context) {
        String hbaseZnode;
        if (!this.hasVersionAtLeast2()) {
            throw new ConfigurationException("HBase major version number must be at least 2 for hbase2sink");
        }
        this.tableName = context.getString("table");
        String cf = context.getString("columnFamily");
        this.batchSize = context.getLong("batchSize", Long.valueOf(100L));
        Context serializerContext = new Context();
        String eventSerializerType = context.getString("serializer");
        Preconditions.checkNotNull((Object)this.tableName, (Object)"Table name cannot be empty, please specify in configuration file");
        Preconditions.checkNotNull((Object)cf, (Object)"Column family cannot be empty, please specify in configuration file");
        if (eventSerializerType == null || eventSerializerType.isEmpty()) {
            eventSerializerType = "org.apache.flume.sink.hbase2.SimpleHBase2EventSerializer";
            logger.info("No serializer defined, Will use default");
        }
        serializerContext.putAll((Map)context.getSubProperties("serializer."));
        this.columnFamily = cf.getBytes(Charsets.UTF_8);
        try {
            Class<?> clazz = Class.forName(eventSerializerType);
            this.serializer = (HBase2EventSerializer)clazz.newInstance();
            this.serializer.configure(serializerContext);
        }
        catch (Exception e) {
            logger.error("Could not instantiate event serializer.", (Throwable)e);
            Throwables.propagate((Throwable)e);
        }
        this.kerberosKeytab = context.getString("kerberosKeytab");
        this.kerberosPrincipal = context.getString("kerberosPrincipal");
        this.enableWal = context.getBoolean("enableWal", Boolean.valueOf(true));
        logger.info("The write to WAL option is set to: " + String.valueOf(this.enableWal));
        if (!this.enableWal) {
            logger.warn("HBase Sink's enableWal configuration is set to false. All writes to HBase will have WAL disabled, and any data in the memstore of this region in the Region Server could be lost!");
        }
        this.batchIncrements = context.getBoolean("coalesceIncrements", HBase2SinkConfigurationConstants.DEFAULT_COALESCE_INCREMENTS);
        if (this.batchIncrements) {
            logger.info("Increment coalescing is enabled. Increments will be buffered.");
        }
        String zkQuorum = context.getString("zookeeperQuorum");
        Integer port = null;
        if (zkQuorum != null && !zkQuorum.isEmpty()) {
            StringBuilder zkBuilder = new StringBuilder();
            logger.info("Using ZK Quorum: " + zkQuorum);
            String[] zkHosts = zkQuorum.split(",");
            int length = zkHosts.length;
            for (int i = 0; i < length; ++i) {
                String[] zkHostAndPort = zkHosts[i].split(":");
                zkBuilder.append(zkHostAndPort[0].trim());
                if (i != length - 1) {
                    zkBuilder.append(",");
                } else {
                    zkQuorum = zkBuilder.toString();
                }
                if (zkHostAndPort[1] == null) {
                    throw new FlumeException("Expected client port for the ZK node!");
                }
                if (port == null) {
                    port = Integer.parseInt(zkHostAndPort[1].trim());
                    continue;
                }
                if (port.equals(Integer.parseInt(zkHostAndPort[1].trim()))) continue;
                throw new FlumeException("All Zookeeper nodes in the quorum must use the same client port.");
            }
            if (port == null) {
                port = 2181;
            }
            this.config.set("hbase.zookeeper.quorum", zkQuorum);
            this.config.setInt("hbase.zookeeper.property.clientPort", port.intValue());
        }
        if ((hbaseZnode = context.getString("znodeParent")) != null && !hbaseZnode.isEmpty()) {
            this.config.set("zookeeper.znode.parent", hbaseZnode);
        }
        this.sinkCounter = new SinkCounter(this.getName());
    }

    public Configuration getConfig() {
        return this.config;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Sink.Status process() throws EventDeliveryException {
        Sink.Status status;
        block13: {
            status = Sink.Status.READY;
            Channel channel = this.getChannel();
            Transaction txn = channel.getTransaction();
            LinkedList<Row> actions = new LinkedList<Row>();
            LinkedList<Increment> incs = new LinkedList<Increment>();
            try {
                long i;
                txn.begin();
                if (this.serializer instanceof BatchAware) {
                    ((BatchAware)((Object)this.serializer)).onBatchStart();
                }
                for (i = 0L; i < this.batchSize; ++i) {
                    Event event = channel.take();
                    if (event == null) {
                        if (i == 0L) {
                            status = Sink.Status.BACKOFF;
                            this.sinkCounter.incrementBatchEmptyCount();
                            break;
                        }
                        this.sinkCounter.incrementBatchUnderflowCount();
                        break;
                    }
                    this.serializer.initialize(event, this.columnFamily);
                    actions.addAll(this.serializer.getActions());
                    incs.addAll(this.serializer.getIncrements());
                }
                if (i == this.batchSize) {
                    this.sinkCounter.incrementBatchCompleteCount();
                }
                this.sinkCounter.addToEventDrainAttemptCount(i);
                this.putEventsAndCommit(actions, incs, txn);
            }
            catch (Throwable e) {
                try {
                    txn.rollback();
                }
                catch (Exception e2) {
                    logger.error("Exception in rollback. Rollback might not have been successful.", (Throwable)e2);
                }
                logger.error("Failed to commit transaction.Transaction rolled back.", e);
                this.sinkCounter.incrementEventWriteOrChannelFail(e);
                if (e instanceof Error || e instanceof RuntimeException) {
                    logger.error("Failed to commit transaction.Transaction rolled back.", e);
                    Throwables.propagate((Throwable)e);
                    break block13;
                }
                logger.error("Failed to commit transaction.Transaction rolled back.", e);
                throw new EventDeliveryException("Failed to commit transaction.Transaction rolled back.", e);
            }
            finally {
                txn.close();
            }
        }
        return status;
    }

    private void putEventsAndCommit(List<Row> actions, List<Increment> incs, Transaction txn) throws Exception {
        this.privilegedExecutor.execute(() -> {
            ArrayList<Mutation> mutations = new ArrayList<Mutation>(actions.size());
            for (Row r : actions) {
                if (r instanceof Put) {
                    ((Put)r).setDurability(this.enableWal ? Durability.USE_DEFAULT : Durability.SKIP_WAL);
                }
                if (r instanceof Increment) {
                    ((Increment)r).setDurability(this.enableWal ? Durability.USE_DEFAULT : Durability.SKIP_WAL);
                }
                if (r instanceof Mutation) {
                    mutations.add((Mutation)r);
                    continue;
                }
                logger.warn("dropping row " + r + " since it is not an Increment or Put");
            }
            this.table.mutate(mutations);
            this.table.flush();
            return null;
        });
        this.privilegedExecutor.execute(() -> {
            List<Increment> processedIncrements = this.batchIncrements ? this.coalesceIncrements((Iterable<Increment>)incs) : incs;
            if (this.debugIncrCallback != null) {
                this.debugIncrCallback.onAfterCoalesce(processedIncrements);
            }
            for (Increment i : processedIncrements) {
                i.setDurability(this.enableWal ? Durability.USE_DEFAULT : Durability.SKIP_WAL);
                this.table.mutate((Mutation)i);
            }
            this.table.flush();
            return null;
        });
        txn.commit();
        this.sinkCounter.addToEventDrainSuccessCount((long)actions.size());
    }

    private Map<byte[], NavigableMap<byte[], Long>> getFamilyMap(Increment inc) {
        Preconditions.checkNotNull((Object)inc, (Object)"Increment required");
        return inc.getFamilyMapOfLongs();
    }

    private List<Increment> coalesceIncrements(Iterable<Increment> incs) {
        Preconditions.checkNotNull(incs, (Object)"List of Increments must not be null");
        TreeMap counters = Maps.newTreeMap((Comparator)Bytes.BYTES_COMPARATOR);
        for (Increment inc : incs) {
            byte[] row = inc.getRow();
            Map<byte[], NavigableMap<byte[], Long>> families = this.getFamilyMap(inc);
            for (Map.Entry<byte[], NavigableMap<byte[], Long>> familyEntry : families.entrySet()) {
                byte[] family = familyEntry.getKey();
                NavigableMap<byte[], Long> qualifiers = familyEntry.getValue();
                for (Map.Entry qualifierEntry : qualifiers.entrySet()) {
                    byte[] qualifier = (byte[])qualifierEntry.getKey();
                    Long count = (Long)qualifierEntry.getValue();
                    this.incrementCounter(counters, row, family, qualifier, count);
                }
            }
        }
        LinkedList coalesced = Lists.newLinkedList();
        for (Map.Entry rowEntry : counters.entrySet()) {
            byte[] row = (byte[])rowEntry.getKey();
            Map families = (Map)rowEntry.getValue();
            Increment inc = new Increment(row);
            for (Map.Entry familyEntry : families.entrySet()) {
                byte[] family = (byte[])familyEntry.getKey();
                NavigableMap qualifiers = (NavigableMap)familyEntry.getValue();
                for (Map.Entry qualifierEntry : qualifiers.entrySet()) {
                    byte[] qualifier = (byte[])qualifierEntry.getKey();
                    long count = (Long)qualifierEntry.getValue();
                    inc.addColumn(family, qualifier, count);
                }
            }
            coalesced.add(inc);
        }
        return coalesced;
    }

    private void incrementCounter(Map<byte[], Map<byte[], NavigableMap<byte[], Long>>> counters, byte[] row, byte[] family, byte[] qualifier, Long count) {
        Map families = counters.computeIfAbsent(row, k -> Maps.newTreeMap((Comparator)Bytes.BYTES_COMPARATOR));
        NavigableMap qualifiers = families.computeIfAbsent(family, k -> Maps.newTreeMap((Comparator)Bytes.BYTES_COMPARATOR));
        qualifiers.merge(qualifier, count, (a, b) -> a + b);
    }

    String getHBbaseVersionString() {
        return VersionInfo.getVersion();
    }

    private int getMajorVersion(String version) throws NumberFormatException {
        return Integer.parseInt(version.split("\\.")[0]);
    }

    private boolean hasVersionAtLeast2() {
        String version = this.getHBbaseVersionString();
        try {
            if (this.getMajorVersion(version) >= 2) {
                return true;
            }
        }
        catch (NumberFormatException ex) {
            logger.error(ex.getMessage());
        }
        logger.error("Invalid HBase version for hbase2sink:" + version);
        return false;
    }

    @VisibleForTesting
    @InterfaceAudience.Private
    HBase2EventSerializer getSerializer() {
        return this.serializer;
    }

    public long getBatchSize() {
        return this.batchSize;
    }

    @VisibleForTesting
    @InterfaceAudience.Private
    static interface DebugIncrementsCallback {
        public void onAfterCoalesce(Iterable<Increment> var1);
    }
}

