/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.sql.parser.hive.ddl;

import org.apache.flink.sql.parser.hive.impl.ParseException;

import org.apache.calcite.sql.SqlIdentifier;
import org.apache.calcite.sql.SqlNode;
import org.apache.calcite.sql.SqlNodeList;
import org.apache.calcite.sql.SqlWriter;
import org.apache.calcite.sql.parser.SqlParserPos;

import static org.apache.flink.sql.parser.hive.ddl.SqlAlterHiveTable.AlterTableOp.CHANGE_TBL_PROPS;

/** ALTER DDL to change properties of a Hive table. */
public class SqlAlterHiveTableProps extends SqlAlterHiveTable {

    private final SqlNodeList origProps;

    public SqlAlterHiveTableProps(
            SqlParserPos pos, SqlIdentifier tableName, SqlNodeList propertyList)
            throws ParseException {
        super(
                CHANGE_TBL_PROPS,
                pos,
                tableName,
                null,
                HiveDDLUtils.checkReservedTableProperties(propertyList));
        HiveDDLUtils.unescapeProperties(propertyList);
        // remove the last property which is the ALTER_TABLE_OP
        this.origProps =
                new SqlNodeList(
                        propertyList.getList().subList(0, propertyList.size() - 1),
                        propertyList.getParserPosition());
    }

    @Override
    public void unparse(SqlWriter writer, int leftPrec, int rightPrec) {
        super.unparse(writer, leftPrec, rightPrec);
        writer.keyword("SET TBLPROPERTIES");
        SqlWriter.Frame withFrame = writer.startList("(", ")");
        for (SqlNode property : origProps) {
            printIndent(writer);
            property.unparse(writer, leftPrec, rightPrec);
        }
        writer.newlineAndIndent();
        writer.endList(withFrame);
    }
}
