/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.flink.table.runtime.join

import org.apache.flink.api.common.state._
import org.apache.flink.api.common.typeinfo.TypeInformation
import org.apache.flink.api.java.tuple.{Tuple2 => JTuple2}
import org.apache.flink.configuration.Configuration
import org.apache.flink.streaming.api.functions.co.CoProcessFunction
import org.apache.flink.table.api.TableConfig
import org.apache.flink.table.runtime.types.CRow
import org.apache.flink.types.Row
import org.apache.flink.util.Collector

/**
  * Connect data for left stream and right stream. Only use for left or right join without
  * non-equal predicates.
  *
  * @param leftType        the input type of left stream
  * @param rightType       the input type of right stream
  * @param genJoinFuncName the function code without any non-equi condition
  * @param genJoinFuncCode the function name without any non-equi condition
  * @param isLeftJoin      the type of join, whether it is the type of left join
  * @param config          configuration that determines runtime behavior
  */
class NonWindowLeftRightJoin(
    leftType: TypeInformation[Row],
    rightType: TypeInformation[Row],
    genJoinFuncName: String,
    genJoinFuncCode: String,
    isLeftJoin: Boolean,
    minRetentionTime: Long,
    maxRetentionTime: Long)
  extends NonWindowOuterJoin(
    leftType,
    rightType,
    genJoinFuncName,
    genJoinFuncCode,
    isLeftJoin,
    minRetentionTime,
    maxRetentionTime) {

  override def open(parameters: Configuration): Unit = {
    super.open(parameters)
    val joinType = if (isLeftJoin) "Left" else "Right"
    LOG.debug(s"Instantiating NonWindow${joinType}OuterJoin")
  }

  /**
    * Puts or Retract an element from the input stream into state and search the other state to
    * output records meet the condition. The input row will be preserved and appended with null, if
    * there is no match. Records will be expired in state if state retention time has been
    * specified.
    */
  override def processElement(
      value: CRow,
      ctx: CoProcessFunction[CRow, CRow, CRow]#Context,
      out: Collector[CRow],
      currentSideState: MapState[Row, JTuple2[Long, Long]],
      otherSideState: MapState[Row, JTuple2[Long, Long]],
      recordFromLeft: Boolean): Unit = {

    val inputRow = value.row
    updateCurrentSide(value, ctx, currentSideState)

    cRowWrapper.reset()
    cRowWrapper.setCollector(out)
    cRowWrapper.setChange(value.change)

    // join other side data
    if (recordFromLeft == isLeftJoin) {
      preservedJoin(inputRow, recordFromLeft, otherSideState)
    } else {
      retractJoin(value, recordFromLeft, currentSideState, otherSideState)
    }
  }
}

