/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.checkpoint;

import org.apache.flink.runtime.checkpoint.InflightDataRescalingDescriptor.InflightDataGateOrPartitionRescalingDescriptor;
import org.apache.flink.runtime.checkpoint.InflightDataRescalingDescriptor.InflightDataGateOrPartitionRescalingDescriptor.MappingType;

import org.apache.flink.shaded.guava18.com.google.common.collect.Sets;

import java.util.Arrays;
import java.util.Set;

/** Helper class to create {@link InflightDataRescalingDescriptor} and {@link RescaleMappings}. */
public class InflightDataRescalingDescriptorUtil {
    public static RescaleMappings mappings(int[]... channelMappings) {
        final int numberOfTargets =
                Arrays.stream(channelMappings)
                                .flatMapToInt(targets -> Arrays.stream(targets))
                                .max()
                                .orElse(-1)
                        + 1;
        return RescaleMappings.of(Arrays.stream(channelMappings), numberOfTargets);
    }

    public static int[] to(int... indexes) {
        return indexes;
    }

    public static <T> T[] array(T... elements) {
        return elements;
    }

    public static <T> Set<T> set(T... elements) {
        return Sets.newHashSet(elements);
    }

    public static InflightDataRescalingDescriptor rescalingDescriptor(
            int[] oldIndices, RescaleMappings[] mappings, Set<Integer> ambiguousSubtasks) {
        return new InflightDataRescalingDescriptor(
                Arrays.stream(mappings)
                        .map(
                                mapping ->
                                        new InflightDataGateOrPartitionRescalingDescriptor(
                                                oldIndices,
                                                mapping,
                                                ambiguousSubtasks,
                                                MappingType.RESCALING))
                        .toArray(InflightDataGateOrPartitionRescalingDescriptor[]::new));
    }
}
