/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.streaming.kafka.test;

import org.apache.flink.api.java.utils.ParameterTool;
import org.apache.flink.streaming.api.datastream.DataStream;
import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;
import org.apache.flink.streaming.connectors.kafka.FlinkKafkaConsumer;
import org.apache.flink.streaming.connectors.kafka.FlinkKafkaProducer;
import org.apache.flink.streaming.connectors.kafka.internals.KeyedSerializationSchemaWrapper;
import org.apache.flink.streaming.kafka.test.base.CustomWatermarkExtractor;
import org.apache.flink.streaming.kafka.test.base.KafkaEvent;
import org.apache.flink.streaming.kafka.test.base.KafkaEventSchema;
import org.apache.flink.streaming.kafka.test.base.KafkaExampleUtil;
import org.apache.flink.streaming.kafka.test.base.RollingAdditionMapper;

/**
 * A simple example that shows how to read from and write to modern Kafka. This will read String
 * messages from the input topic, parse them into a POJO type {@link KafkaEvent}, group by some key,
 * and finally perform a rolling addition on each key for which the results are written back to
 * another topic.
 *
 * <p>This example also demonstrates using a watermark assigner to generate per-partition watermarks
 * directly in the Flink Kafka consumer. For demonstration purposes, it is assumed that the String
 * messages are of formatted as a (word,frequency,timestamp) tuple.
 *
 * <p>Example usage: --input-topic test-input --output-topic test-output --bootstrap.servers
 * localhost:9092 --group.id myconsumer
 */
public class KafkaExample extends KafkaExampleUtil {

    public static void main(String[] args) throws Exception {
        // parse input arguments
        final ParameterTool parameterTool = ParameterTool.fromArgs(args);
        StreamExecutionEnvironment env = KafkaExampleUtil.prepareExecutionEnv(parameterTool);

        DataStream<KafkaEvent> input =
                env.addSource(
                                new FlinkKafkaConsumer<>(
                                                parameterTool.getRequired("input-topic"),
                                                new KafkaEventSchema(),
                                                parameterTool.getProperties())
                                        .assignTimestampsAndWatermarks(
                                                new CustomWatermarkExtractor()))
                        .keyBy("word")
                        .map(new RollingAdditionMapper());

        input.addSink(
                new FlinkKafkaProducer<>(
                        parameterTool.getRequired("output-topic"),
                        new KeyedSerializationSchemaWrapper<>(new KafkaEventSchema()),
                        parameterTool.getProperties(),
                        FlinkKafkaProducer.Semantic.EXACTLY_ONCE));

        env.execute("Modern Kafka Example");
    }
}
