/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.catalog;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import org.apache.flink.annotation.Internal;
import org.apache.flink.table.api.Schema;
import org.apache.flink.table.api.TableException;
import org.apache.flink.table.catalog.CatalogTable;
import org.apache.flink.table.catalog.Column;
import org.apache.flink.table.catalog.ResolvedCatalogTable;
import org.apache.flink.table.catalog.ResolvedSchema;
import org.apache.flink.table.catalog.UniqueConstraint;
import org.apache.flink.table.catalog.WatermarkSpec;
import org.apache.flink.table.catalog.exceptions.CatalogException;
import org.apache.flink.table.expressions.ResolvedExpression;
import org.apache.flink.table.types.DataType;
import org.apache.flink.table.types.logical.LogicalType;
import org.apache.flink.util.Preconditions;

@Internal
public final class CatalogPropertiesUtil {
    public static final String IS_GENERIC = "is_generic";
    public static final String FLINK_PROPERTY_PREFIX = "flink.";
    private static final String SEPARATOR = ".";
    private static final String SCHEMA = "schema";
    private static final String NAME = "name";
    private static final String DATA_TYPE = "data-type";
    private static final String EXPR = "expr";
    private static final String METADATA = "metadata";
    private static final String VIRTUAL = "virtual";
    private static final String PRIMARY_KEY = "primary-key";
    private static final String COLUMNS = "columns";
    private static final String PARTITION = "partition";
    private static final String KEYS = "keys";
    private static final String PARTITION_KEYS = CatalogPropertiesUtil.compoundKey("partition", "keys");
    private static final String WATERMARK = "watermark";
    private static final String WATERMARK_ROWTIME = "rowtime";
    private static final String WATERMARK_STRATEGY = "strategy";
    private static final String WATERMARK_STRATEGY_EXPR = CatalogPropertiesUtil.compoundKey("strategy", "expr");
    private static final String WATERMARK_STRATEGY_DATA_TYPE = CatalogPropertiesUtil.compoundKey("strategy", "data-type");
    private static final String PRIMARY_KEY_NAME = CatalogPropertiesUtil.compoundKey("primary-key", "name");
    private static final String PRIMARY_KEY_COLUMNS = CatalogPropertiesUtil.compoundKey("primary-key", "columns");
    private static final String COMMENT = "comment";

    public static Map<String, String> serializeCatalogTable(ResolvedCatalogTable resolvedTable) {
        try {
            HashMap<String, String> properties = new HashMap<String, String>();
            CatalogPropertiesUtil.serializeResolvedSchema(properties, resolvedTable.getResolvedSchema());
            String comment = resolvedTable.getComment();
            if (comment != null && comment.length() > 0) {
                properties.put(COMMENT, comment);
            }
            CatalogPropertiesUtil.serializePartitionKeys(properties, resolvedTable.getPartitionKeys());
            properties.putAll(resolvedTable.getOptions());
            properties.remove(IS_GENERIC);
            return properties;
        }
        catch (Exception e) {
            throw new CatalogException("Error in serializing catalog table.", e);
        }
    }

    public static CatalogTable deserializeCatalogTable(Map<String, String> properties) {
        try {
            Schema schema = CatalogPropertiesUtil.deserializeSchema(properties);
            String comment = properties.get(COMMENT);
            List<String> partitionKeys = CatalogPropertiesUtil.deserializePartitionKeys(properties);
            Map<String, String> options = CatalogPropertiesUtil.deserializeOptions(properties);
            return CatalogTable.of(schema, comment, partitionKeys, options);
        }
        catch (Exception e) {
            throw new CatalogException("Error in deserializing catalog table.", e);
        }
    }

    private static Map<String, String> deserializeOptions(Map<String, String> map) {
        return map.entrySet().stream().filter(e -> {
            String key = (String)e.getKey();
            return !key.startsWith("schema.") && !key.startsWith(PARTITION_KEYS + SEPARATOR) && !key.equals(COMMENT);
        }).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    private static List<String> deserializePartitionKeys(Map<String, String> map) {
        int partitionCount = CatalogPropertiesUtil.getCount(map, PARTITION_KEYS, NAME);
        ArrayList<String> partitionKeys = new ArrayList<String>();
        for (int i = 0; i < partitionCount; ++i) {
            String partitionNameKey = CatalogPropertiesUtil.compoundKey(PARTITION_KEYS, i, NAME);
            String partitionName = CatalogPropertiesUtil.getValue(map, partitionNameKey);
            partitionKeys.add(partitionName);
        }
        return partitionKeys;
    }

    private static Schema deserializeSchema(Map<String, String> map) {
        Schema.Builder builder = Schema.newBuilder();
        CatalogPropertiesUtil.deserializeColumns(map, builder);
        CatalogPropertiesUtil.deserializeWatermark(map, builder);
        CatalogPropertiesUtil.deserializePrimaryKey(map, builder);
        return builder.build();
    }

    private static void deserializePrimaryKey(Map<String, String> map, Schema.Builder builder) {
        String constraintNameKey = CatalogPropertiesUtil.compoundKey(SCHEMA, PRIMARY_KEY_NAME);
        String columnsKey = CatalogPropertiesUtil.compoundKey(SCHEMA, PRIMARY_KEY_COLUMNS);
        if (map.containsKey(constraintNameKey)) {
            String constraintName = CatalogPropertiesUtil.getValue(map, constraintNameKey);
            String[] columns = CatalogPropertiesUtil.getValue(map, columnsKey, s -> s.split(","));
            builder.primaryKeyNamed(constraintName, columns);
        }
    }

    private static void deserializeWatermark(Map<String, String> map, Schema.Builder builder) {
        String watermarkKey = CatalogPropertiesUtil.compoundKey(SCHEMA, WATERMARK);
        int watermarkCount = CatalogPropertiesUtil.getCount(map, watermarkKey, WATERMARK_ROWTIME);
        for (int i = 0; i < watermarkCount; ++i) {
            String rowtimeKey = CatalogPropertiesUtil.compoundKey(watermarkKey, i, WATERMARK_ROWTIME);
            String exprKey = CatalogPropertiesUtil.compoundKey(watermarkKey, i, WATERMARK_STRATEGY_EXPR);
            String rowtime2 = CatalogPropertiesUtil.getValue(map, rowtimeKey);
            String expr = CatalogPropertiesUtil.getValue(map, exprKey);
            builder.watermark(rowtime2, expr);
        }
    }

    private static void deserializeColumns(Map<String, String> map, Schema.Builder builder) {
        int fieldCount = CatalogPropertiesUtil.getCount(map, SCHEMA, NAME);
        for (int i = 0; i < fieldCount; ++i) {
            String nameKey = CatalogPropertiesUtil.compoundKey(SCHEMA, i, NAME);
            String dataTypeKey = CatalogPropertiesUtil.compoundKey(SCHEMA, i, DATA_TYPE);
            String exprKey = CatalogPropertiesUtil.compoundKey(SCHEMA, i, EXPR);
            String metadataKey = CatalogPropertiesUtil.compoundKey(SCHEMA, i, METADATA);
            String virtualKey = CatalogPropertiesUtil.compoundKey(SCHEMA, i, VIRTUAL);
            String name = CatalogPropertiesUtil.getValue(map, nameKey);
            if (map.containsKey(exprKey)) {
                String expr = CatalogPropertiesUtil.getValue(map, exprKey);
                builder.columnByExpression(name, expr);
                continue;
            }
            if (map.containsKey(metadataKey)) {
                String metadata = CatalogPropertiesUtil.getValue(map, metadataKey);
                String dataType2 = CatalogPropertiesUtil.getValue(map, dataTypeKey);
                boolean isVirtual = CatalogPropertiesUtil.getValue(map, virtualKey, Boolean::parseBoolean);
                if (metadata.equals(name)) {
                    builder.columnByMetadata(name, dataType2, null, isVirtual);
                    continue;
                }
                builder.columnByMetadata(name, dataType2, metadata, isVirtual);
                continue;
            }
            String dataType3 = CatalogPropertiesUtil.getValue(map, dataTypeKey);
            builder.column(name, dataType3);
        }
    }

    private static void serializePartitionKeys(Map<String, String> map, List<String> keys) {
        Preconditions.checkNotNull(keys);
        CatalogPropertiesUtil.putIndexedProperties(map, PARTITION_KEYS, Collections.singletonList(NAME), keys.stream().map(Collections::singletonList).collect(Collectors.toList()));
    }

    private static void serializeResolvedSchema(Map<String, String> map, ResolvedSchema schema) {
        Preconditions.checkNotNull((Object)schema);
        CatalogPropertiesUtil.serializeColumns(map, schema.getColumns());
        CatalogPropertiesUtil.serializeWatermarkSpecs(map, schema.getWatermarkSpecs());
        schema.getPrimaryKey().ifPresent(pk -> CatalogPropertiesUtil.serializePrimaryKey(map, pk));
    }

    private static void serializePrimaryKey(Map<String, String> map, UniqueConstraint constraint) {
        map.put(CatalogPropertiesUtil.compoundKey(SCHEMA, PRIMARY_KEY_NAME), constraint.getName());
        map.put(CatalogPropertiesUtil.compoundKey(SCHEMA, PRIMARY_KEY_COLUMNS), String.join((CharSequence)",", constraint.getColumns()));
    }

    private static void serializeWatermarkSpecs(Map<String, String> map, List<WatermarkSpec> specs) {
        if (!specs.isEmpty()) {
            ArrayList<List<String>> watermarkValues = new ArrayList<List<String>>();
            for (WatermarkSpec spec : specs) {
                watermarkValues.add(Arrays.asList(spec.getRowtimeAttribute(), CatalogPropertiesUtil.serializeResolvedExpression(spec.getWatermarkExpression()), CatalogPropertiesUtil.serializeDataType(spec.getWatermarkExpression().getOutputDataType())));
            }
            CatalogPropertiesUtil.putIndexedProperties(map, CatalogPropertiesUtil.compoundKey(SCHEMA, WATERMARK), Arrays.asList(WATERMARK_ROWTIME, WATERMARK_STRATEGY_EXPR, WATERMARK_STRATEGY_DATA_TYPE), watermarkValues);
        }
    }

    private static void serializeColumns(Map<String, String> map, List<Column> columns) {
        String[] names = CatalogPropertiesUtil.serializeColumnNames(columns);
        String[] dataTypes = CatalogPropertiesUtil.serializeColumnDataTypes(columns);
        String[] expressions = CatalogPropertiesUtil.serializeColumnComputations(columns);
        String[] metadata = CatalogPropertiesUtil.serializeColumnMetadataKeys(columns);
        String[] virtual = CatalogPropertiesUtil.serializeColumnVirtuality(columns);
        ArrayList<List<String>> values = new ArrayList<List<String>>();
        for (int i = 0; i < columns.size(); ++i) {
            values.add(Arrays.asList(names[i], dataTypes[i], expressions[i], metadata[i], virtual[i]));
        }
        CatalogPropertiesUtil.putIndexedProperties(map, SCHEMA, Arrays.asList(NAME, DATA_TYPE, EXPR, METADATA, VIRTUAL), values);
    }

    private static String serializeResolvedExpression(ResolvedExpression resolvedExpression) {
        try {
            return resolvedExpression.asSerializableString();
        }
        catch (TableException e) {
            throw new TableException(String.format("Expression '%s' cannot be stored in a durable catalog. Currently, only SQL expressions have a well-defined string representation that is used to serialize a catalog object into a map of string-based properties.", resolvedExpression.asSummaryString()), e);
        }
    }

    private static String serializeDataType(DataType dataType2) {
        LogicalType type = dataType2.getLogicalType();
        try {
            return type.asSerializableString();
        }
        catch (TableException e) {
            throw new TableException(String.format("Data type '%s' cannot be stored in a durable catalog. Only data types that have a well-defined string representation can be used when serializing a catalog object into a map of string-based properties. This excludes anonymously defined, unregistered types such as structured types in particular.", type.asSummaryString()), e);
        }
    }

    private static String[] serializeColumnNames(List<Column> columns) {
        return (String[])columns.stream().map(Column::getName).toArray(String[]::new);
    }

    private static String[] serializeColumnDataTypes(List<Column> columns) {
        return (String[])columns.stream().map(Column::getDataType).map(CatalogPropertiesUtil::serializeDataType).toArray(String[]::new);
    }

    private static String[] serializeColumnComputations(List<Column> columns) {
        return (String[])columns.stream().map(column -> {
            if (column instanceof Column.ComputedColumn) {
                Column.ComputedColumn c = (Column.ComputedColumn)column;
                return CatalogPropertiesUtil.serializeResolvedExpression(c.getExpression());
            }
            return null;
        }).toArray(String[]::new);
    }

    private static String[] serializeColumnMetadataKeys(List<Column> columns) {
        return (String[])columns.stream().map(column -> {
            if (column instanceof Column.MetadataColumn) {
                Column.MetadataColumn c = (Column.MetadataColumn)column;
                return c.getMetadataKey().orElse(c.getName());
            }
            return null;
        }).toArray(String[]::new);
    }

    private static String[] serializeColumnVirtuality(List<Column> columns) {
        return (String[])columns.stream().map(column -> {
            if (column instanceof Column.MetadataColumn) {
                Column.MetadataColumn c = (Column.MetadataColumn)column;
                return Boolean.toString(c.isVirtual());
            }
            return null;
        }).toArray(String[]::new);
    }

    private static void putIndexedProperties(Map<String, String> map, String key, List<String> subKeys, List<List<String>> subKeyValues) {
        Preconditions.checkNotNull((Object)key);
        Preconditions.checkNotNull(subKeys);
        Preconditions.checkNotNull(subKeyValues);
        for (int idx = 0; idx < subKeyValues.size(); ++idx) {
            List<String> values = subKeyValues.get(idx);
            if (values == null || values.size() != subKeys.size()) {
                throw new IllegalArgumentException("Values must have same arity as keys.");
            }
            if (values.stream().allMatch(Objects::isNull)) {
                throw new IllegalArgumentException("Values must have at least one non-null value.");
            }
            for (int keyIdx = 0; keyIdx < values.size(); ++keyIdx) {
                String value = values.get(keyIdx);
                if (value == null) continue;
                map.put(CatalogPropertiesUtil.compoundKey(key, idx, subKeys.get(keyIdx)), values.get(keyIdx));
            }
        }
    }

    private static int getCount(Map<String, String> map, String key, String suffix) {
        String escapedKey = Pattern.quote(key);
        String escapedSuffix = Pattern.quote(suffix);
        String escapedSeparator = Pattern.quote(SEPARATOR);
        Pattern pattern = Pattern.compile(escapedKey + escapedSeparator + "(\\d+)" + escapedSeparator + escapedSuffix);
        IntStream indexes = map.keySet().stream().flatMapToInt(k -> {
            Matcher matcher = pattern.matcher((CharSequence)k);
            if (matcher.find()) {
                return IntStream.of(Integer.parseInt(matcher.group(1)));
            }
            return IntStream.empty();
        });
        return indexes.max().orElse(-1) + 1;
    }

    private static String getValue(Map<String, String> map, String key) {
        return (String)CatalogPropertiesUtil.getValue(map, key, Function.identity());
    }

    private static <T> T getValue(Map<String, String> map, String key, Function<String, T> parser) {
        String value = map.get(key);
        if (value == null) {
            throw new IllegalArgumentException(String.format("Could not find property key '%s'.", key));
        }
        try {
            return parser.apply(value);
        }
        catch (Exception e) {
            throw new IllegalArgumentException(String.format("Could not parse value for property key '%s': %s", key, value));
        }
    }

    private static String compoundKey(Object ... components) {
        return Stream.of(components).map(Object::toString).collect(Collectors.joining(SEPARATOR));
    }

    private CatalogPropertiesUtil() {
    }
}

