/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.kubernetes.utils;

import org.apache.flink.configuration.BlobServerOptions;
import org.apache.flink.configuration.ConfigOption;
import org.apache.flink.configuration.ConfigOptions;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.HighAvailabilityOptions;
import org.apache.flink.util.FlinkRuntimeException;
import org.apache.flink.util.TestLogger;

import org.junit.Test;

import static org.hamcrest.CoreMatchers.containsString;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;

/** Tests for {@link KubernetesUtils}. */
public class KubernetesUtilsTest extends TestLogger {

    @Test
    public void testParsePortRange() {
        final Configuration cfg = new Configuration();
        cfg.set(BlobServerOptions.PORT, "50100-50200");
        try {
            KubernetesUtils.parsePort(cfg, BlobServerOptions.PORT);
            fail("Should fail with an exception.");
        } catch (FlinkRuntimeException e) {
            assertThat(
                    e.getMessage(),
                    containsString(
                            BlobServerOptions.PORT.key()
                                    + " should be specified to a fixed port. Do not support a range of ports."));
        }
    }

    @Test
    public void testParsePortNull() {
        final Configuration cfg = new Configuration();
        ConfigOption<String> testingPort =
                ConfigOptions.key("test.port").stringType().noDefaultValue();
        try {
            KubernetesUtils.parsePort(cfg, testingPort);
            fail("Should fail with an exception.");
        } catch (NullPointerException e) {
            assertThat(e.getMessage(), containsString(testingPort.key() + " should not be null."));
        }
    }

    @Test
    public void testCheckWithDynamicPort() {
        testCheckAndUpdatePortConfigOption("0", "6123", "6123");
    }

    @Test
    public void testCheckWithFixedPort() {
        testCheckAndUpdatePortConfigOption("6123", "16123", "6123");
    }

    private void testCheckAndUpdatePortConfigOption(
            String port, String fallbackPort, String expectedPort) {
        final Configuration cfg = new Configuration();
        cfg.setString(HighAvailabilityOptions.HA_JOB_MANAGER_PORT_RANGE, port);
        KubernetesUtils.checkAndUpdatePortConfigOption(
                cfg,
                HighAvailabilityOptions.HA_JOB_MANAGER_PORT_RANGE,
                Integer.valueOf(fallbackPort));
        assertEquals(expectedPort, cfg.get(HighAvailabilityOptions.HA_JOB_MANAGER_PORT_RANGE));
    }
}
