/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.runtime.functions;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalAccessor;
import java.util.TimeZone;
import org.apache.calcite.avatica.util.DateTimeUtils;
import org.apache.calcite.avatica.util.TimeUnit;
import org.apache.calcite.avatica.util.TimeUnitRange;
import org.apache.flink.table.api.TableException;
import org.apache.flink.table.data.DecimalData;
import org.apache.flink.table.data.DecimalDataUtils;
import org.apache.flink.table.data.TimestampData;
import org.apache.flink.table.types.logical.DateType;
import org.apache.flink.table.types.logical.LogicalType;
import org.apache.flink.table.types.logical.TimestampType;
import org.apache.flink.table.utils.ThreadLocalCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SqlDateTimeUtils {
    private static final Logger LOG = LoggerFactory.getLogger(SqlDateTimeUtils.class);
    private static final int EPOCH_JULIAN = 2440588;
    private static final long MILLIS_PER_SECOND = 1000L;
    private static final long MILLIS_PER_MINUTE = 60000L;
    private static final long MILLIS_PER_HOUR = 3600000L;
    private static final long MILLIS_PER_DAY = 86400000L;
    private static final String DATE_FORMAT_STRING = "yyyy-MM-dd";
    private static final String TIME_FORMAT_STRING = "HH:mm:ss";
    private static final String TIMESTAMP_FORMAT_STRING = "yyyy-MM-dd HH:mm:ss";
    public static final TimeZone UTC_ZONE = TimeZone.getTimeZone("UTC");
    private static final TimeZone LOCAL_TZ = TimeZone.getDefault();
    private static final String[] DEFAULT_DATETIME_FORMATS = new String[]{"yyyy-MM-dd HH:mm:ss", "yyyy-MM-dd HH:mm:ss.S", "yyyy-MM-dd HH:mm:ss.SS", "yyyy-MM-dd HH:mm:ss.SSS", "yyyy-MM-dd HH:mm:ss.SSSS", "yyyy-MM-dd HH:mm:ss.SSSSS", "yyyy-MM-dd HH:mm:ss.SSSSSS", "yyyy-MM-dd HH:mm:ss.SSSSSSS", "yyyy-MM-dd HH:mm:ss.SSSSSSSS", "yyyy-MM-dd HH:mm:ss.SSSSSSSSS"};
    private static final ThreadLocalCache<String, SimpleDateFormat> FORMATTER_CACHE = new ThreadLocalCache<String, SimpleDateFormat>(){

        @Override
        public SimpleDateFormat getNewInstance(String key) {
            return new SimpleDateFormat(key);
        }
    };
    private static final ThreadLocalCache<String, DateTimeFormatter> DATETIME_FORMATTER_CACHE = new ThreadLocalCache<String, DateTimeFormatter>(){

        @Override
        public DateTimeFormatter getNewInstance(String key) {
            return DateTimeFormatter.ofPattern(key);
        }
    };
    private static final ThreadLocalCache<String, TimeZone> TIMEZONE_CACHE = new ThreadLocalCache<String, TimeZone>(){

        @Override
        public TimeZone getNewInstance(String tz) {
            return TimeZone.getTimeZone(tz);
        }
    };
    private static final DateType REUSE_DATE_TYPE = new DateType();
    private static final TimestampType REUSE_TIMESTAMP_TYPE = new TimestampType(9);

    public static Date internalToDate(int v) {
        long t = (long)v * 86400000L;
        return new Date(t - (long)LOCAL_TZ.getOffset(t));
    }

    public static Time internalToTime(int v) {
        return new Time(v - LOCAL_TZ.getOffset(v));
    }

    public static Timestamp internalToTimestamp(long v) {
        return new Timestamp(v - (long)LOCAL_TZ.getOffset(v));
    }

    public static int dateToInternal(Date date) {
        long ts = date.getTime() + (long)LOCAL_TZ.getOffset(date.getTime());
        return (int)(ts / 86400000L);
    }

    public static int timeToInternal(Time time) {
        long ts = time.getTime() + (long)LOCAL_TZ.getOffset(time.getTime());
        return (int)(ts % 86400000L);
    }

    public static long timestampToInternal(Timestamp ts) {
        long time = ts.getTime();
        return time + (long)LOCAL_TZ.getOffset(time);
    }

    public static int toDate(int v) {
        return v;
    }

    public static long toTimestamp(long v) {
        return v;
    }

    public static long toTimestamp(double v) {
        return (long)v;
    }

    public static long toTimestamp(DecimalData v) {
        return DecimalDataUtils.castToLong(v);
    }

    public static TimestampData toTimestampData(String dateStr) {
        int length = dateStr.length();
        String format = length == 10 ? DATE_FORMAT_STRING : (length >= 21 && length <= 29 ? DEFAULT_DATETIME_FORMATS[length - 20] : DEFAULT_DATETIME_FORMATS[0]);
        return SqlDateTimeUtils.toTimestampData(dateStr, format);
    }

    public static TimestampData toTimestampData(String dateStr, String format) {
        DateTimeFormatter formatter = DATETIME_FORMATTER_CACHE.get(format);
        try {
            TemporalAccessor accessor = formatter.parse(dateStr);
            int year = accessor.isSupported(ChronoField.YEAR) ? accessor.get(ChronoField.YEAR) : 1970;
            int month = accessor.isSupported(ChronoField.MONTH_OF_YEAR) ? accessor.get(ChronoField.MONTH_OF_YEAR) : 1;
            int day = accessor.isSupported(ChronoField.DAY_OF_MONTH) ? accessor.get(ChronoField.DAY_OF_MONTH) : 1;
            int hour = accessor.isSupported(ChronoField.HOUR_OF_DAY) ? accessor.get(ChronoField.HOUR_OF_DAY) : 0;
            int minute = accessor.isSupported(ChronoField.MINUTE_OF_HOUR) ? accessor.get(ChronoField.MINUTE_OF_HOUR) : 0;
            int second = accessor.isSupported(ChronoField.SECOND_OF_MINUTE) ? accessor.get(ChronoField.SECOND_OF_MINUTE) : 0;
            int nanoOfSecond = accessor.isSupported(ChronoField.NANO_OF_SECOND) ? accessor.get(ChronoField.NANO_OF_SECOND) : 0;
            LocalDateTime ldt = LocalDateTime.of(year, month, day, hour, minute, second, nanoOfSecond);
            return TimestampData.fromLocalDateTime(ldt);
        }
        catch (DateTimeParseException e) {
            try {
                dateStr = dateStr.trim();
                int space = dateStr.indexOf(32);
                if (space >= 0) {
                    Timestamp ts = Timestamp.valueOf(dateStr);
                    return TimestampData.fromTimestamp(ts);
                }
                Date dt = Date.valueOf(dateStr);
                return TimestampData.fromLocalDateTime(LocalDateTime.of(dt.toLocalDate(), LocalTime.MIDNIGHT));
            }
            catch (IllegalArgumentException ie) {
                return null;
            }
        }
    }

    public static Long toTimestamp(String dateStr) {
        return SqlDateTimeUtils.toTimestamp(dateStr, UTC_ZONE);
    }

    public static Long toTimestamp(String dateStr, TimeZone tz) {
        int length = dateStr.length();
        String format = length == 10 ? DATE_FORMAT_STRING : (length == 21 ? DEFAULT_DATETIME_FORMATS[1] : (length == 22 ? DEFAULT_DATETIME_FORMATS[2] : (length == 23 ? DEFAULT_DATETIME_FORMATS[3] : DEFAULT_DATETIME_FORMATS[0])));
        return SqlDateTimeUtils.toTimestamp(dateStr, format, tz);
    }

    public static Long toTimestamp(String dateStr, String format, TimeZone tz) {
        SimpleDateFormat formatter = FORMATTER_CACHE.get(format);
        formatter.setTimeZone(tz);
        try {
            return formatter.parse(dateStr).getTime();
        }
        catch (ParseException e) {
            return null;
        }
    }

    public static Long toTimestamp(String dateStr, String format) {
        return SqlDateTimeUtils.toTimestamp(dateStr, format, UTC_ZONE);
    }

    public static Long toTimestampTz(String dateStr, String format, String tzStr) {
        TimeZone tz = TIMEZONE_CACHE.get(tzStr);
        return SqlDateTimeUtils.toTimestamp(dateStr, format, tz);
    }

    public static Long toTimestampTz(String dateStr, String tzStr) {
        return SqlDateTimeUtils.toTimestampTz(dateStr, TIMESTAMP_FORMAT_STRING, tzStr);
    }

    public static int strToDate(String dateStr, String fromFormat) {
        long ts = SqlDateTimeUtils.parseToTimeMillis(dateStr, fromFormat, TimeZone.getTimeZone("UTC"));
        ZoneId zoneId = ZoneId.of("UTC");
        Instant instant = Instant.ofEpochMilli(ts);
        ZonedDateTime zdt = ZonedDateTime.ofInstant(instant, zoneId);
        return DateTimeUtils.ymdToUnixDate(zdt.getYear(), zdt.getMonthValue(), zdt.getDayOfMonth());
    }

    public static String dateFormat(TimestampData ts, String format, ZoneId zoneId) {
        DateTimeFormatter formatter = DATETIME_FORMATTER_CACHE.get(format);
        Instant instant = ts.toInstant();
        return LocalDateTime.ofInstant(instant, zoneId).format(formatter);
    }

    public static String dateFormat(TimestampData ts, String format) {
        return SqlDateTimeUtils.dateFormat(ts, format, ZoneId.of("UTC"));
    }

    public static String dateFormat(TimestampData ts, String format, TimeZone zone) {
        return SqlDateTimeUtils.dateFormat(ts, format, zone.toZoneId());
    }

    public static String dateFormat(long ts, String format, TimeZone tz) {
        SimpleDateFormat formatter = FORMATTER_CACHE.get(format);
        formatter.setTimeZone(tz);
        java.util.Date dateTime = new java.util.Date(ts);
        return formatter.format(dateTime);
    }

    public static String dateFormat(String dateStr, String fromFormat, String toFormat, TimeZone tz) {
        SimpleDateFormat fromFormatter = FORMATTER_CACHE.get(fromFormat);
        fromFormatter.setTimeZone(tz);
        SimpleDateFormat toFormatter = FORMATTER_CACHE.get(toFormat);
        toFormatter.setTimeZone(tz);
        try {
            return toFormatter.format(fromFormatter.parse(dateStr));
        }
        catch (ParseException e) {
            LOG.error("Exception when formatting: '" + dateStr + "' from: '" + fromFormat + "' to: '" + toFormat + "'", (Throwable)e);
            return null;
        }
    }

    public static String dateFormat(String dateStr, String toFormat, TimeZone tz) {
        return SqlDateTimeUtils.dateFormat(dateStr, TIMESTAMP_FORMAT_STRING, toFormat, tz);
    }

    public static String dateFormat(long ts, String format) {
        return SqlDateTimeUtils.dateFormat(ts, format, UTC_ZONE);
    }

    public static String dateFormat(String dateStr, String fromFormat, String toFormat) {
        return SqlDateTimeUtils.dateFormat(dateStr, fromFormat, toFormat, UTC_ZONE);
    }

    public static String dateFormat(String dateStr, String toFormat) {
        return SqlDateTimeUtils.dateFormat(dateStr, toFormat, UTC_ZONE);
    }

    public static String dateFormatTz(long ts, String format, String tzStr) {
        TimeZone tz = TIMEZONE_CACHE.get(tzStr);
        return SqlDateTimeUtils.dateFormat(ts, format, tz);
    }

    public static String dateFormatTz(long ts, String tzStr) {
        return SqlDateTimeUtils.dateFormatTz(ts, TIMESTAMP_FORMAT_STRING, tzStr);
    }

    public static String convertTz(String dateStr, String format, String tzFrom, String tzTo) {
        Long ts = SqlDateTimeUtils.toTimestampTz(dateStr, format, tzFrom);
        if (null != ts) {
            return SqlDateTimeUtils.dateFormatTz(ts, tzTo);
        }
        return null;
    }

    public static String convertTz(String dateStr, String tzFrom, String tzTo) {
        return SqlDateTimeUtils.convertTz(dateStr, TIMESTAMP_FORMAT_STRING, tzFrom, tzTo);
    }

    public static String timestampToString(long ts, int precision) {
        int p = precision <= 3 && precision >= 0 ? precision : 3;
        String format = DEFAULT_DATETIME_FORMATS[p];
        return SqlDateTimeUtils.dateFormat(ts, format, UTC_ZONE);
    }

    public static String timestampToString(long ts, int precision, TimeZone tz) {
        int p = precision <= 3 && precision >= 0 ? precision : 3;
        String format = DEFAULT_DATETIME_FORMATS[p];
        return SqlDateTimeUtils.dateFormat(ts, format, tz);
    }

    public static String unixTimeToString(int time) {
        StringBuilder buf = new StringBuilder(8);
        SqlDateTimeUtils.unixTimeToString(buf, time, 0);
        return buf.toString();
    }

    private static void unixTimeToString(StringBuilder buf, int time, int precision) {
        while (time < 0) {
            time = (int)((long)time + 86400000L);
        }
        int h = time / 3600000;
        int time2 = time % 3600000;
        int m = time2 / 60000;
        int time3 = time2 % 60000;
        int s = time3 / 1000;
        int ms = time3 % 1000;
        SqlDateTimeUtils.int2(buf, h);
        buf.append(':');
        SqlDateTimeUtils.int2(buf, m);
        buf.append(':');
        SqlDateTimeUtils.int2(buf, s);
        if (precision > 0) {
            buf.append('.');
            while (precision > 0) {
                buf.append((char)(48 + ms / 100));
                ms %= 100;
                if ((ms *= 10) == 0) break;
                --precision;
            }
        }
    }

    private static void int2(StringBuilder buf, int i) {
        buf.append((char)(48 + i / 10 % 10));
        buf.append((char)(48 + i % 10));
    }

    private static long parseToTimeMillis(String dateStr, TimeZone tz) {
        String format = dateStr.length() <= 10 ? DATE_FORMAT_STRING : TIMESTAMP_FORMAT_STRING;
        return SqlDateTimeUtils.parseToTimeMillis(dateStr, format, tz) + (long)SqlDateTimeUtils.getMillis(dateStr);
    }

    private static long parseToTimeMillis(String dateStr, String format, TimeZone tz) {
        SimpleDateFormat formatter = FORMATTER_CACHE.get(format);
        formatter.setTimeZone(tz);
        try {
            java.util.Date date = formatter.parse(dateStr);
            return date.getTime();
        }
        catch (ParseException e) {
            LOG.error(String.format("Exception when parsing datetime string '%s' in format '%s'", dateStr, format), (Throwable)e);
            return Long.MIN_VALUE;
        }
    }

    private static int getMillis(String dateStr) {
        int length = dateStr.length();
        if (length == 19) {
            return 0;
        }
        if (length == 21) {
            return Integer.parseInt(dateStr.substring(20)) * 100;
        }
        if (length == 22) {
            return Integer.parseInt(dateStr.substring(20)) * 10;
        }
        if (length >= 23 && length <= 26) {
            return Integer.parseInt(dateStr.substring(20, 23));
        }
        return 0;
    }

    public static int extractYearMonth(TimeUnitRange range2, int v) {
        switch (range2) {
            case YEAR: {
                return v / 12;
            }
            case MONTH: {
                return v % 12;
            }
            case QUARTER: {
                return (v % 12 + 2) / 3;
            }
        }
        throw new UnsupportedOperationException("Unsupported TimeUnitRange: " + (Object)((Object)range2));
    }

    public static long unixTimeExtract(TimeUnitRange range2, int ts) {
        return DateTimeUtils.unixTimeExtract(range2, ts);
    }

    public static long extractFromTimestamp(TimeUnitRange range2, TimestampData ts, TimeZone tz) {
        return SqlDateTimeUtils.convertExtract(range2, ts, REUSE_TIMESTAMP_TYPE, tz);
    }

    private static long convertExtract(TimeUnitRange range2, TimestampData ts, LogicalType type, TimeZone tz) {
        TimeUnit startUnit = range2.startUnit;
        long millisecond = ts.getMillisecond();
        int nanoOfMillisecond = ts.getNanoOfMillisecond();
        long offset = tz.getOffset(millisecond);
        long utcTs = millisecond + offset;
        switch (startUnit) {
            case MILLENNIUM: 
            case CENTURY: 
            case YEAR: 
            case QUARTER: 
            case MONTH: 
            case DAY: 
            case DOW: 
            case DOY: 
            case WEEK: {
                if (type instanceof TimestampType) {
                    long d = SqlDateTimeUtils.divide(utcTs, TimeUnit.DAY.multiplier);
                    return DateTimeUtils.unixDateExtract(range2, d);
                }
                if (type instanceof DateType) {
                    return SqlDateTimeUtils.divide(utcTs, TimeUnit.DAY.multiplier);
                }
                throw new TableException(type + " is unsupported now.");
            }
            case DECADE: {
                throw new TableException("DECADE is unsupported now.");
            }
            case EPOCH: {
                throw new TableException("EPOCH is unsupported now.");
            }
            case MICROSECOND: {
                if (type instanceof TimestampType) {
                    long millis = SqlDateTimeUtils.divide(SqlDateTimeUtils.mod(utcTs, SqlDateTimeUtils.getFactor(startUnit)), startUnit.multiplier);
                    int micros = nanoOfMillisecond / 1000;
                    return millis + (long)micros;
                }
                throw new TableException(type + " is unsupported now.");
            }
            case NANOSECOND: {
                if (type instanceof TimestampType) {
                    long millis = SqlDateTimeUtils.divide(SqlDateTimeUtils.mod(utcTs, SqlDateTimeUtils.getFactor(startUnit)), startUnit.multiplier);
                    return millis + (long)nanoOfMillisecond;
                }
                throw new TableException(type + " is unsupported now.");
            }
        }
        long res = SqlDateTimeUtils.mod(utcTs, SqlDateTimeUtils.getFactor(startUnit));
        res = SqlDateTimeUtils.divide(res, startUnit.multiplier);
        return res;
    }

    private static long divide(long res, BigDecimal value) {
        if (value.equals(BigDecimal.ONE)) {
            return res;
        }
        if (value.compareTo(BigDecimal.ONE) < 0 && value.signum() == 1) {
            BigDecimal reciprocal = BigDecimal.ONE.divide(value, RoundingMode.UNNECESSARY);
            return reciprocal.multiply(BigDecimal.valueOf(res)).longValue();
        }
        return res / value.longValue();
    }

    private static long mod(long res, BigDecimal value) {
        if (value.equals(BigDecimal.ONE)) {
            return res;
        }
        return res % value.longValue();
    }

    private static BigDecimal getFactor(TimeUnit unit) {
        switch (unit) {
            case DAY: {
                return BigDecimal.ONE;
            }
            case HOUR: {
                return TimeUnit.DAY.multiplier;
            }
            case MINUTE: {
                return TimeUnit.HOUR.multiplier;
            }
            case SECOND: {
                return TimeUnit.MINUTE.multiplier;
            }
            case MICROSECOND: 
            case NANOSECOND: 
            case MILLISECOND: {
                return TimeUnit.SECOND.multiplier;
            }
            case YEAR: {
                return BigDecimal.ONE;
            }
            case MONTH: {
                return TimeUnit.YEAR.multiplier;
            }
            case QUARTER: {
                return TimeUnit.YEAR.multiplier;
            }
            case MILLENNIUM: 
            case CENTURY: 
            case DECADE: {
                return BigDecimal.ONE;
            }
        }
        throw new IllegalArgumentException("Invalid start unit.");
    }

    public static long timestampFloor(TimeUnitRange range2, long ts) {
        return SqlDateTimeUtils.timestampFloor(range2, ts, UTC_ZONE);
    }

    public static long timestampFloor(TimeUnitRange range2, long ts, TimeZone tz) {
        long offset = tz.getOffset(ts);
        long utcTs = ts + offset;
        switch (range2) {
            case HOUR: {
                return SqlDateTimeUtils.floor(utcTs, 3600000L) - offset;
            }
            case DAY: {
                return SqlDateTimeUtils.floor(utcTs, 86400000L) - offset;
            }
            case YEAR: 
            case MONTH: 
            case QUARTER: {
                int days = (int)(utcTs / 86400000L + 2440588L);
                return SqlDateTimeUtils.julianDateFloor(range2, days, true) * 86400000L - offset;
            }
        }
        throw new AssertionError((Object)range2);
    }

    public static long timestampCeil(TimeUnitRange range2, long ts) {
        return SqlDateTimeUtils.timestampCeil(range2, ts, UTC_ZONE);
    }

    public static long timestampCeil(TimeUnitRange range2, long ts, TimeZone tz) {
        long offset = tz.getOffset(ts);
        long utcTs = ts + offset;
        switch (range2) {
            case HOUR: {
                return SqlDateTimeUtils.ceil(utcTs, 3600000L) - offset;
            }
            case DAY: {
                return SqlDateTimeUtils.ceil(utcTs, 86400000L) - offset;
            }
            case YEAR: 
            case MONTH: 
            case QUARTER: {
                int days = (int)(utcTs / 86400000L + 2440588L);
                return SqlDateTimeUtils.julianDateFloor(range2, days, false) * 86400000L - offset;
            }
        }
        throw new AssertionError((Object)range2);
    }

    private static long floor(long a, long b) {
        long r = a % b;
        if (r < 0L) {
            return a - r - b;
        }
        return a - r;
    }

    private static long ceil(long a, long b) {
        long r = a % b;
        if (r > 0L) {
            return a - r + b;
        }
        return a - r;
    }

    private static long julianDateFloor(TimeUnitRange range2, int julian, boolean floor) {
        int b = 0;
        int c = 0;
        if (julian > 2299160) {
            int a = julian + 32044;
            b = (4 * a + 3) / 146097;
            c = a - b * 146097 / 4;
        } else {
            b = 0;
            c = julian + 32082;
        }
        int d = (4 * c + 3) / 1461;
        int e = c - 1461 * d / 4;
        int m = (5 * e + 2) / 153;
        int day = e - (153 * m + 2) / 5 + 1;
        int month = m + 3 - 12 * (m / 10);
        int quarter = (month + 2) / 3;
        int year = b * 100 + d - 4800 + m / 10;
        switch (range2) {
            case YEAR: {
                if (!(floor || month <= 1 && day <= 1)) {
                    ++year;
                }
                return DateTimeUtils.ymdToUnixDate(year, 1, 1);
            }
            case MONTH: {
                if (!floor && day > 1) {
                    ++month;
                }
                return DateTimeUtils.ymdToUnixDate(year, month, 1);
            }
            case QUARTER: {
                if (!(floor || month <= 1 && day <= 1)) {
                    ++quarter;
                }
                return DateTimeUtils.ymdToUnixDate(year, quarter * 3 - 2, 1);
            }
        }
        throw new AssertionError((Object)range2);
    }

    public static int dateDiff(long t1, long t2, TimeZone tz) {
        ZoneId zoneId = tz.toZoneId();
        LocalDate ld1 = Instant.ofEpochMilli(t1).atZone(zoneId).toLocalDate();
        LocalDate ld2 = Instant.ofEpochMilli(t2).atZone(zoneId).toLocalDate();
        return (int)ChronoUnit.DAYS.between(ld2, ld1);
    }

    public static int dateDiff(String t1Str, long t2, TimeZone tz) {
        long t1 = SqlDateTimeUtils.parseToTimeMillis(t1Str, tz);
        return SqlDateTimeUtils.dateDiff(t1, t2, tz);
    }

    public static int dateDiff(long t1, String t2Str, TimeZone tz) {
        long t2 = SqlDateTimeUtils.parseToTimeMillis(t2Str, tz);
        return SqlDateTimeUtils.dateDiff(t1, t2, tz);
    }

    public static int dateDiff(String t1Str, String t2Str, TimeZone tz) {
        long t1 = SqlDateTimeUtils.parseToTimeMillis(t1Str, tz);
        long t2 = SqlDateTimeUtils.parseToTimeMillis(t2Str, tz);
        return SqlDateTimeUtils.dateDiff(t1, t2, tz);
    }

    public static int dateDiff(long t1, long t2) {
        return SqlDateTimeUtils.dateDiff(t1, t2, UTC_ZONE);
    }

    public static int dateDiff(String t1Str, long t2) {
        return SqlDateTimeUtils.dateDiff(t1Str, t2, UTC_ZONE);
    }

    public static int dateDiff(long t1, String t2Str) {
        return SqlDateTimeUtils.dateDiff(t1, t2Str, UTC_ZONE);
    }

    public static int dateDiff(String t1Str, String t2Str) {
        return SqlDateTimeUtils.dateDiff(t1Str, t2Str, UTC_ZONE);
    }

    public static String dateSub(String dateStr, int days, TimeZone tz) {
        long ts = SqlDateTimeUtils.parseToTimeMillis(dateStr, tz);
        if (ts == Long.MIN_VALUE) {
            return null;
        }
        return SqlDateTimeUtils.dateSub(ts, days, tz);
    }

    public static String dateSub(long ts, int days, TimeZone tz) {
        ZoneId zoneId = tz.toZoneId();
        Instant instant = Instant.ofEpochMilli(ts);
        ZonedDateTime zdt = ZonedDateTime.ofInstant(instant, zoneId);
        long resultTs = zdt.minusDays(days).toInstant().toEpochMilli();
        return SqlDateTimeUtils.dateFormat(resultTs, DATE_FORMAT_STRING, tz);
    }

    public static String dateSub(String dateStr, int days) {
        return SqlDateTimeUtils.dateSub(dateStr, days, UTC_ZONE);
    }

    public static String dateSub(long ts, int days) {
        return SqlDateTimeUtils.dateSub(ts, days, UTC_ZONE);
    }

    public static String dateAdd(String dateStr, int days, TimeZone tz) {
        long ts = SqlDateTimeUtils.parseToTimeMillis(dateStr, tz);
        if (ts == Long.MIN_VALUE) {
            return null;
        }
        return SqlDateTimeUtils.dateAdd(ts, days, tz);
    }

    public static String dateAdd(long ts, int days, TimeZone tz) {
        ZoneId zoneId = tz.toZoneId();
        Instant instant = Instant.ofEpochMilli(ts);
        ZonedDateTime zdt = ZonedDateTime.ofInstant(instant, zoneId);
        long resultTs = zdt.plusDays(days).toInstant().toEpochMilli();
        return SqlDateTimeUtils.dateFormat(resultTs, DATE_FORMAT_STRING, tz);
    }

    public static String dateAdd(String dateStr, int days) {
        return SqlDateTimeUtils.dateAdd(dateStr, days, UTC_ZONE);
    }

    public static String dateAdd(long ts, int days) {
        return SqlDateTimeUtils.dateAdd(ts, days, UTC_ZONE);
    }

    public static long fromTimestamp(long ts) {
        return ts;
    }

    public static String fromUnixtime(long unixtime, TimeZone tz) {
        return SqlDateTimeUtils.fromUnixtime(unixtime, TIMESTAMP_FORMAT_STRING, tz);
    }

    public static String fromUnixtime(long unixtime, String format, TimeZone tz) {
        SimpleDateFormat formatter = FORMATTER_CACHE.get(format);
        formatter.setTimeZone(tz);
        java.util.Date date = new java.util.Date(unixtime * 1000L);
        try {
            return formatter.format(date);
        }
        catch (Exception e) {
            LOG.error("Exception when formatting.", (Throwable)e);
            return null;
        }
    }

    public static String fromUnixtime(double unixtime, TimeZone tz) {
        return SqlDateTimeUtils.fromUnixtime((long)unixtime, tz);
    }

    public static String fromUnixtime(DecimalData unixtime, TimeZone tz) {
        return SqlDateTimeUtils.fromUnixtime(DecimalDataUtils.castToLong(unixtime), tz);
    }

    public static String fromUnixtime(long unixtime) {
        return SqlDateTimeUtils.fromUnixtime(unixtime, UTC_ZONE);
    }

    public static String fromUnixtime(long unixtime, String format) {
        return SqlDateTimeUtils.fromUnixtime(unixtime, format, UTC_ZONE);
    }

    public static String fromUnixtime(double unixtime) {
        return SqlDateTimeUtils.fromUnixtime(unixtime, UTC_ZONE);
    }

    public static String fromUnixtime(DecimalData unixtime) {
        return SqlDateTimeUtils.fromUnixtime(unixtime, UTC_ZONE);
    }

    public static long unixTimestamp() {
        return System.currentTimeMillis() / 1000L;
    }

    public static long unixTimestamp(String dateStr, TimeZone tz) {
        return SqlDateTimeUtils.unixTimestamp(dateStr, TIMESTAMP_FORMAT_STRING, tz);
    }

    public static long unixTimestamp(String dateStr, String format, TimeZone tz) {
        long ts = SqlDateTimeUtils.parseToTimeMillis(dateStr, format, tz);
        if (ts == Long.MIN_VALUE) {
            return Long.MIN_VALUE;
        }
        return ts / 1000L;
    }

    public static long unixTimestamp(String dateStr) {
        return SqlDateTimeUtils.unixTimestamp(dateStr, UTC_ZONE);
    }

    public static long unixTimestamp(String dateStr, String format) {
        return SqlDateTimeUtils.unixTimestamp(dateStr, format, UTC_ZONE);
    }

    public static long unixTimestamp(long ts) {
        return ts / 1000L;
    }

    public static LocalDate unixDateToLocalDate(int date) {
        return SqlDateTimeUtils.julianToLocalDate(date + 2440588);
    }

    private static LocalDate julianToLocalDate(int julian) {
        int j2 = julian + 32044;
        int g = j2 / 146097;
        int dg = j2 % 146097;
        int c = (dg / 36524 + 1) * 3 / 4;
        int dc = dg - c * 36524;
        int b = dc / 1461;
        int db = dc % 1461;
        int a = (db / 365 + 1) * 3 / 4;
        int da = db - a * 365;
        int y = g * 400 + c * 100 + b * 4 + a;
        int m = (da * 5 + 308) / 153 - 2;
        int d = da - (m + 4) * 153 / 5 + 122;
        int year = y - 4800 + (m + 2) / 12;
        int month = (m + 2) % 12 + 1;
        int day = d + 1;
        return LocalDate.of(year, month, day);
    }

    public static int localDateToUnixDate(LocalDate date) {
        return SqlDateTimeUtils.ymdToUnixDate(date.getYear(), date.getMonthValue(), date.getDayOfMonth());
    }

    private static int ymdToUnixDate(int year, int month, int day) {
        int julian = SqlDateTimeUtils.ymdToJulian(year, month, day);
        return julian - 2440588;
    }

    private static int ymdToJulian(int year, int month, int day) {
        int a = (14 - month) / 12;
        int y = year + 4800 - a;
        int m = month + 12 * a - 3;
        return day + (153 * m + 2) / 5 + 365 * y + y / 4 - y / 100 + y / 400 - 32045;
    }

    public static LocalTime unixTimeToLocalTime(int time) {
        int h = time / 3600000;
        int time2 = time % 3600000;
        int m = time2 / 60000;
        int time3 = time2 % 60000;
        int s = time3 / 1000;
        int ms = time3 % 1000;
        return LocalTime.of(h, m, s, ms * 1000000);
    }

    public static int localTimeToUnixDate(LocalTime time) {
        return time.getHour() * 3600000 + time.getMinute() * 60000 + time.getSecond() * 1000 + time.getNano() / 1000000;
    }

    public static LocalDateTime unixTimestampToLocalDateTime(long timestamp) {
        int date = (int)(timestamp / 86400000L);
        int time = (int)(timestamp % 86400000L);
        if (time < 0) {
            --date;
            time = (int)((long)time + 86400000L);
        }
        LocalDate localDate = SqlDateTimeUtils.unixDateToLocalDate(date);
        LocalTime localTime = SqlDateTimeUtils.unixTimeToLocalTime(time);
        return LocalDateTime.of(localDate, localTime);
    }

    public static long localDateTimeToUnixTimestamp(LocalDateTime dateTime) {
        return SqlDateTimeUtils.unixTimestamp(dateTime.getYear(), dateTime.getMonthValue(), dateTime.getDayOfMonth(), dateTime.getHour(), dateTime.getMinute(), dateTime.getSecond(), dateTime.getNano() / 1000000);
    }

    private static long unixTimestamp(int year, int month, int day, int hour, int minute, int second, int mills) {
        int date = SqlDateTimeUtils.ymdToUnixDate(year, month, day);
        return (long)date * 86400000L + (long)hour * 3600000L + (long)minute * 60000L + (long)second * 1000L + (long)mills;
    }

    public static TimestampData timestampToTimestampWithLocalZone(TimestampData ts, TimeZone tz) {
        return TimestampData.fromInstant(ts.toLocalDateTime().atZone(tz.toZoneId()).toInstant());
    }

    public static TimestampData timestampWithLocalZoneToTimestamp(TimestampData ts, TimeZone tz) {
        return TimestampData.fromLocalDateTime(LocalDateTime.ofInstant(ts.toInstant(), tz.toZoneId()));
    }

    public static int timestampWithLocalZoneToDate(long ts, TimeZone tz) {
        return SqlDateTimeUtils.localDateToUnixDate(LocalDateTime.ofInstant(Instant.ofEpochMilli(ts), tz.toZoneId()).toLocalDate());
    }

    public static int timestampWithLocalZoneToTime(long ts, TimeZone tz) {
        return SqlDateTimeUtils.localTimeToUnixDate(LocalDateTime.ofInstant(Instant.ofEpochMilli(ts), tz.toZoneId()).toLocalTime());
    }

    public static long dateToTimestampWithLocalZone(int date, TimeZone tz) {
        return LocalDateTime.of(SqlDateTimeUtils.unixDateToLocalDate(date), LocalTime.MIDNIGHT).atZone(tz.toZoneId()).toInstant().toEpochMilli();
    }

    public static long timeToTimestampWithLocalZone(int time, TimeZone tz) {
        return SqlDateTimeUtils.unixTimestampToLocalDateTime(time).atZone(tz.toZoneId()).toInstant().toEpochMilli();
    }

    private static boolean isInteger(String s) {
        boolean isInt = s.length() > 0;
        for (int i = 0; i < s.length(); ++i) {
            if (s.charAt(i) >= '0' && s.charAt(i) <= '9') continue;
            isInt = false;
            break;
        }
        return isInt;
    }

    private static boolean isLeapYear(int s) {
        return s % 400 == 0 || s % 4 == 0 && s % 100 != 0;
    }

    private static boolean isIllegalDate(int y, int m, int d) {
        int[] monthOf31Days = new int[]{1, 3, 5, 7, 8, 10, 12};
        if (y < 0 || y > 9999 || m < 1 || m > 12 || d < 1 || d > 31) {
            return false;
        }
        if (!(m != 2 || d <= 28 || SqlDateTimeUtils.isLeapYear(y) && d == 29)) {
            return false;
        }
        if (d == 31) {
            for (int i : monthOf31Days) {
                if (i != m) continue;
                return true;
            }
            return false;
        }
        return true;
    }

    public static Integer dateStringToUnixDate(String s) {
        int d;
        int m;
        int y;
        int hyphen1;
        int ws1 = s.indexOf(" ");
        if (ws1 > 0) {
            s = s.substring(0, ws1);
        }
        if ((hyphen1 = s.indexOf(45)) < 0) {
            if (!SqlDateTimeUtils.isInteger(s.trim())) {
                return null;
            }
            y = Integer.parseInt(s.trim());
            m = 1;
            d = 1;
        } else {
            if (!SqlDateTimeUtils.isInteger(s.substring(0, hyphen1).trim())) {
                return null;
            }
            y = Integer.parseInt(s.substring(0, hyphen1).trim());
            int hyphen2 = s.indexOf(45, hyphen1 + 1);
            if (hyphen2 < 0) {
                if (!SqlDateTimeUtils.isInteger(s.substring(hyphen1 + 1).trim())) {
                    return null;
                }
                m = Integer.parseInt(s.substring(hyphen1 + 1).trim());
                d = 1;
            } else {
                if (!SqlDateTimeUtils.isInteger(s.substring(hyphen1 + 1, hyphen2).trim())) {
                    return null;
                }
                m = Integer.parseInt(s.substring(hyphen1 + 1, hyphen2).trim());
                if (!SqlDateTimeUtils.isInteger(s.substring(hyphen2 + 1).trim())) {
                    return null;
                }
                d = Integer.parseInt(s.substring(hyphen2 + 1).trim());
            }
        }
        if (!SqlDateTimeUtils.isIllegalDate(y, m, d)) {
            return null;
        }
        return DateTimeUtils.ymdToUnixDate(y, m, d);
    }

    public static Integer timeStringToUnixDate(String v) {
        return SqlDateTimeUtils.timeStringToUnixDate(v, 0);
    }

    public static Integer timeStringToUnixDate(String v, int start) {
        int milli;
        int second;
        int minute;
        int hour;
        int timezoneMinute;
        int timezoneHour;
        int colon1 = v.indexOf(58, start);
        int operator = -1;
        int end = v.length();
        int timezone = v.indexOf(45, start);
        if (timezone < 0) {
            timezone = v.indexOf(43, start);
            operator = 1;
        }
        if (timezone < 0) {
            timezoneHour = 0;
            timezoneMinute = 0;
        } else {
            end = timezone;
            int colon3 = v.indexOf(58, timezone);
            if (colon3 < 0) {
                if (!SqlDateTimeUtils.isInteger(v.substring(timezone + 1).trim())) {
                    return null;
                }
                timezoneHour = Integer.parseInt(v.substring(timezone + 1).trim());
                timezoneMinute = 0;
            } else {
                if (!SqlDateTimeUtils.isInteger(v.substring(timezone + 1, colon3).trim())) {
                    return null;
                }
                timezoneHour = Integer.parseInt(v.substring(timezone + 1, colon3).trim());
                if (!SqlDateTimeUtils.isInteger(v.substring(colon3 + 1).trim())) {
                    return null;
                }
                timezoneMinute = Integer.parseInt(v.substring(colon3 + 1).trim());
            }
        }
        if (colon1 < 0) {
            if (!SqlDateTimeUtils.isInteger(v.substring(start, end).trim())) {
                return null;
            }
            hour = Integer.parseInt(v.substring(start, end).trim());
            minute = 1;
            second = 1;
            milli = 0;
        } else {
            if (!SqlDateTimeUtils.isInteger(v.substring(start, colon1).trim())) {
                return null;
            }
            hour = Integer.parseInt(v.substring(start, colon1).trim());
            int colon2 = v.indexOf(58, colon1 + 1);
            if (colon2 < 0) {
                if (!SqlDateTimeUtils.isInteger(v.substring(colon1 + 1, end).trim())) {
                    return null;
                }
                minute = Integer.parseInt(v.substring(colon1 + 1, end).trim());
                second = 1;
                milli = 0;
            } else {
                if (!SqlDateTimeUtils.isInteger(v.substring(colon1 + 1, colon2).trim())) {
                    return null;
                }
                minute = Integer.parseInt(v.substring(colon1 + 1, colon2).trim());
                int dot = v.indexOf(46, colon2);
                if (dot < 0) {
                    if (!SqlDateTimeUtils.isInteger(v.substring(colon2 + 1, end).trim())) {
                        return null;
                    }
                    second = Integer.parseInt(v.substring(colon2 + 1, end).trim());
                    milli = 0;
                } else {
                    if (!SqlDateTimeUtils.isInteger(v.substring(colon2 + 1, dot).trim())) {
                        return null;
                    }
                    second = Integer.parseInt(v.substring(colon2 + 1, dot).trim());
                    milli = SqlDateTimeUtils.parseFraction(v.substring(dot + 1, end).trim(), 100);
                }
            }
        }
        return (hour += operator * timezoneHour) * 3600000 + (minute += operator * timezoneMinute) * 60000 + second * 1000 + milli;
    }

    private static int parseFraction(String v, int multiplier) {
        int r = 0;
        for (int i = 0; i < v.length(); ++i) {
            char c = v.charAt(i);
            int x = c < '0' || c > '9' ? 0 : c - 48;
            r += multiplier * x;
            if (multiplier < 10) {
                if (i + 1 >= v.length() || v.charAt(i + 1) < '5') break;
                ++r;
                break;
            }
            multiplier /= 10;
        }
        return r;
    }

    public static String timestampToString(TimestampData ts, int precision) {
        LocalDateTime ldt = ts.toLocalDateTime();
        String fraction = SqlDateTimeUtils.pad(9, ldt.getNano());
        while (fraction.length() > precision && fraction.endsWith("0")) {
            fraction = fraction.substring(0, fraction.length() - 1);
        }
        StringBuilder ymdhms = SqlDateTimeUtils.ymdhms(new StringBuilder(), ldt.getYear(), ldt.getMonthValue(), ldt.getDayOfMonth(), ldt.getHour(), ldt.getMinute(), ldt.getSecond());
        if (fraction.length() > 0) {
            ymdhms.append(".").append(fraction);
        }
        return ymdhms.toString();
    }

    public static String timestampToString(TimestampData ts, TimeZone tz, int precision) {
        return SqlDateTimeUtils.timestampToString(SqlDateTimeUtils.timestampWithLocalZoneToTimestamp(ts, tz), precision);
    }

    private static String pad(int length, long v) {
        StringBuilder s = new StringBuilder(Long.toString(v));
        while (s.length() < length) {
            s.insert(0, "0");
        }
        return s.toString();
    }

    private static StringBuilder hms(StringBuilder b, int h, int m, int s) {
        SqlDateTimeUtils.int2(b, h);
        b.append(':');
        SqlDateTimeUtils.int2(b, m);
        b.append(':');
        SqlDateTimeUtils.int2(b, s);
        return b;
    }

    private static StringBuilder ymdhms(StringBuilder b, int year, int month, int day, int h, int m, int s) {
        SqlDateTimeUtils.ymd(b, year, month, day);
        b.append(' ');
        SqlDateTimeUtils.hms(b, h, m, s);
        return b;
    }

    private static StringBuilder ymd(StringBuilder b, int year, int month, int day) {
        SqlDateTimeUtils.int4(b, year);
        b.append('-');
        SqlDateTimeUtils.int2(b, month);
        b.append('-');
        SqlDateTimeUtils.int2(b, day);
        return b;
    }

    private static void int4(StringBuilder buf, int i) {
        buf.append((char)(48 + i / 1000 % 10));
        buf.append((char)(48 + i / 100 % 10));
        buf.append((char)(48 + i / 10 % 10));
        buf.append((char)(48 + i % 10));
    }

    public static TimestampData truncate(TimestampData ts, int precision) {
        String fraction = Integer.toString(ts.toLocalDateTime().getNano());
        if (fraction.length() <= precision) {
            return ts;
        }
        if (precision <= 3) {
            return TimestampData.fromEpochMillis(SqlDateTimeUtils.zeroLastDigits(ts.getMillisecond(), 3 - precision));
        }
        return TimestampData.fromEpochMillis(ts.getMillisecond(), (int)SqlDateTimeUtils.zeroLastDigits(ts.getNanoOfMillisecond(), 9 - precision));
    }

    private static long zeroLastDigits(long l, int n) {
        long tenToTheN = (long)Math.pow(10.0, n);
        return l / tenToTheN * tenToTheN;
    }
}

