/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.formats.json;

import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Array;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAccessor;
import java.time.temporal.TemporalQueries;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import org.apache.flink.annotation.Internal;
import org.apache.flink.api.common.serialization.DeserializationSchema;
import org.apache.flink.api.common.typeinfo.TypeInformation;
import org.apache.flink.formats.json.TimeFormats;
import org.apache.flink.formats.json.TimestampFormat;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.databind.JsonNode;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.databind.node.ArrayNode;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.databind.node.TextNode;
import org.apache.flink.table.api.TableException;
import org.apache.flink.table.data.DecimalData;
import org.apache.flink.table.data.GenericArrayData;
import org.apache.flink.table.data.GenericMapData;
import org.apache.flink.table.data.GenericRowData;
import org.apache.flink.table.data.RowData;
import org.apache.flink.table.data.StringData;
import org.apache.flink.table.data.TimestampData;
import org.apache.flink.table.types.logical.ArrayType;
import org.apache.flink.table.types.logical.DecimalType;
import org.apache.flink.table.types.logical.LogicalType;
import org.apache.flink.table.types.logical.LogicalTypeFamily;
import org.apache.flink.table.types.logical.MapType;
import org.apache.flink.table.types.logical.RowType;
import org.apache.flink.table.types.logical.utils.LogicalTypeChecks;
import org.apache.flink.table.types.logical.utils.LogicalTypeUtils;
import org.apache.flink.util.Preconditions;

@Internal
public class JsonRowDataDeserializationSchema
implements DeserializationSchema<RowData> {
    private static final long serialVersionUID = 1L;
    private final boolean failOnMissingField;
    private final boolean ignoreParseErrors;
    private final TypeInformation<RowData> resultTypeInfo;
    private final DeserializationRuntimeConverter runtimeConverter;
    private final ObjectMapper objectMapper = new ObjectMapper();
    private final TimestampFormat timestampFormat;

    public JsonRowDataDeserializationSchema(RowType rowType, TypeInformation<RowData> resultTypeInfo, boolean failOnMissingField, boolean ignoreParseErrors, TimestampFormat timestampFormat) {
        if (ignoreParseErrors && failOnMissingField) {
            throw new IllegalArgumentException("JSON format doesn't support failOnMissingField and ignoreParseErrors are both enabled.");
        }
        this.resultTypeInfo = (TypeInformation)Preconditions.checkNotNull(resultTypeInfo);
        this.failOnMissingField = failOnMissingField;
        this.ignoreParseErrors = ignoreParseErrors;
        this.runtimeConverter = this.createRowConverter((RowType)Preconditions.checkNotNull((Object)rowType));
        this.timestampFormat = timestampFormat;
    }

    public RowData deserialize(byte[] message) throws IOException {
        try {
            JsonNode root = this.objectMapper.readTree(message);
            return (RowData)this.runtimeConverter.convert(root);
        }
        catch (Throwable t) {
            if (this.ignoreParseErrors) {
                return null;
            }
            throw new IOException(String.format("Failed to deserialize JSON '%s'.", new String(message)), t);
        }
    }

    public boolean isEndOfStream(RowData nextElement) {
        return false;
    }

    public TypeInformation<RowData> getProducedType() {
        return this.resultTypeInfo;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        JsonRowDataDeserializationSchema that = (JsonRowDataDeserializationSchema)o;
        return this.failOnMissingField == that.failOnMissingField && this.ignoreParseErrors == that.ignoreParseErrors && this.resultTypeInfo.equals(that.resultTypeInfo) && this.timestampFormat.equals((Object)that.timestampFormat);
    }

    public int hashCode() {
        return Objects.hash(new Object[]{this.failOnMissingField, this.ignoreParseErrors, this.resultTypeInfo, this.timestampFormat});
    }

    private DeserializationRuntimeConverter createConverter(LogicalType type) {
        return this.wrapIntoNullableConverter(this.createNotNullConverter(type));
    }

    private DeserializationRuntimeConverter createNotNullConverter(LogicalType type) {
        switch (type.getTypeRoot()) {
            case NULL: {
                return jsonNode -> null;
            }
            case BOOLEAN: {
                return this::convertToBoolean;
            }
            case TINYINT: {
                return jsonNode -> Byte.parseByte(jsonNode.asText().trim());
            }
            case SMALLINT: {
                return jsonNode -> Short.parseShort(jsonNode.asText().trim());
            }
            case INTEGER: 
            case INTERVAL_YEAR_MONTH: {
                return this::convertToInt;
            }
            case BIGINT: 
            case INTERVAL_DAY_TIME: {
                return this::convertToLong;
            }
            case DATE: {
                return this::convertToDate;
            }
            case TIME_WITHOUT_TIME_ZONE: {
                return this::convertToTime;
            }
            case TIMESTAMP_WITHOUT_TIME_ZONE: {
                return this::convertToTimestamp;
            }
            case FLOAT: {
                return this::convertToFloat;
            }
            case DOUBLE: {
                return this::convertToDouble;
            }
            case CHAR: 
            case VARCHAR: {
                return this::convertToString;
            }
            case BINARY: 
            case VARBINARY: {
                return this::convertToBytes;
            }
            case DECIMAL: {
                return this.createDecimalConverter((DecimalType)type);
            }
            case ARRAY: {
                return this.createArrayConverter((ArrayType)type);
            }
            case MAP: 
            case MULTISET: {
                return this.createMapConverter((MapType)type);
            }
            case ROW: {
                return this.createRowConverter((RowType)type);
            }
        }
        throw new UnsupportedOperationException("Unsupported type: " + type);
    }

    private boolean convertToBoolean(JsonNode jsonNode) {
        if (jsonNode.isBoolean()) {
            return jsonNode.asBoolean();
        }
        return Boolean.parseBoolean(jsonNode.asText().trim());
    }

    private int convertToInt(JsonNode jsonNode) {
        if (jsonNode.canConvertToInt()) {
            return jsonNode.asInt();
        }
        return Integer.parseInt(jsonNode.asText().trim());
    }

    private long convertToLong(JsonNode jsonNode) {
        if (jsonNode.canConvertToLong()) {
            return jsonNode.asLong();
        }
        return Long.parseLong(jsonNode.asText().trim());
    }

    private double convertToDouble(JsonNode jsonNode) {
        if (jsonNode.isDouble()) {
            return jsonNode.asDouble();
        }
        return Double.parseDouble(jsonNode.asText().trim());
    }

    private float convertToFloat(JsonNode jsonNode) {
        if (jsonNode.isDouble()) {
            return (float)jsonNode.asDouble();
        }
        return Float.parseFloat(jsonNode.asText().trim());
    }

    private int convertToDate(JsonNode jsonNode) {
        LocalDate date = DateTimeFormatter.ISO_LOCAL_DATE.parse(jsonNode.asText()).query(TemporalQueries.localDate());
        return (int)date.toEpochDay();
    }

    private int convertToTime(JsonNode jsonNode) {
        TemporalAccessor parsedTime = TimeFormats.SQL_TIME_FORMAT.parse(jsonNode.asText());
        LocalTime localTime = parsedTime.query(TemporalQueries.localTime());
        return localTime.toSecondOfDay() * 1000;
    }

    private TimestampData convertToTimestamp(JsonNode jsonNode) {
        TemporalAccessor parsedTimestamp;
        switch (this.timestampFormat) {
            case SQL: {
                parsedTimestamp = TimeFormats.SQL_TIMESTAMP_FORMAT.parse(jsonNode.asText());
                break;
            }
            case ISO_8601: {
                parsedTimestamp = TimeFormats.ISO8601_TIMESTAMP_FORMAT.parse(jsonNode.asText());
                break;
            }
            default: {
                throw new TableException(String.format("Unsupported timestamp format '%s'. Validator should have checked that.", new Object[]{this.timestampFormat}));
            }
        }
        LocalTime localTime = parsedTimestamp.query(TemporalQueries.localTime());
        LocalDate localDate = parsedTimestamp.query(TemporalQueries.localDate());
        return TimestampData.fromLocalDateTime((LocalDateTime)LocalDateTime.of(localDate, localTime));
    }

    private StringData convertToString(JsonNode jsonNode) {
        return StringData.fromString((String)jsonNode.asText());
    }

    private byte[] convertToBytes(JsonNode jsonNode) {
        try {
            return jsonNode.binaryValue();
        }
        catch (IOException e) {
            throw new JsonParseException("Unable to deserialize byte array.", e);
        }
    }

    private DeserializationRuntimeConverter createDecimalConverter(DecimalType decimalType) {
        int precision = decimalType.getPrecision();
        int scale = decimalType.getScale();
        return jsonNode -> {
            BigDecimal bigDecimal = jsonNode.isBigDecimal() ? jsonNode.decimalValue() : new BigDecimal(jsonNode.asText());
            return DecimalData.fromBigDecimal((BigDecimal)bigDecimal, (int)precision, (int)scale);
        };
    }

    private DeserializationRuntimeConverter createArrayConverter(ArrayType arrayType) {
        DeserializationRuntimeConverter elementConverter = this.createConverter(arrayType.getElementType());
        Class elementClass = LogicalTypeUtils.toInternalConversionClass((LogicalType)arrayType.getElementType());
        return jsonNode -> {
            ArrayNode node = (ArrayNode)jsonNode;
            Object[] array = (Object[])Array.newInstance(elementClass, node.size());
            for (int i = 0; i < node.size(); ++i) {
                JsonNode innerNode = node.get(i);
                array[i] = elementConverter.convert(innerNode);
            }
            return new GenericArrayData(array);
        };
    }

    private DeserializationRuntimeConverter createMapConverter(MapType mapType) {
        LogicalType keyType = mapType.getKeyType();
        if (!LogicalTypeChecks.hasFamily((LogicalType)keyType, (LogicalTypeFamily)LogicalTypeFamily.CHARACTER_STRING)) {
            throw new UnsupportedOperationException("JSON format doesn't support non-string as key type of map. The map type is: " + mapType.asSummaryString());
        }
        DeserializationRuntimeConverter keyConverter = this.createConverter(keyType);
        DeserializationRuntimeConverter valueConverter = this.createConverter(mapType.getValueType());
        return jsonNode -> {
            Iterator fields = jsonNode.fields();
            HashMap<Object, Object> result = new HashMap<Object, Object>();
            while (fields.hasNext()) {
                Map.Entry entry = (Map.Entry)fields.next();
                Object key = keyConverter.convert((JsonNode)TextNode.valueOf((String)((String)entry.getKey())));
                Object value = valueConverter.convert((JsonNode)entry.getValue());
                result.put(key, value);
            }
            return new GenericMapData(result);
        };
    }

    private DeserializationRuntimeConverter createRowConverter(RowType rowType) {
        DeserializationRuntimeConverter[] fieldConverters = (DeserializationRuntimeConverter[])rowType.getFields().stream().map(RowType.RowField::getType).map(this::createConverter).toArray(DeserializationRuntimeConverter[]::new);
        String[] fieldNames = rowType.getFieldNames().toArray(new String[0]);
        return jsonNode -> {
            ObjectNode node = (ObjectNode)jsonNode;
            int arity = fieldNames.length;
            GenericRowData row = new GenericRowData(arity);
            for (int i = 0; i < arity; ++i) {
                String fieldName = fieldNames[i];
                JsonNode field = node.get(fieldName);
                Object convertedField = this.convertField(fieldConverters[i], fieldName, field);
                row.setField(i, convertedField);
            }
            return row;
        };
    }

    private Object convertField(DeserializationRuntimeConverter fieldConverter, String fieldName, JsonNode field) {
        if (field == null) {
            if (this.failOnMissingField) {
                throw new JsonParseException("Could not find field with name '" + fieldName + "'.");
            }
            return null;
        }
        return fieldConverter.convert(field);
    }

    private DeserializationRuntimeConverter wrapIntoNullableConverter(DeserializationRuntimeConverter converter) {
        return jsonNode -> {
            if (jsonNode == null || jsonNode.isNull()) {
                return null;
            }
            try {
                return converter.convert(jsonNode);
            }
            catch (Throwable t) {
                if (!this.ignoreParseErrors) {
                    throw t;
                }
                return null;
            }
        };
    }

    private static final class JsonParseException
    extends RuntimeException {
        private static final long serialVersionUID = 1L;

        public JsonParseException(String message) {
            super(message);
        }

        public JsonParseException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    @FunctionalInterface
    private static interface DeserializationRuntimeConverter
    extends Serializable {
        public Object convert(JsonNode var1);
    }
}

