/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.client.cli;

import java.io.IOError;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.table.api.TableResult;
import org.apache.flink.table.api.TableSchema;
import org.apache.flink.table.client.SqlClientException;
import org.apache.flink.table.client.cli.CliChangelogResultView;
import org.apache.flink.table.client.cli.CliResultView;
import org.apache.flink.table.client.cli.CliStrings;
import org.apache.flink.table.client.cli.CliTableResultView;
import org.apache.flink.table.client.cli.CliTableauResultView;
import org.apache.flink.table.client.cli.CliUtils;
import org.apache.flink.table.client.cli.SqlCommandParser;
import org.apache.flink.table.client.cli.SqlCompleter;
import org.apache.flink.table.client.cli.SqlMultiLineParser;
import org.apache.flink.table.client.gateway.Executor;
import org.apache.flink.table.client.gateway.ProgramTargetDescriptor;
import org.apache.flink.table.client.gateway.ResultDescriptor;
import org.apache.flink.table.client.gateway.SqlExecutionException;
import org.apache.flink.types.Row;
import org.jline.reader.EndOfFileException;
import org.jline.reader.LineReader;
import org.jline.reader.LineReaderBuilder;
import org.jline.reader.MaskingCallback;
import org.jline.reader.UserInterruptException;
import org.jline.terminal.Terminal;
import org.jline.terminal.TerminalBuilder;
import org.jline.utils.AttributedString;
import org.jline.utils.AttributedStringBuilder;
import org.jline.utils.AttributedStyle;
import org.jline.utils.InfoCmp;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CliClient {
    private static final Logger LOG = LoggerFactory.getLogger(CliClient.class);
    private final Executor executor;
    private final String sessionId;
    private final Terminal terminal;
    private final LineReader lineReader;
    private final String prompt;
    private boolean isRunning;
    private static final int PLAIN_TERMINAL_WIDTH = 80;
    private static final int PLAIN_TERMINAL_HEIGHT = 30;
    private static final int SOURCE_MAX_SIZE = 50000;

    @VisibleForTesting
    public CliClient(Terminal terminal, String sessionId, Executor executor, Path historyFilePath) {
        this.terminal = terminal;
        this.sessionId = sessionId;
        this.executor = executor;
        terminal.writer().println();
        terminal.writer().flush();
        this.lineReader = LineReaderBuilder.builder().terminal(terminal).appName("Flink SQL CLI Client").parser(new SqlMultiLineParser()).completer(new SqlCompleter(sessionId, executor)).build();
        this.lineReader.option(LineReader.Option.DISABLE_EVENT_EXPANSION, true);
        this.lineReader.setVariable("errors", 1);
        this.lineReader.option(LineReader.Option.CASE_INSENSITIVE, true);
        if (Files.exists(historyFilePath, new LinkOption[0]) || CliUtils.createFile(historyFilePath)) {
            String msg = "Command history file path: " + historyFilePath;
            System.out.println(msg);
            LOG.info(msg);
            this.lineReader.setVariable("history-file", historyFilePath);
        } else {
            String msg = "Unable to create history file: " + historyFilePath;
            System.out.println(msg);
            LOG.warn(msg);
        }
        this.prompt = new AttributedStringBuilder().style(AttributedStyle.DEFAULT.foreground(2)).append("Flink SQL").style(AttributedStyle.DEFAULT).append("> ").toAnsi();
    }

    public CliClient(String sessionId, Executor executor, Path historyFilePath) {
        this(CliClient.createDefaultTerminal(), sessionId, executor, historyFilePath);
    }

    public Terminal getTerminal() {
        return this.terminal;
    }

    public String getSessionId() {
        return this.sessionId;
    }

    public void clearTerminal() {
        if (this.isPlainTerminal()) {
            for (int i = 0; i < 200; ++i) {
                this.terminal.writer().println();
            }
        } else {
            this.terminal.puts(InfoCmp.Capability.clear_screen, new Object[0]);
        }
    }

    public boolean isPlainTerminal() {
        return this.terminal.getWidth() == 0 && this.terminal.getHeight() == 0;
    }

    public int getWidth() {
        if (this.isPlainTerminal()) {
            return 80;
        }
        return this.terminal.getWidth();
    }

    public int getHeight() {
        if (this.isPlainTerminal()) {
            return 30;
        }
        return this.terminal.getHeight();
    }

    public Executor getExecutor() {
        return this.executor;
    }

    public void open() {
        this.isRunning = true;
        this.terminal.writer().append(CliStrings.MESSAGE_WELCOME);
        while (this.isRunning) {
            String line;
            this.terminal.writer().append("\n");
            this.terminal.flush();
            try {
                line = this.lineReader.readLine(this.prompt, null, (MaskingCallback)null, null);
            }
            catch (UserInterruptException e) {
                continue;
            }
            catch (IOError | EndOfFileException e) {
                break;
            }
            catch (Throwable t) {
                throw new SqlClientException("Could not read from command line.", t);
            }
            if (line == null) continue;
            Optional<SqlCommandParser.SqlCommandCall> cmdCall = this.parseCommand(line);
            cmdCall.ifPresent(this::callCommand);
        }
    }

    public void close() {
        try {
            this.terminal.close();
        }
        catch (IOException e) {
            throw new SqlClientException("Unable to close terminal.", e);
        }
    }

    public boolean submitUpdate(String statement) {
        this.terminal.writer().println(CliStrings.messageInfo("Executing the following statement:").toAnsi());
        this.terminal.writer().println(new AttributedString(statement).toString());
        this.terminal.flush();
        Optional<SqlCommandParser.SqlCommandCall> parsedStatement = this.parseCommand(statement);
        return parsedStatement.map(cmdCall -> {
            switch (cmdCall.command) {
                case INSERT_INTO: 
                case INSERT_OVERWRITE: {
                    return this.callInsert((SqlCommandParser.SqlCommandCall)cmdCall);
                }
            }
            this.printError("Unsupported SQL statement.");
            return false;
        }).orElse(false);
    }

    private Optional<SqlCommandParser.SqlCommandCall> parseCommand(String line) {
        SqlCommandParser.SqlCommandCall parsedLine;
        try {
            parsedLine = SqlCommandParser.parse(this.executor.getSqlParser(this.sessionId), line);
        }
        catch (SqlExecutionException e) {
            this.printExecutionException(e);
            return Optional.empty();
        }
        return Optional.of(parsedLine);
    }

    private void callCommand(SqlCommandParser.SqlCommandCall cmdCall) {
        switch (cmdCall.command) {
            case QUIT: {
                this.callQuit();
                break;
            }
            case CLEAR: {
                this.callClear();
                break;
            }
            case RESET: {
                this.callReset();
                break;
            }
            case SET: {
                this.callSet(cmdCall);
                break;
            }
            case HELP: {
                this.callHelp();
                break;
            }
            case SHOW_CATALOGS: {
                this.callShowCatalogs();
                break;
            }
            case SHOW_DATABASES: {
                this.callShowDatabases();
                break;
            }
            case SHOW_TABLES: {
                this.callShowTables();
                break;
            }
            case SHOW_FUNCTIONS: {
                this.callShowFunctions();
                break;
            }
            case SHOW_MODULES: {
                this.callShowModules();
                break;
            }
            case USE_CATALOG: {
                this.callUseCatalog(cmdCall);
                break;
            }
            case USE: {
                this.callUseDatabase(cmdCall);
                break;
            }
            case DESC: 
            case DESCRIBE: {
                this.callDescribe(cmdCall);
                break;
            }
            case EXPLAIN: {
                this.callExplain(cmdCall);
                break;
            }
            case SELECT: {
                this.callSelect(cmdCall);
                break;
            }
            case INSERT_INTO: 
            case INSERT_OVERWRITE: {
                this.callInsert(cmdCall);
                break;
            }
            case CREATE_TABLE: {
                this.callDdl(cmdCall.operands[0], "Table has been created.");
                break;
            }
            case DROP_TABLE: {
                this.callDdl(cmdCall.operands[0], "Table has been removed.");
                break;
            }
            case CREATE_VIEW: {
                this.callDdl(cmdCall.operands[0], "View has been created.");
                break;
            }
            case DROP_VIEW: {
                this.callDdl(cmdCall.operands[0], "View has been removed.");
                break;
            }
            case CREATE_FUNCTION: {
                this.callDdl(cmdCall.operands[0], "Function has been created.");
                break;
            }
            case DROP_FUNCTION: {
                this.callDdl(cmdCall.operands[0], "Function has been removed.");
                break;
            }
            case ALTER_FUNCTION: {
                this.callDdl(cmdCall.operands[0], "Alter function succeeded!", "Alter function failed!");
                break;
            }
            case SOURCE: {
                this.callSource(cmdCall);
                break;
            }
            case CREATE_DATABASE: {
                this.callDdl(cmdCall.operands[0], "Database has been created.");
                break;
            }
            case DROP_DATABASE: {
                this.callDdl(cmdCall.operands[0], "Database has been removed.");
                break;
            }
            case ALTER_DATABASE: {
                this.callDdl(cmdCall.operands[0], "Alter database succeeded!", "Alter database failed!");
                break;
            }
            case ALTER_TABLE: {
                this.callDdl(cmdCall.operands[0], "Alter table succeeded!", "Alter table failed!");
                break;
            }
            case CREATE_CATALOG: {
                this.callDdl(cmdCall.operands[0], "Catalog has been created.");
                break;
            }
            case DROP_CATALOG: {
                this.callDdl(cmdCall.operands[0], "Catalog has been removed.");
                break;
            }
            default: {
                throw new SqlClientException("Unsupported command: " + (Object)((Object)cmdCall.command));
            }
        }
    }

    private void callQuit() {
        this.printInfo("Exiting Flink SQL CLI Client...");
        this.isRunning = false;
    }

    private void callClear() {
        this.clearTerminal();
    }

    private void callReset() {
        try {
            this.executor.resetSessionProperties(this.sessionId);
        }
        catch (SqlExecutionException e) {
            this.printExecutionException(e);
            return;
        }
        this.printInfo("All session properties have been set to their default values.");
    }

    private void callSet(SqlCommandParser.SqlCommandCall cmdCall) {
        if (cmdCall.operands.length == 0) {
            Map<String, String> properties;
            try {
                properties = this.executor.getSessionProperties(this.sessionId);
            }
            catch (SqlExecutionException e2) {
                this.printExecutionException(e2);
                return;
            }
            if (properties.isEmpty()) {
                this.terminal.writer().println(CliStrings.messageInfo("Result was empty.").toAnsi());
            } else {
                properties.entrySet().stream().map(e -> (String)e.getKey() + "=" + (String)e.getValue()).sorted().forEach(p -> this.terminal.writer().println((String)p));
            }
        } else {
            try {
                this.executor.setSessionProperty(this.sessionId, cmdCall.operands[0], cmdCall.operands[1].trim());
            }
            catch (SqlExecutionException e3) {
                this.printExecutionException(e3);
                return;
            }
            this.terminal.writer().println(CliStrings.messageInfo("Session property has been set.").toAnsi());
        }
        this.terminal.flush();
    }

    private void callHelp() {
        this.terminal.writer().println(CliStrings.MESSAGE_HELP);
        this.terminal.flush();
    }

    private void callShowCatalogs() {
        List<String> catalogs;
        try {
            catalogs = this.executor.listCatalogs(this.sessionId);
        }
        catch (SqlExecutionException e) {
            this.printExecutionException(e);
            return;
        }
        if (catalogs.isEmpty()) {
            this.terminal.writer().println(CliStrings.messageInfo("Result was empty.").toAnsi());
        } else {
            catalogs.forEach(v -> this.terminal.writer().println((String)v));
        }
        this.terminal.flush();
    }

    private void callShowDatabases() {
        List<String> dbs;
        try {
            dbs = this.executor.listDatabases(this.sessionId);
        }
        catch (SqlExecutionException e) {
            this.printExecutionException(e);
            return;
        }
        if (dbs.isEmpty()) {
            this.terminal.writer().println(CliStrings.messageInfo("Result was empty.").toAnsi());
        } else {
            dbs.forEach(v -> this.terminal.writer().println((String)v));
        }
        this.terminal.flush();
    }

    private void callShowTables() {
        List<String> tables;
        try {
            tables = this.executor.listTables(this.sessionId);
        }
        catch (SqlExecutionException e) {
            this.printExecutionException(e);
            return;
        }
        if (tables.isEmpty()) {
            this.terminal.writer().println(CliStrings.messageInfo("Result was empty.").toAnsi());
        } else {
            tables.forEach(v -> this.terminal.writer().println((String)v));
        }
        this.terminal.flush();
    }

    private void callShowFunctions() {
        List<String> functions;
        try {
            functions = this.executor.listFunctions(this.sessionId);
        }
        catch (SqlExecutionException e) {
            this.printExecutionException(e);
            return;
        }
        if (functions.isEmpty()) {
            this.terminal.writer().println(CliStrings.messageInfo("Result was empty.").toAnsi());
        } else {
            Collections.sort(functions);
            functions.forEach(v -> this.terminal.writer().println((String)v));
        }
        this.terminal.flush();
    }

    private void callShowModules() {
        List<String> modules;
        try {
            modules = this.executor.listModules(this.sessionId);
        }
        catch (SqlExecutionException e) {
            this.printExecutionException(e);
            return;
        }
        if (modules.isEmpty()) {
            this.terminal.writer().println(CliStrings.messageInfo("Result was empty.").toAnsi());
        } else {
            modules.forEach(v -> this.terminal.writer().println((String)v));
        }
        this.terminal.flush();
    }

    private void callUseCatalog(SqlCommandParser.SqlCommandCall cmdCall) {
        try {
            this.executor.useCatalog(this.sessionId, cmdCall.operands[0]);
        }
        catch (SqlExecutionException e) {
            this.printExecutionException(e);
            return;
        }
        this.terminal.flush();
    }

    private void callUseDatabase(SqlCommandParser.SqlCommandCall cmdCall) {
        try {
            this.executor.useDatabase(this.sessionId, cmdCall.operands[0]);
        }
        catch (SqlExecutionException e) {
            this.printExecutionException(e);
            return;
        }
        this.terminal.flush();
    }

    private void callDescribe(SqlCommandParser.SqlCommandCall cmdCall) {
        TableSchema schema;
        try {
            schema = this.executor.getTableSchema(this.sessionId, cmdCall.operands[0]);
        }
        catch (SqlExecutionException e) {
            this.printExecutionException(e);
            return;
        }
        this.terminal.writer().println(schema.toString());
        this.terminal.flush();
    }

    private void callExplain(SqlCommandParser.SqlCommandCall cmdCall) {
        String explanation;
        try {
            TableResult tableResult = this.executor.executeSql(this.sessionId, cmdCall.operands[0]);
            explanation = ((Row)tableResult.collect().next()).getField(0).toString();
        }
        catch (SqlExecutionException e) {
            this.printExecutionException(e);
            return;
        }
        this.terminal.writer().println(explanation);
        this.terminal.flush();
    }

    private void callSelect(SqlCommandParser.SqlCommandCall cmdCall) {
        ResultDescriptor resultDesc;
        try {
            resultDesc = this.executor.executeQuery(this.sessionId, cmdCall.operands[0]);
        }
        catch (SqlExecutionException e) {
            this.printExecutionException(e);
            return;
        }
        if (resultDesc.isTableauMode()) {
            try (CliTableauResultView tableauResultView = new CliTableauResultView(this.terminal, this.executor, this.sessionId, resultDesc);){
                if (resultDesc.isMaterialized()) {
                    tableauResultView.displayBatchResults();
                } else {
                    tableauResultView.displayStreamResults();
                }
            }
            catch (SqlExecutionException e) {
                this.printExecutionException(e);
            }
        } else {
            CliResultView view = resultDesc.isMaterialized() ? new CliTableResultView(this, resultDesc) : new CliChangelogResultView(this, resultDesc);
            try {
                view.open();
                this.printInfo("Result retrieval cancelled.");
            }
            catch (SqlExecutionException e) {
                this.printExecutionException(e);
            }
        }
    }

    private boolean callInsert(SqlCommandParser.SqlCommandCall cmdCall) {
        this.printInfo("Submitting SQL update statement to the cluster...");
        try {
            ProgramTargetDescriptor programTarget = this.executor.executeUpdate(this.sessionId, cmdCall.operands[0]);
            this.terminal.writer().println(CliStrings.messageInfo("Table update statement has been successfully submitted to the cluster:").toAnsi());
            this.terminal.writer().println(programTarget.toString());
            this.terminal.flush();
        }
        catch (SqlExecutionException e) {
            this.printExecutionException(e);
            return false;
        }
        return true;
    }

    private void callSource(SqlCommandParser.SqlCommandCall cmdCall) {
        String stmt;
        String pathString = cmdCall.operands[0];
        try {
            Path path = Paths.get(pathString, new String[0]);
            byte[] encoded = Files.readAllBytes(path);
            stmt = new String(encoded, Charset.defaultCharset());
        }
        catch (IOException e) {
            this.printExecutionException(e);
            return;
        }
        if (stmt.length() > 50000) {
            this.printExecutionError("The given file exceeds the maximum number of characters.");
            return;
        }
        this.terminal.writer().println(CliStrings.messageInfo("Executing the following statement:").toAnsi());
        this.terminal.writer().println(new AttributedString(stmt).toString());
        this.terminal.flush();
        Optional<SqlCommandParser.SqlCommandCall> call = this.parseCommand(stmt);
        call.ifPresent(this::callCommand);
    }

    private void callDdl(String ddl, String successMessage) {
        this.callDdl(ddl, successMessage, null);
    }

    private void callDdl(String ddl, String successMessage, String errorMessage) {
        try {
            this.executor.executeSql(this.sessionId, ddl);
            this.printInfo(successMessage);
        }
        catch (SqlExecutionException e) {
            this.printExecutionException(errorMessage, e);
        }
    }

    private void printExecutionException(Throwable t) {
        this.printExecutionException(null, t);
    }

    private void printExecutionException(String message, Throwable t) {
        String finalMessage = message == null ? "Could not execute SQL statement." : "Could not execute SQL statement. " + message;
        this.printException(finalMessage, t);
    }

    private void printExecutionError(String message) {
        this.terminal.writer().println(CliStrings.messageError("Could not execute SQL statement.", message).toAnsi());
        this.terminal.flush();
    }

    private void printException(String message, Throwable t) {
        LOG.warn(message, t);
        this.terminal.writer().println(CliStrings.messageError(message, t).toAnsi());
        this.terminal.flush();
    }

    private void printError(String message) {
        this.terminal.writer().println(CliStrings.messageError(message).toAnsi());
        this.terminal.flush();
    }

    private void printInfo(String message) {
        this.terminal.writer().println(CliStrings.messageInfo(message).toAnsi());
        this.terminal.flush();
    }

    private static Terminal createDefaultTerminal() {
        try {
            return TerminalBuilder.builder().name("Flink SQL CLI Client").build();
        }
        catch (IOException e) {
            throw new SqlClientException("Error opening command line interface.", e);
        }
    }
}

