/** @fileoverview Auto-generated Externs files
 * @externs
 */


/**
 * @const
 * @suppress {duplicate|const} */
var createjs = {};


/**
 * The Graphics class exposes an easy to use API for generating vector drawing instructions and drawing them to a
 * specified context. Note that you can use Graphics without any dependency on the EaselJS framework by calling {{#crossLink "Graphics/draw"}}{{/crossLink}}
 * directly, or it can be used with the {{#crossLink "Shape"}}{{/crossLink}} object to draw vector graphics within the
 * context of an EaselJS display list.
 * 
 * There are two approaches to working with Graphics object: calling methods on a Graphics instance (the "Graphics API"), or
 * instantiating Graphics command objects and adding them to the graphics queue via {{#crossLink "Graphics/append"}}{{/crossLink}}.
 * The former abstracts the latter, simplifying beginning and ending paths, fills, and strokes.
 * 
 * var g = new createjs.Graphics();
 * g.setStrokeStyle(1);
 * g.beginStroke("#000000");
 * g.beginFill("red");
 * g.drawCircle(0,0,30);
 * 
 * All drawing methods in Graphics return the Graphics instance, so they can be chained together. For example,
 * the following line of code would generate the instructions to draw a rectangle with a red stroke and blue fill:
 * 
 * mycreatejs.Graphics.beginStroke("red").beginFill("blue").drawRect(20, 20, 100, 50);
 * 
 * Each graphics API call generates a command object (see below). The last command to be created can be accessed via
 * {{#crossLink "Graphics/command:property"}}{{/crossLink}}:
 * 
 * var fillCommand = mycreatejs.Graphics.beginFill("red").command;
 * // ... later, update the fill style/color:
 * fillCommand.style = "blue";
 * // or change it to a bitmap fill:
 * fillCommand.bitmap(myImage);
 * 
 * For more direct control of rendering, you can instantiate and append command objects to the graphics queue directly. In this case, you
 * need to manage path creation manually, and ensure that fill/stroke is applied to a defined path:
 * 
 * // start a new path. createjs.Graphics.beginCmd is a reusable BeginPath instance:
 * mycreatejs.Graphics.append(createjs.createjs.Graphics.beginCmd);
 * // we need to define the path before applying the fill:
 * var circle = new createjs.createjs.Graphics.Circle(0,0,30);
 * mycreatejs.Graphics.append(circle);
 * // fill the path we just defined:
 * var fill = new createjs.createjs.Graphics.Fill("red");
 * mycreatejs.Graphics.append(fill);
 * 
 * These approaches can be used together, for example to insert a custom command:
 * 
 * mycreatejs.Graphics.beginFill("red");
 * var customCommand = new CustomSpiralCommand(etc);
 * mycreatejs.Graphics.append(customCommand);
 * mycreatejs.Graphics.beginFill("blue");
 * mycreatejs.Graphics.drawCircle(0, 0, 30);
 * 
 * See {{#crossLink "Graphics/append"}}{{/crossLink}} for more info on creating custom commands.
 * 
 * <h4>Tiny API</h4>
 * The Graphics class also includes a "tiny API", which is one or two-letter methods that are shortcuts for all of the
 * Graphics methods. These methods are great for creating compact instructions, and is used by the Toolkit for CreateJS
 * to generate readable code. All tiny methods are marked as protected, so you can view them by enabling protected
 * descriptions in the docs.
 * 
 * <table>
 * <tr><td><b>Tiny</b></td><td><b>Method</b></td><td><b>Tiny</b></td><td><b>Method</b></td></tr>
 * <tr><td>mt</td><td>{{#crossLink "Graphics/moveTo"}}{{/crossLink}} </td>
 * <td>lt</td> <td>{{#crossLink "Graphics/lineTo"}}{{/crossLink}}</td></tr>
 * <tr><td>a/at</td><td>{{#crossLink "Graphics/arc"}}{{/crossLink}} / {{#crossLink "Graphics/arcTo"}}{{/crossLink}} </td>
 * <td>bt</td><td>{{#crossLink "Graphics/bezierCurveTo"}}{{/crossLink}} </td></tr>
 * <tr><td>qt</td><td>{{#crossLink "Graphics/quadraticCurveTo"}}{{/crossLink}} (also curveTo)</td>
 * <td>r</td><td>{{#crossLink "Graphics/rect"}}{{/crossLink}} </td></tr>
 * <tr><td>cp</td><td>{{#crossLink "Graphics/closePath"}}{{/crossLink}} </td>
 * <td>c</td><td>{{#crossLink "Graphics/clear"}}{{/crossLink}} </td></tr>
 * <tr><td>f</td><td>{{#crossLink "Graphics/beginFill"}}{{/crossLink}} </td>
 * <td>lf</td><td>{{#crossLink "Graphics/beginLinearGradientFill"}}{{/crossLink}} </td></tr>
 * <tr><td>rf</td><td>{{#crossLink "Graphics/beginRadialGradientFill"}}{{/crossLink}} </td>
 * <td>bf</td><td>{{#crossLink "Graphics/beginBitmapFill"}}{{/crossLink}} </td></tr>
 * <tr><td>ef</td><td>{{#crossLink "Graphics/endFill"}}{{/crossLink}} </td>
 * <td>ss</td><td>{{#crossLink "Graphics/setStrokeStyle"}}{{/crossLink}} </td></tr>
 * <tr><td>s</td><td>{{#crossLink "Graphics/beginStroke"}}{{/crossLink}} </td>
 * <td>ls</td><td>{{#crossLink "Graphics/beginLinearGradientStroke"}}{{/crossLink}} </td></tr>
 * <tr><td>rs</td><td>{{#crossLink "Graphics/beginRadialGradientStroke"}}{{/crossLink}} </td>
 * <td>bs</td><td>{{#crossLink "Graphics/beginBitmapStroke"}}{{/crossLink}} </td></tr>
 * <tr><td>es</td><td>{{#crossLink "Graphics/endStroke"}}{{/crossLink}} </td>
 * <td>dr</td><td>{{#crossLink "Graphics/drawRect"}}{{/crossLink}} </td></tr>
 * <tr><td>rr</td><td>{{#crossLink "Graphics/drawRoundRect"}}{{/crossLink}} </td>
 * <td>rc</td><td>{{#crossLink "Graphics/drawRoundRectComplex"}}{{/crossLink}} </td></tr>
 * <tr><td>dc</td><td>{{#crossLink "Graphics/drawCircle"}}{{/crossLink}} </td>
 * <td>de</td><td>{{#crossLink "Graphics/drawEllipse"}}{{/crossLink}} </td></tr>
 * <tr><td>dp</td><td>{{#crossLink "Graphics/drawPolyStar"}}{{/crossLink}} </td>
 * <td>p</td><td>{{#crossLink "Graphics/decodePath"}}{{/crossLink}} </td></tr>
 * </table>
 * 
 * Here is the above example, using the tiny API instead.
 * 
 * mycreatejs.Graphics.s("red").f("blue").r(20, 20, 100, 50);
 *
 * @see [Graphics]
 * @constructor
 */
createjs.Graphics = function () {}

    /**
     * Returns the graphics instructions array. Each entry is a graphics command object (ex. createjs.Graphics.Fill, createjs.Graphics.Rect)
     * Modifying the returned array directly is not recommended, and is likely to result in unexpected behaviour.
     * 
     * This property is mainly intended for introspection of the instructions (ex. for graphics export).
     *
     * @type {(Array|null)} 
     * @see [Graphics]
     */
createjs.Graphics.prototype.instructions;

    /**
     * Returns a CSS compatible color string based on the specified RGB numeric color values in the format
     * "rgba(255,255,255,1.0)", or if alpha is null then in the format "rgb(255,255,255)". For example,
     * 
     * createjs.createjs.Graphics.getRGB(50, 100, 150, 0.5);
     * // Returns "rgba(50,100,150,0.5)"
     * 
     * It also supports passing a single hex color value as the first param, and an optional alpha value as the second
     * param. For example,
     * 
     * createjs.createjs.Graphics.getRGB(0xFF00FF, 0.2);
     * // Returns "rgba(255,0,255,0.2)"
     *
     * @param {(number|null)} r The red component for the color, between 0 and 0xFF (255).
     * @param {(number|null)} g The green component for the color, between 0 and 0xFF (255).
     * @param {(number|null)} b The blue component for the color, between 0 and 0xFF (255).
     * @param {number=} opt_alpha The alpha component for the color where 0 is fully transparent and 1 is fully opaque.
     * @see [Graphics]
     * @returns {string} A CSS compatible color string based on the specified RGB numeric color values in the format "rgba(255,255,255,1.0)", or if alpha is null then in the format "rgb(255,255,255)".
     */
createjs.Graphics.getRGB = function (r, g, b, opt_alpha) { return ''; }

    /**
     * Returns a CSS compatible color string based on the specified HSL numeric color values in the format "hsla(360,100,100,1.0)",
     * or if alpha is null then in the format "hsl(360,100,100)".
     * 
     * createjs.createjs.Graphics.getHSL(150, 100, 70);
     * // Returns "hsl(150,100,70)"
     *
     * @param {(number|null)} hue The hue component for the color, between 0 and 360.
     * @param {(number|null)} saturation The saturation component for the color, between 0 and 100.
     * @param {(number|null)} lightness The lightness component for the color, between 0 and 100.
     * @param {number=} opt_alpha The alpha component for the color where 0 is fully transparent and 1 is fully opaque.
     * @see [Graphics]
     * @returns {string} A CSS compatible color string based on the specified HSL numeric color values in the format "hsla(360,100,100,1.0)", or if alpha is null then in the format "hsl(360,100,100)".
     */
createjs.Graphics.getHSL = function (hue, saturation, lightness, opt_alpha) { return ''; }

    /**
     * Stores all graphics commands so they won't be executed in future draws. Calling store() a second time adds to
     * the existing store. This also affects `drawAsPath()`.
     * 
     * This is useful in cases where you are creating vector graphics in an iterative manner (ex. generative art), so
     * that only new graphics need to be drawn (which can provide huge performance benefits), but you wish to retain all
     * of the vector instructions for later use (ex. scaling, modifying, or exporting).
     * 
     * Note that calling store() will force the active path (if any) to be ended in a manner similar to changing
     * the fill or stroke.
     * 
     * For example, consider a application where the user draws lines with the mouse. As each line segment (or collection of
     * segments) are added to a Shape, it can be rasterized using {{#crossLink "DisplayObject/updateCache"}}{{/crossLink}},
     * and then stored, so that it can be redrawn at a different scale when the application is resized, or exported to SVG.
     * 
     * // set up cache:
     * myShape.cache(0,0,500,500,scale);
     * 
     * // when the user drags, draw a new line:
     * myShape.graphics.moveTo(oldX,oldY).lineTo(newX,newY);
     * // then draw it into the existing cache:
     * myShape.updateCache("source-over");
     * // store the new line, so it isn't redrawn next time:
     * myShape.store();
     * 
     * // then, when the window resizes, we can re-render at a different scale:
     * // first, unstore all our lines:
     * myShape.unstore();
     * // then cache using the new scale:
     * myShape.cache(0,0,500,500,newScale);
     * // finally, store the existing commands again:
     * myShape.store();
     *
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.store = function () {  return null; }

    /**
     * Closes the current path, effectively drawing a line from the current drawing point to the first drawing point specified
     * since the fill or stroke was last set. A tiny API method "cp" also exists.
     *
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.closePath = function () {  return null; }

    /**
     * Begins a linear gradient fill defined by the line (x0, y0) to (x1, y1). This ends the current sub-path. For
     * example, the following code defines a black to white vertical gradient ranging from 20px to 120px, and draws a
     * square to display it:
     * 
     * mycreatejs.Graphics.beginLinearGradientFill(["#000","#FFF"], [0, 1], 0, 20, 0, 120).drawRect(20, 20, 120, 120);
     * 
     * A tiny API method "lf" also exists.
     *
     * @param {(Array|null)} colors An array of CSS compatible color values. For example, ["#F00","#00F"] would define a gradient drawing from red to blue.
     * @param {(Array|null)} ratios An array of gradient positions which correspond to the colors. For example, [0.1, 0.9] would draw the first color to 10% then interpolating to the second color at 90%.
     * @param {(number|null)} x0 The position of the first point defining the line that defines the gradient direction and size.
     * @param {(number|null)} y0 The position of the first point defining the line that defines the gradient direction and size.
     * @param {(number|null)} x1 The position of the second point defining the line that defines the gradient direction and size.
     * @param {(number|null)} y1 The position of the second point defining the line that defines the gradient direction and size.
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.beginLinearGradientFill = function (colors, ratios, x0, y0, x1, y1) {  return null; }

    /**
     * Begins a radial gradient stroke. This ends the current sub-path. For example, the following code defines a red to
     * blue radial gradient centered at (100, 100), with a radius of 50, and draws a rectangle to display it:
     * 
     * mycreatejs.Graphics.setStrokeStyle(10)
     * .beginRadialGradientStroke(["#F00","#00F"], [0, 1], 100, 100, 0, 100, 100, 50)
     * .drawRect(50, 90, 150, 110);
     * 
     * A tiny API method "rs" also exists.
     *
     * @param {(Array|null)} colors An array of CSS compatible color values. For example, ["#F00","#00F"] would define a gradient drawing from red to blue.
     * @param {(Array|null)} ratios An array of gradient positions which correspond to the colors. For example, [0.1, 0.9] would draw the first color to 10% then interpolating to the second color at 90%, then draw the second color to 100%.
     * @param {(number|null)} x0 Center position of the inner circle that defines the gradient.
     * @param {(number|null)} y0 Center position of the inner circle that defines the gradient.
     * @param {(number|null)} r0 Radius of the inner circle that defines the gradient.
     * @param {(number|null)} x1 Center position of the outer circle that defines the gradient.
     * @param {(number|null)} y1 Center position of the outer circle that defines the gradient.
     * @param {(number|null)} r1 Radius of the outer circle that defines the gradient.
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.beginRadialGradientStroke = function (colors, ratios, x0, y0, r0, x1, y1, r1) {  return null; }

    /**
     * Draws a rounded rectangle with different corner radii. Supports positive and negative corner radii. A tiny API
     * method "rc" also exists.
     *
     * @param {(number|null)} x The horizontal coordinate to draw the round rect.
     * @param {(number|null)} y The vertical coordinate to draw the round rect.
     * @param {(number|null)} w The width of the round rect.
     * @param {(number|null)} h The height of the round rect.
     * @param {(number|null)} radiusTL Top left corner radius.
     * @param {(number|null)} radiusTR Top right corner radius.
     * @param {(number|null)} radiusBR Bottom right corner radius.
     * @param {(number|null)} radiusBL Bottom left corner radius.
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.drawRoundRectComplex = function (x, y, w, h, radiusTL, radiusTR, radiusBR, radiusBL) {  return null; }

    /**
     * Decodes a compact encoded path string into a series of draw instructions.
     * This format is not intended to be human readable, and is meant for use by authoring tools.
     * The format uses a base64 character set, with each character representing 6 bits, to define a series of draw
     * commands.
     * 
     * Each command is comprised of a single "header" character followed by a variable number of alternating x and y
     * position values. Reading the header bits from left to right (most to least significant): bits 1 to 3 specify the
     * type of operation (0-moveTo, 1-lineTo, 2-quadraticCurveTo, 3-bezierCurveTo, 4-closePath, 5-7 unused). Bit 4
     * indicates whether position values use 12 bits (2 characters) or 18 bits (3 characters), with a one indicating the
     * latter. Bits 5 and 6 are currently unused.
     * 
     * Following the header is a series of 0 (closePath), 2 (moveTo, lineTo), 4 (quadraticCurveTo), or 6 (bezierCurveTo)
     * parameters. These parameters are alternating x/y positions represented by 2 or 3 characters (as indicated by the
     * 4th bit in the command char). These characters consist of a 1 bit sign (1 is negative, 0 is positive), followed
     * by an 11 (2 char) or 17 (3 char) bit integer value. All position values are in tenths of a pixel. Except in the
     * case of move operations which are absolute, this value is a delta from the previous x or y position (as
     * appropriate).
     * 
     * For example, the string "A3cAAMAu4AAA" represents a line starting at -150,0 and ending at 150,0.
     * <br />A - bits 000000. First 3 bits (000) indicate a moveTo operation. 4th bit (0) indicates 2 chars per
     * parameter.
     * <br />n0 - 110111011100. Absolute x position of -150.0px. First bit indicates a negative value, remaining bits
     * indicate 1500 tenths of a pixel.
     * <br />AA - 000000000000. Absolute y position of 0.
     * <br />I - 001100. First 3 bits (001) indicate a lineTo operation. 4th bit (1) indicates 3 chars per parameter.
     * <br />Au4 - 000000101110111000. An x delta of 300.0px, which is added to the previous x value of -150.0px to
     * provide an absolute position of +150.0px.
     * <br />AAA - 000000000000000000. A y delta value of 0.
     * 
     * A tiny API method "p" also exists.
     *
     * @param {string} str The path string to decode.
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.decodePath = function (str) {  return null; }

    /**
     * Draws a bezier curve from the current drawing point to (x, y) using the control points (cp1x, cp1y) and (cp2x,
     * cp2y). For detailed information, read the
     * <a href="http://www.whatwg.org/specs/web-apps/current-work/multipage/the-canvas-element.html#dom-context-2d-beziercurveto">
     * whatwg spec</a>. A tiny API method "bt" also exists.
     *
     * @param {(number|null)} cp1x 
     * @param {(number|null)} cp1y 
     * @param {(number|null)} cp2x 
     * @param {(number|null)} cp2y 
     * @param {(number|null)} x 
     * @param {(number|null)} y 
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.bezierCurveTo = function (cp1x, cp1y, cp2x, cp2y, x, y) {  return null; }

    /**
     * Begins a radial gradient fill. This ends the current sub-path. For example, the following code defines a red to
     * blue radial gradient centered at (100, 100), with a radius of 50, and draws a circle to display it:
     * 
     * mycreatejs.Graphics.beginRadialGradientFill(["#F00","#00F"], [0, 1], 100, 100, 0, 100, 100, 50).drawCircle(100, 100, 50);
     * 
     * A tiny API method "rf" also exists.
     *
     * @param {(Array|null)} colors An array of CSS compatible color values. For example, ["#F00","#00F"] would define a gradient drawing from red to blue.
     * @param {(Array|null)} ratios An array of gradient positions which correspond to the colors. For example, [0.1, 0.9] would draw the first color to 10% then interpolating to the second color at 90%.
     * @param {(number|null)} x0 Center position of the inner circle that defines the gradient.
     * @param {(number|null)} y0 Center position of the inner circle that defines the gradient.
     * @param {(number|null)} r0 Radius of the inner circle that defines the gradient.
     * @param {(number|null)} x1 Center position of the outer circle that defines the gradient.
     * @param {(number|null)} y1 Center position of the outer circle that defines the gradient.
     * @param {(number|null)} r1 Radius of the outer circle that defines the gradient.
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.beginRadialGradientFill = function (colors, ratios, x0, y0, r0, x1, y1, r1) {  return null; }

    /**
     * Draws the display object into the specified context ignoring its visible, alpha, shadow, and transform.
     * Returns true if the draw was handled (useful for overriding functionality).
     * 
     * NOTE: This method is mainly for internal use, though it may be useful for advanced uses.
     *
     * @param {(CanvasRenderingContext2D|null)} ctx The canvas 2D context object to draw into.
     * @param {(Object|null)} data Optional data that is passed to graphics command exec methods. When called from a Shape instance, the shape passes itself as the data parameter. This can be used by custom graphic commands to insert contextual data.
     * @see [Graphics]
     */
createjs.Graphics.prototype.draw = function (ctx, data) {  }

    /**
     * Clears all drawing instructions, effectively resetting this Graphics instance. Any line and fill styles will need
     * to be redefined to draw shapes following a clear call. A tiny API method "c" also exists.
     *
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.clear = function () {  return null; }

    /**
     * Returns true if this Graphics instance has no drawing commands.
     *
     * @see [Graphics]
     * @returns {(boolean|null)} Returns true if this Graphics instance has no drawing commands.
     */
createjs.Graphics.prototype.isEmpty = function () {  return null; }

    /**
     * Draws an arc defined by the radius, startAngle and endAngle arguments, centered at the position (x, y). For
     * example, to draw a full circle with a radius of 20 centered at (100, 100):
     * 
     * arc(100, 100, 20, 0, Math.PI*2);
     * 
     * For detailed information, read the
     * <a href="http://www.whatwg.org/specs/web-apps/current-work/multipage/the-canvas-element.html#dom-context-2d-arc">whatwg spec</a>.
     * A tiny API method "a" also exists.
     *
     * @param {(number|null)} x 
     * @param {(number|null)} y 
     * @param {(number|null)} radius 
     * @param {(number|null)} startAngle Measured in radians.
     * @param {(number|null)} endAngle Measured in radians.
     * @param {(boolean|null)} anticlockwise 
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.arc = function (x, y, radius, startAngle, endAngle, anticlockwise) {  return null; }

    /**
     * Draws a star if pointSize is greater than 0, or a regular polygon if pointSize is 0 with the specified number of
     * points. For example, the following code will draw a familiar 5 pointed star shape centered at 100, 100 and with a
     * radius of 50:
     * 
     * mycreatejs.Graphics.beginFill("#FF0").drawPolyStar(100, 100, 50, 5, 0.6, -90);
     * // Note: -90 makes the first point vertical
     * 
     * A tiny API method "dp" also exists.
     *
     * @param {(number|null)} x Position of the center of the shape.
     * @param {(number|null)} y Position of the center of the shape.
     * @param {(number|null)} radius The outer radius of the shape.
     * @param {(number|null)} sides The number of points on the star or sides on the polygon.
     * @param {(number|null)} pointSize The depth or "pointy-ness" of the star points. A pointSize of 0 will draw a regular polygon (no points), a pointSize of 1 will draw nothing because the points are infinitely pointy.
     * @param {(number|null)} angle The angle of the first point / corner. For example a value of 0 will draw the first point directly to the right of the center.
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.drawPolyStar = function (x, y, radius, sides, pointSize, angle) {  return null; }

    /**
     * @param commit 
     * @see [Graphics]
     */
createjs.Graphics.prototype._updateInstructions = function (commit) {  }

    /**
     * @param fill 
     * @see [Graphics]
     */
createjs.Graphics.prototype._setFill = function (fill) {  }

    /**
     * Begins a pattern fill using the specified image. This ends the current sub-path. A tiny API method "bf" also
     * exists.
     *
     * @param {(HTMLCanvasElement|HTMLImageElement|HTMLVideoElement|null)} image The Image, Canvas, or Video object to use as the pattern.
     * @param {string} repetition Optional. Indicates whether to repeat the image in the fill area. One of "repeat", "repeat-x", "repeat-y", or "no-repeat". Defaults to "repeat". Note that Firefox does not support "repeat-x" or "repeat-y" (latest tests were in FF 20.0), and will default to "repeat".
     * @param {(createjs.Matrix2D|null)} matrix Optional. Specifies a transformation matrix for the bitmap fill. This transformation will be applied relative to the parent transform.
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.beginBitmapFill = function (image, repetition, matrix) {  return null; }

    /**
     * Moves the drawing point to the specified position. A tiny API method "mt" also exists.
     *
     * @param {(number|null)} x The x coordinate the drawing point should move to.
     * @param {(number|null)} y The y coordinate the drawing point should move to.
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls).
     */
createjs.Graphics.prototype.moveTo = function (x, y) {  return null; }

    /**
     * Returns a clone of this Graphics instance. Note that the individual command objects are not cloned.
     *
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} A clone of the current Graphics instance.
     */
createjs.Graphics.prototype.clone = function () {  return null; }

    /**
     * Appends a graphics command object to the graphics queue. Command objects expose an "exec" method
     * that accepts two parameters: the Context2D to operate on, and an arbitrary data object passed into
     * {{#crossLink "Graphics/draw"}}{{/crossLink}}. The latter will usually be the Shape instance that called draw.
     * 
     * This method is used internally by Graphics methods, such as drawCircle, but can also be used directly to insert
     * built-in or custom graphics commands. For example:
     * 
     * // attach data to our shape, so we can access it during the draw:
     * myShape.color = "red";
     * 
     * // append a Circle command object:
     * myShape.graphics.append(new createjs.Graphics.Circle(50, 50, 30));
     * 
     * // append a custom command object with an exec method that sets the fill style
     * // based on the shape's data, and then fills the circle.
     * myShape.graphics.append({exec:function(ctx, shape) {
     * ctx.fillStyle = shape.color;
     * ctx.fill();
     * }});
     *
     * @param {(Object|null)} command A graphics command object exposing an "exec" method.
     * @param {boolean} clean The clean param is primarily for internal use. A value of true indicates that a command does not generate a path that should be stroked or filled.
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.append = function (command, clean) {  return null; }

    /**
     * Begins a linear gradient stroke defined by the line (x0, y0) to (x1, y1). This ends the current sub-path. For
     * example, the following code defines a black to white vertical gradient ranging from 20px to 120px, and draws a
     * square to display it:
     * 
     * mycreatejs.Graphics.setStrokeStyle(10).
     * beginLinearGradientStroke(["#000","#FFF"], [0, 1], 0, 20, 0, 120).drawRect(20, 20, 120, 120);
     * 
     * A tiny API method "ls" also exists.
     *
     * @param {(Array|null)} colors An array of CSS compatible color values. For example, ["#F00","#00F"] would define a gradient drawing from red to blue.
     * @param {(Array|null)} ratios An array of gradient positions which correspond to the colors. For example, [0.1, 0.9] would draw the first color to 10% then interpolating to the second color at 90%.
     * @param {(number|null)} x0 The position of the first point defining the line that defines the gradient direction and size.
     * @param {(number|null)} y0 The position of the first point defining the line that defines the gradient direction and size.
     * @param {(number|null)} x1 The position of the second point defining the line that defines the gradient direction and size.
     * @param {(number|null)} y1 The position of the second point defining the line that defines the gradient direction and size.
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.beginLinearGradientStroke = function (colors, ratios, x0, y0, x1, y1) {  return null; }

    /**
     * Ends the current sub-path, and begins a new one with no fill. Functionally identical to <code>beginFill(null)</code>.
     * A tiny API method "ef" also exists.
     *
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.endFill = function () {  return null; }

    /**
     * Draws a line from the current drawing point to the specified position, which become the new current drawing
     * point. A tiny API method "lt" also exists.
     * 
     * For detailed information, read the
     * <a href="http://www.whatwg.org/specs/web-apps/current-work/multipage/the-canvas-element.html#complex-shapes-(paths)">
     * whatwg spec</a>.
     *
     * @param {(number|null)} x The x coordinate the drawing point should draw to.
     * @param {(number|null)} y The y coordinate the drawing point should draw to.
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.lineTo = function (x, y) {  return null; }

    /**
     * Draws only the path described for this Graphics instance, skipping any non-path instructions, including fill and
     * stroke descriptions. Used for <code>DisplayObject.mask</code> to draw the clipping path, for example.
     * 
     * NOTE: This method is mainly for internal use, though it may be useful for advanced uses.
     *
     * @param {(CanvasRenderingContext2D|null)} ctx The canvas 2D context object to draw into.
     * @see [Graphics]
     */
createjs.Graphics.prototype.drawAsPath = function (ctx) {  }

    /**
     * Begins a pattern fill using the specified image. This ends the current sub-path. Note that unlike bitmap fills,
     * strokes do not currently support a matrix parameter due to limitations in the canvas API. A tiny API method "bs"
     * also exists.
     *
     * @param {(HTMLCanvasElement|HTMLImageElement|HTMLVideoElement|null)} image The Image, Canvas, or Video object to use as the pattern.
     * @param {string=} opt_repetition Optional. Indicates whether to repeat the image in the fill area. One of "repeat", "repeat-x", "repeat-y", or "no-repeat". Defaults to "repeat".
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.beginBitmapStroke = function (image, opt_repetition) {  return null; }

    /**
     * Draws an ellipse (oval) with a specified width (w) and height (h). Similar to {{#crossLink "Graphics/drawCircle"}}{{/crossLink}},
     * except the width and height can be different. A tiny API method "de" also exists.
     *
     * @param {(number|null)} x The left coordinate point of the ellipse. Note that this is different from {{#crossLink "Graphics/drawCircle"}}{{/crossLink}} which draws from center.
     * @param {(number|null)} y The top coordinate point of the ellipse. Note that this is different from {{#crossLink "Graphics/drawCircle"}}{{/crossLink}} which draws from the center.
     * @param {(number|null)} w The height (horizontal diameter) of the ellipse. The horizontal radius will be half of this number.
     * @param {(number|null)} h The width (vertical diameter) of the ellipse. The vertical radius will be half of this number.
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.drawEllipse = function (x, y, w, h) {  return null; }

    /**
     * Draws a circle with the specified radius at (x, y).
     * 
     * var g = new createjs.Graphics();
     * g.setStrokeStyle(1);
     * g.beginStroke(createjs.createjs.Graphics.getRGB(0,0,0));
     * g.beginFill(createjs.createjs.Graphics.getRGB(255,0,0));
     * g.drawCircle(0,0,3);
     * 
     * var s = new createjs.Shape(g);
     * createjs.Graphics.x = 100;
     * createjs.Graphics.y = 100;
     * 
     * stage.addChild(s);
     * stage.update();
     * 
     * A tiny API method "dc" also exists.
     *
     * @param {(number|null)} x x coordinate center point of circle.
     * @param {(number|null)} y y coordinate center point of circle.
     * @param {(number|null)} radius Radius of circle.
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.drawCircle = function (x, y, radius) {  return null; }

    /**
     * Draws a rectangle at (x, y) with the specified width and height using the current fill and/or stroke.
     * For detailed information, read the
     * <a href="http://www.whatwg.org/specs/web-apps/current-work/multipage/the-canvas-element.html#dom-context-2d-rect">
     * whatwg spec</a>. A tiny API method "r" also exists.
     *
     * @param {(number|null)} x 
     * @param {(number|null)} y 
     * @param {(number|null)} w Width of the rectangle
     * @param {(number|null)} h Height of the rectangle
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.rect = function (x, y, w, h) {  return null; }

    /**
     * Begins a stroke with the specified color. This ends the current sub-path. A tiny API method "s" also exists.
     *
     * @param {string} color A CSS compatible color value (ex. "#FF0000", "red", or "rgba(255,0,0,0.5)"). Setting to null will result in no stroke.
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.beginStroke = function (color) {  return null; }

    /**
     * Use the {{#crossLink "Graphics/instructions:property"}}{{/crossLink}} property instead.
     *
     * @see [Graphics]
     * @returns {(Array|null)} 
     */
createjs.Graphics.prototype.getInstructions = function () {  return null; }

    /**
     * Begins a fill with the specified color. This ends the current sub-path. A tiny API method "f" also exists.
     *
     * @param {string} color A CSS compatible color value (ex. "red", "#FF0000", or "rgba(255,0,0,0.5)"). Setting to null will result in no fill.
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.beginFill = function (color) {  return null; }

    /**
     * Unstores any graphics commands that were previously stored using {{#crossLink "Graphics/store"}}{{/crossLink}}
     * so that they will be executed in subsequent draw calls.
     *
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.unstore = function () {  return null; }

    /**
     * Draws a rounded rectangle with all corners with the specified radius.
     *
     * @param {(number|null)} x 
     * @param {(number|null)} y 
     * @param {(number|null)} w 
     * @param {(number|null)} h 
     * @param {(number|null)} radius Corner radius.
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.drawRoundRect = function (x, y, w, h, radius) {  return null; }

    /**
     * @param stroke 
     * @see [Graphics]
     */
createjs.Graphics.prototype._setStroke = function (stroke) {  }

    /**
     * Draws an arc with the specified control points and radius.  For detailed information, read the
     * <a href="http://www.whatwg.org/specs/web-apps/current-work/multipage/the-canvas-element.html#dom-context-2d-arcto">
     * whatwg spec</a>. A tiny API method "at" also exists.
     *
     * @param {(number|null)} x1 
     * @param {(number|null)} y1 
     * @param {(number|null)} x2 
     * @param {(number|null)} y2 
     * @param {(number|null)} radius 
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.arcTo = function (x1, y1, x2, y2, radius) {  return null; }

    /**
     * Returns a string representation of this object.
     *
     * @see [Graphics]
     * @returns {string} a string representation of the instance.
     */
createjs.Graphics.prototype.toString = function () { return ''; }

    /**
     * Sets the stroke style for the current sub-path. Like all drawing methods, this can be chained, so you can define
     * the stroke style and color in a single line of code like so:
     * 
     * mycreatejs.Graphics.setStrokeStyle(8,"round").beginStroke("#F00");
     * 
     * A tiny API method "ss" also exists.
     *
     * @param {(number|null)} thickness The width of the stroke.
     * @param {(number|string)=} opt_caps Indicates the type of caps to use at the end of lines. One of butt, round, or square. Defaults to "butt". Also accepts the values 0 (butt), 1 (round), and 2 (square) for use with the tiny API.
     * @param {(number|string)=} opt_joints Specifies the type of joints that should be used where two lines meet. One of bevel, round, or miter. Defaults to "miter". Also accepts the values 0 (miter), 1 (round), and 2 (bevel) for use with the tiny API.
     * @param {number=} opt_miterLimit If joints is set to "miter", then you can specify a miter limit ratio which controls at what point a mitered joint will be clipped.
     * @param {boolean=} opt_ignoreScale If true, the stroke will be drawn at the specified thickness regardless of active transformations.
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.setStrokeStyle = function (thickness, opt_caps, opt_joints, opt_miterLimit, opt_ignoreScale) {  return null; }

    /**
     * Ends the current sub-path, and begins a new one with no stroke. Functionally identical to <code>beginStroke(null)</code>.
     * A tiny API method "es" also exists.
     *
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.endStroke = function () {  return null; }

    /**
     * Draws a quadratic curve from the current drawing point to (x, y) using the control point (cpx, cpy). For detailed
     * information, read the <a href="http://www.whatwg.org/specs/web-apps/current-work/multipage/the-canvas-element.html#dom-context-2d-quadraticcurveto">
     * whatwg spec</a>. A tiny API method "qt" also exists.
     *
     * @param {(number|null)} cpx 
     * @param {(number|null)} cpy 
     * @param {(number|null)} x 
     * @param {(number|null)} y 
     * @see [Graphics]
     * @returns {(createjs.Graphics|null)} The Graphics instance the method is called on (useful for chaining calls.)
     */
createjs.Graphics.prototype.quadraticCurveTo = function (cpx, cpy, x, y) {  return null; }

