/*
 *
 *  Licensed to the Apache Software Foundation (ASF) under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */

package org.apache.flex.compiler.internal.units.requests;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Collection;
import java.util.Collections;
import java.util.Set;

import org.apache.commons.io.output.StringBuilderWriter;
import org.apache.flex.abc.ABCParser;
import org.apache.flex.abc.print.ABCDumpVisitor;
import org.apache.flex.compiler.internal.embedding.EmbedData;
import org.apache.flex.compiler.problems.ICompilerProblem;
import org.apache.flex.compiler.units.requests.IABCBytesRequestResult;

/**
 * Output wad containing a result of running the ABCGenerator.
 */
public class ABCBytesRequestResult implements IABCBytesRequestResult
{
    /**
     * This constructor is used when code generation completes normally.
     * 
     * @param bytes abc bytes generated by code generator
     * @param problems Collection of problems found by the code generator.
     * @param embeds embedded assets
     */
	public ABCBytesRequestResult(byte[] bytes, ICompilerProblem problems[], Set<EmbedData> embeds)
    {
        assert ((bytes != null) || (problems != null));
        this.bytes = bytes;
        this.problems = problems;
        this.embeds = embeds;
    }

    /**
     * This constructor is used when code generation completes normally with no problems
     * 
     * @param bytes abc bytes generated by code generator
     */
    public ABCBytesRequestResult(byte[] bytes)
    {
        assert bytes != null;
        this.bytes = bytes;
        this.problems = ZEROPROBLEMS;
        this.embeds = Collections.emptySet();
    }
    
    /**
     * This constructor is used when code generation has no byte code
     * 
     * @param problems Collection of problems found by the code generator.
     */
    public ABCBytesRequestResult(ICompilerProblem problems[])
    {
        assert problems != null;
        this.bytes = ZEROBYTES;
        this.problems = problems;
        this.embeds = Collections.emptySet();
    }
    
    /**
     * This constructor is used when code generation completes normally with no problems and no bytes
     */
    public ABCBytesRequestResult()
    {
        this.bytes = ZEROBYTES;
        this.problems = ZEROPROBLEMS;
        this.embeds = Collections.emptySet();
    }

    /**
     * @return abc data generated by ABCGenerator.
     */
    @Override
    public byte[] getABCBytes()
    {
        return bytes;
    }

    /**
     * Returns a list of problems generated by the ABCGenerator while attempting
     * to generate abc data. If the getBytes method of this class returns null,
     * this method should not return null.
     * 
     * @return Collection of compiler problems generated by ABCGenerator while
     * attempting to generate abc data.
     */
    @Override
    public ICompilerProblem[] getProblems()
    {
        return problems;
    }

    @Override
    public Collection<EmbedData> getEmbeds()
    {
        return embeds;
    }

    private final byte[] bytes;
    private final ICompilerProblem[] problems;
    private final Set<EmbedData> embeds;
    
    
    @Override
    public String toString()
    {
        StringBuilder sb = new StringBuilder();
        PrintWriter out = new PrintWriter(new StringBuilderWriter(sb));
        
        ABCParser parser = null;
        try
        {
            parser = new ABCParser(new ByteArrayInputStream(bytes));
        }
        catch (IOException e)
        {
        }
        parser.parseABC(new ABCDumpVisitor(out));
        
        return sb.toString();
    }
}
