/*
 * Decompiled with CFR 0.152.
 */
package org.apache.empire.db.validation;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.empire.data.DataType;
import org.apache.empire.db.DBColumn;
import org.apache.empire.db.DBDatabase;
import org.apache.empire.db.DBRelation;
import org.apache.empire.db.DBTable;
import org.apache.empire.db.DBTableColumn;
import org.apache.empire.db.DBView;
import org.apache.empire.db.validation.DBModelErrorHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DBModelChecker {
    private static final Logger log = LoggerFactory.getLogger(DBModelChecker.class);
    private final Map<String, DBTable> tableMap = new HashMap<String, DBTable>();
    private final DBDatabase remoteDb = new RemoteDatabase();

    public synchronized void checkModel(DBDatabase db, Connection conn, String dbSchema, DBModelErrorHandler handler) {
        try {
            DatabaseMetaData dbMeta = conn.getMetaData();
            this.collectTables(dbMeta, dbSchema, null);
            this.collectColumns(dbMeta, dbSchema, null);
            this.collectPrimaryKeys(dbMeta, dbSchema, null);
            this.collectForeignKeys(dbMeta, dbSchema, null);
            for (DBTable table : db.getTables()) {
                this.checkTable(table, handler);
            }
            for (DBView view : db.getViews()) {
                this.checkView(view, conn, handler);
            }
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }

    protected String getMetaCatalog(String dbSchema) {
        return null;
    }

    protected String getMetaSchemaPattern(String dbSchema) {
        return dbSchema;
    }

    protected boolean isSystemTable(String tableName, ResultSet tableMeta) {
        return tableName.indexOf(36) >= 0;
    }

    protected void collectTables(DatabaseMetaData dbMeta, String dbSchema, String tablePattern) throws SQLException {
        ResultSet dbTables = dbMeta.getTables(this.getMetaCatalog(dbSchema), this.getMetaSchemaPattern(dbSchema), null, new String[]{"TABLE", "VIEW"});
        int count = 0;
        while (dbTables.next()) {
            String tableName = dbTables.getString("TABLE_NAME");
            if (this.isSystemTable(tableName, dbTables)) {
                log.info("Ignoring system table " + tableName);
                continue;
            }
            this.addTable(tableName);
            ++count;
        }
        log.info("{} tables added for schema {}", (Object)count, (Object)dbSchema);
    }

    protected void collectColumns(DatabaseMetaData dbMeta, String dbSchema, String tablePattern) throws SQLException {
        ResultSet dbColumns = dbMeta.getColumns(this.getMetaCatalog(dbSchema), this.getMetaSchemaPattern(dbSchema), null, null);
        while (dbColumns.next()) {
            String tableName = dbColumns.getString("TABLE_NAME");
            DBTable t = this.getTable(tableName);
            if (t == null) {
                log.error("Table not found: {}", (Object)tableName);
                continue;
            }
            this.addColumn(t, dbColumns);
        }
    }

    protected void collectPrimaryKeys(DatabaseMetaData dbMeta, String dbSchema, String tablePattern) throws SQLException {
        for (DBTable t : this.getTables()) {
            ArrayList<String> pkCols = new ArrayList<String>();
            ResultSet primaryKeys = dbMeta.getPrimaryKeys(this.getMetaCatalog(dbSchema), this.getMetaSchemaPattern(dbSchema), t.getName());
            while (primaryKeys.next()) {
                pkCols.add(primaryKeys.getString("COLUMN_NAME"));
            }
            if (pkCols.size() <= 0) continue;
            DBColumn[] keys = new DBColumn[pkCols.size()];
            for (int i = 0; i < keys.length; ++i) {
                keys[i] = t.getColumn(((String)pkCols.get(i)).toUpperCase());
            }
            t.setPrimaryKey(keys);
        }
    }

    protected void collectForeignKeys(DatabaseMetaData dbMeta, String dbSchema, String tablePattern) throws SQLException {
        ResultSet foreignKeys = dbMeta.getImportedKeys(this.getMetaCatalog(dbSchema), this.getMetaSchemaPattern(dbSchema), tablePattern);
        while (foreignKeys.next()) {
            String fkTable = foreignKeys.getString("FKTABLE_NAME");
            String fkColumn = foreignKeys.getString("FKCOLUMN_NAME");
            String pkTable = foreignKeys.getString("PKTABLE_NAME");
            String pkColumn = foreignKeys.getString("PKCOLUMN_NAME");
            String fkName = foreignKeys.getString("FK_NAME");
            DBTableColumn c1 = (DBTableColumn)this.getTable(fkTable).getColumn(fkColumn.toUpperCase());
            DBTableColumn c2 = (DBTableColumn)this.getTable(pkTable).getColumn(pkColumn.toUpperCase());
            DBRelation relation = this.remoteDb.getRelation(fkName);
            if (relation == null) {
                this.addRelation(fkName, c1.referenceOn(c2));
                continue;
            }
            DBRelation.DBReference[] refs = relation.getReferences();
            this.remoteDb.removeRelation(relation);
            DBRelation.DBReference[] newRefs = new DBRelation.DBReference[refs.length + 1];
            DBRelation.DBReference newRef = new DBRelation.DBReference(c1, c2);
            for (int i = 0; i < refs.length; ++i) {
                newRefs[i] = refs[i];
            }
            newRefs[newRefs.length - 1] = newRef;
            this.addRelation(fkName, newRefs);
        }
    }

    protected void checkTable(DBTable table, DBModelErrorHandler handler) {
        DBTable remoteTable = this.getTable(table.getName());
        if (remoteTable == null) {
            handler.itemNotFound(table);
            return;
        }
        this.checkPrimaryKey(table, remoteTable, handler);
        this.checkForeignKeys(table, remoteTable, handler);
        for (DBColumn column : table.getColumns()) {
            DBColumn remoteColumn = remoteTable.getColumn(column.getName());
            if (remoteColumn == null) {
                handler.itemNotFound(column);
                continue;
            }
            this.checkColumn(column, remoteColumn, handler);
        }
    }

    protected void checkPrimaryKey(DBTable table, DBTable remoteTable, DBModelErrorHandler handler) {
        if (table.getPrimaryKey() == null) {
            return;
        }
        if (remoteTable.getPrimaryKey() == null) {
            handler.itemNotFound(table.getPrimaryKey());
            return;
        }
        DBColumn[] pk = table.getPrimaryKey().getColumns();
        DBColumn[] remotePk = remoteTable.getPrimaryKey().getColumns();
        block0: for (DBColumn pkCol : pk) {
            for (DBColumn remotePkCol : remotePk) {
                if (pkCol.getFullName().equalsIgnoreCase(remotePkCol.getFullName())) continue block0;
            }
            handler.primaryKeyColumnMissing(table.getPrimaryKey(), pkCol);
        }
    }

    protected void checkForeignKeys(DBTable table, DBTable remoteTable, DBModelErrorHandler handler) {
        if (table.getForeignKeyRelations().isEmpty()) {
            return;
        }
        List<DBRelation> relations = table.getForeignKeyRelations();
        List<DBRelation> remoteRelations = remoteTable.getForeignKeyRelations();
        block0: for (DBRelation relation : relations) {
            block1: for (DBRelation.DBReference reference : relation.getReferences()) {
                if (reference.getTargetColumn().getRowSet() instanceof DBTable) {
                    DBTable targetTable = (DBTable)reference.getTargetColumn().getRowSet();
                    DBTableColumn targetColumn = reference.getTargetColumn();
                    if (!targetTable.getPrimaryKey().contains(targetColumn)) {
                        log.info("The column " + targetColumn.getName() + " of foreign key {} is not a primary key of table {} and cant be checked because of a limitation in JDBC", (Object)relation.getName(), (Object)targetTable.getName());
                        continue;
                    }
                }
                for (DBRelation remoteRelation : remoteRelations) {
                    for (DBRelation.DBReference remoteReference : remoteRelation.getReferences()) {
                        if (reference.getSourceColumn().getFullName().equalsIgnoreCase(remoteReference.getSourceColumn().getFullName()) && reference.getTargetColumn().getFullName().equalsIgnoreCase(remoteReference.getTargetColumn().getFullName())) continue block1;
                    }
                }
                handler.itemNotFound(relation);
                continue block0;
            }
        }
    }

    protected void checkView(DBView view, Connection conn, DBModelErrorHandler handler) {
        DBTable remoteView = this.getTable(view.getName());
        if (remoteView == null) {
            handler.itemNotFound(view);
            return;
        }
        for (DBColumn column : view.getColumns()) {
            DBColumn remoteColumn = remoteView.getColumn(column.getName());
            if (remoteColumn == null) {
                handler.itemNotFound(column);
                continue;
            }
            this.checkColumnType(column, remoteColumn, handler);
        }
    }

    protected void checkColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        switch (column.getDataType()) {
            case UNKNOWN: {
                this.checkUnknownColumn(column, remoteColumn, handler);
                break;
            }
            case INTEGER: {
                this.checkIntegerColumn(column, remoteColumn, handler);
                break;
            }
            case AUTOINC: {
                this.checkAutoIncColumn(column, remoteColumn, handler);
                break;
            }
            case TEXT: 
            case VARCHAR: {
                this.checkTextColumn(column, remoteColumn, handler);
                break;
            }
            case DATE: 
            case DATETIME: 
            case TIMESTAMP: {
                this.checkDateColumn(column, remoteColumn, handler);
                break;
            }
            case CHAR: {
                this.checkCharColumn(column, remoteColumn, handler);
                break;
            }
            case FLOAT: {
                this.checkFloatColumn(column, remoteColumn, handler);
                break;
            }
            case DECIMAL: {
                this.checkDecimalColumn(column, remoteColumn, handler);
                break;
            }
            case BOOL: {
                this.checkBoolColumn(column, remoteColumn, handler);
                break;
            }
            case CLOB: {
                this.checkClobColumn(column, remoteColumn, handler);
                break;
            }
            case BLOB: {
                this.checkBlobColumn(column, remoteColumn, handler);
                break;
            }
            case UNIQUEID: {
                this.checkUniqueIdColumn(column, remoteColumn, handler);
                break;
            }
            default: {
                throw new RuntimeException("Invalid DataType " + (Object)((Object)column.getDataType()));
            }
        }
    }

    private void checkGenericColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkColumnType(column, remoteColumn, handler);
        this.checkColumnNullable(column, remoteColumn, handler);
        this.checkColumnSize(column, remoteColumn, handler);
    }

    protected void checkColumnType(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        if (column.getDataType() != remoteColumn.getDataType()) {
            handler.columnTypeMismatch(column, remoteColumn.getDataType());
        }
    }

    protected void checkColumnNullable(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        if (column.isRequired() && !remoteColumn.isRequired()) {
            handler.columnNullableMismatch(column, remoteColumn.isRequired());
        }
    }

    protected void checkColumnSize(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        if ((int)column.getSize() != (int)remoteColumn.getSize()) {
            handler.columnSizeMismatch(column, (int)remoteColumn.getSize(), 0);
        }
    }

    protected void checkUnknownColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkGenericColumn(column, remoteColumn, handler);
    }

    protected void checkIntegerColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkGenericColumn(column, remoteColumn, handler);
    }

    protected void checkAutoIncColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkColumnSize(column, remoteColumn, handler);
        this.checkColumnNullable(column, remoteColumn, handler);
    }

    protected void checkTextColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkGenericColumn(column, remoteColumn, handler);
    }

    protected void checkDateColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkColumnNullable(column, remoteColumn, handler);
        if (!remoteColumn.getDataType().isDate()) {
            handler.columnTypeMismatch(column, remoteColumn.getDataType());
        }
    }

    protected void checkCharColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkGenericColumn(column, remoteColumn, handler);
    }

    protected void checkFloatColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkGenericColumn(column, remoteColumn, handler);
    }

    protected void checkDecimalColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkGenericColumn(column, remoteColumn, handler);
        if (column instanceof DBTableColumn && remoteColumn instanceof DBTableColumn) {
            DBTableColumn tableColumn = (DBTableColumn)column;
            DBTableColumn tableRemoteColumn = (DBTableColumn)remoteColumn;
            if (tableColumn.getDecimalScale() != tableRemoteColumn.getDecimalScale()) {
                handler.columnSizeMismatch(column, (int)remoteColumn.getSize(), tableRemoteColumn.getDecimalScale());
            }
        }
    }

    protected void checkBoolColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkColumnType(column, remoteColumn, handler);
        this.checkColumnNullable(column, remoteColumn, handler);
    }

    protected void checkBlobColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkColumnType(column, remoteColumn, handler);
        this.checkColumnNullable(column, remoteColumn, handler);
    }

    protected void checkClobColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkColumnType(column, remoteColumn, handler);
        this.checkColumnNullable(column, remoteColumn, handler);
    }

    protected void checkUniqueIdColumn(DBColumn column, DBColumn remoteColumn, DBModelErrorHandler handler) {
        this.checkGenericColumn(column, remoteColumn, handler);
    }

    protected final Collection<DBTable> getTables() {
        return this.tableMap.values();
    }

    protected final DBTable getTable(String tableName) {
        return this.tableMap.get(tableName.toUpperCase());
    }

    protected void addTable(String tableName) {
        this.tableMap.put(tableName.toUpperCase(), new DBTable(tableName, this.remoteDb));
    }

    protected void addRelation(String relName, DBRelation.DBReference ... references) {
        this.remoteDb.addRelation(relName, references);
    }

    protected DBTableColumn addColumn(DBTable t, ResultSet rs) throws SQLException {
        String name = rs.getString("COLUMN_NAME");
        DataType empireType = this.getEmpireDataType(rs.getInt("DATA_TYPE"));
        double colSize = rs.getInt("COLUMN_SIZE");
        if (empireType == DataType.DECIMAL || empireType == DataType.FLOAT) {
            int decimalDig = rs.getInt("DECIMAL_DIGITS");
            if (decimalDig > 0) {
                try {
                    int intSize = rs.getInt("COLUMN_SIZE");
                    colSize = Double.parseDouble(String.valueOf(intSize) + '.' + decimalDig);
                }
                catch (Exception e) {
                    log.error("Failed to parse decimal digits for column " + name);
                }
            }
            if (colSize < 1.0) {
                empireType = DataType.INTEGER;
            }
        } else if (empireType == DataType.INTEGER || empireType == DataType.CLOB || empireType == DataType.BLOB) {
            colSize = 0.0;
        }
        boolean required = false;
        String defaultValue = rs.getString("COLUMN_DEF");
        if (rs.getString("IS_NULLABLE").equalsIgnoreCase("NO")) {
            required = true;
        }
        if (rs.getInt("DATA_TYPE") == 93 && defaultValue != null && defaultValue.equals("CURRENT_TIMESTAMP")) {
            required = false;
            defaultValue = null;
        }
        DataType originalType = empireType;
        ResultSetMetaData metaData = rs.getMetaData();
        int colCount = metaData.getColumnCount();
        for (int i = 1; i <= colCount; ++i) {
            String colName = metaData.getColumnName(i);
            if ((!colName.equalsIgnoreCase("IS_AUTOINCREMENT") || !rs.getString(i).equalsIgnoreCase("YES")) && (!colName.equals("TYPE_NAME") || !rs.getString(i).matches(".*(?i:identity).*"))) continue;
            empireType = DataType.AUTOINC;
        }
        DBTableColumn col = t.addColumn(name, empireType, colSize, required, (Object)defaultValue);
        if (empireType.equals((Object)DataType.AUTOINC)) {
            col.setAttribute("AutoIncDataType", (Object)originalType);
        }
        return col;
    }

    protected DataType getEmpireDataType(int sqlType) {
        DataType empireType = DataType.UNKNOWN;
        switch (sqlType) {
            case -6: 
            case -5: 
            case 4: 
            case 5: {
                empireType = DataType.INTEGER;
                break;
            }
            case -9: 
            case 12: {
                empireType = DataType.VARCHAR;
                break;
            }
            case 91: {
                empireType = DataType.DATE;
                break;
            }
            case 92: 
            case 93: {
                empireType = DataType.DATETIME;
                break;
            }
            case -15: 
            case 1: {
                empireType = DataType.CHAR;
                break;
            }
            case 6: 
            case 7: 
            case 8: {
                empireType = DataType.FLOAT;
                break;
            }
            case 2: 
            case 3: {
                empireType = DataType.DECIMAL;
                break;
            }
            case -7: 
            case 16: {
                empireType = DataType.BOOL;
                break;
            }
            case -16: 
            case -1: 
            case 2005: {
                empireType = DataType.CLOB;
                break;
            }
            case -4: 
            case -3: 
            case -2: 
            case 2004: {
                empireType = DataType.BLOB;
                break;
            }
            default: {
                empireType = DataType.UNKNOWN;
                log.warn("SQL column type " + sqlType + " not supported.");
            }
        }
        log.debug("Mapping date type " + String.valueOf(sqlType) + " to " + (Object)((Object)empireType));
        return empireType;
    }

    private static class RemoteDatabase
    extends DBDatabase {
        private static final long serialVersionUID = 1L;

        private RemoteDatabase() {
        }
    }
}

