// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.doris.nereids.trees.expressions;

import org.apache.doris.nereids.trees.TreeNode;
import org.apache.doris.nereids.trees.expressions.shape.LeafExpression;
import org.apache.doris.nereids.trees.expressions.visitor.ExpressionVisitor;

import com.google.common.collect.ImmutableList;

import java.util.List;
import java.util.Objects;

/**
 * This represents any expression, it means it equals any expression
 */
public class Any extends Expression implements LeafExpression {

    public static final Any INSTANCE = new Any(ImmutableList.of());

    private Any(Expression... children) {
        super(children);
    }

    private Any(List<Expression> children) {
        super(children);
    }

    @Override
    public <R, C> R accept(ExpressionVisitor<R, C> visitor, C context) {
        return visitor.visitAny(this, context);
    }

    @Override
    public boolean nullable() {
        return false;
    }

    @Override
    public boolean equals(Object o) {
        return true;
    }

    /**
     * Equals with direction
     * Since the equals method in Any is always true, that means Any is equals to others, but not equal in reverse.
     * The expression with Any should always be the first argument.
     */
    public static boolean equals(Expression expressionWithAny, Expression target) {
        return Objects.equals(expressionWithAny, target);
    }

    @Override
    public int hashCode() {
        return 0;
    }

    @Override
    public boolean deepEquals(TreeNode<?> that) {
        return true;
    }
}
