#include <random>
#include <fstream>

#include "CLucene/_ApiHeader.h"
#include "CLucene/analysis/standard95/StandardAnalyzer.h"
#include "test.h"

void testCut(const std::string &str, std::vector<std::string> &tokens) {
  auto standard =
      std::make_unique<lucene::analysis::standard95::StandardAnalyzer>();
  auto tokenizer =
      static_cast<lucene::analysis::standard95::StandardTokenizer *>(
          standard->tokenStream(L"name", nullptr));

  lucene::util::SStringReader<char> reader;
  reader.init(str.data(), str.size(), false);

  tokenizer->reset(&reader);
  CL_NS(analysis)::Token token;
  while (tokenizer->next(&token)) {
    std::string_view term(token.termBuffer<char>(), token.termLength<char>());
    tokens.emplace_back(term);
  }

  delete tokenizer;
}

void testCutLines(std::vector<std::string>& datas, std::vector<std::string> &tokens) {
  auto standard =
      std::make_unique<lucene::analysis::standard95::StandardAnalyzer>();
  standard->useStopWords(false);
  auto tokenizer =
      static_cast<lucene::analysis::standard95::StandardTokenizer *>(
          standard->tokenStream(L"name", nullptr));

  for (auto& data : datas) {
    lucene::util::SStringReader<char> reader;
    reader.init(data.data(), data.size(), false);

    tokenizer->reset(&reader);
    std::string str;
    CL_NS(analysis)::Token token;
    while (tokenizer->next(&token)) {
      std::string_view term(token.termBuffer<char>(), token.termLength<char>());
      str += term;
      str += "|";
    }
    tokens.emplace_back(str);
  }

  delete tokenizer;
}

void testStandardTokenizer(CuTest *tc) {
  std::string str = "Wha\u0301t's this thing do?";
  std::vector<std::string> tokens;
  testCut(str, tokens);

  CLUCENE_ASSERT(tokens[0] == "wha\u0301t's");
  CLUCENE_ASSERT(tokens[1] == "thing");
  CLUCENE_ASSERT(tokens[2] == "do");
}

void testStandardTokenizerMaxTokenLength(CuTest *tc) {
  std::string str = "one two three ";
  std::string longWord;
  for (int i = 0; i < 51; ++i) {
    longWord.append("abcde");
  }
  str += longWord;
  str += " four five six";
  std::vector<std::string> tokens;
  testCut(str, tokens);

  CLUCENE_ASSERT(tokens[0] == "one");
  CLUCENE_ASSERT(tokens[1] == "two");
  CLUCENE_ASSERT(tokens[2] == "three");
  CLUCENE_ASSERT(tokens[3] == longWord);
  CLUCENE_ASSERT(tokens[4] == "four");
  CLUCENE_ASSERT(tokens[5] == "five");
  CLUCENE_ASSERT(tokens[6] == "six");
}

void testStandardTokenizerMaxTokenLength1(CuTest *tc) {
  std::string str = "one two three ";
  std::string longWord;
  for (int i = 0; i < 51; ++i) {
    longWord.append("abcde");
  }
  str += longWord;
  str += "abcde";
  str += " four five six";
  std::vector<std::string> tokens;
  testCut(str, tokens);

  CLUCENE_ASSERT(tokens[0] == "one");
  CLUCENE_ASSERT(tokens[1] == "two");
  CLUCENE_ASSERT(tokens[2] == "three");
  CLUCENE_ASSERT(tokens[3] == longWord);
  CLUCENE_ASSERT(tokens[4] == "abcde");
  CLUCENE_ASSERT(tokens[5] == "four");
  CLUCENE_ASSERT(tokens[6] == "five");
  CLUCENE_ASSERT(tokens[7] == "six");
}

void testUtf8Str(CuTest *tc) {
  std::string str =
      "昨天我参加了一个编程比赛,遇到了一位技术很棒的朋友。His name is David, "
      "他今年28岁,电话号码是13566668888。David来自美国,他的邮箱是david@gmail."
      "com。编程比赛持续了2个小时,最后David赢得了冠军。虽然我只得了第三名,"
      "但我学到了很多编程技巧。David给我推荐了一个学习编程的网站 "
      "www.codingforeveryone.com,"
      "这个网站包含许多免费的在线课程。我们都觉得这个网站很有用,"
      "可以提高编程技能。After the contest, David and I exchanged contacts. "
      "我给David我的身份证号码1234891987654321作为联系方式。我相信通过我的努力,"
      "我的编程水平一定会越来越好。这次参加比赛让我结交了一位知音好友,"
      "也激励了我不断学习进步。";
  std::vector<std::string> tokens;
  testCut(str, tokens);

  std::string old_str;
  for (auto &token : tokens) {
    old_str += token;
    old_str += "|";
  }

  std::string new_str =
      "昨|天|我|参|加|了|一|个|编|程|比|赛|遇|到|了|一|位|技|术|很|棒|的|朋|友|"
      "his|name|david|他|今|年|28|岁|电|话|号|码|是|13566668888|david|来|自|美|"
      "国|他|的|邮|箱|是|david|gmail.com|编|程|比|赛|持|续|了|2|个|小|时|最|后|"
      "david|赢|得|了|冠|军|虽|然|我|只|得|了|第|三|名|但|我|学|到|了|很|多|编|"
      "程|技|巧|david|给|我|推|荐|了|一|个|学|习|编|程|的|网|站|www."
      "codingforeveryone.com|这|个|网|站|包|含|许|多|免|费|的|在|线|课|程|我|"
      "们|都|觉|得|这|个|网|站|很|有|用|可|以|提|高|编|程|技|能|after|contest|"
      "david|i|exchanged|contacts|我|给|david|我|的|身|份|证|号|码|"
      "1234891987654321|作|为|联|系|方|式|我|相|信|通|过|我|的|努|力|我|的|编|"
      "程|水|平|一|定|会|越|来|越|好|这|次|参|加|比|赛|让|我|结|交|了|一|位|知|"
      "音|好|友|也|激|励|了|我|不|断|学|习|进|步|";

  CLUCENE_ASSERT(old_str == new_str);
}

void testArmenian(CuTest *tc) {
  std::string str =
      "Վիքիպեդիայի 13 միլիոն հոդվածները (4,600` հայերեն վիքիպեդիայում) գրվել "
      "են կամավորների կողմից ու համարյա բոլոր հոդվածները կարող է խմբագրել "
      "ցանկաց մարդ ով կարող է բացել Վիքիպեդիայի կայքը։";
  std::vector<std::string> tokens;
  testCut(str, tokens);

  std::vector<std::string> new_tokens = {
      "Վիքիպեդիայի",   "13",     "միլիոն",
      "հոդվածները",    "4,600",  "հայերեն",
      "վիքիպեդիայում", "գրվել",  "են",
      "կամավորների",   "կողմից", "ու",
      "համարյա",       "բոլոր",  "հոդվածները",
      "կարող",         "է",      "խմբագրել",
      "ցանկաց",        "մարդ",   "ով",
      "կարող",         "է",      "բացել",
      "Վիքիպեդիայի",   "կայքը"};

  CLUCENE_ASSERT((tokens == new_tokens));
}

void testAmharic(CuTest *tc) {
  std::string str =
      "ዊኪፔድያ የባለ ብዙ ቋንቋ የተሟላ ትክክለኛና ነጻ መዝገበ ዕውቀት (ኢንሳይክሎፒዲያ) ነው። ማንኛውም";
  std::vector<std::string> tokens;
  testCut(str, tokens);

  std::vector<std::string> new_tokens = {"ዊኪፔድያ", "የባለ",       "ብዙ", "ቋንቋ",
                                         "የተሟላ",  "ትክክለኛና",    "ነጻ", "መዝገበ",
                                         "ዕውቀት",  "ኢንሳይክሎፒዲያ", "ነው", "ማንኛውም"};

  CLUCENE_ASSERT((tokens == new_tokens));
}

void testBengali(CuTest *tc) {
  std::string str =
      "এই বিশ্বকোষ পরিচালনা করে উইকিমিডিয়া ফাউন্ডেশন (একটি অলাভজনক সংস্থা)। "
      "উইকিপিডিয়ার শুরু ১৫ জানুয়ারি, ২০০১ সালে। এখন পর্যন্ত ২০০টিরও বেশী ভাষায় "
      "উইকিপিডিয়া রয়েছে।";
  std::vector<std::string> tokens;
  testCut(str, tokens);

  std::vector<std::string> new_tokens = {
      "এই",         "বিশ্বকোষ", "পরিচালনা", "করে",   "উইকিমিডিয়া",
      "ফাউন্ডেশন",   "একটি",    "অলাভজনক",  "সংস্থা", "উইকিপিডিয়ার",
      "শুরু",         "১৫",      "জানুয়ারি",  "২০০১",  "সালে",
      "এখন",        "পর্যন্ত",   "২০০টিরও",  "বেশী",  "ভাষায়",
      "উইকিপিডিয়া", "রয়েছে"};

  CLUCENE_ASSERT((tokens == new_tokens));
}

void testFarsi(CuTest *tc) {
  std::string str =
      "ویکی پدیای انگلیسی در تاریخ ۲۵ دی ۱۳۷۹ به صورت مکملی برای دانشنامهٔ "
      "تخصصی نوپدیا نوشته شد.";
  std::vector<std::string> tokens;
  testCut(str, tokens);

  std::vector<std::string> new_tokens = {
      "ویکی",     "پدیای", "انگلیسی", "در",    "تاریخ", "۲۵",
      "دی",       "۱۳۷۹",  "به",      "صورت",  "مکملی", "برای",
      "دانشنامهٔ", "تخصصی", "نوپدیا",  "نوشته", "شد"};

  CLUCENE_ASSERT((tokens == new_tokens));
}

void testGreek(CuTest *tc) {
  std::string str =
      "Γράφεται σε συνεργασία από εθελοντές με το λογισμικό wiki, κάτι που "
      "σημαίνει ότι άρθρα μπορεί να προστεθούν ή να αλλάξουν από τον καθένα.";
  std::vector<std::string> tokens;
  testCut(str, tokens);

  std::vector<std::string> new_tokens = {
      "Γράφεται", "σε",        "συνεργασία", "από",  "εθελοντές",  "με",
      "το",       "λογισμικό", "wiki",       "κάτι", "που",        "σημαίνει",
      "ότι",      "άρθρα",     "μπορεί",     "να",   "προστεθούν", "ή",
      "να",       "αλλάξουν",  "από",        "τον",  "καθένα"};

  CLUCENE_ASSERT((tokens == new_tokens));
}

void testThai(CuTest *tc) {
  std::string str = "การที่ได้ต้องแสดงว่างานดี. แล้วเธอจะไปไหน? ๑๒๓๔";
  std::vector<std::string> tokens;
  testCut(str, tokens);

  std::vector<std::string> new_tokens = {"การที่ได้ต้องแสดงว่างานดี", "แล้วเธอจะไปไหน",
                                         "๑๒๓๔"};

  CLUCENE_ASSERT((tokens == new_tokens));
}

void testLao(CuTest *tc) {
  std::string str = "ສາທາລະນະລັດ ປະຊາທິປະໄຕ ປະຊາຊົນລາວ";
  std::vector<std::string> tokens;
  testCut(str, tokens);

  std::vector<std::string> new_tokens = {"ສາທາລະນະລັດ", "ປະຊາທິປະໄຕ",
                                         "ປະຊາຊົນລາວ"};

  CLUCENE_ASSERT((tokens == new_tokens));
}

void testTibetan(CuTest *tc) {
  std::string str =
      "སྣོན་མཛོད་དང་ལས་འདིས་བོད་ཡིག་མི་ཉམས་གོང་འཕེལ་དུ་གཏོང་བར་ཧ་ཅང་དགེ་མཚན་མཆིས་སོ། །";
  std::vector<std::string> tokens;
  testCut(str, tokens);

  std::vector<std::string> new_tokens = {
      "སྣོན",  "མཛོད", "དང",  "ལས", "འདིས", "བོད", "ཡིག", "མི",   "ཉམས", "གོང",
      "འཕེལ", "དུ",   "གཏོང", "བར", "ཧ",   "ཅང", "དགེ", "མཚན", "མཆིས", "སོ"};

  CLUCENE_ASSERT((tokens == new_tokens));
}

static void testChinese(CuTest *tc) {
  std::string str = "我是中国人。 １２３４ Ｔｅｓｔｓ ";
  std::vector<std::string> tokens;
  testCut(str, tokens);

  std::vector<std::string> new_tokens = {"我", "是",       "中",        "国",
                                         "人", "１２３４", "Ｔｅｓｔｓ"};

  CLUCENE_ASSERT((tokens == new_tokens));
}


std::vector<std::string> datas = {
  "{\"qid\": \"qid_1815059893214501395\", \"category\": \"烦恼-恋爱\", \"title\": \"请问深入骨髓地喜欢一个人怎么办我不能确定对方是不是喜欢我，我却想 \", \"desc\": \"我不能确定对方是不是喜欢我，我却想分分秒秒跟他在一起，有谁能告诉我如何能想他少一点\", \"answer\": \"一定要告诉他你很喜欢他 很爱他!!  虽然不知道你和他现在的关系是什么！但如果真的觉得很喜欢就向他表白啊！！起码你努力过了！  女生主动多少占一点优势的！！呵呵  只愿曾经拥有！  到以后就算感情没现在这么强烈了也不会觉得遗憾啊~！  与其每天那么痛苦的想他 恋他 还不如直接告诉他 ！  不要怕回破坏你们现有的感情！因为如果不告诉他  你可能回后悔一辈子！！  \"}",
  "{\"qid\": \"qid_2063849676113062517\", \"category\": \"游戏-完美游戏-诛仙\", \"title\": \"我登陆诛仙2时总说我账号密码错误，但是我打的是正确的，就算不对我? \", \"desc\": \"\", \"answer\": \"被盗号了~我的号在22号那天被盗了，跟你一样情况，link密码与账号错误，我密保都有了呐，邮箱换密码也不行，还被删了号，伤心兼郁闷，呵呵，盗号了。建议跟完美申请把号要回来，或者玩新的号！\"}",
  "{\"qid\": \"qid_6625582808814915192\", \"category\": \"游戏-网络游戏\", \"title\": \"斩魔仙者称号怎么得来的 \", \"desc\": \"斩魔仙者称号怎么得来的\", \"answer\": \"楼主您好，以下为转载：\\r\\r圣诞前热身 来《生肖传说》做斩魔仙者\\r\\r　　一年一度的圣诞节快要来临了，大街小巷商户们都在忙着准备12月25日圣诞的来临。而这时候，一些妖魔也正蠢蠢欲动准备作乱。作为生肖世界肩负维护世界和平、拯救全人类的生肖使者，怎么能不有所行动，为了生肖世界的安定而做防范准备？！\\r\\r　　要让妖魔鬼怪能对你有所心悸，除了自己本身武艺要高强，最好能在妖魔界打出知名度，这样，当你的亲朋好友被妖魔袭击时，只要爆出你的名号，这些妖魔上就会落荒而逃，岂不好哉？那么，“斩魔仙者”这个响亮的称号应该足够能震慑住妖魔，让他们铭记在心了吧！\\r\\r斩魔仙者的称号\\r\\r　　而且，这个“斩魔仙者”的称号并不是人人都能得到的。只有成功挑战70级副本中的隐藏BOSS“羽翼仙”的人才能获得此称号！并且前提条件是在12月18日~12月25日之间第一队成功挑战羽翼仙的人才能获此称号！因此，此称号在全服范围内，是绝对不可能超过5个的！\\r\\r　　要挑战羽翼仙可不是一件容易的事。首先，要在70级副本中打败4个强大的BOSS！在打完副本的第4个BOSS有一定几率获得道具“羽翼真元”，有了羽翼真元后就可以与羽翼仙进行一场战斗。羽翼仙就站在第4个BOSS的旁边，只是没有道具是不能进入战斗的。\\r\\r羽翼仙\\r\\r　　在12月18日~12月25日活动期间成功挑战羽翼仙后的第一支队伍就可以获得兑换“斩魔仙者”的道具——烈火珍珠旗。当然，如果你在这场激烈的战斗中不幸捐躯，那么当然是不会得到这个道具的。得到了这把“烈火珍珠旗”的玩家就可以到NPC燃烧使处兑换称号了！\\r\\r　　这样兼具高强能力和超强人品才能获得的称号，怎么能不人望而生畏，怎么能不让那些妖魔胆怯？想要获得的玩家就快快行动，莫要让人先抢了这全服唯一的“斩魔仙者”称号！\\r\\r如果满意，请采纳。\\r谢谢~\"}",
  "{\"qid\": \"qid_9204493405205415849\", \"category\": \"商业/理财-贸易\", \"title\": \"有哪位好心人上传一份女衬衫的加拿大海关发票给我看一下塞多谢了 \", \"desc\": \"多谢了\", \"answer\": \"我给你信息了\"}",
  "{\"qid\": \"qid_5049427108036202403\", \"category\": \"娱乐-度假旅游\", \"title\": \"想去澳州看亲戚二个星期，怎么去，求教 \", \"desc\": \"想去澳州看亲戚二个星期，怎么去，求教\", \"answer\": \"你看亲戚，申请的是旅游签证676！澳洲旅游签证很容易的。 \\r\\n你让亲戚将他的护照签证页和个人信息页扫描发邮件给你，地址联系电话给你。最好是亲戚给你一封邀请函。\\r\\n你填一个自己的信息表，银行存款证明，单位的准假证明，房产证，户口本等材料递交即可。。 \\r\\n7~8天就可以拿签证，还是一年多次，每次停留30天的签证。 \\r\\n\\r\\n我与你情况一样，3月份刚刚去过，很爽。 \\r\\n不过我怕麻烦，交给旅行社代办的签证，收了几百元。签证+服务费是1200元。省力，英文填表都交给他们了。 \\r\\n\\r\\n澳大利亚签证无需面签的。 \\r\\n\"}",
  "{\"qid\": \"qid_7961760649211429660\", \"category\": \"生活-礼节礼仪\", \"title\": \"送男生什么生日礼物好呢？[思考] \", \"desc\": \"送男生什么生日礼物好呢？[思考] \", \"answer\": \"运动护腕，游戏机，CD，整蛊玩具，男士香水，希望能帮到你，麻烦给“好评”\"}",
  "{\"qid\": \"qid_2507568206590316480\", \"category\": \"教育/科学-外语学习-英语考试\", \"title\": \"英语音节划分问题在重读和非重读音节的相邻处只有一个辅字组时，如果 \", \"desc\": \"在重读和非重读音节的相邻处只有一个辅字组时，如果前面重读音节里的元音是长音（长元音和双元音）则辅字组属于后面一个音节，如果重读音节里的元音是短音（短元音），则辅字组属于重读音节。例如：\\r长音 pa-r fa-ther ze-ro mo-tor Si-mon stu-dent fea-ture e-ven \\r短音 cap-ital moth-er ver-y mod-el Lil-y stud-y feath-er sev-en \\r两元音字母之间有一个辅音字母时,辅音字母归后一音节\\r---------\\r问题一:\\r比喻capital如果按底在重读和非重读音节的相邻处只有一个辅字组时，如果前面重读音节里的元音是长音（长元音和双元音）则辅字组属于后面一个音节，如果重读音节里的元音是短音（短元音），则辅字组属于重读音节。例如：\\r长音 pa-r fa-ther ze-ro mo-tor Si-mon stu-dent fea-ture e-ven \\r短音 cap-ital moth-er ver-y mod-el Lil-y stud-y feath-er sev-en \\r两元音字母之间有一个辅音字母时,辅音字母归后一音节\\r---------\\r问题一:\\r比喻capital如果按底二句话就应该是ca-pi-tal这样划分啊！为什么真实是cap-i-tal这样划分的\\r----------------\\r问题二:\\r还有我在学习英语音节划分中的课程内容，学到这里，有一点我就不明白了，如果先要分辨出：前面重读音节里的元音是长音还是短音，再作辅字组的划分。就必须要先分辨出前面音节是开音节还是闭音节，才知道是长音还是短音了，可再分开/闭音节的过程中又要看此辅字组是在前面音节还是后面音节。\\r----------------------------\\r问题三:\\r为有音标的w和r和v在拼的时候好像拼出来的音是一样的啊这是为什么！D和那个舌头伸出来的音拼出来的也一样!father中的there不是和de一样吗?要怎样才能分得出区别啊！是不是有区别的是我听不出而已\\r-------------\\r问题四:\\r还有就是你们学拼读规则都在那里学习的怎么我在网上找不到这样的视频啊找到的都是文章来的！是不是大家都不重视拼读规则的所以就没那些教程！\\r--------------\", \"answer\": \": \\r音节划分不是像你说的那个样子。\\r你去翻看一下OXFO 字典，里面的音标都是根据元音划分的。还有一点就是，这个划分与词根、前缀、后缀都有关系的。\\r---------------- \\r问题二: \\r亲，我真的想说，多听多说比你这样死钻牛角尖区分这个要管用的多。。。\\r---------------------------- \\r问题三: \\r为有音标的w和r和v在拼的时候好像拼出来的音是一样的啊这是为什么！D和那个舌头伸出来的音拼出来的也一样!father中的there不是和de一样吗?要怎样才能分得出区别啊！是不是有区别的是我听不出而已 \\r\\r怎么会一样呢。。。我觉得多半可能你发音不准啊 你可以比对口型还有送气的方式 多跟外教探讨一下 \\r\\rD TH 这完全是两个不同的音 无论单拼还是放在单词里都是不同的 \\r同学我觉得你多半是发音有问题 如果你发音不准那么你听力肯定好不了 \\r------------- \\r问题四: \\r有一本书 语音基础 蓝皮的 是英语系的语音基础的教科书 很好 如果你想系统地学可以考虑买一本看看\\r--------------\\r补充问题\\r见回答四\"}",
  "{\"qid\": \"qid_3642560876521827750\", \"category\": \"教育/科学-理工学科-建筑学\", \"title\": \"厂房内有吊车，起吊高度怎么定？厂房内有吊车，牛腿高度怎么定？根据 \", \"desc\": \"内有吊车，牛腿高度怎么定？根据什么来定？生产的工艺要求？还是要考虑到其他因素？\", \"answer\": \"你好！\\r\\n\\r\\n首先按厂房内的最大部件确定吊车的起重量和起吊高度，这个一般由工艺专业确定，然后根据吊车样本提供的吊车高度定下吊车轨顶标高，再按厂房跨度、起吊重量等选定吊车梁和吊车轨道连接件，那么牛腿高度 = 吊车轨顶标高 - 吊车梁高度 - 轨道连接件高度。\"}",
  "{\"qid\": \"qid_8990759060620275711\", \"category\": \"育儿-妈妈-分娩期\", \"title\": \"你好！请问有疼痛晕厥史的产妇可以顺产吗？ \", \"desc\": \"你好！请问有疼痛晕厥史的可以顺产吗？\", \"answer\": \"您好，可以尝试一下无痛分娩。\"}",
  "{\"qid\": \"qid_5331366642436435012\", \"category\": \"商业/理财-个人理财\", \"title\": \"投资个人理财产品需注意哪些问题? \", \"desc\": \"\", \"answer\": \"投资个人理财产品哪些问题需注意 \\r\\n\\r\\n第一切记，人民币理财产品不同于储蓄、国债等产品，并非没有风险，“预期收益率也并非“实际收益率”。不论哪个银行，一般都不会承诺“保本保息”之类的最低收益或固定收益，打出的底牌往往是“预期收益率”。无论是哪一种理财产品，它的主要投资对象是-->国债、央行票据和金融债券等。因此基金存在的风险，理财产品照样存在。另一方面，风险来自产品本身是否是合法的、产品设计上是否有缺陷。 \\r\\n\\r\\n第二切记，个人理财产品的高收益通常伴随着高风险。据有关人士介绍，扣除各项成本后，一般目前一年期的理财产品收益率在2.7%—3.1%，如果银行承诺的利率超过这个上限，投资者在购买时，就要谨慎行事了。 \\r\\n\\r\\n第三切记，在购买人民币理财产品时一定要对购买合同和协议进行认真审查。一旦发生问题，购买者有可能处于被动位置。 \\r\\n\"}",
  "{\"qid\": \"qid_4371543678850352818\", \"category\": \"健康-医院\", \"title\": \"女性经常有作乳房自查是否不用每年上医院作体检了？经常自查乳房的， \", \"desc\": \"经常自查乳房的，无异常。不用再去查了吗？\", \"answer\": \"    尽管业界一直提倡女性学会乳房自查，以及早发现异常。但是也存在部分乳腺疾病如有的乳腺癌早期可能不完全有明显肿块触及，或者乳房外形无特殊改变。自查会在一定程度上有漏诊，造成严重后患。\\r\\n    因此，自查也不能代替医院相关医生和器械的科学筛查，建议成年女性每年应去正规医院体检一次。\"}",
  "{\"qid\": \"qid_1573872705600431367\", \"category\": \"文化/艺术-文学\", \"title\": \"\\\"我爱你\\\"古文怎么说? \", \"desc\": \"\\\"\\\"古文怎么说?\", \"answer\": \"吾怜卿,甚矣----我爱你.真的很爱.\\r\\n\\r\\n吾甚怜卿----我非常爱你....\\r\\n\\r\\n吾甚怜卿耶----我特爱你呀!\"}",
  "{\"qid\": \"qid_9187122022980985230\", \"category\": \"教育/科学-人文学科-历史学\", \"title\": \"为什么没有副总书记一职 \", \"desc\": \"为什么没有副总记一职\", \"answer\": \"楼上的说的不对 你说的是八大准备的事情  人家问的是现在事情\\r\\n不设副总书记 是为了保证总书记的权威  副总书记 有些太子和明确二把手的意思 会对天子产生威胁 清朝后期已经不设太子了  现在就更没有必要了\"}",
  "{\"qid\": \"qid_4106106752044617103\", \"category\": \"教育/科学-学习帮助\", \"title\": \"已知a,b都是锐角，tana=4/3,tanb=1/7，求a\", \"desc\": \"已知a,b都是锐角，tana=4/3,tanb=1/7，求a-b=？\", \"answer\": \"tan(a-b)=(tana-tanb)/(1+tanatanb)\\r\\n=(4/3-1/7)/[1+(4/3)(1/7)]\\r\\n=(28-3)/(21+4)=1\\r\\na,b都是锐角,-π/2<a-b<π/2\\r\\n所以a-b=π/4.\"}",
  "{\"qid\": \"qid_7707644486067442493\", \"category\": \"娱乐-花鸟鱼虫\", \"title\": \"请教一下，这是什么植物？谢谢。 \", \"desc\": \"请教一下，这是什么？谢谢。 \", \"answer\": \"这应该是迎春花\"}",
  "{\"qid\": \"qid_525979235196905709\", \"category\": \"游戏-久游游戏-魔力宝贝\", \"title\": \"家族建立必须等级？我看新浪任务说家族建立的必须条件是LV50+， \", \"desc\": \"我看新浪任务说建立的必须条件是LV50+，是家族长必须LV50+？还是家族建立的五个人都必须LV50+啊？\", \"answer\": \"家族建立的时候，对于家族长和家族建立的同伴4人的等级全是无限制的（这个新浪在任务上有个小错误）！\\r\\n那么只是推荐等级为50+而已。家族建立的任务其实并不难，也不是很长，之所以推荐LV50+，主要有两点困难：1.通过地之神管的任务是制作2分钟内料理“美奶滋拌饭”。这里的困难不是2分钟，而是材料问题。主要就是15个瓷碗的拿法（见下），找朋友帮忙拿下15个碗也就没什么了。2.通过火之神管的任务是第二个小困难，如果没有等级好一点的组队的话，其五连战恐怕会困难一点，总体来说还是不难。5人组队还是有相当的胜算的。\\r\\n回到你的问题，结论就是家族的建立无论是族长还是建立的4名同伴，其等级都是无限制的。\\r\\n\\r\\nPS：祝愿魔力愉快，家族兴旺。新年快乐~~！\"}",
  "{\"qid\": \"qid_6001180011544757766\", \"category\": \"健康-五官科-口腔科\", \"title\": \"牙龈包住牙齿怎么办 \", \"desc\": \"包住牙齿怎么办\", \"answer\": \"病情分析：\\n你好,牙龈出血以局部原因最为常见,但也有是其他疾病在牙龈组织上出现的一种症状.\\n指导意见：\\n局部原因为牙龈炎和牙周炎,主要是口腔卫生不佳,牙垢,牙结石,不良修复体,补牙材料,食物嵌塞等局部刺激因素引起牙龈充血肿胀,在刷牙,咬硬物,吃水果时发生牙龈出血,有时可有少许的自发性渗血.将刺激因素去除后,出血即会好转.牙龈外伤主要是指剔牙时造成的牙龈损伤,致使牙龈出血.这样情况应采取正确的剔牙方法,保护牙龈.全身因素有血液病如血友病,白血病,再生障碍性贫血等,有相当一部分患者是最初以牙龈出血而到口腔科就诊的.其他一些慢性疾病亦可导致牙龈出血,如糖尿病,维生素C缺乏及一些消耗性疾病.肝脏疾病可因影响凝血因素的合成,也会有牙龈出血.发现牙龈出血,不可过度紧张,应及时去医院检查,如为局部因素引起,经口腔科治疗即可.如出血量多不易止血时,应到内科诊治,要及时做血液系统检查\"}",
  "{\"qid\": \"qid_3473806824072149837\", \"category\": \"教育/科学-人文学科-哲学\", \"title\": \"加强调的着重号有什么作用？列宁指出：“物质是标志〔客观实在〕的哲 \", \"desc\": \"列宁指出：“物质是标志〔客观实在〕的范畴，这种客观实在是【人通过感觉感知的】，〖它不依赖于我们的感觉而存在〗，【为我们的感觉所复写、摄影、反映】。”\\r我用表示〔　〕下圆点，用【　】代替波浪线，用〖　〗代替下划线，请问列宁的物质定义中的着重号要强调的是什么？\", \"answer\": \"　　辩证唯物主义物质观的重大意义有：\\r\\n①它指出物质是不依赖人的感觉和意识的客观存在，坚持了唯物主义一元论，同唯心主义和二元论划清了界限。\\r\\n②它指出物质是能被人所感觉、认识的，坚持了可知论，同不可知论划清了界限。\\r\\n③它指出客观实在性是物质的唯一特性，这就把物质与具体的物质形态区别开来，坚持了辩证唯物主义，同旧唯物主义划清了界限。\\r\\n④它指出了物质是哲学范畴，物质既包括自然界也包括人类社会，这就揭示了自然界和人类社会的物质统一性，坚持了历史唯物主义，同历史唯心主义划清了界限。\\r\\n\\r\\n客观实在是各种物质现象的普遍本质和唯一特性。\"}",
  "{\"qid\": \"qid_193153085081424041\", \"category\": \"电脑/网络-电脑装机\", \"title\": \"请问这样一个配置需要多大的电源？cpu:AMD羿龙X38450（ \", \"desc\": \"cpu: AMD 羿龙X3 8450（盒）\\r：技嘉 GA-MA770-S3(rev. 2.0)\\r内存：金士顿 2GB DDR2 800\\r硬盘：希捷 320G 7200.11 16M（串口/5年盒）\\r显卡：迪兰恒进 HD4850北极星\\r\\r\\r不知道这套配置怎么样，还有哪些不足，高手帮帮忙，下个星期就攒了\\r听说现在希捷500G的负面问题多多，很多人推荐我用320G的，请问500G的问题真的很多么？\\r电源用TTXP420的合适么？功率是否够用？还是用其他的？帮忙推荐几款电源。。\", \"answer\": \"你这套配置和我现在用的这台机器的差不多，我用的是\\rcpu: AMD 羿龙X2 5000+（黑盒）配九州风神 黑虎鲸散热器\\rcpu OC到3G \\r：捷波悍马 HAO3(AMD 790X rev. 1.0) \\r内存：黑金刚 2 X 1GB DDR2 800 时序调整为（4-4-4-12）默电\\r硬盘：希捷 500G 7200.11 32M（串口/5年盒） \\r显卡：迪兰恒进 HD4850公版 自己更换ZERO 8800U太空梭散热器\\r总的来说这样的配置对付一般的游戏都很轻松，在22寸~24寸的显示器上都可以比较完美的运行绝大部分游戏！你的CPU比我的要好一点，耗电要高一点，其它的和我的配置差不多\\r我开始用的电源是航嘉 磐石400，在这个平台上面一般情况下可以正常使用，但是在我把显卡和CPU同时大幅超频的情况下，电源挂掉了！现在换了个台达的600W的电源\\r所以按你的配置，建议在不打算超频使用的情况下，选择足额输出350W以上的电源，个人建议选择400W左右的预留一些亢余用起来更放心一点！\\r目前个人不大推荐你选择航嘉电源，最近航嘉被暴电源输出虚标功率，建议你看看鑫谷，长城，全汉的大功率电源！\"}",
  "{\"qid\": \"qid_7313842672675064329\", \"category\": \"教育/科学-外语学习\", \"title\": \"翻译3Shedescribestheexperienceof15 \", \"desc\": \"She describes the eerience of 15-year-old Jessica from Burlington. Jessica's leg was cut off at the knee because of cancer. She was learning to wear a prosthesis . Jessica tells about playing soccer . She kicked the ball hard and it flew off in one direction while her artificiShe describes the eerience of 15-year-old Jessica from Burlington. Jessica's leg was cut off at the knee because of cancer. She was learning to wear a prosthesis . Jessica tells about playing soccer . She kicked the ball hard and it flew off in one direction while her artificial leg flew another way. She fell on the ground and laughed . \", \"answer\": \"She describes the experience of 15-year-old Jessica from Burlington. Jessica's leg was cut off at the knee because of cancer. She was learning to wear a prosthesis . Jessica tells about playing soccer . She kicked the ball hard and it flew off in one direction while her artificial leg flew another way. She fell on the ground and laughed .  \\r\\r她叙述了来自Burlington.的１５岁女孩杰西卡的经历．由于癌症的缘故，杰西卡的膝盖以下部分被截肢了．那时她正学着使用一只义肢来替代腿．杰西卡谈到了踢的故事．一次，她对着球猛踢了一脚，球朝着一个方向飞去的同时，她的假腿却飞向了另一个方向．杰西卡摔倒在地，哈哈地笑了起来． \\r  \"}",
  "{\"qid\": \"qid_2496719599303374558\", \"category\": \"娱乐-幽默滑稽\", \"title\": \"网络上真的没有帅哥美女吗?突然间想到的~~以前在哪里看到过是这么? \", \"desc\": \"上真的没有帅哥美女吗?\\r突然间想到的~~以前在哪里看到过是这么说的\", \"answer\": \"美女哪有空来成天上网，都约会去了。。。\\r\\n帅哥知道网上都是恐龙所以也都不来了。\"}",
  "{\"qid\": \"qid_9201495377383673621\", \"category\": \"生活-育儿\", \"title\": \"耳朵怎么了宝宝晚上睡觉总是翻来覆去的,仿佛睡不塌实,而且每次翻身 \", \"desc\": \"晚上睡觉总是翻来覆去的,仿佛睡不塌实,而且每次翻身的时候总用手挖耳朵,找医生看了,也没发现里面有什么,不知道他为什么这样?是不是缺什么维生素什么的?\", \"answer\": \"宝宝多大了，要是4个月左右，属于长牙期间，他挖耳朵是因为牙床痒痛\"}",
  "{\"qid\": \"qid_8085049932951069286\", \"category\": \"生活-购房置业\", \"title\": \"北京房产证过户手续怎么办》夫妻两人一人先走，房产过户另一个人，手 \", \"desc\": \"两人一人先走，房产过户另一个人，手续怎么办？\", \"answer\": \"去公证处 做一个继承公正 然后拿着继承公正去房地产交易中心可以直接发新证\"}",
  "{\"qid\": \"qid_3488727986990973326\", \"category\": \"健康-精神心理科-心理科\", \"title\": \"面烧糊了还能吃吗糊味明显，吃了对健康影响大吗 \", \"desc\": \"糊味明显，吃了对影响大吗\", \"answer\": \"最好别吃，没好处而且也难吃，容易致癌\"}",
  "{\"qid\": \"qid_713239302402366328\", \"category\": \"社会民生-求职就业\", \"title\": \"刚毕业的大学生怎样考职称吗？求教：刚毕业的大学生可以怎样考职称？ \", \"desc\": \"求教：刚毕业的大学生可以怎样考职称？\\r\\r我今年刚毕业 \\r不是很懂有关考职称的事情 \\r请问是不是只能从初级开始考？ \\r可不可以直接考中级？ \\r需要什么资质条件吗？ \\r我现在做的工作是类似于办公室主任的工作 \\r有关企业啊行政人事管理啊之类的\\r单单考计算机和英语职称有用吗\\r我现在可以考什么样的职称?谢谢！ \\r谢谢！\\r\", \"answer\": \"你好：\\r\\n\\r\\n你的工作内容可以报考经济类职称。首先是从初级报考，报考助理经济师（初级），条件是你本科毕业，见习期满一年之后，直接申请评审，不需要考试。\\r\\n\\r\\n今后报考中级就需要考试了。经济师（中级）在你获得助理四年之后就可申请报考。\\r\\n\\r\\n关于职称和职业问题，欢迎浏览我的专题“职场人生”，希望有帮助： \\r\\n\\r\\n祝你顺利！！\\r\\n\\r\\n\\r\\n\"}",
  "{\"qid\": \"qid_4604022711015097001\", \"category\": \"娱乐-动漫\", \"title\": \"求好看的动漫 \", \"desc\": \"求好看的\", \"answer\": \"八犬，狂欢节，K，黑执事，家教，海贼，幸运星.....夏目友人帐，新全职猎人，冰果，秒速五厘米问题儿童都来自异世界 ， 血型君火影忍者、网球王子、名侦探柯南\"}",
  "{\"qid\": \"qid_802484035859939592\", \"category\": \"游戏-巨人游戏-征途\", \"title\": \"为什么我起名字老是不合法?这个游戏不可以起带有特殊符号的名字吗? \", \"desc\": \"这个不可以起带有特殊符号的名字吗?\\r比如:ρs.等︶ㄣ待  裃???喏 ??Zoeヤゞ   ..[ゃ鉒錠﹖\", \"answer\": \"目前只能用中文和英文起，而且中文有些词组也不行\\r\\n像日文，智能abc打出的符号都不行\\r\\n\\r\\n你可以先起玩看看，不行就删，千万别留***的名字，以后很多事都不方便，像推荐人之类的\"}",
  "{\"qid\": \"qid_8757963117572327404\", \"category\": \"健康-精神心理科\", \"title\": \"你相信这个世上有真爱吗？我觉得人在一起久了就会有感情！你们说呢？ \", \"desc\": \"我觉得人在一起久了就会有！你们说呢？\", \"answer\": \"半信半疑,相处时间长啦,那是感情.但那种感情就是一般的那种朋友,人活在世上不可能也不可以没有朋友,那里一点一滴积累下来的感情.是很现实的.但是真爱?这个\\\"爱\\\"太难懂,太复杂啦!\"}",
  "{\"qid\": \"qid_3841177982407043650\", \"category\": \"教育/科学-理工学科-生物学\", \"title\": \"我家公猫为什么晚上乱叫呀~~~ \", \"desc\": \"\", \"answer\": \"是发情呢呵呵.猫发情没有固定时间的,差不多没个月都闹,有的厉害的一个月有半个月在闹猫.所以建议你带它去做绝育吧. \\r\\n做绝育的好处太多了,本身它是公猫对它伤害特别小不像母猫还开膛,当天就能活动了.且做了对它本身有好处,猫发情期是很痛苦的,还会影响食欲. \\r\\n对你的家庭也好,不用整天被它影响睡眠了,绝育过的猫猫更粘人脾气也会变的温顺乖巧,饭量也会增加哦.\"}",
  "{\"qid\": \"qid_6056160307824183948\", \"category\": \"健康-医院\", \"title\": \"你好我家宝宝7多月，这几天拉肚子，去医院检查说是细菌感染肠炎，吃? \", \"desc\": \" 你好我家宝宝7多月，这几天拉肚子，去检查说是细菌感染肠炎，吃了几天的药反反复复，今天拉泡泡\", \"answer\": \"给宝宝吃点妈咪爱\"}",
  "{\"qid\": \"qid_755887206012435380\", \"category\": \"游戏-网易游戏-魔兽世界\", \"title\": \"关于安装台服的问题等CWOW遥遥无期,鄙人也花了一天时间下载了台 \", \"desc\": \"等CWOW遥遥无期,鄙人也花了一天时间下载了台服,我的CWOW装到D盘,台到F盘,可我安装台服的时候,他总是直接跳出CWOW的登陆界面,这是什么问题?\\r另外,希望大家给我个明确,详细的解决方案 谢谢了\", \"answer\": \"你机器里同时有国服和台服是不行的.那样会有冲突的.\\r无法正常运行的.\\r\\r以下办法用下吧.\\r台服客户端下载：  \\r(1)打开CWOW文件夹，找到realmlist.wtf文件，用记事本打开它，先清空里面的内容，再添加“set realmlist  ”（不包括双引号），保存后退出，这样，在每次要进去台服的时候，只需要用WoW.exe来启动，如果是进去CWOW的话，就用Launcher.exe来启动，不会造成冲突。另外，用WoW.exe启动后，登陆的界面是CWOW的界面，这是因为使用的是CWOW的客户端的原因，不用怀疑，直接输入ID密码登陆后进入的就是台服。 ---------------------- \\r(2)如果你已经有CWOW的客户端了，再想装TWOW的时候需要修改注册表。 \\r修改方法： \\r1、　点击　开始 \\r2、　点击　运行 \\r3、　在运行框里输入　regedit4、　在弹出的注册表框里点击　编辑 \\r5、　点击查找 \\r6、　在查找框里输入　Blizzard Entertainment \\r7、　当左边框里搜索到　Blizzard Entertainment　的时候，在左边整个删除，一直搜索到出现对话框里说　注册表搜索完毕　为止 \\r[安装] \\r然后是安装，直接解压，安装就OK了，安装后貌似是1.11.0的版本(视你下载的版本为定)。 --------------------- \\r补丁方面,直接通过程序下载,速度很快,一般能达到80K/S以上的速度(1M ADSL).---------------------------【注册账号】此处资料为转载，仅供参考------  \\r直接申请体验CDK,然后开始录入个人资料. \\r\\r召集令编号: \\rRF070112000032 \\rRF070112000033 \\rRF070112000034 \\rRF070112000035 \\rRF070112000036 \\r注：注册帐号时填写\\\"召集令编号\\\"可获得（14天）30小时体验优惠。 \\r非必选项 \\r录入个人资料这里,请注意一下,请尽量填写个人准确资料,如地区选(其它),以及真实身份证号码,真实住址,还有真实的email地址.(^^我选的住在钓鱼岛...不过地址栏下面的都写的真实的) \\r范例：南海诸岛钓鱼台列屿中华人民共和国江苏省南京市 ...XD~~ \\r\\r至此,你的体验帐号基本申请完毕，可以去游戏里进行下体验和尝试了这样可以安装2个客户端,分别指向CWOW和TWOW.--------------------------【输入法】此处资料为转载，仅供参考----------五笔输入法的朋友，到 \\r  ,下载极点中文输入法，安装后在设置选项里调成输出繁体字。 \\r\\r拼音输入法的朋友，WINDOWS控制面板→区域和语言选项→语言→详细信息→添加→微软拼音输入法3.0，添加后在输入法的选项里调出“简/繁体”的按纽，在新增加的简繁体切换按纽按一下换成繁体即可. \\r（推荐使用微软拼音输入法2003，效果比微软拼音输入法3.0要好，放出地址 \\r  ;下载后直接安装就可以了，会自动覆盖原来的微软拼音输入法3.0） \\r-----------------------【充值月卡】-------------------转为正式帐号： \\r直接冲值月卡即可，不需要CDK，重大更新！~ \\r  ... _id=2&GoPage=1&lis= \\r\\rTWOW最大的优势就是月卡 \\r插件方面,目前大部分的简体插件是能使用在TWOW的，您下载的魔兽精灵依然可以在TWOW使用。 \\r最新补充 \\r1/台湾没有距阵卡，注意帐号安全 \\r2/注意保存密码，共有3种，分别为游戏密码，官网论坛密码，安全码， \\r（本人好久没玩，安全码遗忘中，找客服已经解决） \\r3/台湾每周四早上4点例行维护，一般时间为6小时，大部分情况是提前开机\"}",
  "{\"qid\": \"qid_5012882145821532503\", \"category\": \"娱乐-明星\", \"title\": \"李克勤什么歌好听 \", \"desc\": \"\", \"answer\": \"红日。。。。。\"}",
  "{\"qid\": \"qid_7521787209011105347\", \"category\": \"健康-男科-前列腺科\", \"title\": \"血压140/105我18岁征兵体检时发现有高血压，30岁时偶尔会 \", \"desc\": \"我18岁征兵时发现有高血压，30岁时偶尔会头晕，测量血压为135/90,持续服用过卡托普利，效果不明显，近半年来，我的头晕加重，脖子转动时感觉僵硬，筋疼，服用吲达帕胺片，现在的血压是140/105，我查了一下资料，我就是舒张压太高了，是什么原因呢？应该服用什么药才好呢？\", \"answer\": \"我18岁征兵体检时发现有高血压，30岁时偶尔会头晕，测量血压为135/90,持续服用过卡托普利，效果不明显，近半年来，我的头晕加重，脖子转动时感觉僵硬，筋疼，服用吲达帕胺片，现在的血压是140/105，我查了一下资料，我就是舒张压太高了，是什么原因呢？应该服用什么药才好呢？\\r\\n大概是你当上“官”的原因，现在当官的不是“三高”就是“四高”，简单的说是：不良的生活方式和饮食习惯造成的。\\r\\n应该服用什么药？最好找中医用中药调理。原因简单如下：\\r\\n\\r\\n不仅卡托普利会引起药源性震颤麻痹，不少药物若使用失当也会诱发此病，其实，类似的现象并非罕见，常见的药物有：降血压药：\\r\\n\\r\\n　　利血平是从印度萝芙木提取的生物碱，降压灵是中国萝芙木所含的生物总碱，降压作用较利血平弱。两者的结构相似,其药理作用使交感神经末梢囊泡内递质(去甲肾上腺素和多巴胺)耗竭，因此长期应用易诱发药源性震颤麻痹征和抑郁症。\\r\\n\"}",
  "{\"qid\": \"qid_1336718430447702776\", \"category\": \"生活-育儿\", \"title\": \"有人知道宝宝补锌产品排行榜么？知道的可以说说建议，上面什么产品比? \", \"desc\": \"有人知道补锌产品排行榜么？知道的可以说说建议，上面什么产品比较好！\", \"answer\": \"宝宝补锌产品排行榜？冒失上次看到过，记得有过一个叫新稀宝牌子的补锌产品，是第三代蛋白锌产品，补锌效果是最好的，你可以百度百科新稀宝看看具体资料！\\r\\n\"}",
  "{\"qid\": \"qid_6790807861626896337\", \"category\": \"生活-服装/首饰\", \"title\": \"春装秋装区别我想知道春装和秋装的区别！要详细地说！ \", \"desc\": \"我想知道和秋装的区别！要详细地说！\", \"answer\": \"    最大的区别还是在质地上。据了解，夏装普遍以棉、丝等轻薄材料为主，而观察新上市的秋装，则以针织衫、厚棉材料为主。无论是上装还是裤子，普遍都比夏装增加了不少厚度。颜色也开始偏向秋天色彩，如咖啡、棕、黑等，即使是一些“天性”比较亮丽的颜色，如绿、紫、红等，也从原先的淡色系转向了深色系。而在天气仍有些微热的夏末，这种深色系新装并不为消费者买账，商家表示，春装的艳丽更接近这几天的消费心理。\"}",
  "{\"qid\": \"qid_4781331459651416283\", \"category\": \"生活-生活常识\", \"title\": \"全国的邮政小包价格一样吗 \", \"desc\": \"全国的邮政小包价格一样吗\", \"answer\": \"中国邮政小包所有货物按克收费，分为9个区，每个区代表不同的国家，价格也有所不同，首重50克以内都是5元钱，然后在对应给出折扣，单件货物限制在2KG以内。中国邮政小包所有货物按克收费，分为9个区，每个区代表不同的国家，价格也有所不同，首重50克以内都是5元钱，然后在对应给出折扣，单件货物限制在2KG以内。深圳互联易中国邮政小包价格很不错的，深圳地区都可上门提货的类似于挂号信的小包都是4.5吧，好久没有寄了。不一样哦，分区域的，不过差别倒是都不大。您若感觉满意，请点“好评”，支持我在微问的道路上帮助更多有需要的人~不一样哦，起重资费：起重500克及以内20元； 续重资费 ：（续重每500克或其零数 ）一区 6元 ，二区9元 ， 三区  15元（具体分区方式，请寄件人拨打电话或到当地邮局营业窗口咨询，客服电话11185。）这样比较容易睡着。如果满意我的回答，可点击“对我有用”，谢谢。\"}",
  "{\"qid\": \"qid_6299572732647927935\", \"category\": \"教育/科学-理工学科-生物学\", \"title\": \"我在山西,冬天榕树怎样养植榕树一直掉叶,现在没剩几片了 \", \"desc\": \"一直掉叶,现在没剩几片了\", \"answer\": \"因为榕树喜阳,应该把它放置到有一定阳光的地方,还有冬天气温如果低于五度,叶片容易冻伤,应该把它移到室内。有的树桩盆景还有耐寒或非耐寒性,对非耐寒性的树桩一般冬天还要进入温室维护管理,如榕树、福建茶等。冬天浇水时注意不要湿了叶子和花朵。榕树适应性强，在酸性、中性、微碱性的土壤都能生长，但最好是在微酸、疏松、排水良好、又有一定肥力的土壤中养植。栽种一般在萌芽前的3～4月份进行。\\r\\n\\r\\n    榕树盆景多用紫砂盆，也有用釉陶盆的。盆的深度和样式根据造型而定，悬崖式多用签筒盆，如把盆景置于高脚几架上，也可用普通方盆或圆盆；曲干式、斜干式、提根式、树石式多用中等深度的长方形或椭圆形盆钵。\\r\\n\\r\\n荫棚内养护。秋末初冬，当夜间最低温度降至6一7℃时，应移入室内向阳处越冬，冬季室温以不低于5℃为宜。棕竹喜温暖、湿润、通风并有—定光照的环境，如烈日曝晒、盆内干燥，常造成枝叶枯黄，轻者生长不良，严重者可能枯死。\\r\\n\"}",
  "{\"qid\": \"qid_5121647188856896111\", \"category\": \"教育/科学-升学入学-考研\", \"title\": \"怎样才能够进入外交部工作呢？有什么要求吗？必须是相关的的专业才可? \", \"desc\": \"怎样才能够进入部工作呢？有什么要求吗？必须是相关的的专业才可以吗？需要考公务员吗?熟练掌握一门小语种是不是更有利呢？\", \"answer\": \"1）关于“怎样才能够进入外交部工作呢？”的回答：高中毕业后考入国际关系学院。\\r\\r2）关于“有什么要求吗？”的回答：\\r身体条件除须达到部和卫生部联合颁发的《普通高等学校招生体检工作指导意见》的有关规定外，还需符合下列条件:\\r\\r　　(1)男生身高170cm以上，体重50kg以上;女生身高160cm以上，体重45kg以上，体型匀称;\\r \\r　　(2)左右眼裸眼视力4.3以上(报考英语专业、信息管理与信息系统专业和信息应用专业的考生，左右眼裸眼视力4.8以上)，无色盲、色弱、斜视、对眼等眼疾;\\r\\r　　(3)听觉、嗅觉正常，无影响外语学习的听力系统和发音系统疾病，无口吃;\\r\\r　　(4)心脏、血液、肝功能正常，无甲肝、乙肝和澳抗阳性;无严重急慢性疾病，无传染病和精神疾患等;\\r\\r　　(5)五官端正，面部及身体暴露部位无明显特征和缺陷，无鸡胸，无腋臭，无严重静脉曲张，无明显八字步、罗圈腿，无平足、跛足，无斜肩、纹身、少白头、驼背，无各种残疾，本人及直系血亲无精神病史。\\r以上只是客观条件的要求， 另外最重要的就是成绩。\\r\\r3）关于“必须是相关的的专业才可以吗？”的回答：相关的专业很多。只要你得本专业优秀就可以有机会。\\r\\r4）这个行业不同公务员， 如果你直接考入外交部， 那就没必要再考公务员了！\\r\\r5）任何行业熟练掌握一门小语种都是更有利的。\"}",
  "{\"qid\": \"qid_6432791236741067118\", \"category\": \"商业/理财-基金\", \"title\": \"广发货币基金转聚丰需要几天？ \", \"desc\": \"\", \"answer\": \"在交易日15：00前转换，按当天基金净值计算份额，下一交易日确认。\"}",
  "{\"qid\": \"qid_7204323359827944758\", \"category\": \"生活-美容/塑身-祛斑/祛痘\", \"title\": \"上海简单有效的祛斑方法_祛斑需要多少钱？ \", \"desc\": \"上海简单有效的方法_祛斑需要多少钱？\", \"answer\": \"祛除雀斑一定要采用专业的治疗方法，千万不能盲目的相信偏方治疗，以免耽误自己的病情。目前治疗雀斑的方法有很多，大多数患者会选择激光治疗，因为激光治疗可以准确的定位，将皮肤内的色素沉积击碎，有选择性的吸收，治疗的更加彻底，而且见效快，安全不留疤痕。\\r\\n激光祛除色斑操作简单、效果明显、过程时间短，最多不过十几分钟。每次治疗结束以后，皮肤都要有一个自然吸收的新陈代谢过程，因此治疗的间隔时间一般为2到3个月。根据色素及血管病变的大小、数量及深度，有的治疗一两次就能达到预期的效果，一般则需4到6次才能彻底解决问题。激光祛除色斑效果是一般的祛除色斑产品无法比拟的。\\r\\n激光祛斑是利用激光的光热解原理。激光照射皮肤后，会被皮肤病变部位的色素吸收，而激光产生的高能量就会将这些色素击碎，色素碎屑会随人体的各项循环排除体外，从而达到祛斑的效果。激光祛斑对颜色较深的色素斑、辐射斑等效果是最为显著的。激光治疗能达到比较理想的效果是由激光治疗的原理决定的，激光治疗的原理是激光仪产生一束高强的光束，不同类型的激光可以产生不同颜色的光线。激光的禸眼色是治疗的关键。不同颜色的激光被不同颜色的皮肤吸收，并利用激光能量令染料颗粒崩解汽化，封闭血管，再由身体吸收染料颗粒将其排出体外，色素随之消退。\\r\\n\"}",
  "{\"qid\": \"qid_3109046540611139702\", \"category\": \"游戏-久游游戏-魔力宝贝\", \"title\": \"怀旧我想打阿鲁巴斯，要拿护士帽吗怎么个流程，话说我玩这么久，没直 \", \"desc\": \"怎么个流程，话说我玩这么久，没直接干过2684\", \"answer\": \"没有护士帽也可以直接打阿鲁巴斯...\\r\\n请注意看下面的图片:\"}",
  "{\"qid\": \"qid_3677532485006819107\", \"category\": \"电子数码-照相机/摄像机\", \"title\": \"为什么低端的数码相机采用CMOS传感器，高级一点采用CCD，而更? \", \"desc\": \"\", \"answer\": \"     1、 目前，人们使用的数码相机，数码、拍照手机和摄像头的感光器材主要是分为CCD和COMS两类，相比较而言，CCD的优势在于成像质量好，但是由于制造工艺复杂，只有少数的厂商能够掌握，价格较高。在相同分辨率下，CMOS价格比CCD便宜，但是CMOS器件产生的图像质量相比CCD来说要低一些。到目前为止，市面上绝大多数的消费级别以及高端数码相机都使用CCD作为感应器，CMOS感应器则作为低端产品只应用于一些摄像头上。\\r      2、 然而，CMOS影像传感器也有自身的优点，其电源消耗量比CCD低。CCD为提供优异的影像品质，付出代价即是较高的电源消耗量，为使电荷传输顺畅，噪声降低，需由高压差改善传输效果。但CMOS影像传感器将每一画素的电荷转换成电压，读取前便将其放大，电源消耗量比CCD低。此外，由于CMOS整合性高，使体积大幅缩小，这点是CCD所不能及的。目前CMOS影像传感器首要解决的问题就是降低噪声的产生，所以有的比较高级影象生产企业在解决了CMOS降燥问题后,反而使用CMOS。\\r      3、 未来CMOS影像传感器是否可以改变长久以来被CCD压抑的宿命，往后技术的发展是重要关键。\"}",
  "{\"qid\": \"qid_4173358817197735817\", \"category\": \"游戏-网络游戏\", \"title\": \"攻击别国占领陆地的结果是如何计算的。攻击别国占领陆地的结果是如何 \", \"desc\": \"攻击别国占领陆地的结果是如何计算的？\\r\\n请教高手，有公式计算的么？是如何计算的？\", \"answer\": \"楼上说的不正确，要这么简单楼主就不问了\\r\\n\\r\\n好像和攻打国和被攻打国的实力有关，还有其他因素\"}",
  "{\"qid\": \"qid_6594100490612694209\", \"category\": \"电脑/网络-操作系统/系统故障\", \"title\": \"XP关机时出现结束\\\"SAMPLE\\\"程序对话框，怎么回事？我的Wi \", \"desc\": \"我的 XP遇到关机问题，当我试图关闭系统时出现“正在结束”框，系统在试图定位一个叫 SAMPLE的程序。但是我找不到这个在我的系统中运行的程序，我也无法关闭系统，怎样对付这个问题呢？\", \"answer\": \"你很可能中了蠕虫之类了病毒了。先用杀毒查查吧。\\r这是2001年的消息：（参考网页： ）\\r如果真是此类病毒可以根据下面内容删除相关文件，或找新版瑞星等杀毒软件查杀。\\r娜妲病毒出现变种\\r　　（台北讯）今年9月肆虐网络的娜妲病毒（Nimda）出现了最新的变种病毒。 \\r\\r　　趋势科技表示，娜妲病毒及其变种病毒感染力极强，是非常凶猛的病毒，皆会透过电子邮件、资源分享、微软IIS及WEB浏览等四种方式来传播，目前美国、韩国及菲律宾分别传出感染灾情。台湾证券交易所和期货交易所的电脑已先后中毒。 \\r\\r　　PE-Nimda.E夹带的附加档案为“Sample.exe”，并会在Windows资料夹置入LOAD.EXE或CSRSS.EXE。 \\r\\r　　趋势科技提醒电脑族别开启附件档名为sample.exe的电子邮件，或采用内建“个人防火墙”功能的防毒软件，以彻底封锁恶意电脑病毒及网络蠕虫，有效保护电脑。 \\r\\r　　趋势科技技术支援部经理康宏旭表示，新变种病毒PE-Nimda.E具有电脑蠕虫（Worm）的特性，会利用数种感染途径来进行繁殖及散播，比如：E-mail、资源分享、微软IISServer，及WEB浏览。 \\r\\r　　透过电子邮件传播时，病毒会夹带“sample.exe”档案，一旦档案被执行时，它会在C磁碟的Windows\\\\Temp资料夹下建立 p.exe暂存档，这个档案是e-mail格式，并且包含蠕虫所夹带的病毒档案。 \\r\\r　　根据资料统计，娜妲病毒在全球造成5亿9000万美元的损失。 \\r\\r　　台湾证券交易所股市观测站系统，在前天5时57分时遭娜妲病毒入侵，使得股市观测站中的重大讯息内容及上市公司财报申报，投资人均无法上网查询，证交所昨天上午表示，股市观测站现已恢复查询功能。 \\r\\r　　证交所表示，证交所经检查后发现，股市观测站遭病毒入侵原因是经由内部电子邮件发送所致，由于股市观测站为证交所内部网络系统，在发现被病毒入侵，证交所立即关闭内部网络进行扫毒。\\r\"}",
  "{\"qid\": \"qid_750050624123334282\", \"category\": \"游戏-光宇游戏-问道\", \"title\": \"天府情缘出问题了晚上天府都进不去了给个说法哦~我还在杀怪哈死了掉 \", \"desc\": \"晚上天府都进不去了给个说法哦~我还在杀怪哈死了掉道了怎么办？那个赔GM给说法哦！！！！\", \"answer\": \"你好\\r\\n服务器如果出了问题 需要维护\\r\\n请耐心稍等 等服务器恢复\"}",
  "{\"qid\": \"qid_2419920079104386978\", \"category\": \"娱乐-博彩\", \"title\": \"任九，让我优，让我喜~~~~~一次次买彩，一次次倾听那比分，一次 \", \"desc\": \"一次次买彩，一次次倾听那比分，一次次兴奋，一次次失落，一次次无归而返！最终还是任8~~~~\", \"answer\": \"咱们都是一样的心情  希望你以后有好的运气 保持好的心态 \"}",
  "{\"qid\": \"qid_1615445641236654883\", \"category\": \"电脑/网络-互联网\", \"title\": \"汉城什么时候改名为首尔，为什么？ \", \"desc\": \"汉城什么时候为首尔，为什么？\", \"answer\": \"首尔（：서울），正式名称为首尔特别市（서울 특별시），是韩国的首都，旧称汉城、汉阳等，「首尔」是韩语「서울」的汉语译音名称。2005年1月18日，该市市议会正式通过把其市的中文名称定为「首尔」，并把官方的所有出版物改用新名称，但部份华语地区还使用汉城。 \\r\\r首尔（서울），旧称南京（남경）、汉阳（한양）、汉城（한성）、京城（경성）。自从1945年二战结束韩国独立以后，当地民族情绪高涨，并将其首都改称为서울（英文为Seoul），韩语是「首都」或「京城」的意思，就如中国的首都称为“京”一样。因此，Seoul成为当时韩国国内唯一没有对应汉字名称的地名。中国大陆早期曾经根据韩语发音而音译作“苏乌”，但其后跟随其他华语地区，沿用这个城市在李氏朝鲜时的旧称“汉城”。然而，其他语言都翻译成“SEOUL”或类似的发音 \\r\\r这使日常沟通出现不少混乱，因为首尔市内有不少建筑物分别使用“首尔”或“汉城”为名称，但翻译成中文之后，这些地方名称会变得很相似。一个例子是：首尔市内同时有“汉城大学”及“首尔大学”，假若采用“汉城”作为 Seoul 的首都名称，会使两家大学的译名都变成了“汉城大学”。后来，中国人根据后者的英文名称，在其中文名称加上“国立”二字作分辨。但其实，首尔大学在韩国的知名度比汉城大学高很多，而大学本身的中文出版物，亦自称为“Seoul大学校”。但对於学校以外的机构，译名的问题就给他们造成困扰。2004年，韩国曾经有人发起运动要求解散首尔大学，韩国多份主要报章由於不清楚中国对两家“汉城大学”的区分方式，而多次在他们的中文版新闻中把首尔大学错译为“汉城大学”，而其企划部的柳根培亦无缘无故被“转校”成为汉城大学的发言人。 \\r\\r韩国政府从1992年起就一直在进行“서울”的中文名称的制订工作，拟订的对应中文包括“首尔”和“首午尔”。2005年1月18日，서울市议会通过使用与“SEOUL”发音相近的“首尔”作为这个城市的中文名称，并要求韩国政府公文、出版物、网站、机场和车站、公路路标的中文版本都使用“首尔”来代替“汉城”。但是迄今为止，在中文裏使用最多的还是具有500余年历史的“汉城”这个名称。香港及台湾的传媒大部份已经转用“首尔”这个新的名称；中国大陆官方正式宣布了改名这个消息，但并不常使用“首尔”这个新名。一些民办传媒及报刊先开始采用“首尔”，一些官方媒体（如新华网）也已经开始使用，还有一些处在过渡期，如中国中央电视台写成“汉城（首尔）”，不过中国大陆绝大部份出版物、媒体、政府及商业机关仍延续旧称。 \\r\\r有不少中国人质疑市议会是否有权更改本国首都的汉语名称，并指如果韩国首都的中文译名改变，将使华人世界对于韩国首都的称呼造成混乱。还有一个重要原因是“首尔”主要是根据现时汉语普通话的译音，但汉字是在汉语各方言间，以及韩、日、越南语的一些时候使用的，如果音译的话，会造成很多使用汉字的地区对“首尔”两字读音发生混乱，如粤语读为sau2 yi5，和韩语读音差别很大。而“首尔”两字按韩语中汉字的读音也成了「수이」（Su-i）。随著语音的发展，若干年后，即使普通话“首尔”两字的发音和韩语也可能对应不上，和众多西方音译词在各处的差别一样，可能会造成汉字使用者的困扰。有人提出如果根据韩语서울采用汉字“西尉”（韩语读作서울，即Seoul）则不会有此问题，可以在使用汉字的地区和时间上保持一致。 \\r\\r然而，韩国方面一直对中国大陆的这些想法非常不解。一来这城市是他们的城市，二来他们亦已多次透过各种沟道来解释这次改变的因由。 \\r首尔是韩国首都汉城根据其英文“SEOUL”的音译所改的新的中文名称。 \\r汉城市长李明博2005年1月19日举行记者招待会，宣布将首都的中文名称由“汉城”改为“首尔”。“汉城”一词不再使用。李明博市长的解说词是：绝大多数国家都将“SEOUL”按照与其英文标记相似的发音来称呼。如：汉语中的华盛顿、伦敦也都是根据这些地名的固有发音来标记的；只有汉城的中文名称一直沿用古代名称“汉城”。李明博市长向中国游说：“首尔”同汉城的韩语发音最为接近，并采用多用于外国地名的常用汉字，中国人也容易熟悉这一新名称。 \\r很明显，李市长是从语音学角度切入问题的。但是如果就事论事的话，李明博等韩国官方人员的解释比较牵强。因为即使改换了“汉城”汉语的名称为“首尔”，也存在着翻译上及使用习惯上的混乱问题。况且，汉语中的外国地名也不都是以发音为根据翻译的，如英国的牛津、剑桥等并非完全是音译，美国的盐湖城(SsltLakeCity)、阿肯色州的小石城(LittleRock)等完全是意译。新西兰首都（Wellington）的官方中译名几乎就是错译——“威灵顿”显然比“惠灵顿”更贴切，但新西兰政府似乎从未提出任何异议。中文名称叫什么？应当尊重中国的传统使用习惯。 \\r据考证，“汉城”的称呼沿用了韩国古代历史王朝的用法。1394年，李成桂将都城从开京迁移到了汉阳，正式命名为汉城(Hansung)。这一名称在汉语中至今已经使用了610年。二战结束以后，韩国将汉城称为韩国语的“首都”(Sieur)，英文音译为Seoul，但是韩国的书面汉字仍然写作汉城。因此，可以说，韩国这次改换首都的汉语名称绝非像表面上解释的那样简单，而是包含深刻的原因和其他方面复杂的考虑。 \\r随着19世纪末民族主义的崛起，韩国国内就出现了不能正确认识本民族文化和客观评价中国文化对韩国民族文化影响，而摆脱汉语文化影响的思潮。韩国在二战以后，民族主义思潮进一步发展，曾以法律规定，以韩国的表音字为专用文字。从1970年起，韩国小学、中学教科书中的汉字被取消，完全使用表音文字。 \\r随着韩国经济的崛起，这种极端的民族主义情绪进一步发展，在1988年汉城奥运会召开之前，韩国政府曾经下令取消所有牌匾上的汉字标记，以强调韩国的民族文化。 \\r只是到了1999年2月，金大中总统才下令部分解除对汉字使用的限制。但对于这种解禁措施，韩国国内也存在着激烈的反对势力，他们担心这种措施将导致汉字的泛滥与韩国文字的消亡。 \\r从某种意义上说，韩国改变“汉城”的中文名字是本国民族主义在新形势下的延续和发展的表现。 \\r汉武帝时曾在朝鲜设立了四个郡。“汉城”是中国人六百年前至今一直习惯称谓的名字。韩国过去也一直用汉字写人名和地名。虽然“汉城”之名是由韩国古代的先人所起，但现代的韩国人总觉得不是本国的名称，总觉得“汉城”与中国有瓜葛，容易让人联想到中国的汉朝。对于汉朝，一些韩国人又缺乏正确的历史观，认为是对朝鲜的侵略。正式在这种不良心理情结的指导下，韩国才有意将“汉城”的中文译名更改为“首尔”。 \\r韩国官方为这次改名的解释，仅仅是表面的，这是以国际惯例和便于国际交往来掩盖更加深层的阴暗心理情结，努力摆脱汉字文化对韩国深厚影响的一种尝试。 \\r叫了610年的汉城一下子改名，叫着真让人感觉别扭！就好比纽约（New Youk）突然让叫“牛月克”。你能习惯吗？我是习惯不了。我在很多方面敬佩和尊重韩国人，但是在这一点上，我B4韩国人。太小家子气了！ \\r不可否认，朝鲜民族在历史上深受日本侵略者的奴役之苦，大力弘扬朝鲜本民族的文化，加强自身民族文化的优势地位，努力摆脱外来文化影响等措施，有可以理解的一面，不应该随意扣上狭隘的帽子。 \\r但是，另一方面，韩国自身在保持和发扬本民族文化的同时，对外来文化，特别是博大精深的中国文化，也应该采取扬弃的态度，不应该不分好坏一概拒绝。其实，博大精深的中华文化对朝鲜民族文化的产生、发展都起到了巨大的贡献作用。 \\r在具体对待这次改名的问题上，韩国有权利更改本国首都的汉语译名，但是中国以及其他汉语权的国家也有权接受或不接受这个新译名。接受与不接受，这也是中国与其他汉语国家应有的权利，韩国应当予以尊重。因为对中国等这些国家来说，不仅仅是一个译法上的问题，而涉及了历史习惯、经济费用等多方面的问题。\\r\"}",
  "{\"qid\": \"qid_9108333640282323746\", \"category\": \"电脑/网络-硬件\", \"title\": \"UPS电源工作原理 \", \"desc\": \"\", \"answer\": \"你好：\\r\\n——★UPS电源就是不间断备用电源，其主要功能为：①、当UPS电源检测到供电电源失去（如停电、断电等）时，会立即启动、转换为内部电池供电（逆变）状态，为设备提供应急工作；②、较高档次的UPS电源，兼做稳压器的作用，可以使输出电源电压处于稳定状态，提高供电电压质量。\"}",
  "{\"qid\": \"qid_1313046378804543985\", \"category\": \"电脑/网络-互联网-上网帮助\", \"title\": \"我在上注册，在中文香港界面就可以用用户名登录成功，在英文界面就说 \", \"desc\": \"在界面就说我登陆失败？Why ?\", \"answer\": \"我记得大陆的Ebay与海外是没有数据连接的. 而且大陆的Ebay也不叫 Ebay 而是叫 易趣 . 网址是   (译音就是 易趣 )\\r\\n\\r\\n如果喜欢Ebay的话. 那么只有去非大陆的Ebay了. 否则就去淘宝吧. 个人感觉大陆Ebay用户没淘宝多\"}",
  "{\"qid\": \"qid_5709676122096504533\", \"category\": \"烦恼-交友技巧\", \"title\": \"11月17?_q毖稚齷了U? \", \"desc\": \"\", \"answer\": \"=11月17?_q毖稚齷了\"}",
  "{\"qid\": \"qid_4026459235138627255\", \"category\": \"商业/理财-股票\", \"title\": \"站在2995以上的高岗我全副武装　满仓!　站在2900以上的高岗 \", \"desc\": \"我全副武装\\r　 满仓!\\r　 站在2900以上的高岗\\r　\\r虽然今年是暖冬\\r　可我现 在真的感觉有点凉\\r　心也惶惶\\r　人 也慌慌\\r　　\\r　我用力探起头\\r频频的向山下张望\\r那战友们的影子哟 \\r在落日的余晖中\\r一下流的东西砸出的阴线\\r越拉越长~~~~~~\\r　　\\r我略微地抬头\\r　　 眼帘中便 是 3000以上的风光 \\r　　 很美,但很凄凉\\r　　 就象圣地西藏.\\r　　 广袤无垠\\r　　 万里无疆\\r在 3000以上的山岗\\r昨天这里万马奔腾\\r还高奏着凯歌\\r但想不到啊\\r今天却是这样的冷清 与凄凉\\r　　 我那可 怜的 股我全副武装\\r　 满仓!\\r　 站在2900以上的高岗\\r　\\r虽然今年是暖冬\\r　可我现 在真的感觉有点凉\\r　心也惶惶\\r　人 也慌慌\\r　　\\r　我用力探起头\\r频频的向山下张望\\r那战友们的影子哟 \\r在落日的余晖中\\r一下流的东西砸出的阴线\\r越拉越长~~~~~~\\r　　\\r我略微地抬头\\r　　 眼帘中便 是 3000以上的风光 \\r　　 很美,但很凄凉\\r　　 就象圣地西藏.\\r　　 广袤无垠\\r　　 万里无疆\\r在 3000以上的山岗\\r昨天这里万马奔腾\\r还高奏着凯歌\\r但想不到啊\\r今天却是这样的冷清 与凄凉\\r　　 我那可 怜的 　　试问用那漆黑的嘴唇\\r怎可能描述出股市遍地是金子！\\r２９００以上的高地，何时再回来！\\r２９００的 阵地\\r　　 怎么就上不来一个救兵\\r　　 倒是有只来自天空的飞鹰\\r　　 偶尔的盘旋上来\\r　　 咕 咕的叫 着~~~~~`\\r　　 不要难过\\r　 　不要难过\\r　　 看那万亿的基金，必会冲上来夺取３０００高峰 \\r　　\\r　　 我感动得热 泪盈眶\\r　　 仿佛看到了亲人\\r　　 可是山下突然的枪声\\r　　 射向这只大鹰 \\r　　 耳边响起:\\r　　 谁敢染指股票，我是老大！\\r　　 坚决打压 ，让小散交出票票！！！ \\r　　\\r　　 我绝望着看看天\\r　　 不知道把谁相信\\r　　 光明正大多好啊\\r　　 苍天为何让我买了60 \", \"answer\": \"你看到了\\\"万点\\\"的前程,有眼光!\\r\\n\\r\\n但是,短期波动还是会让你水溅衣衫,湿(诗)人!\"}",
  "{\"qid\": \"qid_4039207394580751973\", \"category\": \"教育/科学-学习帮助\", \"title\": \"急需~~~~~~~~~~~~~~~~~~~~~~~~~afew和 \", \"desc\": \"a  few 和 few       a ltle和little  有什么不同?\", \"answer\": \"a few 和 few  都是修饰可数名词\\r\\n前者是表示肯定，有一些的意思；后者表示否定，表示几乎没有\\r\\na few boys; few boys \\r\\na little和little 都是修饰不可数名词； \\r\\n前者是表示肯定，有一些的意思；后者表示否定，表示几乎没有\\r\\n\\r\\nlittle ink \\r\\n a little ink \\r\\n\"}",
  "{\"qid\": \"qid_538211041202792726\", \"category\": \"游戏-久游游戏-魔力宝贝\", \"title\": \"关于BL~我们队伍平均70级，2魔1传1弓1格，2次BL都过不了 \", \"desc\": \"我们队伍平均70级，2魔1传1弓1格，2次都过不了，请问大家一般打多少次能过啊？这个是不是大部分得靠运气，帮帮小弟谢谢了！\", \"answer\": \"打佛利波罗十大要点:\\r1.5人5攻宠,攻宠请选择昆虫系的!另外至少要会3级以上战栗(在只有2个影子的时候开始吸BL魔)\\r2.物理系的人尽量带斧,请记得必杀很厉害!另外防御很重要,最好上240.\\r3.魔防很重要:重装请选择本人能装备的最高等级的抗魔装\\r4.技能很重要:请过BL(AKS也一样)的传把技能烧烧高,至少不低于(等级/10)特别是超补!!!魔术也一样!\\r5.补给很重要:请一定在让传教在战斗前补满魔法,自己带好300或400的血瓶,遇到F4连续超强的时候血瓶才能帮你顶过去!\\r6.属性很重要,建议佩带全水属性水晶,选有水属性的攻宠,至少保证200以上敏捷以免被闪,攻击顺序很重要由幻(全火属性用冰杀)---暗(全风属性用火杀)---阴(全地属性用风杀)---旋律(全水属性用陨杀)的顺序杀!\\r由于水属性全克火属性,能有30%的加成,不出意外第一回合就可以把幻影秒了!\\r只有3个影子+BL的情况下还是很容易过的!\\r7.种族问题:请注意F4影子是特殊系的,而昆虫系全克特殊系,而转昆虫系的是什么道具呢???答案是:100个古钱到法兰冒险者旅馆3楼交换到的\\\"秋分之竖琴\\\"!!!变成昆虫系以后可以减少影子对人伤害的20%以上!\\r8.网络问题:请要做大任务的人尽量选人少的时间,比如早上6点到8点,比如非节假日,另外还要选择人少的服务器!可以降低掉线几率!\\r9.指挥问题:请事先决定战斗时候听谁的指挥,以免发生争执,任务失败!\\r10.最后是最重要的一点:运气!\\r运气不好第一回合就乱敏F4放8次超强,运气不好打倒F4吸干影子也能掉线!\\r\"}",
  "{\"qid\": \"qid_4093944749221724847\", \"category\": \"文化/艺术-民俗传统\", \"title\": \"歇后语：打破脑壳不叫痛\", \"desc\": \"\", \"answer\": \"打破脑壳不叫痛---死称好汉\"}",
  "{\"qid\": \"qid_8382927358605856893\", \"category\": \"娱乐-博彩\", \"title\": \"大家发现了一个问题了吗？？关于中奖……江西的彩民，每期的投注额绝 \", \"desc\": \"江西的彩民，\\r每期的投注额绝对不是最多的，\\r而是很少，在全国数偏低。\\r\\r但每期的中奖却是最多的，\\r从往期的，\\r近期的超级大乐透，，\\r\\r第一期中了198注二等奖，\\r第二期中了唯一的一等奖……\\r\\r翻翻以往的中奖纪录，\\r不得不为这一现象所惊叹：\\r\\r江西自古称物华天宝，人杰地灵，\\r所言不虚呀！！\\r\\r\", \"answer\": \"謝謝?侵?如此稱? 江西人,\\r\\n\\r\\n\\r\\n我也是江西人之一,呵呵.\\r\\n\\r\\n\\r\\n可目前??] ?纹谥?00万大?.\\r\\n\\r\\n\\r\\n祝好運連連.\\r\\n\\r\\n\\r\\n\\r\\n\\r\\n\"}",
  "{\"qid\": \"qid_6623662053203796023\", \"category\": \"游戏-单机游戏\", \"title\": \"使命召唤2的问题帮帮我谢谢了！从新浪下载的使命召唤2使用了XEZ \", \"desc\": \"从新浪下载的2 使用了XEZL-GZWX-XQZG-GEGU-EEFC序了号 是不是有问题 ? 单机游戏不能玩了，只能玩多人的。\", \"answer\": \"用虚拟光驱载入游戏盘,屏蔽物理光驱(右键单击盘符->属性->硬件->选择物理光驱->属性->不使用设备)就可以了(第一次安装完要重启)\"}",
  "{\"qid\": \"qid_3596130276328985892\", \"category\": \"游戏-久游游戏-魔力宝贝\", \"title\": \"回归魔力.问练功点.回归魔力电信时长,巨蟹的.请问112的敏魔要 \", \"desc\": \"回归魔力\\r时长,巨蟹的.\\r请问112的敏魔要去哪练?\\r还有,请教现在怎么没有卖料理的?\\r自己做什么好?锅么?\\r拿来卖可以卖多少钱?\\r还有,谁的群能 Q 3\\r找不到朋友玩魔力啊,现在人好少~-~\", \"answer\": \"楼上的两位说的都有点道理\\r\\n不过\\r\\n说起来经验。还是GBL排第一。不过需要带大量料理\\r\\n还有炮灰宠。准备工作麻烦\\r\\n但是\\r\\n在T95一般不用带料理。一般魔完就回。\\r\\n关于料理方面。由于高层T的开放。佣兵/方舟/要塞几乎没人\\r\\n导致料理销量的急剧下降\\r\\n这个时候要做东西的话。还是做装备卖的好==>>不推荐做料理\\r\\n综合：建议在T95先练着。料理不带\\r\\n等待T100的开放。估计时间不会太久\\r\\n\"}",
  "{\"qid\": \"qid_819575706465459004\", \"category\": \"社会民生-法律\", \"title\": \"有限公司可以不设股东会，把董事会作为权力机构吗？如题 \", \"desc\": \"如题\", \"answer\": \"看来回答这个问题的，对公司法都只是了解皮毛。在下回答如下：\\r\\n可以不设股东会的有限责任公司至少有以下几种公司：\\r\\n 1、一人有限责任公司；\\r\\n2、中外合资企业；\\r\\n3、中外合作企业；\\r\\n4、外商独资企业；\\r\\n5、国有独资企业。\\r\\n上述企业依法可以不设立股东会，除此之外，其他种类有限责任公司必须设立股东会，这是法律的强制性规定。请参见：《中外合资经营企业法》《中外合作经营企业法》、《外资企业法》和《公司法》的有关规定。\\r\\n此外，即使设立股东会，在某种意义上，董事会仍是企业的权利机构。同时，也可以通过公司章程和股东会决议的方式，将股东会的部分权利授予董事会。\"}",
  "{\"qid\": \"qid_252650478374433459\", \"category\": \"教育/科学-职业教育\", \"title\": \"2009年湖南会计人员继续教育是什么时候？ \", \"desc\": \"\", \"answer\": \"可看下参考资料 \"}",
  "{\"qid\": \"qid_6895139452061137503\", \"category\": \"育儿-宝宝-婴儿期\", \"title\": \"我的奶水不够吃,怎么增加奶水 \", \"desc\": \"\", \"answer\": \"在某些情况下，哺乳会产生奶水不足的现象，但多半都可加以矫正。比方说：\\r\\r1、过早添加配方奶或其它食品--这是造成奶水不足的主因之一。由于宝宝已经吃了其它食物，并不感觉饥饿，便自动减少吸奶的时间，如此一来，乳汁便自动调节、减少产量。\\r\\r2、喂食时间过短--有些妈妈限制哺喂的次数和时间，比方说规定每四个小时才喂一次，或者每次喂食时间过短等，都会造成母奶产量的减少。事实上，哺喂母奶不必有固定的时间表，宝宝饿了就可以吃。而每次哺喂的时间在理想上也应由宝宝自己来决定，不应加以限制。有时候宝宝的嘴一离开妈妈的乳头，妈妈便立刻假设宝宝已经吃饱了。事实上宝宝可能只是想休息一下、喘一口气(吸奶是很累的，有没有听过「使出吃奶的力气」这句话) ，或是因为好奇心想要观察周围的环境等，妈妈可以再试着喂喂看。\\r\\r3、人工挤乳器损坏或不会使用--有些妈妈想要确定自己的乳汁是否足够，于是便想到可以挤挤看，没想到却挤不出多少东西，于是便以为自己的奶水不够。其实挤奶也是需要时间学习的，更何况宝宝的嘴比任何挤奶器更能有效的移出奶水。有时候有些妈妈已经恢复上班工作，便用挤乳器挤出母乳喂食宝宝，没想到过了一阵子却越挤越少，在这种情况下，请先检查人工挤乳器是否损坏，有时候换一个新的机器便可解决问题。不过请妈妈们注意，由于大多数人工挤乳器并不像宝宝的嘴那般具有增加母乳产量的能力，因此在挤的时候千万保持耐心，慢慢来，多花一点时间就可以了。\\r\\r4、婴儿快速生长期(growth spurts)在婴儿约二至三周、六周、以及三个月大左右，婴儿在此时会经历较为快速的生长阶段，此时宝宝会频频要求吸奶。这可以说是宝宝以其本能在增加妈妈的奶水产量，因此若在此时添加其它食物，反而会妨碍奶水的增加。\\r\\r5、授乳妇营养不良--授乳妈妈应该注意营养，摄取适当的热量，不宜过度减轻体重，以免影响乳汁分泌。最好多食用富含蛋白质的食物，进食适量的液体，并注意营养是否均衡。\\r\\r6 、药物影响--授乳妈妈吃含雌激素的避孕药，或因疾病接受某些药物治疗，有时会影响泌乳量，此时可改采其它方式避孕，或避免使用某些药物等。在就诊时，应让医师知道??正在喂母乳。\\r\\r7、母亲睡眠不足、压力过大--为人母的工作十分耗费精神及体力，有时会影响母乳的分泌。建议妈妈们放松心情，多找时间休息，就可以解决暂时奶水不足的现象。食疗方一：金针豆浆 \\r\\r　　原料：金针菜（黄花菜）30g、鲜豆浆250g、白糖20g \\r\\r　　做法：将金针菜用温水润软，煎去花蒂，洗净，放于砂锅中，注入清水100ml，大火烧开后，加入鲜豆浆和白糖，再共煮10分钟。 \\r\\r　　服法：分2次服，早晚各1次，连服3天 \\r\\r　　功能：清热解毒、利尿运乳，适用于产妇缺乳。 \\r\\r食疗方二：花生鲤鱼汤 \\r\\r　　原料：花生仁60g，鲤鱼1尾（重约500g），生姜5g，料酒5g，酱油3g，化猪油10g，精盐3g，味精0.2g \\r\\r　　做法：1.生姜洗净切丝，鲤鱼刮鳞去腮，剖腹去内脏，砍成块，一起放入碗中，加入料酒、酱油和精盐，拌和均匀，腌渍入味。2.花生仁洗净，放于砂锅中，注入清水500ml，小火煮至酥烂，下腌渍好的鲤鱼块、化猪油，再煮20分钟，加味精，调匀。 \\r\\r　　服法：分2次服。趁热食花生和鱼，喝汤。 \\r\\r　　功效：祛湿利水，醒脾和胃，通乳，适用于产后缺乳。 \\r\\r食疗方三：木瓜猪蹄 \\r\\r　　原料：新鲜番木瓜1只（重约300g），猪蹄1只（重约400g），精盐4g，味精0.5g \\r\\r　　做法：1.番木瓜洗净剖开，去籽，切成3cm小块。2.猪蹄烙净余毛，刮洗干净，砍成小块，放于砂锅中，注入清水800ml，盖严，先用大火烧开，撇去浮沫，转用小火炖至八成酥烂时，加入番木瓜片，继用小火炖至酥烂。下盐和味精，调匀。 \\r\\r　　服法：分2次服，早晚各1次。趁温食番木瓜、猪蹄、喝汤。 \\r\\r　　功效：健脾胃，助消化，下乳汁，适用于产妇缺乳。 \"}",
  "{\"qid\": \"qid_3291007687817816835\", \"category\": \"教育/科学-外语学习-英语考试\", \"title\": \"英语\\\"What'sup?\\\"是什么意思? \", \"desc\": \"\", \"answer\": \"\\\"What's up?\\\"就是问对方近来如何，有点像中文里的“最近怎样”，通常没什么事就会回答\\\"Not much\\\"或者\\\"Nothing\\\"。\\r\\n\\r\\n不过，还有一种情况也很常见，就是对方也回答了一句\\\"What's up?\\\"。在这种情况下，\\\"What's up\\\"几乎相当于\\\"Hello\\\"。\\r\\n\\r\\n\\\"What's up?\\\"也常被用来问人家有什么事，比如别人登门拜访，你就可以用，意为“有何贵干哪”。\\r\\n\\r\\n\"}",
  "{\"qid\": \"qid_3661354683987560684\", \"category\": \"烦恼-恋爱\", \"title\": \"jiaoyu如何培养学生阅读能力 \", \"desc\": \"如何培养阅读能力\", \"answer\": \"通过认真学习《新课程标准》，我深深地体会到：阅读能力，对一个学生来说，是一种十分重要的能力，同时也是学生应该具备的一个重要素质。阅读能力的强弱，与学生获取知识，提高学习兴趣，增长见识，以及培养自学能力等方面都有密切联系。小学生阅读能力提高了不但能够加强小学生在阅读中的情感体验，而且可以让学生积累丰富的语文知识，形成良好的语感；能让小学生去理解、鉴赏作品，从而受到高尚情操与趣味的熏陶；同时也能发展他们的个性，丰富他们的精神世界。我认为必须加强小学生的课外阅读，提高小学生的课外阅读的能力。而目前在农村小学，由于受办学条件的影响，学生的课外阅读量少，导致作文能力低，学生阅读兴趣不高，达不到应有的教学效果。如何针对农村的教学实际情况，提高农村小学生的课外阅读能力，我认为应该从以下几方面入手。\\r　一、激发学生的兴趣，让他们喜爱阅读。\\r　 兴趣是最好的老师，是人们从事任何活动的动力。学生有了阅读兴趣，才能从内心深处对课外阅读产生主动需要。因此，在教学中，我用了一些特殊的方式，来激发学生阅读的兴趣，从而让他们快乐地接受自己想要学习的语文知识。\\r教学中，我经常鼓励学生用自己喜欢的方式进行阅读。如教学《海底世界》一课时，让让学生仔细读一读，想想准备用什么方法来学习。通过思考，有的学生说：我准备一句一句地读，边读边想象海底的各种小动物，不理解的地方多读几遍，并且可以对照插图理解。有的说：我准备和同桌比赛朗读，看谁读得好。还有的说：我想围绕“海底真是一个景色奇异，物产丰富的世界！”这句话展开学习，解决了这个问题，这篇课文也就读懂了。总之，在阅读教学中我常常尊重学生的选择，满足学生的愿望，鼓励他们根据自己的体验用自己的思维方式去探究，从而发展自己的个性。\\r　　二、教给他们读书的方法，让学生学会阅读。\\r　 光有兴趣是远远不够的，我们要进一步引导他们不仅爱读，而且会读，要读得更好更有收效。作为教师，我们应该向学生介绍一些行之有效的读书方法。\\r⒈略读法。这种方法的运用一般是根据学生在课内学习或写作上的某种需要，有选择地阅读有关书报的有关篇章或有关部分，以便学以致用。如学习了《太阳》一课，学生为了更全面、更深刻的了解人类与太阳、自然的密切关系，有目的地阅读《人与自然》一书(或上网查阅)。而且要指导学生去图书馆借阅相关的资料，再选择自己所需要的部分，以培养摘录资料和运用资料的能力。同时也丰富了语言积累，提高了阅读兴趣和阅读能力。\\r　⒉精读法。所谓精读法，就是对书报上的某些重点文章，集中精力，逐字逐句由表及里精思熟读的阅读方法。它是培养学生阅读能力最主要最基本的手段。有的文章语言隽永，引经据典，情节生动。我们可以利用这些作品为依据，指导学生精读，要求学生全身心投入，边读、边想、边批注，逐渐养成认真读书的好习惯。\\r　⒊默读法。默读法就是对所读的书报，不发音、不辨读、不转移视线，而是快速地观其概貌。这就要求学生在快速的浏览中，要集中注意力，作出快速的信息处理和消化。利用默读法，可以做到用最少的时间获取尽量多的信息。\\r　⒋摘抄评价法。此阅读法就是在阅读过程中根据自己的需要将有关的词、句、段乃至全篇原文摘抄下来，或对阅读的重点、难点部分划记号，作注释，写评语。文章中富有教育意义的警句格言、精彩生动的词句、段落，可以摘录下来，积存进自己设立的“词库”中，为以后的作文准备了丰富的语言积累。同时还可以对自己订阅的报刊杂志，将其中的好文章剪裁下来，粘贴到自己的读书笔记中。\\r　三、创造条件，让学生快乐阅读。\\r    我要努力为学生创造阅读的条件，让学生在自由的空间，幽雅的环境中舒舒服服的进行阅读。\\r   每个学生的兴趣、个性特点不尽相同，因此他们所习惯的呈现阅读体验的方式也不一样：有的喜欢朗读，有的喜欢表演，有的喜欢表述……我们应该尊重他们的选择。如阅读了《草原》以后，我鼓励学生说：同学们的感受一定很深，请你用自己喜欢的方式表达出来。于是，他们有的深情地诵读，有的和着音乐翩翩起舞，有的高歌一曲，有的写起了小诗，有的和同桌交流……课堂成了学生积极参与、主动表现的舞台。\\r　 四、积极评价，让学生对阅读有不同的看法。\\r   阅读教学应提倡标新立异，允许学生见仁见智地解读作品，对作品进行创造性的思考。\\r   总之，在提高阅读能力时我们要处处让学生体验成功的喜悦。当学生在获得成功后，会更坚持大量的广泛的阅读，良好的读书习惯也就随之形成。同时个人的进步往往又是同学们效仿的范例，这样班里就会形成浓厚的课外阅读氛围，学生的阅读能力也会随之提高。\"}",
  "{\"qid\": \"qid_7324298667277705568\", \"category\": \"电脑/网络-软件-图像处理软件\", \"title\": \"CAD软件的角度取值为整数，1度，2度，3度，4度没有2.1度,? \", \"desc\": \"CAD的角度取值为整数，1度，2度，3度，4度没有2.1度,2.1度,可已设置吗!!!!!!画个52.606度的角画不了!?????\", \"answer\": \"   可以设置！\\r\\n   先正常画出角度，在标注完角度值后，可以点击鼠标右键选择“精度”，再选择合适的精度位数。\\r\\n   也可以在标注样式里把标注精度统一调整，这样以后所有的标注都可以按照这个精度标注了。\"}",
  "{\"qid\": \"qid_2680849588140160855\", \"category\": \"社会民生-法律\", \"title\": \"物流公司发货不按照协议发货照成货物损坏怎么赔偿 \", \"desc\": \"物流发货不按照协议发货照成货物损坏怎么赔偿\", \"answer\": \"一般按照货物价值赔偿。有空加个微博好友给个好评啊！\"}",
  "{\"qid\": \"qid_508534880557955811\", \"category\": \"社会民生-其他社会话题\", \"title\": \"会开车的好处我是个大学生，请问有驾驶证对以后找工作帮助大吗？ \", \"desc\": \"我是个，请问有驾驶证对以后找工作帮助大吗？\", \"answer\": \"你好！\\r\\n\\r\\n艺不亏人，只要是技术学会总比不会好。有驾驶证对以后找工作应该有帮助，如果你打算开车，帮助非常大啊！\\r\\n\\r\\n建议：在读期间还是专心致志的学习你的专业为好。办驾驶证还得住驾校，住驾校得花钱、每年年检还得花钱。如果你的家庭条件不错而且是学习之余有足够的时间也可以办，否则还是毕业后再做打算吧。\"}",
  "{\"qid\": \"qid_8193953555975861150\", \"category\": \"文化/艺术-书画美术\", \"title\": \"何谓普普?来历? \", \"desc\": \"\", \"answer\": \"普普风，美国纽约的时尚休闲品牌； \\r\\n  \\r\\n英文商标FGFEEL是FEEL GOOD FEEL（感受好心情）的缩写，这也正是普普风所倡导的生活理念——自由，阳光，健康，爽朗； 普普风以22-35岁时尚一族的年青人为消费对象，价位适中，它的成衣大多采用进口的休闲布料和具有独特工艺的高科技面料，加上其全新的设计理念——把时装化的合体裁剪和流行元素融入到轻松、自由的休闲之中，从而提高了休闲的品味，也拉近了休闲和时尚之间的距离。产品给人感觉细腻丰富，从服装的设计与裁剪上散发出不可抗拒的时代气息与魅力，整个品牌给人一种大器，加上时尚与稳重融合的美妙感觉，必将受到崇尚新生活的都市时尚一族的青睐； 普普风于2000年8月正式抢滩中国，目前正在深圳市飘二千服装有限公司科学的管理和运作之下，以强大的资金和所倡导的全新的休闲文化为后盾，以纽约，香港两个时尚前沿城市为龙头，向中国的服装市场进发！ 来吧！让我们一起来享受普普风，享受好心情，享受无边的自由！ \"}",
  "{\"qid\": \"qid_5656001177274245606\", \"category\": \"游戏-网络游戏\", \"title\": \"问道游戏有什么办法可以快速刷友好度的 \", \"desc\": \"\\r\\n有什么办法可以快速刷友好度的\", \"answer\": \"开LD打卡一回合清怪即可以打到卡又可以增加友好度。每杀一次怪增加1点友好度 当然有钱的话送玫瑰巧克力等等最快了。胡子任务每天只能做一次！全部综合的话很快就够好友度结婚闹洞房了！！！ \"}",
  "{\"qid\": \"qid_3449485945632669470\", \"category\": \"生活-购物\", \"title\": \"市区哪里还有卖北京冰糖葫芦的[抓狂][抓狂][抓狂][抓狂]？？? \", \"desc\": \"市区哪里还有卖北京冰糖的[抓狂][抓狂][抓狂][抓狂]？？？求解！\", \"answer\": \"牌坊街\"}",
  "{\"qid\": \"qid_624441425152161418\", \"category\": \"生活-购物\", \"title\": \"请问各位大虾？北京哪里有卖橄榄油的？我想买，不知哪里有卖的，那位 \", \"desc\": \"我想买，不知哪里有卖的，那位知道请告诉我。谢谢。\", \"answer\": \"学院路学府园站有一家普尔斯马特超市，里面就有正宗橄榄油卖。\"}",
  "{\"qid\": \"qid_7381878519240599315\", \"category\": \"游戏-网易游戏-魔兽世界\", \"title\": \"50+级最好的双手武器想在50+级的战场呆一段时间,所以求个50 \", \"desc\": \"想在50+级的战场呆一段时间,所以求个50+级的好双手(本人战士)\", \"answer\": \"命运不错，还有就是碎灵\\r\\n\\r\\n最好的60前的双手武器，是厄运贡品 的古树天敌\\r\\n古树天敌\\r\\n拾取后绑定  \\r\\n双手  斧\\r\\n128 - 193伤害  速度 2.70\\r\\n（每秒伤害59.4）  \\r\\n+25 力量  \\r\\n+15 耐力  \\r\\n需要等级 58  \\r\\n装备： 使你造成致命一击的几率提高2%。  \\r\\n     \\r\\n慢慢去刷吧\"}",
  "{\"qid\": \"qid_2339683591001722848\", \"category\": \"商业/理财-基金\", \"title\": \"1.大宗交易的金额与数量有何规定？（一）A股交易数量在50万股（? \", \"desc\": \"1. 大宗交易的金额与数量有何规定？ \\r\\r    （一）A股交易数量在50万股（含）以上，或交易金额在300万元（含）人民币以上；B股交易数量在50万股（含）以上，或交易金额在30万美元（含）以上； 　　（二）交易数量在300万份（含）以上，或交易金额在300万元（含）人民币以上； 　　（三）债券交易数量在2万手（含）以上，或交易金额在2000万元（含）人民币以上。 　　（四）债券回购交易数量在5万手（含）以上，或交易金额在5000万元（含）人民币以上。 \\r \\r    在上海证券交易所（以下简称本所）市场进行的证券单笔买卖达到如下最低限额，可以采用1. 大宗交易的金额与数量有何规定？ \\r\\r    （一）A股交易数量在50万股（含）以上，或交易金额在300万元（含）人民币以上；B股交易数量在50万股（含）以上，或交易金额在30万美元（含）以上； 　　（二）交易数量在300万份（含）以上，或交易金额在300万元（含）人民币以上； 　　（三）债券交易数量在2万手（含）以上，或交易金额在2000万元（含）人民币以上。 　　（四）债券回购交易数量在5万手（含）以上，或交易金额在5000万元（含）人民币以上。 \\r \\r    在上海证券交易所（以下简称本所）市场进行的证券单笔买卖达到如下最低限额，可以采用大宗交易方式： \\r\\r    2.投资者如何进行大宗交易？ \\r\\r    拥有本所专用席位的机构可通过该席位进行大宗交易。\\r\\r　　其他投资者进行大宗交易，应委托其办理指定交易的本所会员办理。 \\r\\r    3.大宗交易的交易时间如何规定？\\r\\r    大宗交易的交易时间为本所交易日的15:00-15:30，本所在上述时间内受理大宗交易申报。 　　大宗交易用户可在交易日的14：30-15：00登陆本所大宗交易电子系统，进行开始前的准备工作；大宗交易用户可在交易日的15：30-16：00通过本所大宗交易电子系统查询当天大宗交易情况或接受当天成交数据． \\r\\r    4．哪些证券不能进行大宗交易？ \\r\\r    本所交易日的15:00前仍处于停牌状态的证券，本所不受理其大宗交易的申报。 \\r\\r    5．大宗交易的申报包括哪些内容？ \\r\\r    大宗交易的申报包括意向申报和成交申报。 意向申报应包括以下内容：\\r\\r 　　（一）证券代码； 　　（二）股东帐号； 　　（三）买卖方向； 　　（四）本所规定的其他内容。 　　意向申报中是否明确交易价格和交易数量，由申报方决定。 　　买方和卖方就大宗交易达成一致后，代表买方和卖方的会员分别通过各自席位（拥有本所专用席位的机构通过该席位）进行成交申报，成交申报应包括以下内容： \\r\\r    （一）证券代码； 　　（二）股东帐号； 　　（三）成交价格； 　　（四）成交数量； 　　（五）买卖方向； 　　（六）买卖对手方席位号； 　　（七）本所规定的其他内容。 　　买方和卖方的上述成交申报中，证券代码、成交价格和成交数量必须一致。 \\r\\r    截止到2008年3月，我国A股市场解禁股份数已达到800亿股之多，从2006年6月份算起，平均每月解禁数量约38亿股。以解禁时点个股股价动态测算，已累计解禁金额约12639亿元，平均每月约解禁601亿元\\r\\r    目前阶段，减持对市场的影响更有可能是通过“挤出效应”来体现的，它占用了全市场一定的资金量，但对个股自身影响并不显著。减持个股需要新的资金承接，这就导致了其它个股上资产配置的下降，从而引发其它个股乃至市场下跌，这与宏观整体观察相一致。此外，目前阶段，减持对市场更多的是一种“心理效应”，减持规模的积累令投资者形成一种心理上的担忧。 \\r\\r    2008年4月起将解禁股数约676亿股，平均每月75亿股，全年峰值出现在8月，月解禁214亿股，月解禁规模是过往水平的近两倍。以2008年3月28日收盘价数据测算，全年解禁金额大约是10546亿元，月解禁金额约1171亿元。如果以过往减持占解禁比例为9%数据测算，每月减持股数约6.8亿股，减持金额大约是105亿元\\r\\r    6. 意向申报时应注意哪些问题？\\r\\r　　意向申报应当真实有效。申报方价格不明确的，视为至少愿以规定的最低价格买入或最高价格卖出；数量不明确的，视为至少愿以最低限额成交。\\r\\r  　买方和卖方根据大宗交易的意向申报信息，就大宗交易的价格和数量等要素进行议价协商。 　　当意向申报被其他参与者接受（包括其他参与者报出比意向申报更优的价格）时，申报方应当至少与一个接受意向申报的参与者进行成交申报。\\r\\r    7. 大宗交易的成交价格如何确定？\\r\\r　　大宗交易的成交价格，由买方和卖方在当日最高和最低成交价格之间确定。该证券当日无成交的，以前收盘价为成交价。\\r\\r    8. 大宗交易的成交申报是否能撤消或变更？\\r\\r　　大宗交易的成交申报须经本所确认。本所确认后，买方和卖方不得撤销或变更成交申报，并必须承认交易结果、履行相关的清算交收义务 \\r\\r    9. 进行大宗交易的帐户（证券帐户和资金帐户）是否应有足额的证券或资金？\\r\\r　　会员应保证大宗交易投资者帐户（包括会员自营账户）实际拥有与意向申报和成交申报相对应的证券或资金。 \\r\\r    10.大宗交易的费用如何收取？\\r\\r　　大宗交易的交易经手费按集中竞价交易方式下同品种证券的交易经手费率标准下浮（其中，股票、基金下浮30%，债券、债券回购下浮10%）， 其它费用同集中竞价交易。 \\r\\r    11.大宗交易的信息如何披露\\r\\r　　大宗交易收盘后，本所在指定媒体上公布每笔大宗交易的成交量、成交价以及买卖双方所在会员营业部的名称（通过本所专用席位达成大宗交易时，为该专用席位所属机构的名称）。大宗交易成交后涉及法定信息披露要求的，买卖双方应依照有关法律法规履行信息披露义务。 \\r\\r    12.大宗交易是否纳入指数计算？\\r\\r　　大宗交易不纳入指数计算， \\r\\r    13.大宗交易的价格是否作为该证券当日的收盘价？\\r\\r　　大宗交易的成交价不作为该证券当日的收盘价，\\r\", \"answer\": \"谢谢!又学了一些新知识.\"}",
  "{\"qid\": \"qid_7086440755587918295\", \"category\": \"生活-生活常识\", \"title\": \"谁用过“水宜生杯”,真的是那么有作用吗?我看电视上的宣传广告，觉? \", \"desc\": \"谁用过“水宜生杯”,真的是那么有作用吗?\\r\\r我看上的宣传广告，觉得有点不可思议。所以上来问问。\", \"answer\": \"听了广告都想信的话，癌症能治愈、糖尿病能治愈、高血压不要长期服药……没有哪种病不能治的，那中国的医院都要关门，医生都要改行了，中国人平均寿命早已领先世界了。\"}",
  "{\"qid\": \"qid_4243935411392610351\", \"category\": \"商业/理财-财务税务\", \"title\": \"退税（营业税，城建税，教育费附加，所得税）分别怎么做帐？ \", \"desc\": \"\", \"answer\": \"国家税务总局规定：\\r\\n收到退回的流转税（增值税、营业税、城建税、教育费附加等）贷：补贴收入；\\r\\n收到退回的企业所得税，贷：所得税。\"}",
  "{\"qid\": \"qid_8472921637205595238\", \"category\": \"游戏-网络游戏\", \"title\": \"我这样的配置会不会卡？游戏登不上去，没试过不知道XP2500+O \", \"desc\": \"2500+ OC 3200+\\rKMS 512M DDR400\\r盈通9800PRO\\rST SATA 7200/8M\\r\", \"answer\": \"除非网络问题，否则肯定不会卡的\\r\\n效果中等也能很流畅进行~\\r\\n系统优化的好的话，开最大效果也是没问题的\"}",
  "{\"qid\": \"qid_5642645166941049746\", \"category\": \"生活-交通出行-问路\", \"title\": \"深圳至赣州怎样走请问深圳开车到赣州应该怎样走啊？请各位可以详细告 \", \"desc\": \"请问深圳到赣州应该怎样走啊？请各位可以详细告诉吗？\", \"answer\": \"从广州走：从广州--惠州--河源--和平--定南--赣州,到了赣州，高速路一路有，指示到南昌只要600余公里，全程高速。 \\r\\r高速具体路段如下: 广州北环--沙河立交--白云隧道--广氮出口--广深高速--火村出口--广惠高速--惠河高速--阿深高速(粤赣高速), \\r\\r千万注意: 1、在广氮出口有指示牌标明广惠,广深高速,下来后,一路都有路标指示是广深高速,不必惊慌，到了火村出口就要注意,一条路去广惠,一条路去广深,走错了就往东莞方向去了,我一同事元旦时就走错了,结果到了东莞再掉头,多走了近百公里。 2、还有就是,上广惠高速到了小金口,下了广惠高速后,有指示牌指明惠河高速,也易走,上了惠河高速后,去赣州就一直走,不要下任何出口就行了,路线都有很明显的标牌,不会错. 3、到了赣州后，有指示牌指明如何去吉安、南昌。 \\r\\r另：粤赣高速正式名称为阿深高速，终点为深圳，起点应该是和平县与江西定南县之间的一个小镇，全程粤赣高速都被标为阿深高速，所以各位不要以为走错了。 \\r    全程限速100公里/小时。 另江西境内隧道较多，限速80还要开大灯。 \\r\\r从深圳走就是直接从梅观上机荷，再转上深惠，再上惠河，就可以直接上阿深高速了。 \\r\"}",
  "{\"qid\": \"qid_5865257258993981788\", \"category\": \"健康-内科-神经内科\", \"title\": \"遵义市治疗女性癫痫需要注意患者的哪些 \", \"desc\": \"市治疗女性癫痫需要注意患者的哪些\", \"answer\": \"遵义市治疗女性癫痫需要注意患者的哪些，有人统计在妊娠前已控制癫痫发作者，50%的人仍可保持不发作，5%的人发作减小，45%的患者发作回数增多，如果在妊娠前发作多于每月1次，其妊娠时则有60%～98%的发作增多，并常在头3个月内发作恶化。妊娠期癫痫变化的原因至今尚未清楚，可能与妊娠毒血症、血管损害、静脉栓塞、妊娠引发脑病和抗癫痫药血浓度下降等有关。有报告癫痫患者妊娠时并发症发病率能增加50%，如孕期女性生殖道流血、妊娠毒血症、早产、剖腹产、自发流产及羊膜炎等，而且婴儿死亡率也高。因此应当提倡神经科和医师密切配合，病人应选择较好的分娩。遵义市治疗女性癫痫需要注意患者的哪些，女性承担着生育下一代的重任。对于女性癫痫患者来讲，怀孕期间不能盲目服药，否则不仅会引起癫痫发作更会对胎儿造成伤害。因此，女性患者若在妊娠期间癫痫发作，一定要谨慎治疗，并在医生的指导下服药，以免引起胎儿畸形。遵义市治疗女性癫痫需要注意患者的哪些，月经周期的变化主要是性激素的水平在体内发生周期性改变，而性激素本身，特别是雌激素对癫痫发作影响是明显的，这样癫痫发作也会随着月经周期的变化而变化。月经加重发作在症状性癫痫中更为突出，那些已经有频频发作者、或有月经前紧张者，更可能在月经前发或在月经性流血时发作。此外，也有一些女性癫痫患者的发作，只出现在月经前或月经期，称之为月经性癫痫。\"}",
  "{\"qid\": \"qid_2561773807125585731\", \"category\": \"健康-外科-肛肠外科\", \"title\": \"工作太累会造成便秘吗?整天干重体力劳动,大便老是不通,排不净 \", \"desc\": \"整天干重体力劳动,大便老是不通,排不净\", \"answer\": \"工作不会直接影响大便，但是由于工作是定时的，当在工作的时候往往会忽略便意，而便意是很短暂的，如果没有及时去解手，就不再有大便的感觉了。时间久了，肠子的运动功能会削弱。\\r\\n\\r\\n另外，身体虚弱的人，包括亚健康的人，内脏功能较差，肠蠕动也慢，常常伴有体虚型便秘，这时候如果用泻药犹如雪上加霜，从中医角度来说，正确的方法是以补促泻，也就是通过滋补让身体强壮，提高肠子运动力，达到治本的目的。\\r\\n\\r\\n干重体力活的人要注意饭后休息至少半小时，在这段时间培养便意，特别是早饭后不论有没有便意都要养成上厕所的习惯，坚持一段时间会让肠子产生反射，便秘问题就可以从根本上解决了。\\r\\n\\r\\n良好的排泄功能于健康很重要，所以，为了健康快快调整生活习惯吧，祝你健康。\"}",
  "{\"qid\": \"qid_6378427847623797285\", \"category\": \"游戏-网络游戏-街头篮球\", \"title\": \"牛C进来回答下上次在4区看到个会后中投后3分的C,不知道怎么会的 \", \"desc\": \"上次在4区看到个会后中投后3分的,不知道怎么会的,哪个人可以告诉下?\", \"answer\": \"BUG\\r\\n进了不算分\\r\\n具体操作：\\r\\n首先上篮，按两下DD键（不要再按D键了）\\r\\n然后等下一次进攻，按W挤人，然后轻按D键做假动作，不停的方向转圈，当人物转到正朝篮下的时候，不停的按D键的，投篮出去\\r\\n可能有时候发不出来，可以多试\"}",
  "{\"qid\": \"qid_5432373825202796041\", \"category\": \"商业/理财-股票\", \"title\": \"帮分析：600527江南高纤今天是否为上涨前的蓄势调整吗？ \", \"desc\": \"帮：600527江南高纤今天是否为上涨前的蓄势调整吗？\", \"answer\": \"该股今天上涨不是蓄势调整,而是反弹出货,止赢.\"}",
  "{\"qid\": \"qid_5212414270058077095\", \"category\": \"体育/运动-足球\", \"title\": \"郝海东现在干什么了？怎么也不听报道了？ \", \"desc\": \"怎么也不听报道了？\", \"answer\": \"郝海东现在有两个角色，一是湖南湘军的老板，还有在外国的一个不知名的球队踢球！\"}",
  "{\"qid\": \"qid_5333488564465164841\", \"category\": \"健康-妇产科-产科\", \"title\": \"怀孕什么时候能感觉到？初期症状都有什么？怀孕什么时候能感觉到？初? \", \"desc\": \"什么时候能感觉到？初期症状都有什么？怀孕什么时候能感觉到？初期症状都有什么？\", \"answer\": \"1、停经：月经过期10日以上怀疑妊娠.2、早孕反应：60％以上的妇女在妊娠6周以后会出现畏寒、头晕、乏力、流涎、嗜睡、食欲不振、喜食酸物或厌食油腻、恶心和晨起呕吐等现象，称早孕反应，持续2个月后自行消失\\n 3、尿频。\\n 4、乳房变化：查体时可见乳房逐渐变大，有胀痛感。\\n 5、妊娠早期乳房增大明显，充血水肿，随着乳腺管的增生，皮下浅静脉清晰可见。6、受精卵着床部位变软，妊娠6－7周后双合诊检查子宫峡部极软，子宫体和宫颈似不相连，称黑加征，是早期妊娠的特征性变化。\\n 7、B超早期检查最早以妊娠囊为判断妊娠标准。\\n 8、超生多普勒检查\\n 9、宫颈粘液检查未见羊齿植物叶状结晶，妊娠的可能性大。\\n 10、妊娠试验：妊娠7－9日后可以查妇女血β－HCG判断妊娠。\\n 11、基础体温BBT的测定：如果高温相持续18日不降，妊娠可能性大；如果持续3周不降，妊娠的可能性更大。\"}",
  "{\"qid\": \"qid_8022711795984531330\", \"category\": \"生活-生活常识\", \"title\": \"吃什么生精比较快吃什么生精快，而且易食用 \", \"desc\": \"吃什么生精快，而且易食用\", \"answer\": \"多食鱼鳔、鱼子、韭菜、小河虾、虾仁、干果仁。\"}",
  "{\"qid\": \"qid_4925683491148619350\", \"category\": \"商业/理财-股票\", \"title\": \"清泉点评之39“角度与变线”以前是准备放在80讲以后进行的，也就 \", \"desc\": \"    以前是准备放在80讲以后进行的，也就是说，准备在介绍完基本的技术分析之后。但大盘运行至目前的敏感阶段，个股纷纷脱离原有通道或上升趋势线，走出了快速上扬行情。容易产生两个误区，一是以原有趋势线为参照，实质就是坐电梯（个股强势两至三根中大阳后会步入调整，有些还会下跌回原趋势线）。二是短线以自己的止赢为目标（比如说8%，10%等等），致使吃不足强势波段，浪费了强势机会和资金使用效率。\\r    \\r    在江恩的技术分析中，十分注重的上升角度。主要参照的有60度，45度，30度。\\r60度以上上升角度的，一般持有为主，不具备做波段的能力。也就是    以前是准备放在80讲以后进行的，也就是说，准备在介绍完基本的技术分析之后。但大盘运行至目前的敏感阶段，个股纷纷脱离原有通道或上升趋势线，走出了快速上扬行情。容易产生两个误区，一是以原有趋势线为参照，实质就是坐电梯（个股强势两至三根中大阳后会步入调整，有些还会下跌回原趋势线）。二是短线以自己的止赢为目标（比如说8%，10%等等），致使吃不足强势波段，浪费了强势机会和资金使用效率。\\r    \\r    在江恩的技术分析中，十分注重的上升角度。主要参照的有60度，45度，30度。\\r60度以上上升角度的，一般持有为主，不具备做波段的能力。也就是说，往往在K线图上的表现形式为，依托5日或10日均线为上升趋势线上扬，只要不有效跌破上升趋势线即可谨慎持有。短线加速时，以涨停方式突破，次日列分时均线止赢，一般会产生不错的利润结果。\\r    45度左右角度上扬的，一般为个股通道运行的常态，总体强势为主，下轨在20日均线附近，上轨为两个重要高点的连线。在回落至通道下轨时，可做短线加仓操作；上行至通道上轨附近时，视成交量的变化，以短线部分减仓为主。个别的，沿趋势线上扬，K线形式上阴阳交错。总体以持有为主。\\r    30度角上扬的，在趋势线上以此30日均线为主，形态上以波浪上扬和台阶上扬为主。类似的个股运行趋势，在操作上参照通道上轨压力及个股运行规律，小波段操作，一般会产生比单纯持股更多的收益效果。\\r    \\r    在上述角度运行的个股，会随着阻力位的有效突破、买气的增加、基本面的改善，发生由量到质的转变，会有改变原趋势线运行的走势出现，这就是变线。小角度的会向大角度变线，强势的个股还有抛物线走势出现。在变线后的两天，一般会有阳线越来越大、6日BIAS指标值超越12、个股走势呈加速形态，这个时候，一般就是列分时均线止赢操作或15分钟线止赢操作的时机了，在分时图上，股价分时有效跌破分时均线同时不能有效站上，或15分钟K线图中MA5与MA10有效死叉时，减磅出局，都可以相对确保个股短线加速时的利润。以600210为例，上述短线操作提示卖出时，可以减磅在8.33元至8.40元附近。\\r    也有一些大角度上扬的个股，在短线乖离率增大时（离中期均线较远），会有一定回档中期均线的要求，往往在个股运行的中位产生一定的震荡，也就是常见的W形、三角型、旗型、矩形（小箱体）的整理形态，之后，按低一个角度的趋势线上扬。目前部分基金重仓，二线蓝筹，运行的就是类似走势。可按中小角度上扬的操作方法即可。\\r\\r\", \"answer\": \"先收藏了，我水平太差，一知半解，不过会好好学习的，谢谢！ \"}",
  "{\"qid\": \"qid_2285648232270123155\", \"category\": \"游戏-手机游戏\", \"title\": \"学手机游戏开发有前途吗？悦成学校好么？现在网络上关于就业，关于手 \", \"desc\": \"现在上关于就业，关于手机游戏开发方面的信息好像很火啊，学手机游戏开发到底有前途没？有一个叫悦成的手机方面的公司经常在眼前出现，它是做什么的？\", \"answer\": \"以前是每个人都有一部手机，现在是每个人基本都有一部智能手机，手机市场会越来越开阔，越来越智能，智能手机背后需要有更多的开发人才进行软件、游戏开发，才凸显智能手机的优势，所以手机游戏开发行业前途无量，不容置疑。\"}",
  "{\"qid\": \"qid_2648433068506518033\", \"category\": \"生活-生活常识\", \"title\": \"有谁知道妆点一生的洗发水去屑效果怎么样？ \", \"desc\": \"\", \"answer\": \"我用过他家产品，效果还不错。\"}",
  "{\"qid\": \"qid_6997163851468800167\", \"category\": \"电子数码-手机/通讯\", \"title\": \"诺基亚E63最低电量指示空格还是一格是不是没电无法开机的时候摁开 \", \"desc\": \"是不是没电无法开机的时候 摁开机键 中间方框按键亮 但无法开机\", \"answer\": \"能显示的最低电量是1格…当电池电量不足以开机时就像你说的闪一下就会变黑了。\"}",
  "{\"qid\": \"qid_7369656993833514153\", \"category\": \"教育/科学-院校信息\", \"title\": \"山东最好的厨师学校 \", \"desc\": \"\", \"answer\": \"山东新东方烹饪学校吧\"}",
  "{\"qid\": \"qid_7320590726058808053\", \"category\": \"游戏-网络游戏\", \"title\": \"末日轻装套3章会涨还是会降，幅度大不大，懂市场的哥哥姐姐能说说吗? \", \"desc\": \"大概会多少钱一套，谢谢了哦\", \"answer\": \"肯定会降了　　但是幅度不会太大　　　３章开了　　末日轻　就会象２章的飞龙一样　普遍　　　　大家都去搞新衣服～～～\"}",
  "{\"qid\": \"qid_3882306860161199895\", \"category\": \"娱乐-博彩\", \"title\": \"256元冲14场火锅而去01切尔西VS维　拉3102西汉姆VS雷 \", \"desc\": \"01  VS 维　拉 31 \\r02 西汉姆 VS 雷　丁 31\\r03 热　刺 VS 富勒姆 3 \\r04 伯明翰 VS 米德尔 31\\r05 德比郡 VS 利物浦 0 \\r06 埃弗顿 VS 博尔顿 3\\r07 桑德兰 VS 曼　联 0 \\r08 维　冈 VS 纽卡斯 3\\r09 朴茨茅 VS 阿森纳 10\\r10 曼　城 VS 布莱克 31 \\r11 切尔西 VS 纽卡斯 3\\r12 热　刺 VS 雷　丁 3 \\r13 西汉姆 VS 曼　联 10 \\r14 埃弗顿 VS 阿森纳 31 \\r\", \"answer\": \"06 埃弗顿 VS 博尔顿 31\\r\\n08 维　冈 VS 纽卡斯 310 \\r\\n10 曼　城 VS 布莱克 30\\r\\n祝你好运\\r\\n\"}",
  "{\"qid\": \"qid_6061260557426028915\", \"category\": \"游戏-单机游戏-反恐精英\", \"title\": \"用STEAM打1.6进有CD的服务器进去后一会就不能动了,请问怎? \", \"desc\": \"以前出现过此问题,我更新了和CS的版本后正常,今天CS已经更新了，但CD没发现有更新?我的是4.31.0,如是CD版本问题请告诉我哪有更新的?\", \"answer\": \"cs.sga.com.cn就有是中文网站~\"}",
  "{\"qid\": \"qid_4899492148390597084\", \"category\": \"生活-美容/塑身\", \"title\": \"怎么把脸洗干净？我怎么觉得每天我的脸洗完后，还不是很干净。皮肤也 \", \"desc\": \"我怎么觉得每天我的脸洗完后，还不是很干净。\\r\\r\\r也不是很好。\\r\\r怎么办？\", \"answer\": \"最好用洗面奶,带沙粒的,对油性皮肤,还有干燥掉皮的皮肤效果不错.\"}",
  "{\"qid\": \"qid_5401910109406337381\", \"category\": \"教育/科学-人文学科-历史学\", \"title\": \"35年前的今天，一个伟人过世了。请您在此说一句心里话＠ \", \"desc\": \"请您在此说一句心里话＠\", \"answer\": \"1976年1月8日共和国第一任总理周恩来在北京逝世，终年78岁。\"}",
  "{\"qid\": \"qid_6950055077148625728\", \"category\": \"烦恼-恋爱\", \"title\": \"初恋小毛孩的疑惑她答应做我女朋友,却不愿公开,为什么呢?是因为怕 \", \"desc\": \"她答应做我女,却不愿公开,为什么呢?是因为怕舆论还是不相信我!\", \"answer\": \"你们都还不大吧！！\\r\\n她既然答应你了饿，我想就没有骗你的必要吧。女生做什么事情一般都是很有原则的，不会拿自己的幸福开玩笑，如果你真心喜欢她，就不该怀疑她\\r\\n或许是她怕知道人太多后，父母和老师也容易只，这样对你俩都不好吧！！！\\r\\n不过，这种问题，你最好和她心平气和的好好谈清楚，沟通最重要嘛！！！瞎想只是增加自己的烦恼，对你们感情的伤害，不会有任何好处的·！·！\"}",
  "{\"qid\": \"qid_3331013172446325757\", \"category\": \"体育/运动-足球\", \"title\": \"请输入您的问题...山东鲁能泰山足球队的世界俱乐部排名是多少? \", \"desc\": \"山东鲁能泰山队的世界俱乐部排名是多少?\", \"answer\": \"前一百名里面没有，追上皇马是不可能的。好比皇马是硕士，鲁能就是小学生。\"}",
  "{\"qid\": \"qid_1103262911379902382\", \"category\": \"游戏-网络游戏\", \"title\": \"诛仙在哪里领激活码啊?在哪里领激活码啊? \", \"desc\": \"在哪里领啊?\", \"answer\": \"去在官网看看 \\r\\n其实你一搜索 那个链接就会出现的o(∩_∩)o...\"}",
  "{\"qid\": \"qid_6579774747972276861\", \"category\": \"健康-人体常识\", \"title\": \"脖子长10CM算长还是短?我15岁，男生，身高167CM,脸圆， \", \"desc\": \"我15岁，男生，身高167M,脸圆，头稍大，身材比较瘦，皮肤白，请问我适合穿那种领口稍大的圆领或V领黑色T恤吗？我个人比较喜欢深色的衣服。\", \"answer\": \"有点儿长了，脖子长的男生穿什么都好看，自信豪迈吧！\"}",
  "{\"qid\": \"qid_4993337915700844386\", \"category\": \"健康-肿瘤科\", \"title\": \"拉肚子以后,身体虚弱,应该吃些什么东西调养,有什么应该注意的?不 \", \"desc\": \"不好意思,昨天好像吃到什么不太的东西了,一直拉肚子,搞的我睡觉也睡不好,索性就起来了, 我该吃些什么东西调理下.\", \"answer\": \"一、饮食治疗目的\\r预防并纠正水及电解质平衡失调；供给充足营养，改善营养状况。避免机械性及性刺激，使肠道得到适当休息，有利于病情早日恢复。\\r\\r二、饮食治疗原则\\r(一)怠性腹泻饮食治疗\\r1．急性期禁食：急性水泻期需暂时禁食，使肠道完全休息。必要时由静脉输液，以防失水过多而脱水。\\r2．清淡流质饮食：不需禁食者，发病初宜给清淡流质饮食。如蛋白水、果汁、米汤、薄面汤等，以咸为主。早期禁牛奶、蔗糖等易产气的流质饮食。有些患者对牛奶不适应，服牛奶后常加重腹泻。\\r3．根据病情调整饮食：排便次数减少，症状缓解后改为低脂流质饮食，或低脂少渣、细软易消化的半流质饮食，如大米粥、藕粉、烂面条、面片等。\\r4．饮食选择：腹泻基本停止后，可供给低脂少渣半流质饮食或软食。少量多餐，以利于消化；如面条、粥、馒头、烂米饭、瘦肉泥等。仍应适当限制含粗纤维多的蔬菜水果等，以后逐渐过渡到普食。\\r5．补充维生素：注意复合维生素B和维生素C补充，如鲜桔汁、果汁、番茄汁、菜汤等。\\r6．饮食禁忌：禁酒，忌肥肉，坚硬及含粗纤维多的蔬菜、生冷瓜果，油脂多的点心及冷饮等。\\r\\r(二)慢性腹泻饮食治疗\\r1．低脂少渣饮食：每天脂肪40g左右，过多不易消化并加重胃肠道负担，刺激胃肠蠕动加重腹泻。故植物油也应限制，并注意烹调方法，以蒸、煮、氽、烩、烧等为主，禁用油煎炸、爆炒、滑溜等。可用食物有瘦肉、鸡、虾、鱼、豆制品等。注意少渣，粗纤维多的食物能刺激肠蠕动，使腹泻加重，当腹泻次数多时最好暂时不吃或尽量少吃蔬菜和水果，可给予鲜果汁、番茄汁以补充维生素；少渣饮食可减少肠蠕动、减轻腹泻，故宜进食细挂面、粥、烂饭等。\\r2．高蛋白高热能：慢性腹泻病程长，常反复发作，影响食物消化吸收，并造成体内贮存的热能消耗。为改善营养状况，应给予高蛋白高热能饮食，并用逐渐加量的方法，如增加过快，营养素不能完全吸收，反而可能加重胃肠道负担。可供给蛋白质lOOg／d左右，热能10．46～12，55MJ(2500～3000kcal)。\\r3．禁忌食物：如粗粮、生冷瓜果、冷拌莱等，含粗纤维多的韭菜、芹菜、榨菜等；坚硬不易消化的肉类如火腿、香肠、腌肉等；刺激性食物如辣椒、烈酒、芥末、辣椒粉，以及肥肉、油酥点心等高脂肪食物。\\r\\r更多参考:\\r \"}",
  "{\"qid\": \"qid_1172162630992884010\", \"category\": \"生活-保健养生\", \"title\": \"生活饮食习惯对人体健康的影响饮食的健康会对人体产生怎样的影响？ \", \"desc\": \" 饮食的会对人体产生怎样的影响？\", \"answer\": \"现在的人工作压力都很大，不按时吃饭的大有人在，暴饮暴食的也很多。这就是为什么现在亚健康的人群巨大的原因。良好的饮食习惯对你以后上了岁数会有很大帮助，会长寿的\"}",
  "{\"qid\": \"qid_744713671292039544\", \"category\": \"生活-美容/塑身-减肥/塑身\", \"title\": \"求推荐减肥药各位[泪][泪][泪][泪] \", \"desc\": \"求推荐药各位[泪][泪][泪][泪]\", \"answer\": \"恩 我最近也在减肥噢，给我的食谱给你参考一下，我个人觉得很有用。\\n早上 一杯牛奶，一个鸡蛋白\\n中午 两勺饭，青菜 汤水可以多吃。但是不要撑着了\\n晚上 一个苹果，或者一块西瓜，你自己喜欢就好。\"}",
  "{\"qid\": \"qid_8294380531769628792\", \"category\": \"娱乐-动漫\", \"title\": \"大家帮我找找这张图~就是海贼王里的银狐篇里的一集,路飞一行人与福 \", \"desc\": \"就是里的银狐篇里的一集,路飞一行人与福克斯(用迟钝光线的那个)比赛,其中一个画面里由于TV制作问题而出现了两个索罗,我想要这张截图,以前见到过~\\r却不慎丢失~请大家帮忙找一找~或是截一张\\r谢谢大家了~\", \"answer\": \"海贼王里\\r\\n\\r\\n两个索罗\"}",
  "{\"qid\": \"qid_2783837286197112772\", \"category\": \"游戏-网络游戏-跑跑卡丁车\", \"title\": \"进入跑跑卡丁车时为什么会出现错误报告每当进入跑跑就出现一个错误报 \", \"desc\": \"每当进入跑跑就出现一个错误报告，是:annot Execute NMCO Server! (Error CODE = 3).这是怎么会事\", \"answer\": \"卸载掉，重新装一下，最好优化大师先优化一下\"}",
  "{\"qid\": \"qid_6893766316173749231\", \"category\": \"电脑/网络\", \"title\": \"视频下载在哪里可以下载到格式为AVI的视频啊？ \", \"desc\": \"在哪里可以到格式为AVI的视频啊？\", \"answer\": \"你得用个转码器,因为AVI的格式也有好几种,就算你下到了AVI格式的,也不一定你的手机就能看.\\r\\n我用的转码器还可以,你试下.\\r\\n下载地址： \\r\\n电影的网站你可以上迅雷上下。\\r\\n转码前按他的提示做个配置。\"}",
  "{\"qid\": \"qid_4348258677050110984\", \"category\": \"健康-人体常识\", \"title\": \"09年查出肝血管瘤1.9\", \"desc\": \"09年查出血管瘤1.9-2.0今年长大了2.5-2.0肝左叶，平时左上腹胀不舒服，医生说是胃胀请问我该怎么办\", \"answer\": \"肝血管瘤是一种常见的肝脏良性肿瘤, 一般小血管瘤多无症状,也可在生气着急等情绪变化、饮酒、劳累后感觉肝区轻微不适，并伴有上腹饱胀、嗳气等症状。 \\r\\n肝血管瘤多数小于4厘米，成人后一般不继续生长，对健康无妨碍，亦不会癌变。所以一般不必治疗。\\r\\n直径大于5 cm者为巨大肝血管瘤，有潜在破裂出血的危险，需要治疗 。 \\r\\n你检查肝血管瘤1.9-2.0 今年长大了2.5-2.0，虽然有所增大，仍属于小血管瘤，目前不需处理，但要定期（半年或一年）复查。\\r\\n左上腹胀不舒服，不除外肝血管瘤所致的可能，可先按胃胀治疗，口服吗丁林1片，日2~3次，饭前半小时服。\\r\\n\"}",
  "{\"qid\": \"qid_6296306809888150600\", \"category\": \"商业/理财-股票\", \"title\": \"恒生指数就是道琼斯吗 \", \"desc\": \"\", \"answer\": \"恒生指数是港股的主要指数，道琼斯是美国股市的重要指数，不是一码事。\"}",
  "{\"qid\": \"qid_5565928575577908328\", \"category\": \"生活-服装/首饰\", \"title\": \"南京哪里有卖钻戒拖的?我有个钻戒好多年了钻石很亮但是戒指非常的污? \", \"desc\": \"南京哪里有卖拖的? 我有个钻戒好多年了 钻石很亮 但是戒指非常的污浊了 想更换钻拖 所以想问下南京哪里可以卖钻拖?  还有 大概费用是多少呢。谢谢回答\", \"answer\": \"不做广告,也不搞这个,帮你在网上查了下,据说中山门地矿所3楼有一家,不过不清楚名字,还有就是金轮新天地负一楼，有家晓何珠宝制作坊,一般金鹰这种大商场珠宝维修点也是可以做的吧应该,~\"}",
  "{\"qid\": \"qid_3811040838065815077\", \"category\": \"生活-起名\", \"title\": \"除了夹腋下，还有没有测量宝宝体温时间短，宝宝也不会抗拒的测量办法? \", \"desc\": \"除了夹腋下，还有没有测量体温时间短，宝宝也不会抗拒的测量办法？\", \"answer\": \"宝宝量体温，除了肛门、腋下，还有颈下。\\r\\n其实，选择在宝宝睡觉的时候量吧。\"}",
  "{\"qid\": \"qid_4075857162645346487\", \"category\": \"教育/科学-职业教育\", \"title\": \"如果觉得自学考试的成绩与自己所做的试卷不符，应怎么处理？是山东自 \", \"desc\": \"是山东的考生，能复查试卷吗？有没有其他办法？\", \"answer\": \"不能复查试卷\\r\\n只能按考试机构规定的时间,向其提出复核申请,由考办工作人员为你复核试卷.\"}",
  "{\"qid\": \"qid_7998567165628609834\", \"category\": \"电脑/网络-软件-网络软件\", \"title\": \"BT下载问题是不是下载与上传的速度不快就不太伤硬盘？我BT的下载 \", \"desc\": \"是不是下载与上传的速度不快就不太伤？我BT的下载速度和上传速度都在50kb/s,这样的速度伤硬盘吗？\", \"answer\": \"因为硬盘有一个规定的擦写速度，如果不超过这个速度话，对硬盘的伤害是很小的，不会影响到硬盘寿命。你的速度在100K左右，对你的硬盘伤害不大，所以不用担心。\"}",
  "{\"qid\": \"qid_8426154873806908170\", \"category\": \"娱乐-电视\", \"title\": \"想着买款节能的液晶电视用。居家过日子，我本着能省则省的原则，买液 \", \"desc\": \"居家过日子，我本着能省则省的原则，买液晶，想选款节能的，听朋友说台湾奇美的很不错，是吗？\", \"answer\": \"楼主免了吧，台湾货可是出了名的低端啊，质量次，买回家做啥啊，再说了，又有哪款液晶电视不节能啊\"}",
  "{\"qid\": \"qid_965145238873261268\", \"category\": \"游戏-完美游戏-诛仙\", \"title\": \"我的5173商品通过核审为什么不能上架？我卖个诛仙账号上面说发布 \", \"desc\": \"我卖个账号 上面说发布完成 为什么不能上架？一点击上架就出现 商品0比上架成功 1笔上架失败`\", \"answer\": \"验证信息失败了 5173会自己登陆下你帐号看邮箱以及其他你所说的信息是否正确  你可一看下  处理结果\"}",
  "{\"qid\": \"qid_3487350800464607887\", \"category\": \"电脑/网络-软件-多媒体软件\", \"title\": \"FLASH中的雪花飘落怎么制作？ \", \"desc\": \"\", \"answer\": \"简单的方法~\\r\\n建立一个引导层--然后在引导层画曲线(这条线就是将来你要让雪花飘落的路径)\\r\\n然后新建一层,把其属性设置为被引导层.\\r\\n之后在这层画一片雪花(有现成的最好) 然后作一个渐变动画,由上到下.使其能够依附于引导线上.\\r\\n再后浏览,这就是一片雪花飘落的动画了~\\r\\n\\r\\n如果要做出来下大雪的场景,CTRL+F8新建一个影片剪辑~把刚才作好的针复制近来.多建立几层,让雪在不同时间出现(应该知道怎么做吧~),回到场景1 在库里找到你刚才做的影片剪辑(打开库的快接键CTRL+L) 复制2个拖动到场景1里面来\\r\\n\\r\\n复制出来的影片剪辑 分别调整大小和透明度~让场景有点景深~之后就OK了~\\r\\n\\r\\n这个方法比较直接一点~如果要用AS编写的话恐怕对新手来说是有点麻烦~不过你可以去闪客学堂里看看或许有对你有帮助的文章或者素材,还有灵感~\\r\\n\\r\\n                                                          ~~歪歪~~\"}",
  "{\"qid\": \"qid_2029610074590556760\", \"category\": \"教育/科学-人文学科-历史学\", \"title\": \"\\\"我们现在都是中国人了，\\\"《华盛顿邮报》备受欢迎的专栏作家Dav? \", \"desc\": \"\\\"我们现在都是中国人了，\\\"《华盛顿邮报》备受欢迎的专栏David　Ignatius写道。\\\"我们有一个名义上的资本主义经济，但当它遇到麻烦时，我们就不信任不受约束的私人市场。中国人正变得更像我们，但同时我们也正变得更像他们。\\\"\\r    美国都学了中国的哪些东西？\", \"answer\": \"学古代的中国儒家文化，道经，孙子兵法。\\r\\n\\r\\n学现在的中国特色社会主义市场经济（政府监管比较强，半计划，半市场）\"}",
  "{\"qid\": \"qid_3577367020099281567\", \"category\": \"健康-精神心理科\", \"title\": \"追求物质生活和追求精神生活矛盾么?常听人们说物质生活过于丰富，人 \", \"desc\": \"常听人们说物质过于丰富，人就不去追求精神生活的更高层次了。宗教中的苦行者放弃了舒适的物质生活甚至是最基本的物质需要，去追求一个纯净的精神世界。那么，追求物质生活和追求精神生活真的相互矛盾么？如果说人是精神的载体，但是从客观上说，人又必须靠物质来支持。但是为什么通常时候高层的物质生活和高层的精神生活不可兼得呢？\", \"answer\": \"个人意见仅供参考,\\r\\n常听人们说物质生活过于丰富，人就不去追求精神生活的更高层次了\\r\\n有一种原因是人的内在品质的问题,黄金,与泥土的区别是很大的,\\r\\n第二,欲望的满足是无止境的,一个欲望的满足,另一个欲望早已迫不及待的需要满足,绵延不绝,无止无境,一般的人是无法摆脱这固定的轮回\\r\\n第三,想想吸毒者,他们的心思会放在哪里呢?当然这例子是有点极端.\\r\\n第四,人在对世界与人生没有一个正确性的认识之前,什么是好,什么是坏是分不清的,很多的人只是看到眼前的利益,眼前的享受,也因此为眼前的利益忙碌奔命,竭尽所能,\\r\\n第五,正是世界观与人生观不同,有些人只相信只有今生,而有些人相信有来生,等等有着完全相反的观念\\r\\n\\r\\n\"}",
  "{\"qid\": \"qid_2489526911613624875\", \"category\": \"健康-内科-内分泌科\", \"title\": \"糖尿病的食疗法 \", \"desc\": \"\", \"answer\": \"\\r糖尿病的真相\\r由于祖国医学研究的滞后，加之一线的医疗工作者对理论上的了解不是很多，所以很多患者并不知道糖尿病到底是如何产生的？为什么是不能治愈的终身疾病？\\r    这里我只针对2型糖尿病做一些解释，但也不代表全部，因为还有许多事情没有搞清楚，但以现有的科研成果，大部分的2型糖尿病患者是有机会真正的和病患剥离的！\\r    我在这里分三个层面来解释，第一个就是我们众所周知的一种解释，在国内这种解释还非常广，并且大部分的治疗方法都是针对此方面，就是直接认为是胰岛出了问题。因为人体内的血糖只有通过胰岛素的控制才能发挥作用，而胰岛分泌的胰岛素少或者质量不好，不能很好的被利用，造成血糖不受控制，所以就得了糖尿病。所以医生就会使用一些药物来刺激你的胰岛分泌更多的胰岛素来满足血糖的需要，但胰岛毕竟是人体的器官，经常性的用药物来刺激就会慢慢让它的功能消失怠尽，不能再分泌了！所以这时候医生会建议你直接打胰岛素来代替自身分泌，这时你的胰岛已经完蛋了！\\r    这时也有医疗工作者提出让患者在病患早期就先打胰岛素，这样可以让胰岛得到充分的休息来恢复分泌功能，但有发现打胰岛素的量很难控制，容易造成低血糖更加危险，另一方面也不能控制到底让胰岛休息多长时间合适，时间长了也一样会使它失去功能，所以这种方法也没有得到普及，不管是刺激胰岛还是替代它，对人的身体都是非常有害的，而且过程烦琐，稍不注意就可能出现更大的问题，所以我们都不建议这样的治疗方法！\\r    还有一种药物是直接作用于你吃的食物，让你产生的糖份直接减少，结果是营养不良的现象开始出现，抵抗力，免疫力都会下降，身体更容易受损。另外，糖本是人不可缺少的东西，少了也不行，所以这种方法也不被看好！\\r    但在这个治疗过程中人们发现，在血糖被控制的情况下并发症还是会来，而且危害极大，这就是医生都会告诉你真正可怕的是并发症的原因，因为你根本阻止不了！\\r    更可怕的是，我国的医疗工作者们在治疗方法上还停留在这个层面！\\r    插一句，现在的医疗工作者大致分为两种，一种就是你在里经常见到的医生或护士，他们是现有技术和药品的使用者，所以他们的认识也只停留在这个层面，他们没有自己独创的治疗方法，他们甚至会直接让你要先树立一种糖尿病绝对不能治愈的思想，然后再调理，还会告诉你所有说能治好病的人都是骗子，但他们又从不会说他们会对你的病没有办法，而他们自己又完全知道自己是无能为力的，种种的矛盾造就了我们现在已知的医疗群体，也使社会对他们的信心和信任在大幅下降，所以有时候“先要问问医生”或“要听医生的话”并不是完全正确的！要慎重挑选医生。另一种就是同时在做科研工作的医学专家，研究各种疾病的新的治疗方法，但在国内由于很多信息和设备都不足，加之大多人都以“药治”的角度寻求解决的方案，所以在慢性病领域已经很久没有什么突破性的东西了，而他们有很多处于闭关自守的状态，特别是中西医之间的难以融合，使很多国外已成型的概念或发现很难利用。第一种基本是授命于第二种，而我们受的教育都是要绝对听从医生的，结果可想而之，所有的人都说治不好，又都不知道为什么！\\r    继续说病，后来人们又发现了一个不能解释的现象，就是很多糖尿病患者的胰岛是正常的，这就和原来的认识发生了冲突，看来还有我们不知道的东西在起作用，到底是什么呢？大约前两三年的时候，有人从国外带来了最新的概念，而这种概念在国外已有多年的历史并且有了相应的产品。“胰岛素抵抗”的说法开始得到人们的了解。原来在我们的血液的细胞当中有一种蛋白质，而胰岛素只有和它相结合才能起到控制血糖的作用，我们把这种蛋白称作受体，如果受体不接受胰岛素的话，你就是分泌多少也是没有用处，来自体外的胰岛素一样没有作用，我们把这种不接受的现象称作“胰岛素抵抗”。而胰岛是它的第一个受害者，可以说我们都冤枉了胰岛，它为了保护我们的身体，不断的分泌胰岛素以为可以解决，但结局只有自己的“气绝身亡”，还要在医生的无知下背上所有的骂名，并且被他们毁掉，真的很可怜，大家千万不要在责怪我们的胰岛了！\\r国外针对此现象开发出胰岛素增敏剂，增强受体的亲和力，以使它更好的和胰岛素相结合，从根源上解决问题。受体的数目是相对恒定的，由于它周围环境的恶化，使它“生了锈”，还有一些发生萎缩、不健全等现象使它不与胰岛素结合，而现有的增敏剂只能解决20%的问题，在胰岛素的协助下起到了到此为止已知范围内最有效的作用，但同样对并发症只有延缓的作用，至于如何起到100%的作用还没有发现，而且因为糖尿病往往不能早期发现，在你发现时，并发症的种子也已经种了下去，这样即使你能恢复所有的受体也只是达到一个降血糖的作用，而我们真正需要解决的是阻止并发症的出现。所以理论上来说，全球范围内还不能治好糖尿病。而这种认知几乎存在于所有的慢性病领域内。\\r而我国的医疗科研工作者对这一层的了解还只停留在“知道有这么一说”的程度上，其它都不明白，所以也没有对应的治疗方法。而增敏剂对于国内人群来说过于昂贵，而且限制人群也很多，医生很少推荐，并为得到广泛的认可和普及。\\r这两个层面是糖尿病领域内的现状，现在我们说一下为什么此方面的研究如此困难和缓慢。前面我提示了一点，现在的医疗研究者的研究角度都是直接针对病患，通过什么技术或产品来直接消除它。比如对于“胰岛素抵抗”就是这样，而对引起它的诱因不屑一顾。“胰岛素抵抗”的出现完全是因为人体内环境的不同程度的污染，太多的身体不需要的杂质环绕在受体的周围，受体怎么可能还正常？我们提出的解决方案有三个：一是清洁受体所处的环境，二是恢复那些不正常的受体的功能，三是促生新受体来替换已经不可恢复的。这样解决的结果让我们消除“胰岛素抵抗”的效率达到了80%，而省下的20%，我们发现它们已经不足以引起并发症，所以我们有了成功治愈的记录，这就是我说的第三个层面。\\r当然，我们也还有很多解决不了，也是因为我们还没有搞透彻，比如1型，比如发现太晚，比如引发“胰岛素抵抗”的另一个原因，有些和受体很相似的物质半路截走了胰岛素等等。\\r写这些的目的是希望大家即使不能接受我们的结论也要对整个治疗糖尿病的技术和产品有一个充分的了解，也承认市场上很多我们不能信任的东西，加之医生极力宣扬糖尿病不可治愈的概念，使很多病友丧失了信心，其实完全不必如此，，前期预防，后期调理都不失为很好的方法，并且糖尿病患者有很多机会活到人的正常寿命，这个过程也不是我们想象的那样痛苦，毕竟我们真正想要的还是这个。\\r我们的技术现在通过三个途径来解决问题，胰岛功能的恢复，“胰岛素抵抗”的消除，并发症的防治。更全面有效的解决问题，而不是针对某一项。\\r蜜蜂体内提取的类胰岛素成分即使单独拿出来就已经可以在某种程度上替代打胰岛素，而且同时还有对胰岛的恢复作用，而这个成分仅仅是其中一个重要的成分，蜂毒能带给我们的还有很多，在这里不一一列举，更详细的内容有更针对性的解释。\\r所以请大家清楚一点，糖尿病不能治愈的说法是片面的，是受很多限制的，包括我们面对强大的反对论调而难以推广，医生在患者的思维里种下了根深蒂固的不可能，而且在无限的延续，我并不是完全怪医生，在他们的所知领域内也只能那样说，但我不希望患者因此放弃努力，更不希望被虚假的广告吓的因噎废食，医学在不断进步中，但从开发到推广的路更加艰难，任何的新生事物都要经过重重阻力才会得到广泛的认可，希望大家能与我们共同把属于我们祖国的光荣的结晶推广开来，这也许是中医在世界范围内得到认可的最好机会！\\r\"}",
  "{\"qid\": \"qid_4941995463447011765\", \"category\": \"烦恼-两性问题\", \"title\": \"把女友分泌出的爱液吃进肚子里要不要紧啊?如题 \", \"desc\": \"如题\", \"answer\": \"当然不要紧。\\r\\n\\r\\n爱液都是人体性器官和腺体的分泌物，如果没有疾病，是没有致病菌的，但是女性阴道内是有一些正常菌群。\\r\\n\\r\\n\"}",
  "{\"qid\": \"qid_7288784539564051995\", \"category\": \"游戏-网络游戏\", \"title\": \"为什么我领不到任务我的等级已经达到任务提示的要求,但当去找任务N \", \"desc\": \"我的等级已经达到任务提示的要求,但当去找任务NP时他们竟然不理我,他们头上的任务标志没有显示.\", \"answer\": \"不可能的  如果不行 去 官方 论坛 发 帖子  \"}",
  "{\"qid\": \"qid_8281396420932914288\", \"category\": \"娱乐-星座/运势\", \"title\": \"1983年2月27日下午3点18分出生的女孩子,命运如何请大师帮 \", \"desc\": \"请大师帮帮忙,看看.重点帮忙看看工作情况,谢谢了\", \"answer\": \"看了你的本命盘，如果你的阳历日期和出生时间都没有错的话，\\r\\n\\r\\n那么我可以断定你是一个很有潜力的人。你早年有些辛苦，中年发迹，晚年大好！在你的生命中可能一路走来都是充满挑战的，但是正因为充满挑战，所以能激发生命中最大的潜能。\\r\\n10宫主合火三合木， 其实你的事业和事业运相当好， 四十岁以后应该会有一番作为， 一句话：前途无量。 你与异性相处时，通常会感受到压力，感情上有些郁闷。  因为你无论在婚姻、家庭、爱情、财务方面都会出现问题。（请问一下：你的父母是不是经常吵架？）另外，你有很好的创业能力。\"}",
  "{\"qid\": \"qid_4032135752540622525\", \"category\": \"商业/理财-股票\", \"title\": \"请教老师000858有什么利空？跌这么多？支撑位是多少？谢谢！ \", \"desc\": \"\", \"answer\": \"000858 五粮液，基本面优良，隶属大盘绩优成长股。机构筹码锁定良好，短期调整结束后，有望延续反弹。运用黄金率0.618分析，短期技术压力在33.80元一线。操作策略，建议持股至上述区域，短期出货。祝好运~！\"}",
  "{\"qid\": \"qid_8738478348727306306\", \"category\": \"游戏-网络游戏\", \"title\": \"怎么电信2不能进啊，老是提示稍候，我都等了3个小时了！ \", \"desc\": \"\", \"answer\": \"网通的就进网通的，电信的进电信的，每周三9：30——12：30服务器例行维护，这个游戏里有提示。\"}",
  "{\"qid\": \"qid_2662891279005095757\", \"category\": \"体育/运动-篮球\", \"title\": \"火箭今年能否进入季后赛第2轮、、、 \", \"desc\": \"、、、\", \"answer\": \"难啊，现在连进季后赛都很悬啊，但还是支持\"}",
  "{\"qid\": \"qid_5110465507302901369\", \"category\": \"生活-美食/烹饪-烹饪方法\", \"title\": \"我要学做四川水豆鼓，请高手教我，说详细点，谢谢。 \", \"desc\": \"我要学做四川豆鼓，请高手教我，说详细点，谢谢。\", \"answer\": \"正规的四川水豆豉到底怎样做，一直不得而知。后来看了本川菜制作的书籍，讲水豆豉的作法基本与我婆婆的一样，先把黄豆泡了,煮熟，然后用棉被保温捂两天，再加姜末、罗卜干、油辣子、盐等即成。最近几年，按照书上的方法,我家几乎年年冬天都要自已做水豆豉。但是，多数时候都不成功，不是捂出来发酸就是发苦,这几年倒掉没捂成功的豆子不知有多少斤,谁敢吃呀！\\r我们就?磨,既然用被子捂,肯定就是让它发酵,发酵有两个条件不可缺少，第一要菌种,第二要适合的温度。我们一没菌种二不知该什么温度,捂出来的豆鼓谁知道用什么菌发的酵，敢随便吃吗？再说,即使菌种对了不知道合适的温度，捂多久用多少层棉被都是瞎碰,当然不能次次成功。联想到馒头的做法,先要用酵母菌和面，在温暖的室温下才能发酵成功，肯定水豆豉也应该有的做法,但菌种和温度是有侍解决的关键。\\r弟弟从日本回来,带回一种叫\\\"纳豆\\\"的食品，尝了后感觉与我们四川的水豆鼓差不多,就差姜末、罗卜干、辣椒油。听说日本已工厂化生产这种\\\"纳豆\\\"，看来日本一定有这种纳豆的菌种,用科学方法控制温度发酵，否则不可能工厂化生产标淮味道的纳豆。听弟弟说，在日本,納豆成为一种传统的营养食品,几乎家家户户都制作每日食用。\\r我们中国四川的水豆豉弄了半天日本也有,甚至还用科学的方法家家户户都能自制。我们哪天也掌握了科学做水豆鼓的方法多好，可以方便做出标准味道的正宗水豆鼓,再也不用去瞎捣鼓。\\r机会来了,我终于找到了纳豆的制作方法,经过自已的制作经验，把简单的四川水豆鼓自制方法告诉大家：\\r1.到伊滕洋华堂去买日本纳豆，成都的伊滕锦华店有卖(在二环路万达广场)，买最便宜的，因为作菌种用只需一点点，大约l4元两小盒。买回放冰箱保存待用。\\r2.到超市里买黄豆，约5元一斤，一次做一斤3口之家可吃好多天的。\\r3.黄豆用温水泡12-24小时。\\r\\r4.用高压锅煮黄豆,注意水要少放,高压锅上汽压阀后,改用小火保持30-45分钟,如压力太大,可把火关熄,待会再打开火煮。目的是把黄豆煮软,用手指轻压即成粉碎末状最佳。如果黄豆煮的火候不够,最后做成的水豆鼓口感一定不好吃。\\r5.用漏勺沥干黄豆水份，稍冷却一会(估计温度为50度时)把黄豆装进有盖的容器中。注意：所有东西一定要保证干净、无油，怕被杂菌污染。黄豆的温度一定不能太高，超过70度纳豆菌会被烫死，太低不易发酵,50度左右正好加入纳豆菌。\\r6.把买的日本纳豆从冰箱里拿出，打开小盒，取出里面的两包小调料不用，把納豆作为菌种放入煮好的约50度温度的黄豆中，用筷子和匀。一般500克黄豆放20-30克菌种，正好一小盒纳豆重20多克。\\r7.准备一纸箱,一电热毯(暖脚电毯最好)，小棉被。在纸箱内用电热毯、棉被做一窝，把和好菌种的黄豆(装容器内)趁热放内窝内，盖上箱盖，打开电热毯，低温档即可。注意：温度不可太高，保持在手感窝内暖暖即够了,不要烫手。箱子上面不用捂死了,纸箱盖上再搭个垫子什么的就可以,让它稍透点气，因为水豆鼓发酵需要空气和一定的湿度，最好选用盖上有小孔的容器，保持通气就可以。\\r8.黄豆在纸箱窝里保温发酵12-14小时就好了，正好一夜。取出来用筷子挑起看看,起大量的丝丝说明成功！尝一颗，不苦不酸,没任何怪味，ok！\\r9.将发酵成功的水豆豉拌上姜末、罗卜干颗颗、盐、油辣椒，调合适自己的口味后，用食品袋分装成一小袋一小袋的，估计每袋份量够自己一家人吃1-2天的。把装好袋的水豆鼓放入冰箱的冷冻室,冻成冰状保存。吃时，头天从冷冻室拿出一袋,自然解冻后，第二天再吃,就是可口的四川水豆豉了。\\r为什么要冷冻成冰状？因为发了酵的水豆鼓在冰箱冷藏室温度下仍会发酵,而发酵过了的会影响口感，放冷冻室冻成冰状不会把发酵菌冻死它,只让它暂停发酵，拿出来自然解冻后口感一样鲜美。但注意，千万不能用微波炉解冻,杀死菌了口感不好。\\r发好酵的水豆鼓会产生很多细长的菌丝，这是正常的，说明你发酵成功了，丝丝多才好,这丝丝可以牵到1米长呢！\\r发好酵的水豆鼓在放调料前千万记住留一小包(约20-30克),什么调料也别加,直接把它放进冰箱冷冻室保存,等下次你再做时，它就成了你的菌种，不用再花十多块的冤枉钱去伊滕买了！\\r怎么样，想试试吗？\"}",
  "{\"qid\": \"qid_9077723349152984281\", \"category\": \"商业/理财-个人理财\", \"title\": \"芯片卡功能？ \", \"desc\": \"\", \"answer\": \"“芯片”通常分为三大类。第一类是CPU芯片，就是指计算机内部对数据进行处理和控制的部件，也是各种数字化智能设备的“主脑”。第二类是存储芯片，主要是用于记录电子产品中的各种格式的数据。第三类是数字多媒体芯片，我们熟知的数码相机、越来越逼真的手机铃声就是通过此类芯片实现的。\\r\\n\"}",
  "{\"qid\": \"qid_1543978678977646490\", \"category\": \"电脑/网络-硬件-显示器\", \"title\": \"如何让投影仪在放映，而电脑依然可以操作其他的不影响投影仪？我前两 \", \"desc\": \"我前两天看到有人这么操作。。\", \"answer\": \"这样你需要有双显示功能的机器，如一般的笔记本都带一个VGA接口，或者双显示卡的台式机、带双显功能的显示卡等。\\r\\n第一步，连接投影在另外一个显示接口上；\\r\\n第二步，在桌面上点击鼠标右键，点属性，点设置，找到第二个显示器，点选将Windows桌面扩展在该显示器上。\\r\\n第三步，点确定；\\r\\n第四步，把要在投影上显示的内容向右拖，就能拖到另外一个显示器也就是投影可以显示了，同时第一个显示器可以做别的而不影响投影显示。\"}",
  "{\"qid\": \"qid_4113197521587359311\", \"category\": \"烦恼-交友技巧\", \"title\": \"大家来看看..我受不了了我是男的.我们宿舍有个同学..不知道什么 \", \"desc\": \"我是男的.我们宿舍有个同学..不知道什么问题...\\r我快疯了啊...我买什么衣服他就跟我买什么衣服一样的/上个月我买了一套服过几天他也买了一套跟我一样的...后来我又买了两条牛仔裤...他也去买了跟我一样的....我受不了了啊....!不知道他是不是变态啊...害我现在都不知道穿什么衣服了啊。..我不喜欢穿跟别人一样的衣服的啊\\r\", \"answer\": \"我大学的时候也和你有类似的经历。\\r\\n我打耳眼，他们也去;我戴隐形眼镜，他们也去配；总之是我买什么东西，他们也跟着买．\\r\\n我明白你的心情，但是他们就是单纯得觉得你的衣服很好．你就把你自己当作他们的流行时尚风向标．其实没有什么的，这是对你品位的认可．就和我觉得你用的洁面乳很好用，所以我也去买一样的．\"}",
  "{\"qid\": \"qid_1986055023312890569\", \"category\": \"电脑/网络-互联网-上网帮助\", \"title\": \"一般的蔓藤彩绘要多少钱啊？怎么算价格的啊？ \", \"desc\": \"怎么算价格的啊？\", \"answer\": \"一般在80—160之间 按平方计算 详情请登录我的博客文章分类里的业务介绍和价格\"}",
  "{\"qid\": \"qid_5729552866040713794\", \"category\": \"社会民生-法律\", \"title\": \"关于立法（）没有权力制定规章。A．国家发展和改革委员会B．国务院 \", \"desc\": \"（ ）没有权力制定规章。 \\rA．国家发展和改革委员会 B．国务院办公厅 \\rC．国家人事部 D．审计署 \\r问题补充:书本上是选A，解释说“国家发展和改革委员会主要职责为拟订并组织实施国民经济和社会发展战略、中长期规划和年度计划；提民经济发展和优化重大经济结构的目标和政策；提出运用各种经济手段和政策的建议；受国务院委托向全国人大作国民经济和社会发展计划的报告”\\r请专业人士解答，并给出具体解释！谢谢！！\", \"answer\": \"选A是正确的。\\r\\n因为，根据《立法法》第七十一条规定：“国务院各部、委员会、中国人民银行、审计署和具有行政管理职能的直属机构，可以根据法律和国务院的行政法规、决定、命令，在本部门的权限范围内，制定规章。\\r\\n部门规章规定的事项应当属于执行法律或者国务院的行政法规、决定、命令的事项。”而国家发展和改革委员会没有“决定、命令”的行政权力。\\r\\n\"}",
  "{\"qid\": \"qid_7784419091660529988\", \"category\": \"教育/科学-理工学科\", \"title\": \"任何事物都有由生到灭的过程。为什么只知道物种在不断灭亡，却没有看? \", \"desc\": \"\", \"answer\": \"物种产生是渐变，物种灭绝很多都是突变。\"}",
  "{\"qid\": \"qid_2043732258234017217\", \"category\": \"健康-肿瘤科-宫颈癌\", \"title\": \"阴道干涩和妇科病有关系吗？自从我生过孩子后，和老公过性生活时，老 \", \"desc\": \"  自从我生过孩子后，和老公过性时，老觉得我阴道干，不像以前那样湿润，这让我很是苦恼。请问这种现象一般由哪些因素造成？谢谢！\", \"answer\": \"阴道干涩是很多年轻女性在性中常见的症状。它会引起性交疼痛、影响性快感；久而久之还会导致性欲减退、性冷淡，最终影响到夫妻和睦。相当一部分人认为，阴道干涩是由体内缺少雌激素导致的。因此，常常要求医生让自己大量补充雌激素。这种做法对她们有害无益。 \\r正处于生育年龄的妇女是一生中性功能最旺盛的时期，所以，大部分人体内都不缺雌激素。那么，导致阴道干涩的原因是什么呢？临床中发现，如果与阴道干涩相伴的，还有口角发炎、皮肤干燥、脱屑等症状，就要考虑是否缺乏维生素B2。维生素B2是人体中一种重要的营养物质，一旦缺乏，会影响皮肤黏膜的代谢，使身体器官失去正常功能。其症状常常表现为口角发炎、眼睑炎、结膜炎、耳鼻黏膜干燥、皮肤干燥、脱屑等。它的缺乏对女性生殖器的损伤尤其严重，表现为阴道壁干燥、黏膜充血和溃破。如果发现自己身上有这种现象，最好先到医院的妇产科就诊。 此外，也可通过食物或药物来补充维生素B2。比如多吃五谷杂粮、带皮的水果等，因为维生素B2大多存在于植物的果皮或壳中。服用维生素B2的补充剂也可让体内营养物质达到平衡，缓解阴道干涩。具体方法是：口服每日3次，每次10毫克.\\r\"}",
  "{\"qid\": \"qid_2585360456584549615\", \"category\": \"教育/科学-理工学科-天文学\", \"title\": \"太阳上面的水是以什么形态存在的？ \", \"desc\": \"\", \"answer\": \"　　太阳表面温度达6000度，水是以氢离子和氢氧根离子形式存在。\\r\\n　　太阳上可能有水！加拿大Waterloo大学的Peter Bernath在1991年研究太阳黑子光谱时，发现了有史以来观测到太阳表面的最低温度。事实上，水在5000K 以下，并不会完全被分离成H+和OH-基，所以，还是有某些极高温的“水蒸气”存在于太阳表面。\"}",
  "{\"qid\": \"qid_7008329957135961451\", \"category\": \"游戏-网易游戏-魔兽世界\", \"title\": \"我在5173上买了一个1500的魔兽帐号。不到3天就被恶意找回！ \", \"desc\": \"我问下，你们注册交易的时候，出售帐号的人有上传身份证号码吗？\\r你们可以提供交易方的银行帐号、固定电话等信息给我吗？我要去公安局报警。\\r不可能出售担保帐号时候担保人的信息都没有吧\\r\", \"answer\": \"你在买号的时候 第一点就应该看卖家提供不提供身份证扫描件,第二要看,改帐号能身份证重置么,第三也就是能转服么和卖家帐号有过交易记录, 如果上述有哪点不满足 该帐号都很危险, 如果就算都满足了,也危险他没事改个密码, 所以买号就看运气了.看卖家人品了\\r\\n\"}",
  "{\"qid\": \"qid_78284520140909447\", \"category\": \"游戏-网易游戏-魔兽世界\", \"title\": \"战士的双手武器是看秒攻,还是攻击上限?我朋友说战士的双手武器只要 \", \"desc\": \"我说战士的双手武器只要秒攻高就好.真是这样的吗?难道不看攻击最高值吗?\", \"answer\": \"你朋友说得是狂战吧\\r\\n如果是武器战就不一定拿DPS高的了\\r\\n\\r\\n武器战的武器上限高好,当然都高是最好的了\\r\\n因为本身技能里有不受武器攻击速度限制的技能(就是瞬发技能)\\r\\n出暴击之后有意想不到的效果\\r\\n\\r\\n战士在战斗中接近敌人的机会并不是很多\\r\\n所一每次机会都很重要\\r\\n平稳输出我认为效果不佳\\r\\n仅在PK中是这样\\r\\nFB就不一样了\\r\\n\"}",
  "{\"qid\": \"qid_2301762798312100171\", \"category\": \"教育/科学-外语学习\", \"title\": \"台商ITC产业.英语ITC是什么全拼名称? \", \"desc\": \"台商C产业.英语ITC是什么全拼名称?\", \"answer\": \"应该是信息技术与通讯产业。Information Technologe and Communication.\\r\\n\\r\\n\"}",
  "{\"qid\": \"qid_8907005866216628656\", \"category\": \"娱乐-博彩\", \"title\": \"好了不说了罗马尼亚伤心罗马尼亚真的不行了不过火锅太大了倍投还有好 \", \"desc\": \"罗尼亚真的不行了不过火锅太大了倍投还有好多单中了没意思祝大家好运，我说过大家没必要刻意博冷火锅就是火锅国家队比赛而且都是强队的比赛，上期胆少这期太多出一二个还是没奖金除法国沙特外\", \"answer\": \"感觉是30。但是现在2：1，这并不稳，如果奥地利再进个，那就是了，感觉要出1，这是我不想要的，不如出0。\"}",
  "{\"qid\": \"qid_2461982717893146507\", \"category\": \"电脑/网络-互联网-上网帮助\", \"title\": \"网络媒体资源网哪个好？ \", \"desc\": \"\", \"answer\": \"看看版主俱乐部吧，那里汇聚了全国最权威的网络媒体资源，能让我们快速找到我们自己急需的媒体资源。\"}",
  "{\"qid\": \"qid_4109011963370125292\", \"category\": \"文化/艺术-历史话题\", \"title\": \"听说北京建都860年了，从什么时候开始的啊，都那些朝代啊？ \", \"desc\": \"听说建都860年了，从什么时候开始的啊，都那些朝代啊？\\r\", \"answer\": \"公元1153年，金主亮贞元元年，由上都迁燕京，北京始成为国都。金/元/明（永乐帝）/一直到现在\"}",
  "{\"qid\": \"qid_8541638805318977815\", \"category\": \"健康-整形美容-眼部整形\", \"title\": \"孩子得了恶性肿瘤？作为妈妈如何救她？尊敬的学诚法师：孩子马上8岁 \", \"desc\": \"尊敬的学诚：\\r孩子马上8岁了，去年夏天诊断为恶性横纹肌肉瘤。经过了半年多的放化疗，现在回家了。作为妈妈，我很心痛，为了帮助孩子，从孩子手术后我就开始吃素，而且开始念佛号。还带孩子去寺院拜佛，为冤亲债主立超度的排位。每天我夜不能寐，很是不安。\\r请问作为妈妈我还能为孩子做些什么？\", \"answer\": \"推己及人，以此心体会天下母亲之心，发愿为拔除所有众生的痛苦而积功累德。\"}",
  "{\"qid\": \"qid_1813420482139576514\", \"category\": \"教育/科学-理工学科-工程技术科学\", \"title\": \"请问热继器的双金属片用什么材料造的？ \", \"desc\": \"请问热继器的双金属片用什么造的？\", \"answer\": \"　　热继电器的测量元件通常用双金属片，它是由主动层和被动层组成。主动层材料采用较高膨胀系数的铁铬镍合金。被动层材料采用膨胀系数很小的铁镍合金。\\r\\n　　 \"}",
  "{\"qid\": \"qid_7695728816446997476\", \"category\": \"娱乐-博彩\", \"title\": \"080期:512元斯图加单0.01阿森纳\", \"desc\": \"01 阿森纳---博尔顿  单“3”足以解决问题（客队3轮来1胜2平，但区分与谁？）\\r02 富勒姆---狼　队 31（主队不败格局）\\r03 曼　城---布莱克 3（两队关系特殊）\\r04 纽卡斯---布莱浦 13（两支升班马对决，看好主队不败）\\r05 西布朗---热　刺  31（热刺40年后再次踏进欧冠赛场，下周中做客出战小组对手德甲劲旅不来梅，联赛分心难免。西布朗本赛季前3战主胜客负，该队从上赛季至今连续13个主场保持不败，面对做客能力不强的热刺，有望保得不败）\\r06 西汉姆---切尔西  01（切尔西过往11次与西汉姆交锋中取得9胜2平，但近两季01 阿森纳---博尔顿  单“3”足以解决问题（客队3轮来1胜2平，但区分与谁？）\\r02 富勒姆---狼　队 31（主队不败格局）\\r03 曼　城---布莱克 3（两队关系特殊）\\r04 纽卡斯---布莱浦 13（两支升班马对决，看好主队不败）\\r05 西布朗---热　刺  31（热刺40年后再次踏进欧冠赛场，下周中做客出战小组对手德甲劲旅不来梅，联赛分心难免。西布朗本赛季前3战主胜客负，该队从上赛季至今连续13个主场保持不败，面对做客能力不强的热刺，有望保得不败）\\r06 西汉姆---切尔西  01（切尔西过往11次与西汉姆交锋中取得9胜2平，但近两季4次交手只取得2胜2平）\\r07 维　冈---桑德兰  10（维冈本赛季前两个主场全败，1球不进失了10球，算上上赛季，维冈近5个主场只赢过1次）\\r08 多　特---沃尔夫  10（多特过往主战狼堡8胜4平2负，但近3个主场2平1负，下周中的欧联小组赛首战，防其分心）\\r09 门　兴---法兰克  03（本场难度大，但客胜不能?G）\\r10 汉　堡---纽伦堡  3（汉堡近9次主战对手8胜1平，优势十分明显。纽伦堡5次做客1胜4负（3次大败），相信主队大炒）\\r11汉诺威---勒　沃 01（汉诺威过往主战勒沃4胜6平5负，近8次主战1胜4平3负。勒沃前锋黑尔默斯和队长罗尔费斯伤愈归队，攻防实力均有所增强，上轮主场惨败堪称奇耻大辱，看客军反弹）\\r12 弗赖堡--斯图加  0（上赛季弗赖堡被斯图加特双杀，且青年军近年来还从未有过开局3连败的先例）\\r13 拜　仁--不来梅 3（Oddset的1.60、3.00、4.00组合下，拜仁5次入选取得全胜，其中2次战胜不来梅）\\r14 切塞纳--AC米兰 0（客让1球中高水,优势特别大,博胆）\\r\", \"answer\": \"除了热刺，其余和楼主同思路！\\r\\n\\r\\n今天看到了朋友们都不看好热刺，看来是要谨慎对待这场了\\r\\n\\r\\n个人觉得热刺下周中客场战不莱梅，本轮预计会演练防反阵型，上轮刚败，为了士气不应连败给保级队啊！\\r\\n\\r\\n说个题外话，热刺为国捐躯2大将，会否和中超青岛队一样上轮疑似受到官哨保护呢，呵呵\\r\\n\\r\\n祝好运\"}",
  "{\"qid\": \"qid_7719774581845631279\", \"category\": \"生活\", \"title\": \"我不知道2011西安市考报名地址在哪,有没有人知道的？ \", \"desc\": \"\", \"answer\": \"我身边考公务员的同学都买的是华图教育的书，还有一些去华图报班了，他们说华图的政法干警培训班开设的比较合理，把民法和基础课程分开来了，所以可以根据自己的实际情况选择一门或者多门去上课，报的班越多优惠越多。他们报班之前也比较了好几家培训机构，最后汇总起来，再加上现场咨询，都决定去报陕西华图的班了。相信大家的眼睛是雪亮的吧~~呵呵，你也可以去陕西华图咨询下，就在小寨汉唐书城8楼呢~我同学就是去那儿咨询的\"}",
  "{\"qid\": \"qid_4034019609632244539\", \"category\": \"电脑/网络-互联网\", \"title\": \"请问新浪微博是最好用的吗？各个网站都有微博，新浪排第几啊？ \", \"desc\": \"各个都有微博，新浪排第几啊？\", \"answer\": \"当然啦 我们都是微博控啊\"}",
  "{\"qid\": \"qid_4586346939984102214\", \"category\": \"生活-购物\", \"title\": \"帮我个忙，我问个价格我想买个取暖气，但又不懂市场价，谁能告诉我大 \", \"desc\": \"我想买个取暖气，但又不懂价，谁能告诉我大概需要多少钱？谢谢了！\", \"answer\": \"商场里品牌的要200多.外面小电器商行,只要60-70元左右,一个取暖气没有必要买好的,到当地小电器商行买就可以了.\"}",
  "{\"qid\": \"qid_2958710616169871282\", \"category\": \"游戏-网络游戏\", \"title\": \"武道会玩拳的技能点怎么加啊？请教高手武道会玩拳的技能点怎么加啊？ \", \"desc\": \"武道会玩拳的技能点怎么加啊？\\r\\n希望高手能告诉我`\\r\\n要说明白点我才懂哦`\\r\\n谢谢咯`\", \"answer\": \"玩拳,前期重力拳加到可以学流星拳(好像是2级,要是觉的练级慢其实可以全部加满),拳法专精理所当然是要全满的,共通的冥想只需要一点技能点就能学满,瞒好用的~拳法强化当然也是要学的~\\r\\n我也只是玩前期的,后期还没研究呵~\"}",
  "{\"qid\": \"qid_3758382171738024593\", \"category\": \"娱乐-博彩\", \"title\": \"麻烦请教怎么传图？谢谢！ \", \"desc\": \"麻烦请教怎么传图？谢谢！\", \"answer\": \"把拍有照片的硬件（如手机，相机）用USB线与电脑连接，回答或提问时点浏览找到这张照片，确定就行了\"}",
  "{\"qid\": \"qid_6231371354979360246\", \"category\": \"娱乐-度假旅游\", \"title\": \"济南附近有没有樱桃园？在哪里？ \", \"desc\": \"附近有没有樱桃园？在哪里？\", \"answer\": \"你好：在仲宫镇南部山区有啊。\"}",
  "{\"qid\": \"qid_3554855727125997778\", \"category\": \"健康-精神心理科-心理科\", \"title\": \"为何现在这些人等候说90后是脑残？你们都是90后是脑残？亲，看看 \", \"desc\": \"你们都是90后是脑残？亲，看看现在的00后比我们还疯狂，你们怎么不说说他们，议论议论他们，我们被针对了这么久，都在慢慢老去了。而00后呢？\", \"answer\": \"90后从出生开始生活环境相对前几代优越许多，没有经历过历史和政治上的动荡，也没有经历过大的经济波动。由于计划生育政策的影响，90后的大部分都是独生子女，除父母外，亲情观相对淡薄，在这样的生长环境里，有相当一部分人比前几代人来说，更显得有些孤僻。今日中国的信息发展迅速，同时由于“经济全球一体化”的原因，全球各地的物品均涌入中国，各国交流日益广泛，而90后相对年轻，对新事物的接受能力较强，在审美观和价值观方面也与前人有很大不同。\\r\\n    总的来说,90后的问题不是你们自己造成的,但是你们真的表现很差!非主流我就不说了,也许这是你们的审美问题.追星,谈恋爱,炫富,攀比,还有那个宁愿汶川地震的,难道这些都是你们应该做的?也许你会说你们80后也有这样的,但是你也要明白比例问题.在你身边的90后们,有几个不是这样的?\\r\\n    我不会称你们脑残,每个人都有尊严.如果你们能够真正成熟起来,向别人证明你们不是\\\"脑残\\\",谁还会这么无聊去说呢?当你们能真正担当起建设祖国的大业时,谁还有资格说呢?\"}",
  "{\"qid\": \"qid_1793283395186071139\", \"category\": \"游戏-网络游戏\", \"title\": \"想做把80级灵枪!!!想做把80级灵枪,有3种选择,1,所有5、 \", \"desc\": \"想做把80级灵枪,有3种选择,1,所有5、金5、TS1700（满）；2，金5、TS1700（满），忽视抗性20；3，所有5，技能10，TS1700（满）。哪种最好，谢谢指教。\\r碧波瑶池，会员2800——3000W。80TS已有1700（满）\", \"answer\": \"除了技能，那两个都可以。用所有相五和忽视抗性加天伤也很好。至于金五配合这两条我不建议。所有相五加高你防和速度更为贴切。\"}",
  "{\"qid\": \"qid_2831811435504024684\", \"category\": \"健康-肿瘤科\", \"title\": \"如何更好的治疗慢性咽炎？我的嗓子有时清凉，有时发哑，医生让我吃三 \", \"desc\": \"我的嗓子有时清凉，有时发哑，让我吃三个月的穿王消炎片，我吃了一个月了，实在是不好吃，我都要吐了。请问有没有更好的方法？\", \"answer\": \"慢性咽炎的治疗： \\r\\r1、主要针对病因，如戒烟戒酒，积极治疗急性咽炎及鼻腔、鼻窦、扁桃体的慢性炎症，改善工作和环境，避免粉尘及有害气体的刺激。加强锻炼，增强体质，预防感冒。 \\r\\r2、病人如有咽干、咽痛可选用一些含片，如华素片、草珊瑚含片、银黄含片、泰乐奇含片、西瓜霜含片等，以减轻或解除症状； \\r\\r3、也可选用各种中成药，如万应胶囊、清咽利喉颗粒、一清胶囊、十味龙胆花颗粒。 \\r\\r4、如果患慢性肥厚性咽炎，咽干、咽部异物感明显时，可采用分次激光、冷冻或电灼治疗。 \\r\\r专家提醒：慢性咽炎一般不需要使用抗生素治疗，因为慢性咽炎并非细菌感染。 \\r\\r因为： \\r\\r1、滥用抗生素可能导致咽喉部正常菌群失调，引起二重感染。 \\r\\r2、每一种抗生素都有全身副作用。滥用抗生素可对人体造成危害。 \\r\\r3、滥用抗生素还能引起细菌耐药。 \\r\\r需要注意的是：慢性咽炎极易反复，症状常发生在疲劳、受凉、烟酒过度、进刺激性食物、气候突变及吸入寒冷空气后。这时病人咽干、咽痛较为剧烈，部分病人还有发热，检查常可见咽部黏膜急性充血、肿胀，血常规检查白细胞增高，中性粒细胞百分率增高。此时，可在医生指导下使用广谱抗生素治疗或根据药敏试验选用相应的抗生素治疗3－5天，急性症状消失后马上停药。同时，病人需休息，多饮水及进食流汁饮食。\\r\\r\"}",
  "{\"qid\": \"qid_7481899923593833557\", \"category\": \"电脑/网络-互联网\", \"title\": \"为什么那么多黑客？老是在邮箱发的乱七八糟信件！ \", \"desc\": \"老是在邮箱发的乱七八糟信件！\", \"answer\": \"黑客不会无聊到望你邮箱里发乱七八糟得东西，你说的都是垃圾广告邮件吧，这个免费邮件的通病，收费邮箱会管理严格一点，很多垃圾邮件会被过滤掉。\"}",
  "{\"qid\": \"qid_1665016037283621083\", \"category\": \"游戏-网络游戏-奇迹世界\", \"title\": \"请问如果红名死了的话包裹里的装备和东西会被暴掉么？如果用了，安全? \", \"desc\": \"请问如果红名死了的话包裹里的装备和东西会被暴掉么？如果用了，安全锁链的话，是不是那里的装备都不会掉落？也就是即使红名了，带上安全锁链挂了就不会掉装备是么？？？\", \"answer\": \"没安全锁链肯定掉\\r\\n有安全锁链肯定不会掉\"}",
  "{\"qid\": \"qid_2384368102589799049\", \"category\": \"游戏-完美游戏-武林外传\", \"title\": \"武林更新问题怎么关闭游戏再打开后总是出现\\\"发现新版本,是否立即升 \", \"desc\": \"怎么关闭再打开后总是出现\\\"发现新版本,是否立即升级\\\"啊?\", \"answer\": \"你更新到最新版本就可以了。过年了又增加了好多活动。\\r\\n双倍经验：\\r\\n\\r\\n具体双倍时间安排如下：\\r\\n2月2日 14:00-18:00\\r\\n2月3日 14:00-18:00\\r\\n2月6日 14:00-18:00\\r\\n2月7日 14:00-18:00\\r\\n2月10日 14:00-18:00\\r\\n2月11日 14:00-18:00\\r\\n2月14日 14:00-18:00\\r\\n2月15日 14:00-18:00 \\r\\n\\r\\n海盗的入侵\\r\\n\\r\\n活动时间：2月1日到2月21日期间 13：00——14：00 15线八里庄\\r\\n来自海盗船的怪物侵袭八里庄，十二上代寡不敌众，形势十分危急！\\r\\n\\r\\n春节活动\\r\\n\\r\\n活动时间：2月5日到2月14日期间 21：00——24：00 15线京城\\r\\n散财童子派送春节大奖，好运隆隆而至，财源滚滚而来！\\r\\n祝你游戏快乐。\"}",
  "{\"qid\": \"qid_34331558253166439\", \"category\": \"游戏-网络游戏\", \"title\": \"法师与精力请问,法师加精力可以提升魔法攻击速度吗?可以提升魔法念 \", \"desc\": \"请问,加精力可以提升魔法攻击速度吗?可以提升魔法念咒速度吗?谢谢\", \"answer\": \"不行\"}",
  "{\"qid\": \"qid_6643157122912268630\", \"category\": \"生活-美容/塑身-护肤\", \"title\": \"黑眼圈越来越重怎么办白天总感觉很累 \", \"desc\": \"黑眼圈越来越重怎么办    白天总感觉很累\", \"answer\": \"多数人的黑眼圈是后天原因，即因经常熬夜、吸烟、清除眼影妆不彻底或睡觉时枕头太低而形成的。怎样消除掉黑眼圈，兹介绍以下几种方法：一、涂蜂蜜法：在洗脸后勿擦干脸上的水分，让其自然干，然后在眼部周围涂上蜂蜜，先按摩几分种，再等10分种后用清水洗净，水不要擦去使其自然干，涂上面霜即可。二、敷酸奶法：用纱布蘸上些酸奶，敷在眼睛周围，每次十分钟。三、热鸡蛋按摩法：将鸡蛋煮熟后去壳，用小毛巾包裹住，合上双眼用鸡蛋按摩眼睛四周，可加快血液循环。四、苹果片敷眼法：将含汁量多的苹果切片，敷双眼每次15分钟。五、土豆片敷眼法：土豆去皮切成约2厘米的厚片，外敷双眼每次5分钟六、同时也可以采用复合****进行黑眼圈的祛除。\"}",
  "{\"qid\": \"qid_1508851283884568158\", \"category\": \"娱乐-幽默滑稽\", \"title\": \"中国境内活得最潇洒的是哪一类人？ \", \"desc\": \"\", \"answer\": \"丁克一族\"}",
  "{\"qid\": \"qid_1966271995448072003\", \"category\": \"健康-儿科\", \"title\": \"胃胀气，打隔，有隔打不出来，吃东西感觉很硬，不消化，不敢吃东西，? \", \"desc\": \"胃胀气，打隔，有隔打不出来，吃东西感觉很硬，不消化，不敢吃东西，五六年之前做过胃镜,是慢性浅表性胃炎，后来吃胃粒康颗粒吃好过，可是烤肉吃多了又犯过几次病到现在，我感觉冬天的时候胃难受比夏天厉害。 应该吃什么药？饮食有什么注意事项？\", \"answer\": \"您好，鉴于您很长时间没有行胃镜检查了，所以建议先行胃镜检查，在明确是否有疾病的基础上再决定是否服用相关药物，不能仅凭症状就自行服用药物，可能会适得其反。\"}",
  "{\"qid\": \"qid_6363440678237764642\", \"category\": \"电脑/网络-软件\", \"title\": \"金山词霸哪个版本比较好用?免费版，可以安装的！ \", \"desc\": \"免费版，可以安装的！\", \"answer\": \" 目前金山词霸2005版本比较稳定好用（2006版本出来不久），如你先安装Acrobat Rreader 6.0，后安装金山词霸2005的话还支持PDF文件的取词翻译。\\r\\n下载地址： \"}",
  "{\"qid\": \"qid_1233180131150352374\", \"category\": \"商业/理财-股票\", \"title\": \"请帮忙解答000825成本18.15,600011成本13.05 \", \"desc\": \"000825成本18.15,600011成本13.05,000777成本32.28,后市如何操作?\", \"answer\": \"000825成本18.15,如果说你是准备中长线持股的话这股票是没有问题的.建议持股观望.\\r\\n600011成本13.05,有调整的迹象,建议先出局等调整到位逢低再介入.\\r\\n000777成本32.28,其业绩属一次性收益而不是主营收益.个人建议逢高出局.\"}",
  "{\"qid\": \"qid_5567485424998867637\", \"category\": \"游戏-完美游戏-完美国际\", \"title\": \"关于洗点问题我是77小武侠。。。灵力一直没加还是5点。。但是穿上 \", \"desc\": \"我是77小武侠。。。灵力一直没加还是5点。。但是穿上海盗装后会加灵。。。79海盗王还会加。。。我现在穿备灵力是9。。我想穿着装备用全能洗点券。。不知道能不能把原始的5灵给洗去啊？？知道的再回答我。。不要给我错误答案，浪费我的元宝\", \"answer\": \"我没洗过 从楼上的大体意思我估计 洗点一次能洗去5点属性 但是起初的5点灵力是绝对洗不掉的 但是你把灵力属性加到8（不穿装备）  再用洗点 一次洗去5点 不就灵剩3点了么 相当于投机取巧的方式 既然有人能洗到3（你必须确定有人无装备灵力是3）  我估计就是这么个情况 我也不大知道哈 \"}",
  "{\"qid\": \"qid_3829681820734565978\", \"category\": \"健康-外科-肝胆外科\", \"title\": \"腱鞘囊肿的危害手腕上长了个腱鞘囊肿倒是一点也不疼，但是腱鞘囊肿对 \", \"desc\": \"手腕上长了个腱鞘囊肿倒是一点也不疼，但是腱鞘囊肿对身体有什么危害吗\", \"answer\": \"腱鞘囊肿的危害表现为患处局部压痛。触摸时皮下饱满并有波动囊样感，伴有腕部无力，不适或疼痛，多为酸痛或放射性痛，可有一定的功能障碍。\\r\\n很多患者被手腕或其他部位的腱鞘囊肿所困扰，不仅会出现疼痛等症状，还会对学习生活造成一定的困扰。女性患者尤其对手腕手背的腱鞘囊肿很在意，疼痛不变的同时还会影响美观。所以腱鞘囊肿的危害是不容忽视的。\\r\\n腱鞘囊肿可自行挤爆，按压，时间久了会自行消退吸收，但是极易感染复发。或采用穿刺抽出囊液，注入肾上腺皮质激素或透明质酸梅的方法对治疗腱鞘囊肿也有一定的疗效。如果手术切除腱鞘囊肿应避免患病关节一个月。\\r\\n中药治疗贴腱囊贴，安全方便，治疗效果很不错。\\r\\n腱鞘囊肿患者要注意对患处的休息，忌辛辣的食物。治愈后要预防腱鞘囊肿的复发，避免过度劳损，适当休息和锻炼。\\r\\n\"}",
  "{\"qid\": \"qid_4029454027761505606\", \"category\": \"游戏-网络游戏\", \"title\": \"请问52级大地穿什么好？是末日还是青狼？都加什么属性，请各位高手? \", \"desc\": \"请问52级大地穿什么好？是末日还是青？都加什么属性，请各位高手指点一下\", \"answer\": \"    要我选择的话，我穿青狼轻。\\r\\n\\r\\n如果你有钱,那么就不要选择夏隆，夏隆属性是不错但是防低。2章闪避毕竟还不是很能体现。（加4的闪避）\\r\\n末日，是套好衣服。如果你不打算纹身的话，末日是你不2的选择。只不过是属性上的加加减减。（减1力量2体质加3敏捷）\\r\\n你可能会问清狼有什么用？加15%施法要来做什么？5%的防御真的那么重要吗？\\r\\n青狼为B顶也就是防最高的衣服和末日一样。但是5%看上去是没什么用，你全部冲到+3，再加BUFF再加5%是多少防？不用我说了吧。在PK的时候真的是闪普攻重要些吗？你末日3点敏=1~2点闪。2点敏加一点闪纹过的都知道，所以要尽量避免单数的浪费。技能命中于闪没关系，所以不管你多少闪都会被技能打中的。当然我不是说大地闪不重要。白木才看不懂我在说什么。\\r\\n\\r\\n15%的施法你PK多就会知道有多重要了，迟缓的效果多用用，出门带把弓。宝藏/骑士/银月等等再不为惧。\\r\\n\\r\\n[信我就顶我咖啡=。=~] \\r\\n\"}",
  "{\"qid\": \"qid_4497604405927257528\", \"category\": \"游戏-网易游戏-魔兽世界\", \"title\": \"LM的公会仓库在哪？我原来是BL的！ \", \"desc\": \"LM的公会仓库在哪？我原来是的！\", \"answer\": \"只要有银行的地方,都有公会仓存.\"}",
  "{\"qid\": \"qid_4757082100615101536\", \"category\": \"游戏-完美游戏-诛仙\", \"title\": \"为什么我挖到失落的宝藏老没交的任务啊天行者那没交任务的对话 \", \"desc\": \"天行者那没交任务的对话\", \"answer\": \"周日的12点到周二维护期间，即使你挖到失落的宝物也交不了。\\r\\n周二维护完以后到周日12点前是交失落的宝物的时间。\"}",
  "{\"qid\": \"qid_2022266574013777018\", \"category\": \"娱乐-明星\", \"title\": \"谁是你心中的大帅哥 \", \"desc\": \"\", \"answer\": \"香港：古天乐  周润发  梁朝伟  任达华 \\r\\n韩国：RINI    宋承宪  \\r\\n台湾：小猪    仔仔\\r\\n国外：小贝\"}",
  "{\"qid\": \"qid_600662082255417032\", \"category\": \"商业/理财-产业信息\", \"title\": \"龙岗最大的净水器厂家是哪个呀 \", \"desc\": \"龙岗最大的厂家是哪个呀\", \"answer\": \"“怡盛源”(www.ysyjs.com)是深圳市怡盛源净水产业有限公司旗下品牌，是中国著名品牌、中国净水器十大品牌之一，怡盛源精于净水材料工艺技术的研发与优化组合、全方位解决及单位的生活饮用水问题，是国内首屈一指的净水器生产厂家。 怡盛源净水器生产厂家，凭借高档次的做工和外观、人性化的设计、众多功能的合理化整合等诸多优点，正受到越来越多消费者的喜爱。怡盛源新技术的应用解决了当前水源污染对消费者饮水健康造成的危害，对缓解居民饮水新方式具有划时代的意义，怡盛源解决家庭饮水难题。 怡盛源净水器采用的是进口滤芯，膜面积相对国产膜其表面微孔均匀、稳定不易污秽、不易断裂、膜通量大、保证出水品质，先进的净化工艺有效地净化我们的自来水，让我们能够喝上健康安全的净化水。美国进口KDF能有效去除水中的重金属和余氯，优质椰壳活性炭高效吸附水中的异色异味，保证出水水质无色无味、甘醇可口。有了怡盛源净水器我们不用再担心自来水发黄和刺鼻的漂白水味道，同时高精度的过滤让水中有害物质被阻挡在净水之外，可怕的红虫线形虫从此与你无缘，怡盛源净水器我看行！ 怡盛源除了净化能力出众其售后服务也非常到位，每一位购买怡盛源净水器的客户怡盛源会定期派遣售后服务人员到用户家中进行水质检测、现场维修、滤芯更换等，让你时时刻刻都能喝上健康安全的饮用水，保证您24小时都能使用的我们产品。混乱的净水市场，品种繁多的净水器品牌，怡盛源做为净水器十大品牌之一，它的出现让消费者多一份选择。 怡盛源让消费者更加健康让家庭更加温馨，怡盛源深圳净水器生产厂家以净化中国水质为己任为消费者提供安全高效的净化产品，怡盛源让消费者多一份选择。公司名称  深圳怡盛源净水产业有限公司地址  广东 深圳 龙岗区《赠人玫瑰手有余香，祝您好运一生一世，如果回答有用，请点“好评”，谢谢^_^!》\"}",
  "{\"qid\": \"qid_2159798797982929645\", \"category\": \"健康-妇产科-产科\", \"title\": \"请推荐几个好看的韩国电影 \", \"desc\": \"\", \"answer\": \"惊喜派对,人形师,恋爱魔术师,谁都有秘密,我的野蛮女友\"}",
  "{\"qid\": \"qid_4436489053185970120\", \"category\": \"烦恼-恋爱\", \"title\": \"她失恋不久，我们有结果吗？我们一直是同班同学，原来我一直很喜欢她 \", \"desc\": \"我们一直是同班同学，原来我一直很喜欢她。\\r可她原来有男友。现在她了，我想向她说明我对她的感情，这样有结果吗？\", \"answer\": \"有!\\r\\n不过先别说,这个时候就算答应你,也不稳定\\r\\n继续对她好\\r\\n爱她\\r\\n过段日子,等她慢慢走出失恋的阴影,也开始喜欢你了\\r\\n再说不迟!\"}",
  "{\"qid\": \"qid_2070352552714657550\", \"category\": \"汽车\", \"title\": \"机动车年检都要带什么证件？ \", \"desc\": \"机年检都要带什么证件？\", \"answer\": \"驾驶证、行驶证、各种税费缴纳证明、身份证、还要把罚款都交了\"}",
  "{\"qid\": \"qid_2372099682189065657\", \"category\": \"健康-人体常识\", \"title\": \"去打胎，请半天假够吗？急！！请经验者回答 \", \"desc\": \"\", \"answer\": \"最好休息一个月。 \\r\\n在做完流产后，应当在医生的指导下，加强自我保护措施，否则可能会对身体健康产生不利影响。这些措施包括： \\r\\n1、适当休息：头三天最好卧床休息，否则可能延长阴道出血时间，半个月内最好避免参加体力劳动和体育锻炼。 \\r\\n2、适当补充营养：要适当补充营养，多吃一些富含蛋白质、维生素的食品，如新鲜水果蔬菜、瘦肉、鲜鱼、蛋类、奶或豆制品等。 \\r\\n3、注意个人卫生：人流后要注意保持个人卫生，否则易造成感染。流产后一个月内应避免性生活；出血期间勤换卫生护垫；半个月内不能进行坐浴，只能淋浴。 \\r\\n4、注意出血情况：流产后的一周左右，阴道内会有红色或淡红色分泌物，其量由多到少，无明显气味。如果出血时间超过15天，出血量大过月经量，特别是分泌物有臭味，并伴有发烧、腹痛等症状，则应及时去医院检查和治疗，以免病情加重。 \\r\\n5、做好避孕工作：一旦恢复性生活，要做好避孕工作，避免再次意外受孕。\"}",
  "{\"qid\": \"qid_2033705502100600130\", \"category\": \"游戏-网页游戏\", \"title\": \"文明度怎样增加的快啊 \", \"desc\": \"文明度怎样增加的快啊\", \"answer\": \"怎么长得都不快....\\r\\n \\r\\n\\r\\n看着里面有建筑提升的声望度，当然如果肯花资源的话就冲神庙，小型会+50文明，大型+250\"}",
  "{\"qid\": \"qid_3558981436792464568\", \"category\": \"教育/科学-理工学科-化学\", \"title\": \"化学题在一个体积的密闭容器放入3L气体R和5L气体Q，在一定条件 \", \"desc\": \"在一个体积的密闭容器放入3L气体R和5L气体Q，在一定条件下发生反应：2R（气）+5Q（气）=4X（气）+nY(气），反应完全后，容器温度不变，混合气体的压强是原来的87.5?,则方程中的n值是（  ）\\rA、2      B、3     C、4     D、5\", \"answer\": \"选A。\\r\\n\\r\\n\\r\\n因为完全反应后 压强改变所以△n改变。也就是2+5≠4+n所以排除B项。再就是完全反映后压强变为原先的87.5%也就是说减小了，那么反应就是向系数和小的方向进行。所以只有A项符合\\r\\n\\r\\n  望采纳。\"}",
  "{\"qid\": \"qid_1732805277774471678\", \"category\": \"商业/理财-股票\", \"title\": \"一道股票计算题，高手请进！题目：我现在购买某一股票，股价是2.0 \", \"desc\": \"题目：我现在购买某一，股价是2.00元，我要购买5000股，三天后，该股票股价上涨为2.10元。\\r在这一次交易中，我一共要交多少的费用？我能赚到的实际是多少钱？\\r\\r麻烦高手把计算过程写清楚一些！谢谢！\", \"answer\": \"你的资金是2*5000=10000\\r\\n手续费+印花税=千分之2.5*10000=25+过户费1.2=26.2\\r\\n以上是买进的费用\\r\\n下面是卖出的费用\\r\\n你的资金是2.10*5000=10500\\r\\n手续费+印花税=千分之2.5*10500=26.25+过户费1.2=27.45\\r\\n注明,有的时候有过户费,有的时候是没的,我的交易基本上没有过户费,但上次就有1.2元了\\r\\n你一共交费是26.2+27.45=53.65\\r\\n你赚了10500-10000=500-53.65=446.35\"}",
  "{\"qid\": \"qid_1373866045137056510\", \"category\": \"生活-美容/塑身\", \"title\": \"去皱眼霜选什么好？ \", \"desc\": \"去皱选什么好？\", \"answer\": \"你可以试试我姐用的这中艾弗莎抗皱眼霜嘛，我看到她才没有好久，也就一两个月吧，现在基本上都看不出来了，说这个是能显著淡化眼部幼纹与细纹的作用，呵呵，你可以去了解一下。\"}",
  "{\"qid\": \"qid_7713653475767665323\", \"category\": \"娱乐-谜语\", \"title\": \"什么车无轮？什么猪无嘴？什么驴无毛？什么屋无门？什么书无字？什么? \", \"desc\": \"\", \"answer\": \"第一个答案：乞朱为吾昭雪 \\r有个典故的，在明太主朱元章贪官当道 ,人们有怨都很难得到公正的审判 有一穷书生, 考试三年而不第. 并且家中田地被占官官相卫他无处伸冤 根本不可能去告御状。 终于他想出个方法 能让太主知道他的怨情 。他出了这么个 ，让别人猜 ，由于很难很快传遍天下 ，也就传到了太主的手里 太主看了后， 便把他招来问你有什么怨情？他把所有的委屈都说了出来 贪官被斩首了 ，可他们总不明白书生怎么告的御状 。后来 ，迷揭开了 所有有的秘密全在那个诗歌里 \\r何车无轮？：古时候，农民犁地，在牛身后的拉的叫犁车，犁车是用来开荒的，那时候，人们管开荒又叫‘起地’，所以‘起’通‘乞’代表乞求的意思 \\r何猪无嘴？：这个自然代表太主本人，猪代表朱字，说太主很少慰问国事，像无嘴一样 \\r何驴无毛？：也代表太主本人，太主本来当过和尚，那是广交贤良，以天下为心，用这句和上面的那句对比，说明太主应该管国事了，所以，这句代表的字就是‘为’字，表示太主该有所为了。 \\r何屋无门：这个屋字通吾字，代表书生自己， \\r何书无字：古代的诏书都是太监念的，而太监根本不认识字，所以，诏书就是用口头来传达的，所以，这句代表的字就是‘昭’ \\r何花无叶：这个很简单了，大家都会想到的，就是雪花了，这句代表的就是‘雪’字 \\r所以，整个句子连起来就是‘乞朱为吾昭雪’ \\r天下之博学者，当数书生，令以后之人没有不为之叹服的。 \\r\\r第二个答案：我爱你一万年 \\r何车无轮：我（我）车无轮。我的车本来就没有轮子，轮子给我卸了，但没有钱买新的轮子，只好放在仓库里。 \\r何猪无嘴：矮（爱）竹无嘴。矮竹，就是我们说的文竹。植物而已，哪里来的嘴巴？ \\r何驴无毛：泥（你）驴无毛。谁都知道。 \\r何屋无门：蚁（一）屋无门。蚂蚁的家是洞口，没有门的。 \\r何书无字：万（万）树无字。多少树都没有字的。 \\r何花无叶：年（年）画无叶。年画而已，和叶子确实没有关系。 \\r\\r第三个答案：老鼠喜欢大米 \\r何车无轮：老车无轮。车老了，要拆掉，拆车是先拆轮子的，可能是由于轮子上有橡胶，可以重新回收的缘故。当然轮子也没有了。 \\r何猪无嘴：紫（子）竹（猪），一种植物。不是动物，当然没有嘴。子猪的子，就是鼠。学过天干地支的人都知道。 \\r何驴无毛：锡驴无毛。为什么呢？无锡古代盛产金属锡，北人用毛驴来运送锡矿，获利丰厚。然而驴经常在锡矿中，容易锡中毒，浑身毛发尽脱。常见运锡之驴无毛。是为锡（喜）驴无毛。 \\r何屋无门：幻（欢）屋无门。典出《金刚经》。 \\r何书无字：大叔无字。大叔是农村的叫法。一般大叔是只有名字，不会专门学文人取什么“字”“号”等等乱七八糟的东西的。 \\r何花无叶：米花无叶。谁都知道。爆米花直接可以吃的，不是什么植物。 \\r合起来就是老鼠喜欢大米(一般正确的答案)。典出智力题目：大米的男朋友是什么？ \\r\\r第四个答案：风雨途中同心 \\r（风）车无轮 \\r（雨）珠（谐 猪）无嘴 \\r（秃）驴无毛 \\r（中）午（谐 屋）无门 \\r（桐）树（谐 书）无字 \\r（心）花无叶 \\r风雨秃中桐心 谐 风雨途中同心 \\r\"}",
  "{\"qid\": \"qid_9193441501481640502\", \"category\": \"教育/科学-理工学科-心理学\", \"title\": \"怎样理解“驾驶它的是我们自己无忧无虑的心灵”这句话的含义是苏教版 \", \"desc\": \"是苏教版的语文，请速度回答，PLEASE！！！！！\", \"answer\": \"小船随着人的心意前行，人们的心灵在大海上自由驰骋，心灵和大海已经完全融合在一起。      \\r\\n给分\\r\\n谢谢\\r\\n\"}",
  "{\"qid\": \"qid_9149020487491891510\", \"category\": \"商业/理财-财务税务\", \"title\": \"财务ERP软件；是什么啊？主要是操作些什么啊？大致怎么用啊？怎么? \", \"desc\": \"\", \"answer\": \"ERP (Enterprise Resources Planning, “企业资源计划”系统) 是由不同功能模块组成的软件包，它采用计算机和网络通讯技术的最新成就，是旨在对企业的多种资源进行计划和实现物流、资金流、信息流 (“三流”) 合一的操作平台和管理信息系统。\\r\\nERP中的财务管理涉及到\\\"总账,UFO报表,固定资产,应收款管理,应付款管理,专家财务评估,公司对帐,网上银行,现金流量表,票据通,报帐中心,网上报销,WEB财务\\\"等等,它不只涉及到公司的财务管理还和其他模块有接口(这才是他的应用价值所在),如和采购管理,分销管理,存货核算,库存管理都有接口,可自动处理应收\\\\应付单据,库存盘盈\\\\盘亏.它将公司的资金流高度集成在一个系统中,即时掌控企业的财务运营情况.\\r\\n除了财务会计外,ERP还涉及到管理会计,如资金管理,成本管理,项目管理,预算管理等等. \\r\\n你购买了软件，就会有相应的培训的，这个比单纯的财务软件要复杂一些。\\r\\n\"}",
  "{\"qid\": \"qid_283192057145502214\", \"category\": \"游戏-网络游戏\", \"title\": \"精神专注怎么给自己用啊？我刚刚学会，可惜不能给自己用，怎么办啊？ \", \"desc\": \"专注怎么给自己用啊？我刚刚学会，可惜不能给自己用，怎么办啊？\", \"answer\": \"点击屏幕左上角的你的名字，然后在点技能啊~这样就可以了\"}",
  "{\"qid\": \"qid_8515339931774021497\", \"category\": \"游戏-单机游戏-轩辕剑\", \"title\": \"轩辕剑4如何用法宝 \", \"desc\": \"\", \"answer\": \"一般来说法宝是可以让你用一些特殊技能的，在轮到你行动时，在命令菜单中可用\\r\\n被动技能的（如死亡复活），在遇到相应状况，法宝自动起作用\"}",
  "{\"qid\": \"qid_3624710425439488528\", \"category\": \"游戏-盛大游戏-传奇世界\", \"title\": \"搞个白金账号要什么东西 \", \"desc\": \"\", \"answer\": \"抽奖等 或买书 有RMB买也可以\"}",
  "{\"qid\": \"qid_4519308873129627169\", \"category\": \"娱乐-宠物\", \"title\": \"我家的吉娃娃指甲长的太长,也买了专用的剪指钳,但是它就是不让剪.? \", \"desc\": \"\", \"answer\": \"我家狗狗也是不让剪,主要原因是因为狗狗的指甲都很敏感.所以我买了一个宠物专用的\\\"雷达罩\\\" 学名叫什么我也不知道,只是觉得那东西戴在狗狗身上,就跟一个雷达一样.狗狗戴上那个一个,你让你家人扶住他的脖子,让他别看到你在剪他的指甲,他自然就安静了.然后剪的时候尽量别剪太深,但是下手要快,不要犹豫,否则他会警觉的.反正我家的狗狗以后剪指甲的时候就再也没闹过.去宠物美容太远,每次还花钱,不值得.其他的也不一定能管用,这招其实挺管用的.一般小商品市场都有卖,而且不贵\"}",
  "{\"qid\": \"qid_8811094182532219078\", \"category\": \"商业/理财-财务税务\", \"title\": \"限股需要缴纳个人所得税吗？去年转让的限售股需要缴纳个人所得税吗？ \", \"desc\": \"去年转让的限售股需要缴纳个人所得税吗？\\r\\n\", \"answer\": \"根据《财政部 国家税务总局 证监会关于个人转让上市公司限售股所得征收个人所得税有关问题的通知》（财税［2009］167号）的规定，自2010年1月1日起，对个人转让限售股取得的所得，按照“财产转让所得”适用20％的比例税率征收个人所得税。 \\r\\n\\r\\n个人转让限售股，以每次转让限售股转让收入，减除股票原值和合理税费后的余额，为应纳税所得额。即 \\r\\n\\r\\n应纳税所得额＝限售股转让收入－（限售股原值＋合理税费） \\r\\n\\r\\n应纳税额＝应纳税所得额×20％ \\r\\n\\r\\n限售股原值是指限售股买入时的买入价及按照规定缴纳的有关费用。合理税费是指转让限售股过程中发生的印花税、佣金、过户费等与交易相关的税费。 \\r\\n\\r\\n如果纳税人未能提供完整、真实的限售股原值凭证，不能准确计算限售股原值的，主管税务机关一律按限售股转让收入的15％核定限售股原值及合理税费。 \\r\\n\\r\\n\"}",
  "{\"qid\": \"qid_478271179941814867\", \"category\": \"娱乐-宠物\", \"title\": \"狗狗用的东西有店铺推荐吗？谢谢了。[可爱] \", \"desc\": \"用的东西有店铺推荐吗？谢谢了。[可爱]\", \"answer\": \"直接上淘宝买吧  很多的哦   你家的是什么狗狗呢  请点击下方的好评哦 谢谢\"}",
  "{\"qid\": \"qid_2306408878195592765\", \"category\": \"电脑/网络-硬件\", \"title\": \"技嘉m61Pme\", \"desc\": \" \\r： 2G\\r主板：技嘉m61Pme-s2\\r现在想配一个显卡，不知道用什么的好？ 大家介绍一下。。\\r性价比要高点的，当然价格要便宜的。最好控制在500左右。 \", \"answer\": \"技嘉M61PME-S2采用nVIDIA GeForce6100-430单芯片组，集成GeForce 6100显示核心支持AM2接口CPU不算太老还可以用，跟你配置相匹配的显卡500元内：\\r\\n1.铭瑄 GT240高清版\\r\\n核心频率 550MHz \\r\\n显存频率 2000MHz \\r\\n显存类型 GDDR3 \\r\\n显存容量 512 \\r\\n显存位宽 128bit \\r\\n显存描述 GDDR3显存 \\r\\n显存速度 1ns \\r\\n小影霸GF4后羿版：跟上面差不多\\r\\n2.昂达HD5550 512MB GD3 双敏火旋风2 HD5550 V1024小牛版\\r\\n显存类型：GDDR3\\r\\n核心频率：550MHz\\r\\n显存频率：1600MHz\\r\\n显存容量：512\\r\\n显存位宽 128bit\\r\\n显存速度：1.1ns\\r\\n双敏火旋风2 HD5550 V1024小牛版：跟上面差不多\\r\\n上述是比较新的显卡，够用就行\"}",
  "{\"qid\": \"qid_3313689307220512171\", \"category\": \"健康-内科-肾内科\", \"title\": \"请大家帮忙大谢诸位了生殖器尿道中间部位海面脆骨上有一个小黄豆大小 \", \"desc\": \"生殖器尿道中间部位海面脆骨上有一个小黄豆大小的小疙瘩 在里面从外面用手摸 能感觉到不痛不痒 不妨碍小便 两年多了是什麽病 怎样治疗 还有左耳朵根儿长了两个小瘤 有一次发炎了去医院开了点儿黄金膏 抹了 消肿了 黄金膏是黑色的抹完后粉瘤也变黑了 怕手术 不理它没事吧 我岁数36 结婚8年 身高1.67 体重68公斤 谢诸位\", \"answer\": \"您好\\r\\n病情盲目猜测没意义\\r\\n建议应及早到医院进一步检查明确\\r\\n检查后遵医嘱对症采取针对性的治疗即可！\"}",
  "{\"qid\": \"qid_2038844104592063340\", \"category\": \"电脑/网络-互联网-上网帮助\", \"title\": \"考试CPA考过的请指教经验！ \", \"desc\": \"考过的请指教经验！\", \"answer\": \"这个容易办，紧扣教材，万变不离其中\"}",
  "{\"qid\": \"qid_7209364090176418758\", \"category\": \"生活-美容/塑身-化妆\", \"title\": \"韩智敏代言的梦妆BB霜效果怎么样？我想买呢！！一直很喜欢韩智敏， \", \"desc\": \"一直很喜欢韩智敏，人长得靓丽，也水当当。她代言的梦妆品牌我也一直很喜欢，最近广告常播的梦妆BB霜，我正打算买来试试，不知道效果怎么样呢？求真相！谢谢啦！\", \"answer\": \"楼主，很巧哟，我现在正在用的就是梦妆的BB霜呢，它分清爽型和滋润型两种，所以你要根据自己肌肤的类型来选择。我选的是那款滋润型的多效BB矿物修容霜，防晒指数SPF20/PA+，我觉得很适合我的皮肤。它里面含有智臻繁花复合体（用过她家多效系列的护肤品，所以对这个成分很熟悉哦～），能够帮助增强肌肤弹性，有效改善肌肤黯沉及瑕疵，最近朋友都说我肤色看起来亮很多，整个皮肤感觉QQ的，很有弹性。嘿嘿~还有那个复合矿物元素，是天然的成分哦，不会让色素在脸上沉积，长时间使用都不用担心会伤害皮肤。而且质地比较柔滑盈润，不容易堆积，在脸上轻轻一抹，就化开了。用它修饰肤色，想要拥有一个完美裸妆肤完全不是梦啊。所以我化妆的时候遮瑕膏、粉底什么的都省了，直接用它，一步就搞定了。它还有很好的防晒效果，UVB、UVA都能全部隔离，到户外或者是面对电脑工作，带着它，特别的方便和实用，所以不得不心水它啊~！\"}",
  "{\"qid\": \"qid_4925855189856708274\", \"category\": \"电脑/网络-互联网-上网帮助\", \"title\": \"请问怎样才能获的更高积分 \", \"desc\": \"\", \"answer\": \"多问多答,发挥想象力,当然不要乱答,不然别人会扔鸡蛋了\\r\\n总之这里是个帮助别人的地方,争强好胜是可以的,但是不要用非法手段\"}",
  "{\"qid\": \"qid_5438551119871866263\", \"category\": \"教育/科学-理工学科-数学\", \"title\": \"这数学题你会吗?关于x的方程x^2\", \"desc\": \"关于x的方程x^2-x+m在[-1,1]上有实数解,则m的取值是?\", \"answer\": \"连个方程都没有!!!\\r\\n令f(x)=x^2-x+m,讨论:\\r\\n①如果在[-1,1]上只有一个根,需要f(-1)*f(1)<=0\\r\\n  所以得到-2<=m<=0\\r\\n②如果在[-1,1]上有两个根,需要:\\r\\n  对称轴在[-1,1]上(已满足),判别式为非负!f(1)>=0,f(-1)>=0\\r\\n  得到0<=m<=1/4\\r\\n所以综上,m的范围是-2<=m<=1/4\"}",
  "{\"qid\": \"qid_655643042658033111\", \"category\": \"游戏-单机游戏-魔兽争霸\", \"title\": \"1.21版本什么时候出？？ \", \"desc\": \"1.21版本什么时候出？？\", \"answer\": \"不知道啊\\r\\n应该不会太晚吧\\r\\n1.20的玩着也不错啊!!\"}",
  "{\"qid\": \"qid_1081892684767137198\", \"category\": \"生活-育儿\", \"title\": \"3岁小孩拉肚子，拉出白浆状有点咳嗽，咳嗽后容易拉出白色浆状物，这 \", \"desc\": \"有点，咳嗽后容易拉出白色浆状物，这是怎么回事？她肚子不痛。\", \"answer\": \"带宝宝去化验一下大便。\"}",
  "{\"qid\": \"qid_8725600614872540890\", \"category\": \"游戏-腾讯游戏-QQ幻想\", \"title\": \"现在新出的法宝药师四十五级.想去买个法宝.去什么地方可以买得到. \", \"desc\": \"四十五级.\\r\\r想去买个法宝.\\r去什么地方可以买得到.\\r要花多少钱.\\r还有就是买什么最好.\\r买来是带什么地方?\\r是武器那里吗?\", \"answer\": \"   想买法宝到楼兰城郊地图最左上脚荣光使者那里买，你45要是名誉有1001就买个清气净璃瓶，没有1001就买个碧如意吧！价钱不贵，具体多少我忘了，但你肯定买的起，法宝带在更新后一个新的空格。\\r\\n   现在最高的都是60级的仙器套装。\"}",
  "{\"qid\": \"qid_5527707343652499605\", \"category\": \"生活-保健养生\", \"title\": \"关于EFT保健品事，还要发行股票是真的？我妈妈正在做EFT，谁知 \", \"desc\": \"我妈妈正在做EFT，谁知道这到底是不是骗人的把戏？最近又说EFT要发行，得赶紧买货好成为会员，才能享受到将来的股票大量收益等等，谁能告诉我此事的后果？\", \"answer\": \"我做过.千万不要买!因为直销员和公司之间的关系不对等,消息也可能是假的,只是公司为了促销.过一阵子再看看!\\r\\n电子商务双轨制公司,价格太贵,不符合中国国情.\\r\\n在美国,老板叫秦杰.\\r\\n和\\r\\n安旗公司一样,如果走传统开店的路,就会被工商局查封!对这些公司不要太相信!\"}",
  "{\"qid\": \"qid_2120993978482268954\", \"category\": \"教育/科学-学习帮助\", \"title\": \"商鞅变法和北魏孝文帝改革都获得成功的主要原因A顺应历史发展趋势B \", \"desc\": \"A顺应发展趋势\\rB符合封建统治阶级的利益\\rC代表广大人民的利益\\rD得到统治者的支持\\r答案：B为什么ACD不对？\", \"answer\": \"1商鞅变法符合秦国的国情,实质上变法是为了维护君主专制,加强维护国君统治人民的权力,维护国君至高无上的地位.因此得到秦孝公的大力支持,取到变法成功的第一步.\\r\\n2秦国当时是一个农业落后,氏族势力很深的国家.商鞅,主要起内容是发展农业生产,符合秦国的需要.即成功的第二步.\\r\\n3推行法家思想,万事皆决于法,在一定程度上缓和秦国的国内矛盾.\\r\\n所以AB正确\\r\\nB项,当时商鞅变法得到秦孝公的支持.北魏孝文帝改革,随着国内矛盾和民族矛盾上升,北魏统治者认识当只有进行汉化改革,才符合自己统治.\\r\\nC无论当时怎么变法,都是维护统治阶级利益 ,不可能代表广大人民利益.\\r\\nD商鞅变法得到秦孝公支持.北魏孝文帝改革,自身就是统治者.\"}",
  "{\"qid\": \"qid_9110737097744485302\", \"category\": \"生活-起名\", \"title\": \"饮酒与怀孕我想生个猪宝宝，请问什么时候开始已经不能再喝酒了，因为 \", \"desc\": \"我想生个宝宝，请问什么时候开始已经不能再喝酒了，因为前几天还喝了酒\", \"answer\": \"喝酒后4周再进行此项计划，而且这其中不能再碰酒了。\"}",
  "{\"qid\": \"qid_3913105430061039865\", \"category\": \"娱乐-博彩\", \"title\": \"请教亚盘各位大师，网上哪里可以玩澳门的上下盘呢？谢谢了 \", \"desc\": \"各位大师，网上哪里可以玩澳门的上下盘呢？谢谢了\", \"answer\": \"可以玩，但提款不方便\"}",
  "{\"qid\": \"qid_8325162146787472205\", \"category\": \"娱乐-博彩\", \"title\": \"第一次发图。。。（图2）请高手进来批一下。。第一次发图。拍摄不好 \", \"desc\": \"第一次发图。不好虽见笑```谢谢进来帮批示的朋友。。：）））\\r\", \"answer\": \"6 12场感觉挺悬的...祝好\"}"
};

std::vector<std::string> tokens = {
  "qid|qid_1815059893214501395|category|烦|恼|恋|爱|title|请|问|深|入|骨|髓|地|喜|欢|一|个|人|怎|么|办|我|不|能|确|定|对|方|是|不|是|喜|欢|我|我|却|想|desc|我|不|能|确|定|对|方|是|不|是|喜|欢|我|我|却|想|分|分|秒|秒|跟|他|在|一|起|有|谁|能|告|诉|我|如|何|能|想|他|少|一|点|answer|一|定|要|告|诉|他|你|很|喜|欢|他|很|爱|他|虽|然|不|知|道|你|和|他|现|在|的|关|系|是|什|么|但|如|果|真|的|觉|得|很|喜|欢|就|向|他|表|白|啊|起|码|你|努|力|过|了|女|生|主|动|多|少|占|一|点|优|势|的|呵|呵|只|愿|曾|经|拥|有|到|以|后|就|算|感|情|没|现|在|这|么|强|烈|了|也|不|会|觉|得|遗|憾|啊|与|其|每|天|那|么|痛|苦|的|想|他|恋|他|还|不|如|直|接|告|诉|他|不|要|怕|回|破|坏|你|们|现|有|的|感|情|因|为|如|果|不|告|诉|他|你|可|能|回|后|悔|一|辈|子|",
  "qid|qid_2063849676113062517|category|游|戏|完|美|游|戏|诛|仙|title|我|登|陆|诛|仙|2|时|总|说|我|账|号|密|码|错|误|但|是|我|打|的|是|正|确|的|就|算|不|对|我|desc|answer|被|盗|号|了|我|的|号|在|22|号|那|天|被|盗|了|跟|你|一|样|情|况|link|密|码|与|账|号|错|误|我|密|保|都|有|了|呐|邮|箱|换|密|码|也|不|行|还|被|删|了|号|伤|心|兼|郁|闷|呵|呵|盗|号|了|建|议|跟|完|美|申|请|把|号|要|回|来|或|者|玩|新|的|号|",
  "qid|qid_6625582808814915192|category|游|戏|网|络|游|戏|title|斩|魔|仙|者|称|号|怎|么|得|来|的|desc|斩|魔|仙|者|称|号|怎|么|得|来|的|answer|楼|主|您|好|以|下|为|转|载|r|r|圣|诞|前|热|身|来|生|肖|传|说|做|斩|魔|仙|者|r|r|一|年|一|度|的|圣|诞|节|快|要|来|临|了|大|街|小|巷|商|户|们|都|在|忙|着|准|备|12|月|25|日|圣|诞|的|来|临|而|这|时|候|一|些|妖|魔|也|正|蠢|蠢|欲|动|准|备|作|乱|作|为|生|肖|世|界|肩|负|维|护|世|界|和|平|拯|救|全|人|类|的|生|肖|使|者|怎|么|能|不|有|所|行|动|为|了|生|肖|世|界|的|安|定|而|做|防|范|准|备|r|r|要|让|妖|魔|鬼|怪|能|对|你|有|所|心|悸|除|了|自|己|本|身|武|艺|要|高|强|最|好|能|在|妖|魔|界|打|出|知|名|度|这|样|当|你|的|亲|朋|好|友|被|妖|魔|袭|击|时|只|要|爆|出|你|的|名|号|这|些|妖|魔|上|就|会|落|荒|而|逃|岂|不|好|哉|那|么|斩|魔|仙|者|这|个|响|亮|的|称|号|应|该|足|够|能|震|慑|住|妖|魔|让|他|们|铭|记|在|心|了|吧|r|r|斩|魔|仙|者|的|称|号|r|r|而|且|这|个|斩|魔|仙|者|的|称|号|并|不|是|人|人|都|能|得|到|的|只|有|成|功|挑|战|70|级|副|本|中|的|隐|藏|boss|羽|翼|仙|的|人|才|能|获|得|此|称|号|并|且|前|提|条|件|是|在|12|月|18|日|12|月|25|日|之|间|第|一|队|成|功|挑|战|羽|翼|仙|的|人|才|能|获|此|称|号|因|此|此|称|号|在|全|服|范|围|内|是|绝|对|不|可|能|超|过|5|个|的|r|r|要|挑|战|羽|翼|仙|可|不|是|一|件|容|易|的|事|首|先|要|在|70|级|副|本|中|打|败|4|个|强|大|的|boss|在|打|完|副|本|的|第|4|个|boss|有|一|定|几|率|获|得|道|具|羽|翼|真|元|有|了|羽|翼|真|元|后|就|可|以|与|羽|翼|仙|进|行|一|场|战|斗|羽|翼|仙|就|站|在|第|4|个|boss|的|旁|边|只|是|没|有|道|具|是|不|能|进|入|战|斗|的|r|r|羽|翼|仙|r|r|在|12|月|18|日|12|月|25|日|活|动|期|间|成|功|挑|战|羽|翼|仙|后|的|第|一|支|队|伍|就|可|以|获|得|兑|换|斩|魔|仙|者|的|道|具|烈|火|珍|珠|旗|当|然|如|果|你|在|这|场|激|烈|的|战|斗|中|不|幸|捐|躯|那|么|当|然|是|不|会|得|到|这|个|道|具|的|得|到|了|这|把|烈|火|珍|珠|旗|的|玩|家|就|可|以|到|npc|燃|烧|使|处|兑|换|称|号|了|r|r|这|样|兼|具|高|强|能|力|和|超|强|人|品|才|能|获|得|的|称|号|怎|么|能|不|人|望|而|生|畏|怎|么|能|不|让|那|些|妖|魔|胆|怯|想|要|获|得|的|玩|家|就|快|快|行|动|莫|要|让|人|先|抢|了|这|全|服|唯|一|的|斩|魔|仙|者|称|号|r|r|如|果|满|意|请|采|纳|r|谢|谢|",
  "qid|qid_9204493405205415849|category|商|业|理|财|贸|易|title|有|哪|位|好|心|人|上|传|一|份|女|衬|衫|的|加|拿|大|海|关|发|票|给|我|看|一|下|塞|多|谢|了|desc|多|谢|了|answer|我|给|你|信|息|了|",
  "qid|qid_5049427108036202403|category|娱|乐|度|假|旅|游|title|想|去|澳|州|看|亲|戚|二|个|星|期|怎|么|去|求|教|desc|想|去|澳|州|看|亲|戚|二|个|星|期|怎|么|去|求|教|answer|你|看|亲|戚|申|请|的|是|旅|游|签|证|676|澳|洲|旅|游|签|证|很|容|易|的|r|n|你|让|亲|戚|将|他|的|护|照|签|证|页|和|个|人|信|息|页|扫|描|发|邮|件|给|你|地|址|联|系|电|话|给|你|最|好|是|亲|戚|给|你|一|封|邀|请|函|r|n|你|填|一|个|自|己|的|信|息|表|银|行|存|款|证|明|单|位|的|准|假|证|明|房|产|证|户|口|本|等|材|料|递|交|即|可|r|n7|8|天|就|可|以|拿|签|证|还|是|一|年|多|次|每|次|停|留|30|天|的|签|证|r|n|r|n|我|与|你|情|况|一|样|3|月|份|刚|刚|去|过|很|爽|r|n|不|过|我|怕|麻|烦|交|给|旅|行|社|代|办|的|签|证|收|了|几|百|元|签|证|服|务|费|是|1200|元|省|力|英|文|填|表|都|交|给|他|们|了|r|n|r|n|澳|大|利|亚|签|证|无|需|面|签|的|r|n|",
  "qid|qid_7961760649211429660|category|生|活|礼|节|礼|仪|title|送|男|生|什|么|生|日|礼|物|好|呢|思|考|desc|送|男|生|什|么|生|日|礼|物|好|呢|思|考|answer|运|动|护|腕|游|戏|机|cd|整|蛊|玩|具|男|士|香|水|希|望|能|帮|到|你|麻|烦|给|好|评|",
  "qid|qid_2507568206590316480|category|教|育|科|学|外|语|学|习|英|语|考|试|title|英|语|音|节|划|分|问|题|在|重|读|和|非|重|读|音|节|的|相|邻|处|只|有|一|个|辅|字|组|时|如|果|desc|在|重|读|和|非|重|读|音|节|的|相|邻|处|只|有|一|个|辅|字|组|时|如|果|前|面|重|读|音|节|里|的|元|音|是|长|音|长|元|音|和|双|元|音|则|辅|字|组|属|于|后|面|一|个|音|节|如|果|重|读|音|节|里|的|元|音|是|短|音|短|元|音|则|辅|字|组|属|于|重|读|音|节|例|如|r|长|音|pa|r|fa|ther|ze|ro|mo|tor|si|mon|stu|dent|fea|ture|e|ven|r|短|音|cap|ital|moth|er|ver|y|mod|el|lil|y|stud|y|feath|er|sev|en|r|两|元|音|字|母|之|间|有|一|个|辅|音|字|母|时|辅|音|字|母|归|后|一|音|节|r|r|问|题|一|r|比|喻|capital|如|果|按|底|在|重|读|和|非|重|读|音|节|的|相|邻|处|只|有|一|个|辅|字|组|时|如|果|前|面|重|读|音|节|里|的|元|音|是|长|音|长|元|音|和|双|元|音|则|辅|字|组|属|于|后|面|一|个|音|节|如|果|重|读|音|节|里|的|元|音|是|短|音|短|元|音|则|辅|字|组|属|于|重|读|音|节|例|如|r|长|音|pa|r|fa|ther|ze|ro|mo|tor|si|mon|stu|dent|fea|ture|e|ven|r|短|音|cap|ital|moth|er|ver|y|mod|el|lil|y|stud|y|feath|er|sev|en|r|两|元|音|字|母|之|间|有|一|个|辅|音|字|母|时|辅|音|字|母|归|后|一|音|节|r|r|问|题|一|r|比|喻|capital|如|果|按|底|二|句|话|就|应|该|是|ca|pi|tal|这|样|划|分|啊|为|什|么|真|实|是|cap|i|tal|这|样|划|分|的|r|r|问|题|二|r|还|有|我|在|学|习|英|语|音|节|划|分|中|的|课|程|内|容|学|到|这|里|有|一|点|我|就|不|明|白|了|如|果|先|要|分|辨|出|前|面|重|读|音|节|里|的|元|音|是|长|音|还|是|短|音|再|作|辅|字|组|的|划|分|就|必|须|要|先|分|辨|出|前|面|音|节|是|开|音|节|还|是|闭|音|节|才|知|道|是|长|音|还|是|短|音|了|可|再|分|开|闭|音|节|的|过|程|中|又|要|看|此|辅|字|组|是|在|前|面|音|节|还|是|后|面|音|节|r|r|问|题|三|r|为|有|音|标|的|w|和|r|和|v|在|拼|的|时|候|好|像|拼|出|来|的|音|是|一|样|的|啊|这|是|为|什|么|d|和|那|个|舌|头|伸|出|来|的|音|拼|出|来|的|也|一|样|father|中|的|there|不|是|和|de|一|样|吗|要|怎|样|才|能|分|得|出|区|别|啊|是|不|是|有|区|别|的|是|我|听|不|出|而|已|r|r|问|题|四|r|还|有|就|是|你|们|学|拼|读|规|则|都|在|那|里|学|习|的|怎|么|我|在|网|上|找|不|到|这|样|的|视|频|啊|找|到|的|都|是|文|章|来|的|是|不|是|大|家|都|不|重|视|拼|读|规|则|的|所|以|就|没|那|些|教|程|r|answer|r|音|节|划|分|不|是|像|你|说|的|那|个|样|子|r|你|去|翻|看|一|下|oxfo|字|典|里|面|的|音|标|都|是|根|据|元|音|划|分|的|还|有|一|点|就|是|这|个|划|分|与|词|根|前|缀|后|缀|都|有|关|系|的|r|r|问|题|二|r|亲|我|真|的|想|说|多|听|多|说|比|你|这|样|死|钻|牛|角|尖|区|分|这|个|要|管|用|的|多|r|r|问|题|三|r|为|有|音|标|的|w|和|r|和|v|在|拼|的|时|候|好|像|拼|出|来|的|音|是|一|样|的|啊|这|是|为|什|么|d|和|那|个|舌|头|伸|出|来|的|音|拼|出|来|的|也|一|样|father|中|的|there|不|是|和|de|一|样|吗|要|怎|样|才|能|分|得|出|区|别|啊|是|不|是|有|区|别|的|是|我|听|不|出|而|已|r|r|怎|么|会|一|样|呢|我|觉|得|多|半|可|能|你|发|音|不|准|啊|你|可|以|比|对|口|型|还|有|送|气|的|方|式|多|跟|外|教|探|讨|一|下|r|rd|th|这|完|全|是|两|个|不|同|的|音|无|论|单|拼|还|是|放|在|单|词|里|都|是|不|同|的|r|同|学|我|觉|得|你|多|半|是|发|音|有|问|题|如|果|你|发|音|不|准|那|么|你|听|力|肯|定|好|不|了|r|r|问|题|四|r|有|一|本|书|语|音|基|础|蓝|皮|的|是|英|语|系|的|语|音|基|础|的|教|科|书|很|好|如|果|你|想|系|统|地|学|可|以|考|虑|买|一|本|看|看|r|r|补|充|问|题|r|见|回|答|四|",
  "qid|qid_3642560876521827750|category|教|育|科|学|理|工|学|科|建|筑|学|title|厂|房|内|有|吊|车|起|吊|高|度|怎|么|定|厂|房|内|有|吊|车|牛|腿|高|度|怎|么|定|根|据|desc|内|有|吊|车|牛|腿|高|度|怎|么|定|根|据|什|么|来|定|生|产|的|工|艺|要|求|还|是|要|考|虑|到|其|他|因|素|answer|你|好|r|n|r|n|首|先|按|厂|房|内|的|最|大|部|件|确|定|吊|车|的|起|重|量|和|起|吊|高|度|这|个|一|般|由|工|艺|专|业|确|定|然|后|根|据|吊|车|样|本|提|供|的|吊|车|高|度|定|下|吊|车|轨|顶|标|高|再|按|厂|房|跨|度|起|吊|重|量|等|选|定|吊|车|梁|和|吊|车|轨|道|连|接|件|那|么|牛|腿|高|度|吊|车|轨|顶|标|高|吊|车|梁|高|度|轨|道|连|接|件|高|度|",
  "qid|qid_8990759060620275711|category|育|儿|妈|妈|分|娩|期|title|你|好|请|问|有|疼|痛|晕|厥|史|的|产|妇|可|以|顺|产|吗|desc|你|好|请|问|有|疼|痛|晕|厥|史|的|可|以|顺|产|吗|answer|您|好|可|以|尝|试|一|下|无|痛|分|娩|",
  "qid|qid_5331366642436435012|category|商|业|理|财|个|人|理|财|title|投|资|个|人|理|财|产|品|需|注|意|哪|些|问|题|desc|answer|投|资|个|人|理|财|产|品|哪|些|问|题|需|注|意|r|n|r|n|第|一|切|记|人|民|币|理|财|产|品|不|同|于|储|蓄|国|债|等|产|品|并|非|没|有|风|险|预|期|收|益|率|也|并|非|实|际|收|益|率|不|论|哪|个|银|行|一|般|都|不|会|承|诺|保|本|保|息|之|类|的|最|低|收|益|或|固|定|收|益|打|出|的|底|牌|往|往|是|预|期|收|益|率|无|论|是|哪|一|种|理|财|产|品|它|的|主|要|投|资|对|象|是|国|债|央|行|票|据|和|金|融|债|券|等|因|此|基|金|存|在|的|风|险|理|财|产|品|照|样|存|在|另|一|方|面|风|险|来|自|产|品|本|身|是|否|是|合|法|的|产|品|设|计|上|是|否|有|缺|陷|r|n|r|n|第|二|切|记|个|人|理|财|产|品|的|高|收|益|通|常|伴|随|着|高|风|险|据|有|关|人|士|介|绍|扣|除|各|项|成|本|后|一|般|目|前|一|年|期|的|理|财|产|品|收|益|率|在|2.7|3.1|如|果|银|行|承|诺|的|利|率|超|过|这|个|上|限|投|资|者|在|购|买|时|就|要|谨|慎|行|事|了|r|n|r|n|第|三|切|记|在|购|买|人|民|币|理|财|产|品|时|一|定|要|对|购|买|合|同|和|协|议|进|行|认|真|审|查|一|旦|发|生|问|题|购|买|者|有|可|能|处|于|被|动|位|置|r|n|",
  "qid|qid_4371543678850352818|category|健|康|医|院|title|女|性|经|常|有|作|乳|房|自|查|是|否|不|用|每|年|上|医|院|作|体|检|了|经|常|自|查|乳|房|的|desc|经|常|自|查|乳|房|的|无|异|常|不|用|再|去|查|了|吗|answer|尽|管|业|界|一|直|提|倡|女|性|学|会|乳|房|自|查|以|及|早|发|现|异|常|但|是|也|存|在|部|分|乳|腺|疾|病|如|有|的|乳|腺|癌|早|期|可|能|不|完|全|有|明|显|肿|块|触|及|或|者|乳|房|外|形|无|特|殊|改|变|自|查|会|在|一|定|程|度|上|有|漏|诊|造|成|严|重|后|患|r|n|因|此|自|查|也|不|能|代|替|医|院|相|关|医|生|和|器|械|的|科|学|筛|查|建|议|成|年|女|性|每|年|应|去|正|规|医|院|体|检|一|次|",
  "qid|qid_1573872705600431367|category|文|化|艺|术|文|学|title|我|爱|你|古|文|怎|么|说|desc|古|文|怎|么|说|answer|吾|怜|卿|甚|矣|我|爱|你|真|的|很|爱|r|n|r|n|吾|甚|怜|卿|我|非|常|爱|你|r|n|r|n|吾|甚|怜|卿|耶|我|特|爱|你|呀|",
  "qid|qid_9187122022980985230|category|教|育|科|学|人|文|学|科|历|史|学|title|为|什|么|没|有|副|总|书|记|一|职|desc|为|什|么|没|有|副|总|记|一|职|answer|楼|上|的|说|的|不|对|你|说|的|是|八|大|准|备|的|事|情|人|家|问|的|是|现|在|事|情|r|n|不|设|副|总|书|记|是|为|了|保|证|总|书|记|的|权|威|副|总|书|记|有|些|太|子|和|明|确|二|把|手|的|意|思|会|对|天|子|产|生|威|胁|清|朝|后|期|已|经|不|设|太|子|了|现|在|就|更|没|有|必|要|了|",
  "qid|qid_4106106752044617103|category|教|育|科|学|学|习|帮|助|title|已|知|a|b|都|是|锐|角|tana|4|3|tanb|1|7|求|a|desc|已|知|a|b|都|是|锐|角|tana|4|3|tanb|1|7|求|a|b|answer|tan|a|b|tana|tanb|1|tanatanb|r|n|4|3|1|7|1|4|3|1|7|r|n|28|3|21|4|1|r|na|b|都|是|锐|角|π|2|a|b|π|2|r|n|所|以|a|b|π|4|",
  "qid|qid_7707644486067442493|category|娱|乐|花|鸟|鱼|虫|title|请|教|一|下|这|是|什|么|植|物|谢|谢|desc|请|教|一|下|这|是|什|么|谢|谢|answer|这|应|该|是|迎|春|花|",
  "qid|qid_525979235196905709|category|游|戏|久|游|游|戏|魔|力|宝|贝|title|家|族|建|立|必|须|等|级|我|看|新|浪|任|务|说|家|族|建|立|的|必|须|条|件|是|lv50|desc|我|看|新|浪|任|务|说|建|立|的|必|须|条|件|是|lv50|是|家|族|长|必|须|lv50|还|是|家|族|建|立|的|五|个|人|都|必|须|lv50|啊|answer|家|族|建|立|的|时|候|对|于|家|族|长|和|家|族|建|立|的|同|伴|4|人|的|等|级|全|是|无|限|制|的|这|个|新|浪|在|任|务|上|有|个|小|错|误|r|n|那|么|只|是|推|荐|等|级|为|50|而|已|家|族|建|立|的|任|务|其|实|并|不|难|也|不|是|很|长|之|所|以|推|荐|lv50|主|要|有|两|点|困|难|1|通|过|地|之|神|管|的|任|务|是|制|作|2|分|钟|内|料|理|美|奶|滋|拌|饭|这|里|的|困|难|不|是|2|分|钟|而|是|材|料|问|题|主|要|就|是|15|个|瓷|碗|的|拿|法|见|下|找|朋|友|帮|忙|拿|下|15|个|碗|也|就|没|什|么|了|2|通|过|火|之|神|管|的|任|务|是|第|二|个|小|困|难|如|果|没|有|等|级|好|一|点|的|组|队|的|话|其|五|连|战|恐|怕|会|困|难|一|点|总|体|来|说|还|是|不|难|5|人|组|队|还|是|有|相|当|的|胜|算|的|r|n|回|到|你|的|问|题|结|论|就|是|家|族|的|建|立|无|论|是|族|长|还|是|建|立|的|4|名|同|伴|其|等|级|都|是|无|限|制|的|r|n|r|nps|祝|愿|魔|力|愉|快|家|族|兴|旺|新|年|快|乐|",
  "qid|qid_6001180011544757766|category|健|康|五|官|科|口|腔|科|title|牙|龈|包|住|牙|齿|怎|么|办|desc|包|住|牙|齿|怎|么|办|answer|病|情|分|析|n|你|好|牙|龈|出|血|以|局|部|原|因|最|为|常|见|但|也|有|是|其|他|疾|病|在|牙|龈|组|织|上|出|现|的|一|种|症|状|n|指|导|意|见|n|局|部|原|因|为|牙|龈|炎|和|牙|周|炎|主|要|是|口|腔|卫|生|不|佳|牙|垢|牙|结|石|不|良|修|复|体|补|牙|材|料|食|物|嵌|塞|等|局|部|刺|激|因|素|引|起|牙|龈|充|血|肿|胀|在|刷|牙|咬|硬|物|吃|水|果|时|发|生|牙|龈|出|血|有|时|可|有|少|许|的|自|发|性|渗|血|将|刺|激|因|素|去|除|后|出|血|即|会|好|转|牙|龈|外|伤|主|要|是|指|剔|牙|时|造|成|的|牙|龈|损|伤|致|使|牙|龈|出|血|这|样|情|况|应|采|取|正|确|的|剔|牙|方|法|保|护|牙|龈|全|身|因|素|有|血|液|病|如|血|友|病|白|血|病|再|生|障|碍|性|贫|血|等|有|相|当|一|部|分|患|者|是|最|初|以|牙|龈|出|血|而|到|口|腔|科|就|诊|的|其|他|一|些|慢|性|疾|病|亦|可|导|致|牙|龈|出|血|如|糖|尿|病|维|生|素|c|缺|乏|及|一|些|消|耗|性|疾|病|肝|脏|疾|病|可|因|影|响|凝|血|因|素|的|合|成|也|会|有|牙|龈|出|血|发|现|牙|龈|出|血|不|可|过|度|紧|张|应|及|时|去|医|院|检|查|如|为|局|部|因|素|引|起|经|口|腔|科|治|疗|即|可|如|出|血|量|多|不|易|止|血|时|应|到|内|科|诊|治|要|及|时|做|血|液|系|统|检|查|",
  "qid|qid_3473806824072149837|category|教|育|科|学|人|文|学|科|哲|学|title|加|强|调|的|着|重|号|有|什|么|作|用|列|宁|指|出|物|质|是|标|志|客|观|实|在|的|哲|desc|列|宁|指|出|物|质|是|标|志|客|观|实|在|的|范|畴|这|种|客|观|实|在|是|人|通|过|感|觉|感|知|的|它|不|依|赖|于|我|们|的|感|觉|而|存|在|为|我|们|的|感|觉|所|复|写|摄|影|反|映|r|我|用|表|示|下|圆|点|用|代|替|波|浪|线|用|代|替|下|划|线|请|问|列|宁|的|物|质|定|义|中|的|着|重|号|要|强|调|的|是|什|么|answer|辩|证|唯|物|主|义|物|质|观|的|重|大|意|义|有|r|n|它|指|出|物|质|是|不|依|赖|人|的|感|觉|和|意|识|的|客|观|存|在|坚|持|了|唯|物|主|义|一|元|论|同|唯|心|主|义|和|二|元|论|划|清|了|界|限|r|n|它|指|出|物|质|是|能|被|人|所|感|觉|认|识|的|坚|持|了|可|知|论|同|不|可|知|论|划|清|了|界|限|r|n|它|指|出|客|观|实|在|性|是|物|质|的|唯|一|特|性|这|就|把|物|质|与|具|体|的|物|质|形|态|区|别|开|来|坚|持|了|辩|证|唯|物|主|义|同|旧|唯|物|主|义|划|清|了|界|限|r|n|它|指|出|了|物|质|是|哲|学|范|畴|物|质|既|包|括|自|然|界|也|包|括|人|类|社|会|这|就|揭|示|了|自|然|界|和|人|类|社|会|的|物|质|统|一|性|坚|持|了|历|史|唯|物|主|义|同|历|史|唯|心|主|义|划|清|了|界|限|r|n|r|n|客|观|实|在|是|各|种|物|质|现|象|的|普|遍|本|质|和|唯|一|特|性|",
  "qid|qid_193153085081424041|category|电|脑|网|络|电|脑|装|机|title|请|问|这|样|一|个|配|置|需|要|多|大|的|电|源|cpu:amd|羿|龙|x38450|desc|cpu|amd|羿|龙|x3|8450|盒|r|技|嘉|ga|ma770|s3|rev|2.0|r|内|存|金|士|顿|2gb|ddr2|800|r|硬|盘|希|捷|320g|7200.11|16m|串|口|5|年|盒|r|显|卡|迪|兰|恒|进|hd4850|北|极|星|r|r|r|不|知|道|这|套|配|置|怎|么|样|还|有|哪|些|不|足|高|手|帮|帮|忙|下|个|星|期|就|攒|了|r|听|说|现|在|希|捷|500g|的|负|面|问|题|多|多|很|多|人|推|荐|我|用|320g|的|请|问|500g|的|问|题|真|的|很|多|么|r|电|源|用|ttxp420|的|合|适|么|功|率|是|否|够|用|还|是|用|其|他|的|帮|忙|推|荐|几|款|电|源|answer|你|这|套|配|置|和|我|现|在|用|的|这|台|机|器|的|差|不|多|我|用|的|是|rcpu|amd|羿|龙|x2|5000|黑|盒|配|九|州|风|神|黑|虎|鲸|散|热|器|rcpu|oc|到|3g|r|捷|波|悍|马|hao3|amd|790x|rev|1.0|r|内|存|黑|金|刚|2|x|1gb|ddr2|800|时|序|调|整|为|4|4|4|12|默|电|r|硬|盘|希|捷|500g|7200.11|32m|串|口|5|年|盒|r|显|卡|迪|兰|恒|进|hd4850|公|版|自|己|更|换|zero|8800u|太|空|梭|散|热|器|r|总|的|来|说|这|样|的|配|置|对|付|一|般|的|游|戏|都|很|轻|松|在|22|寸|24|寸|的|显|示|器|上|都|可|以|比|较|完|美|的|运|行|绝|大|部|分|游|戏|你|的|cpu|比|我|的|要|好|一|点|耗|电|要|高|一|点|其|它|的|和|我|的|配|置|差|不|多|r|我|开|始|用|的|电|源|是|航|嘉|磐|石|400|在|这|个|平|台|上|面|一|般|情|况|下|可|以|正|常|使|用|但|是|在|我|把|显|卡|和|cpu|同|时|大|幅|超|频|的|情|况|下|电|源|挂|掉|了|现|在|换|了|个|台|达|的|600w|的|电|源|r|所|以|按|你|的|配|置|建|议|在|不|打|算|超|频|使|用|的|情|况|下|选|择|足|额|输|出|350w|以|上|的|电|源|个|人|建|议|选|择|400w|左|右|的|预|留|一|些|亢|余|用|起|来|更|放|心|一|点|r|目|前|个|人|不|大|推|荐|你|选|择|航|嘉|电|源|最|近|航|嘉|被|暴|电|源|输|出|虚|标|功|率|建|议|你|看|看|鑫|谷|长|城|全|汉|的|大|功|率|电|源|",
  "qid|qid_7313842672675064329|category|教|育|科|学|外|语|学|习|title|翻|译|3shedescribestheexperienceof15|desc|she|describes|the|eerience|of|15|year|old|jessica|from|burlington|jessica's|leg|was|cut|off|at|the|knee|because|of|cancer|she|was|learning|to|wear|a|prosthesis|jessica|tells|about|playing|soccer|she|kicked|the|ball|hard|and|it|flew|off|in|one|direction|while|her|artificishe|describes|the|eerience|of|15|year|old|jessica|from|burlington|jessica's|leg|was|cut|off|at|the|knee|because|of|cancer|she|was|learning|to|wear|a|prosthesis|jessica|tells|about|playing|soccer|she|kicked|the|ball|hard|and|it|flew|off|in|one|direction|while|her|artificial|leg|flew|another|way|she|fell|on|the|ground|and|laughed|answer|she|describes|the|experience|of|15|year|old|jessica|from|burlington|jessica's|leg|was|cut|off|at|the|knee|because|of|cancer|she|was|learning|to|wear|a|prosthesis|jessica|tells|about|playing|soccer|she|kicked|the|ball|hard|and|it|flew|off|in|one|direction|while|her|artificial|leg|flew|another|way|she|fell|on|the|ground|and|laughed|r|r|她|叙|述|了|来|自|burlington|的|１５|岁|女|孩|杰|西|卡|的|经|历|由|于|癌|症|的|缘|故|杰|西|卡|的|膝|盖|以|下|部|分|被|截|肢|了|那|时|她|正|学|着|使|用|一|只|义|肢|来|替|代|腿|杰|西|卡|谈|到|了|踢|的|故|事|一|次|她|对|着|球|猛|踢|了|一|脚|球|朝|着|一|个|方|向|飞|去|的|同|时|她|的|假|腿|却|飞|向|了|另|一|个|方|向|杰|西|卡|摔|倒|在|地|哈|哈|地|笑|了|起|来|r|",
  "qid|qid_2496719599303374558|category|娱|乐|幽|默|滑|稽|title|网|络|上|真|的|没|有|帅|哥|美|女|吗|突|然|间|想|到|的|以|前|在|哪|里|看|到|过|是|这|么|desc|上|真|的|没|有|帅|哥|美|女|吗|r|突|然|间|想|到|的|以|前|在|哪|里|看|到|过|是|这|么|说|的|answer|美|女|哪|有|空|来|成|天|上|网|都|约|会|去|了|r|n|帅|哥|知|道|网|上|都|是|恐|龙|所|以|也|都|不|来|了|",
  "qid|qid_9201495377383673621|category|生|活|育|儿|title|耳|朵|怎|么|了|宝|宝|晚|上|睡|觉|总|是|翻|来|覆|去|的|仿|佛|睡|不|塌|实|而|且|每|次|翻|身|desc|晚|上|睡|觉|总|是|翻|来|覆|去|的|仿|佛|睡|不|塌|实|而|且|每|次|翻|身|的|时|候|总|用|手|挖|耳|朵|找|医|生|看|了|也|没|发|现|里|面|有|什|么|不|知|道|他|为|什|么|这|样|是|不|是|缺|什|么|维|生|素|什|么|的|answer|宝|宝|多|大|了|要|是|4|个|月|左|右|属|于|长|牙|期|间|他|挖|耳|朵|是|因|为|牙|床|痒|痛|",
  "qid|qid_8085049932951069286|category|生|活|购|房|置|业|title|北|京|房|产|证|过|户|手|续|怎|么|办|夫|妻|两|人|一|人|先|走|房|产|过|户|另|一|个|人|手|desc|两|人|一|人|先|走|房|产|过|户|另|一|个|人|手|续|怎|么|办|answer|去|公|证|处|做|一|个|继|承|公|正|然|后|拿|着|继|承|公|正|去|房|地|产|交|易|中|心|可|以|直|接|发|新|证|",
  "qid|qid_3488727986990973326|category|健|康|精|神|心|理|科|心|理|科|title|面|烧|糊|了|还|能|吃|吗|糊|味|明|显|吃|了|对|健|康|影|响|大|吗|desc|糊|味|明|显|吃|了|对|影|响|大|吗|answer|最|好|别|吃|没|好|处|而|且|也|难|吃|容|易|致|癌|",
  "qid|qid_713239302402366328|category|社|会|民|生|求|职|就|业|title|刚|毕|业|的|大|学|生|怎|样|考|职|称|吗|求|教|刚|毕|业|的|大|学|生|可|以|怎|样|考|职|称|desc|求|教|刚|毕|业|的|大|学|生|可|以|怎|样|考|职|称|r|r|我|今|年|刚|毕|业|r|不|是|很|懂|有|关|考|职|称|的|事|情|r|请|问|是|不|是|只|能|从|初|级|开|始|考|r|可|不|可|以|直|接|考|中|级|r|需|要|什|么|资|质|条|件|吗|r|我|现|在|做|的|工|作|是|类|似|于|办|公|室|主|任|的|工|作|r|有|关|企|业|啊|行|政|人|事|管|理|啊|之|类|的|r|单|单|考|计|算|机|和|英|语|职|称|有|用|吗|r|我|现|在|可|以|考|什|么|样|的|职|称|谢|谢|r|谢|谢|r|answer|你|好|r|n|r|n|你|的|工|作|内|容|可|以|报|考|经|济|类|职|称|首|先|是|从|初|级|报|考|报|考|助|理|经|济|师|初|级|条|件|是|你|本|科|毕|业|见|习|期|满|一|年|之|后|直|接|申|请|评|审|不|需|要|考|试|r|n|r|n|今|后|报|考|中|级|就|需|要|考|试|了|经|济|师|中|级|在|你|获|得|助|理|四|年|之|后|就|可|申|请|报|考|r|n|r|n|关|于|职|称|和|职|业|问|题|欢|迎|浏|览|我|的|专|题|职|场|人|生|希|望|有|帮|助|r|n|r|n|祝|你|顺|利|r|n|r|n|r|n|",
  "qid|qid_4604022711015097001|category|娱|乐|动|漫|title|求|好|看|的|动|漫|desc|求|好|看|的|answer|八|犬|狂|欢|节|k|黑|执|事|家|教|海|贼|幸|运|星|夏|目|友|人|帐|新|全|职|猎|人|冰|果|秒|速|五|厘|米|问|题|儿|童|都|来|自|异|世|界|血|型|君|火|影|忍|者|网|球|王|子|名|侦|探|柯|南|",
  "qid|qid_802484035859939592|category|游|戏|巨|人|游|戏|征|途|title|为|什|么|我|起|名|字|老|是|不|合|法|这|个|游|戏|不|可|以|起|带|有|特|殊|符|号|的|名|字|吗|desc|这|个|不|可|以|起|带|有|特|殊|符|号|的|名|字|吗|r|比|如|ρs|等|ㄣ|待|裃|喏|zoe|ヤ|ゞ|ゃ|鉒|錠|answer|目|前|只|能|用|中|文|和|英|文|起|而|且|中|文|有|些|词|组|也|不|行|r|n|像|日|文|智|能|abc|打|出|的|符|号|都|不|行|r|n|r|n|你|可|以|先|起|玩|看|看|不|行|就|删|千|万|别|留|的|名|字|以|后|很|多|事|都|不|方|便|像|推|荐|人|之|类|的|",
  "qid|qid_8757963117572327404|category|健|康|精|神|心|理|科|title|你|相|信|这|个|世|上|有|真|爱|吗|我|觉|得|人|在|一|起|久|了|就|会|有|感|情|你|们|说|呢|desc|我|觉|得|人|在|一|起|久|了|就|会|有|你|们|说|呢|answer|半|信|半|疑|相|处|时|间|长|啦|那|是|感|情|但|那|种|感|情|就|是|一|般|的|那|种|朋|友|人|活|在|世|上|不|可|能|也|不|可|以|没|有|朋|友|那|里|一|点|一|滴|积|累|下|来|的|感|情|是|很|现|实|的|但|是|真|爱|这|个|爱|太|难|懂|太|复|杂|啦|",
  "qid|qid_3841177982407043650|category|教|育|科|学|理|工|学|科|生|物|学|title|我|家|公|猫|为|什|么|晚|上|乱|叫|呀|desc|answer|是|发|情|呢|呵|呵|猫|发|情|没|有|固|定|时|间|的|差|不|多|没|个|月|都|闹|有|的|厉|害|的|一|个|月|有|半|个|月|在|闹|猫|所|以|建|议|你|带|它|去|做|绝|育|吧|r|n|做|绝|育|的|好|处|太|多|了|本|身|它|是|公|猫|对|它|伤|害|特|别|小|不|像|母|猫|还|开|膛|当|天|就|能|活|动|了|且|做|了|对|它|本|身|有|好|处|猫|发|情|期|是|很|痛|苦|的|还|会|影|响|食|欲|r|n|对|你|的|家|庭|也|好|不|用|整|天|被|它|影|响|睡|眠|了|绝|育|过|的|猫|猫|更|粘|人|脾|气|也|会|变|的|温|顺|乖|巧|饭|量|也|会|增|加|哦|",
  "qid|qid_6056160307824183948|category|健|康|医|院|title|你|好|我|家|宝|宝|7|多|月|这|几|天|拉|肚|子|去|医|院|检|查|说|是|细|菌|感|染|肠|炎|吃|desc|你|好|我|家|宝|宝|7|多|月|这|几|天|拉|肚|子|去|检|查|说|是|细|菌|感|染|肠|炎|吃|了|几|天|的|药|反|反|复|复|今|天|拉|泡|泡|answer|给|宝|宝|吃|点|妈|咪|爱|",
  "qid|qid_755887206012435380|category|游|戏|网|易|游|戏|魔|兽|世|界|title|关|于|安|装|台|服|的|问|题|等|cwow|遥|遥|无|期|鄙|人|也|花|了|一|天|时|间|下|载|了|台|desc|等|cwow|遥|遥|无|期|鄙|人|也|花|了|一|天|时|间|下|载|了|台|服|我|的|cwow|装|到|d|盘|台|到|f|盘|可|我|安|装|台|服|的|时|候|他|总|是|直|接|跳|出|cwow|的|登|陆|界|面|这|是|什|么|问|题|r|另|外|希|望|大|家|给|我|个|明|确|详|细|的|解|决|方|案|谢|谢|了|answer|你|机|器|里|同|时|有|国|服|和|台|服|是|不|行|的|那|样|会|有|冲|突|的|r|无|法|正|常|运|行|的|r|r|以|下|办|法|用|下|吧|r|台|服|客|户|端|下|载|r|1|打|开|cwow|文|件|夹|找|到|realmlist.wtf|文|件|用|记|事|本|打|开|它|先|清|空|里|面|的|内|容|再|添|加|set|realmlist|不|包|括|双|引|号|保|存|后|退|出|这|样|在|每|次|要|进|去|台|服|的|时|候|只|需|要|用|wow.exe|来|启|动|如|果|是|进|去|cwow|的|话|就|用|launcher.exe|来|启|动|不|会|造|成|冲|突|另|外|用|wow.exe|启|动|后|登|陆|的|界|面|是|cwow|的|界|面|这|是|因|为|使|用|的|是|cwow|的|客|户|端|的|原|因|不|用|怀|疑|直|接|输|入|id|密|码|登|陆|后|进|入|的|就|是|台|服|r|2|如|果|你|已|经|有|cwow|的|客|户|端|了|再|想|装|twow|的|时|候|需|要|修|改|注|册|表|r|修|改|方|法|r1|点|击|开|始|r2|点|击|运|行|r3|在|运|行|框|里|输|入|regedit4|在|弹|出|的|注|册|表|框|里|点|击|编|辑|r5|点|击|查|找|r6|在|查|找|框|里|输|入|blizzard|entertainment|r7|当|左|边|框|里|搜|索|到|blizzard|entertainment|的|时|候|在|左|边|整|个|删|除|一|直|搜|索|到|出|现|对|话|框|里|说|注|册|表|搜|索|完|毕|为|止|r|安|装|r|然|后|是|安|装|直|接|解|压|安|装|就|ok|了|安|装|后|貌|似|是|1.11.0|的|版|本|视|你|下|载|的|版|本|为|定|r|补|丁|方|面|直|接|通|过|程|序|下|载|速|度|很|快|一|般|能|达|到|80k|s|以|上|的|速|度|1m|adsl|注|册|账|号|此|处|资|料|为|转|载|仅|供|参|考|r|直|接|申|请|体|验|cdk|然|后|开|始|录|入|个|人|资|料|r|r|召|集|令|编|号|rrf070112000032|rrf070112000033|rrf070112000034|rrf070112000035|rrf070112000036|r|注|注|册|帐|号|时|填|写|召|集|令|编|号|可|获|得|14|天|30|小|时|体|验|优|惠|r|非|必|选|项|r|录|入|个|人|资|料|这|里|请|注|意|一|下|请|尽|量|填|写|个|人|准|确|资|料|如|地|区|选|其|它|以|及|真|实|身|份|证|号|码|真|实|住|址|还|有|真|实|的|email|地|址|我|选|的|住|在|钓|鱼|岛|不|过|地|址|栏|下|面|的|都|写|的|真|实|的|r|范|例|南|海|诸|岛|钓|鱼|台|列|屿|中|华|人|民|共|和|国|江|苏|省|南|京|市|xd|r|r|至|此|你|的|体|验|帐|号|基|本|申|请|完|毕|可|以|去|游|戏|里|进|行|下|体|验|和|尝|试|了|这|样|可|以|安|装|2|个|客|户|端|分|别|指|向|cwow|和|twow|输|入|法|此|处|资|料|为|转|载|仅|供|参|考|五|笔|输|入|法|的|朋|友|到|r|下|载|极|点|中|文|输|入|法|安|装|后|在|设|置|选|项|里|调|成|输|出|繁|体|字|r|r|拼|音|输|入|法|的|朋|友|windows|控|制|面|板|区|域|和|语|言|选|项|语|言|详|细|信|息|添|加|微|软|拼|音|输|入|法|3.0|添|加|后|在|输|入|法|的|选|项|里|调|出|简|繁|体|的|按|纽|在|新|增|加|的|简|繁|体|切|换|按|纽|按|一|下|换|成|繁|体|即|可|r|推|荐|使|用|微|软|拼|音|输|入|法|2003|效|果|比|微|软|拼|音|输|入|法|3.0|要|好|放|出|地|址|r|下|载|后|直|接|安|装|就|可|以|了|会|自|动|覆|盖|原|来|的|微|软|拼|音|输|入|法|3.0|r|充|值|月|卡|转|为|正|式|帐|号|r|直|接|冲|值|月|卡|即|可|不|需|要|cdk|重|大|更|新|r|_id|2|gopage|1|lis|r|rtwow|最|大|的|优|势|就|是|月|卡|r|插|件|方|面|目|前|大|部|分|的|简|体|插|件|是|能|使|用|在|twow|的|您|下|载|的|魔|兽|精|灵|依|然|可|以|在|twow|使|用|r|最|新|补|充|r1|台|湾|没|有|距|阵|卡|注|意|帐|号|安|全|r2|注|意|保|存|密|码|共|有|3|种|分|别|为|游|戏|密|码|官|网|论|坛|密|码|安|全|码|r|本|人|好|久|没|玩|安|全|码|遗|忘|中|找|客|服|已|经|解|决|r3|台|湾|每|周|四|早|上|4|点|例|行|维|护|一|般|时|间|为|6|小|时|大|部|分|情|况|是|提|前|开|机|",
  "qid|qid_5012882145821532503|category|娱|乐|明|星|title|李|克|勤|什|么|歌|好|听|desc|answer|红|日|",
  "qid|qid_7521787209011105347|category|健|康|男|科|前|列|腺|科|title|血|压|140|105|我|18|岁|征|兵|体|检|时|发|现|有|高|血|压|30|岁|时|偶|尔|会|desc|我|18|岁|征|兵|时|发|现|有|高|血|压|30|岁|时|偶|尔|会|头|晕|测|量|血|压|为|135|90|持|续|服|用|过|卡|托|普|利|效|果|不|明|显|近|半|年|来|我|的|头|晕|加|重|脖|子|转|动|时|感|觉|僵|硬|筋|疼|服|用|吲|达|帕|胺|片|现|在|的|血|压|是|140|105|我|查|了|一|下|资|料|我|就|是|舒|张|压|太|高|了|是|什|么|原|因|呢|应|该|服|用|什|么|药|才|好|呢|answer|我|18|岁|征|兵|体|检|时|发|现|有|高|血|压|30|岁|时|偶|尔|会|头|晕|测|量|血|压|为|135|90|持|续|服|用|过|卡|托|普|利|效|果|不|明|显|近|半|年|来|我|的|头|晕|加|重|脖|子|转|动|时|感|觉|僵|硬|筋|疼|服|用|吲|达|帕|胺|片|现|在|的|血|压|是|140|105|我|查|了|一|下|资|料|我|就|是|舒|张|压|太|高|了|是|什|么|原|因|呢|应|该|服|用|什|么|药|才|好|呢|r|n|大|概|是|你|当|上|官|的|原|因|现|在|当|官|的|不|是|三|高|就|是|四|高|简|单|的|说|是|不|良|的|生|活|方|式|和|饮|食|习|惯|造|成|的|r|n|应|该|服|用|什|么|药|最|好|找|中|医|用|中|药|调|理|原|因|简|单|如|下|r|n|r|n|不|仅|卡|托|普|利|会|引|起|药|源|性|震|颤|麻|痹|不|少|药|物|若|使|用|失|当|也|会|诱|发|此|病|其|实|类|似|的|现|象|并|非|罕|见|常|见|的|药|物|有|降|血|压|药|r|n|r|n|利|血|平|是|从|印|度|萝|芙|木|提|取|的|生|物|碱|降|压|灵|是|中|国|萝|芙|木|所|含|的|生|物|总|碱|降|压|作|用|较|利|血|平|弱|两|者|的|结|构|相|似|其|药|理|作|用|使|交|感|神|经|末|梢|囊|泡|内|递|质|去|甲|肾|上|腺|素|和|多|巴|胺|耗|竭|因|此|长|期|应|用|易|诱|发|药|源|性|震|颤|麻|痹|征|和|抑|郁|症|r|n|",
  "qid|qid_1336718430447702776|category|生|活|育|儿|title|有|人|知|道|宝|宝|补|锌|产|品|排|行|榜|么|知|道|的|可|以|说|说|建|议|上|面|什|么|产|品|比|desc|有|人|知|道|补|锌|产|品|排|行|榜|么|知|道|的|可|以|说|说|建|议|上|面|什|么|产|品|比|较|好|answer|宝|宝|补|锌|产|品|排|行|榜|冒|失|上|次|看|到|过|记|得|有|过|一|个|叫|新|稀|宝|牌|子|的|补|锌|产|品|是|第|三|代|蛋|白|锌|产|品|补|锌|效|果|是|最|好|的|你|可|以|百|度|百|科|新|稀|宝|看|看|具|体|资|料|r|n|",
  "qid|qid_6790807861626896337|category|生|活|服|装|首|饰|title|春|装|秋|装|区|别|我|想|知|道|春|装|和|秋|装|的|区|别|要|详|细|地|说|desc|我|想|知|道|和|秋|装|的|区|别|要|详|细|地|说|answer|最|大|的|区|别|还|是|在|质|地|上|据|了|解|夏|装|普|遍|以|棉|丝|等|轻|薄|材|料|为|主|而|观|察|新|上|市|的|秋|装|则|以|针|织|衫|厚|棉|材|料|为|主|无|论|是|上|装|还|是|裤|子|普|遍|都|比|夏|装|增|加|了|不|少|厚|度|颜|色|也|开|始|偏|向|秋|天|色|彩|如|咖|啡|棕|黑|等|即|使|是|一|些|天|性|比|较|亮|丽|的|颜|色|如|绿|紫|红|等|也|从|原|先|的|淡|色|系|转|向|了|深|色|系|而|在|天|气|仍|有|些|微|热|的|夏|末|这|种|深|色|系|新|装|并|不|为|消|费|者|买|账|商|家|表|示|春|装|的|艳|丽|更|接|近|这|几|天|的|消|费|心|理|",
  "qid|qid_4781331459651416283|category|生|活|生|活|常|识|title|全|国|的|邮|政|小|包|价|格|一|样|吗|desc|全|国|的|邮|政|小|包|价|格|一|样|吗|answer|中|国|邮|政|小|包|所|有|货|物|按|克|收|费|分|为|9|个|区|每|个|区|代|表|不|同|的|国|家|价|格|也|有|所|不|同|首|重|50|克|以|内|都|是|5|元|钱|然|后|在|对|应|给|出|折|扣|单|件|货|物|限|制|在|2kg|以|内|中|国|邮|政|小|包|所|有|货|物|按|克|收|费|分|为|9|个|区|每|个|区|代|表|不|同|的|国|家|价|格|也|有|所|不|同|首|重|50|克|以|内|都|是|5|元|钱|然|后|在|对|应|给|出|折|扣|单|件|货|物|限|制|在|2kg|以|内|深|圳|互|联|易|中|国|邮|政|小|包|价|格|很|不|错|的|深|圳|地|区|都|可|上|门|提|货|的|类|似|于|挂|号|信|的|小|包|都|是|4.5|吧|好|久|没|有|寄|了|不|一|样|哦|分|区|域|的|不|过|差|别|倒|是|都|不|大|您|若|感|觉|满|意|请|点|好|评|支|持|我|在|微|问|的|道|路|上|帮|助|更|多|有|需|要|的|人|不|一|样|哦|起|重|资|费|起|重|500|克|及|以|内|20|元|续|重|资|费|续|重|每|500|克|或|其|零|数|一|区|6|元|二|区|9|元|三|区|15|元|具|体|分|区|方|式|请|寄|件|人|拨|打|电|话|或|到|当|地|邮|局|营|业|窗|口|咨|询|客|服|电|话|11185|这|样|比|较|容|易|睡|着|如|果|满|意|我|的|回|答|可|点|击|对|我|有|用|谢|谢|",
  "qid|qid_6299572732647927935|category|教|育|科|学|理|工|学|科|生|物|学|title|我|在|山|西|冬|天|榕|树|怎|样|养|植|榕|树|一|直|掉|叶|现|在|没|剩|几|片|了|desc|一|直|掉|叶|现|在|没|剩|几|片|了|answer|因|为|榕|树|喜|阳|应|该|把|它|放|置|到|有|一|定|阳|光|的|地|方|还|有|冬|天|气|温|如|果|低|于|五|度|叶|片|容|易|冻|伤|应|该|把|它|移|到|室|内|有|的|树|桩|盆|景|还|有|耐|寒|或|非|耐|寒|性|对|非|耐|寒|性|的|树|桩|一|般|冬|天|还|要|进|入|温|室|维|护|管|理|如|榕|树|福|建|茶|等|冬|天|浇|水|时|注|意|不|要|湿|了|叶|子|和|花|朵|榕|树|适|应|性|强|在|酸|性|中|性|微|碱|性|的|土|壤|都|能|生|长|但|最|好|是|在|微|酸|疏|松|排|水|良|好|又|有|一|定|肥|力|的|土|壤|中|养|植|栽|种|一|般|在|萌|芽|前|的|3|4|月|份|进|行|r|n|r|n|榕|树|盆|景|多|用|紫|砂|盆|也|有|用|釉|陶|盆|的|盆|的|深|度|和|样|式|根|据|造|型|而|定|悬|崖|式|多|用|签|筒|盆|如|把|盆|景|置|于|高|脚|几|架|上|也|可|用|普|通|方|盆|或|圆|盆|曲|干|式|斜|干|式|提|根|式|树|石|式|多|用|中|等|深|度|的|长|方|形|或|椭|圆|形|盆|钵|r|n|r|n|荫|棚|内|养|护|秋|末|初|冬|当|夜|间|最|低|温|度|降|至|6|一|7|时|应|移|入|室|内|向|阳|处|越|冬|冬|季|室|温|以|不|低|于|5|为|宜|棕|竹|喜|温|暖|湿|润|通|风|并|有|定|光|照|的|环|境|如|烈|日|曝|晒|盆|内|干|燥|常|造|成|枝|叶|枯|黄|轻|者|生|长|不|良|严|重|者|可|能|枯|死|r|n|",
  "qid|qid_5121647188856896111|category|教|育|科|学|升|学|入|学|考|研|title|怎|样|才|能|够|进|入|外|交|部|工|作|呢|有|什|么|要|求|吗|必|须|是|相|关|的|的|专|业|才|可|desc|怎|样|才|能|够|进|入|部|工|作|呢|有|什|么|要|求|吗|必|须|是|相|关|的|的|专|业|才|可|以|吗|需|要|考|公|务|员|吗|熟|练|掌|握|一|门|小|语|种|是|不|是|更|有|利|呢|answer|1|关|于|怎|样|才|能|够|进|入|外|交|部|工|作|呢|的|回|答|高|中|毕|业|后|考|入|国|际|关|系|学|院|r|r2|关|于|有|什|么|要|求|吗|的|回|答|r|身|体|条|件|除|须|达|到|部|和|卫|生|部|联|合|颁|发|的|普|通|高|等|学|校|招|生|体|检|工|作|指|导|意|见|的|有|关|规|定|外|还|需|符|合|下|列|条|件|r|r|1|男|生|身|高|170cm|以|上|体|重|50kg|以|上|女|生|身|高|160cm|以|上|体|重|45kg|以|上|体|型|匀|称|r|r|2|左|右|眼|裸|眼|视|力|4.3|以|上|报|考|英|语|专|业|信|息|管|理|与|信|息|系|统|专|业|和|信|息|应|用|专|业|的|考|生|左|右|眼|裸|眼|视|力|4.8|以|上|无|色|盲|色|弱|斜|视|对|眼|等|眼|疾|r|r|3|听|觉|嗅|觉|正|常|无|影|响|外|语|学|习|的|听|力|系|统|和|发|音|系|统|疾|病|无|口|吃|r|r|4|心|脏|血|液|肝|功|能|正|常|无|甲|肝|乙|肝|和|澳|抗|阳|性|无|严|重|急|慢|性|疾|病|无|传|染|病|和|精|神|疾|患|等|r|r|5|五|官|端|正|面|部|及|身|体|暴|露|部|位|无|明|显|特|征|和|缺|陷|无|鸡|胸|无|腋|臭|无|严|重|静|脉|曲|张|无|明|显|八|字|步|罗|圈|腿|无|平|足|跛|足|无|斜|肩|纹|身|少|白|头|驼|背|无|各|种|残|疾|本|人|及|直|系|血|亲|无|精|神|病|史|r|以|上|只|是|客|观|条|件|的|要|求|另|外|最|重|要|的|就|是|成|绩|r|r3|关|于|必|须|是|相|关|的|的|专|业|才|可|以|吗|的|回|答|相|关|的|专|业|很|多|只|要|你|得|本|专|业|优|秀|就|可|以|有|机|会|r|r4|这|个|行|业|不|同|公|务|员|如|果|你|直|接|考|入|外|交|部|那|就|没|必|要|再|考|公|务|员|了|r|r5|任|何|行|业|熟|练|掌|握|一|门|小|语|种|都|是|更|有|利|的|",
  "qid|qid_6432791236741067118|category|商|业|理|财|基|金|title|广|发|货|币|基|金|转|聚|丰|需|要|几|天|desc|answer|在|交|易|日|15|00|前|转|换|按|当|天|基|金|净|值|计|算|份|额|下|一|交|易|日|确|认|",
  "qid|qid_7204323359827944758|category|生|活|美|容|塑|身|祛|斑|祛|痘|title|上|海|简|单|有|效|的|祛|斑|方|法|祛|斑|需|要|多|少|钱|desc|上|海|简|单|有|效|的|方|法|祛|斑|需|要|多|少|钱|answer|祛|除|雀|斑|一|定|要|采|用|专|业|的|治|疗|方|法|千|万|不|能|盲|目|的|相|信|偏|方|治|疗|以|免|耽|误|自|己|的|病|情|目|前|治|疗|雀|斑|的|方|法|有|很|多|大|多|数|患|者|会|选|择|激|光|治|疗|因|为|激|光|治|疗|可|以|准|确|的|定|位|将|皮|肤|内|的|色|素|沉|积|击|碎|有|选|择|性|的|吸|收|治|疗|的|更|加|彻|底|而|且|见|效|快|安|全|不|留|疤|痕|r|n|激|光|祛|除|色|斑|操|作|简|单|效|果|明|显|过|程|时|间|短|最|多|不|过|十|几|分|钟|每|次|治|疗|结|束|以|后|皮|肤|都|要|有|一|个|自|然|吸|收|的|新|陈|代|谢|过|程|因|此|治|疗|的|间|隔|时|间|一|般|为|2|到|3|个|月|根|据|色|素|及|血|管|病|变|的|大|小|数|量|及|深|度|有|的|治|疗|一|两|次|就|能|达|到|预|期|的|效|果|一|般|则|需|4|到|6|次|才|能|彻|底|解|决|问|题|激|光|祛|除|色|斑|效|果|是|一|般|的|祛|除|色|斑|产|品|无|法|比|拟|的|r|n|激|光|祛|斑|是|利|用|激|光|的|光|热|解|原|理|激|光|照|射|皮|肤|后|会|被|皮|肤|病|变|部|位|的|色|素|吸|收|而|激|光|产|生|的|高|能|量|就|会|将|这|些|色|素|击|碎|色|素|碎|屑|会|随|人|体|的|各|项|循|环|排|除|体|外|从|而|达|到|祛|斑|的|效|果|激|光|祛|斑|对|颜|色|较|深|的|色|素|斑|辐|射|斑|等|效|果|是|最|为|显|著|的|激|光|治|疗|能|达|到|比|较|理|想|的|效|果|是|由|激|光|治|疗|的|原|理|决|定|的|激|光|治|疗|的|原|理|是|激|光|仪|产|生|一|束|高|强|的|光|束|不|同|类|型|的|激|光|可|以|产|生|不|同|颜|色|的|光|线|激|光|的|禸|眼|色|是|治|疗|的|关|键|不|同|颜|色|的|激|光|被|不|同|颜|色|的|皮|肤|吸|收|并|利|用|激|光|能|量|令|染|料|颗|粒|崩|解|汽|化|封|闭|血|管|再|由|身|体|吸|收|染|料|颗|粒|将|其|排|出|体|外|色|素|随|之|消|退|r|n|",
  "qid|qid_3109046540611139702|category|游|戏|久|游|游|戏|魔|力|宝|贝|title|怀|旧|我|想|打|阿|鲁|巴|斯|要|拿|护|士|帽|吗|怎|么|个|流|程|话|说|我|玩|这|么|久|没|直|desc|怎|么|个|流|程|话|说|我|玩|这|么|久|没|直|接|干|过|2684|answer|没|有|护|士|帽|也|可|以|直|接|打|阿|鲁|巴|斯|r|n|请|注|意|看|下|面|的|图|片|",
  "qid|qid_3677532485006819107|category|电|子|数|码|照|相|机|摄|像|机|title|为|什|么|低|端|的|数|码|相|机|采|用|cmos|传|感|器|高|级|一|点|采|用|ccd|而|更|desc|answer|1|目|前|人|们|使|用|的|数|码|相|机|数|码|拍|照|手|机|和|摄|像|头|的|感|光|器|材|主|要|是|分|为|ccd|和|coms|两|类|相|比|较|而|言|ccd|的|优|势|在|于|成|像|质|量|好|但|是|由|于|制|造|工|艺|复|杂|只|有|少|数|的|厂|商|能|够|掌|握|价|格|较|高|在|相|同|分|辨|率|下|cmos|价|格|比|ccd|便|宜|但|是|cmos|器|件|产|生|的|图|像|质|量|相|比|ccd|来|说|要|低|一|些|到|目|前|为|止|市|面|上|绝|大|多|数|的|消|费|级|别|以|及|高|端|数|码|相|机|都|使|用|ccd|作|为|感|应|器|cmos|感|应|器|则|作|为|低|端|产|品|只|应|用|于|一|些|摄|像|头|上|r|2|然|而|cmos|影|像|传|感|器|也|有|自|身|的|优|点|其|电|源|消|耗|量|比|ccd|低|ccd|为|提|供|优|异|的|影|像|品|质|付|出|代|价|即|是|较|高|的|电|源|消|耗|量|为|使|电|荷|传|输|顺|畅|噪|声|降|低|需|由|高|压|差|改|善|传|输|效|果|但|cmos|影|像|传|感|器|将|每|一|画|素|的|电|荷|转|换|成|电|压|读|取|前|便|将|其|放|大|电|源|消|耗|量|比|ccd|低|此|外|由|于|cmos|整|合|性|高|使|体|积|大|幅|缩|小|这|点|是|ccd|所|不|能|及|的|目|前|cmos|影|像|传|感|器|首|要|解|决|的|问|题|就|是|降|低|噪|声|的|产|生|所|以|有|的|比|较|高|级|影|象|生|产|企|业|在|解|决|了|cmos|降|燥|问|题|后|反|而|使|用|cmos|r|3|未|来|cmos|影|像|传|感|器|是|否|可|以|改|变|长|久|以|来|被|ccd|压|抑|的|宿|命|往|后|技|术|的|发|展|是|重|要|关|键|",
  "qid|qid_4173358817197735817|category|游|戏|网|络|游|戏|title|攻|击|别|国|占|领|陆|地|的|结|果|是|如|何|计|算|的|攻|击|别|国|占|领|陆|地|的|结|果|是|如|何|desc|攻|击|别|国|占|领|陆|地|的|结|果|是|如|何|计|算|的|r|n|请|教|高|手|有|公|式|计|算|的|么|是|如|何|计|算|的|answer|楼|上|说|的|不|正|确|要|这|么|简|单|楼|主|就|不|问|了|r|n|r|n|好|像|和|攻|打|国|和|被|攻|打|国|的|实|力|有|关|还|有|其|他|因|素|",
  "qid|qid_6594100490612694209|category|电|脑|网|络|操|作|系|统|系|统|故|障|title|xp|关|机|时|出|现|结|束|sample|程|序|对|话|框|怎|么|回|事|我|的|wi|desc|我|的|xp|遇|到|关|机|问|题|当|我|试|图|关|闭|系|统|时|出|现|正|在|结|束|框|系|统|在|试|图|定|位|一|个|叫|sample|的|程|序|但|是|我|找|不|到|这|个|在|我|的|系|统|中|运|行|的|程|序|我|也|无|法|关|闭|系|统|怎|样|对|付|这|个|问|题|呢|answer|你|很|可|能|中|了|蠕|虫|之|类|了|病|毒|了|先|用|杀|毒|查|查|吧|r|这|是|2001|年|的|消|息|参|考|网|页|r|如|果|真|是|此|类|病|毒|可|以|根|据|下|面|内|容|删|除|相|关|文|件|或|找|新|版|瑞|星|等|杀|毒|软|件|查|杀|r|娜|妲|病|毒|出|现|变|种|r|台|北|讯|今|年|9|月|肆|虐|网|络|的|娜|妲|病|毒|nimda|出|现|了|最|新|的|变|种|病|毒|r|r|趋|势|科|技|表|示|娜|妲|病|毒|及|其|变|种|病|毒|感|染|力|极|强|是|非|常|凶|猛|的|病|毒|皆|会|透|过|电|子|邮|件|资|源|分|享|微|软|iis|及|web|浏|览|等|四|种|方|式|来|传|播|目|前|美|国|韩|国|及|菲|律|宾|分|别|传|出|感|染|灾|情|台|湾|证|券|交|易|所|和|期|货|交|易|所|的|电|脑|已|先|后|中|毒|r|r|pe|nimda.e|夹|带|的|附|加|档|案|为|sample.exe|并|会|在|windows|资|料|夹|置|入|load.exe|或|csrss.exe|r|r|趋|势|科|技|提|醒|电|脑|族|别|开|启|附|件|档|名|为|sample.exe|的|电|子|邮|件|或|采|用|内|建|个|人|防|火|墙|功|能|的|防|毒|软|件|以|彻|底|封|锁|恶|意|电|脑|病|毒|及|网|络|蠕|虫|有|效|保|护|电|脑|r|r|趋|势|科|技|技|术|支|援|部|经|理|康|宏|旭|表|示|新|变|种|病|毒|pe|nimda.e|具|有|电|脑|蠕|虫|worm|的|特|性|会|利|用|数|种|感|染|途|径|来|进|行|繁|殖|及|散|播|比|如|e|mail|资|源|分|享|微|软|iisserver|及|web|浏|览|r|r|透|过|电|子|邮|件|传|播|时|病|毒|会|夹|带|sample.exe|档|案|一|旦|档|案|被|执|行|时|它|会|在|c|磁|碟|的|windows|temp|资|料|夹|下|建|立|p.exe|暂|存|档|这|个|档|案|是|e|mail|格|式|并|且|包|含|蠕|虫|所|夹|带|的|病|毒|档|案|r|r|根|据|资|料|统|计|娜|妲|病|毒|在|全|球|造|成|5|亿|9000|万|美|元|的|损|失|r|r|台|湾|证|券|交|易|所|股|市|观|测|站|系|统|在|前|天|5|时|57|分|时|遭|娜|妲|病|毒|入|侵|使|得|股|市|观|测|站|中|的|重|大|讯|息|内|容|及|上|市|公|司|财|报|申|报|投|资|人|均|无|法|上|网|查|询|证|交|所|昨|天|上|午|表|示|股|市|观|测|站|现|已|恢|复|查|询|功|能|r|r|证|交|所|表|示|证|交|所|经|检|查|后|发|现|股|市|观|测|站|遭|病|毒|入|侵|原|因|是|经|由|内|部|电|子|邮|件|发|送|所|致|由|于|股|市|观|测|站|为|证|交|所|内|部|网|络|系|统|在|发|现|被|病|毒|入|侵|证|交|所|立|即|关|闭|内|部|网|络|进|行|扫|毒|r|",
  "qid|qid_750050624123334282|category|游|戏|光|宇|游|戏|问|道|title|天|府|情|缘|出|问|题|了|晚|上|天|府|都|进|不|去|了|给|个|说|法|哦|我|还|在|杀|怪|哈|死|了|掉|desc|晚|上|天|府|都|进|不|去|了|给|个|说|法|哦|我|还|在|杀|怪|哈|死|了|掉|道|了|怎|么|办|那|个|赔|gm|给|说|法|哦|answer|你|好|r|n|服|务|器|如|果|出|了|问|题|需|要|维|护|r|n|请|耐|心|稍|等|等|服|务|器|恢|复|",
  "qid|qid_2419920079104386978|category|娱|乐|博|彩|title|任|九|让|我|优|让|我|喜|一|次|次|买|彩|一|次|次|倾|听|那|比|分|一|次|desc|一|次|次|买|彩|一|次|次|倾|听|那|比|分|一|次|次|兴|奋|一|次|次|失|落|一|次|次|无|归|而|返|最|终|还|是|任|8|answer|咱|们|都|是|一|样|的|心|情|希|望|你|以|后|有|好|的|运|气|保|持|好|的|心|态|",
  "qid|qid_1615445641236654883|category|电|脑|网|络|互|联|网|title|汉|城|什|么|时|候|改|名|为|首|尔|为|什|么|desc|汉|城|什|么|时|候|为|首|尔|为|什|么|answer|首|尔|서울|正|式|名|称|为|首|尔|特|别|市|서울|특별시|是|韩|国|的|首|都|旧|称|汉|城|汉|阳|等|首|尔|是|韩|语|서울|的|汉|语|译|音|名|称|2005|年|1|月|18|日|该|市|市|议|会|正|式|通|过|把|其|市|的|中|文|名|称|定|为|首|尔|并|把|官|方|的|所|有|出|版|物|改|用|新|名|称|但|部|份|华|语|地|区|还|使|用|汉|城|r|r|首|尔|서울|旧|称|南|京|남경|汉|阳|한양|汉|城|한성|京|城|경성|自|从|1945|年|二|战|结|束|韩|国|独|立|以|后|当|地|民|族|情|绪|高|涨|并|将|其|首|都|改|称|为|서울|英|文|为|seoul|韩|语|是|首|都|或|京|城|的|意|思|就|如|中|国|的|首|都|称|为|京|一|样|因|此|seoul|成|为|当|时|韩|国|国|内|唯|一|没|有|对|应|汉|字|名|称|的|地|名|中|国|大|陆|早|期|曾|经|根|据|韩|语|发|音|而|音|译|作|苏|乌|但|其|后|跟|随|其|他|华|语|地|区|沿|用|这|个|城|市|在|李|氏|朝|鲜|时|的|旧|称|汉|城|然|而|其|他|语|言|都|翻|译|成|seoul|或|类|似|的|发|音|r|r|这|使|日|常|沟|通|出|现|不|少|混|乱|因|为|首|尔|市|内|有|不|少|建|筑|物|分|别|使|用|首|尔|或|汉|城|为|名|称|但|翻|译|成|中|文|之|后|这|些|地|方|名|称|会|变|得|很|相|似|一|个|例|子|是|首|尔|市|内|同|时|有|汉|城|大|学|及|首|尔|大|学|假|若|采|用|汉|城|作|为|seoul|的|首|都|名|称|会|使|两|家|大|学|的|译|名|都|变|成|了|汉|城|大|学|后|来|中|国|人|根|据|后|者|的|英|文|名|称|在|其|中|文|名|称|加|上|国|立|二|字|作|分|辨|但|其|实|首|尔|大|学|在|韩|国|的|知|名|度|比|汉|城|大|学|高|很|多|而|大|学|本|身|的|中|文|出|版|物|亦|自|称|为|seoul|大|学|校|但|对|於|学|校|以|外|的|机|构|译|名|的|问|题|就|给|他|们|造|成|困|扰|2004|年|韩|国|曾|经|有|人|发|起|运|动|要|求|解|散|首|尔|大|学|韩|国|多|份|主|要|报|章|由|於|不|清|楚|中|国|对|两|家|汉|城|大|学|的|区|分|方|式|而|多|次|在|他|们|的|中|文|版|新|闻|中|把|首|尔|大|学|错|译|为|汉|城|大|学|而|其|企|划|部|的|柳|根|培|亦|无|缘|无|故|被|转|校|成|为|汉|城|大|学|的|发|言|人|r|r|韩|国|政|府|从|1992|年|起|就|一|直|在|进|行|서울|的|中|文|名|称|的|制|订|工|作|拟|订|的|对|应|中|文|包|括|首|尔|和|首|午|尔|2005|年|1|月|18|日|서울|市|议|会|通|过|使|用|与|seoul|发|音|相|近|的|首|尔|作|为|这|个|城|市|的|中|文|名|称|并|要|求|韩|国|政|府|公|文|出|版|物|网|站|机|场|和|车|站|公|路|路|标|的|中|文|版|本|都|使|用|首|尔|来|代|替|汉|城|但|是|迄|今|为|止|在|中|文|裏|使|用|最|多|的|还|是|具|有|500|余|年|历|史|的|汉|城|这|个|名|称|香|港|及|台|湾|的|传|媒|大|部|份|已|经|转|用|首|尔|这|个|新|的|名|称|中|国|大|陆|官|方|正|式|宣|布|了|改|名|这|个|消|息|但|并|不|常|使|用|首|尔|这|个|新|名|一|些|民|办|传|媒|及|报|刊|先|开|始|采|用|首|尔|一|些|官|方|媒|体|如|新|华|网|也|已|经|开|始|使|用|还|有|一|些|处|在|过|渡|期|如|中|国|中|央|电|视|台|写|成|汉|城|首|尔|不|过|中|国|大|陆|绝|大|部|份|出|版|物|媒|体|政|府|及|商|业|机|关|仍|延|续|旧|称|r|r|有|不|少|中|国|人|质|疑|市|议|会|是|否|有|权|更|改|本|国|首|都|的|汉|语|名|称|并|指|如|果|韩|国|首|都|的|中|文|译|名|改|变|将|使|华|人|世|界|对|于|韩|国|首|都|的|称|呼|造|成|混|乱|还|有|一|个|重|要|原|因|是|首|尔|主|要|是|根|据|现|时|汉|语|普|通|话|的|译|音|但|汉|字|是|在|汉|语|各|方|言|间|以|及|韩|日|越|南|语|的|一|些|时|候|使|用|的|如|果|音|译|的|话|会|造|成|很|多|使|用|汉|字|的|地|区|对|首|尔|两|字|读|音|发|生|混|乱|如|粤|语|读|为|sau2|yi5|和|韩|语|读|音|差|别|很|大|而|首|尔|两|字|按|韩|语|中|汉|字|的|读|音|也|成|了|수이|su|i|随|著|语|音|的|发|展|若|干|年|后|即|使|普|通|话|首|尔|两|字|的|发|音|和|韩|语|也|可|能|对|应|不|上|和|众|多|西|方|音|译|词|在|各|处|的|差|别|一|样|可|能|会|造|成|汉|字|使|用|者|的|困|扰|有|人|提|出|如|果|根|据|韩|语|서울|采|用|汉|字|西|尉|韩|语|读|作|서울|即|seoul|则|不|会|有|此|问|题|可|以|在|使|用|汉|字|的|地|区|和|时|间|上|保|持|一|致|r|r|然|而|韩|国|方|面|一|直|对|中|国|大|陆|的|这|些|想|法|非|常|不|解|一|来|这|城|市|是|他|们|的|城|市|二|来|他|们|亦|已|多|次|透|过|各|种|沟|道|来|解|释|这|次|改|变|的|因|由|r|首|尔|是|韩|国|首|都|汉|城|根|据|其|英|文|seoul|的|音|译|所|改|的|新|的|中|文|名|称|r|汉|城|市|长|李|明|博|2005|年|1|月|19|日|举|行|记|者|招|待|会|宣|布|将|首|都|的|中|文|名|称|由|汉|城|改|为|首|尔|汉|城|一|词|不|再|使|用|李|明|博|市|长|的|解|说|词|是|绝|大|多|数|国|家|都|将|seoul|按|照|与|其|英|文|标|记|相|似|的|发|音|来|称|呼|如|汉|语|中|的|华|盛|顿|伦|敦|也|都|是|根|据|这|些|地|名|的|固|有|发|音|来|标|记|的|只|有|汉|城|的|中|文|名|称|一|直|沿|用|古|代|名|称|汉|城|李|明|博|市|长|向|中|国|游|说|首|尔|同|汉|城|的|韩|语|发|音|最|为|接|近|并|采|用|多|用|于|外|国|地|名|的|常|用|汉|字|中|国|人|也|容|易|熟|悉|这|一|新|名|称|r|很|明|显|李|市|长|是|从|语|音|学|角|度|切|入|问|题|的|但|是|如|果|就|事|论|事|的|话|李|明|博|等|韩|国|官|方|人|员|的|解|释|比|较|牵|强|因|为|即|使|改|换|了|汉|城|汉|语|的|名|称|为|首|尔|也|存|在|着|翻|译|上|及|使|用|习|惯|上|的|混|乱|问|题|况|且|汉|语|中|的|外|国|地|名|也|不|都|是|以|发|音|为|根|据|翻|译|的|如|英|国|的|牛|津|剑|桥|等|并|非|完|全|是|音|译|美|国|的|盐|湖|城|ssltlakecity|阿|肯|色|州|的|小|石|城|littlerock|等|完|全|是|意|译|新|西|兰|首|都|wellington|的|官|方|中|译|名|几|乎|就|是|错|译|威|灵|顿|显|然|比|惠|灵|顿|更|贴|切|但|新|西|兰|政|府|似|乎|从|未|提|出|任|何|异|议|中|文|名|称|叫|什|么|应|当|尊|重|中|国|的|传|统|使|用|习|惯|r|据|考|证|汉|城|的|称|呼|沿|用|了|韩|国|古|代|历|史|王|朝|的|用|法|1394|年|李|成|桂|将|都|城|从|开|京|迁|移|到|了|汉|阳|正|式|命|名|为|汉|城|hansung|这|一|名|称|在|汉|语|中|至|今|已|经|使|用|了|610|年|二|战|结|束|以|后|韩|国|将|汉|城|称|为|韩|国|语|的|首|都|sieur|英|文|音|译|为|seoul|但|是|韩|国|的|书|面|汉|字|仍|然|写|作|汉|城|因|此|可|以|说|韩|国|这|次|改|换|首|都|的|汉|语|名|称|绝|非|像|表|面|上|解|释|的|那|样|简|单|而|是|包|含|深|刻|的|原|因|和|其|他|方|面|复|杂|的|考|虑|r|随|着|19|世|纪|末|民|族|主|义|的|崛|起|韩|国|国|内|就|出|现|了|不|能|正|确|认|识|本|民|族|文|化|和|客|观|评|价|中|国|文|化|对|韩|国|民|族|文|化|影|响|而|摆|脱|汉|语|文|化|影|响|的|思|潮|韩|国|在|二|战|以|后|民|族|主|义|思|潮|进|一|步|发|展|曾|以|法|律|规|定|以|韩|国|的|表|音|字|为|专|用|文|字|从|1970|年|起|韩|国|小|学|中|学|教|科|书|中|的|汉|字|被|取|消|完|全|使|用|表|音|文|字|r|随|着|韩|国|经|济|的|崛|起|这|种|极|端|的|民|族|主|义|情|绪|进|一|步|发|展|在|1988|年|汉|城|奥|运|会|召|开|之|前|韩|国|政|府|曾|经|下|令|取|消|所|有|牌|匾|上|的|汉|字|标|记|以|强|调|韩|国|的|民|族|文|化|r|只|是|到|了|1999|年|2|月|金|大|中|总|统|才|下|令|部|分|解|除|对|汉|字|使|用|的|限|制|但|对|于|这|种|解|禁|措|施|韩|国|国|内|也|存|在|着|激|烈|的|反|对|势|力|他|们|担|心|这|种|措|施|将|导|致|汉|字|的|泛|滥|与|韩|国|文|字|的|消|亡|r|从|某|种|意|义|上|说|韩|国|改|变|汉|城|的|中|文|名|字|是|本|国|民|族|主|义|在|新|形|势|下|的|延|续|和|发|展|的|表|现|r|汉|武|帝|时|曾|在|朝|鲜|设|立|了|四|个|郡|汉|城|是|中|国|人|六|百|年|前|至|今|一|直|习|惯|称|谓|的|名|字|韩|国|过|去|也|一|直|用|汉|字|写|人|名|和|地|名|虽|然|汉|城|之|名|是|由|韩|国|古|代|的|先|人|所|起|但|现|代|的|韩|国|人|总|觉|得|不|是|本|国|的|名|称|总|觉|得|汉|城|与|中|国|有|瓜|葛|容|易|让|人|联|想|到|中|国|的|汉|朝|对|于|汉|朝|一|些|韩|国|人|又|缺|乏|正|确|的|历|史|观|认|为|是|对|朝|鲜|的|侵|略|正|式|在|这|种|不|良|心|理|情|结|的|指|导|下|韩|国|才|有|意|将|汉|城|的|中|文|译|名|更|改|为|首|尔|r|韩|国|官|方|为|这|次|改|名|的|解|释|仅|仅|是|表|面|的|这|是|以|国|际|惯|例|和|便|于|国|际|交|往|来|掩|盖|更|加|深|层|的|阴|暗|心|理|情|结|努|力|摆|脱|汉|字|文|化|对|韩|国|深|厚|影|响|的|一|种|尝|试|r|叫|了|610|年|的|汉|城|一|下|子|改|名|叫|着|真|让|人|感|觉|别|扭|就|好|比|纽|约|new|youk|突|然|让|叫|牛|月|克|你|能|习|惯|吗|我|是|习|惯|不|了|我|在|很|多|方|面|敬|佩|和|尊|重|韩|国|人|但|是|在|这|一|点|上|我|b4|韩|国|人|太|小|家|子|气|了|r|不|可|否|认|朝|鲜|民|族|在|历|史|上|深|受|日|本|侵|略|者|的|奴|役|之|苦|大|力|弘|扬|朝|鲜|本|民|族|的|文|化|加|强|自|身|民|族|文|化|的|优|势|地|位|努|力|摆|脱|外|来|文|化|影|响|等|措|施|有|可|以|理|解|的|一|面|不|应|该|随|意|扣|上|狭|隘|的|帽|子|r|但|是|另|一|方|面|韩|国|自|身|在|保|持|和|发|扬|本|民|族|文|化|的|同|时|对|外|来|文|化|特|别|是|博|大|精|深|的|中|国|文|化|也|应|该|采|取|扬|弃|的|态|度|不|应|该|不|分|好|坏|一|概|拒|绝|其|实|博|大|精|深|的|中|华|文|化|对|朝|鲜|民|族|文|化|的|产|生|发|展|都|起|到|了|巨|大|的|贡|献|作|用|r|在|具|体|对|待|这|次|改|名|的|问|题|上|韩|国|有|权|利|更|改|本|国|首|都|的|汉|语|译|名|但|是|中|国|以|及|其|他|汉|语|权|的|国|家|也|有|权|接|受|或|不|接|受|这|个|新|译|名|接|受|与|不|接|受|这|也|是|中|国|与|其|他|汉|语|国|家|应|有|的|权|利|韩|国|应|当|予|以|尊|重|因|为|对|中|国|等|这|些|国|家|来|说|不|仅|仅|是|一|个|译|法|上|的|问|题|而|涉|及|了|历|史|习|惯|经|济|费|用|等|多|方|面|的|问|题|r|",
  "qid|qid_9108333640282323746|category|电|脑|网|络|硬|件|title|ups|电|源|工|作|原|理|desc|answer|你|好|r|n|★|ups|电|源|就|是|不|间|断|备|用|电|源|其|主|要|功|能|为|当|ups|电|源|检|测|到|供|电|电|源|失|去|如|停|电|断|电|等|时|会|立|即|启|动|转|换|为|内|部|电|池|供|电|逆|变|状|态|为|设|备|提|供|应|急|工|作|较|高|档|次|的|ups|电|源|兼|做|稳|压|器|的|作|用|可|以|使|输|出|电|源|电|压|处|于|稳|定|状|态|提|高|供|电|电|压|质|量|",
  "qid|qid_1313046378804543985|category|电|脑|网|络|互|联|网|上|网|帮|助|title|我|在|上|注|册|在|中|文|香|港|界|面|就|可|以|用|用|户|名|登|录|成|功|在|英|文|界|面|就|说|desc|在|界|面|就|说|我|登|陆|失|败|why|answer|我|记|得|大|陆|的|ebay|与|海|外|是|没|有|数|据|连|接|的|而|且|大|陆|的|ebay|也|不|叫|ebay|而|是|叫|易|趣|网|址|是|译|音|就|是|易|趣|r|n|r|n|如|果|喜|欢|ebay|的|话|那|么|只|有|去|非|大|陆|的|ebay|了|否|则|就|去|淘|宝|吧|个|人|感|觉|大|陆|ebay|用|户|没|淘|宝|多|",
  "qid|qid_5709676122096504533|category|烦|恼|交|友|技|巧|title|11|月|17|_q|毖|稚|齷|了|u|desc|answer|11|月|17|_q|毖|稚|齷|了|",
  "qid|qid_4026459235138627255|category|商|业|理|财|股|票|title|站|在|2995|以|上|的|高|岗|我|全|副|武|装|满|仓|站|在|2900|以|上|的|高|岗|desc|我|全|副|武|装|r|满|仓|r|站|在|2900|以|上|的|高|岗|r|r|虽|然|今|年|是|暖|冬|r|可|我|现|在|真|的|感|觉|有|点|凉|r|心|也|惶|惶|r|人|也|慌|慌|r|r|我|用|力|探|起|头|r|频|频|的|向|山|下|张|望|r|那|战|友|们|的|影|子|哟|r|在|落|日|的|余|晖|中|r|一|下|流|的|东|西|砸|出|的|阴|线|r|越|拉|越|长|r|r|我|略|微|地|抬|头|r|眼|帘|中|便|是|3000|以|上|的|风|光|r|很|美|但|很|凄|凉|r|就|象|圣|地|西|藏|r|广|袤|无|垠|r|万|里|无|疆|r|在|3000|以|上|的|山|岗|r|昨|天|这|里|万|马|奔|腾|r|还|高|奏|着|凯|歌|r|但|想|不|到|啊|r|今|天|却|是|这|样|的|冷|清|与|凄|凉|r|我|那|可|怜|的|股|我|全|副|武|装|r|满|仓|r|站|在|2900|以|上|的|高|岗|r|r|虽|然|今|年|是|暖|冬|r|可|我|现|在|真|的|感|觉|有|点|凉|r|心|也|惶|惶|r|人|也|慌|慌|r|r|我|用|力|探|起|头|r|频|频|的|向|山|下|张|望|r|那|战|友|们|的|影|子|哟|r|在|落|日|的|余|晖|中|r|一|下|流|的|东|西|砸|出|的|阴|线|r|越|拉|越|长|r|r|我|略|微|地|抬|头|r|眼|帘|中|便|是|3000|以|上|的|风|光|r|很|美|但|很|凄|凉|r|就|象|圣|地|西|藏|r|广|袤|无|垠|r|万|里|无|疆|r|在|3000|以|上|的|山|岗|r|昨|天|这|里|万|马|奔|腾|r|还|高|奏|着|凯|歌|r|但|想|不|到|啊|r|今|天|却|是|这|样|的|冷|清|与|凄|凉|r|我|那|可|怜|的|试|问|用|那|漆|黑|的|嘴|唇|r|怎|可|能|描|述|出|股|市|遍|地|是|金|子|r２９００|以|上|的|高|地|何|时|再|回|来|r２９００|的|阵|地|r|怎|么|就|上|不|来|一|个|救|兵|r|倒|是|有|只|来|自|天|空|的|飞|鹰|r|偶|尔|的|盘|旋|上|来|r|咕|咕|的|叫|着|r|不|要|难|过|r|不|要|难|过|r|看|那|万|亿|的|基|金|必|会|冲|上|来|夺|取|３０００|高|峰|r|r|我|感|动|得|热|泪|盈|眶|r|仿|佛|看|到|了|亲|人|r|可|是|山|下|突|然|的|枪|声|r|射|向|这|只|大|鹰|r|耳|边|响|起|r|谁|敢|染|指|股|票|我|是|老|大|r|坚|决|打|压|让|小|散|交|出|票|票|r|r|我|绝|望|着|看|看|天|r|不|知|道|把|谁|相|信|r|光|明|正|大|多|好|啊|r|苍|天|为|何|让|我|买|了|60|answer|你|看|到|了|万|点|的|前|程|有|眼|光|r|n|r|n|但|是|短|期|波|动|还|是|会|让|你|水|溅|衣|衫|湿|诗|人|",
  "qid|qid_4039207394580751973|category|教|育|科|学|学|习|帮|助|title|急|需|afew|和|desc|a|few|和|few|a|ltle|和|little|有|什|么|不|同|answer|a|few|和|few|都|是|修|饰|可|数|名|词|r|n|前|者|是|表|示|肯|定|有|一|些|的|意|思|后|者|表|示|否|定|表|示|几|乎|没|有|r|na|few|boys|few|boys|r|na|little|和|little|都|是|修|饰|不|可|数|名|词|r|n|前|者|是|表|示|肯|定|有|一|些|的|意|思|后|者|表|示|否|定|表|示|几|乎|没|有|r|n|r|nlittle|ink|r|n|a|little|ink|r|n|",
  "qid|qid_538211041202792726|category|游|戏|久|游|游|戏|魔|力|宝|贝|title|关|于|bl|我|们|队|伍|平|均|70|级|2|魔|1|传|1|弓|1|格|2|次|bl|都|过|不|了|desc|我|们|队|伍|平|均|70|级|2|魔|1|传|1|弓|1|格|2|次|都|过|不|了|请|问|大|家|一|般|打|多|少|次|能|过|啊|这|个|是|不|是|大|部|分|得|靠|运|气|帮|帮|小|弟|谢|谢|了|answer|打|佛|利|波|罗|十|大|要|点|r1.5|人|5|攻|宠|攻|宠|请|选|择|昆|虫|系|的|另|外|至|少|要|会|3|级|以|上|战|栗|在|只|有|2|个|影|子|的|时|候|开|始|吸|bl|魔|r2|物|理|系|的|人|尽|量|带|斧|请|记|得|必|杀|很|厉|害|另|外|防|御|很|重|要|最|好|上|240|r3|魔|防|很|重|要|重|装|请|选|择|本|人|能|装|备|的|最|高|等|级|的|抗|魔|装|r4|技|能|很|重|要|请|过|bl|aks|也|一|样|的|传|把|技|能|烧|烧|高|至|少|不|低|于|等|级|10|特|别|是|超|补|魔|术|也|一|样|r5|补|给|很|重|要|请|一|定|在|让|传|教|在|战|斗|前|补|满|魔|法|自|己|带|好|300|或|400|的|血|瓶|遇|到|f4|连|续|超|强|的|时|候|血|瓶|才|能|帮|你|顶|过|去|r6|属|性|很|重|要|建|议|佩|带|全|水|属|性|水|晶|选|有|水|属|性|的|攻|宠|至|少|保|证|200|以|上|敏|捷|以|免|被|闪|攻|击|顺|序|很|重|要|由|幻|全|火|属|性|用|冰|杀|暗|全|风|属|性|用|火|杀|阴|全|地|属|性|用|风|杀|旋|律|全|水|属|性|用|陨|杀|的|顺|序|杀|r|由|于|水|属|性|全|克|火|属|性|能|有|30|的|加|成|不|出|意|外|第|一|回|合|就|可|以|把|幻|影|秒|了|r|只|有|3|个|影|子|bl|的|情|况|下|还|是|很|容|易|过|的|r7|种|族|问|题|请|注|意|f4|影|子|是|特|殊|系|的|而|昆|虫|系|全|克|特|殊|系|而|转|昆|虫|系|的|是|什|么|道|具|呢|答|案|是|100|个|古|钱|到|法|兰|冒|险|者|旅|馆|3|楼|交|换|到|的|秋|分|之|竖|琴|变|成|昆|虫|系|以|后|可|以|减|少|影|子|对|人|伤|害|的|20|以|上|r8|网|络|问|题|请|要|做|大|任|务|的|人|尽|量|选|人|少|的|时|间|比|如|早|上|6|点|到|8|点|比|如|非|节|假|日|另|外|还|要|选|择|人|少|的|服|务|器|可|以|降|低|掉|线|几|率|r9|指|挥|问|题|请|事|先|决|定|战|斗|时|候|听|谁|的|指|挥|以|免|发|生|争|执|任|务|失|败|r10|最|后|是|最|重|要|的|一|点|运|气|r|运|气|不|好|第|一|回|合|就|乱|敏|f4|放|8|次|超|强|运|气|不|好|打|倒|f4|吸|干|影|子|也|能|掉|线|r|",
  "qid|qid_4093944749221724847|category|文|化|艺|术|民|俗|传|统|title|歇|后|语|打|破|脑|壳|不|叫|痛|desc|answer|打|破|脑|壳|不|叫|痛|死|称|好|汉|",
  "qid|qid_8382927358605856893|category|娱|乐|博|彩|title|大|家|发|现|了|一|个|问|题|了|吗|关|于|中|奖|江|西|的|彩|民|每|期|的|投|注|额|绝|desc|江|西|的|彩|民|r|每|期|的|投|注|额|绝|对|不|是|最|多|的|r|而|是|很|少|在|全|国|数|偏|低|r|r|但|每|期|的|中|奖|却|是|最|多|的|r|从|往|期|的|r|近|期|的|超|级|大|乐|透|r|r|第|一|期|中|了|198|注|二|等|奖|r|第|二|期|中|了|唯|一|的|一|等|奖|r|r|翻|翻|以|往|的|中|奖|纪|录|r|不|得|不|为|这|一|现|象|所|惊|叹|r|r|江|西|自|古|称|物|华|天|宝|人|杰|地|灵|r|所|言|不|虚|呀|r|r|answer|謝|謝|侵|如|此|稱|江|西|人|r|n|r|n|r|n|我|也|是|江|西|人|之|一|呵|呵|r|n|r|n|r|n|可|目|前|纹|谥|00|万|大|r|n|r|n|r|n|祝|好|運|連|連|r|n|r|n|r|n|r|n|r|n|",
  "qid|qid_6623662053203796023|category|游|戏|单|机|游|戏|title|使|命|召|唤|2|的|问|题|帮|帮|我|谢|谢|了|从|新|浪|下|载|的|使|命|召|唤|2|使|用|了|xez|desc|从|新|浪|下|载|的|2|使|用|了|xezl|gzwx|xqzg|gegu|eefc|序|了|号|是|不|是|有|问|题|单|机|游|戏|不|能|玩|了|只|能|玩|多|人|的|answer|用|虚|拟|光|驱|载|入|游|戏|盘|屏|蔽|物|理|光|驱|右|键|单|击|盘|符|属|性|硬|件|选|择|物|理|光|驱|属|性|不|使|用|设|备|就|可|以|了|第|一|次|安|装|完|要|重|启|",
  "qid|qid_3596130276328985892|category|游|戏|久|游|游|戏|魔|力|宝|贝|title|回|归|魔|力|问|练|功|点|回|归|魔|力|电|信|时|长|巨|蟹|的|请|问|112|的|敏|魔|要|desc|回|归|魔|力|r|时|长|巨|蟹|的|r|请|问|112|的|敏|魔|要|去|哪|练|r|还|有|请|教|现|在|怎|么|没|有|卖|料|理|的|r|自|己|做|什|么|好|锅|么|r|拿|来|卖|可|以|卖|多|少|钱|r|还|有|谁|的|群|能|q|3|r|找|不|到|朋|友|玩|魔|力|啊|现|在|人|好|少|answer|楼|上|的|两|位|说|的|都|有|点|道|理|r|n|不|过|r|n|说|起|来|经|验|还|是|gbl|排|第|一|不|过|需|要|带|大|量|料|理|r|n|还|有|炮|灰|宠|准|备|工|作|麻|烦|r|n|但|是|r|n|在|t95|一|般|不|用|带|料|理|一|般|魔|完|就|回|r|n|关|于|料|理|方|面|由|于|高|层|t|的|开|放|佣|兵|方|舟|要|塞|几|乎|没|人|r|n|导|致|料|理|销|量|的|急|剧|下|降|r|n|这|个|时|候|要|做|东|西|的|话|还|是|做|装|备|卖|的|好|不|推|荐|做|料|理|r|n|综|合|建|议|在|t95|先|练|着|料|理|不|带|r|n|等|待|t100|的|开|放|估|计|时|间|不|会|太|久|r|n|",
  "qid|qid_819575706465459004|category|社|会|民|生|法|律|title|有|限|公|司|可|以|不|设|股|东|会|把|董|事|会|作|为|权|力|机|构|吗|如|题|desc|如|题|answer|看|来|回|答|这|个|问|题|的|对|公|司|法|都|只|是|了|解|皮|毛|在|下|回|答|如|下|r|n|可|以|不|设|股|东|会|的|有|限|责|任|公|司|至|少|有|以|下|几|种|公|司|r|n|1|一|人|有|限|责|任|公|司|r|n2|中|外|合|资|企|业|r|n3|中|外|合|作|企|业|r|n4|外|商|独|资|企|业|r|n5|国|有|独|资|企|业|r|n|上|述|企|业|依|法|可|以|不|设|立|股|东|会|除|此|之|外|其|他|种|类|有|限|责|任|公|司|必|须|设|立|股|东|会|这|是|法|律|的|强|制|性|规|定|请|参|见|中|外|合|资|经|营|企|业|法|中|外|合|作|经|营|企|业|法|外|资|企|业|法|和|公|司|法|的|有|关|规|定|r|n|此|外|即|使|设|立|股|东|会|在|某|种|意|义|上|董|事|会|仍|是|企|业|的|权|利|机|构|同|时|也|可|以|通|过|公|司|章|程|和|股|东|会|决|议|的|方|式|将|股|东|会|的|部|分|权|利|授|予|董|事|会|",
  "qid|qid_252650478374433459|category|教|育|科|学|职|业|教|育|title|2009|年|湖|南|会|计|人|员|继|续|教|育|是|什|么|时|候|desc|answer|可|看|下|参|考|资|料|",
  "qid|qid_6895139452061137503|category|育|儿|宝|宝|婴|儿|期|title|我|的|奶|水|不|够|吃|怎|么|增|加|奶|水|desc|answer|在|某|些|情|况|下|哺|乳|会|产|生|奶|水|不|足|的|现|象|但|多|半|都|可|加|以|矫|正|比|方|说|r|r1|过|早|添|加|配|方|奶|或|其|它|食|品|这|是|造|成|奶|水|不|足|的|主|因|之|一|由|于|宝|宝|已|经|吃|了|其|它|食|物|并|不|感|觉|饥|饿|便|自|动|减|少|吸|奶|的|时|间|如|此|一|来|乳|汁|便|自|动|调|节|减|少|产|量|r|r2|喂|食|时|间|过|短|有|些|妈|妈|限|制|哺|喂|的|次|数|和|时|间|比|方|说|规|定|每|四|个|小|时|才|喂|一|次|或|者|每|次|喂|食|时|间|过|短|等|都|会|造|成|母|奶|产|量|的|减|少|事|实|上|哺|喂|母|奶|不|必|有|固|定|的|时|间|表|宝|宝|饿|了|就|可|以|吃|而|每|次|哺|喂|的|时|间|在|理|想|上|也|应|由|宝|宝|自|己|来|决|定|不|应|加|以|限|制|有|时|候|宝|宝|的|嘴|一|离|开|妈|妈|的|乳|头|妈|妈|便|立|刻|假|设|宝|宝|已|经|吃|饱|了|事|实|上|宝|宝|可|能|只|是|想|休|息|一|下|喘|一|口|气|吸|奶|是|很|累|的|有|没|有|听|过|使|出|吃|奶|的|力|气|这|句|话|或|是|因|为|好|奇|心|想|要|观|察|周|围|的|环|境|等|妈|妈|可|以|再|试|着|喂|喂|看|r|r3|人|工|挤|乳|器|损|坏|或|不|会|使|用|有|些|妈|妈|想|要|确|定|自|己|的|乳|汁|是|否|足|够|于|是|便|想|到|可|以|挤|挤|看|没|想|到|却|挤|不|出|多|少|东|西|于|是|便|以|为|自|己|的|奶|水|不|够|其|实|挤|奶|也|是|需|要|时|间|学|习|的|更|何|况|宝|宝|的|嘴|比|任|何|挤|奶|器|更|能|有|效|的|移|出|奶|水|有|时|候|有|些|妈|妈|已|经|恢|复|上|班|工|作|便|用|挤|乳|器|挤|出|母|乳|喂|食|宝|宝|没|想|到|过|了|一|阵|子|却|越|挤|越|少|在|这|种|情|况|下|请|先|检|查|人|工|挤|乳|器|是|否|损|坏|有|时|候|换|一|个|新|的|机|器|便|可|解|决|问|题|不|过|请|妈|妈|们|注|意|由|于|大|多|数|人|工|挤|乳|器|并|不|像|宝|宝|的|嘴|那|般|具|有|增|加|母|乳|产|量|的|能|力|因|此|在|挤|的|时|候|千|万|保|持|耐|心|慢|慢|来|多|花|一|点|时|间|就|可|以|了|r|r4|婴|儿|快|速|生|长|期|growth|spurts|在|婴|儿|约|二|至|三|周|六|周|以|及|三|个|月|大|左|右|婴|儿|在|此|时|会|经|历|较|为|快|速|的|生|长|阶|段|此|时|宝|宝|会|频|频|要|求|吸|奶|这|可|以|说|是|宝|宝|以|其|本|能|在|增|加|妈|妈|的|奶|水|产|量|因|此|若|在|此|时|添|加|其|它|食|物|反|而|会|妨|碍|奶|水|的|增|加|r|r5|授|乳|妇|营|养|不|良|授|乳|妈|妈|应|该|注|意|营|养|摄|取|适|当|的|热|量|不|宜|过|度|减|轻|体|重|以|免|影|响|乳|汁|分|泌|最|好|多|食|用|富|含|蛋|白|质|的|食|物|进|食|适|量|的|液|体|并|注|意|营|养|是|否|均|衡|r|r6|药|物|影|响|授|乳|妈|妈|吃|含|雌|激|素|的|避|孕|药|或|因|疾|病|接|受|某|些|药|物|治|疗|有|时|会|影|响|泌|乳|量|此|时|可|改|采|其|它|方|式|避|孕|或|避|免|使|用|某|些|药|物|等|在|就|诊|时|应|让|医|师|知|道|正|在|喂|母|乳|r|r7|母|亲|睡|眠|不|足|压|力|过|大|为|人|母|的|工|作|十|分|耗|费|精|神|及|体|力|有|时|会|影|响|母|乳|的|分|泌|建|议|妈|妈|们|放|松|心|情|多|找|时|间|休|息|就|可|以|解|决|暂|时|奶|水|不|足|的|现|象|食|疗|方|一|金|针|豆|浆|r|r|原|料|金|针|菜|黄|花|菜|30g|鲜|豆|浆|250g|白|糖|20g|r|r|做|法|将|金|针|菜|用|温|水|润|软|煎|去|花|蒂|洗|净|放|于|砂|锅|中|注|入|清|水|100ml|大|火|烧|开|后|加|入|鲜|豆|浆|和|白|糖|再|共|煮|10|分|钟|r|r|服|法|分|2|次|服|早|晚|各|1|次|连|服|3|天|r|r|功|能|清|热|解|毒|利|尿|运|乳|适|用|于|产|妇|缺|乳|r|r|食|疗|方|二|花|生|鲤|鱼|汤|r|r|原|料|花|生|仁|60g|鲤|鱼|1|尾|重|约|500g|生|姜|5g|料|酒|5g|酱|油|3g|化|猪|油|10g|精|盐|3g|味|精|0.2g|r|r|做|法|1|生|姜|洗|净|切|丝|鲤|鱼|刮|鳞|去|腮|剖|腹|去|内|脏|砍|成|块|一|起|放|入|碗|中|加|入|料|酒|酱|油|和|精|盐|拌|和|均|匀|腌|渍|入|味|2|花|生|仁|洗|净|放|于|砂|锅|中|注|入|清|水|500ml|小|火|煮|至|酥|烂|下|腌|渍|好|的|鲤|鱼|块|化|猪|油|再|煮|20|分|钟|加|味|精|调|匀|r|r|服|法|分|2|次|服|趁|热|食|花|生|和|鱼|喝|汤|r|r|功|效|祛|湿|利|水|醒|脾|和|胃|通|乳|适|用|于|产|后|缺|乳|r|r|食|疗|方|三|木|瓜|猪|蹄|r|r|原|料|新|鲜|番|木|瓜|1|只|重|约|300g|猪|蹄|1|只|重|约|400g|精|盐|4g|味|精|0.5g|r|r|做|法|1|番|木|瓜|洗|净|剖|开|去|籽|切|成|3cm|小|块|2|猪|蹄|烙|净|余|毛|刮|洗|干|净|砍|成|小|块|放|于|砂|锅|中|注|入|清|水|800ml|盖|严|先|用|大|火|烧|开|撇|去|浮|沫|转|用|小|火|炖|至|八|成|酥|烂|时|加|入|番|木|瓜|片|继|用|小|火|炖|至|酥|烂|下|盐|和|味|精|调|匀|r|r|服|法|分|2|次|服|早|晚|各|1|次|趁|温|食|番|木|瓜|猪|蹄|喝|汤|r|r|功|效|健|脾|胃|助|消|化|下|乳|汁|适|用|于|产|妇|缺|乳|",
  "qid|qid_3291007687817816835|category|教|育|科|学|外|语|学|习|英|语|考|试|title|英|语|what'sup|是|什|么|意|思|desc|answer|what's|up|就|是|问|对|方|近|来|如|何|有|点|像|中|文|里|的|最|近|怎|样|通|常|没|什|么|事|就|会|回|答|not|much|或|者|nothing|r|n|r|n|不|过|还|有|一|种|情|况|也|很|常|见|就|是|对|方|也|回|答|了|一|句|what's|up|在|这|种|情|况|下|what's|up|几|乎|相|当|于|hello|r|n|r|n|what's|up|也|常|被|用|来|问|人|家|有|什|么|事|比|如|别|人|登|门|拜|访|你|就|可|以|用|意|为|有|何|贵|干|哪|r|n|r|n|",
  "qid|qid_3661354683987560684|category|烦|恼|恋|爱|title|jiaoyu|如|何|培|养|学|生|阅|读|能|力|desc|如|何|培|养|阅|读|能|力|answer|通|过|认|真|学|习|新|课|程|标|准|我|深|深|地|体|会|到|阅|读|能|力|对|一|个|学|生|来|说|是|一|种|十|分|重|要|的|能|力|同|时|也|是|学|生|应|该|具|备|的|一|个|重|要|素|质|阅|读|能|力|的|强|弱|与|学|生|获|取|知|识|提|高|学|习|兴|趣|增|长|见|识|以|及|培|养|自|学|能|力|等|方|面|都|有|密|切|联|系|小|学|生|阅|读|能|力|提|高|了|不|但|能|够|加|强|小|学|生|在|阅|读|中|的|情|感|体|验|而|且|可|以|让|学|生|积|累|丰|富|的|语|文|知|识|形|成|良|好|的|语|感|能|让|小|学|生|去|理|解|鉴|赏|作|品|从|而|受|到|高|尚|情|操|与|趣|味|的|熏|陶|同|时|也|能|发|展|他|们|的|个|性|丰|富|他|们|的|精|神|世|界|我|认|为|必|须|加|强|小|学|生|的|课|外|阅|读|提|高|小|学|生|的|课|外|阅|读|的|能|力|而|目|前|在|农|村|小|学|由|于|受|办|学|条|件|的|影|响|学|生|的|课|外|阅|读|量|少|导|致|作|文|能|力|低|学|生|阅|读|兴|趣|不|高|达|不|到|应|有|的|教|学|效|果|如|何|针|对|农|村|的|教|学|实|际|情|况|提|高|农|村|小|学|生|的|课|外|阅|读|能|力|我|认|为|应|该|从|以|下|几|方|面|入|手|r|一|激|发|学|生|的|兴|趣|让|他|们|喜|爱|阅|读|r|兴|趣|是|最|好|的|老|师|是|人|们|从|事|任|何|活|动|的|动|力|学|生|有|了|阅|读|兴|趣|才|能|从|内|心|深|处|对|课|外|阅|读|产|生|主|动|需|要|因|此|在|教|学|中|我|用|了|一|些|特|殊|的|方|式|来|激|发|学|生|阅|读|的|兴|趣|从|而|让|他|们|快|乐|地|接|受|自|己|想|要|学|习|的|语|文|知|识|r|教|学|中|我|经|常|鼓|励|学|生|用|自|己|喜|欢|的|方|式|进|行|阅|读|如|教|学|海|底|世|界|一|课|时|让|让|学|生|仔|细|读|一|读|想|想|准|备|用|什|么|方|法|来|学|习|通|过|思|考|有|的|学|生|说|我|准|备|一|句|一|句|地|读|边|读|边|想|象|海|底|的|各|种|小|动|物|不|理|解|的|地|方|多|读|几|遍|并|且|可|以|对|照|插|图|理|解|有|的|说|我|准|备|和|同|桌|比|赛|朗|读|看|谁|读|得|好|还|有|的|说|我|想|围|绕|海|底|真|是|一|个|景|色|奇|异|物|产|丰|富|的|世|界|这|句|话|展|开|学|习|解|决|了|这|个|问|题|这|篇|课|文|也|就|读|懂|了|总|之|在|阅|读|教|学|中|我|常|常|尊|重|学|生|的|选|择|满|足|学|生|的|愿|望|鼓|励|他|们|根|据|自|己|的|体|验|用|自|己|的|思|维|方|式|去|探|究|从|而|发|展|自|己|的|个|性|r|二|教|给|他|们|读|书|的|方|法|让|学|生|学|会|阅|读|r|光|有|兴|趣|是|远|远|不|够|的|我|们|要|进|一|步|引|导|他|们|不|仅|爱|读|而|且|会|读|要|读|得|更|好|更|有|收|效|作|为|教|师|我|们|应|该|向|学|生|介|绍|一|些|行|之|有|效|的|读|书|方|法|r|略|读|法|这|种|方|法|的|运|用|一|般|是|根|据|学|生|在|课|内|学|习|或|写|作|上|的|某|种|需|要|有|选|择|地|阅|读|有|关|书|报|的|有|关|篇|章|或|有|关|部|分|以|便|学|以|致|用|如|学|习|了|太|阳|一|课|学|生|为|了|更|全|面|更|深|刻|的|了|解|人|类|与|太|阳|自|然|的|密|切|关|系|有|目|的|地|阅|读|人|与|自|然|一|书|或|上|网|查|阅|而|且|要|指|导|学|生|去|图|书|馆|借|阅|相|关|的|资|料|再|选|择|自|己|所|需|要|的|部|分|以|培|养|摘|录|资|料|和|运|用|资|料|的|能|力|同|时|也|丰|富|了|语|言|积|累|提|高|了|阅|读|兴|趣|和|阅|读|能|力|r|精|读|法|所|谓|精|读|法|就|是|对|书|报|上|的|某|些|重|点|文|章|集|中|精|力|逐|字|逐|句|由|表|及|里|精|思|熟|读|的|阅|读|方|法|它|是|培|养|学|生|阅|读|能|力|最|主|要|最|基|本|的|手|段|有|的|文|章|语|言|隽|永|引|经|据|典|情|节|生|动|我|们|可|以|利|用|这|些|作|品|为|依|据|指|导|学|生|精|读|要|求|学|生|全|身|心|投|入|边|读|边|想|边|批|注|逐|渐|养|成|认|真|读|书|的|好|习|惯|r|默|读|法|默|读|法|就|是|对|所|读|的|书|报|不|发|音|不|辨|读|不|转|移|视|线|而|是|快|速|地|观|其|概|貌|这|就|要|求|学|生|在|快|速|的|浏|览|中|要|集|中|注|意|力|作|出|快|速|的|信|息|处|理|和|消|化|利|用|默|读|法|可|以|做|到|用|最|少|的|时|间|获|取|尽|量|多|的|信|息|r|摘|抄|评|价|法|此|阅|读|法|就|是|在|阅|读|过|程|中|根|据|自|己|的|需|要|将|有|关|的|词|句|段|乃|至|全|篇|原|文|摘|抄|下|来|或|对|阅|读|的|重|点|难|点|部|分|划|记|号|作|注|释|写|评|语|文|章|中|富|有|教|育|意|义|的|警|句|格|言|精|彩|生|动|的|词|句|段|落|可|以|摘|录|下|来|积|存|进|自|己|设|立|的|词|库|中|为|以|后|的|作|文|准|备|了|丰|富|的|语|言|积|累|同|时|还|可|以|对|自|己|订|阅|的|报|刊|杂|志|将|其|中|的|好|文|章|剪|裁|下|来|粘|贴|到|自|己|的|读|书|笔|记|中|r|三|创|造|条|件|让|学|生|快|乐|阅|读|r|我|要|努|力|为|学|生|创|造|阅|读|的|条|件|让|学|生|在|自|由|的|空|间|幽|雅|的|环|境|中|舒|舒|服|服|的|进|行|阅|读|r|每|个|学|生|的|兴|趣|个|性|特|点|不|尽|相|同|因|此|他|们|所|习|惯|的|呈|现|阅|读|体|验|的|方|式|也|不|一|样|有|的|喜|欢|朗|读|有|的|喜|欢|表|演|有|的|喜|欢|表|述|我|们|应|该|尊|重|他|们|的|选|择|如|阅|读|了|草|原|以|后|我|鼓|励|学|生|说|同|学|们|的|感|受|一|定|很|深|请|你|用|自|己|喜|欢|的|方|式|表|达|出|来|于|是|他|们|有|的|深|情|地|诵|读|有|的|和|着|音|乐|翩|翩|起|舞|有|的|高|歌|一|曲|有|的|写|起|了|小|诗|有|的|和|同|桌|交|流|课|堂|成|了|学|生|积|极|参|与|主|动|表|现|的|舞|台|r|四|积|极|评|价|让|学|生|对|阅|读|有|不|同|的|看|法|r|阅|读|教|学|应|提|倡|标|新|立|异|允|许|学|生|见|仁|见|智|地|解|读|作|品|对|作|品|进|行|创|造|性|的|思|考|r|总|之|在|提|高|阅|读|能|力|时|我|们|要|处|处|让|学|生|体|验|成|功|的|喜|悦|当|学|生|在|获|得|成|功|后|会|更|坚|持|大|量|的|广|泛|的|阅|读|良|好|的|读|书|习|惯|也|就|随|之|形|成|同|时|个|人|的|进|步|往|往|又|是|同|学|们|效|仿|的|范|例|这|样|班|里|就|会|形|成|浓|厚|的|课|外|阅|读|氛|围|学|生|的|阅|读|能|力|也|会|随|之|提|高|",
  "qid|qid_7324298667277705568|category|电|脑|网|络|软|件|图|像|处|理|软|件|title|cad|软|件|的|角|度|取|值|为|整|数|1|度|2|度|3|度|4|度|没|有|2.1|度|desc|cad|的|角|度|取|值|为|整|数|1|度|2|度|3|度|4|度|没|有|2.1|度|2.1|度|可|已|设|置|吗|画|个|52.606|度|的|角|画|不|了|answer|可|以|设|置|r|n|先|正|常|画|出|角|度|在|标|注|完|角|度|值|后|可|以|点|击|鼠|标|右|键|选|择|精|度|再|选|择|合|适|的|精|度|位|数|r|n|也|可|以|在|标|注|样|式|里|把|标|注|精|度|统|一|调|整|这|样|以|后|所|有|的|标|注|都|可|以|按|照|这|个|精|度|标|注|了|",
  "qid|qid_2680849588140160855|category|社|会|民|生|法|律|title|物|流|公|司|发|货|不|按|照|协|议|发|货|照|成|货|物|损|坏|怎|么|赔|偿|desc|物|流|发|货|不|按|照|协|议|发|货|照|成|货|物|损|坏|怎|么|赔|偿|answer|一|般|按|照|货|物|价|值|赔|偿|有|空|加|个|微|博|好|友|给|个|好|评|啊|",
  "qid|qid_508534880557955811|category|社|会|民|生|其|他|社|会|话|题|title|会|开|车|的|好|处|我|是|个|大|学|生|请|问|有|驾|驶|证|对|以|后|找|工|作|帮|助|大|吗|desc|我|是|个|请|问|有|驾|驶|证|对|以|后|找|工|作|帮|助|大|吗|answer|你|好|r|n|r|n|艺|不|亏|人|只|要|是|技|术|学|会|总|比|不|会|好|有|驾|驶|证|对|以|后|找|工|作|应|该|有|帮|助|如|果|你|打|算|开|车|帮|助|非|常|大|啊|r|n|r|n|建|议|在|读|期|间|还|是|专|心|致|志|的|学|习|你|的|专|业|为|好|办|驾|驶|证|还|得|住|驾|校|住|驾|校|得|花|钱|每|年|年|检|还|得|花|钱|如|果|你|的|家|庭|条|件|不|错|而|且|是|学|习|之|余|有|足|够|的|时|间|也|可|以|办|否|则|还|是|毕|业|后|再|做|打|算|吧|",
  "qid|qid_8193953555975861150|category|文|化|艺|术|书|画|美|术|title|何|谓|普|普|来|历|desc|answer|普|普|风|美|国|纽|约|的|时|尚|休|闲|品|牌|r|n|r|n|英|文|商|标|fgfeel|是|feel|good|feel|感|受|好|心|情|的|缩|写|这|也|正|是|普|普|风|所|倡|导|的|生|活|理|念|自|由|阳|光|健|康|爽|朗|普|普|风|以|22|35|岁|时|尚|一|族|的|年|青|人|为|消|费|对|象|价|位|适|中|它|的|成|衣|大|多|采|用|进|口|的|休|闲|布|料|和|具|有|独|特|工|艺|的|高|科|技|面|料|加|上|其|全|新|的|设|计|理|念|把|时|装|化|的|合|体|裁|剪|和|流|行|元|素|融|入|到|轻|松|自|由|的|休|闲|之|中|从|而|提|高|了|休|闲|的|品|味|也|拉|近|了|休|闲|和|时|尚|之|间|的|距|离|产|品|给|人|感|觉|细|腻|丰|富|从|服|装|的|设|计|与|裁|剪|上|散|发|出|不|可|抗|拒|的|时|代|气|息|与|魅|力|整|个|品|牌|给|人|一|种|大|器|加|上|时|尚|与|稳|重|融|合|的|美|妙|感|觉|必|将|受|到|崇|尚|新|生|活|的|都|市|时|尚|一|族|的|青|睐|普|普|风|于|2000|年|8|月|正|式|抢|滩|中|国|目|前|正|在|深|圳|市|飘|二|千|服|装|有|限|公|司|科|学|的|管|理|和|运|作|之|下|以|强|大|的|资|金|和|所|倡|导|的|全|新|的|休|闲|文|化|为|后|盾|以|纽|约|香|港|两|个|时|尚|前|沿|城|市|为|龙|头|向|中|国|的|服|装|市|场|进|发|来|吧|让|我|们|一|起|来|享|受|普|普|风|享|受|好|心|情|享|受|无|边|的|自|由|",
  "qid|qid_5656001177274245606|category|游|戏|网|络|游|戏|title|问|道|游|戏|有|什|么|办|法|可|以|快|速|刷|友|好|度|的|desc|r|n|有|什|么|办|法|可|以|快|速|刷|友|好|度|的|answer|开|ld|打|卡|一|回|合|清|怪|即|可|以|打|到|卡|又|可|以|增|加|友|好|度|每|杀|一|次|怪|增|加|1|点|友|好|度|当|然|有|钱|的|话|送|玫|瑰|巧|克|力|等|等|最|快|了|胡|子|任|务|每|天|只|能|做|一|次|全|部|综|合|的|话|很|快|就|够|好|友|度|结|婚|闹|洞|房|了|",
  "qid|qid_3449485945632669470|category|生|活|购|物|title|市|区|哪|里|还|有|卖|北|京|冰|糖|葫|芦|的|抓|狂|抓|狂|抓|狂|抓|狂|desc|市|区|哪|里|还|有|卖|北|京|冰|糖|的|抓|狂|抓|狂|抓|狂|抓|狂|求|解|answer|牌|坊|街|",
  "qid|qid_624441425152161418|category|生|活|购|物|title|请|问|各|位|大|虾|北|京|哪|里|有|卖|橄|榄|油|的|我|想|买|不|知|哪|里|有|卖|的|那|位|desc|我|想|买|不|知|哪|里|有|卖|的|那|位|知|道|请|告|诉|我|谢|谢|answer|学|院|路|学|府|园|站|有|一|家|普|尔|斯|马|特|超|市|里|面|就|有|正|宗|橄|榄|油|卖|",
  "qid|qid_7381878519240599315|category|游|戏|网|易|游|戏|魔|兽|世|界|title|50|级|最|好|的|双|手|武|器|想|在|50|级|的|战|场|呆|一|段|时|间|所|以|求|个|50|desc|想|在|50|级|的|战|场|呆|一|段|时|间|所|以|求|个|50|级|的|好|双|手|本|人|战|士|answer|命|运|不|错|还|有|就|是|碎|灵|r|n|r|n|最|好|的|60|前|的|双|手|武|器|是|厄|运|贡|品|的|古|树|天|敌|r|n|古|树|天|敌|r|n|拾|取|后|绑|定|r|n|双|手|斧|r|n128|193|伤|害|速|度|2.70|r|n|每|秒|伤|害|59.4|r|n|25|力|量|r|n|15|耐|力|r|n|需|要|等|级|58|r|n|装|备|使|你|造|成|致|命|一|击|的|几|率|提|高|2|r|n|r|n|慢|慢|去|刷|吧|",
  "qid|qid_2339683591001722848|category|商|业|理|财|基|金|title|1|大|宗|交|易|的|金|额|与|数|量|有|何|规|定|一|a|股|交|易|数|量|在|50|万|股|desc|1|大|宗|交|易|的|金|额|与|数|量|有|何|规|定|r|r|一|a|股|交|易|数|量|在|50|万|股|含|以|上|或|交|易|金|额|在|300|万|元|含|人|民|币|以|上|b|股|交|易|数|量|在|50|万|股|含|以|上|或|交|易|金|额|在|30|万|美|元|含|以|上|二|交|易|数|量|在|300|万|份|含|以|上|或|交|易|金|额|在|300|万|元|含|人|民|币|以|上|三|债|券|交|易|数|量|在|2|万|手|含|以|上|或|交|易|金|额|在|2000|万|元|含|人|民|币|以|上|四|债|券|回|购|交|易|数|量|在|5|万|手|含|以|上|或|交|易|金|额|在|5000|万|元|含|人|民|币|以|上|r|r|在|上|海|证|券|交|易|所|以|下|简|称|本|所|市|场|进|行|的|证|券|单|笔|买|卖|达|到|如|下|最|低|限|额|可|以|采|用|1|大|宗|交|易|的|金|额|与|数|量|有|何|规|定|r|r|一|a|股|交|易|数|量|在|50|万|股|含|以|上|或|交|易|金|额|在|300|万|元|含|人|民|币|以|上|b|股|交|易|数|量|在|50|万|股|含|以|上|或|交|易|金|额|在|30|万|美|元|含|以|上|二|交|易|数|量|在|300|万|份|含|以|上|或|交|易|金|额|在|300|万|元|含|人|民|币|以|上|三|债|券|交|易|数|量|在|2|万|手|含|以|上|或|交|易|金|额|在|2000|万|元|含|人|民|币|以|上|四|债|券|回|购|交|易|数|量|在|5|万|手|含|以|上|或|交|易|金|额|在|5000|万|元|含|人|民|币|以|上|r|r|在|上|海|证|券|交|易|所|以|下|简|称|本|所|市|场|进|行|的|证|券|单|笔|买|卖|达|到|如|下|最|低|限|额|可|以|采|用|大|宗|交|易|方|式|r|r|2|投|资|者|如|何|进|行|大|宗|交|易|r|r|拥|有|本|所|专|用|席|位|的|机|构|可|通|过|该|席|位|进|行|大|宗|交|易|r|r|其|他|投|资|者|进|行|大|宗|交|易|应|委|托|其|办|理|指|定|交|易|的|本|所|会|员|办|理|r|r|3|大|宗|交|易|的|交|易|时|间|如|何|规|定|r|r|大|宗|交|易|的|交|易|时|间|为|本|所|交|易|日|的|15|00|15|30|本|所|在|上|述|时|间|内|受|理|大|宗|交|易|申|报|大|宗|交|易|用|户|可|在|交|易|日|的|14|30|15|00|登|陆|本|所|大|宗|交|易|电|子|系|统|进|行|开|始|前|的|准|备|工|作|大|宗|交|易|用|户|可|在|交|易|日|的|15|30|16|00|通|过|本|所|大|宗|交|易|电|子|系|统|查|询|当|天|大|宗|交|易|情|况|或|接|受|当|天|成|交|数|据|r|r|4|哪|些|证|券|不|能|进|行|大|宗|交|易|r|r|本|所|交|易|日|的|15|00|前|仍|处|于|停|牌|状|态|的|证|券|本|所|不|受|理|其|大|宗|交|易|的|申|报|r|r|5|大|宗|交|易|的|申|报|包|括|哪|些|内|容|r|r|大|宗|交|易|的|申|报|包|括|意|向|申|报|和|成|交|申|报|意|向|申|报|应|包|括|以|下|内|容|r|r|一|证|券|代|码|二|股|东|帐|号|三|买|卖|方|向|四|本|所|规|定|的|其|他|内|容|意|向|申|报|中|是|否|明|确|交|易|价|格|和|交|易|数|量|由|申|报|方|决|定|买|方|和|卖|方|就|大|宗|交|易|达|成|一|致|后|代|表|买|方|和|卖|方|的|会|员|分|别|通|过|各|自|席|位|拥|有|本|所|专|用|席|位|的|机|构|通|过|该|席|位|进|行|成|交|申|报|成|交|申|报|应|包|括|以|下|内|容|r|r|一|证|券|代|码|二|股|东|帐|号|三|成|交|价|格|四|成|交|数|量|五|买|卖|方|向|六|买|卖|对|手|方|席|位|号|七|本|所|规|定|的|其|他|内|容|买|方|和|卖|方|的|上|述|成|交|申|报|中|证|券|代|码|成|交|价|格|和|成|交|数|量|必|须|一|致|r|r|截|止|到|2008|年|3|月|我|国|a|股|市|场|解|禁|股|份|数|已|达|到|800|亿|股|之|多|从|2006|年|6|月|份|算|起|平|均|每|月|解|禁|数|量|约|38|亿|股|以|解|禁|时|点|个|股|股|价|动|态|测|算|已|累|计|解|禁|金|额|约|12639|亿|元|平|均|每|月|约|解|禁|601|亿|元|r|r|目|前|阶|段|减|持|对|市|场|的|影|响|更|有|可|能|是|通|过|挤|出|效|应|来|体|现|的|它|占|用|了|全|市|场|一|定|的|资|金|量|但|对|个|股|自|身|影|响|并|不|显|著|减|持|个|股|需|要|新|的|资|金|承|接|这|就|导|致|了|其|它|个|股|上|资|产|配|置|的|下|降|从|而|引|发|其|它|个|股|乃|至|市|场|下|跌|这|与|宏|观|整|体|观|察|相|一|致|此|外|目|前|阶|段|减|持|对|市|场|更|多|的|是|一|种|心|理|效|应|减|持|规|模|的|积|累|令|投|资|者|形|成|一|种|心|理|上|的|担|忧|r|r|2008|年|4|月|起|将|解|禁|股|数|约|676|亿|股|平|均|每|月|75|亿|股|全|年|峰|值|出|现|在|8|月|月|解|禁|214|亿|股|月|解|禁|规|模|是|过|往|水|平|的|近|两|倍|以|2008|年|3|月|28|日|收|盘|价|数|据|测|算|全|年|解|禁|金|额|大|约|是|10546|亿|元|月|解|禁|金|额|约|1171|亿|元|如|果|以|过|往|减|持|占|解|禁|比|例|为|9|数|据|测|算|每|月|减|持|股|数|约|6.8|亿|股|减|持|金|额|大|约|是|105|亿|元|r|r|6|意|向|申|报|时|应|注|意|哪|些|问|题|r|r|意|向|申|报|应|当|真|实|有|效|申|报|方|价|格|不|明|确|的|视|为|至|少|愿|以|规|定|的|最|低|价|格|买|入|或|最|高|价|格|卖|出|数|量|不|明|确|的|视|为|至|少|愿|以|最|低|限|额|成|交|r|r|买|方|和|卖|方|根|据|大|宗|交|易|的|意|向|申|报|信|息|就|大|宗|交|易|的|价|格|和|数|量|等|要|素|进|行|议|价|协|商|当|意|向|申|报|被|其|他|参|与|者|接|受|包|括|其|他|参|与|者|报|出|比|意|向|申|报|更|优|的|价|格|时|申|报|方|应|当|至|少|与|一|个|接|受|意|向|申|报|的|参|与|者|进|行|成|交|申|报|r|r|7|大|宗|交|易|的|成|交|价|格|如|何|确|定|r|r|大|宗|交|易|的|成|交|价|格|由|买|方|和|卖|方|在|当|日|最|高|和|最|低|成|交|价|格|之|间|确|定|该|证|券|当|日|无|成|交|的|以|前|收|盘|价|为|成|交|价|r|r|8|大|宗|交|易|的|成|交|申|报|是|否|能|撤|消|或|变|更|r|r|大|宗|交|易|的|成|交|申|报|须|经|本|所|确|认|本|所|确|认|后|买|方|和|卖|方|不|得|撤|销|或|变|更|成|交|申|报|并|必|须|承|认|交|易|结|果|履|行|相|关|的|清|算|交|收|义|务|r|r|9|进|行|大|宗|交|易|的|帐|户|证|券|帐|户|和|资|金|帐|户|是|否|应|有|足|额|的|证|券|或|资|金|r|r|会|员|应|保|证|大|宗|交|易|投|资|者|帐|户|包|括|会|员|自|营|账|户|实|际|拥|有|与|意|向|申|报|和|成|交|申|报|相|对|应|的|证|券|或|资|金|r|r|10|大|宗|交|易|的|费|用|如|何|收|取|r|r|大|宗|交|易|的|交|易|经|手|费|按|集|中|竞|价|交|易|方|式|下|同|品|种|证|券|的|交|易|经|手|费|率|标|准|下|浮|其|中|股|票|基|金|下|浮|30|债|券|债|券|回|购|下|浮|10|其|它|费|用|同|集|中|竞|价|交|易|r|r|11|大|宗|交|易|的|信|息|如|何|披|露|r|r|大|宗|交|易|收|盘|后|本|所|在|指|定|媒|体|上|公|布|每|笔|大|宗|交|易|的|成|交|量|成|交|价|以|及|买|卖|双|方|所|在|会|员|营|业|部|的|名|称|通|过|本|所|专|用|席|位|达|成|大|宗|交|易|时|为|该|专|用|席|位|所|属|机|构|的|名|称|大|宗|交|易|成|交|后|涉|及|法|定|信|息|披|露|要|求|的|买|卖|双|方|应|依|照|有|关|法|律|法|规|履|行|信|息|披|露|义|务|r|r|12|大|宗|交|易|是|否|纳|入|指|数|计|算|r|r|大|宗|交|易|不|纳|入|指|数|计|算|r|r|13|大|宗|交|易|的|价|格|是|否|作|为|该|证|券|当|日|的|收|盘|价|r|r|大|宗|交|易|的|成|交|价|不|作|为|该|证|券|当|日|的|收|盘|价|r|answer|谢|谢|又|学|了|一|些|新|知|识|",
  "qid|qid_7086440755587918295|category|生|活|生|活|常|识|title|谁|用|过|水|宜|生|杯|真|的|是|那|么|有|作|用|吗|我|看|电|视|上|的|宣|传|广|告|觉|desc|谁|用|过|水|宜|生|杯|真|的|是|那|么|有|作|用|吗|r|r|我|看|上|的|宣|传|广|告|觉|得|有|点|不|可|思|议|所|以|上|来|问|问|answer|听|了|广|告|都|想|信|的|话|癌|症|能|治|愈|糖|尿|病|能|治|愈|高|血|压|不|要|长|期|服|药|没|有|哪|种|病|不|能|治|的|那|中|国|的|医|院|都|要|关|门|医|生|都|要|改|行|了|中|国|人|平|均|寿|命|早|已|领|先|世|界|了|",
  "qid|qid_4243935411392610351|category|商|业|理|财|财|务|税|务|title|退|税|营|业|税|城|建|税|教|育|费|附|加|所|得|税|分|别|怎|么|做|帐|desc|answer|国|家|税|务|总|局|规|定|r|n|收|到|退|回|的|流|转|税|增|值|税|营|业|税|城|建|税|教|育|费|附|加|等|贷|补|贴|收|入|r|n|收|到|退|回|的|企|业|所|得|税|贷|所|得|税|",
  "qid|qid_8472921637205595238|category|游|戏|网|络|游|戏|title|我|这|样|的|配|置|会|不|会|卡|游|戏|登|不|上|去|没|试|过|不|知|道|xp2500|o|desc|2500|oc|3200|rkms|512m|ddr400|r|盈|通|9800pro|rst|sata|7200|8m|r|answer|除|非|网|络|问|题|否|则|肯|定|不|会|卡|的|r|n|效|果|中|等|也|能|很|流|畅|进|行|r|n|系|统|优|化|的|好|的|话|开|最|大|效|果|也|是|没|问|题|的|",
  "qid|qid_5642645166941049746|category|生|活|交|通|出|行|问|路|title|深|圳|至|赣|州|怎|样|走|请|问|深|圳|开|车|到|赣|州|应|该|怎|样|走|啊|请|各|位|可|以|详|细|告|desc|请|问|深|圳|到|赣|州|应|该|怎|样|走|啊|请|各|位|可|以|详|细|告|诉|吗|answer|从|广|州|走|从|广|州|惠|州|河|源|和|平|定|南|赣|州|到|了|赣|州|高|速|路|一|路|有|指|示|到|南|昌|只|要|600|余|公|里|全|程|高|速|r|r|高|速|具|体|路|段|如|下|广|州|北|环|沙|河|立|交|白|云|隧|道|广|氮|出|口|广|深|高|速|火|村|出|口|广|惠|高|速|惠|河|高|速|阿|深|高|速|粤|赣|高|速|r|r|千|万|注|意|1|在|广|氮|出|口|有|指|示|牌|标|明|广|惠|广|深|高|速|下|来|后|一|路|都|有|路|标|指|示|是|广|深|高|速|不|必|惊|慌|到|了|火|村|出|口|就|要|注|意|一|条|路|去|广|惠|一|条|路|去|广|深|走|错|了|就|往|东|莞|方|向|去|了|我|一|同|事|元|旦|时|就|走|错|了|结|果|到|了|东|莞|再|掉|头|多|走|了|近|百|公|里|2|还|有|就|是|上|广|惠|高|速|到|了|小|金|口|下|了|广|惠|高|速|后|有|指|示|牌|指|明|惠|河|高|速|也|易|走|上|了|惠|河|高|速|后|去|赣|州|就|一|直|走|不|要|下|任|何|出|口|就|行|了|路|线|都|有|很|明|显|的|标|牌|不|会|错|3|到|了|赣|州|后|有|指|示|牌|指|明|如|何|去|吉|安|南|昌|r|r|另|粤|赣|高|速|正|式|名|称|为|阿|深|高|速|终|点|为|深|圳|起|点|应|该|是|和|平|县|与|江|西|定|南|县|之|间|的|一|个|小|镇|全|程|粤|赣|高|速|都|被|标|为|阿|深|高|速|所|以|各|位|不|要|以|为|走|错|了|r|全|程|限|速|100|公|里|小|时|另|江|西|境|内|隧|道|较|多|限|速|80|还|要|开|大|灯|r|r|从|深|圳|走|就|是|直|接|从|梅|观|上|机|荷|再|转|上|深|惠|再|上|惠|河|就|可|以|直|接|上|阿|深|高|速|了|r|",
  "qid|qid_5865257258993981788|category|健|康|内|科|神|经|内|科|title|遵|义|市|治|疗|女|性|癫|痫|需|要|注|意|患|者|的|哪|些|desc|市|治|疗|女|性|癫|痫|需|要|注|意|患|者|的|哪|些|answer|遵|义|市|治|疗|女|性|癫|痫|需|要|注|意|患|者|的|哪|些|有|人|统|计|在|妊|娠|前|已|控|制|癫|痫|发|作|者|50|的|人|仍|可|保|持|不|发|作|5|的|人|发|作|减|小|45|的|患|者|发|作|回|数|增|多|如|果|在|妊|娠|前|发|作|多|于|每|月|1|次|其|妊|娠|时|则|有|60|98|的|发|作|增|多|并|常|在|头|3|个|月|内|发|作|恶|化|妊|娠|期|癫|痫|变|化|的|原|因|至|今|尚|未|清|楚|可|能|与|妊|娠|毒|血|症|血|管|损|害|静|脉|栓|塞|妊|娠|引|发|脑|病|和|抗|癫|痫|药|血|浓|度|下|降|等|有|关|有|报|告|癫|痫|患|者|妊|娠|时|并|发|症|发|病|率|能|增|加|50|如|孕|期|女|性|生|殖|道|流|血|妊|娠|毒|血|症|早|产|剖|腹|产|自|发|流|产|及|羊|膜|炎|等|而|且|婴|儿|死|亡|率|也|高|因|此|应|当|提|倡|神|经|科|和|医|师|密|切|配|合|病|人|应|选|择|较|好|的|分|娩|遵|义|市|治|疗|女|性|癫|痫|需|要|注|意|患|者|的|哪|些|女|性|承|担|着|生|育|下|一|代|的|重|任|对|于|女|性|癫|痫|患|者|来|讲|怀|孕|期|间|不|能|盲|目|服|药|否|则|不|仅|会|引|起|癫|痫|发|作|更|会|对|胎|儿|造|成|伤|害|因|此|女|性|患|者|若|在|妊|娠|期|间|癫|痫|发|作|一|定|要|谨|慎|治|疗|并|在|医|生|的|指|导|下|服|药|以|免|引|起|胎|儿|畸|形|遵|义|市|治|疗|女|性|癫|痫|需|要|注|意|患|者|的|哪|些|月|经|周|期|的|变|化|主|要|是|性|激|素|的|水|平|在|体|内|发|生|周|期|性|改|变|而|性|激|素|本|身|特|别|是|雌|激|素|对|癫|痫|发|作|影|响|是|明|显|的|这|样|癫|痫|发|作|也|会|随|着|月|经|周|期|的|变|化|而|变|化|月|经|加|重|发|作|在|症|状|性|癫|痫|中|更|为|突|出|那|些|已|经|有|频|频|发|作|者|或|有|月|经|前|紧|张|者|更|可|能|在|月|经|前|发|或|在|月|经|性|流|血|时|发|作|此|外|也|有|一|些|女|性|癫|痫|患|者|的|发|作|只|出|现|在|月|经|前|或|月|经|期|称|之|为|月|经|性|癫|痫|",
  "qid|qid_2561773807125585731|category|健|康|外|科|肛|肠|外|科|title|工|作|太|累|会|造|成|便|秘|吗|整|天|干|重|体|力|劳|动|大|便|老|是|不|通|排|不|净|desc|整|天|干|重|体|力|劳|动|大|便|老|是|不|通|排|不|净|answer|工|作|不|会|直|接|影|响|大|便|但|是|由|于|工|作|是|定|时|的|当|在|工|作|的|时|候|往|往|会|忽|略|便|意|而|便|意|是|很|短|暂|的|如|果|没|有|及|时|去|解|手|就|不|再|有|大|便|的|感|觉|了|时|间|久|了|肠|子|的|运|动|功|能|会|削|弱|r|n|r|n|另|外|身|体|虚|弱|的|人|包|括|亚|健|康|的|人|内|脏|功|能|较|差|肠|蠕|动|也|慢|常|常|伴|有|体|虚|型|便|秘|这|时|候|如|果|用|泻|药|犹|如|雪|上|加|霜|从|中|医|角|度|来|说|正|确|的|方|法|是|以|补|促|泻|也|就|是|通|过|滋|补|让|身|体|强|壮|提|高|肠|子|运|动|力|达|到|治|本|的|目|的|r|n|r|n|干|重|体|力|活|的|人|要|注|意|饭|后|休|息|至|少|半|小|时|在|这|段|时|间|培|养|便|意|特|别|是|早|饭|后|不|论|有|没|有|便|意|都|要|养|成|上|厕|所|的|习|惯|坚|持|一|段|时|间|会|让|肠|子|产|生|反|射|便|秘|问|题|就|可|以|从|根|本|上|解|决|了|r|n|r|n|良|好|的|排|泄|功|能|于|健|康|很|重|要|所|以|为|了|健|康|快|快|调|整|生|活|习|惯|吧|祝|你|健|康|",
  "qid|qid_6378427847623797285|category|游|戏|网|络|游|戏|街|头|篮|球|title|牛|c|进|来|回|答|下|上|次|在|4|区|看|到|个|会|后|中|投|后|3|分|的|c|不|知|道|怎|么|会|的|desc|上|次|在|4|区|看|到|个|会|后|中|投|后|3|分|的|不|知|道|怎|么|会|的|哪|个|人|可|以|告|诉|下|answer|bug|r|n|进|了|不|算|分|r|n|具|体|操|作|r|n|首|先|上|篮|按|两|下|dd|键|不|要|再|按|d|键|了|r|n|然|后|等|下|一|次|进|攻|按|w|挤|人|然|后|轻|按|d|键|做|假|动|作|不|停|的|方|向|转|圈|当|人|物|转|到|正|朝|篮|下|的|时|候|不|停|的|按|d|键|的|投|篮|出|去|r|n|可|能|有|时|候|发|不|出|来|可|以|多|试|",
  "qid|qid_5432373825202796041|category|商|业|理|财|股|票|title|帮|分|析|600527|江|南|高|纤|今|天|是|否|为|上|涨|前|的|蓄|势|调|整|吗|desc|帮|600527|江|南|高|纤|今|天|是|否|为|上|涨|前|的|蓄|势|调|整|吗|answer|该|股|今|天|上|涨|不|是|蓄|势|调|整|而|是|反|弹|出|货|止|赢|",
  "qid|qid_5212414270058077095|category|体|育|运|动|足|球|title|郝|海|东|现|在|干|什|么|了|怎|么|也|不|听|报|道|了|desc|怎|么|也|不|听|报|道|了|answer|郝|海|东|现|在|有|两|个|角|色|一|是|湖|南|湘|军|的|老|板|还|有|在|外|国|的|一|个|不|知|名|的|球|队|踢|球|",
  "qid|qid_5333488564465164841|category|健|康|妇|产|科|产|科|title|怀|孕|什|么|时|候|能|感|觉|到|初|期|症|状|都|有|什|么|怀|孕|什|么|时|候|能|感|觉|到|初|desc|什|么|时|候|能|感|觉|到|初|期|症|状|都|有|什|么|怀|孕|什|么|时|候|能|感|觉|到|初|期|症|状|都|有|什|么|answer|1|停|经|月|经|过|期|10|日|以|上|怀|疑|妊|娠|2|早|孕|反|应|60|以|上|的|妇|女|在|妊|娠|6|周|以|后|会|出|现|畏|寒|头|晕|乏|力|流|涎|嗜|睡|食|欲|不|振|喜|食|酸|物|或|厌|食|油|腻|恶|心|和|晨|起|呕|吐|等|现|象|称|早|孕|反|应|持|续|2|个|月|后|自|行|消|失|n|3|尿|频|n|4|乳|房|变|化|查|体|时|可|见|乳|房|逐|渐|变|大|有|胀|痛|感|n|5|妊|娠|早|期|乳|房|增|大|明|显|充|血|水|肿|随|着|乳|腺|管|的|增|生|皮|下|浅|静|脉|清|晰|可|见|6|受|精|卵|着|床|部|位|变|软|妊|娠|6|7|周|后|双|合|诊|检|查|子|宫|峡|部|极|软|子|宫|体|和|宫|颈|似|不|相|连|称|黑|加|征|是|早|期|妊|娠|的|特|征|性|变|化|n|7|b|超|早|期|检|查|最|早|以|妊|娠|囊|为|判|断|妊|娠|标|准|n|8|超|生|多|普|勒|检|查|n|9|宫|颈|粘|液|检|查|未|见|羊|齿|植|物|叶|状|结|晶|妊|娠|的|可|能|性|大|n|10|妊|娠|试|验|妊|娠|7|9|日|后|可|以|查|妇|女|血|β|hcg|判|断|妊|娠|n|11|基|础|体|温|bbt|的|测|定|如|果|高|温|相|持|续|18|日|不|降|妊|娠|可|能|性|大|如|果|持|续|3|周|不|降|妊|娠|的|可|能|性|更|大|",
  "qid|qid_8022711795984531330|category|生|活|生|活|常|识|title|吃|什|么|生|精|比|较|快|吃|什|么|生|精|快|而|且|易|食|用|desc|吃|什|么|生|精|快|而|且|易|食|用|answer|多|食|鱼|鳔|鱼|子|韭|菜|小|河|虾|虾|仁|干|果|仁|",
  "qid|qid_4925683491148619350|category|商|业|理|财|股|票|title|清|泉|点|评|之|39|角|度|与|变|线|以|前|是|准|备|放|在|80|讲|以|后|进|行|的|也|就|desc|以|前|是|准|备|放|在|80|讲|以|后|进|行|的|也|就|是|说|准|备|在|介|绍|完|基|本|的|技|术|分|析|之|后|但|大|盘|运|行|至|目|前|的|敏|感|阶|段|个|股|纷|纷|脱|离|原|有|通|道|或|上|升|趋|势|线|走|出|了|快|速|上|扬|行|情|容|易|产|生|两|个|误|区|一|是|以|原|有|趋|势|线|为|参|照|实|质|就|是|坐|电|梯|个|股|强|势|两|至|三|根|中|大|阳|后|会|步|入|调|整|有|些|还|会|下|跌|回|原|趋|势|线|二|是|短|线|以|自|己|的|止|赢|为|目|标|比|如|说|8|10|等|等|致|使|吃|不|足|强|势|波|段|浪|费|了|强|势|机|会|和|资|金|使|用|效|率|r|r|在|江|恩|的|技|术|分|析|中|十|分|注|重|的|上|升|角|度|主|要|参|照|的|有|60|度|45|度|30|度|r60|度|以|上|上|升|角|度|的|一|般|持|有|为|主|不|具|备|做|波|段|的|能|力|也|就|是|以|前|是|准|备|放|在|80|讲|以|后|进|行|的|也|就|是|说|准|备|在|介|绍|完|基|本|的|技|术|分|析|之|后|但|大|盘|运|行|至|目|前|的|敏|感|阶|段|个|股|纷|纷|脱|离|原|有|通|道|或|上|升|趋|势|线|走|出|了|快|速|上|扬|行|情|容|易|产|生|两|个|误|区|一|是|以|原|有|趋|势|线|为|参|照|实|质|就|是|坐|电|梯|个|股|强|势|两|至|三|根|中|大|阳|后|会|步|入|调|整|有|些|还|会|下|跌|回|原|趋|势|线|二|是|短|线|以|自|己|的|止|赢|为|目|标|比|如|说|8|10|等|等|致|使|吃|不|足|强|势|波|段|浪|费|了|强|势|机|会|和|资|金|使|用|效|率|r|r|在|江|恩|的|技|术|分|析|中|十|分|注|重|的|上|升|角|度|主|要|参|照|的|有|60|度|45|度|30|度|r60|度|以|上|上|升|角|度|的|一|般|持|有|为|主|不|具|备|做|波|段|的|能|力|也|就|是|说|往|往|在|k|线|图|上|的|表|现|形|式|为|依|托|5|日|或|10|日|均|线|为|上|升|趋|势|线|上|扬|只|要|不|有|效|跌|破|上|升|趋|势|线|即|可|谨|慎|持|有|短|线|加|速|时|以|涨|停|方|式|突|破|次|日|列|分|时|均|线|止|赢|一|般|会|产|生|不|错|的|利|润|结|果|r|45|度|左|右|角|度|上|扬|的|一|般|为|个|股|通|道|运|行|的|常|态|总|体|强|势|为|主|下|轨|在|20|日|均|线|附|近|上|轨|为|两|个|重|要|高|点|的|连|线|在|回|落|至|通|道|下|轨|时|可|做|短|线|加|仓|操|作|上|行|至|通|道|上|轨|附|近|时|视|成|交|量|的|变|化|以|短|线|部|分|减|仓|为|主|个|别|的|沿|趋|势|线|上|扬|k|线|形|式|上|阴|阳|交|错|总|体|以|持|有|为|主|r|30|度|角|上|扬|的|在|趋|势|线|上|以|此|30|日|均|线|为|主|形|态|上|以|波|浪|上|扬|和|台|阶|上|扬|为|主|类|似|的|个|股|运|行|趋|势|在|操|作|上|参|照|通|道|上|轨|压|力|及|个|股|运|行|规|律|小|波|段|操|作|一|般|会|产|生|比|单|纯|持|股|更|多|的|收|益|效|果|r|r|在|上|述|角|度|运|行|的|个|股|会|随|着|阻|力|位|的|有|效|突|破|买|气|的|增|加|基|本|面|的|改|善|发|生|由|量|到|质|的|转|变|会|有|改|变|原|趋|势|线|运|行|的|走|势|出|现|这|就|是|变|线|小|角|度|的|会|向|大|角|度|变|线|强|势|的|个|股|还|有|抛|物|线|走|势|出|现|在|变|线|后|的|两|天|一|般|会|有|阳|线|越|来|越|大|6|日|bias|指|标|值|超|越|12|个|股|走|势|呈|加|速|形|态|这|个|时|候|一|般|就|是|列|分|时|均|线|止|赢|操|作|或|15|分|钟|线|止|赢|操|作|的|时|机|了|在|分|时|图|上|股|价|分|时|有|效|跌|破|分|时|均|线|同|时|不|能|有|效|站|上|或|15|分|钟|k|线|图|中|ma5|与|ma10|有|效|死|叉|时|减|磅|出|局|都|可|以|相|对|确|保|个|股|短|线|加|速|时|的|利|润|以|600210|为|例|上|述|短|线|操|作|提|示|卖|出|时|可|以|减|磅|在|8.33|元|至|8.40|元|附|近|r|也|有|一|些|大|角|度|上|扬|的|个|股|在|短|线|乖|离|率|增|大|时|离|中|期|均|线|较|远|会|有|一|定|回|档|中|期|均|线|的|要|求|往|往|在|个|股|运|行|的|中|位|产|生|一|定|的|震|荡|也|就|是|常|见|的|w|形|三|角|型|旗|型|矩|形|小|箱|体|的|整|理|形|态|之|后|按|低|一|个|角|度|的|趋|势|线|上|扬|目|前|部|分|基|金|重|仓|二|线|蓝|筹|运|行|的|就|是|类|似|走|势|可|按|中|小|角|度|上|扬|的|操|作|方|法|即|可|r|r|answer|先|收|藏|了|我|水|平|太|差|一|知|半|解|不|过|会|好|好|学|习|的|谢|谢|",
  "qid|qid_2285648232270123155|category|游|戏|手|机|游|戏|title|学|手|机|游|戏|开|发|有|前|途|吗|悦|成|学|校|好|么|现|在|网|络|上|关|于|就|业|关|于|手|desc|现|在|上|关|于|就|业|关|于|手|机|游|戏|开|发|方|面|的|信|息|好|像|很|火|啊|学|手|机|游|戏|开|发|到|底|有|前|途|没|有|一|个|叫|悦|成|的|手|机|方|面|的|公|司|经|常|在|眼|前|出|现|它|是|做|什|么|的|answer|以|前|是|每|个|人|都|有|一|部|手|机|现|在|是|每|个|人|基|本|都|有|一|部|智|能|手|机|手|机|市|场|会|越|来|越|开|阔|越|来|越|智|能|智|能|手|机|背|后|需|要|有|更|多|的|开|发|人|才|进|行|软|件|游|戏|开|发|才|凸|显|智|能|手|机|的|优|势|所|以|手|机|游|戏|开|发|行|业|前|途|无|量|不|容|置|疑|",
  "qid|qid_2648433068506518033|category|生|活|生|活|常|识|title|有|谁|知|道|妆|点|一|生|的|洗|发|水|去|屑|效|果|怎|么|样|desc|answer|我|用|过|他|家|产|品|效|果|还|不|错|",
  "qid|qid_6997163851468800167|category|电|子|数|码|手|机|通|讯|title|诺|基|亚|e63|最|低|电|量|指|示|空|格|还|是|一|格|是|不|是|没|电|无|法|开|机|的|时|候|摁|开|desc|是|不|是|没|电|无|法|开|机|的|时|候|摁|开|机|键|中|间|方|框|按|键|亮|但|无|法|开|机|answer|能|显|示|的|最|低|电|量|是|1|格|当|电|池|电|量|不|足|以|开|机|时|就|像|你|说|的|闪|一|下|就|会|变|黑|了|",
  "qid|qid_7369656993833514153|category|教|育|科|学|院|校|信|息|title|山|东|最|好|的|厨|师|学|校|desc|answer|山|东|新|东|方|烹|饪|学|校|吧|",
  "qid|qid_7320590726058808053|category|游|戏|网|络|游|戏|title|末|日|轻|装|套|3|章|会|涨|还|是|会|降|幅|度|大|不|大|懂|市|场|的|哥|哥|姐|姐|能|说|说|吗|desc|大|概|会|多|少|钱|一|套|谢|谢|了|哦|answer|肯|定|会|降|了|但|是|幅|度|不|会|太|大|３|章|开|了|末|日|轻|就|会|象|２|章|的|飞|龙|一|样|普|遍|大|家|都|去|搞|新|衣|服|",
  "qid|qid_3882306860161199895|category|娱|乐|博|彩|title|256|元|冲|14|场|火|锅|而|去|01|切|尔|西|vs|维|拉|3102|西|汉|姆|vs|雷|desc|01|vs|维|拉|31|r02|西|汉|姆|vs|雷|丁|31|r03|热|刺|vs|富|勒|姆|3|r04|伯|明|翰|vs|米|德|尔|31|r05|德|比|郡|vs|利|物|浦|0|r06|埃|弗|顿|vs|博|尔|顿|3|r07|桑|德|兰|vs|曼|联|0|r08|维|冈|vs|纽|卡|斯|3|r09|朴|茨|茅|vs|阿|森|纳|10|r10|曼|城|vs|布|莱|克|31|r11|切|尔|西|vs|纽|卡|斯|3|r12|热|刺|vs|雷|丁|3|r13|西|汉|姆|vs|曼|联|10|r14|埃|弗|顿|vs|阿|森|纳|31|r|answer|06|埃|弗|顿|vs|博|尔|顿|31|r|n08|维|冈|vs|纽|卡|斯|310|r|n10|曼|城|vs|布|莱|克|30|r|n|祝|你|好|运|r|n|",
  "qid|qid_6061260557426028915|category|游|戏|单|机|游|戏|反|恐|精|英|title|用|steam|打|1.6|进|有|cd|的|服|务|器|进|去|后|一|会|就|不|能|动|了|请|问|怎|desc|以|前|出|现|过|此|问|题|我|更|新|了|和|cs|的|版|本|后|正|常|今|天|cs|已|经|更|新|了|但|cd|没|发|现|有|更|新|我|的|是|4.31.0|如|是|cd|版|本|问|题|请|告|诉|我|哪|有|更|新|的|answer|cs.sga.com.cn|就|有|是|中|文|网|站|",
  "qid|qid_4899492148390597084|category|生|活|美|容|塑|身|title|怎|么|把|脸|洗|干|净|我|怎|么|觉|得|每|天|我|的|脸|洗|完|后|还|不|是|很|干|净|皮|肤|也|desc|我|怎|么|觉|得|每|天|我|的|脸|洗|完|后|还|不|是|很|干|净|r|r|r|也|不|是|很|好|r|r|怎|么|办|answer|最|好|用|洗|面|奶|带|沙|粒|的|对|油|性|皮|肤|还|有|干|燥|掉|皮|的|皮|肤|效|果|不|错|",
  "qid|qid_5401910109406337381|category|教|育|科|学|人|文|学|科|历|史|学|title|35|年|前|的|今|天|一|个|伟|人|过|世|了|请|您|在|此|说|一|句|心|里|话|desc|请|您|在|此|说|一|句|心|里|话|answer|1976|年|1|月|8|日|共|和|国|第|一|任|总|理|周|恩|来|在|北|京|逝|世|终|年|78|岁|",
  "qid|qid_6950055077148625728|category|烦|恼|恋|爱|title|初|恋|小|毛|孩|的|疑|惑|她|答|应|做|我|女|朋|友|却|不|愿|公|开|为|什|么|呢|是|因|为|怕|desc|她|答|应|做|我|女|却|不|愿|公|开|为|什|么|呢|是|因|为|怕|舆|论|还|是|不|相|信|我|answer|你|们|都|还|不|大|吧|r|n|她|既|然|答|应|你|了|饿|我|想|就|没|有|骗|你|的|必|要|吧|女|生|做|什|么|事|情|一|般|都|是|很|有|原|则|的|不|会|拿|自|己|的|幸|福|开|玩|笑|如|果|你|真|心|喜|欢|她|就|不|该|怀|疑|她|r|n|或|许|是|她|怕|知|道|人|太|多|后|父|母|和|老|师|也|容|易|只|这|样|对|你|俩|都|不|好|吧|r|n|不|过|这|种|问|题|你|最|好|和|她|心|平|气|和|的|好|好|谈|清|楚|沟|通|最|重|要|嘛|瞎|想|只|是|增|加|自|己|的|烦|恼|对|你|们|感|情|的|伤|害|不|会|有|任|何|好|处|的|",
  "qid|qid_3331013172446325757|category|体|育|运|动|足|球|title|请|输|入|您|的|问|题|山|东|鲁|能|泰|山|足|球|队|的|世|界|俱|乐|部|排|名|是|多|少|desc|山|东|鲁|能|泰|山|队|的|世|界|俱|乐|部|排|名|是|多|少|answer|前|一|百|名|里|面|没|有|追|上|皇|马|是|不|可|能|的|好|比|皇|马|是|硕|士|鲁|能|就|是|小|学|生|",
  "qid|qid_1103262911379902382|category|游|戏|网|络|游|戏|title|诛|仙|在|哪|里|领|激|活|码|啊|在|哪|里|领|激|活|码|啊|desc|在|哪|里|领|啊|answer|去|在|官|网|看|看|r|n|其|实|你|一|搜|索|那|个|链|接|就|会|出|现|的|o|o|",
  "qid|qid_6579774747972276861|category|健|康|人|体|常|识|title|脖|子|长|10cm|算|长|还|是|短|我|15|岁|男|生|身|高|167cm|脸|圆|desc|我|15|岁|男|生|身|高|167m|脸|圆|头|稍|大|身|材|比|较|瘦|皮|肤|白|请|问|我|适|合|穿|那|种|领|口|稍|大|的|圆|领|或|v|领|黑|色|t|恤|吗|我|个|人|比|较|喜|欢|深|色|的|衣|服|answer|有|点|儿|长|了|脖|子|长|的|男|生|穿|什|么|都|好|看|自|信|豪|迈|吧|",
  "qid|qid_4993337915700844386|category|健|康|肿|瘤|科|title|拉|肚|子|以|后|身|体|虚|弱|应|该|吃|些|什|么|东|西|调|养|有|什|么|应|该|注|意|的|不|desc|不|好|意|思|昨|天|好|像|吃|到|什|么|不|太|的|东|西|了|一|直|拉|肚|子|搞|的|我|睡|觉|也|睡|不|好|索|性|就|起|来|了|我|该|吃|些|什|么|东|西|调|理|下|answer|一|饮|食|治|疗|目|的|r|预|防|并|纠|正|水|及|电|解|质|平|衡|失|调|供|给|充|足|营|养|改|善|营|养|状|况|避|免|机|械|性|及|性|刺|激|使|肠|道|得|到|适|当|休|息|有|利|于|病|情|早|日|恢|复|r|r|二|饮|食|治|疗|原|则|r|一|怠|性|腹|泻|饮|食|治|疗|r1|急|性|期|禁|食|急|性|水|泻|期|需|暂|时|禁|食|使|肠|道|完|全|休|息|必|要|时|由|静|脉|输|液|以|防|失|水|过|多|而|脱|水|r2|清|淡|流|质|饮|食|不|需|禁|食|者|发|病|初|宜|给|清|淡|流|质|饮|食|如|蛋|白|水|果|汁|米|汤|薄|面|汤|等|以|咸|为|主|早|期|禁|牛|奶|蔗|糖|等|易|产|气|的|流|质|饮|食|有|些|患|者|对|牛|奶|不|适|应|服|牛|奶|后|常|加|重|腹|泻|r3|根|据|病|情|调|整|饮|食|排|便|次|数|减|少|症|状|缓|解|后|改|为|低|脂|流|质|饮|食|或|低|脂|少|渣|细|软|易|消|化|的|半|流|质|饮|食|如|大|米|粥|藕|粉|烂|面|条|面|片|等|r4|饮|食|选|择|腹|泻|基|本|停|止|后|可|供|给|低|脂|少|渣|半|流|质|饮|食|或|软|食|少|量|多|餐|以|利|于|消|化|如|面|条|粥|馒|头|烂|米|饭|瘦|肉|泥|等|仍|应|适|当|限|制|含|粗|纤|维|多|的|蔬|菜|水|果|等|以|后|逐|渐|过|渡|到|普|食|r5|补|充|维|生|素|注|意|复|合|维|生|素|b|和|维|生|素|c|补|充|如|鲜|桔|汁|果|汁|番|茄|汁|菜|汤|等|r6|饮|食|禁|忌|禁|酒|忌|肥|肉|坚|硬|及|含|粗|纤|维|多|的|蔬|菜|生|冷|瓜|果|油|脂|多|的|点|心|及|冷|饮|等|r|r|二|慢|性|腹|泻|饮|食|治|疗|r1|低|脂|少|渣|饮|食|每|天|脂|肪|40g|左|右|过|多|不|易|消|化|并|加|重|胃|肠|道|负|担|刺|激|胃|肠|蠕|动|加|重|腹|泻|故|植|物|油|也|应|限|制|并|注|意|烹|调|方|法|以|蒸|煮|氽|烩|烧|等|为|主|禁|用|油|煎|炸|爆|炒|滑|溜|等|可|用|食|物|有|瘦|肉|鸡|虾|鱼|豆|制|品|等|注|意|少|渣|粗|纤|维|多|的|食|物|能|刺|激|肠|蠕|动|使|腹|泻|加|重|当|腹|泻|次|数|多|时|最|好|暂|时|不|吃|或|尽|量|少|吃|蔬|菜|和|水|果|可|给|予|鲜|果|汁|番|茄|汁|以|补|充|维|生|素|少|渣|饮|食|可|减|少|肠|蠕|动|减|轻|腹|泻|故|宜|进|食|细|挂|面|粥|烂|饭|等|r2|高|蛋|白|高|热|能|慢|性|腹|泻|病|程|长|常|反|复|发|作|影|响|食|物|消|化|吸|收|并|造|成|体|内|贮|存|的|热|能|消|耗|为|改|善|营|养|状|况|应|给|予|高|蛋|白|高|热|能|饮|食|并|用|逐|渐|加|量|的|方|法|如|增|加|过|快|营|养|素|不|能|完|全|吸|收|反|而|可|能|加|重|胃|肠|道|负|担|可|供|给|蛋|白|质|loog|d|左|右|热|能|10．46|12，55mj|2500|3000kcal|r3|禁|忌|食|物|如|粗|粮|生|冷|瓜|果|冷|拌|莱|等|含|粗|纤|维|多|的|韭|菜|芹|菜|榨|菜|等|坚|硬|不|易|消|化|的|肉|类|如|火|腿|香|肠|腌|肉|等|刺|激|性|食|物|如|辣|椒|烈|酒|芥|末|辣|椒|粉|以|及|肥|肉|油|酥|点|心|等|高|脂|肪|食|物|r|r|更|多|参|考|r|",
  "qid|qid_1172162630992884010|category|生|活|保|健|养|生|title|生|活|饮|食|习|惯|对|人|体|健|康|的|影|响|饮|食|的|健|康|会|对|人|体|产|生|怎|样|的|影|响|desc|饮|食|的|会|对|人|体|产|生|怎|样|的|影|响|answer|现|在|的|人|工|作|压|力|都|很|大|不|按|时|吃|饭|的|大|有|人|在|暴|饮|暴|食|的|也|很|多|这|就|是|为|什|么|现|在|亚|健|康|的|人|群|巨|大|的|原|因|良|好|的|饮|食|习|惯|对|你|以|后|上|了|岁|数|会|有|很|大|帮|助|会|长|寿|的|",
  "qid|qid_744713671292039544|category|生|活|美|容|塑|身|减|肥|塑|身|title|求|推|荐|减|肥|药|各|位|泪|泪|泪|泪|desc|求|推|荐|药|各|位|泪|泪|泪|泪|answer|恩|我|最|近|也|在|减|肥|噢|给|我|的|食|谱|给|你|参|考|一|下|我|个|人|觉|得|很|有|用|n|早|上|一|杯|牛|奶|一|个|鸡|蛋|白|n|中|午|两|勺|饭|青|菜|汤|水|可|以|多|吃|但|是|不|要|撑|着|了|n|晚|上|一|个|苹|果|或|者|一|块|西|瓜|你|自|己|喜|欢|就|好|",
  "qid|qid_8294380531769628792|category|娱|乐|动|漫|title|大|家|帮|我|找|找|这|张|图|就|是|海|贼|王|里|的|银|狐|篇|里|的|一|集|路|飞|一|行|人|与|福|desc|就|是|里|的|银|狐|篇|里|的|一|集|路|飞|一|行|人|与|福|克|斯|用|迟|钝|光|线|的|那|个|比|赛|其|中|一|个|画|面|里|由|于|tv|制|作|问|题|而|出|现|了|两|个|索|罗|我|想|要|这|张|截|图|以|前|见|到|过|r|却|不|慎|丢|失|请|大|家|帮|忙|找|一|找|或|是|截|一|张|r|谢|谢|大|家|了|answer|海|贼|王|里|r|n|r|n|两|个|索|罗|",
  "qid|qid_2783837286197112772|category|游|戏|网|络|游|戏|跑|跑|卡|丁|车|title|进|入|跑|跑|卡|丁|车|时|为|什|么|会|出|现|错|误|报|告|每|当|进|入|跑|跑|就|出|现|一|个|错|误|报|desc|每|当|进|入|跑|跑|就|出|现|一|个|错|误|报|告|是|annot|execute|nmco|server|error|code|3|这|是|怎|么|会|事|answer|卸|载|掉|重|新|装|一|下|最|好|优|化|大|师|先|优|化|一|下|",
  "qid|qid_6893766316173749231|category|电|脑|网|络|title|视|频|下|载|在|哪|里|可|以|下|载|到|格|式|为|avi|的|视|频|啊|desc|在|哪|里|可|以|到|格|式|为|avi|的|视|频|啊|answer|你|得|用|个|转|码|器|因|为|avi|的|格|式|也|有|好|几|种|就|算|你|下|到|了|avi|格|式|的|也|不|一|定|你|的|手|机|就|能|看|r|n|我|用|的|转|码|器|还|可|以|你|试|下|r|n|下|载|地|址|r|n|电|影|的|网|站|你|可|以|上|迅|雷|上|下|r|n|转|码|前|按|他|的|提|示|做|个|配|置|",
  "qid|qid_4348258677050110984|category|健|康|人|体|常|识|title|09|年|查|出|肝|血|管|瘤|1.9|desc|09|年|查|出|血|管|瘤|1.9|2.0|今|年|长|大|了|2.5|2.0|肝|左|叶|平|时|左|上|腹|胀|不|舒|服|医|生|说|是|胃|胀|请|问|我|该|怎|么|办|answer|肝|血|管|瘤|是|一|种|常|见|的|肝|脏|良|性|肿|瘤|一|般|小|血|管|瘤|多|无|症|状|也|可|在|生|气|着|急|等|情|绪|变|化|饮|酒|劳|累|后|感|觉|肝|区|轻|微|不|适|并|伴|有|上|腹|饱|胀|嗳|气|等|症|状|r|n|肝|血|管|瘤|多|数|小|于|4|厘|米|成|人|后|一|般|不|继|续|生|长|对|健|康|无|妨|碍|亦|不|会|癌|变|所|以|一|般|不|必|治|疗|r|n|直|径|大|于|5|cm|者|为|巨|大|肝|血|管|瘤|有|潜|在|破|裂|出|血|的|危|险|需|要|治|疗|r|n|你|检|查|肝|血|管|瘤|1.9|2.0|今|年|长|大|了|2.5|2.0|虽|然|有|所|增|大|仍|属|于|小|血|管|瘤|目|前|不|需|处|理|但|要|定|期|半|年|或|一|年|复|查|r|n|左|上|腹|胀|不|舒|服|不|除|外|肝|血|管|瘤|所|致|的|可|能|可|先|按|胃|胀|治|疗|口|服|吗|丁|林|1|片|日|2|3|次|饭|前|半|小|时|服|r|n|",
  "qid|qid_6296306809888150600|category|商|业|理|财|股|票|title|恒|生|指|数|就|是|道|琼|斯|吗|desc|answer|恒|生|指|数|是|港|股|的|主|要|指|数|道|琼|斯|是|美|国|股|市|的|重|要|指|数|不|是|一|码|事|",
  "qid|qid_5565928575577908328|category|生|活|服|装|首|饰|title|南|京|哪|里|有|卖|钻|戒|拖|的|我|有|个|钻|戒|好|多|年|了|钻|石|很|亮|但|是|戒|指|非|常|的|污|desc|南|京|哪|里|有|卖|拖|的|我|有|个|钻|戒|好|多|年|了|钻|石|很|亮|但|是|戒|指|非|常|的|污|浊|了|想|更|换|钻|拖|所|以|想|问|下|南|京|哪|里|可|以|卖|钻|拖|还|有|大|概|费|用|是|多|少|呢|谢|谢|回|答|answer|不|做|广|告|也|不|搞|这|个|帮|你|在|网|上|查|了|下|据|说|中|山|门|地|矿|所|3|楼|有|一|家|不|过|不|清|楚|名|字|还|有|就|是|金|轮|新|天|地|负|一|楼|有|家|晓|何|珠|宝|制|作|坊|一|般|金|鹰|这|种|大|商|场|珠|宝|维|修|点|也|是|可|以|做|的|吧|应|该|",
  "qid|qid_3811040838065815077|category|生|活|起|名|title|除|了|夹|腋|下|还|有|没|有|测|量|宝|宝|体|温|时|间|短|宝|宝|也|不|会|抗|拒|的|测|量|办|法|desc|除|了|夹|腋|下|还|有|没|有|测|量|体|温|时|间|短|宝|宝|也|不|会|抗|拒|的|测|量|办|法|answer|宝|宝|量|体|温|除|了|肛|门|腋|下|还|有|颈|下|r|n|其|实|选|择|在|宝|宝|睡|觉|的|时|候|量|吧|",
  "qid|qid_4075857162645346487|category|教|育|科|学|职|业|教|育|title|如|果|觉|得|自|学|考|试|的|成|绩|与|自|己|所|做|的|试|卷|不|符|应|怎|么|处|理|是|山|东|自|desc|是|山|东|的|考|生|能|复|查|试|卷|吗|有|没|有|其|他|办|法|answer|不|能|复|查|试|卷|r|n|只|能|按|考|试|机|构|规|定|的|时|间|向|其|提|出|复|核|申|请|由|考|办|工|作|人|员|为|你|复|核|试|卷|",
  "qid|qid_7998567165628609834|category|电|脑|网|络|软|件|网|络|软|件|title|bt|下|载|问|题|是|不|是|下|载|与|上|传|的|速|度|不|快|就|不|太|伤|硬|盘|我|bt|的|下|载|desc|是|不|是|下|载|与|上|传|的|速|度|不|快|就|不|太|伤|我|bt|的|下|载|速|度|和|上|传|速|度|都|在|50kb|s|这|样|的|速|度|伤|硬|盘|吗|answer|因|为|硬|盘|有|一|个|规|定|的|擦|写|速|度|如|果|不|超|过|这|个|速|度|话|对|硬|盘|的|伤|害|是|很|小|的|不|会|影|响|到|硬|盘|寿|命|你|的|速|度|在|100k|左|右|对|你|的|硬|盘|伤|害|不|大|所|以|不|用|担|心|",
  "qid|qid_8426154873806908170|category|娱|乐|电|视|title|想|着|买|款|节|能|的|液|晶|电|视|用|居|家|过|日|子|我|本|着|能|省|则|省|的|原|则|买|液|desc|居|家|过|日|子|我|本|着|能|省|则|省|的|原|则|买|液|晶|想|选|款|节|能|的|听|朋|友|说|台|湾|奇|美|的|很|不|错|是|吗|answer|楼|主|免|了|吧|台|湾|货|可|是|出|了|名|的|低|端|啊|质|量|次|买|回|家|做|啥|啊|再|说|了|又|有|哪|款|液|晶|电|视|不|节|能|啊|",
  "qid|qid_965145238873261268|category|游|戏|完|美|游|戏|诛|仙|title|我|的|5173|商|品|通|过|核|审|为|什|么|不|能|上|架|我|卖|个|诛|仙|账|号|上|面|说|发|布|desc|我|卖|个|账|号|上|面|说|发|布|完|成|为|什|么|不|能|上|架|一|点|击|上|架|就|出|现|商|品|0|比|上|架|成|功|1|笔|上|架|失|败|answer|验|证|信|息|失|败|了|5173|会|自|己|登|陆|下|你|帐|号|看|邮|箱|以|及|其|他|你|所|说|的|信|息|是|否|正|确|你|可|一|看|下|处|理|结|果|",
  "qid|qid_3487350800464607887|category|电|脑|网|络|软|件|多|媒|体|软|件|title|flash|中|的|雪|花|飘|落|怎|么|制|作|desc|answer|简|单|的|方|法|r|n|建|立|一|个|引|导|层|然|后|在|引|导|层|画|曲|线|这|条|线|就|是|将|来|你|要|让|雪|花|飘|落|的|路|径|r|n|然|后|新|建|一|层|把|其|属|性|设|置|为|被|引|导|层|r|n|之|后|在|这|层|画|一|片|雪|花|有|现|成|的|最|好|然|后|作|一|个|渐|变|动|画|由|上|到|下|使|其|能|够|依|附|于|引|导|线|上|r|n|再|后|浏|览|这|就|是|一|片|雪|花|飘|落|的|动|画|了|r|n|r|n|如|果|要|做|出|来|下|大|雪|的|场|景|ctrl|f8|新|建|一|个|影|片|剪|辑|把|刚|才|作|好|的|针|复|制|近|来|多|建|立|几|层|让|雪|在|不|同|时|间|出|现|应|该|知|道|怎|么|做|吧|回|到|场|景|1|在|库|里|找|到|你|刚|才|做|的|影|片|剪|辑|打|开|库|的|快|接|键|ctrl|l|复|制|2|个|拖|动|到|场|景|1|里|面|来|r|n|r|n|复|制|出|来|的|影|片|剪|辑|分|别|调|整|大|小|和|透|明|度|让|场|景|有|点|景|深|之|后|就|ok|了|r|n|r|n|这|个|方|法|比|较|直|接|一|点|如|果|要|用|as|编|写|的|话|恐|怕|对|新|手|来|说|是|有|点|麻|烦|不|过|你|可|以|去|闪|客|学|堂|里|看|看|或|许|有|对|你|有|帮|助|的|文|章|或|者|素|材|还|有|灵|感|r|n|r|n|歪|歪|",
  "qid|qid_2029610074590556760|category|教|育|科|学|人|文|学|科|历|史|学|title|我|们|现|在|都|是|中|国|人|了|华|盛|顿|邮|报|备|受|欢|迎|的|专|栏|作|家|dav|desc|我|们|现|在|都|是|中|国|人|了|华|盛|顿|邮|报|备|受|欢|迎|的|专|栏|david|ignatius|写|道|我|们|有|一|个|名|义|上|的|资|本|主|义|经|济|但|当|它|遇|到|麻|烦|时|我|们|就|不|信|任|不|受|约|束|的|私|人|市|场|中|国|人|正|变|得|更|像|我|们|但|同|时|我|们|也|正|变|得|更|像|他|们|r|美|国|都|学|了|中|国|的|哪|些|东|西|answer|学|古|代|的|中|国|儒|家|文|化|道|经|孙|子|兵|法|r|n|r|n|学|现|在|的|中|国|特|色|社|会|主|义|市|场|经|济|政|府|监|管|比|较|强|半|计|划|半|市|场|",
  "qid|qid_3577367020099281567|category|健|康|精|神|心|理|科|title|追|求|物|质|生|活|和|追|求|精|神|生|活|矛|盾|么|常|听|人|们|说|物|质|生|活|过|于|丰|富|人|desc|常|听|人|们|说|物|质|过|于|丰|富|人|就|不|去|追|求|精|神|生|活|的|更|高|层|次|了|宗|教|中|的|苦|行|者|放|弃|了|舒|适|的|物|质|生|活|甚|至|是|最|基|本|的|物|质|需|要|去|追|求|一|个|纯|净|的|精|神|世|界|那|么|追|求|物|质|生|活|和|追|求|精|神|生|活|真|的|相|互|矛|盾|么|如|果|说|人|是|精|神|的|载|体|但|是|从|客|观|上|说|人|又|必|须|靠|物|质|来|支|持|但|是|为|什|么|通|常|时|候|高|层|的|物|质|生|活|和|高|层|的|精|神|生|活|不|可|兼|得|呢|answer|个|人|意|见|仅|供|参|考|r|n|常|听|人|们|说|物|质|生|活|过|于|丰|富|人|就|不|去|追|求|精|神|生|活|的|更|高|层|次|了|r|n|有|一|种|原|因|是|人|的|内|在|品|质|的|问|题|黄|金|与|泥|土|的|区|别|是|很|大|的|r|n|第|二|欲|望|的|满|足|是|无|止|境|的|一|个|欲|望|的|满|足|另|一|个|欲|望|早|已|迫|不|及|待|的|需|要|满|足|绵|延|不|绝|无|止|无|境|一|般|的|人|是|无|法|摆|脱|这|固|定|的|轮|回|r|n|第|三|想|想|吸|毒|者|他|们|的|心|思|会|放|在|哪|里|呢|当|然|这|例|子|是|有|点|极|端|r|n|第|四|人|在|对|世|界|与|人|生|没|有|一|个|正|确|性|的|认|识|之|前|什|么|是|好|什|么|是|坏|是|分|不|清|的|很|多|的|人|只|是|看|到|眼|前|的|利|益|眼|前|的|享|受|也|因|此|为|眼|前|的|利|益|忙|碌|奔|命|竭|尽|所|能|r|n|第|五|正|是|世|界|观|与|人|生|观|不|同|有|些|人|只|相|信|只|有|今|生|而|有|些|人|相|信|有|来|生|等|等|有|着|完|全|相|反|的|观|念|r|n|r|n|",
  "qid|qid_2489526911613624875|category|健|康|内|科|内|分|泌|科|title|糖|尿|病|的|食|疗|法|desc|answer|r|糖|尿|病|的|真|相|r|由|于|祖|国|医|学|研|究|的|滞|后|加|之|一|线|的|医|疗|工|作|者|对|理|论|上|的|了|解|不|是|很|多|所|以|很|多|患|者|并|不|知|道|糖|尿|病|到|底|是|如|何|产|生|的|为|什|么|是|不|能|治|愈|的|终|身|疾|病|r|这|里|我|只|针|对|2|型|糖|尿|病|做|一|些|解|释|但|也|不|代|表|全|部|因|为|还|有|许|多|事|情|没|有|搞|清|楚|但|以|现|有|的|科|研|成|果|大|部|分|的|2|型|糖|尿|病|患|者|是|有|机|会|真|正|的|和|病|患|剥|离|的|r|我|在|这|里|分|三|个|层|面|来|解|释|第|一|个|就|是|我|们|众|所|周|知|的|一|种|解|释|在|国|内|这|种|解|释|还|非|常|广|并|且|大|部|分|的|治|疗|方|法|都|是|针|对|此|方|面|就|是|直|接|认|为|是|胰|岛|出|了|问|题|因|为|人|体|内|的|血|糖|只|有|通|过|胰|岛|素|的|控|制|才|能|发|挥|作|用|而|胰|岛|分|泌|的|胰|岛|素|少|或|者|质|量|不|好|不|能|很|好|的|被|利|用|造|成|血|糖|不|受|控|制|所|以|就|得|了|糖|尿|病|所|以|医|生|就|会|使|用|一|些|药|物|来|刺|激|你|的|胰|岛|分|泌|更|多|的|胰|岛|素|来|满|足|血|糖|的|需|要|但|胰|岛|毕|竟|是|人|体|的|器|官|经|常|性|的|用|药|物|来|刺|激|就|会|慢|慢|让|它|的|功|能|消|失|怠|尽|不|能|再|分|泌|了|所|以|这|时|候|医|生|会|建|议|你|直|接|打|胰|岛|素|来|代|替|自|身|分|泌|这|时|你|的|胰|岛|已|经|完|蛋|了|r|这|时|也|有|医|疗|工|作|者|提|出|让|患|者|在|病|患|早|期|就|先|打|胰|岛|素|这|样|可|以|让|胰|岛|得|到|充|分|的|休|息|来|恢|复|分|泌|功|能|但|有|发|现|打|胰|岛|素|的|量|很|难|控|制|容|易|造|成|低|血|糖|更|加|危|险|另|一|方|面|也|不|能|控|制|到|底|让|胰|岛|休|息|多|长|时|间|合|适|时|间|长|了|也|一|样|会|使|它|失|去|功|能|所|以|这|种|方|法|也|没|有|得|到|普|及|不|管|是|刺|激|胰|岛|还|是|替|代|它|对|人|的|身|体|都|是|非|常|有|害|的|而|且|过|程|烦|琐|稍|不|注|意|就|可|能|出|现|更|大|的|问|题|所|以|我|们|都|不|建|议|这|样|的|治|疗|方|法|r|还|有|一|种|药|物|是|直|接|作|用|于|你|吃|的|食|物|让|你|产|生|的|糖|份|直|接|减|少|结|果|是|营|养|不|良|的|现|象|开|始|出|现|抵|抗|力|免|疫|力|都|会|下|降|身|体|更|容|易|受|损|另|外|糖|本|是|人|不|可|缺|少|的|东|西|少|了|也|不|行|所|以|这|种|方|法|也|不|被|看|好|r|但|在|这|个|治|疗|过|程|中|人|们|发|现|在|血|糖|被|控|制|的|情|况|下|并|发|症|还|是|会|来|而|且|危|害|极|大|这|就|是|医|生|都|会|告|诉|你|真|正|可|怕|的|是|并|发|症|的|原|因|因|为|你|根|本|阻|止|不|了|r|更|可|怕|的|是|我|国|的|医|疗|工|作|者|们|在|治|疗|方|法|上|还|停|留|在|这|个|层|面|r|插|一|句|现|在|的|医|疗|工|作|者|大|致|分|为|两|种|一|种|就|是|你|在|里|经|常|见|到|的|医|生|或|护|士|他|们|是|现|有|技|术|和|药|品|的|使|用|者|所|以|他|们|的|认|识|也|只|停|留|在|这|个|层|面|他|们|没|有|自|己|独|创|的|治|疗|方|法|他|们|甚|至|会|直|接|让|你|要|先|树|立|一|种|糖|尿|病|绝|对|不|能|治|愈|的|思|想|然|后|再|调|理|还|会|告|诉|你|所|有|说|能|治|好|病|的|人|都|是|骗|子|但|他|们|又|从|不|会|说|他|们|会|对|你|的|病|没|有|办|法|而|他|们|自|己|又|完|全|知|道|自|己|是|无|能|为|力|的|种|种|的|矛|盾|造|就|了|我|们|现|在|已|知|的|医|疗|群|体|也|使|社|会|对|他|们|的|信|心|和|信|任|在|大|幅|下|降|所|以|有|时|候|先|要|问|问|医|生|或|要|听|医|生|的|话|并|不|是|完|全|正|确|的|要|慎|重|挑|选|医|生|另|一|种|就|是|同|时|在|做|科|研|工|作|的|医|学|专|家|研|究|各|种|疾|病|的|新|的|治|疗|方|法|但|在|国|内|由|于|很|多|信|息|和|设|备|都|不|足|加|之|大|多|人|都|以|药|治|的|角|度|寻|求|解|决|的|方|案|所|以|在|慢|性|病|领|域|已|经|很|久|没|有|什|么|突|破|性|的|东|西|了|而|他|们|有|很|多|处|于|闭|关|自|守|的|状|态|特|别|是|中|西|医|之|间|的|难|以|融|合|使|很|多|国|外|已|成|型|的|概|念|或|发|现|很|难|利|用|第|一|种|基|本|是|授|命|于|第|二|种|而|我|们|受|的|教|育|都|是|要|绝|对|听|从|医|生|的|结|果|可|想|而|之|所|有|的|人|都|说|治|不|好|又|都|不|知|道|为|什|么|r|继|续|说|病|后|来|人|们|又|发|现|了|一|个|不|能|解|释|的|现|象|就|是|很|多|糖|尿|病|患|者|的|胰|岛|是|正|常|的|这|就|和|原|来|的|认|识|发|生|了|冲|突|看|来|还|有|我|们|不|知|道|的|东|西|在|起|作|用|到|底|是|什|么|呢|大|约|前|两|三|年|的|时|候|有|人|从|国|外|带|来|了|最|新|的|概|念|而|这|种|概|念|在|国|外|已|有|多|年|的|历|史|并|且|有|了|相|应|的|产|品|胰|岛|素|抵|抗|的|说|法|开|始|得|到|人|们|的|了|解|原|来|在|我|们|的|血|液|的|细|胞|当|中|有|一|种|蛋|白|质|而|胰|岛|素|只|有|和|它|相|结|合|才|能|起|到|控|制|血|糖|的|作|用|我|们|把|这|种|蛋|白|称|作|受|体|如|果|受|体|不|接|受|胰|岛|素|的|话|你|就|是|分|泌|多|少|也|是|没|有|用|处|来|自|体|外|的|胰|岛|素|一|样|没|有|作|用|我|们|把|这|种|不|接|受|的|现|象|称|作|胰|岛|素|抵|抗|而|胰|岛|是|它|的|第|一|个|受|害|者|可|以|说|我|们|都|冤|枉|了|胰|岛|它|为|了|保|护|我|们|的|身|体|不|断|的|分|泌|胰|岛|素|以|为|可|以|解|决|但|结|局|只|有|自|己|的|气|绝|身|亡|还|要|在|医|生|的|无|知|下|背|上|所|有|的|骂|名|并|且|被|他|们|毁|掉|真|的|很|可|怜|大|家|千|万|不|要|在|责|怪|我|们|的|胰|岛|了|r|国|外|针|对|此|现|象|开|发|出|胰|岛|素|增|敏|剂|增|强|受|体|的|亲|和|力|以|使|它|更|好|的|和|胰|岛|素|相|结|合|从|根|源|上|解|决|问|题|受|体|的|数|目|是|相|对|恒|定|的|由|于|它|周|围|环|境|的|恶|化|使|它|生|了|锈|还|有|一|些|发|生|萎|缩|不|健|全|等|现|象|使|它|不|与|胰|岛|素|结|合|而|现|有|的|增|敏|剂|只|能|解|决|20|的|问|题|在|胰|岛|素|的|协|助|下|起|到|了|到|此|为|止|已|知|范|围|内|最|有|效|的|作|用|但|同|样|对|并|发|症|只|有|延|缓|的|作|用|至|于|如|何|起|到|100|的|作|用|还|没|有|发|现|而|且|因|为|糖|尿|病|往|往|不|能|早|期|发|现|在|你|发|现|时|并|发|症|的|种|子|也|已|经|种|了|下|去|这|样|即|使|你|能|恢|复|所|有|的|受|体|也|只|是|达|到|一|个|降|血|糖|的|作|用|而|我|们|真|正|需|要|解|决|的|是|阻|止|并|发|症|的|出|现|所|以|理|论|上|来|说|全|球|范|围|内|还|不|能|治|好|糖|尿|病|而|这|种|认|知|几|乎|存|在|于|所|有|的|慢|性|病|领|域|内|r|而|我|国|的|医|疗|科|研|工|作|者|对|这|一|层|的|了|解|还|只|停|留|在|知|道|有|这|么|一|说|的|程|度|上|其|它|都|不|明|白|所|以|也|没|有|对|应|的|治|疗|方|法|而|增|敏|剂|对|于|国|内|人|群|来|说|过|于|昂|贵|而|且|限|制|人|群|也|很|多|医|生|很|少|推|荐|并|为|得|到|广|泛|的|认|可|和|普|及|r|这|两|个|层|面|是|糖|尿|病|领|域|内|的|现|状|现|在|我|们|说|一|下|为|什|么|此|方|面|的|研|究|如|此|困|难|和|缓|慢|前|面|我|提|示|了|一|点|现|在|的|医|疗|研|究|者|的|研|究|角|度|都|是|直|接|针|对|病|患|通|过|什|么|技|术|或|产|品|来|直|接|消|除|它|比|如|对|于|胰|岛|素|抵|抗|就|是|这|样|而|对|引|起|它|的|诱|因|不|屑|一|顾|胰|岛|素|抵|抗|的|出|现|完|全|是|因|为|人|体|内|环|境|的|不|同|程|度|的|污|染|太|多|的|身|体|不|需|要|的|杂|质|环|绕|在|受|体|的|周|围|受|体|怎|么|可|能|还|正|常|我|们|提|出|的|解|决|方|案|有|三|个|一|是|清|洁|受|体|所|处|的|环|境|二|是|恢|复|那|些|不|正|常|的|受|体|的|功|能|三|是|促|生|新|受|体|来|替|换|已|经|不|可|恢|复|的|这|样|解|决|的|结|果|让|我|们|消|除|胰|岛|素|抵|抗|的|效|率|达|到|了|80|而|省|下|的|20|我|们|发|现|它|们|已|经|不|足|以|引|起|并|发|症|所|以|我|们|有|了|成|功|治|愈|的|记|录|这|就|是|我|说|的|第|三|个|层|面|r|当|然|我|们|也|还|有|很|多|解|决|不|了|也|是|因|为|我|们|还|没|有|搞|透|彻|比|如|1|型|比|如|发|现|太|晚|比|如|引|发|胰|岛|素|抵|抗|的|另|一|个|原|因|有|些|和|受|体|很|相|似|的|物|质|半|路|截|走|了|胰|岛|素|等|等|r|写|这|些|的|目|的|是|希|望|大|家|即|使|不|能|接|受|我|们|的|结|论|也|要|对|整|个|治|疗|糖|尿|病|的|技|术|和|产|品|有|一|个|充|分|的|了|解|也|承|认|市|场|上|很|多|我|们|不|能|信|任|的|东|西|加|之|医|生|极|力|宣|扬|糖|尿|病|不|可|治|愈|的|概|念|使|很|多|病|友|丧|失|了|信|心|其|实|完|全|不|必|如|此|前|期|预|防|后|期|调|理|都|不|失|为|很|好|的|方|法|并|且|糖|尿|病|患|者|有|很|多|机|会|活|到|人|的|正|常|寿|命|这|个|过|程|也|不|是|我|们|想|象|的|那|样|痛|苦|毕|竟|我|们|真|正|想|要|的|还|是|这|个|r|我|们|的|技|术|现|在|通|过|三|个|途|径|来|解|决|问|题|胰|岛|功|能|的|恢|复|胰|岛|素|抵|抗|的|消|除|并|发|症|的|防|治|更|全|面|有|效|的|解|决|问|题|而|不|是|针|对|某|一|项|r|蜜|蜂|体|内|提|取|的|类|胰|岛|素|成|分|即|使|单|独|拿|出|来|就|已|经|可|以|在|某|种|程|度|上|替|代|打|胰|岛|素|而|且|同|时|还|有|对|胰|岛|的|恢|复|作|用|而|这|个|成|分|仅|仅|是|其|中|一|个|重|要|的|成|分|蜂|毒|能|带|给|我|们|的|还|有|很|多|在|这|里|不|一|一|列|举|更|详|细|的|内|容|有|更|针|对|性|的|解|释|r|所|以|请|大|家|清|楚|一|点|糖|尿|病|不|能|治|愈|的|说|法|是|片|面|的|是|受|很|多|限|制|的|包|括|我|们|面|对|强|大|的|反|对|论|调|而|难|以|推|广|医|生|在|患|者|的|思|维|里|种|下|了|根|深|蒂|固|的|不|可|能|而|且|在|无|限|的|延|续|我|并|不|是|完|全|怪|医|生|在|他|们|的|所|知|领|域|内|也|只|能|那|样|说|但|我|不|希|望|患|者|因|此|放|弃|努|力|更|不|希|望|被|虚|假|的|广|告|吓|的|因|噎|废|食|医|学|在|不|断|进|步|中|但|从|开|发|到|推|广|的|路|更|加|艰|难|任|何|的|新|生|事|物|都|要|经|过|重|重|阻|力|才|会|得|到|广|泛|的|认|可|希|望|大|家|能|与|我|们|共|同|把|属|于|我|们|祖|国|的|光|荣|的|结|晶|推|广|开|来|这|也|许|是|中|医|在|世|界|范|围|内|得|到|认|可|的|最|好|机|会|r|",
  "qid|qid_4941995463447011765|category|烦|恼|两|性|问|题|title|把|女|友|分|泌|出|的|爱|液|吃|进|肚|子|里|要|不|要|紧|啊|如|题|desc|如|题|answer|当|然|不|要|紧|r|n|r|n|爱|液|都|是|人|体|性|器|官|和|腺|体|的|分|泌|物|如|果|没|有|疾|病|是|没|有|致|病|菌|的|但|是|女|性|阴|道|内|是|有|一|些|正|常|菌|群|r|n|r|n|",
  "qid|qid_7288784539564051995|category|游|戏|网|络|游|戏|title|为|什|么|我|领|不|到|任|务|我|的|等|级|已|经|达|到|任|务|提|示|的|要|求|但|当|去|找|任|务|n|desc|我|的|等|级|已|经|达|到|任|务|提|示|的|要|求|但|当|去|找|任|务|np|时|他|们|竟|然|不|理|我|他|们|头|上|的|任|务|标|志|没|有|显|示|answer|不|可|能|的|如|果|不|行|去|官|方|论|坛|发|帖|子|",
  "qid|qid_8281396420932914288|category|娱|乐|星|座|运|势|title|1983|年|2|月|27|日|下|午|3|点|18|分|出|生|的|女|孩|子|命|运|如|何|请|大|师|帮|desc|请|大|师|帮|帮|忙|看|看|重|点|帮|忙|看|看|工|作|情|况|谢|谢|了|answer|看|了|你|的|本|命|盘|如|果|你|的|阳|历|日|期|和|出|生|时|间|都|没|有|错|的|话|r|n|r|n|那|么|我|可|以|断|定|你|是|一|个|很|有|潜|力|的|人|你|早|年|有|些|辛|苦|中|年|发|迹|晚|年|大|好|在|你|的|生|命|中|可|能|一|路|走|来|都|是|充|满|挑|战|的|但|是|正|因|为|充|满|挑|战|所|以|能|激|发|生|命|中|最|大|的|潜|能|r|n10|宫|主|合|火|三|合|木|其|实|你|的|事|业|和|事|业|运|相|当|好|四|十|岁|以|后|应|该|会|有|一|番|作|为|一|句|话|前|途|无|量|你|与|异|性|相|处|时|通|常|会|感|受|到|压|力|感|情|上|有|些|郁|闷|因|为|你|无|论|在|婚|姻|家|庭|爱|情|财|务|方|面|都|会|出|现|问|题|请|问|一|下|你|的|父|母|是|不|是|经|常|吵|架|另|外|你|有|很|好|的|创|业|能|力|",
  "qid|qid_4032135752540622525|category|商|业|理|财|股|票|title|请|教|老|师|000858|有|什|么|利|空|跌|这|么|多|支|撑|位|是|多|少|谢|谢|desc|answer|000858|五|粮|液|基|本|面|优|良|隶|属|大|盘|绩|优|成|长|股|机|构|筹|码|锁|定|良|好|短|期|调|整|结|束|后|有|望|延|续|反|弹|运|用|黄|金|率|0.618|分|析|短|期|技|术|压|力|在|33.80|元|一|线|操|作|策|略|建|议|持|股|至|上|述|区|域|短|期|出|货|祝|好|运|",
  "qid|qid_8738478348727306306|category|游|戏|网|络|游|戏|title|怎|么|电|信|2|不|能|进|啊|老|是|提|示|稍|候|我|都|等|了|3|个|小|时|了|desc|answer|网|通|的|就|进|网|通|的|电|信|的|进|电|信|的|每|周|三|9|30|12|30|服|务|器|例|行|维|护|这|个|游|戏|里|有|提|示|",
  "qid|qid_2662891279005095757|category|体|育|运|动|篮|球|title|火|箭|今|年|能|否|进|入|季|后|赛|第|2|轮|desc|answer|难|啊|现|在|连|进|季|后|赛|都|很|悬|啊|但|还|是|支|持|",
  "qid|qid_5110465507302901369|category|生|活|美|食|烹|饪|烹|饪|方|法|title|我|要|学|做|四|川|水|豆|鼓|请|高|手|教|我|说|详|细|点|谢|谢|desc|我|要|学|做|四|川|豆|鼓|请|高|手|教|我|说|详|细|点|谢|谢|answer|正|规|的|四|川|水|豆|豉|到|底|怎|样|做|一|直|不|得|而|知|后|来|看|了|本|川|菜|制|作|的|书|籍|讲|水|豆|豉|的|作|法|基|本|与|我|婆|婆|的|一|样|先|把|黄|豆|泡|了|煮|熟|然|后|用|棉|被|保|温|捂|两|天|再|加|姜|末|罗|卜|干|油|辣|子|盐|等|即|成|最|近|几|年|按|照|书|上|的|方|法|我|家|几|乎|年|年|冬|天|都|要|自|已|做|水|豆|豉|但|是|多|数|时|候|都|不|成|功|不|是|捂|出|来|发|酸|就|是|发|苦|这|几|年|倒|掉|没|捂|成|功|的|豆|子|不|知|有|多|少|斤|谁|敢|吃|呀|r|我|们|就|磨|既|然|用|被|子|捂|肯|定|就|是|让|它|发|酵|发|酵|有|两|个|条|件|不|可|缺|少|第|一|要|菌|种|第|二|要|适|合|的|温|度|我|们|一|没|菌|种|二|不|知|该|什|么|温|度|捂|出|来|的|豆|鼓|谁|知|道|用|什|么|菌|发|的|酵|敢|随|便|吃|吗|再|说|即|使|菌|种|对|了|不|知|道|合|适|的|温|度|捂|多|久|用|多|少|层|棉|被|都|是|瞎|碰|当|然|不|能|次|次|成|功|联|想|到|馒|头|的|做|法|先|要|用|酵|母|菌|和|面|在|温|暖|的|室|温|下|才|能|发|酵|成|功|肯|定|水|豆|豉|也|应|该|有|的|做|法|但|菌|种|和|温|度|是|有|侍|解|决|的|关|键|r|弟|弟|从|日|本|回|来|带|回|一|种|叫|纳|豆|的|食|品|尝|了|后|感|觉|与|我|们|四|川|的|水|豆|鼓|差|不|多|就|差|姜|末|罗|卜|干|辣|椒|油|听|说|日|本|已|工|厂|化|生|产|这|种|纳|豆|看|来|日|本|一|定|有|这|种|纳|豆|的|菌|种|用|科|学|方|法|控|制|温|度|发|酵|否|则|不|可|能|工|厂|化|生|产|标|淮|味|道|的|纳|豆|听|弟|弟|说|在|日|本|納|豆|成|为|一|种|传|统|的|营|养|食|品|几|乎|家|家|户|户|都|制|作|每|日|食|用|r|我|们|中|国|四|川|的|水|豆|豉|弄|了|半|天|日|本|也|有|甚|至|还|用|科|学|的|方|法|家|家|户|户|都|能|自|制|我|们|哪|天|也|掌|握|了|科|学|做|水|豆|鼓|的|方|法|多|好|可|以|方|便|做|出|标|准|味|道|的|正|宗|水|豆|鼓|再|也|不|用|去|瞎|捣|鼓|r|机|会|来|了|我|终|于|找|到|了|纳|豆|的|制|作|方|法|经|过|自|已|的|制|作|经|验|把|简|单|的|四|川|水|豆|鼓|自|制|方|法|告|诉|大|家|r1|到|伊|滕|洋|华|堂|去|买|日|本|纳|豆|成|都|的|伊|滕|锦|华|店|有|卖|在|二|环|路|万|达|广|场|买|最|便|宜|的|因|为|作|菌|种|用|只|需|一|点|点|大|约|l4|元|两|小|盒|买|回|放|冰|箱|保|存|待|用|r2|到|超|市|里|买|黄|豆|约|5|元|一|斤|一|次|做|一|斤|3|口|之|家|可|吃|好|多|天|的|r3|黄|豆|用|温|水|泡|12|24|小|时|r|r4|用|高|压|锅|煮|黄|豆|注|意|水|要|少|放|高|压|锅|上|汽|压|阀|后|改|用|小|火|保|持|30|45|分|钟|如|压|力|太|大|可|把|火|关|熄|待|会|再|打|开|火|煮|目|的|是|把|黄|豆|煮|软|用|手|指|轻|压|即|成|粉|碎|末|状|最|佳|如|果|黄|豆|煮|的|火|候|不|够|最|后|做|成|的|水|豆|鼓|口|感|一|定|不|好|吃|r5|用|漏|勺|沥|干|黄|豆|水|份|稍|冷|却|一|会|估|计|温|度|为|50|度|时|把|黄|豆|装|进|有|盖|的|容|器|中|注|意|所|有|东|西|一|定|要|保|证|干|净|无|油|怕|被|杂|菌|污|染|黄|豆|的|温|度|一|定|不|能|太|高|超|过|70|度|纳|豆|菌|会|被|烫|死|太|低|不|易|发|酵|50|度|左|右|正|好|加|入|纳|豆|菌|r6|把|买|的|日|本|纳|豆|从|冰|箱|里|拿|出|打|开|小|盒|取|出|里|面|的|两|包|小|调|料|不|用|把|納|豆|作|为|菌|种|放|入|煮|好|的|约|50|度|温|度|的|黄|豆|中|用|筷|子|和|匀|一|般|500|克|黄|豆|放|20|30|克|菌|种|正|好|一|小|盒|纳|豆|重|20|多|克|r7|准|备|一|纸|箱|一|电|热|毯|暖|脚|电|毯|最|好|小|棉|被|在|纸|箱|内|用|电|热|毯|棉|被|做|一|窝|把|和|好|菌|种|的|黄|豆|装|容|器|内|趁|热|放|内|窝|内|盖|上|箱|盖|打|开|电|热|毯|低|温|档|即|可|注|意|温|度|不|可|太|高|保|持|在|手|感|窝|内|暖|暖|即|够|了|不|要|烫|手|箱|子|上|面|不|用|捂|死|了|纸|箱|盖|上|再|搭|个|垫|子|什|么|的|就|可|以|让|它|稍|透|点|气|因|为|水|豆|鼓|发|酵|需|要|空|气|和|一|定|的|湿|度|最|好|选|用|盖|上|有|小|孔|的|容|器|保|持|通|气|就|可|以|r8|黄|豆|在|纸|箱|窝|里|保|温|发|酵|12|14|小|时|就|好|了|正|好|一|夜|取|出|来|用|筷|子|挑|起|看|看|起|大|量|的|丝|丝|说|明|成|功|尝|一|颗|不|苦|不|酸|没|任|何|怪|味|ok|r9|将|发|酵|成|功|的|水|豆|豉|拌|上|姜|末|罗|卜|干|颗|颗|盐|油|辣|椒|调|合|适|自|己|的|口|味|后|用|食|品|袋|分|装|成|一|小|袋|一|小|袋|的|估|计|每|袋|份|量|够|自|己|一|家|人|吃|1|2|天|的|把|装|好|袋|的|水|豆|鼓|放|入|冰|箱|的|冷|冻|室|冻|成|冰|状|保|存|吃|时|头|天|从|冷|冻|室|拿|出|一|袋|自|然|解|冻|后|第|二|天|再|吃|就|是|可|口|的|四|川|水|豆|豉|了|r|为|什|么|要|冷|冻|成|冰|状|因|为|发|了|酵|的|水|豆|鼓|在|冰|箱|冷|藏|室|温|度|下|仍|会|发|酵|而|发|酵|过|了|的|会|影|响|口|感|放|冷|冻|室|冻|成|冰|状|不|会|把|发|酵|菌|冻|死|它|只|让|它|暂|停|发|酵|拿|出|来|自|然|解|冻|后|口|感|一|样|鲜|美|但|注|意|千|万|不|能|用|微|波|炉|解|冻|杀|死|菌|了|口|感|不|好|r|发|好|酵|的|水|豆|鼓|会|产|生|很|多|细|长|的|菌|丝|这|是|正|常|的|说|明|你|发|酵|成|功|了|丝|丝|多|才|好|这|丝|丝|可|以|牵|到|1|米|长|呢|r|发|好|酵|的|水|豆|鼓|在|放|调|料|前|千|万|记|住|留|一|小|包|约|20|30|克|什|么|调|料|也|别|加|直|接|把|它|放|进|冰|箱|冷|冻|室|保|存|等|下|次|你|再|做|时|它|就|成|了|你|的|菌|种|不|用|再|花|十|多|块|的|冤|枉|钱|去|伊|滕|买|了|r|怎|么|样|想|试|试|吗|",
  "qid|qid_9077723349152984281|category|商|业|理|财|个|人|理|财|title|芯|片|卡|功|能|desc|answer|芯|片|通|常|分|为|三|大|类|第|一|类|是|cpu|芯|片|就|是|指|计|算|机|内|部|对|数|据|进|行|处|理|和|控|制|的|部|件|也|是|各|种|数|字|化|智|能|设|备|的|主|脑|第|二|类|是|存|储|芯|片|主|要|是|用|于|记|录|电|子|产|品|中|的|各|种|格|式|的|数|据|第|三|类|是|数|字|多|媒|体|芯|片|我|们|熟|知|的|数|码|相|机|越|来|越|逼|真|的|手|机|铃|声|就|是|通|过|此|类|芯|片|实|现|的|r|n|",
  "qid|qid_1543978678977646490|category|电|脑|网|络|硬|件|显|示|器|title|如|何|让|投|影|仪|在|放|映|而|电|脑|依|然|可|以|操|作|其|他|的|不|影|响|投|影|仪|我|前|两|desc|我|前|两|天|看|到|有|人|这|么|操|作|answer|这|样|你|需|要|有|双|显|示|功|能|的|机|器|如|一|般|的|笔|记|本|都|带|一|个|vga|接|口|或|者|双|显|示|卡|的|台|式|机|带|双|显|功|能|的|显|示|卡|等|r|n|第|一|步|连|接|投|影|在|另|外|一|个|显|示|接|口|上|r|n|第|二|步|在|桌|面|上|点|击|鼠|标|右|键|点|属|性|点|设|置|找|到|第|二|个|显|示|器|点|选|将|windows|桌|面|扩|展|在|该|显|示|器|上|r|n|第|三|步|点|确|定|r|n|第|四|步|把|要|在|投|影|上|显|示|的|内|容|向|右|拖|就|能|拖|到|另|外|一|个|显|示|器|也|就|是|投|影|可|以|显|示|了|同|时|第|一|个|显|示|器|可|以|做|别|的|而|不|影|响|投|影|显|示|",
  "qid|qid_4113197521587359311|category|烦|恼|交|友|技|巧|title|大|家|来|看|看|我|受|不|了|了|我|是|男|的|我|们|宿|舍|有|个|同|学|不|知|道|什|么|desc|我|是|男|的|我|们|宿|舍|有|个|同|学|不|知|道|什|么|问|题|r|我|快|疯|了|啊|我|买|什|么|衣|服|他|就|跟|我|买|什|么|衣|服|一|样|的|上|个|月|我|买|了|一|套|服|过|几|天|他|也|买|了|一|套|跟|我|一|样|的|后|来|我|又|买|了|两|条|牛|仔|裤|他|也|去|买|了|跟|我|一|样|的|我|受|不|了|了|啊|不|知|道|他|是|不|是|变|态|啊|害|我|现|在|都|不|知|道|穿|什|么|衣|服|了|啊|我|不|喜|欢|穿|跟|别|人|一|样|的|衣|服|的|啊|r|answer|我|大|学|的|时|候|也|和|你|有|类|似|的|经|历|r|n|我|打|耳|眼|他|们|也|去|我|戴|隐|形|眼|镜|他|们|也|去|配|总|之|是|我|买|什|么|东|西|他|们|也|跟|着|买|r|n|我|明|白|你|的|心|情|但|是|他|们|就|是|单|纯|得|觉|得|你|的|衣|服|很|好|你|就|把|你|自|己|当|作|他|们|的|流|行|时|尚|风|向|标|其|实|没|有|什|么|的|这|是|对|你|品|位|的|认|可|就|和|我|觉|得|你|用|的|洁|面|乳|很|好|用|所|以|我|也|去|买|一|样|的|",
  "qid|qid_1986055023312890569|category|电|脑|网|络|互|联|网|上|网|帮|助|title|一|般|的|蔓|藤|彩|绘|要|多|少|钱|啊|怎|么|算|价|格|的|啊|desc|怎|么|算|价|格|的|啊|answer|一|般|在|80|160|之|间|按|平|方|计|算|详|情|请|登|录|我|的|博|客|文|章|分|类|里|的|业|务|介|绍|和|价|格|",
  "qid|qid_5729552866040713794|category|社|会|民|生|法|律|title|关|于|立|法|没|有|权|力|制|定|规|章|a|国|家|发|展|和|改|革|委|员|会|b|国|务|院|desc|没|有|权|力|制|定|规|章|ra|国|家|发|展|和|改|革|委|员|会|b|国|务|院|办|公|厅|rc|国|家|人|事|部|d|审|计|署|r|问|题|补|充|书|本|上|是|选|a|解|释|说|国|家|发|展|和|改|革|委|员|会|主|要|职|责|为|拟|订|并|组|织|实|施|国|民|经|济|和|社|会|发|展|战|略|中|长|期|规|划|和|年|度|计|划|提|民|经|济|发|展|和|优|化|重|大|经|济|结|构|的|目|标|和|政|策|提|出|运|用|各|种|经|济|手|段|和|政|策|的|建|议|受|国|务|院|委|托|向|全|国|人|大|作|国|民|经|济|和|社|会|发|展|计|划|的|报|告|r|请|专|业|人|士|解|答|并|给|出|具|体|解|释|谢|谢|answer|选|a|是|正|确|的|r|n|因|为|根|据|立|法|法|第|七|十|一|条|规|定|国|务|院|各|部|委|员|会|中|国|人|民|银|行|审|计|署|和|具|有|行|政|管|理|职|能|的|直|属|机|构|可|以|根|据|法|律|和|国|务|院|的|行|政|法|规|决|定|命|令|在|本|部|门|的|权|限|范|围|内|制|定|规|章|r|n|部|门|规|章|规|定|的|事|项|应|当|属|于|执|行|法|律|或|者|国|务|院|的|行|政|法|规|决|定|命|令|的|事|项|而|国|家|发|展|和|改|革|委|员|会|没|有|决|定|命|令|的|行|政|权|力|r|n|",
  "qid|qid_7784419091660529988|category|教|育|科|学|理|工|学|科|title|任|何|事|物|都|有|由|生|到|灭|的|过|程|为|什|么|只|知|道|物|种|在|不|断|灭|亡|却|没|有|看|desc|answer|物|种|产|生|是|渐|变|物|种|灭|绝|很|多|都|是|突|变|",
  "qid|qid_2043732258234017217|category|健|康|肿|瘤|科|宫|颈|癌|title|阴|道|干|涩|和|妇|科|病|有|关|系|吗|自|从|我|生|过|孩|子|后|和|老|公|过|性|生|活|时|老|desc|自|从|我|生|过|孩|子|后|和|老|公|过|性|时|老|觉|得|我|阴|道|干|不|像|以|前|那|样|湿|润|这|让|我|很|是|苦|恼|请|问|这|种|现|象|一|般|由|哪|些|因|素|造|成|谢|谢|answer|阴|道|干|涩|是|很|多|年|轻|女|性|在|性|中|常|见|的|症|状|它|会|引|起|性|交|疼|痛|影|响|性|快|感|久|而|久|之|还|会|导|致|性|欲|减|退|性|冷|淡|最|终|影|响|到|夫|妻|和|睦|相|当|一|部|分|人|认|为|阴|道|干|涩|是|由|体|内|缺|少|雌|激|素|导|致|的|因|此|常|常|要|求|医|生|让|自|己|大|量|补|充|雌|激|素|这|种|做|法|对|她|们|有|害|无|益|r|正|处|于|生|育|年|龄|的|妇|女|是|一|生|中|性|功|能|最|旺|盛|的|时|期|所|以|大|部|分|人|体|内|都|不|缺|雌|激|素|那|么|导|致|阴|道|干|涩|的|原|因|是|什|么|呢|临|床|中|发|现|如|果|与|阴|道|干|涩|相|伴|的|还|有|口|角|发|炎|皮|肤|干|燥|脱|屑|等|症|状|就|要|考|虑|是|否|缺|乏|维|生|素|b2|维|生|素|b2|是|人|体|中|一|种|重|要|的|营|养|物|质|一|旦|缺|乏|会|影|响|皮|肤|黏|膜|的|代|谢|使|身|体|器|官|失|去|正|常|功|能|其|症|状|常|常|表|现|为|口|角|发|炎|眼|睑|炎|结|膜|炎|耳|鼻|黏|膜|干|燥|皮|肤|干|燥|脱|屑|等|它|的|缺|乏|对|女|性|生|殖|器|的|损|伤|尤|其|严|重|表|现|为|阴|道|壁|干|燥|黏|膜|充|血|和|溃|破|如|果|发|现|自|己|身|上|有|这|种|现|象|最|好|先|到|医|院|的|妇|产|科|就|诊|此|外|也|可|通|过|食|物|或|药|物|来|补|充|维|生|素|b2|比|如|多|吃|五|谷|杂|粮|带|皮|的|水|果|等|因|为|维|生|素|b2|大|多|存|在|于|植|物|的|果|皮|或|壳|中|服|用|维|生|素|b2|的|补|充|剂|也|可|让|体|内|营|养|物|质|达|到|平|衡|缓|解|阴|道|干|涩|具|体|方|法|是|口|服|每|日|3|次|每|次|10|毫|克|r|",
  "qid|qid_2585360456584549615|category|教|育|科|学|理|工|学|科|天|文|学|title|太|阳|上|面|的|水|是|以|什|么|形|态|存|在|的|desc|answer|太|阳|表|面|温|度|达|6000|度|水|是|以|氢|离|子|和|氢|氧|根|离|子|形|式|存|在|r|n|太|阳|上|可|能|有|水|加|拿|大|waterloo|大|学|的|peter|bernath|在|1991|年|研|究|太|阳|黑|子|光|谱|时|发|现|了|有|史|以|来|观|测|到|太|阳|表|面|的|最|低|温|度|事|实|上|水|在|5000k|以|下|并|不|会|完|全|被|分|离|成|h|和|oh|基|所|以|还|是|有|某|些|极|高|温|的|水|蒸|气|存|在|于|太|阳|表|面|",
  "qid|qid_7008329957135961451|category|游|戏|网|易|游|戏|魔|兽|世|界|title|我|在|5173|上|买|了|一|个|1500|的|魔|兽|帐|号|不|到|3|天|就|被|恶|意|找|回|desc|我|问|下|你|们|注|册|交|易|的|时|候|出|售|帐|号|的|人|有|上|传|身|份|证|号|码|吗|r|你|们|可|以|提|供|交|易|方|的|银|行|帐|号|固|定|电|话|等|信|息|给|我|吗|我|要|去|公|安|局|报|警|r|不|可|能|出|售|担|保|帐|号|时|候|担|保|人|的|信|息|都|没|有|吧|r|answer|你|在|买|号|的|时|候|第|一|点|就|应|该|看|卖|家|提|供|不|提|供|身|份|证|扫|描|件|第|二|要|看|改|帐|号|能|身|份|证|重|置|么|第|三|也|就|是|能|转|服|么|和|卖|家|帐|号|有|过|交|易|记|录|如|果|上|述|有|哪|点|不|满|足|该|帐|号|都|很|危|险|如|果|就|算|都|满|足|了|也|危|险|他|没|事|改|个|密|码|所|以|买|号|就|看|运|气|了|看|卖|家|人|品|了|r|n|",
  "qid|qid_78284520140909447|category|游|戏|网|易|游|戏|魔|兽|世|界|title|战|士|的|双|手|武|器|是|看|秒|攻|还|是|攻|击|上|限|我|朋|友|说|战|士|的|双|手|武|器|只|要|desc|我|说|战|士|的|双|手|武|器|只|要|秒|攻|高|就|好|真|是|这|样|的|吗|难|道|不|看|攻|击|最|高|值|吗|answer|你|朋|友|说|得|是|狂|战|吧|r|n|如|果|是|武|器|战|就|不|一|定|拿|dps|高|的|了|r|n|r|n|武|器|战|的|武|器|上|限|高|好|当|然|都|高|是|最|好|的|了|r|n|因|为|本|身|技|能|里|有|不|受|武|器|攻|击|速|度|限|制|的|技|能|就|是|瞬|发|技|能|r|n|出|暴|击|之|后|有|意|想|不|到|的|效|果|r|n|r|n|战|士|在|战|斗|中|接|近|敌|人|的|机|会|并|不|是|很|多|r|n|所|一|每|次|机|会|都|很|重|要|r|n|平|稳|输|出|我|认|为|效|果|不|佳|r|n|仅|在|pk|中|是|这|样|r|nfb|就|不|一|样|了|r|n|",
  "qid|qid_2301762798312100171|category|教|育|科|学|外|语|学|习|title|台|商|itc|产|业|英|语|itc|是|什|么|全|拼|名|称|desc|台|商|c|产|业|英|语|itc|是|什|么|全|拼|名|称|answer|应|该|是|信|息|技|术|与|通|讯|产|业|information|technologe|and|communication|r|n|r|n|",
  "qid|qid_8907005866216628656|category|娱|乐|博|彩|title|好|了|不|说|了|罗|马|尼|亚|伤|心|罗|马|尼|亚|真|的|不|行|了|不|过|火|锅|太|大|了|倍|投|还|有|好|desc|罗|尼|亚|真|的|不|行|了|不|过|火|锅|太|大|了|倍|投|还|有|好|多|单|中|了|没|意|思|祝|大|家|好|运|我|说|过|大|家|没|必|要|刻|意|博|冷|火|锅|就|是|火|锅|国|家|队|比|赛|而|且|都|是|强|队|的|比|赛|上|期|胆|少|这|期|太|多|出|一|二|个|还|是|没|奖|金|除|法|国|沙|特|外|answer|感|觉|是|30|但|是|现|在|2|1|这|并|不|稳|如|果|奥|地|利|再|进|个|那|就|是|了|感|觉|要|出|1|这|是|我|不|想|要|的|不|如|出|0|",
  "qid|qid_2461982717893146507|category|电|脑|网|络|互|联|网|上|网|帮|助|title|网|络|媒|体|资|源|网|哪|个|好|desc|answer|看|看|版|主|俱|乐|部|吧|那|里|汇|聚|了|全|国|最|权|威|的|网|络|媒|体|资|源|能|让|我|们|快|速|找|到|我|们|自|己|急|需|的|媒|体|资|源|",
  "qid|qid_4109011963370125292|category|文|化|艺|术|历|史|话|题|title|听|说|北|京|建|都|860|年|了|从|什|么|时|候|开|始|的|啊|都|那|些|朝|代|啊|desc|听|说|建|都|860|年|了|从|什|么|时|候|开|始|的|啊|都|那|些|朝|代|啊|r|answer|公|元|1153|年|金|主|亮|贞|元|元|年|由|上|都|迁|燕|京|北|京|始|成|为|国|都|金|元|明|永|乐|帝|一|直|到|现|在|",
  "qid|qid_8541638805318977815|category|健|康|整|形|美|容|眼|部|整|形|title|孩|子|得|了|恶|性|肿|瘤|作|为|妈|妈|如|何|救|她|尊|敬|的|学|诚|法|师|孩|子|马|上|8|岁|desc|尊|敬|的|学|诚|r|孩|子|马|上|8|岁|了|去|年|夏|天|诊|断|为|恶|性|横|纹|肌|肉|瘤|经|过|了|半|年|多|的|放|化|疗|现|在|回|家|了|作|为|妈|妈|我|很|心|痛|为|了|帮|助|孩|子|从|孩|子|手|术|后|我|就|开|始|吃|素|而|且|开|始|念|佛|号|还|带|孩|子|去|寺|院|拜|佛|为|冤|亲|债|主|立|超|度|的|排|位|每|天|我|夜|不|能|寐|很|是|不|安|r|请|问|作|为|妈|妈|我|还|能|为|孩|子|做|些|什|么|answer|推|己|及|人|以|此|心|体|会|天|下|母|亲|之|心|发|愿|为|拔|除|所|有|众|生|的|痛|苦|而|积|功|累|德|",
  "qid|qid_1813420482139576514|category|教|育|科|学|理|工|学|科|工|程|技|术|科|学|title|请|问|热|继|器|的|双|金|属|片|用|什|么|材|料|造|的|desc|请|问|热|继|器|的|双|金|属|片|用|什|么|造|的|answer|热|继|电|器|的|测|量|元|件|通|常|用|双|金|属|片|它|是|由|主|动|层|和|被|动|层|组|成|主|动|层|材|料|采|用|较|高|膨|胀|系|数|的|铁|铬|镍|合|金|被|动|层|材|料|采|用|膨|胀|系|数|很|小|的|铁|镍|合|金|r|n|",
  "qid|qid_7695728816446997476|category|娱|乐|博|彩|title|080|期|512|元|斯|图|加|单|0.01|阿|森|纳|desc|01|阿|森|纳|博|尔|顿|单|3|足|以|解|决|问|题|客|队|3|轮|来|1|胜|2|平|但|区|分|与|谁|r02|富|勒|姆|狼|队|31|主|队|不|败|格|局|r03|曼|城|布|莱|克|3|两|队|关|系|特|殊|r04|纽|卡|斯|布|莱|浦|13|两|支|升|班|马|对|决|看|好|主|队|不|败|r05|西|布|朗|热|刺|31|热|刺|40|年|后|再|次|踏|进|欧|冠|赛|场|下|周|中|做|客|出|战|小|组|对|手|德|甲|劲|旅|不|来|梅|联|赛|分|心|难|免|西|布|朗|本|赛|季|前|3|战|主|胜|客|负|该|队|从|上|赛|季|至|今|连|续|13|个|主|场|保|持|不|败|面|对|做|客|能|力|不|强|的|热|刺|有|望|保|得|不|败|r06|西|汉|姆|切|尔|西|01|切|尔|西|过|往|11|次|与|西|汉|姆|交|锋|中|取|得|9|胜|2|平|但|近|两|季|01|阿|森|纳|博|尔|顿|单|3|足|以|解|决|问|题|客|队|3|轮|来|1|胜|2|平|但|区|分|与|谁|r02|富|勒|姆|狼|队|31|主|队|不|败|格|局|r03|曼|城|布|莱|克|3|两|队|关|系|特|殊|r04|纽|卡|斯|布|莱|浦|13|两|支|升|班|马|对|决|看|好|主|队|不|败|r05|西|布|朗|热|刺|31|热|刺|40|年|后|再|次|踏|进|欧|冠|赛|场|下|周|中|做|客|出|战|小|组|对|手|德|甲|劲|旅|不|来|梅|联|赛|分|心|难|免|西|布|朗|本|赛|季|前|3|战|主|胜|客|负|该|队|从|上|赛|季|至|今|连|续|13|个|主|场|保|持|不|败|面|对|做|客|能|力|不|强|的|热|刺|有|望|保|得|不|败|r06|西|汉|姆|切|尔|西|01|切|尔|西|过|往|11|次|与|西|汉|姆|交|锋|中|取|得|9|胜|2|平|但|近|两|季|4|次|交|手|只|取|得|2|胜|2|平|r07|维|冈|桑|德|兰|10|维|冈|本|赛|季|前|两|个|主|场|全|败|1|球|不|进|失|了|10|球|算|上|上|赛|季|维|冈|近|5|个|主|场|只|赢|过|1|次|r08|多|特|沃|尔|夫|10|多|特|过|往|主|战|狼|堡|8|胜|4|平|2|负|但|近|3|个|主|场|2|平|1|负|下|周|中|的|欧|联|小|组|赛|首|战|防|其|分|心|r09|门|兴|法|兰|克|03|本|场|难|度|大|但|客|胜|不|能|g|r10|汉|堡|纽|伦|堡|3|汉|堡|近|9|次|主|战|对|手|8|胜|1|平|优|势|十|分|明|显|纽|伦|堡|5|次|做|客|1|胜|4|负|3|次|大|败|相|信|主|队|大|炒|r11|汉|诺|威|勒|沃|01|汉|诺|威|过|往|主|战|勒|沃|4|胜|6|平|5|负|近|8|次|主|战|1|胜|4|平|3|负|勒|沃|前|锋|黑|尔|默|斯|和|队|长|罗|尔|费|斯|伤|愈|归|队|攻|防|实|力|均|有|所|增|强|上|轮|主|场|惨|败|堪|称|奇|耻|大|辱|看|客|军|反|弹|r12|弗|赖|堡|斯|图|加|0|上|赛|季|弗|赖|堡|被|斯|图|加|特|双|杀|且|青|年|军|近|年|来|还|从|未|有|过|开|局|3|连|败|的|先|例|r13|拜|仁|不|来|梅|3|oddset|的|1.60|3.00|4.00|组|合|下|拜|仁|5|次|入|选|取|得|全|胜|其|中|2|次|战|胜|不|来|梅|r14|切|塞|纳|ac|米|兰|0|客|让|1|球|中|高|水|优|势|特|别|大|博|胆|r|answer|除|了|热|刺|其|余|和|楼|主|同|思|路|r|n|r|n|今|天|看|到|了|朋|友|们|都|不|看|好|热|刺|看|来|是|要|谨|慎|对|待|这|场|了|r|n|r|n|个|人|觉|得|热|刺|下|周|中|客|场|战|不|莱|梅|本|轮|预|计|会|演|练|防|反|阵|型|上|轮|刚|败|为|了|士|气|不|应|连|败|给|保|级|队|啊|r|n|r|n|说|个|题|外|话|热|刺|为|国|捐|躯|2|大|将|会|否|和|中|超|青|岛|队|一|样|上|轮|疑|似|受|到|官|哨|保|护|呢|呵|呵|r|n|r|n|祝|好|运|",
  "qid|qid_7719774581845631279|category|生|活|title|我|不|知|道|2011|西|安|市|考|报|名|地|址|在|哪|有|没|有|人|知|道|的|desc|answer|我|身|边|考|公|务|员|的|同|学|都|买|的|是|华|图|教|育|的|书|还|有|一|些|去|华|图|报|班|了|他|们|说|华|图|的|政|法|干|警|培|训|班|开|设|的|比|较|合|理|把|民|法|和|基|础|课|程|分|开|来|了|所|以|可|以|根|据|自|己|的|实|际|情|况|选|择|一|门|或|者|多|门|去|上|课|报|的|班|越|多|优|惠|越|多|他|们|报|班|之|前|也|比|较|了|好|几|家|培|训|机|构|最|后|汇|总|起|来|再|加|上|现|场|咨|询|都|决|定|去|报|陕|西|华|图|的|班|了|相|信|大|家|的|眼|睛|是|雪|亮|的|吧|呵|呵|你|也|可|以|去|陕|西|华|图|咨|询|下|就|在|小|寨|汉|唐|书|城|8|楼|呢|我|同|学|就|是|去|那|儿|咨|询|的|",
  "qid|qid_4034019609632244539|category|电|脑|网|络|互|联|网|title|请|问|新|浪|微|博|是|最|好|用|的|吗|各|个|网|站|都|有|微|博|新|浪|排|第|几|啊|desc|各|个|都|有|微|博|新|浪|排|第|几|啊|answer|当|然|啦|我|们|都|是|微|博|控|啊|",
  "qid|qid_4586346939984102214|category|生|活|购|物|title|帮|我|个|忙|我|问|个|价|格|我|想|买|个|取|暖|气|但|又|不|懂|市|场|价|谁|能|告|诉|我|大|desc|我|想|买|个|取|暖|气|但|又|不|懂|价|谁|能|告|诉|我|大|概|需|要|多|少|钱|谢|谢|了|answer|商|场|里|品|牌|的|要|200|多|外|面|小|电|器|商|行|只|要|60|70|元|左|右|一|个|取|暖|气|没|有|必|要|买|好|的|到|当|地|小|电|器|商|行|买|就|可|以|了|",
  "qid|qid_2958710616169871282|category|游|戏|网|络|游|戏|title|武|道|会|玩|拳|的|技|能|点|怎|么|加|啊|请|教|高|手|武|道|会|玩|拳|的|技|能|点|怎|么|加|啊|desc|武|道|会|玩|拳|的|技|能|点|怎|么|加|啊|r|n|希|望|高|手|能|告|诉|我|r|n|要|说|明|白|点|我|才|懂|哦|r|n|谢|谢|咯|answer|玩|拳|前|期|重|力|拳|加|到|可|以|学|流|星|拳|好|像|是|2|级|要|是|觉|的|练|级|慢|其|实|可|以|全|部|加|满|拳|法|专|精|理|所|当|然|是|要|全|满|的|共|通|的|冥|想|只|需|要|一|点|技|能|点|就|能|学|满|瞒|好|用|的|拳|法|强|化|当|然|也|是|要|学|的|r|n|我|也|只|是|玩|前|期|的|后|期|还|没|研|究|呵|",
  "qid|qid_3758382171738024593|category|娱|乐|博|彩|title|麻|烦|请|教|怎|么|传|图|谢|谢|desc|麻|烦|请|教|怎|么|传|图|谢|谢|answer|把|拍|有|照|片|的|硬|件|如|手|机|相|机|用|usb|线|与|电|脑|连|接|回|答|或|提|问|时|点|浏|览|找|到|这|张|照|片|确|定|就|行|了|",
  "qid|qid_6231371354979360246|category|娱|乐|度|假|旅|游|title|济|南|附|近|有|没|有|樱|桃|园|在|哪|里|desc|附|近|有|没|有|樱|桃|园|在|哪|里|answer|你|好|在|仲|宫|镇|南|部|山|区|有|啊|",
  "qid|qid_3554855727125997778|category|健|康|精|神|心|理|科|心|理|科|title|为|何|现|在|这|些|人|等|候|说|90|后|是|脑|残|你|们|都|是|90|后|是|脑|残|亲|看|看|desc|你|们|都|是|90|后|是|脑|残|亲|看|看|现|在|的|00|后|比|我|们|还|疯|狂|你|们|怎|么|不|说|说|他|们|议|论|议|论|他|们|我|们|被|针|对|了|这|么|久|都|在|慢|慢|老|去|了|而|00|后|呢|answer|90|后|从|出|生|开|始|生|活|环|境|相|对|前|几|代|优|越|许|多|没|有|经|历|过|历|史|和|政|治|上|的|动|荡|也|没|有|经|历|过|大|的|经|济|波|动|由|于|计|划|生|育|政|策|的|影|响|90|后|的|大|部|分|都|是|独|生|子|女|除|父|母|外|亲|情|观|相|对|淡|薄|在|这|样|的|生|长|环|境|里|有|相|当|一|部|分|人|比|前|几|代|人|来|说|更|显|得|有|些|孤|僻|今|日|中|国|的|信|息|发|展|迅|速|同|时|由|于|经|济|全|球|一|体|化|的|原|因|全|球|各|地|的|物|品|均|涌|入|中|国|各|国|交|流|日|益|广|泛|而|90|后|相|对|年|轻|对|新|事|物|的|接|受|能|力|较|强|在|审|美|观|和|价|值|观|方|面|也|与|前|人|有|很|大|不|同|r|n|总|的|来|说|90|后|的|问|题|不|是|你|们|自|己|造|成|的|但|是|你|们|真|的|表|现|很|差|非|主|流|我|就|不|说|了|也|许|这|是|你|们|的|审|美|问|题|追|星|谈|恋|爱|炫|富|攀|比|还|有|那|个|宁|愿|汶|川|地|震|的|难|道|这|些|都|是|你|们|应|该|做|的|也|许|你|会|说|你|们|80|后|也|有|这|样|的|但|是|你|也|要|明|白|比|例|问|题|在|你|身|边|的|90|后|们|有|几|个|不|是|这|样|的|r|n|我|不|会|称|你|们|脑|残|每|个|人|都|有|尊|严|如|果|你|们|能|够|真|正|成|熟|起|来|向|别|人|证|明|你|们|不|是|脑|残|谁|还|会|这|么|无|聊|去|说|呢|当|你|们|能|真|正|担|当|起|建|设|祖|国|的|大|业|时|谁|还|有|资|格|说|呢|",
  "qid|qid_1793283395186071139|category|游|戏|网|络|游|戏|title|想|做|把|80|级|灵|枪|想|做|把|80|级|灵|枪|有|3|种|选|择|1|所|有|5|desc|想|做|把|80|级|灵|枪|有|3|种|选|择|1|所|有|5|金|5|ts1700|满|2|金|5|ts1700|满|忽|视|抗|性|20；3|所|有|5|技|能|10|ts1700|满|哪|种|最|好|谢|谢|指|教|r|碧|波|瑶|池|会|员|2800|3000w|80ts|已|有|1700|满|answer|除|了|技|能|那|两|个|都|可|以|用|所|有|相|五|和|忽|视|抗|性|加|天|伤|也|很|好|至|于|金|五|配|合|这|两|条|我|不|建|议|所|有|相|五|加|高|你|防|和|速|度|更|为|贴|切|",
  "qid|qid_2831811435504024684|category|健|康|肿|瘤|科|title|如|何|更|好|的|治|疗|慢|性|咽|炎|我|的|嗓|子|有|时|清|凉|有|时|发|哑|医|生|让|我|吃|三|desc|我|的|嗓|子|有|时|清|凉|有|时|发|哑|让|我|吃|三|个|月|的|穿|王|消|炎|片|我|吃|了|一|个|月|了|实|在|是|不|好|吃|我|都|要|吐|了|请|问|有|没|有|更|好|的|方|法|answer|慢|性|咽|炎|的|治|疗|r|r1|主|要|针|对|病|因|如|戒|烟|戒|酒|积|极|治|疗|急|性|咽|炎|及|鼻|腔|鼻|窦|扁|桃|体|的|慢|性|炎|症|改|善|工|作|和|环|境|避|免|粉|尘|及|有|害|气|体|的|刺|激|加|强|锻|炼|增|强|体|质|预|防|感|冒|r|r2|病|人|如|有|咽|干|咽|痛|可|选|用|一|些|含|片|如|华|素|片|草|珊|瑚|含|片|银|黄|含|片|泰|乐|奇|含|片|西|瓜|霜|含|片|等|以|减|轻|或|解|除|症|状|r|r3|也|可|选|用|各|种|中|成|药|如|万|应|胶|囊|清|咽|利|喉|颗|粒|一|清|胶|囊|十|味|龙|胆|花|颗|粒|r|r4|如|果|患|慢|性|肥|厚|性|咽|炎|咽|干|咽|部|异|物|感|明|显|时|可|采|用|分|次|激|光|冷|冻|或|电|灼|治|疗|r|r|专|家|提|醒|慢|性|咽|炎|一|般|不|需|要|使|用|抗|生|素|治|疗|因|为|慢|性|咽|炎|并|非|细|菌|感|染|r|r|因|为|r|r1|滥|用|抗|生|素|可|能|导|致|咽|喉|部|正|常|菌|群|失|调|引|起|二|重|感|染|r|r2|每|一|种|抗|生|素|都|有|全|身|副|作|用|滥|用|抗|生|素|可|对|人|体|造|成|危|害|r|r3|滥|用|抗|生|素|还|能|引|起|细|菌|耐|药|r|r|需|要|注|意|的|是|慢|性|咽|炎|极|易|反|复|症|状|常|发|生|在|疲|劳|受|凉|烟|酒|过|度|进|刺|激|性|食|物|气|候|突|变|及|吸|入|寒|冷|空|气|后|这|时|病|人|咽|干|咽|痛|较|为|剧|烈|部|分|病|人|还|有|发|热|检|查|常|可|见|咽|部|黏|膜|急|性|充|血|肿|胀|血|常|规|检|查|白|细|胞|增|高|中|性|粒|细|胞|百|分|率|增|高|此|时|可|在|医|生|指|导|下|使|用|广|谱|抗|生|素|治|疗|或|根|据|药|敏|试|验|选|用|相|应|的|抗|生|素|治|疗|3|5|天|急|性|症|状|消|失|后|马|上|停|药|同|时|病|人|需|休|息|多|饮|水|及|进|食|流|汁|饮|食|r|r|",
  "qid|qid_7481899923593833557|category|电|脑|网|络|互|联|网|title|为|什|么|那|么|多|黑|客|老|是|在|邮|箱|发|的|乱|七|八|糟|信|件|desc|老|是|在|邮|箱|发|的|乱|七|八|糟|信|件|answer|黑|客|不|会|无|聊|到|望|你|邮|箱|里|发|乱|七|八|糟|得|东|西|你|说|的|都|是|垃|圾|广|告|邮|件|吧|这|个|免|费|邮|件|的|通|病|收|费|邮|箱|会|管|理|严|格|一|点|很|多|垃|圾|邮|件|会|被|过|滤|掉|",
  "qid|qid_1665016037283621083|category|游|戏|网|络|游|戏|奇|迹|世|界|title|请|问|如|果|红|名|死|了|的|话|包|裹|里|的|装|备|和|东|西|会|被|暴|掉|么|如|果|用|了|安|全|desc|请|问|如|果|红|名|死|了|的|话|包|裹|里|的|装|备|和|东|西|会|被|暴|掉|么|如|果|用|了|安|全|锁|链|的|话|是|不|是|那|里|的|装|备|都|不|会|掉|落|也|就|是|即|使|红|名|了|带|上|安|全|锁|链|挂|了|就|不|会|掉|装|备|是|么|answer|没|安|全|锁|链|肯|定|掉|r|n|有|安|全|锁|链|肯|定|不|会|掉|",
  "qid|qid_2384368102589799049|category|游|戏|完|美|游|戏|武|林|外|传|title|武|林|更|新|问|题|怎|么|关|闭|游|戏|再|打|开|后|总|是|出|现|发|现|新|版|本|是|否|立|即|升|desc|怎|么|关|闭|再|打|开|后|总|是|出|现|发|现|新|版|本|是|否|立|即|升|级|啊|answer|你|更|新|到|最|新|版|本|就|可|以|了|过|年|了|又|增|加|了|好|多|活|动|r|n|双|倍|经|验|r|n|r|n|具|体|双|倍|时|间|安|排|如|下|r|n2|月|2|日|14|00|18|00|r|n2|月|3|日|14|00|18|00|r|n2|月|6|日|14|00|18|00|r|n2|月|7|日|14|00|18|00|r|n2|月|10|日|14|00|18|00|r|n2|月|11|日|14|00|18|00|r|n2|月|14|日|14|00|18|00|r|n2|月|15|日|14|00|18|00|r|n|r|n|海|盗|的|入|侵|r|n|r|n|活|动|时|间|2|月|1|日|到|2|月|21|日|期|间|13|00|14|00|15|线|八|里|庄|r|n|来|自|海|盗|船|的|怪|物|侵|袭|八|里|庄|十|二|上|代|寡|不|敌|众|形|势|十|分|危|急|r|n|r|n|春|节|活|动|r|n|r|n|活|动|时|间|2|月|5|日|到|2|月|14|日|期|间|21|00|24|00|15|线|京|城|r|n|散|财|童|子|派|送|春|节|大|奖|好|运|隆|隆|而|至|财|源|滚|滚|而|来|r|n|祝|你|游|戏|快|乐|",
  "qid|qid_34331558253166439|category|游|戏|网|络|游|戏|title|法|师|与|精|力|请|问|法|师|加|精|力|可|以|提|升|魔|法|攻|击|速|度|吗|可|以|提|升|魔|法|念|desc|请|问|加|精|力|可|以|提|升|魔|法|攻|击|速|度|吗|可|以|提|升|魔|法|念|咒|速|度|吗|谢|谢|answer|不|行|",
  "qid|qid_6643157122912268630|category|生|活|美|容|塑|身|护|肤|title|黑|眼|圈|越|来|越|重|怎|么|办|白|天|总|感|觉|很|累|desc|黑|眼|圈|越|来|越|重|怎|么|办|白|天|总|感|觉|很|累|answer|多|数|人|的|黑|眼|圈|是|后|天|原|因|即|因|经|常|熬|夜|吸|烟|清|除|眼|影|妆|不|彻|底|或|睡|觉|时|枕|头|太|低|而|形|成|的|怎|样|消|除|掉|黑|眼|圈|兹|介|绍|以|下|几|种|方|法|一|涂|蜂|蜜|法|在|洗|脸|后|勿|擦|干|脸|上|的|水|分|让|其|自|然|干|然|后|在|眼|部|周|围|涂|上|蜂|蜜|先|按|摩|几|分|种|再|等|10|分|种|后|用|清|水|洗|净|水|不|要|擦|去|使|其|自|然|干|涂|上|面|霜|即|可|二|敷|酸|奶|法|用|纱|布|蘸|上|些|酸|奶|敷|在|眼|睛|周|围|每|次|十|分|钟|三|热|鸡|蛋|按|摩|法|将|鸡|蛋|煮|熟|后|去|壳|用|小|毛|巾|包|裹|住|合|上|双|眼|用|鸡|蛋|按|摩|眼|睛|四|周|可|加|快|血|液|循|环|四|苹|果|片|敷|眼|法|将|含|汁|量|多|的|苹|果|切|片|敷|双|眼|每|次|15|分|钟|五|土|豆|片|敷|眼|法|土|豆|去|皮|切|成|约|2|厘|米|的|厚|片|外|敷|双|眼|每|次|5|分|钟|六|同|时|也|可|以|采|用|复|合|进|行|黑|眼|圈|的|祛|除|",
  "qid|qid_1508851283884568158|category|娱|乐|幽|默|滑|稽|title|中|国|境|内|活|得|最|潇|洒|的|是|哪|一|类|人|desc|answer|丁|克|一|族|",
  "qid|qid_1966271995448072003|category|健|康|儿|科|title|胃|胀|气|打|隔|有|隔|打|不|出|来|吃|东|西|感|觉|很|硬|不|消|化|不|敢|吃|东|西|desc|胃|胀|气|打|隔|有|隔|打|不|出|来|吃|东|西|感|觉|很|硬|不|消|化|不|敢|吃|东|西|五|六|年|之|前|做|过|胃|镜|是|慢|性|浅|表|性|胃|炎|后|来|吃|胃|粒|康|颗|粒|吃|好|过|可|是|烤|肉|吃|多|了|又|犯|过|几|次|病|到|现|在|我|感|觉|冬|天|的|时|候|胃|难|受|比|夏|天|厉|害|应|该|吃|什|么|药|饮|食|有|什|么|注|意|事|项|answer|您|好|鉴|于|您|很|长|时|间|没|有|行|胃|镜|检|查|了|所|以|建|议|先|行|胃|镜|检|查|在|明|确|是|否|有|疾|病|的|基|础|上|再|决|定|是|否|服|用|相|关|药|物|不|能|仅|凭|症|状|就|自|行|服|用|药|物|可|能|会|适|得|其|反|",
  "qid|qid_6363440678237764642|category|电|脑|网|络|软|件|title|金|山|词|霸|哪|个|版|本|比|较|好|用|免|费|版|可|以|安|装|的|desc|免|费|版|可|以|安|装|的|answer|目|前|金|山|词|霸|2005|版|本|比|较|稳|定|好|用|2006|版|本|出|来|不|久|如|你|先|安|装|acrobat|rreader|6.0|后|安|装|金|山|词|霸|2005|的|话|还|支|持|pdf|文|件|的|取|词|翻|译|r|n|下|载|地|址|",
  "qid|qid_1233180131150352374|category|商|业|理|财|股|票|title|请|帮|忙|解|答|000825|成|本|18.15,600011|成|本|13.05|desc|000825|成|本|18.15,600011|成|本|13.05,000777|成|本|32.28|后|市|如|何|操|作|answer|000825|成|本|18.15|如|果|说|你|是|准|备|中|长|线|持|股|的|话|这|股|票|是|没|有|问|题|的|建|议|持|股|观|望|r|n600011|成|本|13.05|有|调|整|的|迹|象|建|议|先|出|局|等|调|整|到|位|逢|低|再|介|入|r|n000777|成|本|32.28|其|业|绩|属|一|次|性|收|益|而|不|是|主|营|收|益|个|人|建|议|逢|高|出|局|",
  "qid|qid_5567485424998867637|category|游|戏|完|美|游|戏|完|美|国|际|title|关|于|洗|点|问|题|我|是|77|小|武|侠|灵|力|一|直|没|加|还|是|5|点|但|是|穿|上|desc|我|是|77|小|武|侠|灵|力|一|直|没|加|还|是|5|点|但|是|穿|上|海|盗|装|后|会|加|灵|79|海|盗|王|还|会|加|我|现|在|穿|备|灵|力|是|9|我|想|穿|着|装|备|用|全|能|洗|点|券|不|知|道|能|不|能|把|原|始|的|5|灵|给|洗|去|啊|知|道|的|再|回|答|我|不|要|给|我|错|误|答|案|浪|费|我|的|元|宝|answer|我|没|洗|过|从|楼|上|的|大|体|意|思|我|估|计|洗|点|一|次|能|洗|去|5|点|属|性|但|是|起|初|的|5|点|灵|力|是|绝|对|洗|不|掉|的|但|是|你|把|灵|力|属|性|加|到|8|不|穿|装|备|再|用|洗|点|一|次|洗|去|5|点|不|就|灵|剩|3|点|了|么|相|当|于|投|机|取|巧|的|方|式|既|然|有|人|能|洗|到|3|你|必|须|确|定|有|人|无|装|备|灵|力|是|3|我|估|计|就|是|这|么|个|情|况|我|也|不|大|知|道|哈|",
  "qid|qid_3829681820734565978|category|健|康|外|科|肝|胆|外|科|title|腱|鞘|囊|肿|的|危|害|手|腕|上|长|了|个|腱|鞘|囊|肿|倒|是|一|点|也|不|疼|但|是|腱|鞘|囊|肿|对|desc|手|腕|上|长|了|个|腱|鞘|囊|肿|倒|是|一|点|也|不|疼|但|是|腱|鞘|囊|肿|对|身|体|有|什|么|危|害|吗|answer|腱|鞘|囊|肿|的|危|害|表|现|为|患|处|局|部|压|痛|触|摸|时|皮|下|饱|满|并|有|波|动|囊|样|感|伴|有|腕|部|无|力|不|适|或|疼|痛|多|为|酸|痛|或|放|射|性|痛|可|有|一|定|的|功|能|障|碍|r|n|很|多|患|者|被|手|腕|或|其|他|部|位|的|腱|鞘|囊|肿|所|困|扰|不|仅|会|出|现|疼|痛|等|症|状|还|会|对|学|习|生|活|造|成|一|定|的|困|扰|女|性|患|者|尤|其|对|手|腕|手|背|的|腱|鞘|囊|肿|很|在|意|疼|痛|不|变|的|同|时|还|会|影|响|美|观|所|以|腱|鞘|囊|肿|的|危|害|是|不|容|忽|视|的|r|n|腱|鞘|囊|肿|可|自|行|挤|爆|按|压|时|间|久|了|会|自|行|消|退|吸|收|但|是|极|易|感|染|复|发|或|采|用|穿|刺|抽|出|囊|液|注|入|肾|上|腺|皮|质|激|素|或|透|明|质|酸|梅|的|方|法|对|治|疗|腱|鞘|囊|肿|也|有|一|定|的|疗|效|如|果|手|术|切|除|腱|鞘|囊|肿|应|避|免|患|病|关|节|一|个|月|r|n|中|药|治|疗|贴|腱|囊|贴|安|全|方|便|治|疗|效|果|很|不|错|r|n|腱|鞘|囊|肿|患|者|要|注|意|对|患|处|的|休|息|忌|辛|辣|的|食|物|治|愈|后|要|预|防|腱|鞘|囊|肿|的|复|发|避|免|过|度|劳|损|适|当|休|息|和|锻|炼|r|n|",
  "qid|qid_4029454027761505606|category|游|戏|网|络|游|戏|title|请|问|52|级|大|地|穿|什|么|好|是|末|日|还|是|青|狼|都|加|什|么|属|性|请|各|位|高|手|desc|请|问|52|级|大|地|穿|什|么|好|是|末|日|还|是|青|都|加|什|么|属|性|请|各|位|高|手|指|点|一|下|answer|要|我|选|择|的|话|我|穿|青|狼|轻|r|n|r|n|如|果|你|有|钱|那|么|就|不|要|选|择|夏|隆|夏|隆|属|性|是|不|错|但|是|防|低|2|章|闪|避|毕|竟|还|不|是|很|能|体|现|加|4|的|闪|避|r|n|末|日|是|套|好|衣|服|如|果|你|不|打|算|纹|身|的|话|末|日|是|你|不|2|的|选|择|只|不|过|是|属|性|上|的|加|加|减|减|减|1|力|量|2|体|质|加|3|敏|捷|r|n|你|可|能|会|问|清|狼|有|什|么|用|加|15|施|法|要|来|做|什|么|5|的|防|御|真|的|那|么|重|要|吗|r|n|青|狼|为|b|顶|也|就|是|防|最|高|的|衣|服|和|末|日|一|样|但|是|5|看|上|去|是|没|什|么|用|你|全|部|冲|到|3|再|加|buff|再|加|5|是|多|少|防|不|用|我|说|了|吧|在|pk|的|时|候|真|的|是|闪|普|攻|重|要|些|吗|你|末|日|3|点|敏|1|2|点|闪|2|点|敏|加|一|点|闪|纹|过|的|都|知|道|所|以|要|尽|量|避|免|单|数|的|浪|费|技|能|命|中|于|闪|没|关|系|所|以|不|管|你|多|少|闪|都|会|被|技|能|打|中|的|当|然|我|不|是|说|大|地|闪|不|重|要|白|木|才|看|不|懂|我|在|说|什|么|r|n|r|n15|的|施|法|你|pk|多|就|会|知|道|有|多|重|要|了|迟|缓|的|效|果|多|用|用|出|门|带|把|弓|宝|藏|骑|士|银|月|等|等|再|不|为|惧|r|n|r|n|信|我|就|顶|我|咖|啡|r|n|",
  "qid|qid_4497604405927257528|category|游|戏|网|易|游|戏|魔|兽|世|界|title|lm|的|公|会|仓|库|在|哪|我|原|来|是|bl|的|desc|lm|的|公|会|仓|库|在|哪|我|原|来|是|的|answer|只|要|有|银|行|的|地|方|都|有|公|会|仓|存|",
  "qid|qid_4757082100615101536|category|游|戏|完|美|游|戏|诛|仙|title|为|什|么|我|挖|到|失|落|的|宝|藏|老|没|交|的|任|务|啊|天|行|者|那|没|交|任|务|的|对|话|desc|天|行|者|那|没|交|任|务|的|对|话|answer|周|日|的|12|点|到|周|二|维|护|期|间|即|使|你|挖|到|失|落|的|宝|物|也|交|不|了|r|n|周|二|维|护|完|以|后|到|周|日|12|点|前|是|交|失|落|的|宝|物|的|时|间|",
  "qid|qid_2022266574013777018|category|娱|乐|明|星|title|谁|是|你|心|中|的|大|帅|哥|desc|answer|香|港|古|天|乐|周|润|发|梁|朝|伟|任|达|华|r|n|韩|国|rini|宋|承|宪|r|n|台|湾|小|猪|仔|仔|r|n|国|外|小|贝|",
  "qid|qid_600662082255417032|category|商|业|理|财|产|业|信|息|title|龙|岗|最|大|的|净|水|器|厂|家|是|哪|个|呀|desc|龙|岗|最|大|的|厂|家|是|哪|个|呀|answer|怡|盛|源|www.ysyjs.com|是|深|圳|市|怡|盛|源|净|水|产|业|有|限|公|司|旗|下|品|牌|是|中|国|著|名|品|牌|中|国|净|水|器|十|大|品|牌|之|一|怡|盛|源|精|于|净|水|材|料|工|艺|技|术|的|研|发|与|优|化|组|合|全|方|位|解|决|及|单|位|的|生|活|饮|用|水|问|题|是|国|内|首|屈|一|指|的|净|水|器|生|产|厂|家|怡|盛|源|净|水|器|生|产|厂|家|凭|借|高|档|次|的|做|工|和|外|观|人|性|化|的|设|计|众|多|功|能|的|合|理|化|整|合|等|诸|多|优|点|正|受|到|越|来|越|多|消|费|者|的|喜|爱|怡|盛|源|新|技|术|的|应|用|解|决|了|当|前|水|源|污|染|对|消|费|者|饮|水|健|康|造|成|的|危|害|对|缓|解|居|民|饮|水|新|方|式|具|有|划|时|代|的|意|义|怡|盛|源|解|决|家|庭|饮|水|难|题|怡|盛|源|净|水|器|采|用|的|是|进|口|滤|芯|膜|面|积|相|对|国|产|膜|其|表|面|微|孔|均|匀|稳|定|不|易|污|秽|不|易|断|裂|膜|通|量|大|保|证|出|水|品|质|先|进|的|净|化|工|艺|有|效|地|净|化|我|们|的|自|来|水|让|我|们|能|够|喝|上|健|康|安|全|的|净|化|水|美|国|进|口|kdf|能|有|效|去|除|水|中|的|重|金|属|和|余|氯|优|质|椰|壳|活|性|炭|高|效|吸|附|水|中|的|异|色|异|味|保|证|出|水|水|质|无|色|无|味|甘|醇|可|口|有|了|怡|盛|源|净|水|器|我|们|不|用|再|担|心|自|来|水|发|黄|和|刺|鼻|的|漂|白|水|味|道|同|时|高|精|度|的|过|滤|让|水|中|有|害|物|质|被|阻|挡|在|净|水|之|外|可|怕|的|红|虫|线|形|虫|从|此|与|你|无|缘|怡|盛|源|净|水|器|我|看|行|怡|盛|源|除|了|净|化|能|力|出|众|其|售|后|服|务|也|非|常|到|位|每|一|位|购|买|怡|盛|源|净|水|器|的|客|户|怡|盛|源|会|定|期|派|遣|售|后|服|务|人|员|到|用|户|家|中|进|行|水|质|检|测|现|场|维|修|滤|芯|更|换|等|让|你|时|时|刻|刻|都|能|喝|上|健|康|安|全|的|饮|用|水|保|证|您|24|小|时|都|能|使|用|的|我|们|产|品|混|乱|的|净|水|市|场|品|种|繁|多|的|净|水|器|品|牌|怡|盛|源|做|为|净|水|器|十|大|品|牌|之|一|它|的|出|现|让|消|费|者|多|一|份|选|择|怡|盛|源|让|消|费|者|更|加|健|康|让|家|庭|更|加|温|馨|怡|盛|源|深|圳|净|水|器|生|产|厂|家|以|净|化|中|国|水|质|为|己|任|为|消|费|者|提|供|安|全|高|效|的|净|化|产|品|怡|盛|源|让|消|费|者|多|一|份|选|择|公|司|名|称|深|圳|怡|盛|源|净|水|产|业|有|限|公|司|地|址|广|东|深|圳|龙|岗|区|赠|人|玫|瑰|手|有|余|香|祝|您|好|运|一|生|一|世|如|果|回|答|有|用|请|点|好|评|谢|谢|",
  "qid|qid_2159798797982929645|category|健|康|妇|产|科|产|科|title|请|推|荐|几|个|好|看|的|韩|国|电|影|desc|answer|惊|喜|派|对|人|形|师|恋|爱|魔|术|师|谁|都|有|秘|密|我|的|野|蛮|女|友|",
  "qid|qid_4436489053185970120|category|烦|恼|恋|爱|title|她|失|恋|不|久|我|们|有|结|果|吗|我|们|一|直|是|同|班|同|学|原|来|我|一|直|很|喜|欢|她|desc|我|们|一|直|是|同|班|同|学|原|来|我|一|直|很|喜|欢|她|r|可|她|原|来|有|男|友|现|在|她|了|我|想|向|她|说|明|我|对|她|的|感|情|这|样|有|结|果|吗|answer|有|r|n|不|过|先|别|说|这|个|时|候|就|算|答|应|你|也|不|稳|定|r|n|继|续|对|她|好|r|n|爱|她|r|n|过|段|日|子|等|她|慢|慢|走|出|失|恋|的|阴|影|也|开|始|喜|欢|你|了|r|n|再|说|不|迟|",
  "qid|qid_2070352552714657550|category|汽|车|title|机|动|车|年|检|都|要|带|什|么|证|件|desc|机|年|检|都|要|带|什|么|证|件|answer|驾|驶|证|行|驶|证|各|种|税|费|缴|纳|证|明|身|份|证|还|要|把|罚|款|都|交|了|",
  "qid|qid_2372099682189065657|category|健|康|人|体|常|识|title|去|打|胎|请|半|天|假|够|吗|急|请|经|验|者|回|答|desc|answer|最|好|休|息|一|个|月|r|n|在|做|完|流|产|后|应|当|在|医|生|的|指|导|下|加|强|自|我|保|护|措|施|否|则|可|能|会|对|身|体|健|康|产|生|不|利|影|响|这|些|措|施|包|括|r|n1|适|当|休|息|头|三|天|最|好|卧|床|休|息|否|则|可|能|延|长|阴|道|出|血|时|间|半|个|月|内|最|好|避|免|参|加|体|力|劳|动|和|体|育|锻|炼|r|n2|适|当|补|充|营|养|要|适|当|补|充|营|养|多|吃|一|些|富|含|蛋|白|质|维|生|素|的|食|品|如|新|鲜|水|果|蔬|菜|瘦|肉|鲜|鱼|蛋|类|奶|或|豆|制|品|等|r|n3|注|意|个|人|卫|生|人|流|后|要|注|意|保|持|个|人|卫|生|否|则|易|造|成|感|染|流|产|后|一|个|月|内|应|避|免|性|生|活|出|血|期|间|勤|换|卫|生|护|垫|半|个|月|内|不|能|进|行|坐|浴|只|能|淋|浴|r|n4|注|意|出|血|情|况|流|产|后|的|一|周|左|右|阴|道|内|会|有|红|色|或|淡|红|色|分|泌|物|其|量|由|多|到|少|无|明|显|气|味|如|果|出|血|时|间|超|过|15|天|出|血|量|大|过|月|经|量|特|别|是|分|泌|物|有|臭|味|并|伴|有|发|烧|腹|痛|等|症|状|则|应|及|时|去|医|院|检|查|和|治|疗|以|免|病|情|加|重|r|n5|做|好|避|孕|工|作|一|旦|恢|复|性|生|活|要|做|好|避|孕|工|作|避|免|再|次|意|外|受|孕|",
  "qid|qid_2033705502100600130|category|游|戏|网|页|游|戏|title|文|明|度|怎|样|增|加|的|快|啊|desc|文|明|度|怎|样|增|加|的|快|啊|answer|怎|么|长|得|都|不|快|r|n|r|n|r|n|看|着|里|面|有|建|筑|提|升|的|声|望|度|当|然|如|果|肯|花|资|源|的|话|就|冲|神|庙|小|型|会|50|文|明|大|型|250|",
  "qid|qid_3558981436792464568|category|教|育|科|学|理|工|学|科|化|学|title|化|学|题|在|一|个|体|积|的|密|闭|容|器|放|入|3l|气|体|r|和|5l|气|体|q|在|一|定|条|件|desc|在|一|个|体|积|的|密|闭|容|器|放|入|3l|气|体|r|和|5l|气|体|q|在|一|定|条|件|下|发|生|反|应|2r|气|5q|气|4x|气|ny|气|反|应|完|全|后|容|器|温|度|不|变|混|合|气|体|的|压|强|是|原|来|的|87.5|则|方|程|中|的|n|值|是|ra|2|b|3|c|4|d|5|answer|选|a|r|n|r|n|r|n|因|为|完|全|反|应|后|压|强|改|变|所|以|n|改|变|也|就|是|2|5|4|n|所|以|排|除|b|项|再|就|是|完|全|反|映|后|压|强|变|为|原|先|的|87.5|也|就|是|说|减|小|了|那|么|反|应|就|是|向|系|数|和|小|的|方|向|进|行|所|以|只|有|a|项|符|合|r|n|r|n|望|采|纳|",
  "qid|qid_1732805277774471678|category|商|业|理|财|股|票|title|一|道|股|票|计|算|题|高|手|请|进|题|目|我|现|在|购|买|某|一|股|票|股|价|是|2.0|desc|题|目|我|现|在|购|买|某|一|股|价|是|2.00|元|我|要|购|买|5000|股|三|天|后|该|股|票|股|价|上|涨|为|2.10|元|r|在|这|一|次|交|易|中|我|一|共|要|交|多|少|的|费|用|我|能|赚|到|的|实|际|是|多|少|钱|r|r|麻|烦|高|手|把|计|算|过|程|写|清|楚|一|些|谢|谢|answer|你|的|资|金|是|2|5000|10000|r|n|手|续|费|印|花|税|千|分|之|2.5|10000|25|过|户|费|1.2|26.2|r|n|以|上|是|买|进|的|费|用|r|n|下|面|是|卖|出|的|费|用|r|n|你|的|资|金|是|2.10|5000|10500|r|n|手|续|费|印|花|税|千|分|之|2.5|10500|26.25|过|户|费|1.2|27.45|r|n|注|明|有|的|时|候|有|过|户|费|有|的|时|候|是|没|的|我|的|交|易|基|本|上|没|有|过|户|费|但|上|次|就|有|1.2|元|了|r|n|你|一|共|交|费|是|26.2|27.45|53.65|r|n|你|赚|了|10500|10000|500|53.65|446.35|",
  "qid|qid_1373866045137056510|category|生|活|美|容|塑|身|title|去|皱|眼|霜|选|什|么|好|desc|去|皱|选|什|么|好|answer|你|可|以|试|试|我|姐|用|的|这|中|艾|弗|莎|抗|皱|眼|霜|嘛|我|看|到|她|才|没|有|好|久|也|就|一|两|个|月|吧|现|在|基|本|上|都|看|不|出|来|了|说|这|个|是|能|显|著|淡|化|眼|部|幼|纹|与|细|纹|的|作|用|呵|呵|你|可|以|去|了|解|一|下|",
  "qid|qid_7713653475767665323|category|娱|乐|谜|语|title|什|么|车|无|轮|什|么|猪|无|嘴|什|么|驴|无|毛|什|么|屋|无|门|什|么|书|无|字|什|么|desc|answer|第|一|个|答|案|乞|朱|为|吾|昭|雪|r|有|个|典|故|的|在|明|太|主|朱|元|章|贪|官|当|道|人|们|有|怨|都|很|难|得|到|公|正|的|审|判|有|一|穷|书|生|考|试|三|年|而|不|第|并|且|家|中|田|地|被|占|官|官|相|卫|他|无|处|伸|冤|根|本|不|可|能|去|告|御|状|终|于|他|想|出|个|方|法|能|让|太|主|知|道|他|的|怨|情|他|出|了|这|么|个|让|别|人|猜|由|于|很|难|很|快|传|遍|天|下|也|就|传|到|了|太|主|的|手|里|太|主|看|了|后|便|把|他|招|来|问|你|有|什|么|怨|情|他|把|所|有|的|委|屈|都|说|了|出|来|贪|官|被|斩|首|了|可|他|们|总|不|明|白|书|生|怎|么|告|的|御|状|后|来|迷|揭|开|了|所|有|有|的|秘|密|全|在|那|个|诗|歌|里|r|何|车|无|轮|古|时|候|农|民|犁|地|在|牛|身|后|的|拉|的|叫|犁|车|犁|车|是|用|来|开|荒|的|那|时|候|人|们|管|开|荒|又|叫|起|地|所|以|起|通|乞|代|表|乞|求|的|意|思|r|何|猪|无|嘴|这|个|自|然|代|表|太|主|本|人|猪|代|表|朱|字|说|太|主|很|少|慰|问|国|事|像|无|嘴|一|样|r|何|驴|无|毛|也|代|表|太|主|本|人|太|主|本|来|当|过|和|尚|那|是|广|交|贤|良|以|天|下|为|心|用|这|句|和|上|面|的|那|句|对|比|说|明|太|主|应|该|管|国|事|了|所|以|这|句|代|表|的|字|就|是|为|字|表|示|太|主|该|有|所|为|了|r|何|屋|无|门|这|个|屋|字|通|吾|字|代|表|书|生|自|己|r|何|书|无|字|古|代|的|诏|书|都|是|太|监|念|的|而|太|监|根|本|不|认|识|字|所|以|诏|书|就|是|用|口|头|来|传|达|的|所|以|这|句|代|表|的|字|就|是|昭|r|何|花|无|叶|这|个|很|简|单|了|大|家|都|会|想|到|的|就|是|雪|花|了|这|句|代|表|的|就|是|雪|字|r|所|以|整|个|句|子|连|起|来|就|是|乞|朱|为|吾|昭|雪|r|天|下|之|博|学|者|当|数|书|生|令|以|后|之|人|没|有|不|为|之|叹|服|的|r|r|第|二|个|答|案|我|爱|你|一|万|年|r|何|车|无|轮|我|我|车|无|轮|我|的|车|本|来|就|没|有|轮|子|轮|子|给|我|卸|了|但|没|有|钱|买|新|的|轮|子|只|好|放|在|仓|库|里|r|何|猪|无|嘴|矮|爱|竹|无|嘴|矮|竹|就|是|我|们|说|的|文|竹|植|物|而|已|哪|里|来|的|嘴|巴|r|何|驴|无|毛|泥|你|驴|无|毛|谁|都|知|道|r|何|屋|无|门|蚁|一|屋|无|门|蚂|蚁|的|家|是|洞|口|没|有|门|的|r|何|书|无|字|万|万|树|无|字|多|少|树|都|没|有|字|的|r|何|花|无|叶|年|年|画|无|叶|年|画|而|已|和|叶|子|确|实|没|有|关|系|r|r|第|三|个|答|案|老|鼠|喜|欢|大|米|r|何|车|无|轮|老|车|无|轮|车|老|了|要|拆|掉|拆|车|是|先|拆|轮|子|的|可|能|是|由|于|轮|子|上|有|橡|胶|可|以|重|新|回|收|的|缘|故|当|然|轮|子|也|没|有|了|r|何|猪|无|嘴|紫|子|竹|猪|一|种|植|物|不|是|动|物|当|然|没|有|嘴|子|猪|的|子|就|是|鼠|学|过|天|干|地|支|的|人|都|知|道|r|何|驴|无|毛|锡|驴|无|毛|为|什|么|呢|无|锡|古|代|盛|产|金|属|锡|北|人|用|毛|驴|来|运|送|锡|矿|获|利|丰|厚|然|而|驴|经|常|在|锡|矿|中|容|易|锡|中|毒|浑|身|毛|发|尽|脱|常|见|运|锡|之|驴|无|毛|是|为|锡|喜|驴|无|毛|r|何|屋|无|门|幻|欢|屋|无|门|典|出|金|刚|经|r|何|书|无|字|大|叔|无|字|大|叔|是|农|村|的|叫|法|一|般|大|叔|是|只|有|名|字|不|会|专|门|学|文|人|取|什|么|字|号|等|等|乱|七|八|糟|的|东|西|的|r|何|花|无|叶|米|花|无|叶|谁|都|知|道|爆|米|花|直|接|可|以|吃|的|不|是|什|么|植|物|r|合|起|来|就|是|老|鼠|喜|欢|大|米|一|般|正|确|的|答|案|典|出|智|力|题|目|大|米|的|男|朋|友|是|什|么|r|r|第|四|个|答|案|风|雨|途|中|同|心|r|风|车|无|轮|r|雨|珠|谐|猪|无|嘴|r|秃|驴|无|毛|r|中|午|谐|屋|无|门|r|桐|树|谐|书|无|字|r|心|花|无|叶|r|风|雨|秃|中|桐|心|谐|风|雨|途|中|同|心|r|",
  "qid|qid_9193441501481640502|category|教|育|科|学|理|工|学|科|心|理|学|title|怎|样|理|解|驾|驶|它|的|是|我|们|自|己|无|忧|无|虑|的|心|灵|这|句|话|的|含|义|是|苏|教|版|desc|是|苏|教|版|的|语|文|请|速|度|回|答|please|answer|小|船|随|着|人|的|心|意|前|行|人|们|的|心|灵|在|大|海|上|自|由|驰|骋|心|灵|和|大|海|已|经|完|全|融|合|在|一|起|r|n|给|分|r|n|谢|谢|r|n|",
  "qid|qid_9149020487491891510|category|商|业|理|财|财|务|税|务|title|财|务|erp|软|件|是|什|么|啊|主|要|是|操|作|些|什|么|啊|大|致|怎|么|用|啊|怎|么|desc|answer|erp|enterprise|resources|planning|企|业|资|源|计|划|系|统|是|由|不|同|功|能|模|块|组|成|的|软|件|包|它|采|用|计|算|机|和|网|络|通|讯|技|术|的|最|新|成|就|是|旨|在|对|企|业|的|多|种|资|源|进|行|计|划|和|实|现|物|流|资|金|流|信|息|流|三|流|合|一|的|操|作|平|台|和|管|理|信|息|系|统|r|nerp|中|的|财|务|管|理|涉|及|到|总|账|ufo|报|表|固|定|资|产|应|收|款|管|理|应|付|款|管|理|专|家|财|务|评|估|公|司|对|帐|网|上|银|行|现|金|流|量|表|票|据|通|报|帐|中|心|网|上|报|销|web|财|务|等|等|它|不|只|涉|及|到|公|司|的|财|务|管|理|还|和|其|他|模|块|有|接|口|这|才|是|他|的|应|用|价|值|所|在|如|和|采|购|管|理|分|销|管|理|存|货|核|算|库|存|管|理|都|有|接|口|可|自|动|处|理|应|收|应|付|单|据|库|存|盘|盈|盘|亏|它|将|公|司|的|资|金|流|高|度|集|成|在|一|个|系|统|中|即|时|掌|控|企|业|的|财|务|运|营|情|况|r|n|除|了|财|务|会|计|外|erp|还|涉|及|到|管|理|会|计|如|资|金|管|理|成|本|管|理|项|目|管|理|预|算|管|理|等|等|r|n|你|购|买|了|软|件|就|会|有|相|应|的|培|训|的|这|个|比|单|纯|的|财|务|软|件|要|复|杂|一|些|r|n|",
  "qid|qid_283192057145502214|category|游|戏|网|络|游|戏|title|精|神|专|注|怎|么|给|自|己|用|啊|我|刚|刚|学|会|可|惜|不|能|给|自|己|用|怎|么|办|啊|desc|专|注|怎|么|给|自|己|用|啊|我|刚|刚|学|会|可|惜|不|能|给|自|己|用|怎|么|办|啊|answer|点|击|屏|幕|左|上|角|的|你|的|名|字|然|后|在|点|技|能|啊|这|样|就|可|以|了|",
  "qid|qid_8515339931774021497|category|游|戏|单|机|游|戏|轩|辕|剑|title|轩|辕|剑|4|如|何|用|法|宝|desc|answer|一|般|来|说|法|宝|是|可|以|让|你|用|一|些|特|殊|技|能|的|在|轮|到|你|行|动|时|在|命|令|菜|单|中|可|用|r|n|被|动|技|能|的|如|死|亡|复|活|在|遇|到|相|应|状|况|法|宝|自|动|起|作|用|",
  "qid|qid_3624710425439488528|category|游|戏|盛|大|游|戏|传|奇|世|界|title|搞|个|白|金|账|号|要|什|么|东|西|desc|answer|抽|奖|等|或|买|书|有|rmb|买|也|可|以|",
  "qid|qid_4519308873129627169|category|娱|乐|宠|物|title|我|家|的|吉|娃|娃|指|甲|长|的|太|长|也|买|了|专|用|的|剪|指|钳|但|是|它|就|是|不|让|剪|desc|answer|我|家|狗|狗|也|是|不|让|剪|主|要|原|因|是|因|为|狗|狗|的|指|甲|都|很|敏|感|所|以|我|买|了|一|个|宠|物|专|用|的|雷|达|罩|学|名|叫|什|么|我|也|不|知|道|只|是|觉|得|那|东|西|戴|在|狗|狗|身|上|就|跟|一|个|雷|达|一|样|狗|狗|戴|上|那|个|一|个|你|让|你|家|人|扶|住|他|的|脖|子|让|他|别|看|到|你|在|剪|他|的|指|甲|他|自|然|就|安|静|了|然|后|剪|的|时|候|尽|量|别|剪|太|深|但|是|下|手|要|快|不|要|犹|豫|否|则|他|会|警|觉|的|反|正|我|家|的|狗|狗|以|后|剪|指|甲|的|时|候|就|再|也|没|闹|过|去|宠|物|美|容|太|远|每|次|还|花|钱|不|值|得|其|他|的|也|不|一|定|能|管|用|这|招|其|实|挺|管|用|的|一|般|小|商|品|市|场|都|有|卖|而|且|不|贵|",
  "qid|qid_8811094182532219078|category|商|业|理|财|财|务|税|务|title|限|股|需|要|缴|纳|个|人|所|得|税|吗|去|年|转|让|的|限|售|股|需|要|缴|纳|个|人|所|得|税|吗|desc|去|年|转|让|的|限|售|股|需|要|缴|纳|个|人|所|得|税|吗|r|n|answer|根|据|财|政|部|国|家|税|务|总|局|证|监|会|关|于|个|人|转|让|上|市|公|司|限|售|股|所|得|征|收|个|人|所|得|税|有|关|问|题|的|通|知|财|税|2009|167|号|的|规|定|自|2010|年|1|月|1|日|起|对|个|人|转|让|限|售|股|取|得|的|所|得|按|照|财|产|转|让|所|得|适|用|20|的|比|例|税|率|征|收|个|人|所|得|税|r|n|r|n|个|人|转|让|限|售|股|以|每|次|转|让|限|售|股|转|让|收|入|减|除|股|票|原|值|和|合|理|税|费|后|的|余|额|为|应|纳|税|所|得|额|即|r|n|r|n|应|纳|税|所|得|额|限|售|股|转|让|收|入|限|售|股|原|值|合|理|税|费|r|n|r|n|应|纳|税|额|应|纳|税|所|得|额|20|r|n|r|n|限|售|股|原|值|是|指|限|售|股|买|入|时|的|买|入|价|及|按|照|规|定|缴|纳|的|有|关|费|用|合|理|税|费|是|指|转|让|限|售|股|过|程|中|发|生|的|印|花|税|佣|金|过|户|费|等|与|交|易|相|关|的|税|费|r|n|r|n|如|果|纳|税|人|未|能|提|供|完|整|真|实|的|限|售|股|原|值|凭|证|不|能|准|确|计|算|限|售|股|原|值|的|主|管|税|务|机|关|一|律|按|限|售|股|转|让|收|入|的|15|核|定|限|售|股|原|值|及|合|理|税|费|r|n|r|n|",
  "qid|qid_478271179941814867|category|娱|乐|宠|物|title|狗|狗|用|的|东|西|有|店|铺|推|荐|吗|谢|谢|了|可|爱|desc|用|的|东|西|有|店|铺|推|荐|吗|谢|谢|了|可|爱|answer|直|接|上|淘|宝|买|吧|很|多|的|哦|你|家|的|是|什|么|狗|狗|呢|请|点|击|下|方|的|好|评|哦|谢|谢|",
  "qid|qid_2306408878195592765|category|电|脑|网|络|硬|件|title|技|嘉|m61pme|desc|r|2g|r|主|板|技|嘉|m61pme|s2|r|现|在|想|配|一|个|显|卡|不|知|道|用|什|么|的|好|大|家|介|绍|一|下|r|性|价|比|要|高|点|的|当|然|价|格|要|便|宜|的|最|好|控|制|在|500|左|右|answer|技|嘉|m61pme|s2|采|用|nvidia|geforce6100|430|单|芯|片|组|集|成|geforce|6100|显|示|核|心|支|持|am2|接|口|cpu|不|算|太|老|还|可|以|用|跟|你|配|置|相|匹|配|的|显|卡|500|元|内|r|n1|铭|瑄|gt240|高|清|版|r|n|核|心|频|率|550mhz|r|n|显|存|频|率|2000mhz|r|n|显|存|类|型|gddr3|r|n|显|存|容|量|512|r|n|显|存|位|宽|128bit|r|n|显|存|描|述|gddr3|显|存|r|n|显|存|速|度|1ns|r|n|小|影|霸|gf4|后|羿|版|跟|上|面|差|不|多|r|n2|昂|达|hd5550|512mb|gd3|双|敏|火|旋|风|2|hd5550|v1024|小|牛|版|r|n|显|存|类|型|gddr3|r|n|核|心|频|率|550mhz|r|n|显|存|频|率|1600mhz|r|n|显|存|容|量|512|r|n|显|存|位|宽|128bit|r|n|显|存|速|度|1.1ns|r|n|双|敏|火|旋|风|2|hd5550|v1024|小|牛|版|跟|上|面|差|不|多|r|n|上|述|是|比|较|新|的|显|卡|够|用|就|行|",
  "qid|qid_3313689307220512171|category|健|康|内|科|肾|内|科|title|请|大|家|帮|忙|大|谢|诸|位|了|生|殖|器|尿|道|中|间|部|位|海|面|脆|骨|上|有|一|个|小|黄|豆|大|小|desc|生|殖|器|尿|道|中|间|部|位|海|面|脆|骨|上|有|一|个|小|黄|豆|大|小|的|小|疙|瘩|在|里|面|从|外|面|用|手|摸|能|感|觉|到|不|痛|不|痒|不|妨|碍|小|便|两|年|多|了|是|什|麽|病|怎|样|治|疗|还|有|左|耳|朵|根|儿|长|了|两|个|小|瘤|有|一|次|发|炎|了|去|医|院|开|了|点|儿|黄|金|膏|抹|了|消|肿|了|黄|金|膏|是|黑|色|的|抹|完|后|粉|瘤|也|变|黑|了|怕|手|术|不|理|它|没|事|吧|我|岁|数|36|结|婚|8|年|身|高|1.67|体|重|68|公|斤|谢|诸|位|answer|您|好|r|n|病|情|盲|目|猜|测|没|意|义|r|n|建|议|应|及|早|到|医|院|进|一|步|检|查|明|确|r|n|检|查|后|遵|医|嘱|对|症|采|取|针|对|性|的|治|疗|即|可|",
  "qid|qid_2038844104592063340|category|电|脑|网|络|互|联|网|上|网|帮|助|title|考|试|cpa|考|过|的|请|指|教|经|验|desc|考|过|的|请|指|教|经|验|answer|这|个|容|易|办|紧|扣|教|材|万|变|不|离|其|中|",
  "qid|qid_7209364090176418758|category|生|活|美|容|塑|身|化|妆|title|韩|智|敏|代|言|的|梦|妆|bb|霜|效|果|怎|么|样|我|想|买|呢|一|直|很|喜|欢|韩|智|敏|desc|一|直|很|喜|欢|韩|智|敏|人|长|得|靓|丽|也|水|当|当|她|代|言|的|梦|妆|品|牌|我|也|一|直|很|喜|欢|最|近|广|告|常|播|的|梦|妆|bb|霜|我|正|打|算|买|来|试|试|不|知|道|效|果|怎|么|样|呢|求|真|相|谢|谢|啦|answer|楼|主|很|巧|哟|我|现|在|正|在|用|的|就|是|梦|妆|的|bb|霜|呢|它|分|清|爽|型|和|滋|润|型|两|种|所|以|你|要|根|据|自|己|肌|肤|的|类|型|来|选|择|我|选|的|是|那|款|滋|润|型|的|多|效|bb|矿|物|修|容|霜|防|晒|指|数|spf20|pa|我|觉|得|很|适|合|我|的|皮|肤|它|里|面|含|有|智|臻|繁|花|复|合|体|用|过|她|家|多|效|系|列|的|护|肤|品|所|以|对|这|个|成|分|很|熟|悉|哦|能|够|帮|助|增|强|肌|肤|弹|性|有|效|改|善|肌|肤|黯|沉|及|瑕|疵|最|近|朋|友|都|说|我|肤|色|看|起|来|亮|很|多|整|个|皮|肤|感|觉|qq|的|很|有|弹|性|嘿|嘿|还|有|那|个|复|合|矿|物|元|素|是|天|然|的|成|分|哦|不|会|让|色|素|在|脸|上|沉|积|长|时|间|使|用|都|不|用|担|心|会|伤|害|皮|肤|而|且|质|地|比|较|柔|滑|盈|润|不|容|易|堆|积|在|脸|上|轻|轻|一|抹|就|化|开|了|用|它|修|饰|肤|色|想|要|拥|有|一|个|完|美|裸|妆|肤|完|全|不|是|梦|啊|所|以|我|化|妆|的|时|候|遮|瑕|膏|粉|底|什|么|的|都|省|了|直|接|用|它|一|步|就|搞|定|了|它|还|有|很|好|的|防|晒|效|果|uvb|uva|都|能|全|部|隔|离|到|户|外|或|者|是|面|对|电|脑|工|作|带|着|它|特|别|的|方|便|和|实|用|所|以|不|得|不|心|水|它|啊|",
  "qid|qid_4925855189856708274|category|电|脑|网|络|互|联|网|上|网|帮|助|title|请|问|怎|样|才|能|获|的|更|高|积|分|desc|answer|多|问|多|答|发|挥|想|象|力|当|然|不|要|乱|答|不|然|别|人|会|扔|鸡|蛋|了|r|n|总|之|这|里|是|个|帮|助|别|人|的|地|方|争|强|好|胜|是|可|以|的|但|是|不|要|用|非|法|手|段|",
  "qid|qid_5438551119871866263|category|教|育|科|学|理|工|学|科|数|学|title|这|数|学|题|你|会|吗|关|于|x|的|方|程|x|2|desc|关|于|x|的|方|程|x|2|x|m|在|1,1|上|有|实|数|解|则|m|的|取|值|是|answer|连|个|方|程|都|没|有|r|n|令|f|x|x|2|x|m|讨|论|r|n|如|果|在|1,1|上|只|有|一|个|根|需|要|f|1|f|1|0|r|n|所|以|得|到|2|m|0|r|n|如|果|在|1,1|上|有|两|个|根|需|要|r|n|对|称|轴|在|1,1|上|已|满|足|判|别|式|为|非|负|f|1|0|f|1|0|r|n|得|到|0|m|1|4|r|n|所|以|综|上|m|的|范|围|是|2|m|1|4|",
  "qid|qid_655643042658033111|category|游|戏|单|机|游|戏|魔|兽|争|霸|title|1.21|版|本|什|么|时|候|出|desc|1.21|版|本|什|么|时|候|出|answer|不|知|道|啊|r|n|应|该|不|会|太|晚|吧|r|n1.20|的|玩|着|也|不|错|啊|",
  "qid|qid_1081892684767137198|category|生|活|育|儿|title|3|岁|小|孩|拉|肚|子|拉|出|白|浆|状|有|点|咳|嗽|咳|嗽|后|容|易|拉|出|白|色|浆|状|物|这|desc|有|点|咳|嗽|后|容|易|拉|出|白|色|浆|状|物|这|是|怎|么|回|事|她|肚|子|不|痛|answer|带|宝|宝|去|化|验|一|下|大|便|",
  "qid|qid_8725600614872540890|category|游|戏|腾|讯|游|戏|qq|幻|想|title|现|在|新|出|的|法|宝|药|师|四|十|五|级|想|去|买|个|法|宝|去|什|么|地|方|可|以|买|得|到|desc|四|十|五|级|r|r|想|去|买|个|法|宝|r|去|什|么|地|方|可|以|买|得|到|r|要|花|多|少|钱|r|还|有|就|是|买|什|么|最|好|r|买|来|是|带|什|么|地|方|r|是|武|器|那|里|吗|answer|想|买|法|宝|到|楼|兰|城|郊|地|图|最|左|上|脚|荣|光|使|者|那|里|买|你|45|要|是|名|誉|有|1001|就|买|个|清|气|净|璃|瓶|没|有|1001|就|买|个|碧|如|意|吧|价|钱|不|贵|具|体|多|少|我|忘|了|但|你|肯|定|买|的|起|法|宝|带|在|更|新|后|一|个|新|的|空|格|r|n|现|在|最|高|的|都|是|60|级|的|仙|器|套|装|",
  "qid|qid_5527707343652499605|category|生|活|保|健|养|生|title|关|于|eft|保|健|品|事|还|要|发|行|股|票|是|真|的|我|妈|妈|正|在|做|eft|谁|知|desc|我|妈|妈|正|在|做|eft|谁|知|道|这|到|底|是|不|是|骗|人|的|把|戏|最|近|又|说|eft|要|发|行|得|赶|紧|买|货|好|成|为|会|员|才|能|享|受|到|将|来|的|股|票|大|量|收|益|等|等|谁|能|告|诉|我|此|事|的|后|果|answer|我|做|过|千|万|不|要|买|因|为|直|销|员|和|公|司|之|间|的|关|系|不|对|等|消|息|也|可|能|是|假|的|只|是|公|司|为|了|促|销|过|一|阵|子|再|看|看|r|n|电|子|商|务|双|轨|制|公|司|价|格|太|贵|不|符|合|中|国|国|情|r|n|在|美|国|老|板|叫|秦|杰|r|n|和|r|n|安|旗|公|司|一|样|如|果|走|传|统|开|店|的|路|就|会|被|工|商|局|查|封|对|这|些|公|司|不|要|太|相|信|",
  "qid|qid_2120993978482268954|category|教|育|科|学|学|习|帮|助|title|商|鞅|变|法|和|北|魏|孝|文|帝|改|革|都|获|得|成|功|的|主|要|原|因|a|顺|应|历|史|发|展|趋|势|b|desc|a|顺|应|发|展|趋|势|rb|符|合|封|建|统|治|阶|级|的|利|益|rc|代|表|广|大|人|民|的|利|益|rd|得|到|统|治|者|的|支|持|r|答|案|b|为|什|么|acd|不|对|answer|1|商|鞅|变|法|符|合|秦|国|的|国|情|实|质|上|变|法|是|为|了|维|护|君|主|专|制|加|强|维|护|国|君|统|治|人|民|的|权|力|维|护|国|君|至|高|无|上|的|地|位|因|此|得|到|秦|孝|公|的|大|力|支|持|取|到|变|法|成|功|的|第|一|步|r|n2|秦|国|当|时|是|一|个|农|业|落|后|氏|族|势|力|很|深|的|国|家|商|鞅|主|要|起|内|容|是|发|展|农|业|生|产|符|合|秦|国|的|需|要|即|成|功|的|第|二|步|r|n3|推|行|法|家|思|想|万|事|皆|决|于|法|在|一|定|程|度|上|缓|和|秦|国|的|国|内|矛|盾|r|n|所|以|ab|正|确|r|nb|项|当|时|商|鞅|变|法|得|到|秦|孝|公|的|支|持|北|魏|孝|文|帝|改|革|随|着|国|内|矛|盾|和|民|族|矛|盾|上|升|北|魏|统|治|者|认|识|当|只|有|进|行|汉|化|改|革|才|符|合|自|己|统|治|r|nc|无|论|当|时|怎|么|变|法|都|是|维|护|统|治|阶|级|利|益|不|可|能|代|表|广|大|人|民|利|益|r|nd|商|鞅|变|法|得|到|秦|孝|公|支|持|北|魏|孝|文|帝|改|革|自|身|就|是|统|治|者|",
  "qid|qid_9110737097744485302|category|生|活|起|名|title|饮|酒|与|怀|孕|我|想|生|个|猪|宝|宝|请|问|什|么|时|候|开|始|已|经|不|能|再|喝|酒|了|因|为|desc|我|想|生|个|宝|宝|请|问|什|么|时|候|开|始|已|经|不|能|再|喝|酒|了|因|为|前|几|天|还|喝|了|酒|answer|喝|酒|后|4|周|再|进|行|此|项|计|划|而|且|这|其|中|不|能|再|碰|酒|了|",
  "qid|qid_3913105430061039865|category|娱|乐|博|彩|title|请|教|亚|盘|各|位|大|师|网|上|哪|里|可|以|玩|澳|门|的|上|下|盘|呢|谢|谢|了|desc|各|位|大|师|网|上|哪|里|可|以|玩|澳|门|的|上|下|盘|呢|谢|谢|了|answer|可|以|玩|但|提|款|不|方|便|",
  "qid|qid_8325162146787472205|category|娱|乐|博|彩|title|第|一|次|发|图|图|2|请|高|手|进|来|批|一|下|第|一|次|发|图|拍|摄|不|好|desc|第|一|次|发|图|不|好|虽|见|笑|谢|谢|进|来|帮|批|示|的|朋|友|r|answer|6|12|场|感|觉|挺|悬|的|祝|好|"
};

static void testCompLucene95(CuTest *tc) {
  std::vector<std::string> new_tokens;
  testCutLines(datas, new_tokens);

  CLUCENE_ASSERT((tokens == new_tokens));
}

CuSuite *teststandard95(void) {
  CuSuite *suite = CuSuiteNew(_T("CLucene Standard95 Test"));

  SUITE_ADD_TEST(suite, testStandardTokenizer);
  SUITE_ADD_TEST(suite, testStandardTokenizerMaxTokenLength);
  SUITE_ADD_TEST(suite, testStandardTokenizerMaxTokenLength1);
  SUITE_ADD_TEST(suite, testUtf8Str);
  SUITE_ADD_TEST(suite, testArmenian);
  SUITE_ADD_TEST(suite, testAmharic);
  SUITE_ADD_TEST(suite, testBengali);
  SUITE_ADD_TEST(suite, testFarsi);
  SUITE_ADD_TEST(suite, testGreek);
  SUITE_ADD_TEST(suite, testThai);
  SUITE_ADD_TEST(suite, testLao);
  SUITE_ADD_TEST(suite, testTibetan);
  SUITE_ADD_TEST(suite, testChinese);
  SUITE_ADD_TEST(suite, testCompLucene95);
  return suite;
}