// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.doris.nereids.trees.expressions.functions.scalar;

import org.apache.doris.catalog.FunctionSignature;
import org.apache.doris.nereids.trees.expressions.Expression;
import org.apache.doris.nereids.trees.expressions.functions.AlwaysNotNullable;
import org.apache.doris.nereids.trees.expressions.functions.ExplicitlyCastableSignature;
import org.apache.doris.nereids.trees.expressions.visitor.ExpressionVisitor;
import org.apache.doris.nereids.types.ArrayType;
import org.apache.doris.nereids.types.BigIntType;
import org.apache.doris.nereids.types.BooleanType;
import org.apache.doris.nereids.types.DataType;
import org.apache.doris.nereids.types.DateTimeType;
import org.apache.doris.nereids.types.DateTimeV2Type;
import org.apache.doris.nereids.types.DateType;
import org.apache.doris.nereids.types.DateV2Type;
import org.apache.doris.nereids.types.DecimalV2Type;
import org.apache.doris.nereids.types.DecimalV3Type;
import org.apache.doris.nereids.types.DoubleType;
import org.apache.doris.nereids.types.FloatType;
import org.apache.doris.nereids.types.IntegerType;
import org.apache.doris.nereids.types.LargeIntType;
import org.apache.doris.nereids.types.SmallIntType;
import org.apache.doris.nereids.types.StringType;
import org.apache.doris.nereids.types.TinyIntType;
import org.apache.doris.nereids.types.VarcharType;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;

import java.util.List;

/**
 * ScalarFunction 'array'. This class is generated by GenerateFunction.
 */
public class Array extends ScalarFunction
        implements ExplicitlyCastableSignature, AlwaysNotNullable {

    public static final List<FunctionSignature> SIGNATURES = ImmutableList.of(
            FunctionSignature.ret(ArrayType.SYSTEM_DEFAULT).args(),
            FunctionSignature.ret(ArrayType.of(BooleanType.INSTANCE)).varArgs(BooleanType.INSTANCE),
            FunctionSignature.ret(ArrayType.of(TinyIntType.INSTANCE)).varArgs(TinyIntType.INSTANCE),
            FunctionSignature.ret(ArrayType.of(SmallIntType.INSTANCE)).varArgs(SmallIntType.INSTANCE),
            FunctionSignature.ret(ArrayType.of(IntegerType.INSTANCE)).varArgs(IntegerType.INSTANCE),
            FunctionSignature.ret(ArrayType.of(BigIntType.INSTANCE)).varArgs(BigIntType.INSTANCE),
            FunctionSignature.ret(ArrayType.of(LargeIntType.INSTANCE)).varArgs(LargeIntType.INSTANCE),
            FunctionSignature.ret(ArrayType.of(DateTimeType.INSTANCE)).varArgs(DateTimeType.INSTANCE),
            FunctionSignature.ret(ArrayType.of(DateType.INSTANCE)).varArgs(DateType.INSTANCE),
            FunctionSignature.ret(ArrayType.of(DateTimeV2Type.SYSTEM_DEFAULT)).varArgs(DateTimeV2Type.SYSTEM_DEFAULT),
            FunctionSignature.ret(ArrayType.of(DateV2Type.INSTANCE)).varArgs(DateV2Type.INSTANCE),
            FunctionSignature.ret(ArrayType.of(FloatType.INSTANCE)).varArgs(FloatType.INSTANCE),
            FunctionSignature.ret(ArrayType.of(DoubleType.INSTANCE)).varArgs(DoubleType.INSTANCE),
            FunctionSignature.ret(ArrayType.of(DecimalV2Type.SYSTEM_DEFAULT)).varArgs(DecimalV2Type.SYSTEM_DEFAULT),
            FunctionSignature.ret(ArrayType.of(DecimalV3Type.WILDCARD)).varArgs(DecimalV3Type.WILDCARD),
            FunctionSignature.ret(ArrayType.of(VarcharType.SYSTEM_DEFAULT)).varArgs(VarcharType.SYSTEM_DEFAULT),
            FunctionSignature.ret(ArrayType.of(StringType.INSTANCE)).varArgs(StringType.INSTANCE)
    );

    /**
     * constructor with 0 or more arguments.
     */
    public Array(Expression... varArgs) {
        super("array", varArgs);
    }

    @Override
    public FunctionSignature computeSignature(FunctionSignature signature) {
        if (arity() > 0) {
            signature = signature.withReturnType(ArrayType.of(getArgumentType(0)));
        }
        return super.computeSignature(signature);
    }

    @Override
    protected Expression uncheckedCastTo(DataType targetType) {
        if (targetType instanceof ArrayType) {
            ArrayType arrayType = (ArrayType) targetType;
            DataType castItemType = arrayType.getItemType();
            boolean hasCast = false;
            List<Expression> newArguments = Lists.newArrayList();
            for (int i = 0; i < arity(); i++) {
                Expression argument = getArgument(i);
                if (!argument.getDataType().equals(castItemType)) {
                    hasCast = true;
                    newArguments.add(argument.castTo(castItemType));
                } else {
                    newArguments.add(argument);
                }
            }
            return hasCast ? new Array(newArguments.stream().toArray(Expression[]::new)) : this;
        }
        return super.uncheckedCastTo(targetType);
    }

    /**
     * withChildren.
     */
    @Override
    public Array withChildren(List<Expression> children) {
        return new Array(children.stream().toArray(Expression[]::new));
    }

    @Override
    public <R, C> R accept(ExpressionVisitor<R, C> visitor, C context) {
        return visitor.visitArray(this, context);
    }

    @Override
    public List<FunctionSignature> getSignatures() {
        return SIGNATURES;
    }
}
