// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.


package org.apache.doris.datasource.hive.event;

import org.apache.doris.catalog.Env;
import org.apache.doris.common.Config;
import org.apache.doris.common.util.MasterDaemon;
import org.apache.doris.datasource.CatalogIf;
import org.apache.doris.datasource.HMSClientException;
import org.apache.doris.datasource.HMSExternalCatalog;

import org.apache.hadoop.hive.metastore.api.NoSuchObjectException;
import org.apache.hadoop.hive.metastore.api.NotificationEvent;
import org.apache.hadoop.hive.metastore.api.NotificationEventResponse;
import org.apache.hadoop.hive.metastore.messaging.MessageDeserializer;
import org.apache.hadoop.hive.metastore.messaging.json.JSONMessageDeserializer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.util.Collections;
import java.util.List;

/**
 * A metastore event is a instance of the class
 * {@link NotificationEvent}. Metastore can be
 * configured, to work with Listeners which are called on various DDL operations like
 * create/alter/drop operations on database, table, partition etc. Each event has a unique
 * incremental id and the generated events are be fetched from Metastore to get
 * incremental updates to the metadata stored in Hive metastore using the the public API
 * <code>get_next_notification</code> These events could be generated by external
 * Metastore clients like Apache Hive or Apache Spark configured to talk with the same metastore.
 * <p>
 * This class is used to poll metastore for such events at a given frequency. By observing
 * such events, we can take appropriate action on the {@link org.apache.doris.datasource.hive.HiveMetaStoreCache}
 * (refresh/invalidate/add/remove) so that represents the latest information
 * available in metastore. We keep track of the last synced event id in each polling
 * iteration so the next batch can be requested appropriately. The current batch size is
 * constant and set to {@link org.apache.doris.common.Config#hms_events_batch_size_per_rpc}.
 */
public class MetastoreEventsProcessor extends MasterDaemon {
    private static final Logger LOG = LogManager.getLogger(MetastoreEventsProcessor.class);
    public static final String HMS_ADD_THRIFT_OBJECTS_IN_EVENTS_CONFIG_KEY =
            "hive.metastore.notifications.add.thrift.objects";

    // for deserializing from JSON strings from metastore event
    private static final MessageDeserializer JSON_MESSAGE_DESERIALIZER = new JSONMessageDeserializer();
    // for deserializing from GZIP JSON strings from metastore event
    // (some HDP Hive and CDH Hive versions use this format)
    private static final MessageDeserializer GZIP_JSON_MESSAGE_DESERIALIZER = new GzipJSONMessageDeserializer();
    private static final String GZIP_JSON_FORMAT_PREFIX = "gzip";

    // event factory which is used to get or create MetastoreEvents
    private final MetastoreEventFactory metastoreEventFactory;

    private boolean isRunning;

    public MetastoreEventsProcessor() {
        super(MetastoreEventsProcessor.class.getName(), Config.hms_events_polling_interval_ms);
        this.metastoreEventFactory = new MetastoreEventFactory();
        this.isRunning = false;
    }

    /**
     * Fetch the next batch of NotificationEvents from metastore. The default batch size is
     * <code>{@link Config#hms_events_batch_size_per_rpc}</code>
     */
    private List<NotificationEvent> getNextHMSEvents(HMSExternalCatalog hmsExternalCatalog) {
        LOG.debug("Start to pull events on catalog [{}]", hmsExternalCatalog.getName());
        NotificationEventResponse response = hmsExternalCatalog.getNextEventResponse(hmsExternalCatalog);

        if (response == null) {
            return Collections.emptyList();
        }
        return response.getEvents();
    }

    private void doExecute(List<MetastoreEvent> events, HMSExternalCatalog hmsExternalCatalog) {
        for (MetastoreEvent event : events) {
            try {
                event.process();
            } catch (HMSClientException hmsClientException) {
                if (hmsClientException.getCause() != null
                        && hmsClientException.getCause() instanceof NoSuchObjectException) {
                    LOG.warn(event.debugString("Failed to process event and skip"), hmsClientException);
                } else {
                    hmsExternalCatalog.setLastSyncedEventId(event.getEventId() - 1);
                    throw hmsClientException;
                }
            } catch (Exception e) {
                hmsExternalCatalog.setLastSyncedEventId(event.getEventId() - 1);
                throw e;
            }
        }
    }

    /**
     * Process the given list of notification events. Useful for tests which provide a list of events
     */
    private void processEvents(List<NotificationEvent> events, HMSExternalCatalog hmsExternalCatalog) {
        //transfer
        List<MetastoreEvent> metastoreEvents = metastoreEventFactory.getMetastoreEvents(events, hmsExternalCatalog);
        doExecute(metastoreEvents, hmsExternalCatalog);
        hmsExternalCatalog.setLastSyncedEventId(events.get(events.size() - 1).getEventId());
    }

    @Override
    protected void runAfterCatalogReady() {
        if (isRunning) {
            LOG.warn("Last task not finished,ignore current task.");
            return;
        }
        isRunning = true;
        try {
            realRun();
        } catch (Exception ex) {
            LOG.warn("Task failed", ex);
        }
        isRunning = false;
    }

    private void realRun() {
        List<Long> catalogIds = Env.getCurrentEnv().getCatalogMgr().getCatalogIds();
        for (Long catalogId : catalogIds) {
            CatalogIf catalog = Env.getCurrentEnv().getCatalogMgr().getCatalog(catalogId);
            if (catalog instanceof HMSExternalCatalog) {
                HMSExternalCatalog hmsExternalCatalog = (HMSExternalCatalog) catalog;
                List<NotificationEvent> events = Collections.emptyList();
                try {
                    events = getNextHMSEvents(hmsExternalCatalog);
                    if (!events.isEmpty()) {
                        LOG.info("Events size are {} on catalog [{}]", events.size(),
                                hmsExternalCatalog.getName());
                        processEvents(events, hmsExternalCatalog);
                    }
                } catch (MetastoreNotificationFetchException e) {
                    LOG.warn("Failed to fetch hms events on {}. msg: ", hmsExternalCatalog.getName(), e);
                } catch (Exception ex) {
                    LOG.warn("Failed to process hive metastore [{}] events .",
                            hmsExternalCatalog.getName(), ex);
                }
            }
        }
    }

    public static MessageDeserializer getMessageDeserializer(String messageFormat) {
        if (messageFormat != null && messageFormat.startsWith(GZIP_JSON_FORMAT_PREFIX)) {
            return GZIP_JSON_MESSAGE_DESERIALIZER;
        }
        return JSON_MESSAGE_DESERIALIZER;
    }
}
