/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.obs;

import com.google.common.base.Preconditions;
import com.obs.services.ObsClient;
import com.obs.services.exception.ObsException;
import com.obs.services.model.GetObjectRequest;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.fs.CanSetReadahead;
import org.apache.hadoop.fs.FSInputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.obs.OBSFileSystem;
import org.apache.hadoop.fs.obs.OBSInputPolicy;
import org.apache.hadoop.fs.obs.OBSInstrumentation;
import org.apache.hadoop.fs.obs.OBSUtils;
import org.slf4j.Logger;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class OBSInputStream
extends FSInputStream
implements CanSetReadahead {
    private static final int RETRY_TIME = 3;
    private static final int DELAY_TIME = 100;
    private long pos;
    private volatile boolean closed;
    private InputStream wrappedStream;
    private final FileSystem.Statistics stats;
    private final ObsClient client;
    private final String bucket;
    private final String key;
    private final long contentLength;
    private final String uri;
    public static final Logger LOG = OBSFileSystem.LOG;
    private final OBSInstrumentation.InputStreamStatistics streamStatistics;
    private final OBSInputPolicy inputPolicy;
    private long readahead = 65536L;
    private long nextReadPos;
    private long contentRangeFinish;
    private long contentRangeStart;

    public OBSInputStream(String bucket, String key, long contentLength, ObsClient client, FileSystem.Statistics stats, OBSInstrumentation instrumentation, long readahead, OBSInputPolicy inputPolicy) {
        Preconditions.checkArgument((boolean)StringUtils.isNotEmpty((String)bucket), (Object)"No Bucket");
        Preconditions.checkArgument((boolean)StringUtils.isNotEmpty((String)key), (Object)"No Key");
        Preconditions.checkArgument((contentLength >= 0L ? 1 : 0) != 0, (Object)"Negative content length");
        this.bucket = bucket;
        this.key = key;
        this.contentLength = contentLength;
        this.client = client;
        this.stats = stats;
        this.uri = "obs://" + this.bucket + "/" + this.key;
        this.streamStatistics = instrumentation.newInputStreamStatistics();
        this.inputPolicy = inputPolicy;
        this.setReadahead(readahead);
    }

    private synchronized void reopen(String reason, long targetPos, long length) throws IOException {
        if (this.wrappedStream != null) {
            this.closeStream("reopen(" + reason + ")", this.contentRangeFinish);
        }
        this.contentRangeFinish = OBSInputStream.calculateRequestLimit(this.inputPolicy, targetPos, length, this.contentLength, this.readahead);
        LOG.debug("reopen({}) for {} range[{}-{}], length={}, streamPosition={}, nextReadPosition={}", new Object[]{this.uri, reason, targetPos, this.contentRangeFinish, length, this.pos, this.nextReadPos});
        this.streamStatistics.streamOpened();
        try {
            GetObjectRequest request = new GetObjectRequest(this.bucket, this.key);
            request.setRangeStart(Long.valueOf(targetPos));
            request.setRangeEnd(Long.valueOf(this.contentRangeFinish));
            this.wrappedStream = this.client.getObject(request).getObjectContent();
            this.contentRangeStart = targetPos;
            if (this.wrappedStream == null) {
                throw new IOException("Null IO stream from reopen of (" + reason + ") " + this.uri);
            }
        }
        catch (ObsException e) {
            throw OBSUtils.translateException("Reopen at position " + targetPos, this.uri, e);
        }
        this.pos = targetPos;
    }

    public synchronized long getPos() throws IOException {
        return this.nextReadPos < 0L ? 0L : this.nextReadPos;
    }

    public synchronized void seek(long targetPos) throws IOException {
        this.checkNotClosed();
        if (targetPos < 0L) {
            throw new EOFException("Cannot seek to a negative offset " + targetPos);
        }
        if (this.contentLength <= 0L) {
            return;
        }
        this.nextReadPos = targetPos;
    }

    private void seekQuietly(long positiveTargetPos) {
        try {
            this.seek(positiveTargetPos);
        }
        catch (IOException ioe) {
            LOG.debug("Ignoring IOE on seek of {} to {}", new Object[]{this.uri, positiveTargetPos, ioe});
        }
    }

    private void seekInStream(long targetPos, long length) throws IOException {
        this.checkNotClosed();
        if (this.wrappedStream == null) {
            return;
        }
        long diff = targetPos - this.pos;
        if (diff > 0L) {
            boolean skipForward;
            int available = this.wrappedStream.available();
            long forwardSeekRange = Math.max(this.readahead, (long)available);
            long remainingInCurrentRequest = this.remainingInCurrentRequest();
            long forwardSeekLimit = Math.min(remainingInCurrentRequest, forwardSeekRange);
            boolean bl = skipForward = remainingInCurrentRequest > 0L && diff <= forwardSeekLimit;
            if (skipForward) {
                LOG.debug("Forward seek on {}, of {} bytes", (Object)this.uri, (Object)diff);
                this.streamStatistics.seekForwards(diff);
                long skipped = this.wrappedStream.skip(diff);
                if (skipped > 0L) {
                    this.pos += skipped;
                    this.incrementBytesRead(diff);
                }
                if (this.pos == targetPos) {
                    return;
                }
                LOG.warn("Failed to seek on {} to {}. Current position {}", new Object[]{this.uri, targetPos, this.pos});
            }
        } else if (diff < 0L) {
            this.streamStatistics.seekBackwards(diff);
        } else if (this.remainingInCurrentRequest() > 0L) {
            return;
        }
        this.closeStream("seekInStream()", this.contentRangeFinish);
        this.pos = targetPos;
    }

    public boolean seekToNewSource(long targetPos) throws IOException {
        return false;
    }

    private void lazySeek(long targetPos, long len) throws IOException {
        for (int i = 0; i < 3; ++i) {
            try {
                this.seekInStream(targetPos, len);
                if (this.wrappedStream != null) break;
                this.reopen("read from new offset", targetPos, len);
                break;
            }
            catch (EOFException e) {
                LOG.warn("IOException occurred in lazySeek", (Throwable)e);
                if (i == 2) {
                    throw e;
                }
                try {
                    Thread.sleep(100L);
                    continue;
                }
                catch (InterruptedException ie) {
                    throw e;
                }
            }
        }
    }

    private void incrementBytesRead(long bytesRead) {
        this.streamStatistics.bytesRead(bytesRead);
        if (this.stats != null && bytesRead > 0L) {
            this.stats.incrementBytesRead(bytesRead);
        }
    }

    public synchronized int read() throws IOException {
        int byteRead;
        this.checkNotClosed();
        if (this.contentLength == 0L || this.nextReadPos >= this.contentLength) {
            return -1;
        }
        try {
            this.lazySeek(this.nextReadPos, 1L);
            byteRead = this.wrappedStream.read();
        }
        catch (EOFException e) {
            return -1;
        }
        catch (IOException e) {
            this.onReadFailure(e, 1);
            byteRead = this.wrappedStream.read();
        }
        if (byteRead >= 0) {
            ++this.pos;
            ++this.nextReadPos;
        }
        if (byteRead >= 0) {
            this.incrementBytesRead(1L);
        }
        return byteRead;
    }

    private void onReadFailure(IOException ioe, int length) throws IOException {
        LOG.info("Got exception while trying to read from stream {} trying to recover: " + ioe, (Object)this.uri);
        LOG.debug("While trying to read from stream {}", (Object)this.uri, (Object)ioe);
        this.streamStatistics.readException();
        this.reopen("failure recovery", this.pos, length);
    }

    public synchronized int read(byte[] buf, int off, int len) throws IOException {
        int bytesRead;
        this.checkNotClosed();
        this.validatePositionedReadArgs(this.nextReadPos, buf, off, len);
        if (len == 0) {
            return 0;
        }
        if (this.contentLength == 0L || this.nextReadPos >= this.contentLength) {
            return -1;
        }
        try {
            this.lazySeek(this.nextReadPos, len);
        }
        catch (EOFException e) {
            return -1;
        }
        try {
            this.streamStatistics.readOperationStarted(this.nextReadPos, len);
            bytesRead = this.wrappedStream.read(buf, off, len);
        }
        catch (EOFException e) {
            this.onReadFailure(e, len);
            return -1;
        }
        catch (IOException e) {
            this.onReadFailure(e, len);
            bytesRead = this.wrappedStream.read(buf, off, len);
        }
        if (bytesRead > 0) {
            this.pos += (long)bytesRead;
            this.nextReadPos += (long)bytesRead;
        }
        this.incrementBytesRead(bytesRead);
        this.streamStatistics.readOperationCompleted(len, bytesRead);
        return bytesRead;
    }

    private void checkNotClosed() throws IOException {
        if (this.closed) {
            throw new IOException(this.uri + ": " + "Stream is closed!");
        }
    }

    public synchronized void close() throws IOException {
        if (!this.closed) {
            this.closed = true;
            try {
                this.closeStream("close() operation", this.contentRangeFinish);
                super.close();
            }
            finally {
                this.streamStatistics.close();
            }
        }
    }

    private void closeStream(String reason, long length) {
        if (this.wrappedStream != null) {
            try {
                this.wrappedStream.close();
                this.streamStatistics.streamClose();
            }
            catch (IOException e) {
                LOG.debug("When closing {} stream for {}", new Object[]{this.uri, reason, e});
            }
            LOG.debug("Stream {} : {}; streamPos={}, nextReadPos={}, request range {}-{} length={}", new Object[]{this.uri, reason, this.pos, this.nextReadPos, this.contentRangeStart, this.contentRangeFinish, length});
            this.wrappedStream = null;
        }
    }

    @InterfaceStability.Unstable
    public synchronized boolean resetConnection() throws IOException {
        boolean connectionOpen;
        this.checkNotClosed();
        boolean bl = connectionOpen = this.wrappedStream != null;
        if (connectionOpen) {
            LOG.info("Forced reset of connection to {}", (Object)this.uri);
            this.closeStream("reset()", this.contentRangeFinish);
        }
        return connectionOpen;
    }

    public synchronized int available() throws IOException {
        this.checkNotClosed();
        long remaining = this.remainingInFile();
        if (remaining > Integer.MAX_VALUE) {
            return Integer.MAX_VALUE;
        }
        return (int)remaining;
    }

    @InterfaceAudience.Private
    @InterfaceStability.Unstable
    public synchronized long remainingInFile() {
        return this.contentLength - this.pos;
    }

    @InterfaceAudience.Private
    @InterfaceStability.Unstable
    public synchronized long remainingInCurrentRequest() {
        return this.contentRangeFinish - this.pos;
    }

    @InterfaceAudience.Private
    @InterfaceStability.Unstable
    public synchronized long getContentRangeFinish() {
        return this.contentRangeFinish;
    }

    @InterfaceAudience.Private
    @InterfaceStability.Unstable
    public synchronized long getContentRangeStart() {
        return this.contentRangeStart;
    }

    public boolean markSupported() {
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @InterfaceStability.Unstable
    public String toString() {
        String s = this.streamStatistics.toString();
        OBSInputStream oBSInputStream = this;
        synchronized (oBSInputStream) {
            StringBuilder sb = new StringBuilder("OBSInputStream{");
            sb.append(this.uri);
            sb.append(" wrappedStream=").append(this.wrappedStream != null ? "open" : "closed");
            sb.append(" read policy=").append((Object)this.inputPolicy);
            sb.append(" pos=").append(this.pos);
            sb.append(" nextReadPos=").append(this.nextReadPos);
            sb.append(" contentLength=").append(this.contentLength);
            sb.append(" contentRangeStart=").append(this.contentRangeStart);
            sb.append(" contentRangeFinish=").append(this.contentRangeFinish);
            sb.append(" remainingInCurrentRequest=").append(this.remainingInCurrentRequest());
            sb.append('\n').append(s);
            sb.append('}');
            return sb.toString();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void readFully(long position, byte[] buffer, int offset, int length) throws IOException {
        this.checkNotClosed();
        this.validatePositionedReadArgs(position, buffer, offset, length);
        this.streamStatistics.readFullyOperationStarted(position, length);
        if (length == 0) {
            return;
        }
        OBSInputStream oBSInputStream = this;
        synchronized (oBSInputStream) {
            long oldPos = this.getPos();
            try {
                int nbytes;
                this.seek(position);
                for (int nread = 0; nread < length; nread += nbytes) {
                    nbytes = this.read(buffer, offset + nread, length - nread);
                    if (nbytes >= 0) continue;
                    throw new EOFException("End of file reached before reading fully.");
                }
            }
            finally {
                this.seekQuietly(oldPos);
            }
        }
    }

    @InterfaceAudience.Private
    @InterfaceStability.Unstable
    public OBSInstrumentation.InputStreamStatistics getS3AStreamStatistics() {
        return this.streamStatistics;
    }

    public synchronized void setReadahead(Long readahead) {
        if (readahead == null) {
            this.readahead = 65536L;
        } else {
            Preconditions.checkArgument((readahead >= 0L ? 1 : 0) != 0, (Object)"Negative readahead value");
            this.readahead = readahead;
        }
    }

    public synchronized long getReadahead() {
        return this.readahead;
    }

    static long calculateRequestLimit(OBSInputPolicy inputPolicy, long targetPos, long length, long contentLength, long readahead) {
        long rangeLimit;
        switch (inputPolicy) {
            case Random: {
                rangeLimit = length < 0L ? contentLength : targetPos + Math.max(readahead, length);
                break;
            }
            case Sequential: {
                rangeLimit = contentLength;
                break;
            }
            default: {
                rangeLimit = contentLength;
            }
        }
        rangeLimit = Math.min(contentLength, rangeLimit);
        return rangeLimit;
    }
}

