/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.obs;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.SettableFuture;
import com.obs.services.ObsClient;
import com.obs.services.exception.ObsException;
import com.obs.services.model.AbortMultipartUploadRequest;
import com.obs.services.model.CompleteMultipartUploadRequest;
import com.obs.services.model.CompleteMultipartUploadResult;
import com.obs.services.model.CopyObjectRequest;
import com.obs.services.model.CopyObjectResult;
import com.obs.services.model.CopyPartRequest;
import com.obs.services.model.CopyPartResult;
import com.obs.services.model.DeleteObjectsRequest;
import com.obs.services.model.DeleteObjectsResult;
import com.obs.services.model.InitiateMultipartUploadRequest;
import com.obs.services.model.InitiateMultipartUploadResult;
import com.obs.services.model.KeyAndVersion;
import com.obs.services.model.ListMultipartUploadsRequest;
import com.obs.services.model.ListObjectsRequest;
import com.obs.services.model.MultipartUpload;
import com.obs.services.model.MultipartUploadListing;
import com.obs.services.model.ObjectListing;
import com.obs.services.model.ObjectMetadata;
import com.obs.services.model.ObsObject;
import com.obs.services.model.PartEtag;
import com.obs.services.model.PutObjectRequest;
import com.obs.services.model.PutObjectResult;
import com.obs.services.model.UploadPartRequest;
import com.obs.services.model.UploadPartResult;
import com.obs.services.model.fs.FSStatusEnum;
import com.obs.services.model.fs.GetAttributeRequest;
import com.obs.services.model.fs.GetBucketFSStatusRequest;
import com.obs.services.model.fs.GetBucketFSStatusResult;
import com.obs.services.model.fs.NewFileRequest;
import com.obs.services.model.fs.NewFolderRequest;
import com.obs.services.model.fs.ObsFSAttribute;
import com.obs.services.model.fs.ObsFSFile;
import com.obs.services.model.fs.ObsFSFolder;
import com.obs.services.model.fs.RenameRequest;
import com.obs.services.model.fs.RenameResult;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.EnumSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.GlobalStorageStatistics;
import org.apache.hadoop.fs.InvalidRequestException;
import org.apache.hadoop.fs.LocalDirAllocator;
import org.apache.hadoop.fs.LocalFileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.fs.PathIOException;
import org.apache.hadoop.fs.PathIsNotEmptyDirectoryException;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.StorageStatistics;
import org.apache.hadoop.fs.obs.BlockingThreadPoolExecutorService;
import org.apache.hadoop.fs.obs.Constants;
import org.apache.hadoop.fs.obs.Listing;
import org.apache.hadoop.fs.obs.OBSBlockOutputStream;
import org.apache.hadoop.fs.obs.OBSDataBlocks;
import org.apache.hadoop.fs.obs.OBSFileStatus;
import org.apache.hadoop.fs.obs.OBSInputPolicy;
import org.apache.hadoop.fs.obs.OBSInputStream;
import org.apache.hadoop.fs.obs.OBSInstrumentation;
import org.apache.hadoop.fs.obs.OBSLoginHelper;
import org.apache.hadoop.fs.obs.OBSOutputStream;
import org.apache.hadoop.fs.obs.OBSReadaheadInputStream;
import org.apache.hadoop.fs.obs.OBSStorageStatistics;
import org.apache.hadoop.fs.obs.OBSUtils;
import org.apache.hadoop.fs.obs.ObsClientFactory;
import org.apache.hadoop.fs.obs.RenameFailedException;
import org.apache.hadoop.fs.obs.SemaphoredDelegatingExecutor;
import org.apache.hadoop.fs.obs.Statistic;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.util.Progressable;
import org.apache.hadoop.util.ReflectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class OBSFileSystem
extends FileSystem {
    public static final int DEFAULT_BLOCKSIZE = 0x2000000;
    private URI uri;
    private Path workingDir;
    private String username;
    private ObsClient obs;
    private boolean enablePosix = false;
    private boolean enableMultiObjectsDeleteRecursion = true;
    private boolean renameSupportEmptyDestinationFolder = true;
    private String bucket;
    private int maxKeys;
    private Listing listing;
    private long partSize;
    private boolean enableMultiObjectsDelete;
    private ListeningExecutorService boundedThreadPool;
    private ListeningExecutorService boundedCopyThreadPool;
    private ListeningExecutorService boundedDeleteThreadPool;
    private ThreadPoolExecutor unboundedReadThreadPool;
    private ExecutorService unboundedThreadPool;
    private ListeningExecutorService boundedCopyPartThreadPool;
    private long multiPartThreshold;
    public static final Logger LOG = LoggerFactory.getLogger(OBSFileSystem.class);
    private static final Logger PROGRESS = LoggerFactory.getLogger((String)"org.apache.hadoop.fs.obs.OBSFileSystem.Progress");
    private LocalDirAllocator directoryAllocator;
    private String serverSideEncryptionAlgorithm;
    private OBSInstrumentation instrumentation;
    private OBSStorageStatistics storageStatistics;
    private long readAhead;
    private OBSInputPolicy inputPolicy;
    private final AtomicBoolean closed = new AtomicBoolean(false);
    private int MAX_ENTRIES_TO_DELETE;
    private boolean blockUploadEnabled;
    private String blockOutputBuffer;
    private OBSDataBlocks.BlockFactory blockFactory;
    private int blockOutputActiveBlocks;
    private int bufferPartSize;
    private long bufferMaxRange;
    private boolean readaheadInputStreamEnabled = false;
    private long copyPartSize;
    private int maxCopyPartThreads;
    private int maxCopyPartQueue;
    private long keepAliveTime;

    public void initialize(URI name, Configuration originalConf) throws IOException {
        this.uri = OBSLoginHelper.buildFSURI(name);
        this.bucket = name.getHost();
        Configuration conf = OBSUtils.propagateBucketOptions(originalConf, this.bucket);
        OBSUtils.patchSecurityCredentialProviders(conf);
        super.initialize(name, conf);
        this.setConf(conf);
        try {
            this.instrumentation = new OBSInstrumentation(name);
            this.username = UserGroupInformation.getCurrentUser().getShortUserName();
            this.workingDir = new Path("/user", this.username).makeQualified(this.uri, this.getWorkingDirectory());
            Class obsClientFactoryClass = conf.getClass("fs.obs.s3.client.factory.impl", Constants.DEFAULT_OBS_CLIENT_FACTORY_IMPL, ObsClientFactory.class);
            this.obs = ((ObsClientFactory)ReflectionUtils.newInstance((Class)obsClientFactoryClass, (Configuration)conf)).createObsClient(name);
            this.maxKeys = OBSUtils.intOption(conf, "fs.obs.paging.maximum", 1000, 1);
            this.listing = new Listing(this);
            this.partSize = OBSUtils.getMultipartSizeProperty(conf, "fs.obs.multipart.size", 0x6400000L);
            this.multiPartThreshold = OBSUtils.getMultipartSizeProperty(conf, "fs.obs.multipart.threshold", Integer.MAX_VALUE);
            OBSUtils.longBytesOption(conf, "fs.obs.block.size", 0x2000000L, 1L);
            this.enableMultiObjectsDelete = conf.getBoolean("fs.obs.multiobjectdelete.enable", true);
            this.MAX_ENTRIES_TO_DELETE = conf.getInt("fs.obs.multiobjectdelete.maximum", 1000);
            this.enableMultiObjectsDeleteRecursion = conf.getBoolean("fs.obs.multiobjectdelete.recursion", true);
            this.renameSupportEmptyDestinationFolder = conf.getBoolean("fs.obs.rename.to_empty_folder", true);
            this.readAhead = OBSUtils.longBytesOption(conf, "fs.obs.readahead.range", 65536L, 0L);
            this.storageStatistics = (OBSStorageStatistics)GlobalStorageStatistics.INSTANCE.put("OBSStorageStatistics", new GlobalStorageStatistics.StorageStatisticsProvider(){

                public StorageStatistics provide() {
                    return new OBSStorageStatistics();
                }
            });
            int maxThreads = conf.getInt("fs.obs.threads.max", 20);
            int maxCopyThreads = conf.getInt("fs.obs.copy.threads.max", 20);
            int maxDeleteThreads = conf.getInt("fs.obs.delete.threads.max", 10);
            int maxCopyQueue = OBSUtils.intOption(conf, "fs.obs.copy.queue.max", 10, 1);
            int maxDeleteQueue = OBSUtils.intOption(conf, "fs.obs.delete.queue.max", 10, 1);
            if (maxThreads < 2) {
                LOG.warn("fs.obs.threads.max must be at least 2: forcing to 2.");
                maxThreads = 2;
            }
            int maxReadThreads = conf.getInt("fs.obs.threads.read.max", 20);
            int coreReadThreads = conf.getInt("fs.obs.threads.read.core", 5);
            if (maxReadThreads < 2) {
                LOG.warn("fs.obs.threads.max must be at least 2: forcing to 2.");
                maxReadThreads = 2;
            }
            this.bufferPartSize = OBSUtils.intOption(conf, "fs.obs.buffer.part.size", 65536, 1024);
            this.bufferMaxRange = OBSUtils.intOption(conf, "fs.obs.buffer.max.range", 0x1400000, 1024);
            int totalTasks = OBSUtils.intOption(conf, "fs.obs.max.total.tasks", 20, 1);
            this.keepAliveTime = OBSUtils.longOption(conf, "fs.obs.threads.keepalivetime", 60L, 0L);
            this.copyPartSize = OBSUtils.longOption(conf, "fs.obs.copypart.size", 0x6400000L, 0L);
            this.maxCopyPartThreads = conf.getInt("fs.obs.copypart.threads.max", 20);
            this.maxCopyPartQueue = OBSUtils.intOption(conf, "fs.obs.copypart.queue.max", 10, 1);
            this.boundedThreadPool = BlockingThreadPoolExecutorService.newInstance(maxThreads, maxThreads + totalTasks, this.keepAliveTime, TimeUnit.SECONDS, "obs-transfer-shared");
            this.boundedCopyThreadPool = new SemaphoredDelegatingExecutor((ListeningExecutorService)BlockingThreadPoolExecutorService.newInstance(maxCopyThreads, maxCopyThreads + maxCopyQueue, this.keepAliveTime, TimeUnit.SECONDS, "obs-copy-transfer-shared"), maxCopyQueue, true);
            this.boundedDeleteThreadPool = new SemaphoredDelegatingExecutor((ListeningExecutorService)BlockingThreadPoolExecutorService.newInstance(maxDeleteThreads, maxDeleteThreads + maxDeleteQueue, this.keepAliveTime, TimeUnit.SECONDS, "obs-delete-transfer-shared"), maxDeleteQueue, true);
            this.unboundedThreadPool = new ThreadPoolExecutor(maxThreads, Integer.MAX_VALUE, this.keepAliveTime, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(), BlockingThreadPoolExecutorService.newDaemonThreadFactory("obs-transfer-unbounded"));
            this.readaheadInputStreamEnabled = conf.getBoolean("fs.obs.readahead.inputstream.enabled", false);
            if (this.readaheadInputStreamEnabled) {
                this.unboundedReadThreadPool = new ThreadPoolExecutor(coreReadThreads, maxReadThreads, this.keepAliveTime, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(), BlockingThreadPoolExecutorService.newDaemonThreadFactory("obs-transfer-read-unbounded"));
            }
            this.boundedCopyPartThreadPool = new SemaphoredDelegatingExecutor((ListeningExecutorService)BlockingThreadPoolExecutorService.newInstance(this.maxCopyPartThreads, this.maxCopyPartThreads + this.maxCopyPartQueue, this.keepAliveTime, TimeUnit.SECONDS, "obs-copy-part-transfer-shared"), this.maxCopyPartQueue, true);
            this.initTransferManager();
            this.initCannedAcls(conf);
            this.verifyBucketExists();
            this.getBucketFsStatus();
            this.initMultipartUploads(conf);
            this.serverSideEncryptionAlgorithm = conf.getTrimmed("fs.obs.server-side-encryption-algorithm");
            LOG.debug("Using encryption {}", (Object)this.serverSideEncryptionAlgorithm);
            this.inputPolicy = OBSInputPolicy.getPolicy(conf.getTrimmed("fs.obs.experimental.input.fadvise", "normal"));
            this.blockUploadEnabled = conf.getBoolean("fs.obs.fast.upload", true);
            if (this.blockUploadEnabled) {
                this.blockOutputBuffer = conf.getTrimmed("fs.obs.fast.upload.buffer", "disk");
                this.partSize = OBSUtils.ensureOutputParameterInRange("fs.obs.multipart.size", this.partSize);
                this.blockFactory = OBSDataBlocks.createFactory(this, this.blockOutputBuffer);
                this.blockOutputActiveBlocks = OBSUtils.intOption(conf, "fs.obs.fast.upload.active.blocks", 4, 1);
                LOG.debug("Using OBSBlockOutputStream with buffer = {}; block={}; queue limit={}", new Object[]{this.blockOutputBuffer, this.partSize, this.blockOutputActiveBlocks});
            } else {
                LOG.debug("Using OBSOutputStream");
            }
        }
        catch (ObsException e) {
            throw OBSUtils.translateException("initializing ", new Path(name), e);
        }
    }

    protected void verifyBucketExists() throws FileNotFoundException, IOException {
        try {
            if (!this.obs.headBucket(this.bucket)) {
                throw new FileNotFoundException("Bucket " + this.bucket + " does not exist");
            }
        }
        catch (ObsException e) {
            throw OBSUtils.translateException("doesBucketExist", this.bucket, e);
        }
    }

    public boolean getBucketFsStatus(String bucketName) throws FileNotFoundException, IOException {
        try {
            GetBucketFSStatusRequest getBucketFsStatusRequest = new GetBucketFSStatusRequest();
            getBucketFsStatusRequest.setBucketName(bucketName);
            GetBucketFSStatusResult getBucketFsStatusResult = this.obs.getBucketFSStatus(getBucketFsStatusRequest);
            FSStatusEnum fsStatus = getBucketFsStatusResult.getStatus();
            return fsStatus != null && fsStatus == FSStatusEnum.ENABLED;
        }
        catch (ObsException e) {
            LOG.error(e.toString());
            throw OBSUtils.translateException("getBucketFsStatus", this.bucket, e);
        }
    }

    private void getBucketFsStatus() throws FileNotFoundException, IOException {
        this.enablePosix = this.getBucketFsStatus(this.bucket);
    }

    public boolean isFsBucket() {
        return this.enablePosix;
    }

    public OBSInstrumentation getInstrumentation() {
        return this.instrumentation;
    }

    private void initTransferManager() {
    }

    private void initCannedAcls(Configuration conf) {
    }

    private void initMultipartUploads(Configuration conf) throws IOException {
        boolean purgeExistingMultipart = conf.getBoolean("fs.obs.multipart.purge", false);
        long purgeExistingMultipartAge = OBSUtils.longOption(conf, "fs.obs.multipart.purge.age", 86400L, 0L);
        if (purgeExistingMultipart) {
            Date purgeBefore = new Date(new Date().getTime() - purgeExistingMultipartAge * 1000L);
            try {
                ListMultipartUploadsRequest request;
                MultipartUploadListing uploadListing = this.obs.listMultipartUploads(new ListMultipartUploadsRequest(this.bucket));
                do {
                    for (MultipartUpload upload : uploadListing.getMultipartTaskList()) {
                        if (upload.getInitiatedDate().compareTo(purgeBefore) >= 0) continue;
                        this.obs.abortMultipartUpload(new AbortMultipartUploadRequest(this.bucket, upload.getObjectKey(), upload.getUploadId()));
                    }
                    request = new ListMultipartUploadsRequest(this.bucket);
                    request.setUploadIdMarker(uploadListing.getNextUploadIdMarker());
                    request.setKeyMarker(uploadListing.getNextKeyMarker());
                } while ((uploadListing = this.obs.listMultipartUploads(request)).isTruncated());
            }
            catch (ObsException e) {
                if (e.getResponseCode() == 403) {
                    this.instrumentation.errorIgnored();
                    LOG.debug("Failed to purging multipart uploads against {}, FS may be read only", (Object)this.bucket, (Object)e);
                }
                throw OBSUtils.translateException("purging multipart uploads", this.bucket, e);
            }
        }
    }

    public String getScheme() {
        return "obs";
    }

    public URI getUri() {
        return this.uri;
    }

    public int getDefaultPort() {
        return -1;
    }

    @VisibleForTesting
    public ObsClient getObsClient() {
        return this.obs;
    }

    @VisibleForTesting
    long getReadAheadRange() {
        return this.readAhead;
    }

    @InterfaceStability.Unstable
    public OBSInputPolicy getInputPolicy() {
        return this.inputPolicy;
    }

    synchronized File createTmpFileForWrite(String pathStr, long size, Configuration conf) throws IOException {
        if (this.directoryAllocator == null) {
            String bufferDir = conf.get("fs.obs.buffer.dir") != null ? "fs.obs.buffer.dir" : "hadoop.tmp.dir";
            this.directoryAllocator = new LocalDirAllocator(bufferDir);
        }
        return this.directoryAllocator.createTmpFileForWrite(pathStr, size, conf);
    }

    public String getBucket() {
        return this.bucket;
    }

    @InterfaceStability.Unstable
    public void setInputPolicy(OBSInputPolicy inputPolicy) {
        Objects.requireNonNull(inputPolicy, "Null inputStrategy");
        LOG.debug("Setting input strategy: {}", (Object)inputPolicy);
        this.inputPolicy = inputPolicy;
    }

    private String pathToKey(Path path) {
        if (!path.isAbsolute()) {
            path = new Path(this.workingDir, path);
        }
        if (path.toUri().getScheme() != null && path.toUri().getPath().isEmpty()) {
            return "";
        }
        return path.toUri().getPath().substring(1);
    }

    private String maybeAddTrailingSlash(String key) {
        if (!key.isEmpty() && !key.endsWith("/")) {
            return key + '/';
        }
        return key;
    }

    private Path keyToPath(String key) {
        return new Path("/" + key);
    }

    Path keyToQualifiedPath(String key) {
        return this.qualify(this.keyToPath(key));
    }

    Path qualify(Path path) {
        return path.makeQualified(this.uri, this.workingDir);
    }

    public void checkPath(Path path) {
        OBSLoginHelper.checkPath(this.getConf(), this.getUri(), path, this.getDefaultPort());
    }

    protected URI canonicalizeUri(URI rawUri) {
        return OBSLoginHelper.canonicalizeUri(rawUri, this.getDefaultPort());
    }

    public FSDataInputStream open(Path f, int bufferSize) throws IOException {
        LOG.debug("Opening '{}' for reading.", (Object)f);
        OBSFileStatus fileStatus = this.getFileStatus(f);
        if (fileStatus.isDirectory()) {
            throw new FileNotFoundException("Can't open " + f + " because it is a directory");
        }
        if (this.readaheadInputStreamEnabled) {
            return new FSDataInputStream((InputStream)((Object)new OBSReadaheadInputStream(this.bucket, this.pathToKey(f), fileStatus.getLen(), this.obs, this.statistics, this.instrumentation, this.readAhead, this.inputPolicy, this.unboundedReadThreadPool, this.bufferPartSize, this.bufferMaxRange)));
        }
        return new FSDataInputStream((InputStream)((Object)new OBSInputStream(this.bucket, this.pathToKey(f), fileStatus.getLen(), this.obs, this.statistics, this.instrumentation, this.readAhead, this.inputPolicy)));
    }

    public FSDataOutputStream create(Path f, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        long startTime = System.nanoTime();
        String key = this.pathToKey(f);
        OBSFileStatus status = null;
        try {
            status = this.getFileStatus(f);
            if (status.isDirectory()) {
                throw new FileAlreadyExistsException(f + " is a directory");
            }
            if (!overwrite) {
                throw new FileAlreadyExistsException(f + " already exists");
            }
            LOG.debug("create: Overwriting file {}", (Object)f);
        }
        catch (FileNotFoundException e) {
            LOG.debug("create: Creating new file {}", (Object)f);
        }
        this.instrumentation.filesCreatedTotal(1L);
        FSDataOutputStream output = this.blockUploadEnabled ? new FSDataOutputStream((OutputStream)new OBSBlockOutputStream(this, key, (ExecutorService)((Object)new SemaphoredDelegatingExecutor(this.boundedThreadPool, this.blockOutputActiveBlocks, true)), progress, this.partSize, this.blockFactory, this.instrumentation.newOutputStreamStatistics(this.statistics), new OBSWriteOperationHelper(key)), null) : new FSDataOutputStream((OutputStream)new OBSOutputStream(this.getConf(), this, key, progress), null);
        long delay = System.nanoTime() - startTime;
        this.instrumentation.filesCreated(1L, delay);
        return output;
    }

    public FSDataOutputStream createNonRecursive(Path path, FsPermission permission, EnumSet<CreateFlag> flags, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        Path parent = path.getParent();
        if (parent != null && !this.getFileStatus(parent).isDirectory()) {
            throw new FileAlreadyExistsException("Not a directory: " + parent);
        }
        return this.create(path, permission, flags.contains(CreateFlag.OVERWRITE), bufferSize, replication, blockSize, progress);
    }

    public FSDataOutputStream append(Path f, int bufferSize, Progressable progress) throws IOException {
        throw new IOException("Not supported");
    }

    public boolean rename(Path src, Path dst) throws IOException {
        try {
            return this.innerRename(src, dst);
        }
        catch (ObsException e) {
            throw OBSUtils.translateException("rename(" + src + ", " + dst + ")", src, e);
        }
        catch (RenameFailedException e) {
            LOG.error(e.getMessage());
            return e.getExitCode();
        }
        catch (FileNotFoundException e) {
            LOG.error(e.toString());
            return false;
        }
    }

    private boolean innerRename(Path src, Path dst) throws RenameFailedException, FileNotFoundException, IOException, ObsException {
        boolean dstFolderExisted;
        OBSFileStatus dstStatus;
        OBSFileStatus srcStatus;
        String dstKey;
        String srcKey;
        block23: {
            LOG.debug("Rename path {} to {}", (Object)src, (Object)dst);
            this.incrementStatistic(Statistic.INVOCATION_RENAME);
            srcKey = this.pathToKey(src);
            dstKey = this.pathToKey(dst);
            if (srcKey.isEmpty()) {
                throw new RenameFailedException(src, dst, "source is root directory");
            }
            if (dstKey.isEmpty()) {
                throw new RenameFailedException(src, dst, "dest is root directory");
            }
            srcStatus = this.getFileStatus(src);
            if (srcKey.equals(dstKey)) {
                LOG.error("rename: src and dest refer to the same file or directory: {}", (Object)dst);
                throw new RenameFailedException(src, dst, "source and dest refer to the same file or directory").withExitCode(srcStatus.isFile());
            }
            dstStatus = null;
            dstFolderExisted = false;
            try {
                dstStatus = this.getFileStatus(dst);
                if (srcStatus.isDirectory()) {
                    if (dstStatus.isFile()) {
                        throw new RenameFailedException(src, dst, "source is a directory and dest is a file").withExitCode(srcStatus.isFile());
                    }
                    if (!this.renameSupportEmptyDestinationFolder) {
                        throw new RenameFailedException(src, dst, "destination is an existed directory").withExitCode(false);
                    }
                    if (!dstStatus.isEmptyDirectory()) {
                        throw new RenameFailedException(src, dst, "Destination is a non-empty directory").withExitCode(false);
                    }
                    dstFolderExisted = true;
                } else {
                    if (dstStatus.isFile()) {
                        throw new RenameFailedException(src, dst, "Cannot rename onto an existing file").withExitCode(false);
                    }
                    dstFolderExisted = true;
                }
            }
            catch (FileNotFoundException e) {
                LOG.error("rename: destination path {} not found", (Object)dst);
                if (this.enablePosix && !srcStatus.isDirectory() && dstKey.endsWith("/")) {
                    throw new RenameFailedException(src, dst, "source is a file but destination directory is not existed");
                }
                Path parent = dst.getParent();
                if (this.pathToKey(parent).isEmpty()) break block23;
                try {
                    OBSFileStatus dstParentStatus = this.getFileStatus(dst.getParent());
                    if (!dstParentStatus.isDirectory()) {
                        throw new RenameFailedException(src, dst, "destination parent is not a directory");
                    }
                }
                catch (FileNotFoundException e2) {
                    throw new RenameFailedException(src, dst, "destination has no parent ");
                }
            }
        }
        if (srcStatus.isFile()) {
            LOG.debug("rename: renaming file {} to {}", (Object)src, (Object)dst);
            if (dstStatus != null && dstStatus.isDirectory()) {
                String newDstKey = dstKey;
                if (!newDstKey.endsWith("/")) {
                    newDstKey = newDstKey + "/";
                }
                String filename = srcKey.substring(this.pathToKey(src.getParent()).length() + 1);
                dstKey = newDstKey = newDstKey + filename;
            }
            this.renameFile(srcKey, dstKey, srcStatus);
        } else {
            LOG.debug("rename: renaming directory {} to {}", (Object)src, (Object)dst);
            if (!dstKey.endsWith("/")) {
                dstKey = dstKey + "/";
            }
            if (!srcKey.endsWith("/")) {
                srcKey = srcKey + "/";
            }
            if (dstKey.startsWith(srcKey)) {
                throw new RenameFailedException(srcKey, dstKey, "cannot rename a directory to a subdirectory o fitself ");
            }
            this.renameFolder(srcKey, dstKey, dstFolderExisted, dstStatus);
        }
        if (src.getParent() != dst.getParent()) {
            this.deleteUnnecessaryFakeDirectories(dst.getParent());
            this.createFakeDirectoryIfNecessary(src.getParent());
        }
        return true;
    }

    private void renameFile(String srcKey, String dstKey, OBSFileStatus srcStatus) throws IOException {
        this.instrumentation.frontendFilesRenamedTotal(1L);
        long startTime = System.nanoTime();
        if (this.enablePosix) {
            this.fsRenameFile(srcKey, dstKey);
        } else {
            this.copyFile(srcKey, dstKey, srcStatus.getLen());
            this.innerDelete(srcStatus, false);
        }
        long delay = System.nanoTime() - startTime;
        this.instrumentation.frontendFilesRenamed(1L, delay);
        this.instrumentation.filesRenamed(1L, delay);
        LOG.debug("OBSFileSystem rename: " + this.instrumentation.frontendFilesRenamedToString() + ", {src=" + srcKey + ", dst=" + dstKey + ", delay=" + delay + "}");
    }

    private void renameFolder(String srcKey, String dstKey, boolean dstFolderExisted, OBSFileStatus dstStatus) throws IOException {
        this.instrumentation.frontendDirectoriesRenamedTotal(dstFolderExisted, 1L);
        long startTime = System.nanoTime();
        if (this.enablePosix) {
            this.fsRenameFolder(dstFolderExisted, srcKey, dstKey);
        } else {
            this.instrumentation.directoriesRenamedTotal(1L);
            ArrayList<KeyAndVersion> keysToDelete = new ArrayList<KeyAndVersion>();
            if (dstStatus != null && dstStatus.isEmptyDirectory()) {
                keysToDelete.add(new KeyAndVersion(dstKey));
            }
            long listStartTime = System.nanoTime();
            ListObjectsRequest request = new ListObjectsRequest();
            request.setBucketName(this.bucket);
            request.setPrefix(srcKey);
            request.setMaxKeys(this.maxKeys);
            ObjectListing objects = this.listObjects(request);
            this.instrumentation.listObjectsInRename(System.nanoTime() - listStartTime);
            LinkedList<ListenableFuture<DeleteObjectsResult>> deletefutures = new LinkedList<ListenableFuture<DeleteObjectsResult>>();
            LinkedList<ListenableFuture<CopyObjectResult>> copyfutures = new LinkedList<ListenableFuture<CopyObjectResult>>();
            while (true) {
                for (ObsObject summary : objects.getObjects()) {
                    keysToDelete.add(new KeyAndVersion(summary.getObjectKey()));
                    String newDstKey = dstKey + summary.getObjectKey().substring(srcKey.length());
                    copyfutures.add(this.copyFileAsync(summary.getObjectKey(), newDstKey, summary.getMetadata().getContentLength()));
                    if (keysToDelete.size() != this.MAX_ENTRIES_TO_DELETE) continue;
                    this.waitAllCopyFinished(copyfutures);
                    copyfutures.clear();
                    deletefutures.add(this.removeKeysAsync(keysToDelete, true, false));
                }
                if (!objects.isTruncated()) break;
                listStartTime = System.nanoTime();
                objects = this.continueListObjects(objects);
                this.instrumentation.listObjectsInRename(System.nanoTime() - listStartTime);
            }
            if (!keysToDelete.isEmpty()) {
                this.waitAllCopyFinished(copyfutures);
                copyfutures.clear();
                deletefutures.add(this.removeKeysAsync(keysToDelete, false, false));
            }
            try {
                Futures.allAsList(deletefutures).get();
            }
            catch (InterruptedException e) {
                LOG.warn("Interrupted while copying objects (delete)");
                throw new InterruptedIOException("Interrupted while copying objects (delete)");
            }
            catch (ExecutionException e) {
                if (e.getCause() instanceof ObsException) {
                    throw (ObsException)e.getCause();
                }
                if (e.getCause() instanceof IOException) {
                    throw (IOException)e.getCause();
                }
                throw new ObsException("unknown error while copying objects (delete)", e.getCause());
            }
        }
        long delay = System.nanoTime() - startTime;
        this.instrumentation.frontendDirectoriesRenamed(dstFolderExisted, delay);
        this.instrumentation.directoriesRenamed(1L, delay);
        LOG.debug("OBSFileSystem rename: " + this.instrumentation.frontendDirectoriesRenamedToString() + ", {src=" + srcKey + ", dst=" + dstKey + ", delay=" + delay + "}");
    }

    private void waitAllCopyFinished(List<ListenableFuture<CopyObjectResult>> copyfutures) throws InterruptedIOException {
        try {
            Futures.allAsList(copyfutures).get();
        }
        catch (InterruptedException e) {
            LOG.warn("Interrupted while copying objects (copy)");
            throw new InterruptedIOException("Interrupted while copying objects (copy)");
        }
        catch (ExecutionException e) {
            e.printStackTrace();
        }
    }

    private void fsRenameFile(String src, String dst) throws IOException, ObsException {
        LOG.debug("RenameFile path {} to {}", (Object)src, (Object)dst);
        try {
            final RenameRequest renameObjectRequest = new RenameRequest();
            renameObjectRequest.setBucketName(this.bucket);
            renameObjectRequest.setObjectKey(src);
            renameObjectRequest.setNewObjectKey(dst);
            Future<RenameResult> future = this.unboundedThreadPool.submit(new Callable<RenameResult>(){

                @Override
                public RenameResult call() throws ObsException {
                    return OBSFileSystem.this.obs.renameFile(renameObjectRequest);
                }
            });
            try {
                future.get();
                this.incrementWriteOperations();
            }
            catch (InterruptedException e) {
                throw new InterruptedIOException("Interrupted renaming " + src + " to " + dst + ", cancelling");
            }
            catch (ExecutionException e) {
                if (e.getCause() instanceof ObsException) {
                    throw (ObsException)e.getCause();
                }
                throw new ObsException("obs exception: ", e.getCause());
            }
        }
        catch (ObsException e) {
            throw OBSUtils.translateException("renameFile(" + src + ", " + dst + ")", src, e);
        }
    }

    private void fsRenameToNewFolder(String src, String dst) throws IOException, ObsException {
        LOG.debug("RenameFolder path {} to {}", (Object)src, (Object)dst);
        try {
            final RenameRequest renameObjectRequest = new RenameRequest();
            renameObjectRequest.setBucketName(this.bucket);
            renameObjectRequest.setObjectKey(src);
            renameObjectRequest.setNewObjectKey(dst);
            ListenableFuture future = this.boundedThreadPool.submit((Callable)new Callable<RenameResult>(){

                @Override
                public RenameResult call() throws ObsException {
                    return OBSFileSystem.this.obs.renameFolder(renameObjectRequest);
                }
            });
            try {
                future.get();
                this.incrementWriteOperations();
            }
            catch (InterruptedException e) {
                throw new InterruptedIOException("Interrupted renaming " + src + " to " + dst + ", cancelling");
            }
            catch (ExecutionException e) {
                if (e.getCause() instanceof ObsException) {
                    throw (ObsException)e.getCause();
                }
                throw new ObsException("obs exception: ", e.getCause());
            }
        }
        catch (ObsException e) {
            throw OBSUtils.translateException("renameFile(" + src + ", " + dst + ")", src, e);
        }
    }

    private void fsRenameAllSubObjectsToOldFolder(String srcKey, String dstKey) throws IOException {
        int maxKeyNum = this.maxKeys;
        long listStartTime = System.nanoTime();
        ListObjectsRequest request = this.createListObjectsRequest(srcKey, "/", maxKeyNum);
        ObjectListing objects = this.listObjects(request);
        this.instrumentation.listObjectsInRename(System.nanoTime() - listStartTime);
        while (true) {
            for (ObsObject sonSrcObject : objects.getObjects()) {
                String sonSrcKey = sonSrcObject.getObjectKey();
                String sonDstKey = dstKey + sonSrcKey.substring(srcKey.length());
                if (sonSrcKey.equals(srcKey)) continue;
                this.fsRenameToNewObject(sonSrcKey, sonDstKey);
            }
            for (String sonSrcKey : objects.getCommonPrefixes()) {
                String sonDstKey = dstKey + sonSrcKey.substring(srcKey.length());
                if (sonSrcKey.equals(srcKey)) continue;
                this.fsRenameToNewObject(sonSrcKey, sonDstKey);
            }
            if (!objects.isTruncated()) break;
            listStartTime = System.nanoTime();
            objects = this.continueListObjects(objects);
            this.instrumentation.listObjectsInRename(System.nanoTime() - listStartTime);
        }
    }

    private void fsRenameToNewObject(String srcKey, String dstKey) throws IOException {
        if (srcKey.endsWith("/")) {
            this.fsRenameToNewFolder(srcKey, dstKey);
        } else {
            this.fsRenameFile(srcKey, dstKey);
        }
    }

    private void fsRenameFolder(boolean dstFolderIsExisted, String srcKey, String dstKey) throws IOException {
        LOG.debug("RenameFolder path {} to {}, dstFolderIsExisted={}", new Object[]{srcKey, dstKey, dstFolderIsExisted});
        if (!dstFolderIsExisted) {
            this.fsRenameToNewFolder(srcKey, dstKey);
        } else {
            this.fsRenameAllSubObjectsToOldFolder(srcKey, dstKey);
            this.deleteObject(srcKey, true);
        }
    }

    @VisibleForTesting
    public ObjectMetadata getObjectMetadata(Path path) throws IOException {
        return this.getObjectMetadata(this.pathToKey(path));
    }

    protected void incrementStatistic(Statistic statistic) {
        this.incrementStatistic(statistic, 1L);
    }

    protected void incrementStatistic(Statistic statistic, long count) {
        this.instrumentation.incrementCounter(statistic, count);
        this.storageStatistics.incrementCounter(statistic, count);
    }

    protected void decrementGauge(Statistic statistic, long count) {
        this.instrumentation.decrementGauge(statistic, count);
    }

    protected void incrementGauge(Statistic statistic, long count) {
        this.instrumentation.incrementGauge(statistic, count);
    }

    public OBSStorageStatistics getStorageStatistics() {
        return this.storageStatistics;
    }

    protected ObjectMetadata getObjectMetadata(String key) {
        this.incrementStatistic(Statistic.OBJECT_METADATA_REQUESTS);
        ObjectMetadata meta = this.obs.getObjectMetadata(this.bucket, key);
        this.incrementReadOperations();
        return meta;
    }

    protected ObjectListing listObjects(ListObjectsRequest request) {
        this.incrementStatistic(Statistic.OBJECT_LIST_REQUESTS);
        this.incrementReadOperations();
        return this.obs.listObjects(request);
    }

    protected ObjectListing continueListObjects(ObjectListing objects) {
        String delimiter = objects.getDelimiter();
        int maxKeyNum = objects.getMaxKeys();
        this.incrementStatistic(Statistic.OBJECT_CONTINUE_LIST_REQUESTS);
        this.incrementReadOperations();
        ListObjectsRequest request = new ListObjectsRequest();
        request.setMarker(objects.getNextMarker());
        request.setBucketName(this.bucket);
        request.setPrefix(objects.getPrefix());
        if (maxKeyNum > 0 && maxKeyNum < this.maxKeys) {
            request.setMaxKeys(maxKeyNum);
        } else {
            request.setMaxKeys(this.maxKeys);
        }
        if (delimiter != null) {
            request.setDelimiter(delimiter);
        }
        return this.obs.listObjects(request);
    }

    public void incrementReadOperations() {
        this.statistics.incrementReadOps(1);
    }

    public void incrementWriteOperations() {
        this.statistics.incrementWriteOps(1);
    }

    private void deleteObject(String key, boolean isFolder) throws InvalidRequestException {
        this.blockRootDelete(key);
        this.incrementWriteOperations();
        this.incrementStatistic(Statistic.OBJECT_DELETE_REQUESTS);
        long startTime = System.nanoTime();
        this.obs.deleteObject(this.bucket, key);
        long delay = System.nanoTime() - startTime;
        if (isFolder) {
            this.instrumentation.directoriesDeleted(1L, delay);
        } else {
            this.instrumentation.filesDeleted(1L, delay);
        }
    }

    private void blockRootDelete(String key) throws InvalidRequestException {
        if (key.isEmpty() || "/".equals(key)) {
            throw new InvalidRequestException("Bucket " + this.bucket + " cannot be deleted");
        }
    }

    private void deleteObjects(DeleteObjectsRequest deleteRequest) {
        long startTime = System.nanoTime();
        this.incrementWriteOperations();
        this.incrementStatistic(Statistic.OBJECT_DELETE_REQUESTS, 1L);
        this.obs.deleteObjects(deleteRequest);
        long delay = System.nanoTime() - startTime;
        this.instrumentation.batchDeleted(1L, delay);
    }

    public PutObjectRequest newPutObjectRequest(String key, ObjectMetadata metadata, File srcfile) {
        Preconditions.checkNotNull((Object)srcfile);
        PutObjectRequest putObjectRequest = new PutObjectRequest(this.bucket, key, srcfile);
        putObjectRequest.setMetadata(metadata);
        return putObjectRequest;
    }

    private PutObjectRequest newPutObjectRequest(String key, ObjectMetadata metadata, InputStream inputStream) {
        Preconditions.checkNotNull((Object)inputStream);
        PutObjectRequest putObjectRequest = new PutObjectRequest(this.bucket, key, inputStream);
        putObjectRequest.setMetadata(metadata);
        return putObjectRequest;
    }

    public ObjectMetadata newObjectMetadata() {
        ObjectMetadata om = new ObjectMetadata();
        return om;
    }

    public ObjectMetadata newObjectMetadata(long length) {
        ObjectMetadata om = this.newObjectMetadata();
        if (length >= 0L) {
            om.setContentLength(Long.valueOf(length));
        }
        return om;
    }

    public Future putObject(final PutObjectRequest putObjectRequest) {
        long len = putObjectRequest.getFile() != null ? putObjectRequest.getFile().length() : putObjectRequest.getMetadata().getContentLength().longValue();
        this.incrementPutStartStatistics(len);
        Future<PutObjectResult> future = null;
        future = this.unboundedThreadPool.submit(new Callable<PutObjectResult>(){

            @Override
            public PutObjectResult call() throws ObsException {
                return OBSFileSystem.this.obs.putObject(putObjectRequest);
            }
        });
        return future;
    }

    public PutObjectResult putObjectDirect(PutObjectRequest putObjectRequest) throws ObsException {
        long len = putObjectRequest.getFile() != null ? putObjectRequest.getFile().length() : putObjectRequest.getMetadata().getContentLength().longValue();
        this.incrementPutStartStatistics(len);
        try {
            PutObjectResult result = this.obs.putObject(putObjectRequest);
            this.incrementPutCompletedStatistics(true, len);
            return result;
        }
        catch (ObsException e) {
            this.incrementPutCompletedStatistics(false, len);
            throw e;
        }
    }

    public UploadPartResult uploadPart(UploadPartRequest request) throws ObsException {
        long len = request.getPartSize();
        this.incrementPutStartStatistics(len);
        try {
            UploadPartResult uploadPartResult = this.obs.uploadPart(request);
            this.incrementPutCompletedStatistics(true, len);
            return uploadPartResult;
        }
        catch (ObsException e) {
            this.incrementPutCompletedStatistics(false, len);
            throw e;
        }
    }

    public void incrementPutStartStatistics(long bytes) {
        LOG.debug("PUT start {} bytes", (Object)bytes);
        this.incrementWriteOperations();
        this.incrementStatistic(Statistic.OBJECT_PUT_REQUESTS);
        this.incrementGauge(Statistic.OBJECT_PUT_REQUESTS_ACTIVE, 1L);
        if (bytes > 0L) {
            this.incrementGauge(Statistic.OBJECT_PUT_BYTES_PENDING, bytes);
        }
    }

    public void incrementPutCompletedStatistics(boolean success, long bytes) {
        LOG.debug("PUT completed success={}; {} bytes", (Object)success, (Object)bytes);
        this.incrementWriteOperations();
        if (bytes > 0L) {
            this.incrementStatistic(Statistic.OBJECT_PUT_BYTES, bytes);
            this.decrementGauge(Statistic.OBJECT_PUT_BYTES_PENDING, bytes);
        }
        this.incrementStatistic(Statistic.OBJECT_PUT_REQUESTS_COMPLETED);
        this.decrementGauge(Statistic.OBJECT_PUT_REQUESTS_ACTIVE, 1L);
    }

    public void incrementPutProgressStatistics(String key, long bytes) {
        PROGRESS.debug("PUT {}: {} bytes", (Object)key, (Object)bytes);
        this.incrementWriteOperations();
        if (bytes > 0L) {
            this.statistics.incrementBytesWritten(bytes);
        }
    }

    private ListenableFuture<DeleteObjectsResult> removeKeysAsync(List<KeyAndVersion> keysToDelete, boolean clearKeys, boolean deleteFakeDir) throws ObsException, InvalidRequestException {
        ListenableFuture future2;
        if (keysToDelete.isEmpty()) {
            SettableFuture future2 = SettableFuture.create();
            future2.set(null);
            return (ListenableFuture)future2;
        }
        for (KeyAndVersion keyVersion : keysToDelete) {
            this.blockRootDelete(keyVersion.getKey());
        }
        if (this.enableMultiObjectsDelete) {
            this.instrumentation.batchDeletedTotal(1L);
            final DeleteObjectsRequest deleteObjectsRequest = new DeleteObjectsRequest(this.bucket);
            deleteObjectsRequest.setKeyAndVersions(keysToDelete.toArray(new KeyAndVersion[keysToDelete.size()]));
            future2 = this.boundedDeleteThreadPool.submit((Callable)new Callable<DeleteObjectsResult>(){

                @Override
                public DeleteObjectsResult call() throws Exception {
                    OBSFileSystem.this.deleteObjects(deleteObjectsRequest);
                    return null;
                }
            });
        } else {
            final ArrayList<KeyAndVersion> keys = new ArrayList<KeyAndVersion>(keysToDelete);
            future2 = this.boundedDeleteThreadPool.submit((Callable)new Callable<DeleteObjectsResult>(){

                @Override
                public DeleteObjectsResult call() throws Exception {
                    for (KeyAndVersion keyVersion : keys) {
                        OBSFileSystem.this.deleteObject(keyVersion.getKey(), keyVersion.getKey().endsWith("/"));
                    }
                    keys.clear();
                    return null;
                }
            });
        }
        if (!deleteFakeDir) {
            this.instrumentation.filesDeletedTotal(keysToDelete.size());
        } else {
            this.instrumentation.fakeDirsDeletedTotal(keysToDelete.size());
        }
        if (clearKeys) {
            keysToDelete.clear();
        }
        return future2;
    }

    private void removeKeys(List<KeyAndVersion> keysToDelete, boolean clearKeys) throws InvalidRequestException {
        this.removeKeys(keysToDelete, clearKeys, false);
    }

    private void removeKeys(List<KeyAndVersion> keysToDelete, boolean clearKeys, boolean checkRootDelete) throws InvalidRequestException {
        if (keysToDelete.isEmpty()) {
            return;
        }
        if (checkRootDelete) {
            for (KeyAndVersion keyVersion : keysToDelete) {
                this.blockRootDelete(keyVersion.getKey());
            }
        }
        if (!this.enableMultiObjectsDelete) {
            for (KeyAndVersion keyVersion : keysToDelete) {
                this.deleteObject(keyVersion.getKey(), keyVersion.getKey().endsWith("/"));
            }
        } else if (keysToDelete.size() <= this.MAX_ENTRIES_TO_DELETE) {
            DeleteObjectsRequest deleteObjectsRequest = new DeleteObjectsRequest(this.bucket);
            deleteObjectsRequest.setKeyAndVersions(keysToDelete.toArray(new KeyAndVersion[keysToDelete.size()]));
            this.deleteObjects(deleteObjectsRequest);
        } else {
            ArrayList<KeyAndVersion> keys = new ArrayList<KeyAndVersion>(this.MAX_ENTRIES_TO_DELETE);
            for (KeyAndVersion key : keysToDelete) {
                keys.add(key);
                if (keys.size() != this.MAX_ENTRIES_TO_DELETE) continue;
                this.removeKeys(keys, true, false);
            }
            this.removeKeys(keys, true, false);
        }
        if (clearKeys) {
            keysToDelete.clear();
        }
    }

    public boolean delete(Path f, boolean recursive) throws IOException {
        try {
            return this.innerDelete(this.getFileStatus(f), recursive);
        }
        catch (FileNotFoundException e) {
            LOG.debug("Couldn't delete {} - does not exist", (Object)f);
            this.instrumentation.errorIgnored();
            return false;
        }
        catch (ObsException e) {
            throw OBSUtils.translateException("delete", f, e);
        }
    }

    private boolean innerDelete(OBSFileStatus status, boolean recursive) throws IOException, ObsException {
        long startTime;
        if (this.enablePosix) {
            return this.fsDelete(status, recursive);
        }
        Path f = status.getPath();
        LOG.debug("delete: path {} - recursive {}", (Object)f, (Object)recursive);
        String key = this.pathToKey(f);
        if (status.isDirectory()) {
            LOG.debug("delete: Path is a directory: {} - recursive {}", (Object)f, (Object)recursive);
            this.instrumentation.frontendDirectoryDeletedTotal(1L);
            startTime = System.nanoTime();
            if (!key.endsWith("/")) {
                key = key + "/";
            }
            if (key.equals("/")) {
                return this.rejectRootDirectoryDelete(status, recursive);
            }
            if (!recursive && !status.isEmptyDirectory()) {
                throw new PathIsNotEmptyDirectoryException(f.toString());
            }
            if (status.isEmptyDirectory()) {
                LOG.debug("delete: Deleting fake empty directory {} - recursive {}", (Object)f, (Object)recursive);
                this.deleteObject(key, true);
            } else {
                LOG.debug("delete: Deleting objects for directory prefix {} - recursive {}", (Object)f, (Object)recursive);
                String delimiter = recursive ? null : "/";
                ListObjectsRequest request = this.createListObjectsRequest(key, delimiter);
                ObjectListing objects = this.listObjects(request);
                ArrayList<KeyAndVersion> keys = new ArrayList<KeyAndVersion>(objects.getObjects().size());
                while (true) {
                    for (ObsObject summary : objects.getObjects()) {
                        keys.add(new KeyAndVersion(summary.getObjectKey()));
                        LOG.debug("Got object to delete {}", (Object)summary.getObjectKey());
                        if (keys.size() != this.MAX_ENTRIES_TO_DELETE) continue;
                        this.removeKeys(keys, true, true);
                    }
                    if (!objects.isTruncated()) break;
                    objects = this.continueListObjects(objects);
                }
                if (!keys.isEmpty()) {
                    this.removeKeys(keys, false, true);
                }
            }
            long delay = System.nanoTime() - startTime;
            this.instrumentation.frontendDirectoryDeleted(1L, delay);
        } else {
            LOG.debug("delete: Path is a file");
            this.instrumentation.frontendFileDeletedTotal(1L);
            startTime = System.nanoTime();
            this.deleteObject(key, false);
            long delay = System.nanoTime() - startTime;
            this.instrumentation.frontendFileDeleted(1L, delay);
        }
        Path parent = f.getParent();
        if (parent != null) {
            this.createFakeDirectoryIfNecessary(parent);
        }
        return true;
    }

    private boolean fsDelete(OBSFileStatus status, boolean recursive) throws IOException, ObsException {
        Path f = status.getPath();
        LOG.debug("Delete path {} - recursive {}", (Object)f, (Object)recursive);
        String key = this.pathToKey(f);
        if (!status.isDirectory()) {
            LOG.debug("delete: Path is a file");
            this.instrumentation.frontendFileDeletedTotal(1L);
            long startTime = System.nanoTime();
            this.deleteObject(key, false);
            long delay = System.nanoTime() - startTime;
            this.instrumentation.frontendFileDeleted(1L, delay);
        } else {
            LOG.debug("delete: Path is a directory: {} - recursive {}", (Object)f, (Object)recursive);
            this.instrumentation.frontendDirectoryDeletedTotal(1L);
            if (!key.endsWith("/")) {
                key = key + "/";
            }
            if (key.equals("/")) {
                return this.rejectRootDirectoryDelete(status, recursive);
            }
            if (!recursive && !status.isEmptyDirectory()) {
                LOG.warn("delete: Path is not empty: {} - recursive {}", (Object)f, (Object)recursive);
                throw new PathIsNotEmptyDirectoryException(f.toString());
            }
            if (status.isEmptyDirectory()) {
                LOG.debug("delete: Deleting fake empty directory {} - recursive {}", (Object)f, (Object)recursive);
                this.deleteObject(key, true);
            } else {
                LOG.debug("delete: Deleting objects for directory prefix {} to delete - recursive {}", (Object)f, (Object)recursive);
                if (this.enableMultiObjectsDeleteRecursion) {
                    this.fsRecursivelyDelete(key, true);
                } else {
                    this.fsNonRecursivelyDelete(f);
                }
            }
        }
        Path parent = f.getParent();
        if (parent != null) {
            this.createFakeDirectoryIfNecessary(parent);
        }
        return true;
    }

    private void fsNonRecursivelyDelete(Path parent) throws IOException, ObsException {
        FileStatus[] arFileStatus = this.innerListStatus(parent, true);
        this.fsRemoveKeys(arFileStatus);
        this.deleteObject(this.pathToKey(parent), true);
    }

    private void fsRemoveKeys(FileStatus[] arFileStatus) throws ObsException, InvalidRequestException {
        if (arFileStatus.length <= 0) {
            return;
        }
        String key = "";
        for (int i = 0; i < arFileStatus.length; ++i) {
            key = this.pathToKey(arFileStatus[i].getPath());
            this.blockRootDelete(key);
        }
        this.fsRemoveKeysByDepth(arFileStatus);
    }

    private void fsRemoveKeysByDepth(FileStatus[] arFileStatus) throws ObsException, InvalidRequestException {
        if (arFileStatus.length <= 0) {
            return;
        }
        int filesNum = 0;
        int foldersNum = 0;
        String key = "";
        int depth = Integer.MAX_VALUE;
        ArrayList<KeyAndVersion> leafKeys = new ArrayList<KeyAndVersion>(this.MAX_ENTRIES_TO_DELETE);
        for (int idx = arFileStatus.length - 1; idx >= 0; --idx) {
            if (leafKeys.size() >= this.MAX_ENTRIES_TO_DELETE) {
                this.removeKeys(leafKeys, true);
            }
            key = this.pathToKey(arFileStatus[idx].getPath());
            if (!arFileStatus[idx].isDirectory()) {
                leafKeys.add(new KeyAndVersion(key, null));
                ++filesNum;
                continue;
            }
            int keyDepth = OBSFileSystem.fsGetObjectKeyDepth(key);
            if (keyDepth == depth) {
                leafKeys.add(new KeyAndVersion(key, null));
                ++foldersNum;
                continue;
            }
            if (keyDepth < depth) {
                this.removeKeys(leafKeys, true);
                depth = keyDepth;
                leafKeys.add(new KeyAndVersion(key, null));
                ++foldersNum;
                continue;
            }
            LOG.warn("The objects list is invalid because it isn't sorted by path depth.");
            throw new ObsException("System failure");
        }
        this.removeKeys(leafKeys, true);
    }

    private int fsRemoveKeysByDepth(List<KeyAndVersion> keys) throws ObsException, InvalidRequestException {
        if (keys.size() <= 0) {
            return 0;
        }
        int filesNum = 0;
        int foldersNum = 0;
        String key = "";
        int depth = Integer.MAX_VALUE;
        ArrayList<KeyAndVersion> leafKeys = new ArrayList<KeyAndVersion>(this.MAX_ENTRIES_TO_DELETE);
        for (int idx = keys.size() - 1; idx >= 0; --idx) {
            if (leafKeys.size() >= this.MAX_ENTRIES_TO_DELETE) {
                this.removeKeys(leafKeys, true);
            }
            if (!(key = keys.get(idx).getKey()).endsWith("/")) {
                leafKeys.add(new KeyAndVersion(key, null));
                ++filesNum;
                continue;
            }
            int keyDepth = OBSFileSystem.fsGetObjectKeyDepth(key);
            if (keyDepth == depth) {
                leafKeys.add(new KeyAndVersion(key, null));
                ++foldersNum;
                continue;
            }
            if (keyDepth < depth) {
                this.removeKeys(leafKeys, true);
                depth = keyDepth;
                leafKeys.add(new KeyAndVersion(key, null));
                ++foldersNum;
                continue;
            }
            LOG.warn("The objects list is invalid because it isn't sorted by path depth.");
            throw new ObsException("System failure");
        }
        this.removeKeys(leafKeys, true);
        return filesNum + foldersNum;
    }

    private static int fsGetObjectKeyDepth(String key) {
        int depth = 0;
        int idx = key.indexOf(47, 0);
        while (idx >= 0) {
            ++depth;
            idx = key.indexOf(47, idx + 1);
        }
        return key.endsWith("/") ? depth - 1 : depth;
    }

    public int fsRecursivelyDelete(String parentKey, boolean deleteParent, int recursionScale) throws IOException {
        ArrayList<KeyAndVersion> folders = new ArrayList<KeyAndVersion>(this.MAX_ENTRIES_TO_DELETE);
        ArrayList<KeyAndVersion> files = new ArrayList<KeyAndVersion>(this.MAX_ENTRIES_TO_DELETE);
        ListObjectsRequest request = this.createListObjectsRequest(parentKey, null, this.maxKeys);
        ObjectListing objects = this.listObjects(request);
        int delNum = 0;
        while (true) {
            for (ObsObject sonObject : objects.getObjects()) {
                String sonObjectKey = sonObject.getObjectKey();
                if (sonObjectKey.length() == parentKey.length()) continue;
                delNum += this.fsRemoveSonObject(sonObjectKey, files, folders);
            }
            if (folders.size() >= recursionScale) {
                delNum += files.size();
                this.removeKeys(files, true);
                delNum += this.fsRecursivelyDeleteDeepest(folders);
            }
            if (!objects.isTruncated()) break;
            objects = this.continueListObjects(objects);
        }
        delNum += files.size();
        this.removeKeys(files, true);
        delNum += this.fsRemoveKeysByDepth(folders);
        if (deleteParent) {
            this.deleteObject(parentKey, true);
            ++delNum;
        }
        return delNum;
    }

    public int fsRecursivelyDelete(String parentKey, boolean deleteParent) throws IOException {
        return this.fsRecursivelyDelete(parentKey, deleteParent, this.MAX_ENTRIES_TO_DELETE * 2);
    }

    private int fsRemoveSonObject(String sonObjectKey, List<KeyAndVersion> files, List<KeyAndVersion> folders) throws IOException {
        if (!sonObjectKey.endsWith("/")) {
            return this.fsRemoveFile(sonObjectKey, files);
        }
        folders.add(new KeyAndVersion(sonObjectKey));
        return 0;
    }

    private int fsRemoveFile(String sonObjectKey, List<KeyAndVersion> files) throws IOException {
        files.add(new KeyAndVersion(sonObjectKey));
        if (files.size() == this.MAX_ENTRIES_TO_DELETE) {
            this.removeKeys(files, true);
            return this.MAX_ENTRIES_TO_DELETE;
        }
        return 0;
    }

    private List<KeyAndVersion> fsExtractDeepestFolders(List<KeyAndVersion> folders) throws ObsException {
        if (folders.isEmpty()) {
            return null;
        }
        int start = folders.size() - 1;
        KeyAndVersion folder = folders.get(start);
        int deepestDepth = OBSFileSystem.fsGetObjectKeyDepth(folder.getKey());
        int currDepth = 0;
        --start;
        while (start >= 0) {
            folder = folders.get(start);
            currDepth = OBSFileSystem.fsGetObjectKeyDepth(folder.getKey());
            if (currDepth != deepestDepth) {
                if (currDepth < deepestDepth) {
                    ++start;
                    break;
                }
                LOG.warn("The folders list is invalid because it isn't sorted by path depth.");
                throw new ObsException("System failure");
            }
            --start;
        }
        if (start < 0) {
            start = 0;
        }
        int deepestFoldersNum = folders.size() - start;
        ArrayList<KeyAndVersion> deepestFolders = new ArrayList<KeyAndVersion>(Math.min(folders.size(), deepestFoldersNum));
        for (int i = folders.size() - 1; i >= start; --i) {
            folder = folders.get(i);
            deepestFolders.add(folder);
            folders.remove(i);
        }
        return deepestFolders;
    }

    private int fsRecursivelyDeleteDeepest(List<KeyAndVersion> folders) throws IOException {
        int delNum = 0;
        List<KeyAndVersion> deepestFolders = this.fsExtractDeepestFolders(folders);
        for (KeyAndVersion folder : deepestFolders) {
            delNum += this.fsRecursivelyDelete(folder.getKey(), false);
        }
        this.removeKeys(deepestFolders, false);
        return delNum += deepestFolders.size();
    }

    private boolean rejectRootDirectoryDelete(OBSFileStatus status, boolean recursive) throws IOException {
        LOG.info("obs delete the {} root directory of {}", (Object)this.bucket, (Object)recursive);
        boolean emptyRoot = status.isEmptyDirectory();
        if (emptyRoot) {
            return true;
        }
        if (recursive) {
            return false;
        }
        throw new PathIOException(this.bucket, "Cannot delete root path");
    }

    private void createFakeDirectoryIfNecessary(Path f) throws IOException, ObsException {
        if (this.enablePosix) {
            return;
        }
        String key = this.pathToKey(f);
        if (!key.isEmpty() && !this.exists(f)) {
            LOG.debug("Creating new fake directory at {}", (Object)f);
            this.createFakeDirectory(key);
        }
    }

    public FileStatus[] listStatus(Path f) throws FileNotFoundException, IOException {
        try {
            return this.innerListStatus(f, false);
        }
        catch (ObsException e) {
            throw OBSUtils.translateException("listStatus", f, e);
        }
    }

    public FileStatus[] innerListStatus(Path f, boolean recursive) throws FileNotFoundException, IOException, ObsException {
        Path path = this.qualify(f);
        String key = this.pathToKey(path);
        LOG.debug("List status for path: {}", (Object)path);
        this.incrementStatistic(Statistic.INVOCATION_LIST_STATUS);
        OBSFileStatus fileStatus = this.getFileStatus(path);
        if (fileStatus.isDirectory()) {
            if (!key.isEmpty()) {
                key = key + '/';
            }
            String delimiter = recursive ? null : "/";
            ListObjectsRequest request = this.createListObjectsRequest(key, delimiter);
            LOG.debug("listStatus: doing listObjects for directory {} - recursive {}", (Object)f, (Object)recursive);
            Listing.FileStatusListingIterator files = this.listing.createFileStatusListingIterator(path, request, Listing.ACCEPT_ALL, new Listing.AcceptAllButSelfAndS3nDirs(path));
            ArrayList<FileStatus> result = new ArrayList<FileStatus>(files.getBatchSize());
            while (files.hasNext()) {
                result.add(files.next());
            }
            return result.toArray(new FileStatus[result.size()]);
        }
        LOG.debug("Adding: rd (not a dir): {}", (Object)path);
        FileStatus[] stats = new FileStatus[]{fileStatus};
        return stats;
    }

    public List<String> fsGetSubObjects(Path f) throws IOException {
        Path path = this.qualify(f);
        String parentKey = this.pathToKey(path);
        parentKey = parentKey.endsWith("/") ? parentKey : parentKey + "/";
        ArrayList<String> keys = new ArrayList<String>(100);
        String delimiter = null;
        ListObjectsRequest request = this.createListObjectsRequest(parentKey, delimiter, this.maxKeys);
        ObjectListing objects = this.listObjects(request);
        while (true) {
            for (ObsObject son : objects.getObjects()) {
                String sonKey = son.getObjectKey();
                if (sonKey.equals(parentKey)) continue;
                keys.add(sonKey);
            }
            for (String sonKey : objects.getCommonPrefixes()) {
                if (sonKey.equals(parentKey)) continue;
                keys.add(sonKey);
            }
            if (!objects.isTruncated()) break;
            objects = this.continueListObjects(objects);
        }
        return keys;
    }

    private ListObjectsRequest createListObjectsRequest(String key, String delimiter) {
        return this.createListObjectsRequest(key, delimiter, -1);
    }

    private ListObjectsRequest createListObjectsRequest(String key, String delimiter, int maxKeyNum) {
        ListObjectsRequest request = new ListObjectsRequest();
        request.setBucketName(this.bucket);
        if (maxKeyNum > 0 && maxKeyNum < this.maxKeys) {
            request.setMaxKeys(maxKeyNum);
        } else {
            request.setMaxKeys(this.maxKeys);
        }
        request.setPrefix(key);
        if (delimiter != null) {
            request.setDelimiter(delimiter);
        }
        return request;
    }

    public void setWorkingDirectory(Path newDir) {
        this.workingDir = newDir;
    }

    public Path getWorkingDirectory() {
        return this.workingDir;
    }

    public String getUsername() {
        return this.username;
    }

    public boolean mkdirs(Path path, FsPermission permission) throws IOException, FileAlreadyExistsException {
        try {
            return this.innerMkdirs(path, permission);
        }
        catch (ObsException e) {
            throw OBSUtils.translateException("innerMkdirs", path, e);
        }
    }

    private boolean innerMkdirs(Path f, FsPermission permission) throws IOException, FileAlreadyExistsException, ObsException {
        LOG.debug("Making directory: {}", (Object)f);
        this.incrementStatistic(Statistic.INVOCATION_MKDIRS);
        try {
            OBSFileStatus fileStatus = this.getFileStatus(f);
            if (fileStatus.isDirectory()) {
                return true;
            }
            throw new FileAlreadyExistsException("Path is a file: " + f);
        }
        catch (FileNotFoundException e) {
            Path fPart = f.getParent();
            do {
                try {
                    OBSFileStatus fileStatus = this.getFileStatus(fPart);
                    if (fileStatus.isDirectory()) break;
                    if (!fileStatus.isFile()) continue;
                    throw new FileAlreadyExistsException(String.format("Can't make directory for path '%s' since it is a file.", fPart));
                }
                catch (FileNotFoundException fnfe) {
                    this.instrumentation.errorIgnored();
                }
            } while ((fPart = fPart.getParent()) != null);
            String key = this.pathToKey(f);
            this.createFakeDirectory(key);
            return true;
        }
    }

    public OBSFileStatus getFileStatus(Path f) throws IOException {
        String key;
        Path path;
        block12: {
            if (this.enablePosix) {
                return this.fsGetObjectStatus(f);
            }
            this.incrementStatistic(Statistic.INVOCATION_GET_FILE_STATUS);
            path = this.qualify(f);
            key = this.pathToKey(path);
            LOG.debug("Getting path status for {}  ({})", (Object)path, (Object)key);
            if (!key.isEmpty()) {
                try {
                    ObjectMetadata meta = this.getObjectMetadata(key);
                    if (OBSUtils.objectRepresentsDirectory(key, meta.getContentLength())) {
                        LOG.debug("Found exact file: fake directory");
                        return new OBSFileStatus(true, path, this.username);
                    }
                    LOG.debug("Found exact file: normal file");
                    return new OBSFileStatus(meta.getContentLength(), OBSUtils.dateToLong(meta.getLastModified()), path, this.getDefaultBlockSize(path), this.username);
                }
                catch (ObsException e) {
                    if (e.getResponseCode() != 404) {
                        throw OBSUtils.translateException("getFileStatus", path, e);
                    }
                    if (key.endsWith("/")) break block12;
                    String newKey = key + "/";
                    try {
                        ObjectMetadata meta = this.getObjectMetadata(newKey);
                        if (OBSUtils.objectRepresentsDirectory(newKey, meta.getContentLength())) {
                            LOG.debug("Found file (with /): fake directory");
                            return new OBSFileStatus(true, path, this.username);
                        }
                        LOG.debug("Found file (with /): real file? should not happen: {}", (Object)key);
                        return new OBSFileStatus(meta.getContentLength(), OBSUtils.dateToLong(meta.getLastModified()), path, this.getDefaultBlockSize(path), this.username);
                    }
                    catch (ObsException e2) {
                        if (e2.getResponseCode() == 404) break block12;
                        throw OBSUtils.translateException("getFileStatus", newKey, e2);
                    }
                }
            }
        }
        try {
            boolean isEmpty = this.isFolderEmpty(key);
            return new OBSFileStatus(isEmpty, path, this.username);
        }
        catch (ObsException e) {
            if (e.getResponseCode() != 404) {
                throw OBSUtils.translateException("getFileStatus", key, e);
            }
            LOG.error("Not Found: {}", (Object)path);
            throw new FileNotFoundException("No such file or directory: " + path);
        }
    }

    public boolean isFolderEmpty(String key, ObjectListing objects) {
        int count = objects.getObjects().size();
        if (count >= 2 || count == 1 && !((ObsObject)objects.getObjects().get(0)).getObjectKey().equals(key)) {
            return false;
        }
        count = objects.getCommonPrefixes().size();
        return count < 2 && (count != 1 || ((String)objects.getCommonPrefixes().get(0)).equals(key));
    }

    public boolean isFolderEmpty(String key) throws FileNotFoundException, ObsException {
        key = this.maybeAddTrailingSlash(key);
        ListObjectsRequest request = new ListObjectsRequest();
        request.setBucketName(this.bucket);
        request.setPrefix(key);
        request.setDelimiter("/");
        request.setMaxKeys(3);
        ObjectListing objects = this.listObjects(request);
        if (!objects.getCommonPrefixes().isEmpty() || !objects.getObjectSummaries().isEmpty()) {
            if (this.enablePosix && this.isFolderEmpty(key, objects)) {
                LOG.debug("Found empty directory {}", (Object)key);
                return true;
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("Found path as directory (with /): {}/{}", (Object)objects.getCommonPrefixes().size(), (Object)objects.getObjects().size());
                for (ObsObject summary : objects.getObjects()) {
                    LOG.debug("Summary: {} {}", (Object)summary.getObjectKey(), (Object)summary.getMetadata().getContentLength());
                }
                for (String prefix : objects.getCommonPrefixes()) {
                    LOG.debug("Prefix: {}", (Object)prefix);
                }
            }
            LOG.debug("Found non-empty directory {}", (Object)key);
            return false;
        }
        if (key.isEmpty()) {
            LOG.debug("Found root directory");
            return true;
        }
        if (this.enablePosix) {
            LOG.debug("Found empty directory {}", (Object)key);
            return true;
        }
        LOG.debug("Not Found: {}", (Object)key);
        throw new FileNotFoundException("No such file or directory: " + key);
    }

    public OBSFileStatus fsGetObjectStatus(Path f) throws FileNotFoundException, IOException {
        this.incrementStatistic(Statistic.INVOCATION_GET_FILE_STATUS);
        Path path = this.qualify(f);
        String key = this.pathToKey(path);
        LOG.debug("Getting path status for {}  ({})", (Object)path, (Object)key);
        if (key.isEmpty()) {
            LOG.debug("Found root directory");
            boolean isEmpty = this.isFolderEmpty(key);
            return new OBSFileStatus(isEmpty, path, this.username);
        }
        try {
            GetAttributeRequest getAttrRequest = new GetAttributeRequest(this.bucket, key);
            ObsFSAttribute meta = this.obs.getAttribute(getAttrRequest);
            if (OBSFileSystem.fsIsFolder(meta)) {
                LOG.debug("Found file (with /): fake directory");
                boolean isEmpty = this.isFolderEmpty(key);
                return new OBSFileStatus(isEmpty, path, this.username);
            }
            LOG.debug("Found file (with /): real file? should not happen: {}", (Object)key);
            return new OBSFileStatus(meta.getContentLength(), OBSUtils.dateToLong(meta.getLastModified()), path, this.getDefaultBlockSize(path), this.username);
        }
        catch (ObsException e) {
            if (e.getResponseCode() != 404) {
                throw OBSUtils.translateException("getFileStatus", path, e);
            }
            LOG.debug("Not Found: {}", (Object)path);
            throw new FileNotFoundException("No such file or directory: " + path);
        }
    }

    public static boolean fsIsFolder(ObsFSAttribute attr) {
        int S_IFDIR = 16384;
        int mode = attr.getMode();
        return (mode & 0x4000) != 0;
    }

    public void copyFromLocalFile(boolean delSrc, boolean overwrite, Path src, Path dst) throws IOException {
        try {
            this.innerCopyFromLocalFile(delSrc, overwrite, src, dst);
        }
        catch (ObsException e) {
            throw OBSUtils.translateException("copyFromLocalFile(" + src + ", " + dst + ")", src, e);
        }
    }

    private void innerCopyFromLocalFile(boolean delSrc, boolean overwrite, Path src, Path dst) throws IOException, FileAlreadyExistsException, ObsException {
        this.incrementStatistic(Statistic.INVOCATION_COPY_FROM_LOCAL_FILE);
        String key = this.pathToKey(dst);
        if (!overwrite && this.exists(dst)) {
            throw new FileAlreadyExistsException(dst + " already exists");
        }
        LOG.debug("Copying local file from {} to {}", (Object)src, (Object)dst);
        LocalFileSystem local = OBSFileSystem.getLocal((Configuration)this.getConf());
        File srcfile = local.pathToFile(src);
        ObjectMetadata om = this.newObjectMetadata(srcfile.length());
        PutObjectRequest putObjectRequest = this.newPutObjectRequest(key, om, srcfile);
        Future future = this.putObject(putObjectRequest);
        try {
            future.get();
            this.incrementPutCompletedStatistics(true, srcfile.length());
        }
        catch (InterruptedException e) {
            this.incrementPutCompletedStatistics(false, srcfile.length());
            throw new InterruptedIOException("Interrupted copying " + src + " to " + dst + ", cancelling");
        }
        catch (ExecutionException e) {
            this.incrementPutCompletedStatistics(false, srcfile.length());
            e.printStackTrace();
        }
        this.finishedWrite(key);
        if (delSrc) {
            local.delete(src, false);
        }
    }

    public void close() throws IOException {
        if (this.closed.getAndSet(true)) {
            return;
        }
        super.close();
    }

    public String getCanonicalServiceName() {
        return null;
    }

    private ListenableFuture<CopyObjectResult> copyFileAsync(final String srcKey, final String dstKey, final long size) {
        return this.boundedCopyThreadPool.submit((Callable)new Callable<CopyObjectResult>(){

            @Override
            public CopyObjectResult call() throws Exception {
                OBSFileSystem.this.copyFile(srcKey, dstKey, size);
                return null;
            }
        });
    }

    private void copyFile(final String srcKey, final String dstKey, long size) throws IOException, InterruptedIOException, ObsException {
        long startTime = System.nanoTime();
        if (LOG.isDebugEnabled()) {
            LOG.debug("copyFile {} -> {} ", (Object)srcKey, (Object)dstKey);
        }
        try {
            long objectSize = size;
            if (objectSize > this.copyPartSize) {
                InitiateMultipartUploadRequest request = new InitiateMultipartUploadRequest(this.bucket, dstKey);
                InitiateMultipartUploadResult result = this.obs.initiateMultipartUpload(request);
                final String uploadId = result.getUploadId();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Multipart copy file, uploadId: {}", (Object)uploadId);
                }
                long partCount = objectSize % this.copyPartSize == 0L ? objectSize / this.copyPartSize : objectSize / this.copyPartSize + 1L;
                final List partEtags = Collections.synchronizedList(new ArrayList());
                ArrayList<ListenableFuture> partCopyFutures = new ArrayList<ListenableFuture>();
                int i = 0;
                while ((long)i < partCount) {
                    final long rangeStart = (long)i * this.copyPartSize;
                    final long rangeEnd = (long)(i + 1) == partCount ? objectSize - 1L : rangeStart + this.copyPartSize - 1L;
                    final int partNumber = i + 1;
                    partCopyFutures.add(this.boundedCopyPartThreadPool.submit(new Runnable(){

                        @Override
                        public void run() {
                            CopyPartRequest request = new CopyPartRequest();
                            request.setUploadId(uploadId);
                            request.setSourceBucketName(OBSFileSystem.this.bucket);
                            request.setSourceObjectKey(srcKey);
                            request.setDestinationBucketName(OBSFileSystem.this.bucket);
                            request.setDestinationObjectKey(dstKey);
                            request.setByteRangeStart(Long.valueOf(rangeStart));
                            request.setByteRangeEnd(Long.valueOf(rangeEnd));
                            request.setPartNumber(partNumber);
                            try {
                                CopyPartResult result = OBSFileSystem.this.obs.copyPart(request);
                                partEtags.add(new PartEtag(result.getEtag(), Integer.valueOf(result.getPartNumber())));
                                if (LOG.isDebugEnabled()) {
                                    LOG.debug("Multipart copy file, uploadId: {}, Part#{} done.", (Object)uploadId, (Object)partNumber);
                                }
                            }
                            catch (ObsException e) {
                                LOG.error("Multipart copy file exception.", (Throwable)e);
                            }
                        }
                    }));
                    ++i;
                }
                try {
                    Futures.allAsList(partCopyFutures).get();
                }
                catch (InterruptedException e) {
                    LOG.warn("Interrupted while copying objects (copy)");
                    throw new InterruptedIOException("Interrupted while copying objects (copy)");
                }
                catch (ExecutionException e) {
                    LOG.error("Multipart copy file exception.", (Throwable)e);
                }
                if ((long)partEtags.size() != partCount) {
                    LOG.error("partEtags({}) is not equals partCount({}).", (Object)partEtags.size(), (Object)partCount);
                    throw new IllegalStateException("Upload multiparts fail due to some parts are not finished yet");
                }
                Collections.sort(partEtags, new Comparator<PartEtag>(){

                    @Override
                    public int compare(PartEtag o1, PartEtag o2) {
                        return o1.getPartNumber() - o2.getPartNumber();
                    }
                });
                CompleteMultipartUploadRequest completeMultipartUploadRequest = new CompleteMultipartUploadRequest(this.bucket, dstKey, uploadId, partEtags);
                this.obs.completeMultipartUpload(completeMultipartUploadRequest);
            } else {
                ObjectMetadata srcom = this.getObjectMetadata(srcKey);
                ObjectMetadata dstom = this.cloneObjectMetadata(srcom);
                CopyObjectRequest copyObjectRequest = new CopyObjectRequest(this.bucket, srcKey, this.bucket, dstKey);
                copyObjectRequest.setNewObjectMetadata(dstom);
                this.obs.copyObject(copyObjectRequest);
            }
            this.incrementWriteOperations();
        }
        catch (ObsException e) {
            throw OBSUtils.translateException("copyFile(" + srcKey + ", " + dstKey + ")", srcKey, e);
        }
        long delay = System.nanoTime() - startTime;
        this.instrumentation.filesCopied(1L, delay, size);
    }

    public void finishedWrite(String key) {
        LOG.debug("Finished write to {}", (Object)key);
        this.deleteUnnecessaryFakeDirectories(this.keyToPath(key).getParent());
    }

    private void deleteUnnecessaryFakeDirectories(Path path) {
        block5: {
            if (this.enablePosix) {
                return;
            }
            ArrayList<KeyAndVersion> keysToRemove = new ArrayList<KeyAndVersion>();
            while (!path.isRoot()) {
                String key = this.pathToKey(path);
                key = key.endsWith("/") ? key : key + "/";
                keysToRemove.add(new KeyAndVersion(key));
                path = path.getParent();
            }
            try {
                this.removeKeys(keysToRemove, false);
            }
            catch (ObsException | InvalidRequestException e) {
                this.instrumentation.errorIgnored();
                if (!LOG.isDebugEnabled()) break block5;
                StringBuilder sb = new StringBuilder();
                for (KeyAndVersion kv : keysToRemove) {
                    sb.append(kv.getKey()).append(",");
                }
                LOG.debug("While deleting keys {} ", (Object)sb.toString(), (Object)e);
            }
        }
    }

    private void createFakeDirectory(String objectName) throws ObsException, InterruptedIOException {
        if (this.enablePosix) {
            this.fsCreateFolder(objectName);
            return;
        }
        if (!objectName.endsWith("/")) {
            this.createEmptyObject(objectName + "/");
        } else {
            this.createEmptyObject(objectName);
        }
    }

    private void createEmptyObject(String objectName) throws ObsException, InterruptedIOException {
        InputStream im = new InputStream(){

            @Override
            public int read() throws IOException {
                return -1;
            }
        };
        this.instrumentation.directoriesCreatedTotal(1L);
        long startTime = System.nanoTime();
        PutObjectRequest putObjectRequest = this.newPutObjectRequest(objectName, this.newObjectMetadata(0L), im);
        Future upload = this.putObject(putObjectRequest);
        try {
            upload.get();
            this.incrementPutCompletedStatistics(true, 0L);
        }
        catch (InterruptedException e) {
            this.incrementPutCompletedStatistics(false, 0L);
            throw new InterruptedIOException("Interrupted creating " + objectName);
        }
        catch (ExecutionException e) {
            this.incrementPutCompletedStatistics(false, 0L);
            if (e.getCause() instanceof ObsException) {
                throw (ObsException)e.getCause();
            }
            throw new ObsException("obs exception: ", e.getCause());
        }
        this.incrementPutProgressStatistics(objectName, 0L);
        long delay = System.nanoTime() - startTime;
        this.instrumentation.directoriesCreated(1L, delay);
    }

    private void fsCreateFile(String objectName) throws ObsException, InterruptedIOException {
        this.instrumentation.filesCreatedTotal(1L);
        long startTime = System.nanoTime();
        try {
            final NewFileRequest newFileRequest = new NewFileRequest(this.bucket, objectName);
            long len = newFileRequest.getObjectKey().length();
            this.incrementPutStartStatistics(len);
            ListenableFuture future = null;
            future = this.boundedThreadPool.submit((Callable)new Callable<ObsFSFile>(){

                @Override
                public ObsFSFile call() throws ObsException {
                    return OBSFileSystem.this.obs.newFile(newFileRequest);
                }
            });
            future.get();
            this.incrementPutCompletedStatistics(true, 0L);
        }
        catch (InterruptedException e) {
            this.incrementPutCompletedStatistics(false, 0L);
            throw new InterruptedIOException("Interrupted creating " + objectName);
        }
        catch (ExecutionException e) {
            this.incrementPutCompletedStatistics(false, 0L);
            if (e.getCause() instanceof ObsException) {
                throw (ObsException)e.getCause();
            }
            throw new ObsException("obs exception: ", e.getCause());
        }
        this.incrementPutProgressStatistics(objectName, 0L);
        long delay = System.nanoTime() - startTime;
        this.instrumentation.filesCreated(1L, delay);
    }

    private void fsCreateFolder(String objectName) throws ObsException, InterruptedIOException {
        this.instrumentation.directoriesCreatedTotal(1L);
        long startTime = System.nanoTime();
        try {
            final NewFolderRequest newFolderRequest = new NewFolderRequest(this.bucket, objectName);
            long len = newFolderRequest.getObjectKey().length();
            this.incrementPutStartStatistics(len);
            ListenableFuture future = null;
            future = this.boundedThreadPool.submit((Callable)new Callable<ObsFSFolder>(){

                @Override
                public ObsFSFolder call() throws ObsException {
                    return OBSFileSystem.this.obs.newFolder(newFolderRequest);
                }
            });
            future.get();
            this.incrementPutCompletedStatistics(true, 0L);
        }
        catch (InterruptedException e) {
            this.incrementPutCompletedStatistics(false, 0L);
            throw new InterruptedIOException("Interrupted creating " + objectName);
        }
        catch (ExecutionException e) {
            this.incrementPutCompletedStatistics(false, 0L);
            if (e.getCause() instanceof ObsException) {
                throw (ObsException)e.getCause();
            }
            throw new ObsException("obs exception: ", e.getCause());
        }
        this.incrementPutProgressStatistics(objectName, 0L);
        long delay = System.nanoTime() - startTime;
        this.instrumentation.directoriesCreated(1L, delay);
    }

    private ObjectMetadata cloneObjectMetadata(ObjectMetadata source) {
        ObjectMetadata ret = this.newObjectMetadata(source.getContentLength());
        if (source.getContentEncoding() != null) {
            ret.setContentEncoding(source.getContentEncoding());
        }
        return ret;
    }

    @Deprecated
    public long getDefaultBlockSize() {
        return this.getConf().getLongBytes("fs.obs.block.size", 0x2000000L);
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("OBSFileSystem{");
        sb.append("uri=").append(this.uri);
        sb.append(", workingDir=").append(this.workingDir);
        sb.append(", inputPolicy=").append((Object)this.inputPolicy);
        sb.append(", partSize=").append(this.partSize);
        sb.append(", enableMultiObjectsDelete=").append(this.enableMultiObjectsDelete);
        sb.append(", maxKeys=").append(this.maxKeys);
        sb.append(", readAhead=").append(this.readAhead);
        sb.append(", blockSize=").append(this.getDefaultBlockSize());
        sb.append(", multiPartThreshold=").append(this.multiPartThreshold);
        if (this.serverSideEncryptionAlgorithm != null) {
            sb.append(", serverSideEncryptionAlgorithm='").append(this.serverSideEncryptionAlgorithm).append('\'');
        }
        if (this.blockFactory != null) {
            sb.append(", blockFactory=").append(this.blockFactory);
        }
        sb.append(", boundedExecutor=").append(this.boundedThreadPool);
        sb.append(", unboundedExecutor=").append(this.unboundedThreadPool);
        sb.append(", statistics {").append(this.statistics).append("}");
        sb.append(", metrics {").append(this.instrumentation.dump("{", "=", "} ", true)).append("}");
        sb.append('}');
        return sb.toString();
    }

    public long getPartitionSize() {
        return this.partSize;
    }

    public long getMultiPartThreshold() {
        return this.multiPartThreshold;
    }

    int getMaxKeys() {
        return this.maxKeys;
    }

    public FileStatus[] globStatus(Path pathPattern) throws IOException {
        this.incrementStatistic(Statistic.INVOCATION_GLOB_STATUS);
        return super.globStatus(pathPattern);
    }

    public FileStatus[] globStatus(Path pathPattern, PathFilter filter) throws IOException {
        this.incrementStatistic(Statistic.INVOCATION_GLOB_STATUS);
        return super.globStatus(pathPattern, filter);
    }

    public boolean exists(Path f) throws IOException {
        this.incrementStatistic(Statistic.INVOCATION_EXISTS);
        return super.exists(f);
    }

    public boolean isDirectory(Path f) throws IOException {
        this.incrementStatistic(Statistic.INVOCATION_IS_DIRECTORY);
        return super.isDirectory(f);
    }

    public boolean isFile(Path f) throws IOException {
        this.incrementStatistic(Statistic.INVOCATION_IS_FILE);
        return super.isFile(f);
    }

    public RemoteIterator<LocatedFileStatus> listFiles(Path f, boolean recursive) throws FileNotFoundException, IOException {
        this.incrementStatistic(Statistic.INVOCATION_LIST_FILES);
        Path path = this.qualify(f);
        LOG.debug("listFiles({}, {})", (Object)path, (Object)recursive);
        try {
            OBSFileStatus fileStatus = this.getFileStatus(path);
            if (fileStatus.isFile()) {
                LOG.debug("Path is a file");
                return new Listing.SingleStatusRemoteIterator(this.toLocatedFileStatus(fileStatus));
            }
            LOG.debug("listFiles: doing listFiles of directory {} - recursive {}", (Object)path, (Object)recursive);
            String key = this.maybeAddTrailingSlash(this.pathToKey(path));
            String delimiter = recursive ? null : "/";
            LOG.debug("Requesting all entries under {} with delimiter '{}'", (Object)key, (Object)delimiter);
            return this.listing.createLocatedFileStatusIterator(this.listing.createFileStatusListingIterator(path, this.createListObjectsRequest(key, delimiter), Listing.ACCEPT_ALL, new Listing.AcceptFilesOnly(path)));
        }
        catch (ObsException e) {
            throw OBSUtils.translateException("listFiles", path, e);
        }
    }

    public RemoteIterator<LocatedFileStatus> listLocatedStatus(Path f) throws FileNotFoundException, IOException {
        return this.listLocatedStatus(f, Listing.ACCEPT_ALL);
    }

    public RemoteIterator<LocatedFileStatus> listLocatedStatus(Path f, PathFilter filter) throws FileNotFoundException, IOException {
        this.incrementStatistic(Statistic.INVOCATION_LIST_LOCATED_STATUS);
        Path path = this.qualify(f);
        LOG.debug("listLocatedStatus({}, {}", (Object)path, (Object)filter);
        try {
            OBSFileStatus fileStatus = this.getFileStatus(path);
            if (fileStatus.isFile()) {
                LOG.debug("Path is a file");
                return new Listing.SingleStatusRemoteIterator(filter.accept(path) ? this.toLocatedFileStatus(fileStatus) : null);
            }
            String key = this.maybeAddTrailingSlash(this.pathToKey(path));
            return this.listing.createLocatedFileStatusIterator(this.listing.createFileStatusListingIterator(path, this.createListObjectsRequest(key, "/"), filter, new Listing.AcceptAllButSelfAndS3nDirs(path)));
        }
        catch (ObsException e) {
            throw OBSUtils.translateException("listLocatedStatus", path, e);
        }
    }

    LocatedFileStatus toLocatedFileStatus(FileStatus status) throws IOException {
        return new LocatedFileStatus(status, status.isFile() ? this.getFileBlockLocations(status, 0L, status.getLen()) : null);
    }

    final class OBSWriteOperationHelper {
        private final String key;

        private OBSWriteOperationHelper(String key) {
            this.key = key;
        }

        PutObjectRequest newPutRequest(InputStream inputStream, long length) {
            PutObjectRequest request = OBSFileSystem.this.newPutObjectRequest(this.key, this.newObjectMetadata(length), inputStream);
            return request;
        }

        PutObjectRequest newPutRequest(File sourceFile) {
            int length = (int)sourceFile.length();
            PutObjectRequest request = OBSFileSystem.this.newPutObjectRequest(this.key, this.newObjectMetadata(length), sourceFile);
            return request;
        }

        void writeSuccessful() {
            OBSFileSystem.this.finishedWrite(this.key);
        }

        void writeFailed(Exception e) {
            LOG.debug("Write to {} failed", (Object)this, (Object)e);
        }

        public ObjectMetadata newObjectMetadata(long length) {
            return OBSFileSystem.this.newObjectMetadata(length);
        }

        String initiateMultiPartUpload() throws IOException {
            LOG.debug("Initiating Multipart upload");
            InitiateMultipartUploadRequest initiateMPURequest = new InitiateMultipartUploadRequest(OBSFileSystem.this.bucket, this.key);
            initiateMPURequest.setMetadata(this.newObjectMetadata(-1L));
            try {
                return OBSFileSystem.this.obs.initiateMultipartUpload(initiateMPURequest).getUploadId();
            }
            catch (ObsException ace) {
                throw OBSUtils.translateException("initiate MultiPartUpload", this.key, ace);
            }
        }

        CompleteMultipartUploadResult completeMultipartUpload(String uploadId, List<PartEtag> partETags) throws ObsException {
            Preconditions.checkNotNull((Object)uploadId);
            Preconditions.checkNotNull(partETags);
            Preconditions.checkArgument((!partETags.isEmpty() ? 1 : 0) != 0, (Object)"No partitions have been uploaded");
            LOG.debug("Completing multipart upload {} with {} parts", (Object)uploadId, (Object)partETags.size());
            return OBSFileSystem.this.obs.completeMultipartUpload(new CompleteMultipartUploadRequest(OBSFileSystem.this.bucket, this.key, uploadId, new ArrayList<PartEtag>(partETags)));
        }

        void abortMultipartUpload(String uploadId) throws ObsException {
            LOG.debug("Aborting multipart upload {}", (Object)uploadId);
            OBSFileSystem.this.obs.abortMultipartUpload(new AbortMultipartUploadRequest(OBSFileSystem.this.bucket, this.key, uploadId));
        }

        UploadPartRequest newUploadPartRequest(String uploadId, int partNumber, int size, InputStream uploadStream, File sourceFile) {
            Preconditions.checkNotNull((Object)uploadId);
            Preconditions.checkArgument((boolean)(uploadStream != null ^ sourceFile != null), (Object)"Data source");
            Preconditions.checkArgument((size > 0 ? 1 : 0) != 0, (String)"Invalid partition size %s", (Object[])new Object[]{size});
            Preconditions.checkArgument((partNumber > 0 && partNumber <= 10000 ? 1 : 0) != 0);
            LOG.debug("Creating part upload request for {} #{} size {}", new Object[]{uploadId, partNumber, size});
            UploadPartRequest request = new UploadPartRequest();
            request.setUploadId(uploadId);
            request.setBucketName(OBSFileSystem.this.bucket);
            request.setObjectKey(this.key);
            request.setPartSize(Long.valueOf(size));
            request.setPartNumber(partNumber);
            if (uploadStream != null) {
                request.setInput(uploadStream);
            } else {
                request.setFile(sourceFile);
            }
            return request;
        }

        public String toString() {
            StringBuilder sb = new StringBuilder("{bucket=").append(OBSFileSystem.this.bucket);
            sb.append(", key='").append(this.key).append('\'');
            sb.append('}');
            return sb.toString();
        }

        PutObjectResult putObject(PutObjectRequest putObjectRequest) throws IOException {
            try {
                return OBSFileSystem.this.putObjectDirect(putObjectRequest);
            }
            catch (ObsException e) {
                throw OBSUtils.translateException("put", putObjectRequest.getObjectKey(), e);
            }
        }
    }
}

