/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.cosn;

import java.io.IOException;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.cosn.buffer.CosNBufferFactory;
import org.apache.hadoop.fs.cosn.buffer.CosNBufferType;
import org.apache.hadoop.fs.cosn.buffer.CosNByteBuffer;
import org.apache.hadoop.fs.cosn.buffer.CosNDirectBufferFactory;
import org.apache.hadoop.fs.cosn.buffer.CosNMappedBufferFactory;
import org.apache.hadoop.fs.cosn.buffer.CosNNonDirectBufferFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class BufferPool {
    private static final Logger LOG = LoggerFactory.getLogger(BufferPool.class);
    private static BufferPool ourInstance = new BufferPool();
    private long partSize = 0L;
    private long totalBufferSize = 0L;
    private CosNBufferType bufferType;
    private CosNBufferFactory bufferFactory;
    private BlockingQueue<CosNByteBuffer> bufferPool;
    private AtomicInteger referCount = new AtomicInteger(0);
    private AtomicBoolean isInitialize = new AtomicBoolean(false);

    public static BufferPool getInstance() {
        return ourInstance;
    }

    private BufferPool() {
    }

    public synchronized void initialize(Configuration conf) throws IOException {
        LOG.debug("Initialize the buffer pool.");
        if (this.isInitialize.get()) {
            LOG.debug("Buffer pool: [{}] is initialized and referenced once. current reference count: [{}].", (Object)this, (Object)this.referCount);
            this.referCount.incrementAndGet();
            return;
        }
        this.partSize = conf.getLong("fs.cosn.upload.part.size", 0x800000L);
        if (this.partSize < 0x100000L || this.partSize > 0x80000000L) {
            String exceptionMsg = String.format("The block size of CosN is limited to %d to %d. current block size: %d", 0x100000L, 0x80000000L, this.partSize);
            throw new IllegalArgumentException(exceptionMsg);
        }
        this.bufferType = CosNBufferType.typeFactory(conf.get("fs.cosn.upload.buffer", "mapped_disk"));
        if (null == this.bufferType || CosNBufferType.NON_DIRECT_MEMORY != this.bufferType && CosNBufferType.DIRECT_MEMORY != this.bufferType && CosNBufferType.MAPPED_DISK != this.bufferType) {
            LOG.warn("The [{}] option is set incorrectly, using the default settings: [{}].", (Object)"fs.cosn.upload.buffer", (Object)"mapped_disk");
        }
        this.totalBufferSize = conf.get("fs.cosn.upload.buffer.size") == null ? conf.getLong("fs.cosn.buffer.size", -1L) : conf.getLong("fs.cosn.upload.buffer.size", -1L);
        if (this.totalBufferSize < 0L && -1L != this.totalBufferSize) {
            String errMsg = String.format("Negative buffer size: %d", this.totalBufferSize);
            throw new IllegalArgumentException(errMsg);
        }
        if (this.totalBufferSize == -1L) {
            LOG.info("{} is set to -1, so the 'mapped_disk' buffer will be used by default.", (Object)"fs.cosn.upload.buffer.size");
            this.bufferType = CosNBufferType.MAPPED_DISK;
        }
        LOG.info("The type of the upload buffer pool is [{}]. Buffer size:[{}]", (Object)this.bufferType, (Object)this.totalBufferSize);
        if (this.bufferType == CosNBufferType.NON_DIRECT_MEMORY) {
            this.bufferFactory = new CosNNonDirectBufferFactory();
        } else if (this.bufferType == CosNBufferType.DIRECT_MEMORY) {
            this.bufferFactory = new CosNDirectBufferFactory();
        } else if (this.bufferType == CosNBufferType.MAPPED_DISK) {
            String tmpDir = conf.get("fs.cosn.tmp.dir", "/tmp/hadoop_cos");
            boolean deleteOnExit = conf.getBoolean("fs.cosn.map_disk.delete_on_exit.enabled", true);
            this.bufferFactory = new CosNMappedBufferFactory(tmpDir, deleteOnExit);
        } else {
            String exceptionMsg = String.format("The type of the upload buffer is invalid. buffer type: %s", new Object[]{this.bufferType});
            throw new IllegalArgumentException(exceptionMsg);
        }
        if (this.totalBufferSize > 0L && (CosNBufferType.NON_DIRECT_MEMORY == this.bufferType || CosNBufferType.DIRECT_MEMORY == this.bufferType || CosNBufferType.MAPPED_DISK == this.bufferType)) {
            int bufferNumber = (int)(this.totalBufferSize / this.partSize);
            if (bufferNumber == 0) {
                String errMsg = String.format("The buffer size: [%d] is at least greater than or equal to the size of a block: [%d]", this.totalBufferSize, this.partSize);
                throw new IllegalArgumentException(errMsg);
            }
            LOG.info("Initialize the {} buffer pool. size: {}", (Object)this.bufferType, (Object)bufferNumber);
            this.bufferPool = new LinkedBlockingQueue<CosNByteBuffer>(bufferNumber);
            for (int i = 0; i < bufferNumber; ++i) {
                CosNByteBuffer cosNByteBuffer = this.bufferFactory.create((int)this.partSize);
                if (null == cosNByteBuffer) {
                    String exceptionMsg = String.format("create buffer failed. buffer type: %s, buffer factory: %s", this.bufferType.getName(), this.bufferFactory.getClass().getName());
                    throw new IOException(exceptionMsg);
                }
                this.bufferPool.add(cosNByteBuffer);
            }
        }
        this.referCount.incrementAndGet();
        this.isInitialize.set(true);
    }

    private void checkInitialize() throws IOException {
        if (!this.isInitialize.get()) {
            throw new IOException("The buffer pool has not been initialized yet");
        }
        if (-1L != this.totalBufferSize && null == this.bufferPool) {
            throw new IOException("The buffer pool is null, but the size is not -1(unlimited).");
        }
    }

    public CosNByteBuffer getBuffer(int bufferSize) throws IOException, InterruptedException {
        this.checkInitialize();
        LOG.debug("Get a buffer[size: {}, current buffer size: {}]. Thread [id: {}, name: {}].", new Object[]{bufferSize, this.totalBufferSize, Thread.currentThread().getId(), Thread.currentThread().getName()});
        if (bufferSize > 0 && (long)bufferSize <= this.partSize) {
            if (-1L == this.totalBufferSize) {
                return this.bufferFactory.create(bufferSize);
            }
            return this.bufferPool.poll(Long.MAX_VALUE, TimeUnit.SECONDS);
        }
        String exceptionMsg = String.format("Parameter buffer size out of range: 1 to %d", this.partSize);
        throw new IOException(exceptionMsg);
    }

    public void returnBuffer(CosNByteBuffer buffer) throws IOException {
        LOG.debug("Return a buffer. Thread[id: {}, name: {}].", (Object)Thread.currentThread().getId(), (Object)Thread.currentThread().getName());
        if (null == buffer) {
            LOG.error("The buffer returned is null. Ignore it.");
            return;
        }
        this.checkInitialize();
        if (-1L == this.totalBufferSize) {
            LOG.debug("No buffer pool is maintained, and release the buffer directly.");
            this.bufferFactory.release(buffer);
        } else {
            LOG.debug("Return the buffer to the buffer pool.");
            buffer.clear();
            if (!this.bufferPool.offer(buffer)) {
                LOG.error("Return the buffer to buffer pool failed.");
            }
        }
    }

    public synchronized void close() {
        LOG.info("Close a buffer pool instance.");
        if (!this.isInitialize.get()) {
            LOG.warn("The buffer pool has been closed. no changes would be execute.");
            return;
        }
        if (this.referCount.decrementAndGet() > 0) {
            return;
        }
        LOG.info("Begin to release the buffers.");
        if (null != this.bufferPool) {
            for (CosNByteBuffer buffer : this.bufferPool) {
                this.bufferFactory.release(buffer);
            }
            this.bufferPool.clear();
        }
        if (this.referCount.get() == 0) {
            this.isInitialize.set(false);
        }
    }
}

