/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import com.google.common.base.Preconditions;
import com.qcloud.chdfs.fs.CHDFSHadoopFileSystemAdapter;
import com.qcloud.chdfs.permission.RangerAccessType;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CosNFileSystem;
import org.apache.hadoop.fs.CosNUtils;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileChecksum;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.NativeFileSystemStore;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RangerCredentialsClient;
import org.apache.hadoop.fs.XAttrSetFlag;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.util.Progressable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Stable
public class CosFileSystem
extends FileSystem {
    static final Logger LOG = LoggerFactory.getLogger(CosFileSystem.class);
    static final String SCHEME = "cosn";
    private NativeFileSystemStore nativeStore;
    private boolean isPosixFSStore;
    private boolean isDefaultNativeStore;
    private volatile boolean initialized = false;
    private boolean isPosixUseOFSRanger;
    private boolean isPosixImpl = false;
    private FileSystem actualImplFS = null;
    private URI uri;
    private Path workingDir;
    private RangerCredentialsClient rangerCredentialsClient;

    public CosFileSystem() {
    }

    public CosFileSystem(NativeFileSystemStore nativeStore) {
        this.nativeStore = nativeStore;
        this.isDefaultNativeStore = false;
    }

    public String getScheme() {
        return SCHEME;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public void initialize(URI uri, Configuration conf) throws IOException {
        super.initialize(uri, conf);
        this.setConf(conf);
        UserGroupInformation.setConfiguration((Configuration)conf);
        String bucket = uri.getHost();
        this.uri = URI.create(uri.getScheme() + "://" + uri.getAuthority());
        this.workingDir = new Path("/user", System.getProperty("user.name")).makeQualified(this.uri, this.getWorkingDirectory());
        if (null == this.nativeStore) {
            this.nativeStore = CosNUtils.createDefaultStore(conf);
            this.nativeStore.initialize(uri, conf);
            this.isDefaultNativeStore = true;
        }
        this.rangerCredentialsClient = this.nativeStore.getRangerCredentialsClient();
        this.isPosixUseOFSRanger = this.getConf().getBoolean("fs.cosn.posix.bucket.use_ofs_ranger.enabled", false);
        this.checkCustomAuth(conf);
        this.isPosixFSStore = this.nativeStore.headBucket(bucket).isMergeBucket();
        LOG.info("The cos bucket {} bucket.", (Object)(this.isPosixFSStore ? "is the posix" : "is the normal"));
        if (this.isPosixFSStore) {
            String posixBucketFSImpl = this.getConf().get("fs.cosn.posix_bucket.fs.impl");
            if (null == posixBucketFSImpl) {
                posixBucketFSImpl = "com.qcloud.chdfs.fs.CHDFSHadoopFileSystemAdapter";
                this.getConf().set("fs.cosn.posix_bucket.fs.impl", "com.qcloud.chdfs.fs.CHDFSHadoopFileSystemAdapter");
            }
            LOG.info("The posix bucket [{}] use the class [{}] as the filesystem implementation, use each ranger [{}]", new Object[]{bucket, posixBucketFSImpl, this.isPosixUseOFSRanger});
            this.actualImplFS = CosFileSystem.getActualFileSystemByClassName(posixBucketFSImpl);
            if (this.actualImplFS instanceof CosNFileSystem) {
                this.nativeStore.isPosixBucket(true);
                ((CosNFileSystem)this.actualImplFS).withStore(this.nativeStore).withBucket(bucket).withPosixBucket(this.isPosixFSStore).withRangerCredentialsClient(this.rangerCredentialsClient);
            } else {
                if (!(this.actualImplFS instanceof CHDFSHadoopFileSystemAdapter)) throw new IOException(String.format("The posix bucket does not currently support the implementation [%s].", posixBucketFSImpl));
                this.isPosixImpl = true;
                this.passThroughRangerConfig();
                this.transferOfsConfig();
            }
        } else {
            this.actualImplFS = CosFileSystem.getActualFileSystemByClassName("org.apache.hadoop.fs.CosNFileSystem");
            this.nativeStore.isPosixBucket(false);
            ((CosNFileSystem)this.actualImplFS).withStore(this.nativeStore).withBucket(bucket).withPosixBucket(this.isPosixFSStore).withRangerCredentialsClient(this.rangerCredentialsClient);
        }
        this.actualImplFS.initialize(uri, conf);
        this.initialized = true;
    }

    private static FileSystem getActualFileSystemByClassName(String className) throws IOException {
        try {
            Class<?> actualClass = Class.forName(className);
            return (FileSystem)actualClass.newInstance();
        }
        catch (ClassNotFoundException | IllegalAccessException | InstantiationException e) {
            String errMsg = String.format("load class failed, className: %s", className);
            LOG.error(errMsg, (Throwable)e);
            throw new IOException(errMsg, e);
        }
    }

    public Path getHomeDirectory() {
        return this.actualImplFS.getHomeDirectory();
    }

    public FSDataOutputStream append(Path f, int bufferSize, Progressable progress) throws IOException {
        LOG.debug("append file [{}] in COS.", (Object)f);
        this.checkInitialized();
        this.checkPermission(f, RangerAccessType.WRITE);
        return this.actualImplFS.append(f, bufferSize, progress);
    }

    public boolean truncate(Path f, long newLength) throws IOException {
        LOG.debug("truncate file [{}] in COS.", (Object)f);
        this.checkInitialized();
        this.checkPermission(f, RangerAccessType.WRITE);
        return this.actualImplFS.truncate(f, newLength);
    }

    public FSDataOutputStream create(Path f, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        LOG.debug("Creating a new file [{}] in COS.", (Object)f);
        this.checkInitialized();
        this.checkPermission(f, RangerAccessType.WRITE);
        return this.actualImplFS.create(f, permission, overwrite, bufferSize, replication, blockSize, progress);
    }

    public boolean delete(Path f, boolean recursive) throws IOException {
        LOG.debug("Ready to delete path: {}. recursive: {}.", (Object)f, (Object)recursive);
        this.checkInitialized();
        this.checkPermission(f, RangerAccessType.DELETE);
        return this.actualImplFS.delete(f, recursive);
    }

    public FileStatus getFileStatus(Path f) throws IOException {
        LOG.debug("Get file status: {}.", (Object)f);
        this.checkInitialized();
        return this.actualImplFS.getFileStatus(f);
    }

    public URI getUri() {
        return this.actualImplFS.getUri();
    }

    public FileStatus[] listStatus(Path f) throws FileNotFoundException, IOException {
        LOG.debug("list status:" + f);
        this.checkInitialized();
        this.checkPermission(f, RangerAccessType.LIST);
        return this.actualImplFS.listStatus(f);
    }

    public boolean mkdirs(Path f, FsPermission permission) throws IOException {
        LOG.debug("mkdirs path: {}.", (Object)f);
        this.checkInitialized();
        this.checkPermission(f, RangerAccessType.WRITE);
        return this.actualImplFS.mkdirs(f, permission);
    }

    public FSDataInputStream open(Path f, int bufferSize) throws IOException {
        LOG.debug("Open file [{}] to read, buffer [{}]", (Object)f, (Object)bufferSize);
        this.checkInitialized();
        this.checkPermission(f, RangerAccessType.READ);
        return this.actualImplFS.open(f, bufferSize);
    }

    public boolean rename(Path src, Path dst) throws IOException {
        LOG.debug("Rename the source path [{}] to the dest path [{}].", (Object)src, (Object)dst);
        this.checkInitialized();
        this.checkPermission(src, RangerAccessType.DELETE);
        this.checkPermission(dst, RangerAccessType.WRITE);
        return this.actualImplFS.rename(src, dst);
    }

    public long getDefaultBlockSize() {
        return this.actualImplFS.getDefaultBlockSize();
    }

    public void setWorkingDirectory(Path newDir) {
        this.workingDir = newDir;
        this.actualImplFS.setWorkingDirectory(newDir);
    }

    public Path getWorkingDirectory() {
        return this.workingDir;
    }

    public FileChecksum getFileChecksum(Path f, long length) throws IOException {
        LOG.debug("call the checksum for the path: {}.", (Object)f);
        this.checkInitialized();
        this.checkPermission(f, RangerAccessType.READ);
        Preconditions.checkArgument((length >= 0L ? 1 : 0) != 0);
        return this.actualImplFS.getFileChecksum(f, length);
    }

    public void setXAttr(Path f, String name, byte[] value, EnumSet<XAttrSetFlag> flag) throws IOException {
        LOG.debug("set XAttr: {}.", (Object)f);
        this.checkInitialized();
        this.checkPermission(f, RangerAccessType.WRITE);
        this.actualImplFS.setXAttr(f, name, value, flag);
    }

    public byte[] getXAttr(Path f, String name) throws IOException {
        LOG.debug("get XAttr: {}.", (Object)f);
        this.checkInitialized();
        this.checkPermission(f, RangerAccessType.READ);
        return this.actualImplFS.getXAttr(f, name);
    }

    public Map<String, byte[]> getXAttrs(Path f, List<String> names) throws IOException {
        LOG.debug("get XAttrs: {}.", (Object)f);
        this.checkInitialized();
        this.checkPermission(f, RangerAccessType.READ);
        return this.actualImplFS.getXAttrs(f, names);
    }

    public Map<String, byte[]> getXAttrs(Path f) throws IOException {
        LOG.debug("get XAttrs: {}.", (Object)f);
        this.checkInitialized();
        this.checkPermission(f, RangerAccessType.READ);
        return this.actualImplFS.getXAttrs(f);
    }

    public void removeXAttr(Path f, String name) throws IOException {
        LOG.debug("remove XAttr: {}.", (Object)f);
        this.checkInitialized();
        this.checkPermission(f, RangerAccessType.WRITE);
        this.actualImplFS.removeXAttr(f, name);
    }

    public List<String> listXAttrs(Path f) throws IOException {
        LOG.debug("list XAttrs: {}.", (Object)f);
        this.checkInitialized();
        this.checkPermission(f, RangerAccessType.READ);
        return this.actualImplFS.listXAttrs(f);
    }

    public Token<?> getDelegationToken(String renewer) throws IOException {
        LOG.info("getDelegationToken, renewer: {}, stack: {}", (Object)renewer, (Object)Arrays.toString(Thread.currentThread().getStackTrace()).replace(',', '\n'));
        if (this.useOFSRanger()) {
            return this.actualImplFS.getDelegationToken(renewer);
        }
        Token<?> token = this.rangerCredentialsClient.doGetDelegationToken(renewer);
        if (token != null) {
            return token;
        }
        return super.getDelegationToken(renewer);
    }

    public NativeFileSystemStore getStore() {
        return this.nativeStore;
    }

    private void passThroughRangerConfig() {
        String ofsRangerKey = "fs.cosn.trsf.".concat("fs.ofs.ranger.enable.flag");
        if (this.useOFSRanger()) {
            this.getConf().setBoolean(ofsRangerKey, true);
            return;
        }
        this.getConf().setBoolean(ofsRangerKey, false);
        if (!this.rangerCredentialsClient.isEnableRangerPluginPermissionCheck()) {
            LOG.info("not enable ranger plugin permission check");
            return;
        }
        if (this.rangerCredentialsClient.getRangerPolicyUrl() != null) {
            String policyUrlKey = "fs.cosn.trsf.".concat("fs.ofs.cosn.ranger.policy.url");
            this.getConf().set(policyUrlKey, this.rangerCredentialsClient.getRangerPolicyUrl());
        }
        if (this.rangerCredentialsClient.getAuthJarMd5() != null) {
            String authJarMd5Key = "fs.cosn.trsf.".concat("fs.ofs.cosn.ranger.auth.jar.md5");
            this.getConf().set(authJarMd5Key, this.rangerCredentialsClient.getAuthJarMd5());
        }
    }

    private void transferOfsConfig() {
        HashMap tmpConf = new HashMap();
        for (Map.Entry entry : this.getConf()) {
            if (((String)entry.getKey()).startsWith("fs.ofs.")) {
                this.getConf().unset((String)entry.getKey());
            }
            if (!((String)entry.getKey()).startsWith("fs.cosn.trsf.")) continue;
            int pos = "fs.cosn.trsf.".length();
            String subConfigKey = ((String)entry.getKey()).substring(pos);
            tmpConf.put(subConfigKey, entry.getValue());
        }
        for (Map.Entry entry : tmpConf.entrySet()) {
            LOG.info("Transfer the ofs config. key: {}, value: {}.", entry.getKey(), entry.getValue());
            this.getConf().set((String)entry.getKey(), (String)entry.getValue());
        }
    }

    public void releaseFileLock(Path f) throws IOException {
        LOG.debug("Release the file lock: {}.", (Object)f);
        this.checkInitialized();
        if (!(this.actualImplFS instanceof CHDFSHadoopFileSystemAdapter)) {
            throw new UnsupportedOperationException("Not supported currently");
        }
        ((CHDFSHadoopFileSystemAdapter)this.actualImplFS).releaseFileLock(f);
    }

    public String getCanonicalServiceName() {
        if (this.useOFSRanger()) {
            return this.actualImplFS.getCanonicalServiceName();
        }
        return this.rangerCredentialsClient.doGetCanonicalServiceName();
    }

    private void checkPermission(Path f, RangerAccessType rangerAccessType) throws IOException {
        if (this.useOFSRanger()) {
            return;
        }
        this.rangerCredentialsClient.doCheckPermission(f, rangerAccessType, this.getOwnerId(), this.getWorkingDirectory());
    }

    private boolean useOFSRanger() {
        return this.isPosixImpl && this.isPosixUseOFSRanger;
    }

    private void checkCustomAuth(Configuration conf) throws IOException {
        this.rangerCredentialsClient.doCheckCustomAuth(conf);
    }

    private String getOwnerId() {
        UserGroupInformation currentUgi;
        try {
            currentUgi = UserGroupInformation.getCurrentUser();
        }
        catch (IOException e) {
            LOG.warn("get current user failed! use user.name prop", (Throwable)e);
            return System.getProperty("user.name");
        }
        String shortUserName = "";
        if (currentUgi != null) {
            shortUserName = currentUgi.getShortUserName();
        }
        if (shortUserName == null) {
            LOG.warn("get short user name failed! use user.name prop");
            shortUserName = System.getProperty("user.name");
        }
        return shortUserName;
    }

    private void checkInitialized() throws IOException {
        if (!this.initialized) {
            throw new IOException("The FileSystem has been closed or not initialized.");
        }
    }

    public void close() throws IOException {
        LOG.info("begin to close cos file system");
        this.actualImplFS.close();
        if (null != this.nativeStore && this.isDefaultNativeStore) {
            this.nativeStore.close();
        }
        this.initialized = false;
    }
}

