// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.doris.nereids.trees.expressions.functions.scalar;

import org.apache.doris.catalog.FunctionSignature;
import org.apache.doris.nereids.trees.expressions.Expression;
import org.apache.doris.nereids.trees.expressions.functions.AlwaysNullable;
import org.apache.doris.nereids.trees.expressions.functions.ExplicitlyCastableSignature;
import org.apache.doris.nereids.trees.expressions.literal.IntegerLiteral;
import org.apache.doris.nereids.trees.expressions.literal.Literal;
import org.apache.doris.nereids.trees.expressions.visitor.ExpressionVisitor;
import org.apache.doris.nereids.types.DataType;
import org.apache.doris.nereids.types.IntegerType;
import org.apache.doris.nereids.types.StringType;
import org.apache.doris.nereids.types.VarcharType;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;

import java.util.List;
import java.util.Optional;

/**
 * ScalarFunction 'substring'. This class is generated by GenerateScalarFunction.
 */
// TODO: to be compatible with BE, we set AlwaysNullable here.
public class Substring extends ScalarFunction
        implements ExplicitlyCastableSignature, AlwaysNullable {

    public static final List<FunctionSignature> SIGNATURES = ImmutableList.of(
            FunctionSignature.ret(VarcharType.SYSTEM_DEFAULT).args(VarcharType.SYSTEM_DEFAULT, IntegerType.INSTANCE),
            FunctionSignature.ret(StringType.INSTANCE).args(StringType.INSTANCE, IntegerType.INSTANCE),
            FunctionSignature.ret(VarcharType.SYSTEM_DEFAULT)
                    .args(VarcharType.SYSTEM_DEFAULT, IntegerType.INSTANCE, IntegerType.INSTANCE),
            FunctionSignature.ret(StringType.INSTANCE)
                    .args(StringType.INSTANCE, IntegerType.INSTANCE, IntegerType.INSTANCE)
    );

    /**
     * constructor with 2 arguments.
     */
    public Substring(Expression arg0, Expression arg1) {
        super("substring", arg0, arg1, Literal.of(Integer.MAX_VALUE));
    }

    /**
     * constructor with 3 arguments.
     */
    public Substring(Expression arg0, Expression arg1, Expression arg2) {
        super("substring", arg0, arg1, arg2);
    }

    @Override
    protected FunctionSignature computeSignature(FunctionSignature signature, List<Expression> arguments) {
        Optional<Expression> length = arguments.size() == 3
                ? Optional.of(arguments.get(2)) : Optional.empty();
        DataType returnType = VarcharType.SYSTEM_DEFAULT;
        if (length.isPresent() && length.get() instanceof IntegerLiteral) {
            returnType = VarcharType.createVarcharType(((IntegerLiteral) length.get()).getValue());
        }
        return signature.withReturnType(returnType);
    }

    public Expression getSource() {
        return child(0);
    }

    public Expression getPosition() {
        return child(1);
    }

    public Optional<Expression> getLength() {
        return arity() == 3 ? Optional.of(child(2)) : Optional.empty();
    }

    /**
     * withChildren.
     */
    @Override
    public Substring withChildren(List<Expression> children) {
        Preconditions.checkArgument(children.size() == 2
                || children.size() == 3);
        if (children.size() == 2) {
            return new Substring(children.get(0), children.get(1));
        } else {
            return new Substring(children.get(0), children.get(1), children.get(2));
        }
    }

    @Override
    public List<FunctionSignature> getSignatures() {
        return SIGNATURES;
    }

    @Override
    public <R, C> R accept(ExpressionVisitor<R, C> visitor, C context) {
        return visitor.visitSubstring(this, context);
    }
}
