/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.broker.hdfs;

import com.google.common.base.Strings;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.InetAddress;
import java.net.URI;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.nio.channels.FileLock;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.doris.broker.hdfs.BrokerConfig;
import org.apache.doris.broker.hdfs.BrokerException;
import org.apache.doris.broker.hdfs.BrokerFileSystem;
import org.apache.doris.broker.hdfs.ClientContextManager;
import org.apache.doris.broker.hdfs.FileSystemIdentity;
import org.apache.doris.common.WildcardURI;
import org.apache.doris.thrift.TBrokerFD;
import org.apache.doris.thrift.TBrokerFileStatus;
import org.apache.doris.thrift.TBrokerOperationStatusCode;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.util.StringUtils;
import org.apache.log4j.Logger;

public class FileSystemManager {
    private static Logger logger = Logger.getLogger((String)FileSystemManager.class.getName());
    private static final String HDFS_SCHEME = "hdfs";
    private static final String S3A_SCHEME = "s3a";
    private static final String KS3_SCHEME = "ks3";
    private static final String USER_NAME_KEY = "username";
    private static final String PASSWORD_KEY = "password";
    private static final String AUTHENTICATION_SIMPLE = "simple";
    private static final String AUTHENTICATION_KERBEROS = "kerberos";
    private static final String KERBEROS_PRINCIPAL = "kerberos_principal";
    private static final String KERBEROS_KEYTAB = "kerberos_keytab";
    private static final String KERBEROS_KEYTAB_CONTENT = "kerberos_keytab_content";
    private static final String DFS_HA_NAMENODE_KERBEROS_PRINCIPAL_PATTERN = "dfs.namenode.kerberos.principal.pattern";
    private static final String DFS_NAMESERVICES_KEY = "dfs.nameservices";
    private static final String DFS_HA_NAMENODES_PREFIX = "dfs.ha.namenodes.";
    private static final String DFS_HA_NAMENODE_RPC_ADDRESS_PREFIX = "dfs.namenode.rpc-address.";
    private static final String DFS_CLIENT_FAILOVER_PROXY_PROVIDER_PREFIX = "dfs.client.failover.proxy.provider.";
    private static final String DEFAULT_DFS_CLIENT_FAILOVER_PROXY_PROVIDER = "org.apache.hadoop.hdfs.server.namenode.ha.ConfiguredFailoverProxyProvider";
    private static final String FS_DEFAULTFS_KEY = "fs.defaultFS";
    private static final String FS_HDFS_IMPL_DISABLE_CACHE = "fs.hdfs.impl.disable.cache";
    private static final String FS_S3A_ACCESS_KEY = "fs.s3a.access.key";
    private static final String FS_S3A_SECRET_KEY = "fs.s3a.secret.key";
    private static final String FS_S3A_ENDPOINT = "fs.s3a.endpoint";
    private static final String FS_S3A_IMPL_DISABLE_CACHE = "fs.s3a.impl.disable.cache";
    private static final String FS_KS3_ACCESS_KEY = "fs.ks3.AccessKey";
    private static final String FS_KS3_SECRET_KEY = "fs.ks3.AccessSecret";
    private static final String FS_KS3_ENDPOINT = "fs.ks3.endpoint";
    private static final String FS_KS3_IMPL = "fs.ks3.impl";
    private static final String FS_KS3_IMPL_DISABLE_CACHE = "fs.ks3.impl.disable.cache";
    private ScheduledExecutorService handleManagementPool = Executors.newScheduledThreadPool(2);
    private int readBufferSize = 131072;
    private int writeBufferSize = 131072;
    private ConcurrentHashMap<FileSystemIdentity, BrokerFileSystem> cachedFileSystem = new ConcurrentHashMap();
    private ClientContextManager clientContextManager = new ClientContextManager(this.handleManagementPool);

    public FileSystemManager() {
        this.readBufferSize = BrokerConfig.hdfs_read_buffer_size_kb << 10;
        this.writeBufferSize = BrokerConfig.hdfs_write_buffer_size_kb << 10;
        this.handleManagementPool.schedule(new FileSystemExpirationChecker(), 0L, TimeUnit.SECONDS);
    }

    private static String preparePrincipal(String originalPrincipal) throws UnknownHostException {
        String finalPrincipal = originalPrincipal;
        String[] components = originalPrincipal.split("[/@]");
        if (components != null && components.length == 3) {
            if (components[1].equals("_HOST")) {
                finalPrincipal = components[0] + "/" + StringUtils.toLowerCase((String)InetAddress.getLocalHost().getCanonicalHostName()) + "@" + components[2];
            } else if (components[1].equals("_IP")) {
                finalPrincipal = components[0] + "/" + InetAddress.getByName(InetAddress.getLocalHost().getCanonicalHostName()).getHostAddress() + "@" + components[2];
            }
        }
        return finalPrincipal;
    }

    public BrokerFileSystem getFileSystem(String path, Map<String, String> properties) {
        WildcardURI pathUri = new WildcardURI(path);
        String scheme = pathUri.getUri().getScheme();
        if (Strings.isNullOrEmpty((String)scheme)) {
            throw new BrokerException(TBrokerOperationStatusCode.INVALID_INPUT_FILE_PATH, "invalid path. scheme is null", new Object[0]);
        }
        BrokerFileSystem brokerFileSystem = null;
        if (scheme.equals(HDFS_SCHEME)) {
            brokerFileSystem = this.getDistributedFileSystem(path, properties);
        } else if (scheme.equals(S3A_SCHEME)) {
            brokerFileSystem = this.getS3AFileSystem(path, properties);
        } else if (scheme.equals(KS3_SCHEME)) {
            brokerFileSystem = this.getKS3FileSystem(path, properties);
        } else {
            throw new BrokerException(TBrokerOperationStatusCode.INVALID_INPUT_FILE_PATH, "invalid path. scheme is not supported", new Object[0]);
        }
        return brokerFileSystem;
    }

    public BrokerFileSystem getDistributedFileSystem(String path, Map<String, String> properties) {
        String kerberosContent;
        WildcardURI pathUri = new WildcardURI(path);
        String host = "hdfs://" + pathUri.getAuthority();
        if (Strings.isNullOrEmpty((String)pathUri.getAuthority())) {
            if (properties.containsKey(FS_DEFAULTFS_KEY)) {
                host = properties.get(FS_DEFAULTFS_KEY);
                logger.info((Object)"no schema and authority in path. use fs.defaultFs");
            } else {
                logger.warn((Object)("invalid hdfs path. authority is null,path:" + path));
                throw new BrokerException(TBrokerOperationStatusCode.INVALID_ARGUMENT, "invalid hdfs path. authority is null", new Object[0]);
            }
        }
        String username = properties.getOrDefault(USER_NAME_KEY, "");
        String password = properties.getOrDefault(PASSWORD_KEY, "");
        String dfsNameServices = properties.getOrDefault(DFS_NAMESERVICES_KEY, "");
        String authentication = properties.getOrDefault("hadoop.security.authentication", AUTHENTICATION_SIMPLE);
        if (Strings.isNullOrEmpty((String)authentication) || !authentication.equals(AUTHENTICATION_SIMPLE) && !authentication.equals(AUTHENTICATION_KERBEROS)) {
            logger.warn((Object)("invalid authentication:" + authentication));
            throw new BrokerException(TBrokerOperationStatusCode.INVALID_ARGUMENT, "invalid authentication:" + authentication, new Object[0]);
        }
        String hdfsUgi = username + "," + password;
        FileSystemIdentity fileSystemIdentity = null;
        BrokerFileSystem fileSystem = null;
        if (authentication.equals(AUTHENTICATION_SIMPLE)) {
            fileSystemIdentity = new FileSystemIdentity(host, hdfsUgi);
        } else {
            kerberosContent = "";
            if (properties.containsKey(KERBEROS_KEYTAB)) {
                kerberosContent = properties.get(KERBEROS_KEYTAB);
            } else if (properties.containsKey(KERBEROS_KEYTAB_CONTENT)) {
                kerberosContent = properties.get(KERBEROS_KEYTAB_CONTENT);
            } else {
                throw new BrokerException(TBrokerOperationStatusCode.INVALID_ARGUMENT, "keytab is required for kerberos authentication", new Object[0]);
            }
            if (!properties.containsKey(KERBEROS_PRINCIPAL)) {
                throw new BrokerException(TBrokerOperationStatusCode.INVALID_ARGUMENT, "principal is required for kerberos authentication", new Object[0]);
            }
            kerberosContent = kerberosContent + properties.get(KERBEROS_PRINCIPAL);
            try {
                MessageDigest digest = MessageDigest.getInstance("md5");
                byte[] result = digest.digest(kerberosContent.getBytes());
                String kerberosUgi = new String(result);
                fileSystemIdentity = new FileSystemIdentity(host, kerberosUgi);
            }
            catch (NoSuchAlgorithmException e) {
                throw new BrokerException(TBrokerOperationStatusCode.INVALID_ARGUMENT, e.getMessage(), new Object[0]);
            }
        }
        this.cachedFileSystem.putIfAbsent(fileSystemIdentity, new BrokerFileSystem(fileSystemIdentity));
        fileSystem = this.cachedFileSystem.get(fileSystemIdentity);
        if (fileSystem == null) {
            return null;
        }
        fileSystem.getLock().lock();
        try {
            BrokerFileSystem ugi;
            if (!this.cachedFileSystem.containsKey(fileSystemIdentity)) {
                kerberosContent = null;
                return kerberosContent;
            }
            if (fileSystem.getDFSFileSystem() == null) {
                logger.info((Object)("create file system for new path: " + path));
                ugi = null;
                HdfsConfiguration conf = new HdfsConfiguration();
                conf.set("ipc.client.fallback-to-simple-auth-allowed", "true");
                String tmpFilePath = null;
                if (authentication.equals(AUTHENTICATION_KERBEROS)) {
                    conf.set("hadoop.security.authentication", AUTHENTICATION_KERBEROS);
                    String principal = FileSystemManager.preparePrincipal(properties.get(KERBEROS_PRINCIPAL));
                    String keytab = "";
                    if (properties.containsKey(KERBEROS_KEYTAB)) {
                        keytab = properties.get(KERBEROS_KEYTAB);
                    } else if (properties.containsKey(KERBEROS_KEYTAB_CONTENT)) {
                        String keytab_content = properties.get(KERBEROS_KEYTAB_CONTENT);
                        byte[] byArray = Base64.getDecoder().decode(keytab_content);
                        long currentTime = System.currentTimeMillis();
                        Random random = new Random(currentTime);
                        int randNumber = random.nextInt(10000);
                        tmpFilePath = "/tmp/." + principal.replace('/', '_') + "_" + Long.toString(currentTime) + "_" + Integer.toString(randNumber) + "_" + Thread.currentThread().getId();
                        logger.info((Object)("create kerberos tmp file" + tmpFilePath));
                        FileOutputStream fileOutputStream = new FileOutputStream(tmpFilePath);
                        FileLock lock = fileOutputStream.getChannel().lock();
                        fileOutputStream.write(byArray);
                        lock.release();
                        fileOutputStream.close();
                        keytab = tmpFilePath;
                    } else {
                        throw new BrokerException(TBrokerOperationStatusCode.INVALID_ARGUMENT, "keytab is required for kerberos authentication", new Object[0]);
                    }
                    UserGroupInformation.setConfiguration((Configuration)conf);
                    ugi = UserGroupInformation.loginUserFromKeytabAndReturnUGI((String)principal, (String)keytab);
                    if (properties.containsKey(KERBEROS_KEYTAB_CONTENT)) {
                        try {
                            File file = new File(tmpFilePath);
                            if (!file.delete()) {
                                logger.warn((Object)("delete tmp file:" + tmpFilePath + " failed"));
                            }
                        }
                        catch (Exception e) {
                            throw new BrokerException(TBrokerOperationStatusCode.FILE_NOT_FOUND, e.getMessage(), new Object[0]);
                        }
                    }
                }
                if (!Strings.isNullOrEmpty((String)dfsNameServices)) {
                    String dfsHaNameNodesKey = DFS_HA_NAMENODES_PREFIX + dfsNameServices;
                    if (!properties.containsKey(dfsHaNameNodesKey)) {
                        throw new BrokerException(TBrokerOperationStatusCode.INVALID_ARGUMENT, "load request missed necessary arguments for ha mode", new Object[0]);
                    }
                    String dfsHaNameNodes = properties.get(dfsHaNameNodesKey);
                    conf.set(DFS_NAMESERVICES_KEY, dfsNameServices);
                    conf.set(dfsHaNameNodesKey, dfsHaNameNodes);
                    String[] nameNodes = dfsHaNameNodes.split(",");
                    if (nameNodes == null) {
                        throw new BrokerException(TBrokerOperationStatusCode.INVALID_ARGUMENT, "invalid " + dfsHaNameNodesKey + " configuration", new Object[0]);
                    }
                    for (String nameNode : nameNodes) {
                        nameNode = nameNode.trim();
                        String nameNodeRpcAddress = DFS_HA_NAMENODE_RPC_ADDRESS_PREFIX + dfsNameServices + "." + nameNode;
                        if (!properties.containsKey(nameNodeRpcAddress)) {
                            throw new BrokerException(TBrokerOperationStatusCode.INVALID_ARGUMENT, "missed " + nameNodeRpcAddress + " configuration", new Object[0]);
                        }
                        conf.set(nameNodeRpcAddress, properties.get(nameNodeRpcAddress));
                    }
                    String string = DFS_CLIENT_FAILOVER_PROXY_PROVIDER_PREFIX + dfsNameServices;
                    if (properties.containsKey(string)) {
                        conf.set(string, properties.get(string));
                    } else {
                        conf.set(string, DEFAULT_DFS_CLIENT_FAILOVER_PROXY_PROVIDER);
                    }
                    if (properties.containsKey(FS_DEFAULTFS_KEY)) {
                        conf.set(FS_DEFAULTFS_KEY, properties.get(FS_DEFAULTFS_KEY));
                    }
                    if (properties.containsKey(DFS_HA_NAMENODE_KERBEROS_PRINCIPAL_PATTERN)) {
                        conf.set(DFS_HA_NAMENODE_KERBEROS_PRINCIPAL_PATTERN, properties.get(DFS_HA_NAMENODE_KERBEROS_PRINCIPAL_PATTERN));
                    }
                }
                conf.set(FS_HDFS_IMPL_DISABLE_CACHE, "true");
                FileSystem dfsFileSystem = null;
                if (authentication.equals(AUTHENTICATION_SIMPLE) && properties.containsKey(USER_NAME_KEY) && !Strings.isNullOrEmpty((String)username)) {
                    ugi = UserGroupInformation.createRemoteUser((String)username);
                    conf.set("hadoop.security.authentication", AUTHENTICATION_SIMPLE);
                    ugi.setAuthenticationMethod(UserGroupInformation.AuthenticationMethod.SIMPLE);
                }
                dfsFileSystem = ugi != null ? (FileSystem)ugi.doAs(() -> FileSystemManager.lambda$getDistributedFileSystem$0(pathUri, (Configuration)conf)) : FileSystem.get((URI)pathUri.getUri(), (Configuration)conf);
                fileSystem.setFileSystem(dfsFileSystem);
            }
            ugi = fileSystem;
            return ugi;
        }
        catch (Exception e) {
            logger.error((Object)("errors while connect to " + path), (Throwable)e);
            throw new BrokerException(TBrokerOperationStatusCode.NOT_AUTHORIZED, (Throwable)e);
        }
        finally {
            fileSystem.getLock().unlock();
        }
    }

    public BrokerFileSystem getS3AFileSystem(String path, Map<String, String> properties) {
        WildcardURI pathUri = new WildcardURI(path);
        String accessKey = properties.getOrDefault(FS_S3A_ACCESS_KEY, "");
        String secretKey = properties.getOrDefault(FS_S3A_SECRET_KEY, "");
        String endpoint = properties.getOrDefault(FS_S3A_ENDPOINT, "");
        String host = "s3a://" + endpoint;
        String disableCache = properties.getOrDefault(FS_S3A_IMPL_DISABLE_CACHE, "true");
        String s3aUgi = accessKey + "," + secretKey;
        FileSystemIdentity fileSystemIdentity = new FileSystemIdentity(host, s3aUgi);
        BrokerFileSystem fileSystem = null;
        this.cachedFileSystem.putIfAbsent(fileSystemIdentity, new BrokerFileSystem(fileSystemIdentity));
        fileSystem = this.cachedFileSystem.get(fileSystemIdentity);
        if (fileSystem == null) {
            return null;
        }
        fileSystem.getLock().lock();
        try {
            BrokerFileSystem conf;
            if (!this.cachedFileSystem.containsKey(fileSystemIdentity)) {
                BrokerFileSystem brokerFileSystem = null;
                return brokerFileSystem;
            }
            if (fileSystem.getDFSFileSystem() == null) {
                logger.info((Object)("create file system for new path " + path));
                conf = new Configuration();
                conf.set(FS_S3A_ACCESS_KEY, accessKey);
                conf.set(FS_S3A_SECRET_KEY, secretKey);
                conf.set(FS_S3A_ENDPOINT, endpoint);
                conf.set(FS_S3A_IMPL_DISABLE_CACHE, disableCache);
                FileSystem s3AFileSystem = FileSystem.get((URI)pathUri.getUri(), (Configuration)conf);
                fileSystem.setFileSystem(s3AFileSystem);
            }
            conf = fileSystem;
            return conf;
        }
        catch (Exception e) {
            logger.error((Object)("errors while connect to " + path), (Throwable)e);
            throw new BrokerException(TBrokerOperationStatusCode.NOT_AUTHORIZED, (Throwable)e);
        }
        finally {
            fileSystem.getLock().unlock();
        }
    }

    public BrokerFileSystem getKS3FileSystem(String path, Map<String, String> properties) {
        WildcardURI pathUri = new WildcardURI(path);
        String accessKey = properties.getOrDefault(FS_KS3_ACCESS_KEY, "");
        String secretKey = properties.getOrDefault(FS_KS3_SECRET_KEY, "");
        String endpoint = properties.getOrDefault(FS_KS3_ENDPOINT, "");
        String disableCache = properties.getOrDefault(FS_KS3_IMPL_DISABLE_CACHE, "true");
        String host = "ks3://" + endpoint + "/" + pathUri.getUri().getHost();
        String ks3aUgi = accessKey + "," + secretKey;
        FileSystemIdentity fileSystemIdentity = new FileSystemIdentity(host, ks3aUgi);
        BrokerFileSystem fileSystem = null;
        this.cachedFileSystem.putIfAbsent(fileSystemIdentity, new BrokerFileSystem(fileSystemIdentity));
        fileSystem = this.cachedFileSystem.get(fileSystemIdentity);
        if (fileSystem == null) {
            return null;
        }
        fileSystem.getLock().lock();
        try {
            BrokerFileSystem conf;
            if (!this.cachedFileSystem.containsKey(fileSystemIdentity)) {
                BrokerFileSystem brokerFileSystem = null;
                return brokerFileSystem;
            }
            if (fileSystem.getDFSFileSystem() == null) {
                logger.info((Object)("could not find file system for path " + path + " create a new one"));
                conf = new Configuration();
                conf.set(FS_KS3_ACCESS_KEY, accessKey);
                conf.set(FS_KS3_SECRET_KEY, secretKey);
                conf.set(FS_KS3_ENDPOINT, endpoint);
                conf.set(FS_KS3_IMPL, "com.ksyun.kmr.hadoop.fs.ks3.Ks3FileSystem");
                conf.set(FS_KS3_IMPL_DISABLE_CACHE, disableCache);
                FileSystem ks3FileSystem = FileSystem.get((URI)pathUri.getUri(), (Configuration)conf);
                fileSystem.setFileSystem(ks3FileSystem);
            }
            conf = fileSystem;
            return conf;
        }
        catch (Exception e) {
            logger.error((Object)("errors while connect to " + path), (Throwable)e);
            throw new BrokerException(TBrokerOperationStatusCode.NOT_AUTHORIZED, (Throwable)e);
        }
        finally {
            fileSystem.getLock().unlock();
        }
    }

    public List<TBrokerFileStatus> listPath(String path, boolean fileNameOnly, Map<String, String> properties) {
        ArrayList<TBrokerFileStatus> resultFileStatus = null;
        WildcardURI pathUri = new WildcardURI(path);
        BrokerFileSystem fileSystem = this.getFileSystem(path, properties);
        Path pathPattern = new Path(pathUri.getPath());
        try {
            FileStatus[] files = fileSystem.getDFSFileSystem().globStatus(pathPattern);
            if (files == null) {
                resultFileStatus = new ArrayList<TBrokerFileStatus>(0);
                return resultFileStatus;
            }
            resultFileStatus = new ArrayList(files.length);
            for (FileStatus fileStatus : files) {
                TBrokerFileStatus brokerFileStatus = new TBrokerFileStatus();
                brokerFileStatus.setIsDir(fileStatus.isDirectory());
                if (fileStatus.isDirectory()) {
                    brokerFileStatus.setIsSplitable(false);
                    brokerFileStatus.setSize(-1L);
                } else {
                    brokerFileStatus.setSize(fileStatus.getLen());
                    brokerFileStatus.setIsSplitable(true);
                }
                if (fileNameOnly) {
                    brokerFileStatus.setPath(fileStatus.getPath().getName());
                } else {
                    brokerFileStatus.setPath(fileStatus.getPath().toString());
                }
                resultFileStatus.add(brokerFileStatus);
            }
        }
        catch (FileNotFoundException e) {
            logger.info((Object)("file not found: " + e.getMessage()));
            throw new BrokerException(TBrokerOperationStatusCode.FILE_NOT_FOUND, (Throwable)e, "file not found", new Object[0]);
        }
        catch (Exception e) {
            logger.error((Object)"errors while get file status ", (Throwable)e);
            fileSystem.closeFileSystem();
            throw new BrokerException(TBrokerOperationStatusCode.TARGET_STORAGE_SERVICE_ERROR, (Throwable)e, "unknown error when get file status", new Object[0]);
        }
        return resultFileStatus;
    }

    public void deletePath(String path, Map<String, String> properties) {
        WildcardURI pathUri = new WildcardURI(path);
        BrokerFileSystem fileSystem = this.getFileSystem(path, properties);
        Path filePath = new Path(pathUri.getPath());
        try {
            fileSystem.getDFSFileSystem().delete(filePath, true);
        }
        catch (IOException e) {
            logger.error((Object)("errors while delete path " + path));
            fileSystem.closeFileSystem();
            throw new BrokerException(TBrokerOperationStatusCode.TARGET_STORAGE_SERVICE_ERROR, (Throwable)e, "delete path {} error", path);
        }
    }

    public void renamePath(String srcPath, String destPath, Map<String, String> properties) {
        WildcardURI srcPathUri = new WildcardURI(srcPath);
        WildcardURI destPathUri = new WildcardURI(destPath);
        if (!srcPathUri.getAuthority().trim().equals(destPathUri.getAuthority().trim())) {
            throw new BrokerException(TBrokerOperationStatusCode.TARGET_STORAGE_SERVICE_ERROR, "only allow rename in same file system", new Object[0]);
        }
        BrokerFileSystem fileSystem = this.getFileSystem(srcPath, properties);
        Path srcfilePath = new Path(srcPathUri.getPath());
        Path destfilePath = new Path(destPathUri.getPath());
        try {
            boolean isRenameSuccess = fileSystem.getDFSFileSystem().rename(srcfilePath, destfilePath);
            if (!isRenameSuccess) {
                throw new BrokerException(TBrokerOperationStatusCode.TARGET_STORAGE_SERVICE_ERROR, "failed to rename path from {} to {}", srcPath, destPath);
            }
        }
        catch (IOException e) {
            logger.error((Object)("errors while rename path from " + srcPath + " to " + destPath));
            fileSystem.closeFileSystem();
            throw new BrokerException(TBrokerOperationStatusCode.TARGET_STORAGE_SERVICE_ERROR, (Throwable)e, "errors while rename {} to {}", srcPath, destPath);
        }
    }

    public boolean checkPathExist(String path, Map<String, String> properties) {
        WildcardURI pathUri = new WildcardURI(path);
        BrokerFileSystem fileSystem = this.getFileSystem(path, properties);
        Path filePath = new Path(pathUri.getPath());
        try {
            boolean isPathExist = fileSystem.getDFSFileSystem().exists(filePath);
            return isPathExist;
        }
        catch (IOException e) {
            logger.error((Object)("errors while check path exist: " + path));
            fileSystem.closeFileSystem();
            throw new BrokerException(TBrokerOperationStatusCode.TARGET_STORAGE_SERVICE_ERROR, (Throwable)e, "errors while check if path {} exist", path);
        }
    }

    public TBrokerFD openReader(String clientId, String path, long startOffset, Map<String, String> properties) {
        WildcardURI pathUri = new WildcardURI(path);
        Path inputFilePath = new Path(pathUri.getPath());
        BrokerFileSystem fileSystem = this.getFileSystem(path, properties);
        try {
            FSDataInputStream fsDataInputStream = fileSystem.getDFSFileSystem().open(inputFilePath, this.readBufferSize);
            fsDataInputStream.seek(startOffset);
            UUID uuid = UUID.randomUUID();
            TBrokerFD fd = FileSystemManager.parseUUIDToFD(uuid);
            this.clientContextManager.putNewInputStream(clientId, fd, fsDataInputStream, fileSystem);
            return fd;
        }
        catch (IOException e) {
            logger.error((Object)"errors while open path", (Throwable)e);
            fileSystem.closeFileSystem();
            throw new BrokerException(TBrokerOperationStatusCode.TARGET_STORAGE_SERVICE_ERROR, (Throwable)e, "could not open file {}", path);
        }
    }

    public ByteBuffer pread(TBrokerFD fd, long offset, long length) {
        FSDataInputStream fsDataInputStream;
        FSDataInputStream fSDataInputStream = fsDataInputStream = this.clientContextManager.getFsDataInputStream(fd);
        synchronized (fSDataInputStream) {
            long currentStreamOffset;
            try {
                currentStreamOffset = fsDataInputStream.getPos();
            }
            catch (IOException e) {
                logger.error((Object)"errors while get file pos from output stream", (Throwable)e);
                throw new BrokerException(TBrokerOperationStatusCode.TARGET_STORAGE_SERVICE_ERROR, "errors while get file pos from output stream", new Object[0]);
            }
            if (currentStreamOffset != offset) {
                logger.debug((Object)("invalid offset, current read offset is " + currentStreamOffset + " is not equal to request offset " + offset + " seek to it"));
                try {
                    fsDataInputStream.seek(offset);
                }
                catch (IOException e) {
                    throw new BrokerException(TBrokerOperationStatusCode.INVALID_INPUT_OFFSET, (Throwable)e, "current read offset {} is not equal to {}, and could not seek to it", currentStreamOffset, offset);
                }
            }
            byte[] buf = length > (long)this.readBufferSize ? new byte[this.readBufferSize] : new byte[(int)length];
            try {
                int readLength = this.readBytesFully(fsDataInputStream, buf);
                if (readLength < 0) {
                    throw new BrokerException(TBrokerOperationStatusCode.END_OF_FILE, "end of file reached", new Object[0]);
                }
                if (logger.isDebugEnabled()) {
                    logger.debug((Object)("read buffer from input stream, buffer size:" + buf.length + ", read length:" + readLength));
                }
                return ByteBuffer.wrap(buf, 0, readLength);
            }
            catch (IOException e) {
                logger.error((Object)"errors while read data from stream", (Throwable)e);
                throw new BrokerException(TBrokerOperationStatusCode.TARGET_STORAGE_SERVICE_ERROR, (Throwable)e, "errors while write data to output stream", new Object[0]);
            }
        }
    }

    public void seek(TBrokerFD fd, long offset) {
        throw new BrokerException(TBrokerOperationStatusCode.OPERATION_NOT_SUPPORTED, "seek this method is not supported", new Object[0]);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void closeReader(TBrokerFD fd) {
        FSDataInputStream fsDataInputStream;
        FSDataInputStream fSDataInputStream = fsDataInputStream = this.clientContextManager.getFsDataInputStream(fd);
        synchronized (fSDataInputStream) {
            try {
                fsDataInputStream.close();
            }
            catch (IOException e) {
                logger.error((Object)"errors while close file input stream", (Throwable)e);
                throw new BrokerException(TBrokerOperationStatusCode.TARGET_STORAGE_SERVICE_ERROR, (Throwable)e, "errors while close file input stream", new Object[0]);
            }
            finally {
                this.clientContextManager.removeInputStream(fd);
            }
        }
    }

    public TBrokerFD openWriter(String clientId, String path, Map<String, String> properties) {
        WildcardURI pathUri = new WildcardURI(path);
        Path inputFilePath = new Path(pathUri.getPath());
        BrokerFileSystem fileSystem = this.getFileSystem(path, properties);
        try {
            FSDataOutputStream fsDataOutputStream = fileSystem.getDFSFileSystem().create(inputFilePath, true, this.writeBufferSize);
            UUID uuid = UUID.randomUUID();
            TBrokerFD fd = FileSystemManager.parseUUIDToFD(uuid);
            this.clientContextManager.putNewOutputStream(clientId, fd, fsDataOutputStream, fileSystem);
            return fd;
        }
        catch (IOException e) {
            logger.error((Object)"errors while open path", (Throwable)e);
            fileSystem.closeFileSystem();
            throw new BrokerException(TBrokerOperationStatusCode.TARGET_STORAGE_SERVICE_ERROR, (Throwable)e, "could not open file {}", path);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void pwrite(TBrokerFD fd, long offset, byte[] data) {
        FSDataOutputStream fsDataOutputStream;
        FSDataOutputStream fSDataOutputStream = fsDataOutputStream = this.clientContextManager.getFsDataOutputStream(fd);
        synchronized (fSDataOutputStream) {
            long currentStreamOffset;
            try {
                currentStreamOffset = fsDataOutputStream.getPos();
            }
            catch (IOException e) {
                logger.error((Object)"errors while get file pos from output stream", (Throwable)e);
                throw new BrokerException(TBrokerOperationStatusCode.TARGET_STORAGE_SERVICE_ERROR, "errors while get file pos from output stream", new Object[0]);
            }
            if (currentStreamOffset != offset) {
                throw new BrokerException(TBrokerOperationStatusCode.INVALID_INPUT_OFFSET, "current outputstream offset is {} not equal to request {}", currentStreamOffset, offset);
            }
            try {
                fsDataOutputStream.write(data);
            }
            catch (IOException e) {
                logger.error((Object)"errors while write data to output stream", (Throwable)e);
                throw new BrokerException(TBrokerOperationStatusCode.TARGET_STORAGE_SERVICE_ERROR, (Throwable)e, "errors while write data to output stream", new Object[0]);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void closeWriter(TBrokerFD fd) {
        FSDataOutputStream fsDataOutputStream;
        FSDataOutputStream fSDataOutputStream = fsDataOutputStream = this.clientContextManager.getFsDataOutputStream(fd);
        synchronized (fSDataOutputStream) {
            try {
                fsDataOutputStream.flush();
                fsDataOutputStream.close();
            }
            catch (IOException e) {
                logger.error((Object)"errors while close file output stream", (Throwable)e);
                throw new BrokerException(TBrokerOperationStatusCode.TARGET_STORAGE_SERVICE_ERROR, (Throwable)e, "errors while close file output stream", new Object[0]);
            }
            finally {
                this.clientContextManager.removeOutputStream(fd);
            }
        }
    }

    public void ping(String clientId) {
        this.clientContextManager.onPing(clientId);
    }

    private static TBrokerFD parseUUIDToFD(UUID uuid) {
        return new TBrokerFD(uuid.getMostSignificantBits(), uuid.getLeastSignificantBits());
    }

    private int readBytesFully(FSDataInputStream is, byte[] dest) throws IOException {
        int availableReadLength;
        int readLength;
        int n;
        for (readLength = 0; readLength < dest.length && (n = is.read(dest, readLength, availableReadLength = dest.length - readLength)) > 0; readLength += n) {
        }
        return readLength;
    }

    private static /* synthetic */ FileSystem lambda$getDistributedFileSystem$0(WildcardURI pathUri, Configuration conf) throws Exception {
        return FileSystem.get((URI)pathUri.getUri(), (Configuration)conf);
    }

    class FileSystemExpirationChecker
    implements Runnable {
        FileSystemExpirationChecker() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                for (BrokerFileSystem fileSystem : FileSystemManager.this.cachedFileSystem.values()) {
                    if (!fileSystem.isExpired(BrokerConfig.client_expire_seconds)) continue;
                    logger.info((Object)("file system " + fileSystem + " is expired, close and remove it"));
                    fileSystem.getLock().lock();
                    try {
                        fileSystem.closeFileSystem();
                    }
                    catch (Throwable t) {
                        logger.error((Object)"errors while close file system", t);
                    }
                    finally {
                        FileSystemManager.this.cachedFileSystem.remove(fileSystem.getIdentity());
                        fileSystem.getLock().unlock();
                    }
                }
            }
            finally {
                FileSystemManager.this.handleManagementPool.schedule(this, 60L, TimeUnit.SECONDS);
            }
        }
    }
}

