/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.task;

import com.google.common.collect.Maps;
import java.util.HashMap;
import java.util.Map;
import org.apache.doris.catalog.Catalog;
import org.apache.doris.catalog.Database;
import org.apache.doris.catalog.DistributionInfo;
import org.apache.doris.catalog.MaterializedIndex;
import org.apache.doris.catalog.OlapTable;
import org.apache.doris.catalog.Partition;
import org.apache.doris.catalog.Tablet;
import org.apache.doris.common.LoadException;
import org.apache.doris.common.MetaNotFoundException;
import org.apache.doris.common.Pair;
import org.apache.doris.load.FailMsg;
import org.apache.doris.load.Load;
import org.apache.doris.load.LoadChecker;
import org.apache.doris.load.LoadJob;
import org.apache.doris.load.PartitionLoadInfo;
import org.apache.doris.load.TableLoadInfo;
import org.apache.doris.load.TabletLoadInfo;
import org.apache.doris.task.MasterTask;
import org.apache.doris.thrift.TEtlState;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public abstract class LoadEtlTask
extends MasterTask {
    private static final Logger LOG = LogManager.getLogger(LoadEtlTask.class);
    protected static final String QUALITY_FAIL_MSG = "quality not good enough to cancel";
    public static final String DPP_NORMAL_ALL = "dpp.norm.ALL";
    public static final String DPP_ABNORMAL_ALL = "dpp.abnorm.ALL";
    protected final LoadJob job;
    protected final Load load;
    protected Database db;

    public LoadEtlTask(LoadJob job) {
        this.job = job;
        this.signature = job.getId();
        this.load = Catalog.getCurrentCatalog().getLoadInstance();
    }

    protected String getErrorMsg() {
        return "etl job fail";
    }

    @Override
    protected void exec() {
        if (this.job.getState() != LoadJob.JobState.ETL) {
            return;
        }
        if (LoadChecker.checkTimeout(this.job)) {
            this.load.cancelLoadJob(this.job, FailMsg.CancelType.TIMEOUT, "etl timeout to cancel");
            return;
        }
        long dbId = this.job.getDbId();
        this.db = Catalog.getCurrentCatalog().getDbNullable(dbId);
        if (this.db == null) {
            this.load.cancelLoadJob(this.job, FailMsg.CancelType.ETL_RUN_FAIL, "db does not exist. id: " + dbId);
            return;
        }
        if (this.job.getProgress() != 100) {
            try {
                this.updateEtlStatus();
            }
            catch (LoadException e) {
                FailMsg.CancelType cancelType = FailMsg.CancelType.ETL_RUN_FAIL;
                if (e.getMessage().equals(QUALITY_FAIL_MSG)) {
                    cancelType = FailMsg.CancelType.ETL_QUALITY_UNSATISFIED;
                }
                LOG.debug("update etl status fail, msg: {}. job: {}", (Object)e.getMessage(), (Object)this.job.toString());
                this.load.cancelLoadJob(this.job, cancelType, e.getMessage());
                return;
            }
        }
        if (this.job.getProgress() == 100) {
            this.tryUpdateLoading();
        }
    }

    private void updateEtlStatus() throws LoadException {
        if (!this.updateJobEtlStatus()) {
            throw new LoadException("update job etl status fail");
        }
        TEtlState state = this.job.getEtlJobStatus().getState();
        switch (state) {
            case FINISHED: {
                this.processEtlFinished();
                break;
            }
            case CANCELLED: {
                throw new LoadException(this.getErrorMsg());
            }
            case RUNNING: {
                this.processEtlRunning();
                break;
            }
            default: {
                LOG.warn("wrong etl job state: {}", (Object)state.name());
            }
        }
    }

    private void processEtlFinished() throws LoadException {
        if (!this.checkDataQuality(this.job.getMaxFilterRatio())) {
            throw new LoadException(QUALITY_FAIL_MSG);
        }
        Map<String, Pair<String, Long>> filePathMap = this.getFilePathMap();
        Map<Long, TabletLoadInfo> idToTabletLoadInfo = this.getTabletLoadInfos(filePathMap);
        this.job.setIdToTabletLoadInfo(idToTabletLoadInfo);
        this.job.setProgress(100);
        this.job.setEtlFinishTimeMs(System.currentTimeMillis());
        this.job.getEtlJobStatus().setFileMap(null);
        LOG.info("etl job finished. job: {}", (Object)this.job);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void tryUpdateLoading() {
        Map<Long, TableLoadInfo> idToTableLoadInfo = this.job.getIdToTableLoadInfo();
        try {
            for (Map.Entry<Long, TableLoadInfo> tableEntry : idToTableLoadInfo.entrySet()) {
                long tableId = tableEntry.getKey();
                OlapTable table = (OlapTable)this.db.getTableOrMetaException(tableId);
                TableLoadInfo tableLoadInfo = tableEntry.getValue();
                Map<Long, PartitionLoadInfo> idToPartitionLoadInfo = tableLoadInfo.getIdToPartitionLoadInfo();
                for (Map.Entry<Long, PartitionLoadInfo> entry : idToPartitionLoadInfo.entrySet()) {
                    long partitionId = entry.getKey();
                    PartitionLoadInfo partitionLoadInfo = entry.getValue();
                    if (!partitionLoadInfo.isNeedLoad()) continue;
                    table.readLock();
                    try {
                        Partition partition = table.getPartition(partitionId);
                        if (partition == null) {
                            throw new MetaNotFoundException("partition does not exist. id: " + partitionId);
                        }
                    }
                    finally {
                        table.readUnlock();
                    }
                    LOG.info("load job id: {}, label: {}, partition info: {}-{}-{}, partition load info: {}", (Object)this.job.getId(), (Object)this.job.getLabel(), (Object)this.db.getId(), (Object)tableId, (Object)partitionId, (Object)partitionLoadInfo);
                }
            }
        }
        catch (MetaNotFoundException e) {
            this.load.cancelLoadJob(this.job, FailMsg.CancelType.ETL_RUN_FAIL, e.getMessage());
            return;
        }
        if (this.load.updateLoadJobState(this.job, LoadJob.JobState.LOADING)) {
            LOG.info("update job state to loading success. job: {}", (Object)this.job);
        } else {
            LOG.warn("update job state to loading failed. job: {}", (Object)this.job);
            if (this.job.getTransactionId() > 0L) {
                LOG.warn("there maybe remaining transaction id {} in transaction table", (Object)this.job.getTransactionId());
            }
        }
    }

    protected String getPartitionIndexBucketString(String filePath) throws LoadException {
        String fileName = filePath.substring(filePath.lastIndexOf("/") + 1);
        String[] fileNameArr = fileName.split("\\.");
        if (fileNameArr.length != 4) {
            throw new LoadException("etl file name format error, name: " + fileName);
        }
        String partitionIndexBucket = fileName.substring(fileName.indexOf(".") + 1);
        return partitionIndexBucket;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected Map<Long, TabletLoadInfo> getTabletLoadInfos(Map<String, Pair<String, Long>> filePathMap) throws LoadException {
        HashMap idToTabletLoadInfo = Maps.newHashMap();
        boolean hasLoadFiles = false;
        Map<Long, TableLoadInfo> idToTableLoadInfo = this.job.getIdToTableLoadInfo();
        for (Map.Entry<Long, TableLoadInfo> tableEntry : idToTableLoadInfo.entrySet()) {
            long tableId = tableEntry.getKey();
            OlapTable table = (OlapTable)this.db.getTableOrException(tableId, s -> new LoadException("table does not exist. id: " + s));
            table.readLock();
            try {
                TableLoadInfo tableLoadInfo = tableEntry.getValue();
                for (Map.Entry<Long, PartitionLoadInfo> partitionEntry : tableLoadInfo.getIdToPartitionLoadInfo().entrySet()) {
                    long partitionId = partitionEntry.getKey();
                    boolean needLoad = false;
                    Partition partition = table.getPartition(partitionId);
                    if (partition == null) {
                        throw new LoadException("partition does not exist. id: " + partitionId);
                    }
                    DistributionInfo distributionInfo = partition.getDistributionInfo();
                    DistributionInfo.DistributionInfoType distributionType = distributionInfo.getType();
                    if (distributionType != DistributionInfo.DistributionInfoType.RANDOM && distributionType != DistributionInfo.DistributionInfoType.HASH) {
                        throw new LoadException("unknown distribution type. type: " + distributionType.name());
                    }
                    for (MaterializedIndex materializedIndex : partition.getMaterializedIndices(MaterializedIndex.IndexExtState.ALL)) {
                        long indexId = materializedIndex.getId();
                        int tabletIndex = 0;
                        for (Tablet tablet : materializedIndex.getTablets()) {
                            long bucket = tabletIndex++;
                            String tableViewBucket = String.format("%d.%d.%d", partitionId, indexId, bucket);
                            String filePath = null;
                            long fileSize = -1L;
                            if (filePathMap.containsKey(tableViewBucket)) {
                                Pair<String, Long> filePair = filePathMap.get(tableViewBucket);
                                filePath = (String)filePair.first;
                                fileSize = (Long)filePair.second;
                                needLoad = true;
                                hasLoadFiles = true;
                            }
                            TabletLoadInfo tabletLoadInfo = new TabletLoadInfo(filePath, fileSize);
                            idToTabletLoadInfo.put(tablet.getId(), tabletLoadInfo);
                        }
                    }
                    partitionEntry.getValue().setNeedLoad(needLoad);
                }
            }
            finally {
                table.readUnlock();
            }
        }
        if (!hasLoadFiles) {
            throw new LoadException("all partitions have no load data");
        }
        return idToTabletLoadInfo;
    }

    protected boolean checkDataQuality(double maxFilterRatio) {
        Map<String, String> counters = this.job.getEtlJobStatus().getCounters();
        if (!counters.containsKey(DPP_NORMAL_ALL) || !counters.containsKey(DPP_ABNORMAL_ALL)) {
            return true;
        }
        long normalNum = Long.parseLong(counters.get(DPP_NORMAL_ALL));
        long abnormalNum = Long.parseLong(counters.get(DPP_ABNORMAL_ALL));
        return !((double)abnormalNum > (double)(abnormalNum + normalNum) * maxFilterRatio);
    }

    protected abstract boolean updateJobEtlStatus();

    protected abstract void processEtlRunning() throws LoadException;

    protected abstract Map<String, Pair<String, Long>> getFilePathMap() throws LoadException;
}

