/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.planner;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import org.apache.doris.analysis.Analyzer;
import org.apache.doris.analysis.BinaryPredicate;
import org.apache.doris.analysis.Expr;
import org.apache.doris.analysis.ExprSubstitutionMap;
import org.apache.doris.analysis.SlotDescriptor;
import org.apache.doris.analysis.SlotId;
import org.apache.doris.analysis.SlotRef;
import org.apache.doris.analysis.TupleId;
import org.apache.doris.catalog.Type;
import org.apache.doris.common.IdGenerator;
import org.apache.doris.common.util.BitUtil;
import org.apache.doris.planner.HashJoinNode;
import org.apache.doris.planner.OlapScanNode;
import org.apache.doris.planner.PlanNode;
import org.apache.doris.planner.RuntimeFilter;
import org.apache.doris.planner.RuntimeFilterId;
import org.apache.doris.planner.ScanNode;
import org.apache.doris.qe.ConnectContext;
import org.apache.doris.qe.SessionVariable;
import org.apache.doris.thrift.TRuntimeFilterMode;
import org.apache.doris.thrift.TRuntimeFilterType;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public final class RuntimeFilterGenerator {
    private static final Logger LOG = LogManager.getLogger(RuntimeFilterGenerator.class);
    private final Map<TupleId, List<RuntimeFilter>> runtimeFiltersByTid = new HashMap<TupleId, List<RuntimeFilter>>();
    private final IdGenerator<RuntimeFilterId> filterIdGenerator = RuntimeFilterId.createGenerator();
    private final FilterSizeLimits bloomFilterSizeLimits;
    private final Analyzer analyzer;
    private final SessionVariable sessionVariable;

    private RuntimeFilterGenerator(Analyzer analyzer) {
        this.analyzer = analyzer;
        this.sessionVariable = ConnectContext.get().getSessionVariable();
        Preconditions.checkNotNull((Object)this.sessionVariable);
        this.bloomFilterSizeLimits = new FilterSizeLimits(this.sessionVariable);
    }

    public static void generateRuntimeFilters(Analyzer analyzer, PlanNode plan) {
        Preconditions.checkNotNull((Object)analyzer);
        int maxNumBloomFilters = ConnectContext.get().getSessionVariable().getRuntimeFiltersMaxNum();
        int runtimeFilterType = ConnectContext.get().getSessionVariable().getRuntimeFilterType();
        Preconditions.checkState((maxNumBloomFilters >= 0 ? 1 : 0) != 0);
        RuntimeFilterGenerator filterGenerator = new RuntimeFilterGenerator(analyzer);
        Preconditions.checkState((runtimeFilterType >= 0 ? 1 : 0) != 0, (Object)"runtimeFilterType not expected");
        Preconditions.checkState((runtimeFilterType <= Arrays.stream(TRuntimeFilterType.values()).mapToInt(TRuntimeFilterType::getValue).sum() ? 1 : 0) != 0, (Object)"runtimeFilterType not expected");
        filterGenerator.generateFilters(plan);
        List<RuntimeFilter> filters = filterGenerator.getRuntimeFilters();
        if (filters.size() > maxNumBloomFilters) {
            filters.sort((a, b) -> {
                double aSelectivity = a.getSelectivity() == -1.0 ? Double.MAX_VALUE : a.getSelectivity();
                double bSelectivity = b.getSelectivity() == -1.0 ? Double.MAX_VALUE : b.getSelectivity();
                return Double.compare(aSelectivity, bSelectivity);
            });
        }
        int numBloomFilters = 0;
        for (RuntimeFilter filter : filters) {
            filter.extractTargetsPosition();
            if (filter.getType() == TRuntimeFilterType.BLOOM) {
                if (numBloomFilters >= maxNumBloomFilters) continue;
                ++numBloomFilters;
            }
            filter.registerToPlan(analyzer);
        }
    }

    public List<RuntimeFilter> getRuntimeFilters() {
        HashSet<RuntimeFilter> resultSet = new HashSet<RuntimeFilter>();
        for (List<RuntimeFilter> filters : this.runtimeFiltersByTid.values()) {
            resultSet.addAll(filters);
        }
        ArrayList resultList = Lists.newArrayList(resultSet);
        resultList.sort((a, b) -> a.getFilterId().compareTo(b.getFilterId()));
        return resultList;
    }

    private void generateFilters(PlanNode root) {
        if (root instanceof HashJoinNode) {
            HashJoinNode joinNode = (HashJoinNode)root;
            ArrayList<BinaryPredicate> joinConjuncts = new ArrayList<BinaryPredicate>();
            if (!(joinNode.getJoinOp().isLeftOuterJoin() || joinNode.getJoinOp().isFullOuterJoin() || joinNode.getJoinOp().isAntiJoin())) {
                joinConjuncts.addAll(joinNode.getEqJoinConjuncts());
            }
            ArrayList<RuntimeFilter> filters = new ArrayList<RuntimeFilter>();
            for (TRuntimeFilterType type : TRuntimeFilterType.values()) {
                if ((this.sessionVariable.getRuntimeFilterType() & type.getValue()) == 0) continue;
                for (int i = 0; i < joinConjuncts.size(); ++i) {
                    Expr conjunct = (Expr)joinConjuncts.get(i);
                    RuntimeFilter filter = RuntimeFilter.create(this.filterIdGenerator, this.analyzer, conjunct, i, joinNode, type, this.bloomFilterSizeLimits);
                    if (filter == null) continue;
                    this.registerRuntimeFilter(filter);
                    filters.add(filter);
                }
            }
            this.generateFilters((PlanNode)root.getChild(0));
            for (RuntimeFilter runtimeFilter : filters) {
                this.finalizeRuntimeFilter(runtimeFilter);
            }
            this.generateFilters((PlanNode)root.getChild(1));
        } else if (root instanceof ScanNode) {
            this.assignRuntimeFilters((ScanNode)root);
        } else {
            for (PlanNode childNode : root.getChildren()) {
                this.generateFilters(childNode);
            }
        }
    }

    private void registerRuntimeFilter(RuntimeFilter filter) {
        Map<TupleId, List<SlotId>> targetSlotsByTid = filter.getTargetSlots();
        Preconditions.checkState((targetSlotsByTid != null && !targetSlotsByTid.isEmpty() ? 1 : 0) != 0);
        for (TupleId tupleId : targetSlotsByTid.keySet()) {
            this.registerRuntimeFilter(filter, tupleId);
        }
    }

    private void registerRuntimeFilter(RuntimeFilter filter, TupleId targetTid) {
        Preconditions.checkState((boolean)filter.getTargetSlots().containsKey(targetTid));
        List filters = this.runtimeFiltersByTid.computeIfAbsent(targetTid, k -> new ArrayList());
        Preconditions.checkState((!filter.isFinalized() ? 1 : 0) != 0);
        filters.add(filter);
    }

    private void finalizeRuntimeFilter(RuntimeFilter runtimeFilter) {
        HashSet<TupleId> targetTupleIds = new HashSet<TupleId>();
        for (RuntimeFilter.RuntimeFilterTarget target : runtimeFilter.getTargets()) {
            targetTupleIds.addAll(target.node.getTupleIds());
        }
        for (TupleId tupleId : runtimeFilter.getTargetSlots().keySet()) {
            if (targetTupleIds.contains(tupleId)) continue;
            this.runtimeFiltersByTid.get(tupleId).remove(runtimeFilter);
        }
        runtimeFilter.markFinalized();
    }

    private void assignRuntimeFilters(ScanNode scanNode) {
        if (!(scanNode instanceof OlapScanNode)) {
            return;
        }
        TupleId tid = scanNode.getTupleIds().get(0);
        if (!this.runtimeFiltersByTid.containsKey(tid)) {
            return;
        }
        String runtimeFilterMode = this.sessionVariable.getRuntimeFilterMode();
        Preconditions.checkState((boolean)Arrays.stream(TRuntimeFilterMode.values()).map(Enum::name).anyMatch(p -> p.equals(runtimeFilterMode.toUpperCase())), (Object)"runtimeFilterMode not expected");
        for (RuntimeFilter filter : this.runtimeFiltersByTid.get(tid)) {
            Expr targetExpr;
            if (filter.isFinalized() || (targetExpr = this.computeTargetExpr(filter, tid)) == null) continue;
            boolean isBoundByKeyColumns = RuntimeFilterGenerator.isBoundByKeyColumns(this.analyzer, targetExpr, scanNode);
            boolean isLocalTarget = RuntimeFilterGenerator.isLocalTarget(filter, scanNode);
            if (runtimeFilterMode.equals(TRuntimeFilterMode.LOCAL.name()) && !isLocalTarget || runtimeFilterMode.equals(TRuntimeFilterMode.REMOTE.name()) && isLocalTarget) continue;
            RuntimeFilter.RuntimeFilterTarget target = new RuntimeFilter.RuntimeFilterTarget(scanNode, targetExpr, isBoundByKeyColumns, isLocalTarget);
            filter.addTarget(target);
        }
    }

    private static boolean isLocalTarget(RuntimeFilter filter, ScanNode targetNode) {
        return targetNode.getFragment().getId().equals(filter.getBuilderNode().getFragment().getId());
    }

    private static boolean isBoundByKeyColumns(Analyzer analyzer, Expr targetExpr, ScanNode targetNode) {
        Preconditions.checkState((boolean)targetExpr.isBoundByTupleIds(targetNode.getTupleIds()));
        ArrayList<SlotId> sids = new ArrayList<SlotId>();
        targetExpr.getIds(null, sids);
        for (SlotId sid : sids) {
            SlotDescriptor slotDesc = analyzer.getSlotDesc(sid);
            if (slotDesc.getColumn() != null && slotDesc.getColumn().isKey()) continue;
            return false;
        }
        return true;
    }

    private Expr computeTargetExpr(RuntimeFilter filter, TupleId targetTid) {
        Expr targetExpr = filter.getOrigTargetExpr();
        if (!targetExpr.isBound(targetTid)) {
            Preconditions.checkState((boolean)filter.getTargetSlots().containsKey(targetTid));
            ExprSubstitutionMap smap = new ExprSubstitutionMap();
            ArrayList exprSlots = new ArrayList();
            targetExpr.collect(SlotRef.class, exprSlots);
            List<SlotId> sids = filter.getTargetSlots().get(targetTid);
            block4: for (SlotRef slotRef : exprSlots) {
                for (SlotId sid : sids) {
                    if (!this.analyzer.hasValueTransfer(slotRef.getSlotId(), sid)) continue;
                    SlotRef newSlotRef = new SlotRef(this.analyzer.getSlotDesc(sid));
                    newSlotRef.analyzeNoThrow(this.analyzer);
                    smap.put(slotRef, newSlotRef);
                    continue block4;
                }
            }
            Preconditions.checkState((exprSlots.size() == smap.size() ? 1 : 0) != 0);
            try {
                targetExpr = targetExpr.substitute(smap, this.analyzer, false);
            }
            catch (Exception e) {
                return null;
            }
        }
        Type srcType = filter.getSrcExpr().getType();
        if (!targetExpr.getType().equals(srcType)) {
            try {
                targetExpr = targetExpr.castTo(srcType);
            }
            catch (Exception e) {
                return null;
            }
        }
        return targetExpr;
    }

    public static class FilterSizeLimits {
        public final long maxVal;
        public final long minVal;
        public final long defaultVal;

        public FilterSizeLimits(SessionVariable sessionVariable) {
            long maxLimit = sessionVariable.getRuntimeBloomFilterMaxSize();
            this.maxVal = BitUtil.roundUpToPowerOf2(maxLimit);
            long minLimit = sessionVariable.getRuntimeBloomFilterMinSize();
            this.minVal = BitUtil.roundUpToPowerOf2(Math.min(minLimit, this.maxVal));
            long defaultValue = sessionVariable.getRuntimeBloomFilterSize();
            defaultValue = Math.max(defaultValue, this.minVal);
            this.defaultVal = BitUtil.roundUpToPowerOf2(Math.min(defaultValue, this.maxVal));
        }
    }
}

