/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.load.sync;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.gson.annotations.SerializedName;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import org.apache.doris.analysis.BinlogDesc;
import org.apache.doris.analysis.ChannelDescription;
import org.apache.doris.analysis.CreateDataSyncJobStmt;
import org.apache.doris.catalog.Catalog;
import org.apache.doris.catalog.Database;
import org.apache.doris.catalog.Table;
import org.apache.doris.common.Config;
import org.apache.doris.common.DdlException;
import org.apache.doris.common.FeConstants;
import org.apache.doris.common.UserException;
import org.apache.doris.common.io.Text;
import org.apache.doris.common.io.Writable;
import org.apache.doris.common.util.LogBuilder;
import org.apache.doris.common.util.LogKey;
import org.apache.doris.common.util.TimeUtils;
import org.apache.doris.load.sync.DataSyncJobType;
import org.apache.doris.load.sync.SyncChannel;
import org.apache.doris.load.sync.SyncFailMsg;
import org.apache.doris.load.sync.canal.CanalSyncJob;
import org.apache.doris.persist.gson.GsonUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public abstract class SyncJob
implements Writable {
    private static final Logger LOG = LogManager.getLogger(SyncJob.class);
    @SerializedName(value="id")
    protected long id;
    @SerializedName(value="dbId")
    protected long dbId;
    @SerializedName(value="jobName")
    protected String jobName;
    @SerializedName(value="channelDescriptions")
    protected List<ChannelDescription> channelDescriptions;
    protected BinlogDesc binlogDesc;
    @SerializedName(value="createTimeMs")
    protected long createTimeMs;
    @SerializedName(value="lastStartTimeMs")
    protected long lastStartTimeMs;
    @SerializedName(value="lastStopTimeMs")
    protected long lastStopTimeMs;
    @SerializedName(value="finishTimeMs")
    protected long finishTimeMs;
    @SerializedName(value="jobState")
    protected JobState jobState;
    @SerializedName(value="failMsg")
    protected SyncFailMsg failMsg;
    @SerializedName(value="dataSyncJobType")
    protected DataSyncJobType dataSyncJobType;
    protected List<SyncChannel> channels;

    public SyncJob(long id, String jobName, long dbId) {
        this.id = id;
        this.dbId = dbId;
        this.jobName = jobName;
        this.jobState = JobState.PENDING;
        this.createTimeMs = System.currentTimeMillis();
        this.lastStartTimeMs = -1L;
        this.lastStopTimeMs = -1L;
        this.finishTimeMs = -1L;
    }

    public static SyncJob fromStmt(long jobId, CreateDataSyncJobStmt stmt) throws DdlException {
        String dbName = stmt.getDbName();
        Database db = Catalog.getCurrentCatalog().getDbOrDdlException(dbName);
        try {
            CanalSyncJob syncJob;
            switch (stmt.getDataSyncJobType()) {
                case CANAL: {
                    syncJob = new CanalSyncJob(jobId, stmt.getJobName(), db.getId());
                    break;
                }
                default: {
                    throw new DdlException("Unknown load job type.");
                }
            }
            syncJob.setChannelDescriptions(stmt.getChannelDescriptions());
            ((SyncJob)syncJob).checkAndSetBinlogInfo(stmt.getBinlogDesc());
            return syncJob;
        }
        catch (Exception e) {
            throw new DdlException(e.getMessage());
        }
    }

    public boolean isCompleted() {
        return this.jobState == JobState.CANCELLED;
    }

    public boolean isPaused() {
        return this.jobState == JobState.PAUSED;
    }

    public boolean isRunning() {
        return this.jobState == JobState.RUNNING;
    }

    public boolean isCancelled() {
        return this.jobState == JobState.CANCELLED;
    }

    public boolean isNeedReschedule() {
        return false;
    }

    public synchronized void updateState(JobState newState, boolean isReplay) throws UserException {
        this.checkStateTransform(newState);
        this.unprotectedUpdateState(newState, isReplay);
    }

    public void unprotectedUpdateState(JobState newState, boolean isReplay) {
        this.jobState = newState;
        switch (newState) {
            case PENDING: {
                break;
            }
            case RUNNING: {
                this.lastStartTimeMs = System.currentTimeMillis();
                break;
            }
            case PAUSED: {
                this.lastStopTimeMs = System.currentTimeMillis();
                break;
            }
            case CANCELLED: {
                this.lastStopTimeMs = System.currentTimeMillis();
                this.finishTimeMs = System.currentTimeMillis();
                break;
            }
            default: {
                Preconditions.checkState((boolean)false, (Object)("wrong job state: " + newState.name()));
            }
        }
        if (!isReplay) {
            SyncJobUpdateStateInfo info = new SyncJobUpdateStateInfo(this.id, this.jobState, this.lastStartTimeMs, this.lastStopTimeMs, this.finishTimeMs, this.failMsg);
            Catalog.getCurrentCatalog().getEditLog().logUpdateSyncJobState(info);
        }
    }

    private void checkStateTransform(JobState newState) throws UserException {
        switch (this.jobState) {
            case PENDING: {
                break;
            }
            case RUNNING: {
                if (newState != JobState.RUNNING) break;
                throw new DdlException("Could not transform " + (Object)((Object)this.jobState) + " to " + (Object)((Object)newState));
            }
            case PAUSED: {
                if (newState != JobState.PAUSED && newState != JobState.RUNNING) break;
                throw new DdlException("Could not transform " + (Object)((Object)this.jobState) + " to " + (Object)((Object)newState));
            }
            case CANCELLED: {
                throw new DdlException("Could not transform " + (Object)((Object)this.jobState) + " to " + (Object)((Object)newState));
            }
        }
    }

    public void checkAndDoUpdate() throws UserException {
        Database database = Catalog.getCurrentCatalog().getDbNullable(this.dbId);
        if (database == null) {
            if (!this.isCompleted()) {
                String msg = "The database has been deleted. Change job state to cancelled";
                LOG.warn(new LogBuilder(LogKey.SYNC_JOB, this.id).add("database", this.dbId).add("msg", msg).build());
                this.cancel(SyncFailMsg.MsgType.SCHEDULE_FAIL, msg);
            }
            return;
        }
        for (ChannelDescription channelDescription : this.channelDescriptions) {
            Table table = database.getTableNullable(channelDescription.getTargetTable());
            if (table != null) continue;
            if (!this.isCompleted()) {
                String msg = "The table has been deleted. Change job state to cancelled";
                LOG.warn(new LogBuilder(LogKey.SYNC_JOB, this.id).add("dbId", this.dbId).add("table", channelDescription.getTargetTable()).add("msg", msg).build());
                this.cancel(SyncFailMsg.MsgType.SCHEDULE_FAIL, msg);
            }
            return;
        }
        if (this.isNeedReschedule()) {
            LOG.info(new LogBuilder(LogKey.SYNC_JOB, this.id).add("msg", "Job need to be scheduled").build());
            this.updateState(JobState.PENDING, false);
        }
    }

    public void checkAndSetBinlogInfo(BinlogDesc binlogDesc) throws DdlException {
        this.binlogDesc = binlogDesc;
    }

    public abstract void execute() throws UserException;

    public void cancel(SyncFailMsg.MsgType msgType, String errMsg) {
    }

    public void pause() throws UserException {
        throw new UserException("not implemented");
    }

    public void resume() throws UserException {
        throw new UserException("not implemented");
    }

    public String getStatus() {
        return "\\N";
    }

    public String getJobConfig() {
        return "\\N";
    }

    public boolean isExpired(long currentTimeMs) {
        if (!this.isCompleted()) {
            return false;
        }
        Preconditions.checkState((this.finishTimeMs != -1L ? 1 : 0) != 0);
        long expireTime = (long)Config.label_keep_max_second * 1000L;
        return currentTimeMs - this.finishTimeMs > expireTime;
    }

    public List<Comparable> getShowInfo() {
        ArrayList jobInfo = Lists.newArrayList();
        jobInfo.add(this.id);
        jobInfo.add(this.jobName);
        jobInfo.add(this.dataSyncJobType.name());
        jobInfo.add(this.jobState.name());
        StringBuilder channelInfo = new StringBuilder();
        if (this.channels != null) {
            for (int i = 0; i < this.channels.size(); ++i) {
                channelInfo.append(this.channels.get(i).getInfo());
                if (i >= this.channels.size() - 1) continue;
                channelInfo.append(", ");
            }
            jobInfo.add(channelInfo.toString());
        } else {
            jobInfo.add(FeConstants.null_string);
        }
        jobInfo.add(this.getStatus());
        jobInfo.add(this.getJobConfig());
        jobInfo.add(TimeUtils.longToTimeString(this.createTimeMs));
        jobInfo.add(TimeUtils.longToTimeString(this.lastStartTimeMs));
        jobInfo.add(TimeUtils.longToTimeString(this.lastStopTimeMs));
        jobInfo.add(TimeUtils.longToTimeString(this.finishTimeMs));
        if (this.failMsg == null) {
            jobInfo.add(FeConstants.null_string);
        } else {
            jobInfo.add(this.failMsg.toString());
        }
        return jobInfo;
    }

    public void replayUpdateSyncJobState(SyncJobUpdateStateInfo info) {
        this.lastStartTimeMs = info.getLastStartTimeMs();
        this.lastStopTimeMs = info.getLastStopTimeMs();
        this.finishTimeMs = info.getFinishTimeMs();
        try {
            this.updateState(info.getJobState(), true);
        }
        catch (UserException e) {
            LOG.error("replay update state error, which should not happen: {}", (Object)e.getMessage());
        }
        LOG.info(new LogBuilder(LogKey.SYNC_JOB, info.getId()).add("desired_state:", (Object)info.getJobState()).add("msg", "Replay update sync job state").build());
    }

    public void write(DataOutput out) throws IOException {
        Text.writeString((DataOutput)out, (String)GsonUtils.GSON.toJson((Object)this, SyncJob.class));
    }

    public static SyncJob read(DataInput in) throws IOException {
        String json = Text.readString((DataInput)in);
        return (SyncJob)GsonUtils.GSON.fromJson(json, SyncJob.class);
    }

    public void setChannelDescriptions(List<ChannelDescription> channelDescriptions) throws DdlException {
        this.channelDescriptions = channelDescriptions;
        HashMap tableMappings = Maps.newHashMap();
        for (ChannelDescription channelDescription : channelDescriptions) {
            String src = channelDescription.getSrcDatabase() + "." + channelDescription.getSrcTableName();
            String tar = channelDescription.getTargetTable();
            tableMappings.put(src, tar);
        }
        HashSet mappingSet = Sets.newHashSet(tableMappings.values());
        if (mappingSet.size() != tableMappings.size() || mappingSet.size() != channelDescriptions.size()) {
            throw new DdlException("The mapping relations between tables should be injective.");
        }
        for (ChannelDescription channelDescription : channelDescriptions) {
            channelDescription.setChannelId(Catalog.getCurrentCatalog().getNextId());
        }
    }

    public long getId() {
        return this.id;
    }

    public long getDbId() {
        return this.dbId;
    }

    public String getJobName() {
        return this.jobName;
    }

    public JobState getJobState() {
        return this.jobState;
    }

    public DataSyncJobType getJobType() {
        return this.dataSyncJobType;
    }

    public SyncFailMsg getFailMsg() {
        return this.failMsg;
    }

    public void setFailMsg(SyncFailMsg failMsg) {
        this.failMsg = failMsg;
    }

    public List<ChannelDescription> getChannelDescriptions() {
        return this.channelDescriptions;
    }

    public long getFinishTimeMs() {
        return this.finishTimeMs;
    }

    public static class SyncJobUpdateStateInfo
    implements Writable {
        @SerializedName(value="id")
        private long id;
        @SerializedName(value="lastStartTimeMs")
        protected long lastStartTimeMs;
        @SerializedName(value="lastStopTimeMs")
        protected long lastStopTimeMs;
        @SerializedName(value="finishTimeMs")
        protected long finishTimeMs;
        @SerializedName(value="jobState")
        protected JobState jobState;
        @SerializedName(value="failMsg")
        protected SyncFailMsg failMsg;

        public SyncJobUpdateStateInfo(long id, JobState jobState, long lastStartTimeMs, long lastStopTimeMs, long finishTimeMs, SyncFailMsg failMsg) {
            this.id = id;
            this.jobState = jobState;
            this.lastStartTimeMs = lastStartTimeMs;
            this.lastStopTimeMs = lastStopTimeMs;
            this.finishTimeMs = finishTimeMs;
            this.failMsg = failMsg;
        }

        public long getId() {
            return this.id;
        }

        public long getLastStartTimeMs() {
            return this.lastStartTimeMs;
        }

        public long getLastStopTimeMs() {
            return this.lastStopTimeMs;
        }

        public long getFinishTimeMs() {
            return this.finishTimeMs;
        }

        public JobState getJobState() {
            return this.jobState;
        }

        public SyncFailMsg getFailMsg() {
            return this.failMsg;
        }

        public String toString() {
            return GsonUtils.GSON.toJson((Object)this);
        }

        public void write(DataOutput out) throws IOException {
            String json = GsonUtils.GSON.toJson((Object)this);
            Text.writeString((DataOutput)out, (String)json);
        }

        public static SyncJobUpdateStateInfo read(DataInput in) throws IOException {
            String json = Text.readString((DataInput)in);
            return (SyncJobUpdateStateInfo)GsonUtils.GSON.fromJson(json, SyncJobUpdateStateInfo.class);
        }
    }

    public static enum JobState {
        PENDING,
        RUNNING,
        PAUSED,
        CANCELLED;

    }
}

