/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.load.routineload;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;
import com.google.common.collect.Queues;
import java.util.List;
import java.util.concurrent.LinkedBlockingQueue;
import org.apache.doris.catalog.Catalog;
import org.apache.doris.common.ClientPool;
import org.apache.doris.common.Config;
import org.apache.doris.common.InternalErrorCode;
import org.apache.doris.common.LoadException;
import org.apache.doris.common.MetaNotFoundException;
import org.apache.doris.common.UserException;
import org.apache.doris.common.util.DebugUtil;
import org.apache.doris.common.util.LogBuilder;
import org.apache.doris.common.util.LogKey;
import org.apache.doris.common.util.MasterDaemon;
import org.apache.doris.load.routineload.ErrorReason;
import org.apache.doris.load.routineload.RoutineLoadJob;
import org.apache.doris.load.routineload.RoutineLoadManager;
import org.apache.doris.load.routineload.RoutineLoadTaskInfo;
import org.apache.doris.system.Backend;
import org.apache.doris.thrift.BackendService;
import org.apache.doris.thrift.TNetworkAddress;
import org.apache.doris.thrift.TRoutineLoadTask;
import org.apache.doris.thrift.TStatus;
import org.apache.doris.thrift.TStatusCode;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class RoutineLoadTaskScheduler
extends MasterDaemon {
    private static final Logger LOG = LogManager.getLogger(RoutineLoadTaskScheduler.class);
    private static final long BACKEND_SLOT_UPDATE_INTERVAL_MS = 10000L;
    private static final long SLOT_FULL_SLEEP_MS = 10000L;
    private RoutineLoadManager routineLoadManager;
    private LinkedBlockingQueue<RoutineLoadTaskInfo> needScheduleTasksQueue = Queues.newLinkedBlockingQueue();
    private long lastBackendSlotUpdateTime = -1L;

    @VisibleForTesting
    public RoutineLoadTaskScheduler() {
        super("Routine load task scheduler", 0L);
        this.routineLoadManager = Catalog.getCurrentCatalog().getRoutineLoadManager();
    }

    public RoutineLoadTaskScheduler(RoutineLoadManager routineLoadManager) {
        super("Routine load task scheduler", 0L);
        this.routineLoadManager = routineLoadManager;
    }

    @Override
    protected void runAfterCatalogReady() {
        try {
            this.process();
        }
        catch (Throwable e) {
            LOG.warn("Failed to process one round of RoutineLoadTaskScheduler", e);
        }
    }

    private void process() throws UserException, InterruptedException {
        this.updateBackendSlotIfNecessary();
        long start = System.currentTimeMillis();
        int idleSlotNum = this.routineLoadManager.getClusterIdleSlotNum();
        if (idleSlotNum == 0) {
            Thread.sleep(10000L);
            return;
        }
        try {
            RoutineLoadTaskInfo routineLoadTaskInfo = this.needScheduleTasksQueue.take();
            if (System.currentTimeMillis() - routineLoadTaskInfo.getLastScheduledTime() < routineLoadTaskInfo.getTimeoutMs()) {
                this.needScheduleTasksQueue.put(routineLoadTaskInfo);
                return;
            }
            this.scheduleOneTask(routineLoadTaskInfo);
        }
        catch (Exception e) {
            LOG.warn("Taking routine load task from queue has been interrupted", (Throwable)e);
            return;
        }
    }

    private void scheduleOneTask(RoutineLoadTaskInfo routineLoadTaskInfo) throws Exception {
        long startTime;
        routineLoadTaskInfo.setLastScheduledTime(System.currentTimeMillis());
        LOG.debug("schedule routine load task info {} for job {}", (Object)routineLoadTaskInfo.id, (Object)routineLoadTaskInfo.getJobId());
        if (!this.routineLoadManager.checkTaskInJob(routineLoadTaskInfo)) {
            LOG.warn(new LogBuilder(LogKey.ROUTINE_LOAD_TASK, routineLoadTaskInfo.getId()).add("error_msg", "task has been abandoned when scheduling task").build());
            return;
        }
        if (!routineLoadTaskInfo.hasMoreDataToConsume()) {
            this.needScheduleTasksQueue.put(routineLoadTaskInfo);
            return;
        }
        try {
            if (!this.allocateTaskToBe(routineLoadTaskInfo)) {
                this.needScheduleTasksQueue.put(routineLoadTaskInfo);
                return;
            }
        }
        catch (UserException e) {
            this.routineLoadManager.getJob(routineLoadTaskInfo.getJobId()).updateState(RoutineLoadJob.JobState.PAUSED, new ErrorReason(e.getErrorCode(), e.getMessage()), false);
            throw e;
        }
        catch (Exception e) {
            this.routineLoadManager.getJob(routineLoadTaskInfo.getJobId()).updateState(RoutineLoadJob.JobState.PAUSED, new ErrorReason(InternalErrorCode.CREATE_TASKS_ERR, "failed to allocate task: " + e.getMessage()), false);
            LOG.warn(new LogBuilder(LogKey.ROUTINE_LOAD_TASK, routineLoadTaskInfo.getId()).add("error_msg", "allocate task encounter exception: " + e.getMessage()).build(), (Throwable)e);
            throw e;
        }
        try {
            if (!routineLoadTaskInfo.beginTxn()) {
                routineLoadTaskInfo.setBeId(-1L);
                this.needScheduleTasksQueue.put(routineLoadTaskInfo);
                return;
            }
        }
        catch (Exception e) {
            routineLoadTaskInfo.setBeId(-1L);
            this.routineLoadManager.getJob(routineLoadTaskInfo.getJobId()).updateState(RoutineLoadJob.JobState.PAUSED, new ErrorReason(InternalErrorCode.CREATE_TASKS_ERR, "failed to begin txn: " + e.getMessage()), false);
            LOG.warn(new LogBuilder(LogKey.ROUTINE_LOAD_TASK, routineLoadTaskInfo.getId()).add("error_msg", "begin task txn encounter exception: " + e.getMessage()).build(), (Throwable)e);
            throw e;
        }
        TRoutineLoadTask tRoutineLoadTask = null;
        try {
            startTime = System.currentTimeMillis();
            tRoutineLoadTask = routineLoadTaskInfo.createRoutineLoadTask();
            LOG.debug("create routine load task cost(ms): {}, job id: {}", (Object)(System.currentTimeMillis() - startTime), (Object)routineLoadTaskInfo.getJobId());
        }
        catch (MetaNotFoundException e) {
            routineLoadTaskInfo.setBeId(-1L);
            this.routineLoadManager.getJob(routineLoadTaskInfo.getJobId()).updateState(RoutineLoadJob.JobState.CANCELLED, new ErrorReason(InternalErrorCode.META_NOT_FOUND_ERR, "meta not found: " + e.getMessage()), false);
            throw e;
        }
        catch (UserException e) {
            routineLoadTaskInfo.setBeId(-1L);
            this.routineLoadManager.getJob(routineLoadTaskInfo.getJobId()).updateState(RoutineLoadJob.JobState.PAUSED, new ErrorReason(e.getErrorCode(), "failed to create task: " + e.getMessage()), false);
            throw e;
        }
        try {
            startTime = System.currentTimeMillis();
            this.submitTask(routineLoadTaskInfo.getBeId(), tRoutineLoadTask);
            LOG.debug("send routine load task cost(ms): {}, job id: {}", (Object)(System.currentTimeMillis() - startTime), (Object)routineLoadTaskInfo.getJobId());
            if (tRoutineLoadTask.isSetKafkaLoadInfo()) {
                LOG.debug("send kafka routine load task {} with partition offset: {}, job: {}", (Object)tRoutineLoadTask.label, (Object)tRoutineLoadTask.kafka_load_info.partition_begin_offset, (Object)tRoutineLoadTask.getJobId());
            }
        }
        catch (LoadException e) {
            LOG.warn("failed to submit routine load task {} to BE: {}, error: {}", (Object)DebugUtil.printId(routineLoadTaskInfo.getId()), (Object)routineLoadTaskInfo.getBeId(), (Object)e.getMessage());
            this.routineLoadManager.getJob(routineLoadTaskInfo.getJobId()).setOtherMsg(e.getMessage());
        }
        routineLoadTaskInfo.setExecuteStartTimeMs(System.currentTimeMillis());
    }

    private void updateBackendSlotIfNecessary() {
        long currentTime = System.currentTimeMillis();
        if (this.lastBackendSlotUpdateTime == -1L || currentTime - this.lastBackendSlotUpdateTime > 10000L) {
            this.routineLoadManager.updateBeIdToMaxConcurrentTasks();
            this.lastBackendSlotUpdateTime = currentTime;
            LOG.debug("update backend max slot for routine load task scheduling. current task num per BE: {}", (Object)Config.max_routine_load_task_num_per_be);
        }
    }

    public void addTaskInQueue(RoutineLoadTaskInfo routineLoadTaskInfo) {
        this.needScheduleTasksQueue.add(routineLoadTaskInfo);
        LOG.debug("total tasks num in routine load task queue: {}", (Object)this.needScheduleTasksQueue.size());
    }

    public void addTasksInQueue(List<RoutineLoadTaskInfo> routineLoadTaskInfoList) {
        this.needScheduleTasksQueue.addAll(routineLoadTaskInfoList);
        LOG.debug("total tasks num in routine load task queue: {}", (Object)this.needScheduleTasksQueue.size());
    }

    private void submitTask(long beId, TRoutineLoadTask tTask) throws LoadException {
        Backend backend = Catalog.getCurrentSystemInfo().getBackend(beId);
        if (backend == null) {
            throw new LoadException("failed to send tasks to backend " + beId + " because not exist");
        }
        TNetworkAddress address = new TNetworkAddress(backend.getHost(), backend.getBePort());
        boolean ok = false;
        BackendService.Client client = null;
        try {
            client = ClientPool.backendPool.borrowObject(address);
            TStatus tStatus = client.submitRoutineLoadTask((List)Lists.newArrayList((Object[])new TRoutineLoadTask[]{tTask}));
            ok = true;
            if (tStatus.getStatusCode() != TStatusCode.OK) {
                throw new LoadException("failed to submit task. error code: " + tStatus.getStatusCode() + ", msg: " + (tStatus.getErrorMsgsSize() > 0 ? (String)tStatus.getErrorMsgs().get(0) : "NaN"));
            }
            LOG.debug("send routine load task {} to BE: {}", (Object)DebugUtil.printId(tTask.id), (Object)beId);
        }
        catch (Exception e) {
            throw new LoadException("failed to send task: " + e.getMessage(), e);
        }
        finally {
            if (ok) {
                ClientPool.backendPool.returnObject(address, client);
            } else {
                ClientPool.backendPool.invalidateObject(address, client);
            }
        }
    }

    private boolean allocateTaskToBe(RoutineLoadTaskInfo routineLoadTaskInfo) throws LoadException {
        long beId = this.routineLoadManager.getAvailableBeForTask(routineLoadTaskInfo.getJobId(), routineLoadTaskInfo.getPreviousBeId(), routineLoadTaskInfo.getClusterName());
        if (beId == -1L) {
            return false;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug(new LogBuilder(LogKey.ROUTINE_LOAD_TASK, routineLoadTaskInfo.getId()).add("job_id", routineLoadTaskInfo.getJobId()).add("previous_be_id", routineLoadTaskInfo.getPreviousBeId()).add("assigned_be_id", beId).build());
        }
        routineLoadTaskInfo.setBeId(beId);
        return true;
    }
}

