/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.load.loadv2;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.doris.PaloFe;
import org.apache.doris.analysis.BrokerDesc;
import org.apache.doris.common.Config;
import org.apache.doris.common.LoadException;
import org.apache.doris.common.UserException;
import org.apache.doris.common.util.BrokerUtil;
import org.apache.doris.thrift.TBrokerFileStatus;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class SparkRepository {
    private static final Logger LOG = LogManager.getLogger(SparkRepository.class);
    public static final String REPOSITORY_DIR = "__spark_repository__";
    public static final String PREFIX_ARCHIVE = "__archive_";
    public static final String PREFIX_LIB = "__lib_";
    public static final String SPARK_DPP_JAR = "spark-dpp-" + Config.spark_dpp_version + "-jar-with-dependencies.jar";
    public static final String SPARK_DPP = "spark-dpp-" + Config.spark_dpp_version + "-jar-with-dependencies";
    public static final String SPARK_2X = "spark-2x";
    private static final String PATH_DELIMITER = "/";
    private static final String FILE_NAME_SEPARATOR = "_";
    private static final String DPP_RESOURCE_DIR = "/spark-dpp/";
    private static final String SPARK_RESOURCE = "/jars/spark-2x.zip";
    private String remoteRepositoryPath;
    private BrokerDesc brokerDesc;
    private String localDppPath;
    private String localSpark2xPath;
    private String currentDppVersion;
    private SparkArchive currentArchive;

    public SparkRepository(String remoteRepositoryPath, BrokerDesc brokerDesc) {
        this.remoteRepositoryPath = remoteRepositoryPath;
        this.brokerDesc = brokerDesc;
        this.currentDppVersion = Config.spark_dpp_version;
        this.currentArchive = new SparkArchive(this.getRemoteArchivePath(this.currentDppVersion), this.currentDppVersion);
        this.localDppPath = PaloFe.DORIS_HOME_DIR + DPP_RESOURCE_DIR + SPARK_DPP_JAR;
        this.localSpark2xPath = !Strings.isNullOrEmpty((String)Config.spark_resource_path) ? Config.spark_resource_path : Config.spark_home_default_dir + SPARK_RESOURCE;
    }

    public void prepare() throws LoadException {
        this.initRepository();
    }

    private void initRepository() throws LoadException {
        LOG.info("start to init remote repository. local dpp: {}", (Object)this.localDppPath);
        boolean needUpload = false;
        boolean needReplace = false;
        if (!Strings.isNullOrEmpty((String)this.remoteRepositoryPath) && this.brokerDesc != null) {
            if (!this.checkCurrentArchiveExists()) {
                needUpload = true;
            } else {
                String remoteArchivePath = this.getRemoteArchivePath(this.currentDppVersion);
                ArrayList libraries = Lists.newArrayList();
                this.getLibraries(remoteArchivePath, libraries);
                if (libraries.size() != 2) {
                    needUpload = true;
                    needReplace = true;
                } else {
                    this.currentArchive.libraries.addAll(libraries);
                    for (SparkLibrary library : this.currentArchive.libraries) {
                        String localMd5sum = null;
                        switch (library.libType) {
                            case DPP: {
                                localMd5sum = this.getMd5String(this.localDppPath);
                                break;
                            }
                            case SPARK2X: {
                                localMd5sum = this.getMd5String(this.localSpark2xPath);
                                break;
                            }
                            default: {
                                Preconditions.checkState((boolean)false, (Object)("wrong library type: " + (Object)((Object)library.libType)));
                            }
                        }
                        if (localMd5sum.equals(library.md5sum)) continue;
                        needUpload = true;
                        needReplace = true;
                        break;
                    }
                }
            }
        }
        if (needUpload) {
            this.uploadArchive(needReplace);
        }
        LOG.info("init spark repository success, current dppVersion={}, archive path={}, libraries size={}", (Object)this.currentDppVersion, (Object)this.currentArchive.remotePath, (Object)this.currentArchive.libraries.size());
    }

    private boolean checkCurrentArchiveExists() {
        boolean result = false;
        Preconditions.checkNotNull((Object)this.remoteRepositoryPath);
        String remotePath = this.getRemoteArchivePath(this.currentDppVersion);
        try {
            result = BrokerUtil.checkPathExist(remotePath, this.brokerDesc);
            LOG.info("check archive exists in repository, {}", (Object)result);
        }
        catch (UserException e) {
            LOG.warn("Failed to check remote archive exist, path={}, version={}", (Object)remotePath, (Object)this.currentDppVersion);
        }
        return result;
    }

    private void uploadArchive(boolean isReplace) throws LoadException {
        try {
            String remoteArchivePath = this.getRemoteArchivePath(this.currentDppVersion);
            if (isReplace) {
                BrokerUtil.deletePath(remoteArchivePath, this.brokerDesc);
                this.currentArchive.libraries.clear();
            }
            String srcFilePath = null;
            srcFilePath = this.localDppPath;
            String fileName = SparkRepository.getFileName(PATH_DELIMITER, srcFilePath);
            String origFilePath = remoteArchivePath + PATH_DELIMITER + SparkRepository.assemblyFileName(PREFIX_LIB, "", fileName, "");
            this.upload(srcFilePath, origFilePath);
            String md5sum = this.getMd5String(srcFilePath);
            long size = this.getFileSize(srcFilePath);
            String destFilePath = remoteArchivePath + PATH_DELIMITER + SparkRepository.assemblyFileName(PREFIX_LIB, md5sum, fileName, "");
            this.rename(origFilePath, destFilePath);
            this.currentArchive.libraries.add(new SparkLibrary(destFilePath, md5sum, SparkLibrary.LibType.DPP, size));
            srcFilePath = this.localSpark2xPath;
            String origFilePath2 = remoteArchivePath + PATH_DELIMITER + SparkRepository.assemblyFileName(PREFIX_LIB, "", SPARK_2X, ".zip");
            this.upload(srcFilePath, origFilePath2);
            String md5sum2 = this.getMd5String(srcFilePath);
            long size2 = this.getFileSize(srcFilePath);
            String destFilePath2 = remoteArchivePath + PATH_DELIMITER + SparkRepository.assemblyFileName(PREFIX_LIB, md5sum2, SPARK_2X, ".zip");
            this.rename(origFilePath2, destFilePath2);
            this.currentArchive.libraries.add(new SparkLibrary(destFilePath2, md5sum2, SparkLibrary.LibType.SPARK2X, size2));
            LOG.info("finished to upload archive to repository, currentDppVersion={}, path={}", (Object)this.currentDppVersion, (Object)remoteArchivePath);
        }
        catch (UserException e) {
            throw new LoadException(e.getMessage());
        }
    }

    private void getLibraries(String remoteArchivePath, List<SparkLibrary> libraries) throws LoadException {
        ArrayList fileStatuses = Lists.newArrayList();
        try {
            BrokerUtil.parseFile(remoteArchivePath + "/*", this.brokerDesc, fileStatuses);
        }
        catch (UserException e) {
            throw new LoadException(e.getMessage());
        }
        for (TBrokerFileStatus fileStatus : fileStatuses) {
            String md5sum;
            String[] lib_arg;
            String fileName = SparkRepository.getFileName(PATH_DELIMITER, fileStatus.path);
            if (!fileName.startsWith(PREFIX_LIB) || (lib_arg = SparkRepository.unwrap(PREFIX_LIB, fileName).split(FILE_NAME_SEPARATOR)).length != 2 || Strings.isNullOrEmpty((String)(md5sum = lib_arg[0]))) continue;
            String type = lib_arg[1];
            SparkLibrary.LibType libType = null;
            if (type.equals(SPARK_DPP)) {
                libType = SparkLibrary.LibType.DPP;
            } else if (type.equals(SPARK_2X)) {
                libType = SparkLibrary.LibType.SPARK2X;
            } else {
                throw new LoadException("Invalid library type: " + type);
            }
            SparkLibrary remoteFile = new SparkLibrary(fileStatus.path, md5sum, libType, fileStatus.size);
            libraries.add(remoteFile);
            LOG.info("get Libraries from remote archive, archive path={}, library={}, md5sum={}, size={}", (Object)remoteArchivePath, (Object)remoteFile.remotePath, (Object)remoteFile.md5sum, (Object)remoteFile.size);
        }
    }

    public String getMd5String(String filePath) throws LoadException {
        File file = new File(filePath);
        String md5sum = null;
        try {
            md5sum = DigestUtils.md5Hex((InputStream)new FileInputStream(file));
            Preconditions.checkNotNull((Object)md5sum);
            LOG.debug("get md5sum from file {}, md5sum={}", (Object)filePath, (Object)md5sum);
            return md5sum;
        }
        catch (FileNotFoundException e) {
            throw new LoadException("file " + filePath + " does not exist");
        }
        catch (IOException e) {
            throw new LoadException("failed to get md5sum from file " + filePath);
        }
    }

    public long getFileSize(String filePath) throws LoadException {
        File file = new File(filePath);
        long size = file.length();
        if (size <= 0L) {
            throw new LoadException("failed to get size from file " + filePath);
        }
        return size;
    }

    private void upload(String srcFilePath, String destFilePath) throws LoadException {
        try {
            BrokerUtil.writeFile(srcFilePath, destFilePath, this.brokerDesc);
            LOG.info("finished to upload file, localPath={}, remotePath={}", (Object)srcFilePath, (Object)destFilePath);
        }
        catch (UserException e) {
            throw new LoadException("failed to upload lib to repository, srcPath=" + srcFilePath + " destPath=" + destFilePath + " message=" + e.getMessage());
        }
    }

    private void rename(String origFilePath, String destFilePath) throws LoadException {
        try {
            BrokerUtil.rename(origFilePath, destFilePath, this.brokerDesc);
            LOG.info("finished to rename file, originPath={}, destPath={}", (Object)origFilePath, (Object)destFilePath);
        }
        catch (UserException e) {
            throw new LoadException("failed to rename file from " + origFilePath + " to " + destFilePath + ", message=" + e.getMessage());
        }
    }

    public SparkArchive getCurrentArchive() {
        return this.currentArchive;
    }

    private static String getFileName(String delimiter, String path) {
        return path.substring(path.lastIndexOf(delimiter) + 1);
    }

    private static String unwrap(String prefix, String fileName) {
        int pos = fileName.lastIndexOf(".");
        return fileName.substring(prefix.length(), pos > 0 ? pos : fileName.length());
    }

    private static String joinPrefix(String prefix, String fileName) {
        return prefix + fileName;
    }

    public static String assemblyFileName(String prefix, String md5sum, String fileName, String suffix) {
        return prefix + md5sum + FILE_NAME_SEPARATOR + fileName + suffix;
    }

    private String getRemoteArchivePath(String version) {
        return Joiner.on((String)PATH_DELIMITER).join((Object)this.remoteRepositoryPath, (Object)SparkRepository.joinPrefix(PREFIX_ARCHIVE, version), new Object[0]);
    }

    public static class SparkLibrary {
        public String remotePath;
        public String md5sum;
        public long size;
        public LibType libType;

        public SparkLibrary(String remotePath, String md5sum, LibType libType, long size) {
            this.remotePath = remotePath;
            this.md5sum = md5sum;
            this.libType = libType;
            this.size = size;
        }

        public static enum LibType {
            DPP,
            SPARK2X;

        }
    }

    public static class SparkArchive {
        public String remotePath;
        public String version;
        public List<SparkLibrary> libraries;

        public SparkArchive(String remotePath, String version) {
            this.remotePath = remotePath;
            this.version = version;
            this.libraries = Lists.newArrayList();
        }

        public SparkLibrary getDppLibrary() {
            SparkLibrary result = null;
            Optional<SparkLibrary> library = this.libraries.stream().filter(lib -> lib.libType == SparkLibrary.LibType.DPP).findFirst();
            if (library.isPresent()) {
                result = library.get();
            }
            return result;
        }

        public SparkLibrary getSpark2xLibrary() {
            SparkLibrary result = null;
            Optional<SparkLibrary> library = this.libraries.stream().filter(lib -> lib.libType == SparkLibrary.LibType.SPARK2X).findFirst();
            if (library.isPresent()) {
                result = library.get();
            }
            return result;
        }
    }
}

