/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.load;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.doris.catalog.Catalog;
import org.apache.doris.catalog.Database;
import org.apache.doris.catalog.MaterializedIndex;
import org.apache.doris.catalog.OlapTable;
import org.apache.doris.catalog.Partition;
import org.apache.doris.catalog.Replica;
import org.apache.doris.catalog.Table;
import org.apache.doris.catalog.Tablet;
import org.apache.doris.catalog.TabletInvertedIndex;
import org.apache.doris.common.Config;
import org.apache.doris.common.UserException;
import org.apache.doris.common.util.MasterDaemon;
import org.apache.doris.common.util.MetaLockUtils;
import org.apache.doris.load.EtlJobType;
import org.apache.doris.load.FailMsg;
import org.apache.doris.load.Load;
import org.apache.doris.load.LoadJob;
import org.apache.doris.load.PartitionLoadInfo;
import org.apache.doris.load.TableLoadInfo;
import org.apache.doris.load.TabletLoadInfo;
import org.apache.doris.task.AgentBatchTask;
import org.apache.doris.task.AgentTaskExecutor;
import org.apache.doris.task.AgentTaskQueue;
import org.apache.doris.task.HadoopLoadEtlTask;
import org.apache.doris.task.HadoopLoadPendingTask;
import org.apache.doris.task.MasterTaskExecutor;
import org.apache.doris.task.PushTask;
import org.apache.doris.thrift.TPriority;
import org.apache.doris.thrift.TPushType;
import org.apache.doris.thrift.TTaskType;
import org.apache.doris.transaction.GlobalTransactionMgr;
import org.apache.doris.transaction.TabletCommitInfo;
import org.apache.doris.transaction.TabletQuorumFailedException;
import org.apache.doris.transaction.TransactionState;
import org.apache.doris.transaction.TransactionStatus;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class LoadChecker
extends MasterDaemon {
    private static final Logger LOG = LogManager.getLogger(LoadChecker.class);
    private static Map<LoadJob.JobState, LoadChecker> checkers = Maps.newHashMap();
    private static Map<LoadJob.JobState, Map<TPriority, MasterTaskExecutor>> executors = Maps.newHashMap();
    private LoadJob.JobState jobState;

    private LoadChecker(LoadJob.JobState jobState, long intervalMs) {
        super("load checker " + jobState.name().toLowerCase(), intervalMs);
        this.jobState = jobState;
    }

    public static void init(long intervalMs) {
        checkers.put(LoadJob.JobState.PENDING, new LoadChecker(LoadJob.JobState.PENDING, intervalMs));
        checkers.put(LoadJob.JobState.ETL, new LoadChecker(LoadJob.JobState.ETL, intervalMs));
        checkers.put(LoadJob.JobState.LOADING, new LoadChecker(LoadJob.JobState.LOADING, intervalMs));
        checkers.put(LoadJob.JobState.QUORUM_FINISHED, new LoadChecker(LoadJob.JobState.QUORUM_FINISHED, intervalMs));
        HashMap pendingPriorityMap = Maps.newHashMap();
        pendingPriorityMap.put(TPriority.NORMAL, new MasterTaskExecutor("load_pending_thread_num_normal_priority", Config.load_pending_thread_num_normal_priority, true));
        pendingPriorityMap.put(TPriority.HIGH, new MasterTaskExecutor("load_pending_thread_num_high_priority", Config.load_pending_thread_num_high_priority, true));
        executors.put(LoadJob.JobState.PENDING, pendingPriorityMap);
        HashMap etlPriorityMap = Maps.newHashMap();
        etlPriorityMap.put(TPriority.NORMAL, new MasterTaskExecutor("load_etl_thread_num_normal_priority", Config.load_etl_thread_num_normal_priority, true));
        etlPriorityMap.put(TPriority.HIGH, new MasterTaskExecutor("load_etl_thread_num_high_priority", Config.load_etl_thread_num_high_priority, true));
        executors.put(LoadJob.JobState.ETL, etlPriorityMap);
    }

    public static void startAll() {
        for (LoadChecker loadChecker : checkers.values()) {
            loadChecker.start();
        }
        for (Map map : executors.values()) {
            for (MasterTaskExecutor masterTaskExecutor : map.values()) {
                masterTaskExecutor.start();
            }
        }
    }

    @Override
    protected void runAfterCatalogReady() {
        LOG.debug("start check load jobs. job state: {}", (Object)this.jobState.name());
        switch (this.jobState) {
            case PENDING: {
                this.runPendingJobs();
                break;
            }
            case ETL: {
                this.runEtlJobs();
                break;
            }
            case LOADING: {
                this.runLoadingJobs();
                break;
            }
            case QUORUM_FINISHED: {
                this.runQuorumFinishedJobs();
                break;
            }
            default: {
                LOG.warn("wrong job state: {}", (Object)this.jobState.name());
            }
        }
    }

    private void runPendingJobs() {
        Load load = Catalog.getCurrentCatalog().getLoadInstance();
        List<LoadJob> pendingJobs = load.getLoadJobs(LoadJob.JobState.PENDING);
        int runningJobNumLimit = Config.load_running_job_num_limit;
        if (runningJobNumLimit > 0 && !pendingJobs.isEmpty()) {
            int runningJobNum = executors.get((Object)LoadJob.JobState.PENDING).get(TPriority.NORMAL).getTaskNum() + executors.get((Object)LoadJob.JobState.PENDING).get(TPriority.HIGH).getTaskNum() + load.getLoadJobs(LoadJob.JobState.ETL).size();
            if (runningJobNum >= runningJobNumLimit) {
                LOG.debug("running load job num {} exceeds system limit {}", (Object)runningJobNum, (Object)runningJobNumLimit);
                return;
            }
            int remain = runningJobNumLimit - runningJobNum;
            if (pendingJobs.size() > remain) {
                pendingJobs = pendingJobs.subList(0, remain);
            }
        }
        for (LoadJob job : pendingJobs) {
            try {
                HadoopLoadPendingTask task = null;
                EtlJobType etlJobType = job.getEtlJobType();
                switch (etlJobType) {
                    case HADOOP: {
                        task = new HadoopLoadPendingTask(job);
                        break;
                    }
                    default: {
                        LOG.warn("unknown etl job type. type: {}, job id: {}, label: {}, db: {}", (Object)etlJobType.name(), (Object)job.getId(), (Object)job.getLabel(), (Object)job.getDbId());
                    }
                }
                if (task == null || !executors.get((Object)LoadJob.JobState.PENDING).get(job.getPriority()).submit(task)) continue;
                LOG.info("run pending job. job: {}", (Object)job);
            }
            catch (Exception e) {
                LOG.warn("run pending job error", (Throwable)e);
            }
        }
    }

    private void runEtlJobs() {
        List<LoadJob> etlJobs = Catalog.getCurrentCatalog().getLoadInstance().getLoadJobs(LoadJob.JobState.ETL);
        for (LoadJob job : etlJobs) {
            try {
                HadoopLoadEtlTask task = null;
                EtlJobType etlJobType = job.getEtlJobType();
                switch (etlJobType) {
                    case HADOOP: {
                        task = new HadoopLoadEtlTask(job);
                        break;
                    }
                    default: {
                        LOG.warn("unknown etl job type. type: {}", (Object)etlJobType.name());
                    }
                }
                if (task == null || !executors.get((Object)LoadJob.JobState.ETL).get(job.getPriority()).submit(task)) continue;
                LOG.info("run etl job. job: {}", (Object)job);
            }
            catch (Exception e) {
                LOG.warn("run etl job error", (Throwable)e);
            }
        }
    }

    private void runLoadingJobs() {
        List<LoadJob> loadingJobs = Catalog.getCurrentCatalog().getLoadInstance().getLoadJobs(LoadJob.JobState.LOADING);
        for (LoadJob job : loadingJobs) {
            try {
                LOG.info("run loading job. job: {}", (Object)job);
                this.runOneLoadingJob(job);
            }
            catch (Exception e) {
                LOG.warn("run loading job error", (Throwable)e);
            }
        }
    }

    private void runOneLoadingJob(LoadJob job) {
        Load load = Catalog.getCurrentCatalog().getLoadInstance();
        long dbId = job.getDbId();
        Database db = Catalog.getCurrentCatalog().getDbNullable(dbId);
        if (db == null) {
            load.cancelLoadJob(job, FailMsg.CancelType.LOAD_RUN_FAIL, "db does not exist. id: " + dbId);
            return;
        }
        ArrayList tableIds = Lists.newArrayList();
        long tableId = job.getTableId();
        if (tableId > 0L) {
            tableIds.add(tableId);
        } else {
            tableIds.addAll(job.getIdToTableLoadInfo().keySet());
        }
        List<Table> tables = null;
        try {
            tables = db.getTablesOnIdOrderOrThrowException(tableIds);
        }
        catch (UserException e) {
            load.cancelLoadJob(job, FailMsg.CancelType.LOAD_RUN_FAIL, "table does not exist. dbId: " + dbId + ", err: " + e.getMessage());
            return;
        }
        if (job.getTransactionId() < 0L) {
            LOG.warn("cancel load job {}  because it is an old type job, user should resubmit it", (Object)job);
            load.cancelLoadJob(job, FailMsg.CancelType.UNKNOWN, "cancelled because system is during upgrade, user should resubmit it");
            return;
        }
        TransactionState state = Catalog.getCurrentGlobalTransactionMgr().getTransactionState(job.getDbId(), job.getTransactionId());
        if (state == null) {
            LOG.warn("cancel load job {}  because could not find transaction state", (Object)job);
            load.cancelLoadJob(job, FailMsg.CancelType.UNKNOWN, "transaction state lost");
            return;
        }
        if (state.getTransactionStatus() == TransactionStatus.ABORTED) {
            load.cancelLoadJob(job, FailMsg.CancelType.LOAD_RUN_FAIL, "job is aborted in transaction manager [" + state + "]");
            return;
        }
        if (state.getTransactionStatus() == TransactionStatus.COMMITTED) {
            job.setProgress(100);
            LOG.debug("job {} is already committed, just wait it to be visible, transaction state {}", (Object)job, (Object)state);
            return;
        }
        if (state.getTransactionStatus() == TransactionStatus.VISIBLE) {
            if (load.updateLoadJobState(job, LoadJob.JobState.FINISHED)) {
                load.clearJob(job, LoadJob.JobState.QUORUM_FINISHED);
            }
            return;
        }
        if (LoadChecker.checkTimeout(job)) {
            load.cancelLoadJob(job, FailMsg.CancelType.TIMEOUT, "loading timeout to cancel");
            return;
        }
        Set<Long> jobTotalTablets = this.submitPushTasks(job, db);
        if (jobTotalTablets == null) {
            load.cancelLoadJob(job, FailMsg.CancelType.LOAD_RUN_FAIL, "submit push tasks fail");
            return;
        }
        Set<Long> fullTablets = job.getFullTablets();
        if (state.isRunning()) {
            job.setProgress(fullTablets.size() * 100 / jobTotalTablets.size());
        } else {
            job.setProgress(100);
        }
        long stragglerTimeout = job.isSyncDeleteJob() ? job.getDeleteJobTimeout() / 2L : (long)(Config.load_straggler_wait_second * 1000);
        HashSet unfinishedTablets = Sets.newHashSet();
        unfinishedTablets.addAll(jobTotalTablets);
        unfinishedTablets.removeAll(job.getQuorumTablets());
        job.setUnfinishedTablets(unfinishedTablets);
        if (job.getQuorumTablets().containsAll(jobTotalTablets)) {
            if (job.getQuorumFinishTimeMs() < 0L) {
                job.setQuorumFinishTimeMs(System.currentTimeMillis());
            }
            if (System.currentTimeMillis() - job.getQuorumFinishTimeMs() > stragglerTimeout || job.getFullTablets().containsAll(jobTotalTablets)) {
                this.tryCommitJob(job, tables);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void tryCommitJob(LoadJob job, List<Table> tables) {
        Load load = Catalog.getCurrentCatalog().getLoadInstance();
        GlobalTransactionMgr globalTransactionMgr = Catalog.getCurrentGlobalTransactionMgr();
        TransactionState transactionState = globalTransactionMgr.getTransactionState(job.getDbId(), job.getTransactionId());
        ArrayList<TabletCommitInfo> tabletCommitInfos = new ArrayList<TabletCommitInfo>();
        try {
            MetaLockUtils.writeLockTablesOrMetaException(tables);
        }
        catch (UserException e) {
            load.cancelLoadJob(job, FailMsg.CancelType.LOAD_RUN_FAIL, "table does not exist. dbId: " + job.getDbId() + ", err: " + e.getMessage());
            return;
        }
        try {
            TabletInvertedIndex invertedIndex = Catalog.getCurrentInvertedIndex();
            for (Replica replica : job.getFinishedReplicas()) {
                Long tabletId = invertedIndex.getTabletIdByReplica(replica.getId());
                if (tabletId == null) {
                    LOG.warn("could not find tablet id for replica {}, the tablet maybe dropped", (Object)replica);
                    continue;
                }
                tabletCommitInfos.add(new TabletCommitInfo(tabletId, replica.getBackendId()));
            }
            globalTransactionMgr.commitTransaction(job.getDbId(), tables, job.getTransactionId(), tabletCommitInfos);
        }
        catch (TabletQuorumFailedException invertedIndex) {
        }
        catch (UserException e) {
            LOG.warn("errors while commit transaction [{}], cancel the job {}, reason is {}", (Object)transactionState.getTransactionId(), (Object)job, (Object)e);
            load.cancelLoadJob(job, FailMsg.CancelType.UNKNOWN, transactionState.getReason());
        }
        finally {
            MetaLockUtils.writeUnlockTables(tables);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Set<Long> submitPushTasks(LoadJob job, Database db) {
        Map<Long, TabletLoadInfo> tabletLoadInfos = job.getIdToTabletLoadInfo();
        boolean needDecompress = job.getEtlJobType() == EtlJobType.HADOOP;
        AgentBatchTask batchTask = new AgentBatchTask();
        HashSet<Long> jobTotalTablets = new HashSet<Long>();
        Map<Long, TableLoadInfo> idToTableLoadInfo = job.getIdToTableLoadInfo();
        for (Map.Entry<Long, TableLoadInfo> tableEntry : idToTableLoadInfo.entrySet()) {
            long tableId = tableEntry.getKey();
            OlapTable table = (OlapTable)db.getTableNullable(tableId);
            if (table == null) {
                LOG.warn("table does not exist. id: {}", (Object)tableId);
                return null;
            }
            TableLoadInfo tableLoadInfo = tableEntry.getValue();
            boolean autoLoadToTwoTablet = true;
            for (Map.Entry<Long, PartitionLoadInfo> partitionEntry : tableLoadInfo.getIdToPartitionLoadInfo().entrySet()) {
                long partitionId = partitionEntry.getKey();
                PartitionLoadInfo partitionLoadInfo = partitionEntry.getValue();
                if (!partitionLoadInfo.isNeedLoad()) continue;
                table.readLock();
                try {
                    Partition partition = table.getPartition(partitionId);
                    if (partition == null) {
                        LOG.warn("partition does not exist. id: {}", (Object)partitionId);
                        Set<Long> set = null;
                        return set;
                    }
                    short replicationNum = table.getPartitionInfo().getReplicaAllocation(partition.getId()).getTotalReplicaNum();
                    List<MaterializedIndex> indices = partition.getMaterializedIndices(MaterializedIndex.IndexExtState.ALL);
                    for (MaterializedIndex index : indices) {
                        long indexId = index.getId();
                        if (!tableLoadInfo.containsIndex(indexId)) continue;
                        for (Tablet tablet : index.getTablets()) {
                            if (!tabletLoadInfos.containsKey(tablet.getId())) continue;
                            jobTotalTablets.add(tablet.getId());
                        }
                        int schemaHash = tableLoadInfo.getIndexSchemaHash(indexId);
                        short quorumNum = (short)(replicationNum / 2 + 1);
                        for (Tablet tablet : index.getTablets()) {
                            long tabletId = tablet.getId();
                            TabletLoadInfo tabletLoadInfo = tabletLoadInfos.get(tabletId);
                            if (tabletLoadInfo == null) continue;
                            String filePath = tabletLoadInfo.getFilePath();
                            long fileSize = tabletLoadInfo.getFileSize();
                            TPushType type = TPushType.LOAD;
                            if (job.isSyncDeleteJob()) {
                                type = TPushType.DELETE;
                            }
                            HashSet<Long> allReplicas = new HashSet<Long>();
                            HashSet<Long> finishedReplicas = new HashSet<Long>();
                            for (Replica replica : tablet.getReplicas()) {
                                long replicaId = replica.getId();
                                allReplicas.add(replicaId);
                                if (!tabletLoadInfo.isReplicaSent(replicaId)) {
                                    PushTask pushTask = new PushTask(job.getResourceInfo(), replica.getBackendId(), db.getId(), tableId, partitionId, indexId, tabletId, replicaId, schemaHash, -1L, filePath, fileSize, 0, job.getId(), type, job.getConditions(), needDecompress, job.getPriority(), TTaskType.REALTIME_PUSH, job.getTransactionId(), Catalog.getCurrentGlobalTransactionMgr().getTransactionIDGenerator().getNextTransactionId());
                                    pushTask.setIsSchemaChanging(autoLoadToTwoTablet);
                                    if (AgentTaskQueue.addTask(pushTask)) {
                                        batchTask.addTask(pushTask);
                                        job.addPushTask(pushTask);
                                        tabletLoadInfo.addSentReplica(replicaId);
                                    }
                                }
                                if (!job.isReplicaFinished(replicaId) || replica.getLastFailedVersion() >= 0L) continue;
                                finishedReplicas.add(replicaId);
                            }
                            if (allReplicas.size() == 0) {
                                LOG.error("invalid situation. tablet is empty. id: {}", (Object)tabletId);
                            }
                            if (finishedReplicas.size() < quorumNum || !tabletLoadInfos.containsKey(tabletId)) continue;
                            job.addQuorumTablet(tabletId);
                            if (finishedReplicas.size() != allReplicas.size()) continue;
                            job.addFullTablet(tabletId);
                        }
                    }
                }
                finally {
                    table.readUnlock();
                }
            }
        }
        if (batchTask.getTaskNum() > 0) {
            AgentTaskExecutor.submit(batchTask);
        }
        return jobTotalTablets;
    }

    private void runQuorumFinishedJobs() {
        List<LoadJob> quorumFinishedJobs = Catalog.getCurrentCatalog().getLoadInstance().getLoadJobs(LoadJob.JobState.QUORUM_FINISHED);
        for (LoadJob job : quorumFinishedJobs) {
            try {
                LOG.info("run quorum finished job. job: {}", (Object)job);
                this.runOneQuorumFinishedJob(job);
            }
            catch (Exception e) {
                LOG.warn("run quorum job error", (Throwable)e);
            }
        }
    }

    private void runOneQuorumFinishedJob(LoadJob job) {
        Load load = Catalog.getCurrentCatalog().getLoadInstance();
        long dbId = job.getDbId();
        Database db = Catalog.getCurrentCatalog().getDbNullable(dbId);
        if (db == null) {
            load.cancelLoadJob(job, FailMsg.CancelType.LOAD_RUN_FAIL, "db does not exist. id: " + dbId);
            return;
        }
        if (load.updateLoadJobState(job, LoadJob.JobState.FINISHED)) {
            load.clearJob(job, LoadJob.JobState.QUORUM_FINISHED);
        }
    }

    public static boolean checkTimeout(LoadJob job) {
        int timeoutSecond = job.getTimeoutSecond();
        if (timeoutSecond == 0) {
            return false;
        }
        long deltaSecond = (System.currentTimeMillis() - job.getCreateTimeMs()) / 1000L;
        return deltaSecond > (long)timeoutSecond;
    }
}

