/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.load;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.Gson;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import org.apache.commons.lang.StringUtils;
import org.apache.doris.PaloFe;
import org.apache.doris.common.Config;
import org.apache.doris.common.FeConstants;
import org.apache.doris.common.LoadException;
import org.apache.doris.common.util.CommandResult;
import org.apache.doris.common.util.Util;
import org.apache.doris.load.DppConfig;
import org.apache.doris.load.EtlStatus;
import org.apache.doris.load.EtlSubmitResult;
import org.apache.doris.thrift.TEtlState;
import org.apache.doris.thrift.TStatus;
import org.apache.doris.thrift.TStatusCode;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class DppScheduler {
    private static final Logger LOG = LogManager.getLogger(DppScheduler.class);
    private static final String HADOOP_CLIENT = PaloFe.DORIS_HOME_DIR + Config.dpp_hadoop_client_path;
    private static final String DPP_OUTPUT_DIR = "export";
    private static final String JOB_CONFIG_DIR = PaloFe.DORIS_HOME_DIR + "/temp/job_conf";
    private static final String JOB_CONFIG_FILE = "jobconfig.json";
    private static final String LOCAL_DPP_DIR = PaloFe.DORIS_HOME_DIR + "/lib/dpp/" + FeConstants.dpp_version;
    private static final int DEFAULT_REDUCE_NUM = 1000;
    private static final long GB = 0x40000000L;
    private static final String ETL_OUTPUT_PATH = "%s%s/%d/%s/%s";
    private static final String ETL_JOB_NAME = "palo2__%s__%s";
    private static final String HADOOP_BISTREAMING_CMD = "%s bistreaming %s -D mapred.job.name=\"%s\" -input %s -output %s -mapper \"sh mapred/mapper.sh\" -reducer \"sh mapred/reducer.sh '\\\"%s\\\"'\" -partitioner com.baidu.sos.mapred.lib.MapIntPartitioner -cacheArchive %s/dpp/x86_64-scm-linux-gnu.tar.gz#tc -cacheArchive %s/dpp/pypy.tar.gz#pypy -cacheArchive %s/dpp/palo_dpp_mr.tar.gz#mapred -numReduceTasks %d -file \"%s\" ";
    private static final String HADOOP_STATUS_CMD = "%s job %s -status %s";
    private static final String HADOOP_KILL_CMD = "%s job %s -kill %s";
    private static final String HADOOP_LS_CMD = "%s fs %s -ls %s";
    private static final String HADOOP_COUNT_CMD = "%s fs %s -count %s";
    private static final String HADOOP_TEST_CMD = "%s fs %s -test %s %s";
    private static final String HADOOP_MKDIR_CMD = "%s fs %s -mkdir %s";
    private static final String HADOOP_RMR_CMD = "%s fs %s -rmr %s";
    private static final String HADOOP_PUT_CMD = "%s fs %s -put %s %s";
    private static final long HADOOP_SPEED_LIMIT_KB = 10240L;
    private static final ConcurrentMap<String, Object> DPP_LOCK_MAP = Maps.newConcurrentMap();
    private String hadoopConfig;
    private String applicationsPath;

    public DppScheduler(DppConfig dppConfig) {
        this.hadoopConfig = this.getHadoopConfigsStr(dppConfig.getHadoopConfigs());
        this.applicationsPath = dppConfig.getFsDefaultName() + dppConfig.getApplicationsPath();
    }

    private String getHadoopConfigsStr(Map<String, String> hadoopConfigs) {
        ArrayList configs = Lists.newArrayList();
        for (Map.Entry<String, String> entry : hadoopConfigs.entrySet()) {
            configs.add(String.format("%s=%s", entry.getKey(), entry.getValue()));
        }
        return String.format("-D %s", StringUtils.join((Collection)configs, (String)" -D "));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public EtlSubmitResult submitEtlJob(long jobId, String loadLabel, String clusterName, String dbName, Map<String, Object> jobConf, int retry) {
        String configDirPath;
        File configDir;
        String etlJobId = null;
        TStatus status = new TStatus();
        status.setStatusCode(TStatusCode.OK);
        ArrayList failMsgs = Lists.newArrayList();
        status.setErrorMsgs((List)failMsgs);
        if (retry > 0) {
            LOG.warn("submit etl retry[{}] > 0. check dpp application", (Object)retry);
            DPP_LOCK_MAP.putIfAbsent(clusterName, new Object());
            Preconditions.checkState((boolean)DPP_LOCK_MAP.containsKey(clusterName));
            Object v = DPP_LOCK_MAP.get(clusterName);
            synchronized (v) {
                try {
                    this.prepareDppApplications();
                }
                catch (LoadException e) {
                    status.setStatusCode(TStatusCode.CANCELLED);
                    failMsgs.add(e.getMessage());
                    return new EtlSubmitResult(status, null);
                }
            }
        }
        if (!Util.deleteDirectory(configDir = new File(configDirPath = JOB_CONFIG_DIR + "/" + jobId))) {
            String errMsg = "delete config dir error. job: " + jobId;
            LOG.warn(errMsg + ", path: {}", (Object)configDirPath);
            status.setStatusCode(TStatusCode.CANCELLED);
            failMsgs.add(errMsg);
            return new EtlSubmitResult(status, null);
        }
        if (!configDir.mkdirs()) {
            String errMsg = "create config file dir error. job: " + jobId;
            LOG.warn(errMsg + ", path: {}", (Object)configDirPath);
            status.setStatusCode(TStatusCode.CANCELLED);
            failMsgs.add(errMsg);
            return new EtlSubmitResult(status, null);
        }
        File configFile = new File(configDirPath + "/" + JOB_CONFIG_FILE);
        BufferedWriter bw = null;
        try {
            bw = new BufferedWriter(new OutputStreamWriter((OutputStream)new FileOutputStream(configFile), "UTF-8"));
            Gson gson = new Gson();
            bw.write(gson.toJson(jobConf));
            bw.flush();
        }
        catch (IOException e) {
            Util.deleteDirectory(configDir);
            String errMsg = "create config file error. job: " + jobId;
            LOG.warn(errMsg + ", file: {}", (Object)(configDirPath + "/" + JOB_CONFIG_FILE));
            status.setStatusCode(TStatusCode.CANCELLED);
            failMsgs.add(errMsg);
            EtlSubmitResult etlSubmitResult = new EtlSubmitResult(status, null);
            return etlSubmitResult;
        }
        finally {
            if (bw != null) {
                try {
                    bw.close();
                }
                catch (IOException e) {
                    LOG.warn("close buffered writer error", (Throwable)e);
                    status.setStatusCode(TStatusCode.CANCELLED);
                    failMsgs.add(e.getMessage());
                    return new EtlSubmitResult(status, null);
                }
            }
        }
        Set<String> inputPaths = this.getInputPaths(jobConf);
        String inputPath = StringUtils.join(inputPaths, (String)" -input ");
        int reduceNumByInputSize = 0;
        try {
            reduceNumByInputSize = this.calcReduceNumByInputSize(inputPaths);
        }
        catch (InputSizeInvalidException e) {
            status.setStatusCode(TStatusCode.CANCELLED);
            failMsgs.add(e.getMessage());
            return new EtlSubmitResult(status, null);
        }
        int reduceNumByTablet = this.calcReduceNumByTablet(jobConf);
        int reduceNum = Math.min(reduceNumByInputSize, reduceNumByTablet);
        LOG.info("calculate reduce num. reduceNum: {}, reduceNumByInputSize: {}, reduceNumByTablet: {}", (Object)reduceNum, (Object)reduceNumByInputSize, (Object)reduceNumByTablet);
        String outputPath = (String)jobConf.get("output_path");
        this.deleteEtlOutputPath(outputPath);
        String etlJobName = String.format(ETL_JOB_NAME, dbName, loadLabel);
        String hadoopRunCmd = String.format(HADOOP_BISTREAMING_CMD, HADOOP_CLIENT, this.hadoopConfig, etlJobName, inputPath, outputPath, this.hadoopConfig, this.applicationsPath, this.applicationsPath, this.applicationsPath, reduceNum, configFile.getAbsolutePath());
        LOG.info(hadoopRunCmd);
        String outputLine = null;
        List<String> hadoopRunCmdList = Util.shellSplit(hadoopRunCmd);
        String[] hadoopRunCmds = hadoopRunCmdList.toArray(new String[0]);
        BufferedReader errorReader = null;
        long startTime = System.currentTimeMillis();
        try {
            Process p = Runtime.getRuntime().exec(hadoopRunCmds);
            errorReader = new BufferedReader(new InputStreamReader(p.getErrorStream()));
            for (int i = 0; i < 1000; ++i) {
                outputLine = errorReader.readLine();
                LOG.info(outputLine);
                if (Strings.isNullOrEmpty((String)outputLine)) {
                    LOG.warn("submit etl job fail. job id: {}, label: {}", (Object)jobId, (Object)loadLabel);
                } else {
                    if (outputLine.toLowerCase().contains("error") || outputLine.toLowerCase().contains("exception")) {
                        failMsgs.add(outputLine);
                    }
                    if (outputLine.indexOf("Running job") == -1) continue;
                    String[] arr = outputLine.split(":");
                    etlJobId = arr[arr.length - 1].trim();
                    p.destroy();
                }
                break;
            }
        }
        catch (IOException e) {
            LOG.warn("submit etl job error", (Throwable)e);
            status.setStatusCode(TStatusCode.CANCELLED);
            failMsgs.add(e.getMessage());
            EtlSubmitResult etlSubmitResult = new EtlSubmitResult(status, null);
            return etlSubmitResult;
        }
        finally {
            Util.deleteDirectory(configDir);
            long endTime = System.currentTimeMillis();
            LOG.info("finished submit hadoop job: {}. cost: {} ms", (Object)jobId, (Object)(endTime - startTime));
            if (errorReader != null) {
                try {
                    errorReader.close();
                }
                catch (IOException e) {
                    LOG.warn("close buffered reader error", (Throwable)e);
                    status.setStatusCode(TStatusCode.CANCELLED);
                    failMsgs.add(e.getMessage());
                    return new EtlSubmitResult(status, null);
                }
            }
        }
        if (etlJobId == null) {
            status.setStatusCode(TStatusCode.CANCELLED);
        }
        return new EtlSubmitResult(status, etlJobId);
    }

    private void prepareDppApplications() throws LoadException {
        String[] envp = new String[]{"LC_ALL=" + Config.locale};
        String hadoopDppDir = this.applicationsPath + "/dpp";
        boolean needUpload = false;
        File dppDir = new File(LOCAL_DPP_DIR);
        if (!dppDir.exists() || !dppDir.isDirectory()) {
            LOG.warn("dpp dir does not exist");
            throw new LoadException("dpp dir does not exist");
        }
        File[] localFiles = dppDir.listFiles();
        String hadoopTestCmd = String.format(HADOOP_TEST_CMD, HADOOP_CLIENT, this.hadoopConfig, "-d", hadoopDppDir);
        LOG.info(hadoopTestCmd);
        CommandResult testResult = Util.executeCommand(hadoopTestCmd, envp);
        if (testResult.getReturnCode() == 0) {
            String hadoopDppFilePath = hadoopDppDir + "/*";
            String hadoopCountCmd = String.format(HADOOP_COUNT_CMD, HADOOP_CLIENT, this.hadoopConfig, hadoopDppFilePath);
            LOG.info(hadoopCountCmd);
            CommandResult countResult = Util.executeCommand(hadoopCountCmd, envp);
            if (countResult.getReturnCode() != 0) {
                LOG.warn("hadoop count error, result: {}", (Object)countResult);
                throw new LoadException("hadoop count error. msg: " + countResult.getStderr());
            }
            HashMap fileMap = Maps.newHashMap();
            String[] fileInfos = countResult.getStdout().split("\n");
            for (String string : fileInfos) {
                String[] fileInfoArr = string.trim().split(" +");
                if (fileInfoArr.length != 4) continue;
                String filePath = fileInfoArr[3];
                String fileName = filePath.substring(filePath.lastIndexOf("/") + 1);
                long size = Long.parseLong(fileInfoArr[2]);
                fileMap.put(fileName, size);
            }
            for (File file : localFiles) {
                if (!file.isFile()) continue;
                String fileName = file.getName();
                if (!fileMap.containsKey(fileName)) {
                    LOG.info("hadoop dpp file does not exist. file: {}", (Object)fileName);
                    needUpload = true;
                } else {
                    long hadoopSize;
                    long localSize = file.length();
                    if (localSize == (hadoopSize = ((Long)fileMap.get(fileName)).longValue())) continue;
                    LOG.info("dpp files size are different. file: {}, local: {}, hadoop: {}", (Object)fileName, (Object)localSize, (Object)hadoopSize);
                    needUpload = true;
                }
                break;
            }
        } else {
            LOG.info("hadoop dir does not exist. dir: {}", (Object)hadoopDppDir);
            needUpload = true;
        }
        if (needUpload) {
            String hadoopRmrCmd = String.format(HADOOP_RMR_CMD, HADOOP_CLIENT, this.hadoopConfig, hadoopDppDir);
            LOG.info(hadoopRmrCmd);
            Util.executeCommand(hadoopRmrCmd, envp);
            String hadoopMkdirCmd = String.format(HADOOP_MKDIR_CMD, HADOOP_CLIENT, this.hadoopConfig, hadoopDppDir);
            LOG.info(hadoopMkdirCmd);
            Util.executeCommand(hadoopMkdirCmd, envp);
            String hadoopPutConfig = this.hadoopConfig + String.format(" -D speed.limit.kb=%d", 10240L);
            String hadoopPutCmd = null;
            CommandResult putResult = null;
            for (File file : localFiles) {
                hadoopPutCmd = String.format(HADOOP_PUT_CMD, HADOOP_CLIENT, hadoopPutConfig, LOCAL_DPP_DIR + "/" + file.getName(), hadoopDppDir);
                LOG.info(hadoopPutCmd);
                putResult = Util.executeCommand(hadoopPutCmd, envp);
                if (putResult.getReturnCode() == 0) continue;
                LOG.warn("hadoop put fail. result: {}", (Object)putResult);
                throw new LoadException("hadoop put fail. msg: " + putResult.getStderr());
            }
        }
    }

    private Set<String> getInputPaths(Map<String, Object> jobConf) {
        HashSet<String> inputPaths = new HashSet<String>();
        Map tables = (Map)jobConf.get("tables");
        for (Map table : tables.values()) {
            Map sourceFileSchema = (Map)table.get("source_file_schema");
            for (Map schema : sourceFileSchema.values()) {
                List fileUrls = (List)schema.get("file_urls");
                inputPaths.addAll(fileUrls);
            }
        }
        return inputPaths;
    }

    private int calcReduceNumByInputSize(Set<String> inputPaths) throws InputSizeInvalidException {
        String[] fileInfos;
        String[] envp = new String[]{"LC_ALL=" + Config.locale};
        int reduceNum = 0;
        String hadoopCountCmd = String.format(HADOOP_COUNT_CMD, HADOOP_CLIENT, this.hadoopConfig, StringUtils.join(inputPaths, (String)" "));
        LOG.info(hadoopCountCmd);
        CommandResult result = Util.executeCommand(hadoopCountCmd, envp);
        if (result.getReturnCode() != 0) {
            LOG.warn("hadoop count error, result: {}", (Object)result);
            return 1000;
        }
        long totalSizeB = 0L;
        for (String fileInfo : fileInfos = result.getStdout().split("\n")) {
            String[] fileInfoArr = fileInfo.trim().split(" +");
            if (fileInfoArr.length != 4) continue;
            totalSizeB += Long.parseLong(fileInfoArr[2]);
        }
        int inputSizeLimitGB = Config.load_input_size_limit_gb;
        if (inputSizeLimitGB != 0 && totalSizeB > (long)inputSizeLimitGB * 0x40000000L) {
            String failMsg = "Input file size[" + (float)totalSizeB / 1.0737418E9f + "GB] exceeds system limit[" + inputSizeLimitGB + "GB]";
            LOG.warn(failMsg);
            throw new InputSizeInvalidException(failMsg);
        }
        if (totalSizeB != 0L) {
            reduceNum = (int)(totalSizeB / Config.dpp_bytes_per_reduce) + 1;
        }
        return reduceNum;
    }

    private int calcReduceNumByTablet(Map<String, Object> jobConf) {
        int reduceNum = 0;
        Map tables = (Map)jobConf.get("tables");
        for (Map table : tables.values()) {
            Map views = (Map)table.get("views");
            for (Map view : views.values()) {
                if (view.containsKey("hash_mod")) {
                    reduceNum += ((Integer)view.get("hash_mod")).intValue();
                    continue;
                }
                if (!view.containsKey("key_ranges")) continue;
                List rangeList = (List)view.get("key_ranges");
                reduceNum += rangeList.size();
            }
        }
        return reduceNum;
    }

    public EtlStatus getEtlJobStatus(String etlJobId) {
        EtlStatus status = new EtlStatus();
        status.setState(TEtlState.RUNNING);
        String hadoopStatusCmd = String.format(HADOOP_STATUS_CMD, HADOOP_CLIENT, this.hadoopConfig, etlJobId);
        LOG.info(hadoopStatusCmd);
        String[] envp = new String[]{"LC_ALL=" + Config.locale};
        CommandResult result = Util.executeCommand(hadoopStatusCmd, envp);
        String stdout = result.getStdout();
        if (result.getReturnCode() != 0) {
            if (stdout != null && stdout.contains("Could not find job")) {
                LOG.warn("cannot find hadoop etl job: {}", (Object)etlJobId);
                status.setState(TEtlState.CANCELLED);
            }
            return status;
        }
        HashMap<String, String> stats = new HashMap<String, String>();
        HashMap<String, String> counters = new HashMap<String, String>();
        String[] stdoutLines = stdout.split("\n");
        String[] array = null;
        for (String line : stdoutLines) {
            array = line.split(":");
            if (array.length == 2) {
                stats.put(array[0].trim(), array[1].trim());
            }
            if ((array = line.split("=")).length != 2) continue;
            counters.put(array[0].trim(), array[1].trim());
        }
        status.setStats(stats);
        status.setCounters(counters);
        for (String key : counters.keySet()) {
            if (!key.startsWith("tracking URL")) continue;
            status.setTrackingUrl(key.substring(14) + "=" + (String)counters.get(key));
            break;
        }
        if (stats.containsKey("job state")) {
            int jobState = Integer.parseInt((String)stats.get("job state"));
            if (jobState == 3 || jobState == 5 || jobState == 6) {
                status.setState(TEtlState.CANCELLED);
            } else if (jobState == 2) {
                status.setState(TEtlState.FINISHED);
            } else {
                status.setState(TEtlState.RUNNING);
            }
        }
        return status;
    }

    public Map<String, Long> getEtlFiles(String outputPath) {
        String[] lsFileResults;
        String[] envp = new String[]{"LC_ALL=" + Config.locale};
        HashMap fileMap = Maps.newHashMap();
        String fileDir = outputPath + "/" + DPP_OUTPUT_DIR;
        String hadoopLsCmd = String.format(HADOOP_LS_CMD, HADOOP_CLIENT, this.hadoopConfig, fileDir);
        LOG.info(hadoopLsCmd);
        CommandResult lsResult = Util.executeCommand(hadoopLsCmd, envp);
        if (lsResult.getReturnCode() != 0) {
            String hadoopTestCmd = String.format(HADOOP_TEST_CMD, HADOOP_CLIENT, this.hadoopConfig, "-d", outputPath);
            LOG.info(hadoopTestCmd);
            CommandResult testResult = Util.executeCommand(hadoopTestCmd, envp);
            if (testResult.getReturnCode() != 0) {
                LOG.info("hadoop dir does not exist. dir: {}", (Object)outputPath);
                return null;
            }
            hadoopTestCmd = String.format(HADOOP_TEST_CMD, HADOOP_CLIENT, this.hadoopConfig, "-d", fileDir);
            LOG.info(hadoopTestCmd);
            testResult = Util.executeCommand(hadoopTestCmd, envp);
            if (testResult.getReturnCode() != 0) {
                LOG.info("hadoop dir does not exist. dir: {}", (Object)fileDir);
                return fileMap;
            }
            return null;
        }
        String stdout = lsResult.getStdout();
        for (String line : lsFileResults = stdout.split("\n")) {
            String[] fileInfos = line.split(" +");
            if (fileInfos.length != 8) continue;
            String filePath = fileInfos[fileInfos.length - 1];
            long fileSize = -1L;
            try {
                fileSize = Long.parseLong(fileInfos[4]);
            }
            catch (NumberFormatException e) {
                LOG.warn("file size format error. line: {}", (Object)line);
            }
            fileMap.put(filePath, fileSize);
        }
        return fileMap;
    }

    public void killEtlJob(String etlJobId) {
        String[] envp = new String[]{"LC_ALL=" + Config.locale};
        String hadoopKillCmd = String.format(HADOOP_KILL_CMD, HADOOP_CLIENT, this.hadoopConfig, etlJobId);
        LOG.info(hadoopKillCmd);
        Util.executeCommand(hadoopKillCmd, envp);
    }

    public void deleteEtlOutputPath(String outputPath) {
        String[] envp = new String[]{"LC_ALL=" + Config.locale};
        String hadoopRmCmd = String.format(HADOOP_RMR_CMD, HADOOP_CLIENT, this.hadoopConfig, outputPath);
        LOG.info(hadoopRmCmd);
        Util.executeCommand(hadoopRmCmd, envp);
    }

    public static String getEtlOutputPath(String fsDefaultName, String outputPath, long dbId, String loadLabel, String etlOutputDir) {
        return String.format(ETL_OUTPUT_PATH, fsDefaultName, outputPath, dbId, loadLabel, etlOutputDir);
    }

    private class InputSizeInvalidException
    extends LoadException {
        public InputSizeInvalidException(String msg) {
            super(msg);
        }
    }
}

