/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.consistency;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import java.util.Iterator;
import java.util.Map;
import org.apache.doris.catalog.Catalog;
import org.apache.doris.catalog.Database;
import org.apache.doris.catalog.MaterializedIndex;
import org.apache.doris.catalog.OlapTable;
import org.apache.doris.catalog.Partition;
import org.apache.doris.catalog.Replica;
import org.apache.doris.catalog.Table;
import org.apache.doris.catalog.Tablet;
import org.apache.doris.catalog.TabletInvertedIndex;
import org.apache.doris.catalog.TabletMeta;
import org.apache.doris.common.Config;
import org.apache.doris.persist.ConsistencyCheckInfo;
import org.apache.doris.qe.ConnectContext;
import org.apache.doris.task.AgentBatchTask;
import org.apache.doris.task.AgentTask;
import org.apache.doris.task.AgentTaskExecutor;
import org.apache.doris.task.AgentTaskQueue;
import org.apache.doris.task.CheckConsistencyTask;
import org.apache.doris.thrift.TResourceInfo;
import org.apache.doris.thrift.TTaskType;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class CheckConsistencyJob {
    private static final Logger LOG = LogManager.getLogger(CheckConsistencyJob.class);
    private static final long CHECK_CONSISTENT_TIME_COST_PER_GIGABYTE_MS = 1000000L;
    private JobState state = JobState.PENDING;
    private long tabletId;
    private Map<Long, Long> checksumMap;
    private int checkedSchemaHash;
    private long checkedVersion;
    private long createTime;
    private long timeoutMs;

    public CheckConsistencyJob(long tabletId) {
        this.tabletId = tabletId;
        this.checksumMap = Maps.newHashMap();
        this.checkedSchemaHash = -1;
        this.checkedVersion = -1L;
        this.createTime = System.currentTimeMillis();
        this.timeoutMs = 0L;
    }

    public JobState getState() {
        return this.state;
    }

    public void setState(JobState state) {
        this.state = state;
    }

    public long getTabletId() {
        return this.tabletId;
    }

    public synchronized void setChecksum(long backendId, long checksum) {
        this.checksumMap.put(backendId, checksum);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean sendTasks() {
        TabletInvertedIndex invertedIndex = Catalog.getCurrentInvertedIndex();
        TabletMeta tabletMeta = invertedIndex.getTabletMeta(this.tabletId);
        if (tabletMeta == null) {
            LOG.debug("tablet[{}] has been removed", (Object)this.tabletId);
            return false;
        }
        Database db = Catalog.getCurrentCatalog().getDbNullable(tabletMeta.getDbId());
        if (db == null) {
            LOG.debug("db[{}] does not exist", (Object)tabletMeta.getDbId());
            return false;
        }
        TResourceInfo resourceInfo = null;
        if (ConnectContext.get() != null) {
            resourceInfo = ConnectContext.get().toResourceCtx();
        }
        Tablet tablet = null;
        AgentBatchTask batchTask = new AgentBatchTask();
        Table table = db.getTableNullable(tabletMeta.getTableId());
        if (table == null) {
            LOG.debug("table[{}] does not exist", (Object)tabletMeta.getTableId());
            return false;
        }
        table.readLock();
        try {
            OlapTable olapTable = (OlapTable)table;
            Partition partition = olapTable.getPartition(tabletMeta.getPartitionId());
            if (partition == null) {
                LOG.debug("partition[{}] does not exist", (Object)tabletMeta.getPartitionId());
                boolean bl = false;
                return bl;
            }
            short replicaNum = olapTable.getPartitionInfo().getReplicaAllocation(partition.getId()).getTotalReplicaNum();
            if (replicaNum == 1) {
                LOG.debug("partition[{}]'s replication num is 1. skip consistency check", (Object)partition.getId());
                boolean bl = false;
                return bl;
            }
            MaterializedIndex index = partition.getIndex(tabletMeta.getIndexId());
            if (index == null) {
                LOG.debug("index[{}] does not exist", (Object)tabletMeta.getIndexId());
                boolean bl = false;
                return bl;
            }
            tablet = index.getTablet(this.tabletId);
            if (tablet == null) {
                LOG.debug("tablet[{}] does not exist", (Object)this.tabletId);
                boolean bl = false;
                return bl;
            }
            this.checkedVersion = partition.getVisibleVersion();
            this.checkedSchemaHash = olapTable.getSchemaHashByIndexId(tabletMeta.getIndexId());
            int sentTaskReplicaNum = 0;
            long maxDataSize = 0L;
            for (Replica replica : tablet.getReplicas()) {
                if (replica.getState() == Replica.ReplicaState.CLONE || replica.getState() == Replica.ReplicaState.DECOMMISSION) continue;
                if (replica.getDataSize() > maxDataSize) {
                    maxDataSize = replica.getDataSize();
                }
                CheckConsistencyTask task = new CheckConsistencyTask(resourceInfo, replica.getBackendId(), tabletMeta.getDbId(), tabletMeta.getTableId(), tabletMeta.getPartitionId(), tabletMeta.getIndexId(), this.tabletId, this.checkedSchemaHash, this.checkedVersion);
                batchTask.addTask(task);
                this.checksumMap.put(replica.getBackendId(), -1L);
                ++sentTaskReplicaNum;
            }
            if (sentTaskReplicaNum < replicaNum / 2 + 1) {
                LOG.info("tablet[{}] does not have enough replica to check.", (Object)this.tabletId);
            } else {
                if (maxDataSize > 0L) {
                    this.timeoutMs = maxDataSize / 1000L / 1000L / 1000L * 1000000L;
                }
                this.timeoutMs = Math.max(this.timeoutMs, Config.check_consistency_default_timeout_second * 1000L);
                this.state = JobState.RUNNING;
            }
        }
        finally {
            table.readUnlock();
        }
        if (this.state != JobState.RUNNING) {
            if (!table.writeLockIfExist()) {
                LOG.debug("table[{}] does not exist", (Object)tabletMeta.getTableId());
                return false;
            }
            try {
                tablet.setCheckedVersion(this.checkedVersion);
            }
            finally {
                table.writeUnlock();
            }
            return false;
        }
        Preconditions.checkState((batchTask.getTaskNum() > 0 ? 1 : 0) != 0);
        for (AgentTask task : batchTask.getAllTasks()) {
            AgentTaskQueue.addTask(task);
        }
        AgentTaskExecutor.submit(batchTask);
        LOG.debug("tablet[{}] send check consistency task. num: {}", (Object)this.tabletId, (Object)batchTask.getTaskNum());
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized int tryFinishJob() {
        if (this.state == JobState.PENDING) {
            return 0;
        }
        TabletMeta tabletMeta = Catalog.getCurrentInvertedIndex().getTabletMeta(this.tabletId);
        if (tabletMeta == null) {
            LOG.warn("tablet[{}] has been removed", (Object)this.tabletId);
            return -1;
        }
        Database db = Catalog.getCurrentCatalog().getDbNullable(tabletMeta.getDbId());
        if (db == null) {
            LOG.warn("db[{}] does not exist", (Object)tabletMeta.getDbId());
            return -1;
        }
        boolean isConsistent = true;
        Table table = db.getTableNullable(tabletMeta.getTableId());
        if (table == null || !table.writeLockIfExist()) {
            LOG.warn("table[{}] does not exist", (Object)tabletMeta.getTableId());
            return -1;
        }
        try {
            OlapTable olapTable = (OlapTable)table;
            Partition partition = olapTable.getPartition(tabletMeta.getPartitionId());
            if (partition == null) {
                LOG.warn("partition[{}] does not exist", (Object)tabletMeta.getPartitionId());
                int n = -1;
                return n;
            }
            MaterializedIndex index = partition.getIndex(tabletMeta.getIndexId());
            if (index == null) {
                LOG.warn("index[{}] does not exist", (Object)tabletMeta.getIndexId());
                int n = -1;
                return n;
            }
            Tablet tablet = index.getTablet(this.tabletId);
            if (tablet == null) {
                LOG.warn("tablet[{}] does not exist", (Object)this.tabletId);
                int n = -1;
                return n;
            }
            if (this.checkedSchemaHash != olapTable.getSchemaHashByIndexId(tabletMeta.getIndexId())) {
                LOG.info("index[{}]'s schema hash has been changed. [{} -> {}]. retry", (Object)tabletMeta.getIndexId(), (Object)this.checkedSchemaHash, (Object)olapTable.getSchemaHashByIndexId(tabletMeta.getIndexId()));
                int n = -1;
                return n;
            }
            if (!this.isTimeout()) {
                boolean isFinished = true;
                Iterator<Map.Entry<Long, Long>> iter = this.checksumMap.entrySet().iterator();
                while (iter.hasNext()) {
                    Map.Entry<Long, Long> entry = iter.next();
                    if (tablet.getReplicaByBackendId(entry.getKey()) == null) {
                        LOG.debug("tablet[{}]'s replica in backend[{}] does not exist. remove from checksumMap", (Object)this.tabletId, (Object)entry.getKey());
                        iter.remove();
                        continue;
                    }
                    if (entry.getValue() != -1L) continue;
                    LOG.debug("tablet[{}] has unfinished replica check sum task. backend[{}]", (Object)this.tabletId, (Object)entry.getKey());
                    isFinished = false;
                }
                if (!isFinished) {
                    int entry = 0;
                    return entry;
                }
                long lastChecksum = -1L;
                for (Map.Entry<Long, Long> entry : this.checksumMap.entrySet()) {
                    long checksum = entry.getValue();
                    if (lastChecksum == -1L) {
                        lastChecksum = checksum;
                        continue;
                    }
                    if (checksum == lastChecksum) continue;
                    isConsistent = false;
                    break;
                }
                if (isConsistent) {
                    LOG.info("tablet[{}] is consistent: {}", (Object)this.tabletId, this.checksumMap.keySet());
                } else {
                    StringBuilder sb = new StringBuilder();
                    sb.append("tablet[").append(this.tabletId).append("] is not consistent: ");
                    for (Map.Entry<Long, Long> entry : this.checksumMap.entrySet()) {
                        sb.append("[").append(entry.getKey()).append("-").append(entry.getValue()).append("]");
                    }
                    sb.append(" [").append(tabletMeta).append("]");
                    LOG.error(sb.toString());
                }
            } else {
                LOG.info("tablet[{}] check consistency job cancelled. timeout", (Object)this.tabletId);
            }
            long lastCheckTime = System.currentTimeMillis();
            db.setLastCheckTime(lastCheckTime);
            olapTable.setLastCheckTime(lastCheckTime);
            partition.setLastCheckTime(lastCheckTime);
            index.setLastCheckTime(lastCheckTime);
            tablet.setLastCheckTime(lastCheckTime);
            tablet.setIsConsistent(isConsistent);
            tablet.setCheckedVersion(this.checkedVersion);
            ConsistencyCheckInfo info = new ConsistencyCheckInfo(db.getId(), table.getId(), partition.getId(), index.getId(), this.tabletId, lastCheckTime, this.checkedVersion, isConsistent);
            Catalog.getCurrentCatalog().getEditLog().logFinishConsistencyCheck(info);
            int n = 1;
            return n;
        }
        finally {
            table.writeUnlock();
        }
    }

    private boolean isTimeout() {
        return this.timeoutMs != 0L && System.currentTimeMillis() - this.createTime >= this.timeoutMs;
    }

    public synchronized void handleFinishedReplica(long backendId, long checksum) {
        if (this.checksumMap.containsKey(backendId)) {
            this.checksumMap.put(backendId, checksum);
        } else {
            LOG.warn("can not find backend[{}] in tablet[{}]'s consistency check job", (Object)backendId, (Object)this.tabletId);
        }
    }

    public synchronized void clear() {
        for (Long backendId : this.checksumMap.keySet()) {
            AgentTaskQueue.removeTask(backendId, TTaskType.CHECK_CONSISTENCY, this.tabletId);
        }
    }

    public static enum JobState {
        PENDING,
        RUNNING;

    }
}

