/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.common.util;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Sets;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import org.apache.doris.analysis.DataSortInfo;
import org.apache.doris.analysis.DateLiteral;
import org.apache.doris.catalog.Column;
import org.apache.doris.catalog.DataProperty;
import org.apache.doris.catalog.KeysType;
import org.apache.doris.catalog.PrimitiveType;
import org.apache.doris.catalog.ReplicaAllocation;
import org.apache.doris.catalog.ScalarType;
import org.apache.doris.catalog.Type;
import org.apache.doris.common.AnalysisException;
import org.apache.doris.common.Config;
import org.apache.doris.common.util.TimeUtils;
import org.apache.doris.resource.Tag;
import org.apache.doris.thrift.TCompressionType;
import org.apache.doris.thrift.TSortType;
import org.apache.doris.thrift.TStorageFormat;
import org.apache.doris.thrift.TStorageMedium;
import org.apache.doris.thrift.TStorageType;
import org.apache.doris.thrift.TTabletType;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class PropertyAnalyzer {
    private static final Logger LOG = LogManager.getLogger(PropertyAnalyzer.class);
    private static final String COMMA_SEPARATOR = ",";
    public static final String PROPERTIES_SHORT_KEY = "short_key";
    public static final String PROPERTIES_REPLICATION_NUM = "replication_num";
    public static final String PROPERTIES_REPLICATION_ALLOCATION = "replication_allocation";
    public static final String PROPERTIES_STORAGE_TYPE = "storage_type";
    public static final String PROPERTIES_STORAGE_MEDIUM = "storage_medium";
    public static final String PROPERTIES_STORAGE_COLDOWN_TIME = "storage_cooldown_time";
    public static final String PROPERTIES_VERSION_INFO = "version_info";
    public static final String PROPERTIES_SCHEMA_VERSION = "schema_version";
    public static final String PROPERTIES_BF_COLUMNS = "bloom_filter_columns";
    public static final String PROPERTIES_BF_FPP = "bloom_filter_fpp";
    private static final double MAX_FPP = 0.05;
    private static final double MIN_FPP = 1.0E-4;
    public static final String PROPERTIES_COLUMN_SEPARATOR = "column_separator";
    public static final String PROPERTIES_LINE_DELIMITER = "line_delimiter";
    public static final String PROPERTIES_COLOCATE_WITH = "colocate_with";
    public static final String PROPERTIES_TIMEOUT = "timeout";
    public static final String PROPERTIES_COMPRESSION = "compression";
    public static final String PROPERTIES_DISTRIBUTION_TYPE = "distribution_type";
    public static final String PROPERTIES_SEND_CLEAR_ALTER_TASK = "send_clear_alter_tasks";
    public static final String PROPERTIES_STORAGE_FORMAT = "storage_format";
    public static final String PROPERTIES_INMEMORY = "in_memory";
    public static final String PROPERTIES_TABLET_TYPE = "tablet_type";
    public static final String PROPERTIES_STRICT_RANGE = "strict_range";
    public static final String PROPERTIES_USE_TEMP_PARTITION_NAME = "use_temp_partition_name";
    public static final String PROPERTIES_TYPE = "type";
    public static final String PROPERTIES_FUNCTION_COLUMN = "function_column";
    public static final String PROPERTIES_SEQUENCE_TYPE = "sequence_type";
    public static final String PROPERTIES_SWAP_TABLE = "swap";
    public static final String TAG_LOCATION = "tag.location";
    public static final String PROPERTIES_DISABLE_QUERY = "disable_query";
    public static final String PROPERTIES_DISABLE_LOAD = "disable_load";

    public static DataProperty analyzeDataProperty(Map<String, String> properties, DataProperty oldDataProperty) throws AnalysisException {
        if (properties == null) {
            return oldDataProperty;
        }
        TStorageMedium storageMedium = null;
        long coolDownTimeStamp = 253402271999000L;
        boolean hasMedium = false;
        boolean hasCooldown = false;
        for (Map.Entry<String, String> entry : properties.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();
            if (!hasMedium && key.equalsIgnoreCase(PROPERTIES_STORAGE_MEDIUM)) {
                hasMedium = true;
                if (value.equalsIgnoreCase(TStorageMedium.SSD.name())) {
                    storageMedium = TStorageMedium.SSD;
                    continue;
                }
                if (value.equalsIgnoreCase(TStorageMedium.HDD.name())) {
                    storageMedium = TStorageMedium.HDD;
                    continue;
                }
                throw new AnalysisException("Invalid storage medium: " + value);
            }
            if (hasCooldown || !key.equalsIgnoreCase(PROPERTIES_STORAGE_COLDOWN_TIME)) continue;
            hasCooldown = true;
            DateLiteral dateLiteral = new DateLiteral(value, (Type)Type.DATETIME);
            coolDownTimeStamp = dateLiteral.unixTimestamp(TimeUtils.getTimeZone());
        }
        if (!hasCooldown && !hasMedium) {
            return oldDataProperty;
        }
        properties.remove(PROPERTIES_STORAGE_MEDIUM);
        properties.remove(PROPERTIES_STORAGE_COLDOWN_TIME);
        if (hasCooldown && !hasMedium) {
            throw new AnalysisException("Invalid data property. storage medium property is not found");
        }
        if (storageMedium == TStorageMedium.HDD && hasCooldown) {
            throw new AnalysisException("Can not assign cooldown timestamp to HDD storage medium");
        }
        long currentTimeMs = System.currentTimeMillis();
        if (storageMedium == TStorageMedium.SSD && hasCooldown && coolDownTimeStamp <= currentTimeMs) {
            throw new AnalysisException("Cooldown time should later than now");
        }
        if (storageMedium == TStorageMedium.SSD && !hasCooldown) {
            coolDownTimeStamp = currentTimeMs + Config.storage_cooldown_second * 1000L;
        }
        Preconditions.checkNotNull(storageMedium);
        return new DataProperty(storageMedium, coolDownTimeStamp);
    }

    public static short analyzeShortKeyColumnCount(Map<String, String> properties) throws AnalysisException {
        short shortKeyColumnCount = -1;
        if (properties != null && properties.containsKey(PROPERTIES_SHORT_KEY)) {
            try {
                shortKeyColumnCount = Short.parseShort(properties.get(PROPERTIES_SHORT_KEY));
            }
            catch (NumberFormatException e) {
                throw new AnalysisException("Short key: " + e.getMessage());
            }
            if (shortKeyColumnCount <= 0) {
                throw new AnalysisException("Short key column count should larger than 0.");
            }
            properties.remove(PROPERTIES_SHORT_KEY);
        }
        return shortKeyColumnCount;
    }

    private static Short analyzeReplicationNum(Map<String, String> properties, String prefix, short oldReplicationNum) throws AnalysisException {
        String propKey;
        Short replicationNum = oldReplicationNum;
        String string = propKey = Strings.isNullOrEmpty((String)prefix) ? PROPERTIES_REPLICATION_NUM : prefix + "." + PROPERTIES_REPLICATION_NUM;
        if (properties != null && properties.containsKey(propKey)) {
            try {
                replicationNum = Short.valueOf(properties.get(propKey));
            }
            catch (Exception e) {
                throw new AnalysisException(e.getMessage());
            }
            if (replicationNum < Config.min_replication_num_per_tablet || replicationNum > Config.max_replication_num_per_tablet) {
                throw new AnalysisException("Replication num should between " + Config.min_replication_num_per_tablet + " and " + Config.max_replication_num_per_tablet);
            }
            properties.remove(propKey);
        }
        return replicationNum;
    }

    public static String analyzeColumnSeparator(Map<String, String> properties, String oldColumnSeparator) {
        String columnSeparator = oldColumnSeparator;
        if (properties != null && properties.containsKey(PROPERTIES_COLUMN_SEPARATOR)) {
            columnSeparator = properties.get(PROPERTIES_COLUMN_SEPARATOR);
            properties.remove(PROPERTIES_COLUMN_SEPARATOR);
        }
        return columnSeparator;
    }

    public static String analyzeLineDelimiter(Map<String, String> properties, String oldLineDelimiter) {
        String lineDelimiter = oldLineDelimiter;
        if (properties != null && properties.containsKey(PROPERTIES_LINE_DELIMITER)) {
            lineDelimiter = properties.get(PROPERTIES_LINE_DELIMITER);
            properties.remove(PROPERTIES_LINE_DELIMITER);
        }
        return lineDelimiter;
    }

    public static TStorageType analyzeStorageType(Map<String, String> properties) throws AnalysisException {
        TStorageType tStorageType = TStorageType.COLUMN;
        if (properties != null && properties.containsKey(PROPERTIES_STORAGE_TYPE)) {
            String storageType = properties.get(PROPERTIES_STORAGE_TYPE);
            if (!storageType.equalsIgnoreCase(TStorageType.COLUMN.name())) {
                throw new AnalysisException("Invalid storage type: " + storageType);
            }
            tStorageType = TStorageType.COLUMN;
            properties.remove(PROPERTIES_STORAGE_TYPE);
        }
        return tStorageType;
    }

    public static TTabletType analyzeTabletType(Map<String, String> properties) throws AnalysisException {
        TTabletType tTabletType = TTabletType.TABLET_TYPE_DISK;
        if (properties != null && properties.containsKey(PROPERTIES_TABLET_TYPE)) {
            String tabletType = properties.get(PROPERTIES_TABLET_TYPE);
            if (tabletType.equalsIgnoreCase("memory")) {
                tTabletType = TTabletType.TABLET_TYPE_MEMORY;
            } else if (tabletType.equalsIgnoreCase("disk")) {
                tTabletType = TTabletType.TABLET_TYPE_DISK;
            } else {
                throw new AnalysisException("Invalid tablet type");
            }
            properties.remove(PROPERTIES_TABLET_TYPE);
        }
        return tTabletType;
    }

    public static long analyzeVersionInfo(Map<String, String> properties) throws AnalysisException {
        long version = 1L;
        if (properties != null && properties.containsKey(PROPERTIES_VERSION_INFO)) {
            String versionInfoStr = properties.get(PROPERTIES_VERSION_INFO);
            try {
                version = Long.parseLong(versionInfoStr);
            }
            catch (NumberFormatException e) {
                throw new AnalysisException("version info number format error: " + versionInfoStr);
            }
            properties.remove(PROPERTIES_VERSION_INFO);
        }
        return version;
    }

    public static int analyzeSchemaVersion(Map<String, String> properties) throws AnalysisException {
        int schemaVersion = 0;
        if (properties != null && properties.containsKey(PROPERTIES_SCHEMA_VERSION)) {
            String schemaVersionStr = properties.get(PROPERTIES_SCHEMA_VERSION);
            try {
                schemaVersion = Integer.parseInt(schemaVersionStr);
            }
            catch (Exception e) {
                throw new AnalysisException("schema version format error");
            }
            properties.remove(PROPERTIES_SCHEMA_VERSION);
        }
        return schemaVersion;
    }

    public static Set<String> analyzeBloomFilterColumns(Map<String, String> properties, List<Column> columns, KeysType keysType) throws AnalysisException {
        HashSet bfColumns = null;
        if (properties != null && properties.containsKey(PROPERTIES_BF_COLUMNS)) {
            bfColumns = Sets.newHashSet();
            String bfColumnsStr = properties.get(PROPERTIES_BF_COLUMNS);
            if (Strings.isNullOrEmpty((String)bfColumnsStr)) {
                return bfColumns;
            }
            String[] bfColumnArr = bfColumnsStr.split(COMMA_SEPARATOR);
            TreeSet bfColumnSet = Sets.newTreeSet((Comparator)String.CASE_INSENSITIVE_ORDER);
            for (String bfColumn : bfColumnArr) {
                bfColumn = bfColumn.trim();
                boolean found = false;
                for (Column column : columns) {
                    if (!column.getName().equalsIgnoreCase(bfColumn)) continue;
                    PrimitiveType type = column.getDataType();
                    if (type == PrimitiveType.TINYINT || type == PrimitiveType.FLOAT || type == PrimitiveType.DOUBLE || type == PrimitiveType.BOOLEAN) {
                        throw new AnalysisException((Object)((Object)type) + " is not supported in bloom filter index. invalid column: " + bfColumn);
                    }
                    if (keysType != KeysType.AGG_KEYS || column.isKey()) {
                        if (!bfColumnSet.add(bfColumn)) {
                            throw new AnalysisException("Reduplicated bloom filter column: " + bfColumn);
                        }
                        bfColumns.add(column.getName());
                        found = true;
                        break;
                    }
                    throw new AnalysisException("Bloom filter index only used in columns of UNIQUE_KEYS/DUP_KEYS table or key columns of AGG_KEYS table. invalid column: " + bfColumn);
                }
                if (found) continue;
                throw new AnalysisException("Bloom filter column does not exist in table. invalid column: " + bfColumn);
            }
            properties.remove(PROPERTIES_BF_COLUMNS);
        }
        return bfColumns;
    }

    public static double analyzeBloomFilterFpp(Map<String, String> properties) throws AnalysisException {
        double bfFpp = 0.0;
        if (properties != null && properties.containsKey(PROPERTIES_BF_FPP)) {
            String bfFppStr = properties.get(PROPERTIES_BF_FPP);
            try {
                bfFpp = Double.parseDouble(bfFppStr);
            }
            catch (NumberFormatException e) {
                throw new AnalysisException("Bloom filter fpp is not Double");
            }
            if (bfFpp < 1.0E-4 || bfFpp > 0.05) {
                throw new AnalysisException("Bloom filter fpp should in [1.0E-4, 0.05]");
            }
            properties.remove(PROPERTIES_BF_FPP);
        }
        return bfFpp;
    }

    public static String analyzeColocate(Map<String, String> properties) throws AnalysisException {
        String colocateGroup = null;
        if (properties != null && properties.containsKey(PROPERTIES_COLOCATE_WITH)) {
            colocateGroup = properties.get(PROPERTIES_COLOCATE_WITH);
            properties.remove(PROPERTIES_COLOCATE_WITH);
        }
        return colocateGroup;
    }

    public static long analyzeTimeout(Map<String, String> properties, long defaultTimeout) throws AnalysisException {
        long timeout = defaultTimeout;
        if (properties != null && properties.containsKey(PROPERTIES_TIMEOUT)) {
            String timeoutStr = properties.get(PROPERTIES_TIMEOUT);
            try {
                timeout = Long.parseLong(timeoutStr);
            }
            catch (NumberFormatException e) {
                throw new AnalysisException("Invalid timeout format: " + timeoutStr);
            }
            properties.remove(PROPERTIES_TIMEOUT);
        }
        return timeout;
    }

    public static TCompressionType analyzeCompressionType(Map<String, String> properties) throws AnalysisException {
        String compressionType = "";
        if (properties == null || !properties.containsKey(PROPERTIES_COMPRESSION)) {
            return TCompressionType.LZ4F;
        }
        compressionType = properties.get(PROPERTIES_COMPRESSION);
        properties.remove(PROPERTIES_COMPRESSION);
        if (compressionType.equalsIgnoreCase("no_compression")) {
            return TCompressionType.NO_COMPRESSION;
        }
        if (compressionType.equalsIgnoreCase("lz4")) {
            return TCompressionType.LZ4;
        }
        if (compressionType.equalsIgnoreCase("lz4f")) {
            return TCompressionType.LZ4F;
        }
        if (compressionType.equalsIgnoreCase("zlib")) {
            return TCompressionType.ZLIB;
        }
        if (compressionType.equalsIgnoreCase("zstd")) {
            return TCompressionType.ZSTD;
        }
        if (compressionType.equalsIgnoreCase("snappy")) {
            return TCompressionType.SNAPPY;
        }
        if (compressionType.equalsIgnoreCase("default_compression")) {
            return TCompressionType.LZ4F;
        }
        throw new AnalysisException("unknown compression type: " + compressionType);
    }

    public static TStorageFormat analyzeStorageFormat(Map<String, String> properties) throws AnalysisException {
        String storageFormat = "";
        if (properties == null || !properties.containsKey(PROPERTIES_STORAGE_FORMAT)) {
            return TStorageFormat.V2;
        }
        storageFormat = properties.get(PROPERTIES_STORAGE_FORMAT);
        properties.remove(PROPERTIES_STORAGE_FORMAT);
        if (storageFormat.equalsIgnoreCase("v1")) {
            throw new AnalysisException("Storage format V1 has been deprecated since version 0.14, please use V2 instead");
        }
        if (storageFormat.equalsIgnoreCase("v2")) {
            return TStorageFormat.V2;
        }
        if (storageFormat.equalsIgnoreCase("default")) {
            return TStorageFormat.V2;
        }
        throw new AnalysisException("unknown storage format: " + storageFormat);
    }

    public static boolean analyzeBooleanProp(Map<String, String> properties, String propKey, boolean defaultVal) {
        if (properties != null && properties.containsKey(propKey)) {
            String val = properties.get(propKey);
            properties.remove(propKey);
            return Boolean.parseBoolean(val);
        }
        return defaultVal;
    }

    public static String analyzeType(Map<String, String> properties) throws AnalysisException {
        String type = null;
        if (properties != null && properties.containsKey(PROPERTIES_TYPE)) {
            type = properties.get(PROPERTIES_TYPE);
            properties.remove(PROPERTIES_TYPE);
        }
        return type;
    }

    public static Type analyzeSequenceType(Map<String, String> properties, KeysType keysType) throws AnalysisException {
        String typeStr = null;
        String propertyName = "function_column.sequence_type";
        if (properties != null && properties.containsKey(propertyName)) {
            typeStr = properties.get(propertyName);
            properties.remove(propertyName);
        }
        if (typeStr == null) {
            return null;
        }
        if (typeStr != null && keysType != KeysType.UNIQUE_KEYS) {
            throw new AnalysisException("sequence column only support UNIQUE_KEYS");
        }
        PrimitiveType type = PrimitiveType.valueOf(typeStr.toUpperCase());
        if (!type.isFixedPointType() && !type.isDateType()) {
            throw new AnalysisException("sequence type only support integer types and date types");
        }
        return ScalarType.createType(type);
    }

    public static Boolean analyzeBackendDisableProperties(Map<String, String> properties, String key, Boolean defaultValue) throws AnalysisException {
        if (properties.containsKey(key)) {
            String value = properties.remove(key);
            return Boolean.valueOf(value);
        }
        return defaultValue;
    }

    public static Tag analyzeBackendTagProperties(Map<String, String> properties, Tag defaultValue) throws AnalysisException {
        if (properties.containsKey(TAG_LOCATION)) {
            String tagVal = properties.remove(TAG_LOCATION);
            return Tag.create("location", tagVal);
        }
        return defaultValue;
    }

    public static ReplicaAllocation analyzeReplicaAllocation(Map<String, String> properties, String prefix) throws AnalysisException {
        String propKey;
        if (properties == null || properties.isEmpty()) {
            return ReplicaAllocation.NOT_SET;
        }
        Short replicaNum = PropertyAnalyzer.analyzeReplicationNum(properties, prefix, (short)0);
        if (replicaNum > 0) {
            return new ReplicaAllocation(replicaNum);
        }
        String string = propKey = Strings.isNullOrEmpty((String)prefix) ? PROPERTIES_REPLICATION_ALLOCATION : prefix + "." + PROPERTIES_REPLICATION_ALLOCATION;
        if (!properties.containsKey(propKey)) {
            return ReplicaAllocation.NOT_SET;
        }
        ReplicaAllocation replicaAlloc = new ReplicaAllocation();
        String allocationVal = properties.remove(propKey);
        allocationVal = allocationVal.replaceAll(" ", "");
        String[] locations = allocationVal.split(COMMA_SEPARATOR);
        int totalReplicaNum = 0;
        for (String location : locations) {
            String[] parts = location.split(":");
            if (parts.length != 2) {
                throw new AnalysisException("Invalid replication allocation property: " + location);
            }
            if (!parts[0].startsWith(TAG_LOCATION)) {
                throw new AnalysisException("Invalid replication allocation tag property: " + location);
            }
            String locationVal = parts[0].substring(TAG_LOCATION.length() + 1);
            if (Strings.isNullOrEmpty((String)locationVal)) {
                throw new AnalysisException("Invalid replication allocation location tag property: " + location);
            }
            Short replicationNum = Short.valueOf(parts[1]);
            replicaAlloc.put(Tag.create("location", locationVal), replicationNum);
            totalReplicaNum += replicationNum.shortValue();
        }
        if (totalReplicaNum < Config.min_replication_num_per_tablet || totalReplicaNum > Config.max_replication_num_per_tablet) {
            throw new AnalysisException("Total replication num should between " + Config.min_replication_num_per_tablet + " and " + Config.max_replication_num_per_tablet);
        }
        if (replicaAlloc.isEmpty()) {
            throw new AnalysisException("Not specified replica allocation property");
        }
        return replicaAlloc;
    }

    public static DataSortInfo analyzeDataSortInfo(Map<String, String> properties, KeysType keyType, int keyCount, TStorageFormat storageFormat) throws AnalysisException {
        if (properties == null || properties.isEmpty()) {
            return new DataSortInfo(TSortType.LEXICAL, keyCount);
        }
        String sortMethod = TSortType.LEXICAL.name();
        if (properties.containsKey("data_sort.sort_type")) {
            sortMethod = properties.remove("data_sort.sort_type");
        }
        TSortType sortType = TSortType.LEXICAL;
        if (sortMethod.equalsIgnoreCase(TSortType.ZORDER.name())) {
            sortType = TSortType.ZORDER;
        } else if (sortMethod.equalsIgnoreCase(TSortType.LEXICAL.name())) {
            sortType = TSortType.LEXICAL;
        } else {
            throw new AnalysisException("only support zorder/lexical method!");
        }
        if (keyType != KeysType.DUP_KEYS && sortType == TSortType.ZORDER) {
            throw new AnalysisException("only duplicate key supports zorder method!");
        }
        if (storageFormat != TStorageFormat.V2 && sortType == TSortType.ZORDER) {
            throw new AnalysisException("only V2 storage format supports zorder method!");
        }
        int colNum = keyCount;
        if (properties.containsKey("data_sort.col_num")) {
            try {
                colNum = Integer.valueOf(properties.remove("data_sort.col_num"));
            }
            catch (Exception e) {
                throw new AnalysisException("param data_sort.col_num error");
            }
        }
        if (sortType == TSortType.ZORDER && (colNum <= 1 || colNum > keyCount)) {
            throw new AnalysisException("z-order needs 2 columns at least, " + keyCount + " columns at most!");
        }
        DataSortInfo dataSortInfo = new DataSortInfo(sortType, colNum);
        return dataSortInfo;
    }
}

