/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.clone;

import com.google.common.base.Preconditions;
import com.google.common.collect.EvictingQueue;
import com.google.common.collect.HashBasedTable;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.collect.Table;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.doris.catalog.Catalog;
import org.apache.doris.catalog.ColocateTableIndex;
import org.apache.doris.catalog.Database;
import org.apache.doris.catalog.DiskInfo;
import org.apache.doris.catalog.MaterializedIndex;
import org.apache.doris.catalog.OlapTable;
import org.apache.doris.catalog.Partition;
import org.apache.doris.catalog.Replica;
import org.apache.doris.catalog.Tablet;
import org.apache.doris.catalog.TabletInvertedIndex;
import org.apache.doris.clone.BackendLoadStatistic;
import org.apache.doris.clone.BalanceStatus;
import org.apache.doris.clone.BeLoadRebalancer;
import org.apache.doris.clone.ClusterLoadStatistic;
import org.apache.doris.clone.PartitionRebalancer;
import org.apache.doris.clone.Rebalancer;
import org.apache.doris.clone.RootPathLoadStatistic;
import org.apache.doris.clone.SchedException;
import org.apache.doris.clone.TabletSchedCtx;
import org.apache.doris.clone.TabletSchedulerStat;
import org.apache.doris.common.AnalysisException;
import org.apache.doris.common.Config;
import org.apache.doris.common.FeConstants;
import org.apache.doris.common.Pair;
import org.apache.doris.common.util.MasterDaemon;
import org.apache.doris.persist.ReplicaPersistInfo;
import org.apache.doris.resource.Tag;
import org.apache.doris.system.Backend;
import org.apache.doris.system.SystemInfoService;
import org.apache.doris.task.AgentBatchTask;
import org.apache.doris.task.AgentTask;
import org.apache.doris.task.AgentTaskExecutor;
import org.apache.doris.task.AgentTaskQueue;
import org.apache.doris.task.CloneTask;
import org.apache.doris.task.DropReplicaTask;
import org.apache.doris.thrift.TFinishTaskRequest;
import org.apache.doris.transaction.DatabaseTransactionMgr;
import org.apache.doris.transaction.TransactionState;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class TabletScheduler
extends MasterDaemon {
    private static final Logger LOG = LogManager.getLogger(TabletScheduler.class);
    private static final int MIN_BATCH_NUM = 50;
    private static final long STAT_UPDATE_INTERVAL_MS = 20000L;
    private static final long SCHEDULE_INTERVAL_MS = 1000L;
    public static final int BALANCE_SLOT_NUM_FOR_PATH = 2;
    private PriorityQueue<TabletSchedCtx> pendingTablets = new PriorityQueue();
    private Set<Long> allTabletIds = Sets.newHashSet();
    private Map<Long, TabletSchedCtx> runningTablets = Maps.newHashMap();
    private Queue<TabletSchedCtx> schedHistory = EvictingQueue.create((int)1000);
    private Map<Long, PathSlot> backendsWorkingSlots = Maps.newConcurrentMap();
    private Table<String, Tag, ClusterLoadStatistic> statisticMap = HashBasedTable.create();
    private long lastStatUpdateTime = 0L;
    private long lastSlotAdjustTime = 0L;
    private Catalog catalog;
    private SystemInfoService infoService;
    private TabletInvertedIndex invertedIndex;
    private ColocateTableIndex colocateTableIndex;
    private TabletSchedulerStat stat;
    private Rebalancer rebalancer;

    public TabletScheduler(Catalog catalog, SystemInfoService infoService, TabletInvertedIndex invertedIndex, TabletSchedulerStat stat, String rebalancerType) {
        super("tablet scheduler", 1000L);
        this.catalog = catalog;
        this.infoService = infoService;
        this.invertedIndex = invertedIndex;
        this.colocateTableIndex = catalog.getColocateTableIndex();
        this.stat = stat;
        this.rebalancer = rebalancerType.equalsIgnoreCase("partition") ? new PartitionRebalancer(infoService, invertedIndex) : new BeLoadRebalancer(infoService, invertedIndex);
    }

    public TabletSchedulerStat getStat() {
        return this.stat;
    }

    private boolean updateWorkingSlots() {
        List<Long> pathHashes;
        ImmutableMap<Long, Backend> backends = this.infoService.getBackendsInCluster(null);
        for (Backend backend : backends.values()) {
            if (backend.hasPathHash() || !backend.isAlive()) continue;
            LOG.info("not all backends have path info");
            return false;
        }
        HashSet deletedBeIds = Sets.newHashSet();
        for (Long beId : this.backendsWorkingSlots.keySet()) {
            if (backends.containsKey((Object)beId)) {
                pathHashes = ((Backend)backends.get((Object)beId)).getDisks().values().stream().filter(v -> v.getState() == DiskInfo.DiskState.ONLINE).map(DiskInfo::getPathHash).collect(Collectors.toList());
                this.backendsWorkingSlots.get(beId).updatePaths(pathHashes);
                continue;
            }
            deletedBeIds.add(beId);
        }
        for (Long beId : deletedBeIds) {
            this.backendsWorkingSlots.remove(beId);
            LOG.info("delete non exist backend: {}", (Object)beId);
        }
        for (Backend be : backends.values()) {
            if (this.backendsWorkingSlots.containsKey(be.getId())) continue;
            pathHashes = be.getDisks().values().stream().map(DiskInfo::getPathHash).collect(Collectors.toList());
            PathSlot slot = new PathSlot(pathHashes, Config.schedule_slot_num_per_path);
            this.backendsWorkingSlots.put(be.getId(), slot);
            LOG.info("add new backend {} with slots num: {}", (Object)be.getId(), (Object)be.getDisks().size());
        }
        return true;
    }

    public Map<Long, PathSlot> getBackendsWorkingSlots() {
        return this.backendsWorkingSlots;
    }

    public synchronized AddResult addTablet(TabletSchedCtx tablet, boolean force) {
        if (!force && Config.disable_tablet_scheduler) {
            return AddResult.DISABLED;
        }
        if (!force && this.containsTablet(tablet.getTabletId())) {
            return AddResult.ALREADY_IN;
        }
        if (!(tablet.getType() == TabletSchedCtx.Type.BALANCE || force || this.pendingTablets.size() <= Config.max_scheduling_tablets && this.runningTablets.size() <= Config.max_scheduling_tablets)) {
            return AddResult.LIMIT_EXCEED;
        }
        this.allTabletIds.add(tablet.getTabletId());
        this.pendingTablets.offer(tablet);
        return AddResult.ADDED;
    }

    public synchronized boolean containsTablet(long tabletId) {
        return this.allTabletIds.contains(tabletId);
    }

    public synchronized void changeTabletsPriorityToVeryHigh(long dbId, long tblId, List<Long> partitionIds) {
        PriorityQueue<TabletSchedCtx> newPendingTablets = new PriorityQueue<TabletSchedCtx>();
        for (TabletSchedCtx tabletCtx : this.pendingTablets) {
            if (tabletCtx.getDbId() == dbId && tabletCtx.getTblId() == tblId && partitionIds.contains(tabletCtx.getPartitionId())) {
                tabletCtx.setOrigPriority(TabletSchedCtx.Priority.VERY_HIGH);
            }
            newPendingTablets.add(tabletCtx);
        }
        this.pendingTablets = newPendingTablets;
    }

    @Override
    protected void runAfterCatalogReady() {
        if (!this.updateWorkingSlots()) {
            return;
        }
        this.updateClusterLoadStatisticsAndPriorityIfNecessary();
        this.schedulePendingTablets();
        this.handleRunningTablets();
        this.selectTabletsForBalance();
        this.stat.counterTabletScheduleRound.incrementAndGet();
    }

    private void updateClusterLoadStatisticsAndPriorityIfNecessary() {
        if (System.currentTimeMillis() - this.lastStatUpdateTime < 20000L) {
            return;
        }
        this.updateClusterLoadStatistic();
        this.rebalancer.updateLoadStatistic(this.statisticMap);
        this.adjustPriorities();
        this.lastStatUpdateTime = System.currentTimeMillis();
    }

    private void updateClusterLoadStatistic() {
        HashBasedTable newStatisticMap = HashBasedTable.create();
        Set<String> clusterNames = this.infoService.getClusterNames();
        for (String clusterName : clusterNames) {
            Set<Tag> tags = this.infoService.getTagsByCluster(clusterName);
            for (Tag tag : tags) {
                ClusterLoadStatistic clusterLoadStatistic = new ClusterLoadStatistic(clusterName, tag, this.infoService, this.invertedIndex);
                clusterLoadStatistic.init();
                newStatisticMap.put((Object)clusterName, (Object)tag, (Object)clusterLoadStatistic);
                LOG.debug("update cluster {} load statistic:\n{}", (Object)clusterName, (Object)clusterLoadStatistic.getBrief());
            }
        }
        this.statisticMap = newStatisticMap;
    }

    public Table<String, Tag, ClusterLoadStatistic> getStatisticMap() {
        return this.statisticMap;
    }

    private synchronized void adjustPriorities() {
        TabletSchedCtx tabletCtx;
        int size = this.pendingTablets.size();
        int changedNum = 0;
        for (int i = 0; i < size && (tabletCtx = this.pendingTablets.poll()) != null; ++i) {
            if (tabletCtx.adjustPriority(this.stat)) {
                ++changedNum;
            }
            this.pendingTablets.add(tabletCtx);
        }
        LOG.debug("adjust priority for all tablets. changed: {}, total: {}", (Object)changedNum, (Object)size);
    }

    private void schedulePendingTablets() {
        long start = System.currentTimeMillis();
        List<TabletSchedCtx> currentBatch = this.getNextTabletCtxBatch();
        LOG.debug("get {} tablets to schedule", (Object)currentBatch.size());
        AgentBatchTask batchTask = new AgentBatchTask();
        for (TabletSchedCtx tabletCtx : currentBatch) {
            try {
                if (Config.disable_tablet_scheduler) {
                    throw new SchedException(SchedException.Status.FINISHED, "tablet scheduler is disabled");
                }
                this.scheduleTablet(tabletCtx, batchTask);
            }
            catch (SchedException e) {
                tabletCtx.increaseFailedSchedCounter();
                tabletCtx.setErrMsg(e.getMessage());
                if (e.getStatus() == SchedException.Status.SCHEDULE_FAILED) {
                    if (tabletCtx.getType() == TabletSchedCtx.Type.BALANCE) {
                        if (Config.disable_balance) {
                            this.finalizeTabletCtx(tabletCtx, TabletSchedCtx.State.CANCELLED, e.getStatus(), "disable balance and " + e.getMessage());
                            continue;
                        }
                        if (tabletCtx.getFailedSchedCounter() > 10) {
                            this.finalizeTabletCtx(tabletCtx, TabletSchedCtx.State.CANCELLED, e.getStatus(), "schedule failed too many times and " + e.getMessage());
                            continue;
                        }
                        tabletCtx.releaseResource(this);
                        this.stat.counterTabletScheduledFailed.incrementAndGet();
                        this.dynamicAdjustPrioAndAddBackToPendingTablets(tabletCtx, e.getMessage());
                        continue;
                    }
                    tabletCtx.releaseResource(this);
                    this.stat.counterTabletScheduledFailed.incrementAndGet();
                    this.dynamicAdjustPrioAndAddBackToPendingTablets(tabletCtx, e.getMessage());
                    continue;
                }
                if (e.getStatus() == SchedException.Status.FINISHED) {
                    this.stat.counterTabletScheduledSucceeded.incrementAndGet();
                    this.finalizeTabletCtx(tabletCtx, TabletSchedCtx.State.FINISHED, e.getStatus(), e.getMessage());
                    continue;
                }
                Preconditions.checkState((e.getStatus() == SchedException.Status.UNRECOVERABLE ? 1 : 0) != 0, (Object)((Object)e.getStatus()));
                this.stat.counterTabletScheduledDiscard.incrementAndGet();
                this.finalizeTabletCtx(tabletCtx, TabletSchedCtx.State.CANCELLED, e.getStatus(), e.getMessage());
                continue;
            }
            catch (Exception e) {
                LOG.warn("got unexpected exception, discard this schedule. tablet: {}", (Object)tabletCtx.getTabletId(), (Object)e);
                this.stat.counterTabletScheduledFailed.incrementAndGet();
                this.finalizeTabletCtx(tabletCtx, TabletSchedCtx.State.UNEXPECTED, SchedException.Status.UNRECOVERABLE, e.getMessage());
                continue;
            }
            Preconditions.checkState((tabletCtx.getState() == TabletSchedCtx.State.RUNNING ? 1 : 0) != 0, (Object)((Object)tabletCtx.getState()));
            this.stat.counterTabletScheduledSucceeded.incrementAndGet();
            this.addToRunningTablets(tabletCtx);
        }
        for (AgentTask task : batchTask.getAllTasks()) {
            if (AgentTaskQueue.addTask(task)) {
                this.stat.counterCloneTask.incrementAndGet();
            }
            LOG.info("add clone task to agent task queue: {}", (Object)task);
        }
        AgentTaskExecutor.submit(batchTask);
        long cost = System.currentTimeMillis() - start;
        this.stat.counterTabletScheduleCostMs.addAndGet(cost);
    }

    private synchronized void addToRunningTablets(TabletSchedCtx tabletCtx) {
        this.runningTablets.put(tabletCtx.getTabletId(), tabletCtx);
    }

    private synchronized TabletSchedCtx takeRunningTablets(long tabletId) {
        return this.runningTablets.remove(tabletId);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void scheduleTablet(TabletSchedCtx tabletCtx, AgentBatchTask batchTask) throws SchedException {
        LOG.debug("schedule tablet: {}, type: {}, status: {}", (Object)tabletCtx.getTabletId(), (Object)tabletCtx.getType(), (Object)tabletCtx.getTabletStatus());
        long currentTime = System.currentTimeMillis();
        tabletCtx.setLastSchedTime(currentTime);
        tabletCtx.setLastVisitedTime(currentTime);
        this.stat.counterTabletScheduled.incrementAndGet();
        Database db = Catalog.getCurrentCatalog().getDbOrException(tabletCtx.getDbId(), s -> new SchedException(SchedException.Status.UNRECOVERABLE, "db " + tabletCtx.getDbId() + " does not exist"));
        OlapTable tbl = (OlapTable)db.getTableOrException(tabletCtx.getTblId(), s -> new SchedException(SchedException.Status.UNRECOVERABLE, "tbl " + tabletCtx.getTblId() + " does not exist"));
        tbl.writeLockOrException(new SchedException(SchedException.Status.UNRECOVERABLE, "table " + tbl.getName() + " does not exist"));
        try {
            Pair<Tablet.TabletStatus, TabletSchedCtx.Priority> statusPair;
            boolean isColocateTable = this.colocateTableIndex.isColocateTable(tbl.getId());
            OlapTable.OlapTableState tableState = tbl.getState();
            Partition partition = tbl.getPartition(tabletCtx.getPartitionId());
            if (partition == null) {
                throw new SchedException(SchedException.Status.UNRECOVERABLE, "partition does not exist");
            }
            MaterializedIndex idx = partition.getIndex(tabletCtx.getIndexId());
            if (idx == null) {
                throw new SchedException(SchedException.Status.UNRECOVERABLE, "index does not exist");
            }
            Tablet tablet = idx.getTablet(tabletCtx.getTabletId());
            Preconditions.checkNotNull((Object)tablet);
            if (isColocateTable) {
                ColocateTableIndex.GroupId groupId = this.colocateTableIndex.getGroup(tbl.getId());
                if (groupId == null) {
                    throw new SchedException(SchedException.Status.UNRECOVERABLE, "colocate group does not exist");
                }
                int tabletOrderIdx = tabletCtx.getTabletOrderIdx();
                if (tabletOrderIdx == -1) {
                    tabletOrderIdx = idx.getTabletOrderIdx(tablet.getId());
                }
                Preconditions.checkState((tabletOrderIdx != -1 ? 1 : 0) != 0);
                Set<Long> backendsSet = this.colocateTableIndex.getTabletBackendsByGroup(groupId, tabletOrderIdx);
                Tablet.TabletStatus st = tablet.getColocateHealthStatus(partition.getVisibleVersion(), tbl.getPartitionInfo().getReplicaAllocation(partition.getId()), backendsSet);
                statusPair = Pair.create(st, TabletSchedCtx.Priority.HIGH);
                tabletCtx.setColocateGroupBackendIds(backendsSet);
            } else {
                List<Long> aliveBeIdsInCluster = this.infoService.getClusterBackendIds(db.getClusterName(), true);
                statusPair = tablet.getHealthStatusWithPriority(this.infoService, tabletCtx.getCluster(), partition.getVisibleVersion(), tbl.getPartitionInfo().getReplicaAllocation(partition.getId()), aliveBeIdsInCluster);
            }
            if (tabletCtx.getType() == TabletSchedCtx.Type.BALANCE && tableState != OlapTable.OlapTableState.NORMAL) {
                throw new SchedException(SchedException.Status.UNRECOVERABLE, "table's state is not NORMAL");
            }
            if (tabletCtx.getType() == TabletSchedCtx.Type.BALANCE) {
                try {
                    DatabaseTransactionMgr dbTransactionMgr = Catalog.getCurrentGlobalTransactionMgr().getDatabaseTransactionMgr(db.getId());
                    for (TransactionState transactionState : dbTransactionMgr.getPreCommittedTxnList()) {
                        if (!transactionState.getTableIdList().contains(tbl.getId())) continue;
                        throw new SchedException(SchedException.Status.UNRECOVERABLE, "There exists PRECOMMITTED transaction related to table");
                    }
                }
                catch (AnalysisException analysisException) {
                    // empty catch block
                }
            }
            if (statusPair.first != Tablet.TabletStatus.VERSION_INCOMPLETE && (partition.getState() != Partition.PartitionState.NORMAL || tableState != OlapTable.OlapTableState.NORMAL) && tableState != OlapTable.OlapTableState.WAITING_STABLE) {
                throw new SchedException(SchedException.Status.UNRECOVERABLE, "table is in alter process, but tablet status is " + ((Tablet.TabletStatus)((Object)statusPair.first)).name());
            }
            tabletCtx.setTabletStatus((Tablet.TabletStatus)((Object)statusPair.first));
            if (statusPair.first == Tablet.TabletStatus.HEALTHY && tabletCtx.getType() == TabletSchedCtx.Type.REPAIR) {
                throw new SchedException(SchedException.Status.UNRECOVERABLE, "tablet is healthy");
            }
            if (statusPair.first != Tablet.TabletStatus.HEALTHY && tabletCtx.getType() == TabletSchedCtx.Type.BALANCE) {
                throw new SchedException(SchedException.Status.UNRECOVERABLE, "tablet is unhealthy when doing balance");
            }
            tabletCtx.setTablet(tablet);
            tabletCtx.setVersionInfo(partition.getVisibleVersion(), partition.getCommittedVersion());
            tabletCtx.setSchemaHash(tbl.getSchemaHashByIndexId(idx.getId()));
            tabletCtx.setStorageMedium(tbl.getPartitionInfo().getDataProperty(partition.getId()).getStorageMedium());
            this.handleTabletByTypeAndStatus((Tablet.TabletStatus)((Object)statusPair.first), tabletCtx, batchTask);
        }
        finally {
            tbl.writeUnlock();
        }
    }

    private void handleTabletByTypeAndStatus(Tablet.TabletStatus status, TabletSchedCtx tabletCtx, AgentBatchTask batchTask) throws SchedException {
        if (tabletCtx.getType() == TabletSchedCtx.Type.REPAIR) {
            switch (status) {
                case REPLICA_MISSING: {
                    this.handleReplicaMissing(tabletCtx, batchTask);
                    break;
                }
                case VERSION_INCOMPLETE: 
                case NEED_FURTHER_REPAIR: {
                    this.handleReplicaVersionIncomplete(tabletCtx, batchTask);
                    break;
                }
                case REPLICA_RELOCATING: {
                    this.handleReplicaRelocating(tabletCtx, batchTask);
                    break;
                }
                case REDUNDANT: {
                    this.handleRedundantReplica(tabletCtx, false);
                    break;
                }
                case FORCE_REDUNDANT: {
                    this.handleRedundantReplica(tabletCtx, true);
                    break;
                }
                case REPLICA_MISSING_IN_CLUSTER: {
                    this.handleReplicaClusterMigration(tabletCtx, batchTask);
                    break;
                }
                case REPLICA_MISSING_FOR_TAG: {
                    this.handleReplicaMissingForTag(tabletCtx, batchTask);
                    break;
                }
                case COLOCATE_MISMATCH: {
                    this.handleColocateMismatch(tabletCtx, batchTask);
                    break;
                }
                case COLOCATE_REDUNDANT: {
                    this.handleColocateRedundant(tabletCtx);
                    break;
                }
                case REPLICA_COMPACTION_TOO_SLOW: {
                    this.handleReplicaTooSlow(tabletCtx);
                    break;
                }
                case UNRECOVERABLE: {
                    throw new SchedException(SchedException.Status.UNRECOVERABLE, "tablet is unrecoverable");
                }
            }
        } else {
            this.doBalance(tabletCtx, batchTask);
        }
    }

    private void handleReplicaMissing(TabletSchedCtx tabletCtx, AgentBatchTask batchTask) throws SchedException {
        this.stat.counterReplicaMissingErr.incrementAndGet();
        if (tabletCtx.compactionRecovered()) {
            return;
        }
        Tag tag = this.chooseProperTag(tabletCtx, true);
        RootPathLoadStatistic destPath = this.chooseAvailableDestPath(tabletCtx, tag, false);
        Preconditions.checkNotNull((Object)destPath);
        tabletCtx.setDest(destPath.getBeId(), destPath.getPathHash());
        tabletCtx.chooseSrcReplica(this.backendsWorkingSlots, -1L);
        batchTask.addTask(tabletCtx.createCloneReplicaAndTask());
    }

    private Tag chooseProperTag(TabletSchedCtx tabletCtx, boolean forMissingReplica) throws SchedException {
        Tablet tablet = tabletCtx.getTablet();
        List<Replica> replicas = tablet.getReplicas();
        Map<Tag, Short> allocMap = tabletCtx.getReplicaAlloc().getAllocMap();
        HashMap currentAllocMap = Maps.newHashMap();
        for (Replica replica : replicas) {
            Backend be = this.infoService.getBackend(replica.getBackendId());
            if (be == null || !be.isScheduleAvailable() || !replica.isAlive() || replica.tooSlow()) continue;
            Short num = currentAllocMap.getOrDefault(be.getTag(), (short)0);
            currentAllocMap.put(be.getTag(), (short)(num + 1));
        }
        for (Map.Entry entry : allocMap.entrySet()) {
            short curNum = currentAllocMap.getOrDefault(entry.getKey(), (short)0);
            if (forMissingReplica && curNum < (Short)entry.getValue()) {
                return (Tag)entry.getKey();
            }
            if (forMissingReplica || curNum <= (Short)entry.getValue()) continue;
            return (Tag)entry.getKey();
        }
        throw new SchedException(SchedException.Status.UNRECOVERABLE, "no proper tag is chose for tablet " + tablet.getId());
    }

    private void handleReplicaVersionIncomplete(TabletSchedCtx tabletCtx, AgentBatchTask batchTask) throws SchedException {
        this.stat.counterReplicaVersionMissingErr.incrementAndGet();
        try {
            tabletCtx.chooseDestReplicaForVersionIncomplete(this.backendsWorkingSlots);
        }
        catch (SchedException e) {
            if (e.getMessage().equals("unable to choose dest replica")) {
                LOG.debug("failed to find version incomplete replica for VERSION_INCOMPLETE task. tablet id: {}, try to find a new backend", (Object)tabletCtx.getTabletId());
                tabletCtx.releaseResource(this, true);
                tabletCtx.setTabletStatus(Tablet.TabletStatus.REPLICA_MISSING);
                this.handleReplicaMissing(tabletCtx, batchTask);
                LOG.debug("succeed to find new backend for VERSION_INCOMPLETE task. tablet id: {}", (Object)tabletCtx.getTabletId());
                return;
            }
            throw e;
        }
        tabletCtx.chooseSrcReplicaForVersionIncomplete(this.backendsWorkingSlots);
        batchTask.addTask(tabletCtx.createCloneReplicaAndTask());
    }

    private void handleReplicaRelocating(TabletSchedCtx tabletCtx, AgentBatchTask batchTask) throws SchedException {
        this.stat.counterReplicaUnavailableErr.incrementAndGet();
        try {
            this.handleReplicaVersionIncomplete(tabletCtx, batchTask);
            LOG.debug("succeed to find version incomplete replica from tablet relocating. tablet id: {}", (Object)tabletCtx.getTabletId());
        }
        catch (SchedException e) {
            if (e.getStatus() == SchedException.Status.SCHEDULE_FAILED) {
                LOG.debug("failed to find version incomplete replica from tablet relocating. tablet id: {}, try to find a new backend", (Object)tabletCtx.getTabletId());
                tabletCtx.releaseResource(this, true);
                tabletCtx.setTabletStatus(Tablet.TabletStatus.REPLICA_MISSING);
                this.handleReplicaMissing(tabletCtx, batchTask);
                LOG.debug("succeed to find new backend for tablet relocating. tablet id: {}", (Object)tabletCtx.getTabletId());
            }
            throw e;
        }
    }

    private void handleRedundantReplica(TabletSchedCtx tabletCtx, boolean force) throws SchedException {
        this.stat.counterReplicaRedundantErr.incrementAndGet();
        if (this.deleteBackendDropped(tabletCtx, force) || this.deleteBadReplica(tabletCtx, force) || this.deleteBackendUnavailable(tabletCtx, force) || this.deleteTooSlowReplica(tabletCtx, force) || this.deleteCloneOrDecommissionReplica(tabletCtx, force) || this.deleteReplicaWithFailedVersion(tabletCtx, force) || this.deleteReplicaWithLowerVersion(tabletCtx, force) || this.deleteReplicaOnSameHost(tabletCtx, force) || this.deleteReplicaNotInCluster(tabletCtx, force) || this.deleteReplicaNotInValidTag(tabletCtx, force) || this.deleteReplicaChosenByRebalancer(tabletCtx, force) || this.deleteReplicaOnHighLoadBackend(tabletCtx, force)) {
            throw new SchedException(SchedException.Status.FINISHED, "redundant replica is deleted");
        }
        throw new SchedException(SchedException.Status.SCHEDULE_FAILED, "unable to delete any redundant replicas");
    }

    private boolean deleteBackendDropped(TabletSchedCtx tabletCtx, boolean force) throws SchedException {
        for (Replica replica : tabletCtx.getReplicas()) {
            long beId = replica.getBackendId();
            if (this.infoService.getBackend(beId) != null) continue;
            this.deleteReplicaInternal(tabletCtx, replica, "backend dropped", force);
            return true;
        }
        return false;
    }

    private boolean deleteBadReplica(TabletSchedCtx tabletCtx, boolean force) throws SchedException {
        for (Replica replica : tabletCtx.getReplicas()) {
            if (!replica.isBad()) continue;
            this.deleteReplicaInternal(tabletCtx, replica, "replica is bad", force);
            return true;
        }
        return false;
    }

    private boolean deleteTooSlowReplica(TabletSchedCtx tabletCtx, boolean force) throws SchedException {
        for (Replica replica : tabletCtx.getReplicas()) {
            if (!replica.tooSlow()) continue;
            this.deleteReplicaInternal(tabletCtx, replica, "replica is too slow", force);
            return true;
        }
        return false;
    }

    private boolean deleteBackendUnavailable(TabletSchedCtx tabletCtx, boolean force) throws SchedException {
        for (Replica replica : tabletCtx.getReplicas()) {
            Backend be = this.infoService.getBackend(replica.getBackendId());
            if (be == null || be.isScheduleAvailable()) continue;
            this.deleteReplicaInternal(tabletCtx, replica, "backend unavailable", force);
            return true;
        }
        return false;
    }

    private boolean deleteCloneOrDecommissionReplica(TabletSchedCtx tabletCtx, boolean force) throws SchedException {
        for (Replica replica : tabletCtx.getReplicas()) {
            if (replica.getState() != Replica.ReplicaState.CLONE && replica.getState() != Replica.ReplicaState.DECOMMISSION) continue;
            this.deleteReplicaInternal(tabletCtx, replica, (Object)((Object)replica.getState()) + " state", force);
            return true;
        }
        return false;
    }

    private boolean deleteReplicaWithFailedVersion(TabletSchedCtx tabletCtx, boolean force) throws SchedException {
        for (Replica replica : tabletCtx.getReplicas()) {
            if (replica.getLastFailedVersion() <= 0L) continue;
            this.deleteReplicaInternal(tabletCtx, replica, "version incomplete", force);
            return true;
        }
        return false;
    }

    private boolean deleteReplicaWithLowerVersion(TabletSchedCtx tabletCtx, boolean force) throws SchedException {
        for (Replica replica : tabletCtx.getReplicas()) {
            if (replica.checkVersionCatchUp(tabletCtx.getCommittedVersion(), false)) continue;
            this.deleteReplicaInternal(tabletCtx, replica, "lower version", force);
            return true;
        }
        return false;
    }

    private boolean deleteReplicaOnSameHost(TabletSchedCtx tabletCtx, boolean force) throws SchedException {
        HashMap hostToReplicas = Maps.newHashMap();
        for (Replica replica : tabletCtx.getReplicas()) {
            Backend be = this.infoService.getBackend(replica.getBackendId());
            if (be == null) {
                return false;
            }
            List replicas = (List)hostToReplicas.get(be.getHost());
            if (replicas == null) {
                replicas = Lists.newArrayList();
                hostToReplicas.put(be.getHost(), replicas);
            }
            replicas.add(replica);
        }
        for (List replicas : hostToReplicas.values()) {
            if (replicas.size() <= 1) continue;
            Tag tag = this.chooseProperTag(tabletCtx, false);
            ClusterLoadStatistic statistic = (ClusterLoadStatistic)this.statisticMap.get((Object)tabletCtx.getCluster(), (Object)tag);
            if (statistic == null) {
                return false;
            }
            return this.deleteFromHighLoadBackend(tabletCtx, replicas, force, statistic);
        }
        return false;
    }

    private boolean deleteReplicaNotInCluster(TabletSchedCtx tabletCtx, boolean force) throws SchedException {
        for (Replica replica : tabletCtx.getReplicas()) {
            Backend be = this.infoService.getBackend(replica.getBackendId());
            if (be == null) {
                return false;
            }
            if (be.getOwnerClusterName().equals(tabletCtx.getCluster())) continue;
            this.deleteReplicaInternal(tabletCtx, replica, "not in cluster", force);
            return true;
        }
        return false;
    }

    private boolean deleteReplicaNotInValidTag(TabletSchedCtx tabletCtx, boolean force) throws SchedException {
        Tablet tablet = tabletCtx.getTablet();
        List<Replica> replicas = tablet.getReplicas();
        Map<Tag, Short> allocMap = tabletCtx.getReplicaAlloc().getAllocMap();
        for (Replica replica : replicas) {
            Backend be = this.infoService.getBackend(replica.getBackendId());
            if (allocMap.containsKey(be.getTag())) continue;
            this.deleteReplicaInternal(tabletCtx, replica, "not in valid tag", force);
            return true;
        }
        return false;
    }

    private boolean deleteReplicaChosenByRebalancer(TabletSchedCtx tabletCtx, boolean force) throws SchedException {
        Long id = this.rebalancer.getToDeleteReplicaId(tabletCtx);
        if (id == -1L) {
            return false;
        }
        Replica chosenReplica = tabletCtx.getTablet().getReplicaById(id);
        if (chosenReplica != null) {
            this.deleteReplicaInternal(tabletCtx, chosenReplica, "src replica of rebalance", force);
            return true;
        }
        return false;
    }

    private boolean deleteReplicaOnHighLoadBackend(TabletSchedCtx tabletCtx, boolean force) throws SchedException {
        Tag tag = this.chooseProperTag(tabletCtx, false);
        ClusterLoadStatistic statistic = (ClusterLoadStatistic)this.statisticMap.get((Object)tabletCtx.getCluster(), (Object)tag);
        if (statistic == null) {
            return false;
        }
        return this.deleteFromHighLoadBackend(tabletCtx, tabletCtx.getReplicas(), force, statistic);
    }

    private boolean deleteFromHighLoadBackend(TabletSchedCtx tabletCtx, List<Replica> replicas, boolean force, ClusterLoadStatistic statistic) throws SchedException {
        Replica chosenReplica = null;
        double maxScore = 0.0;
        for (Replica replica : replicas) {
            BackendLoadStatistic beStatistic = statistic.getBackendLoadStatistic(replica.getBackendId());
            if (beStatistic == null) continue;
            double loadScore = 0.0;
            loadScore = beStatistic.hasMedium(tabletCtx.getStorageMedium()) ? beStatistic.getLoadScore(tabletCtx.getStorageMedium()) : beStatistic.getMixLoadScore();
            if (!(loadScore > maxScore)) continue;
            maxScore = loadScore;
            chosenReplica = replica;
        }
        if (chosenReplica != null) {
            this.deleteReplicaInternal(tabletCtx, chosenReplica, "high load", force);
            return true;
        }
        return false;
    }

    private boolean handleColocateRedundant(TabletSchedCtx tabletCtx) throws SchedException {
        Preconditions.checkNotNull(tabletCtx.getColocateBackendsSet());
        for (Replica replica : tabletCtx.getReplicas()) {
            if (tabletCtx.getColocateBackendsSet().contains(replica.getBackendId()) && !replica.isBad()) continue;
            this.deleteReplicaInternal(tabletCtx, replica, "colocate redundant", false);
            throw new SchedException(SchedException.Status.FINISHED, "colocate redundant replica is deleted");
        }
        throw new SchedException(SchedException.Status.SCHEDULE_FAILED, "unable to delete any colocate redundant replicas");
    }

    private void handleReplicaTooSlow(TabletSchedCtx tabletCtx) throws SchedException {
        Replica chosenReplica = null;
        Replica minReplica = null;
        long maxVersionCount = -1L;
        long minVersionCount = Integer.MAX_VALUE;
        int normalReplicaCount = 0;
        for (Replica replica : tabletCtx.getReplicas()) {
            if (replica.isAlive() && !replica.tooSlow()) {
                ++normalReplicaCount;
            }
            if (replica.getVersionCount() > maxVersionCount) {
                maxVersionCount = replica.getVersionCount();
                chosenReplica = replica;
            }
            if (replica.getVersionCount() >= minVersionCount) continue;
            minVersionCount = replica.getVersionCount();
            minReplica = replica;
        }
        if (chosenReplica != null && !chosenReplica.equals(minReplica) && minReplica.isAlive() && !minReplica.tooSlow() && normalReplicaCount >= 1) {
            chosenReplica.setState(Replica.ReplicaState.COMPACTION_TOO_SLOW);
            LOG.info("set replica id :{} tablet id: {}, backend id: {} to COMPACTION_TOO_SLOW", (Object)chosenReplica.getId(), (Object)tabletCtx.getTablet().getId(), (Object)chosenReplica.getBackendId());
            throw new SchedException(SchedException.Status.FINISHED, "set replica to COMPACTION_TOO_SLOW");
        }
        throw new SchedException(SchedException.Status.FINISHED, "No replica too slow");
    }

    private void deleteReplicaInternal(TabletSchedCtx tabletCtx, Replica replica, String reason, boolean force) throws SchedException {
        if (!force && !Config.enable_force_drop_redundant_replica && replica.getState().canLoad() && replica.getWatermarkTxnId() == -1L && !FeConstants.runningUnitTest) {
            long nextTxnId = Catalog.getCurrentGlobalTransactionMgr().getTransactionIDGenerator().getNextTransactionId();
            replica.setWatermarkTxnId(nextTxnId);
            replica.setState(Replica.ReplicaState.DECOMMISSION);
            tabletCtx.setOrigPriority(TabletSchedCtx.Priority.NORMAL);
            LOG.debug("set replica {} on backend {} of tablet {} state to DECOMMISSION", (Object)replica.getId(), (Object)replica.getBackendId(), (Object)tabletCtx.getTabletId());
            throw new SchedException(SchedException.Status.SCHEDULE_FAILED, "set watermark txn " + nextTxnId);
        }
        if (replica.getState() == Replica.ReplicaState.DECOMMISSION && replica.getWatermarkTxnId() != -1L) {
            long watermarkTxnId = replica.getWatermarkTxnId();
            try {
                if (!Catalog.getCurrentGlobalTransactionMgr().isPreviousTransactionsFinished(watermarkTxnId, tabletCtx.getDbId(), Lists.newArrayList((Object[])new Long[]{tabletCtx.getTblId()}))) {
                    throw new SchedException(SchedException.Status.SCHEDULE_FAILED, "wait txn before " + watermarkTxnId + " to be finished");
                }
            }
            catch (AnalysisException e) {
                throw new SchedException(SchedException.Status.UNRECOVERABLE, e.getMessage());
            }
        }
        tabletCtx.deleteReplica(replica);
        if (force) {
            this.sendDeleteReplicaTask(replica.getBackendId(), tabletCtx.getTabletId(), tabletCtx.getSchemaHash());
        }
        ReplicaPersistInfo info = ReplicaPersistInfo.createForDelete(tabletCtx.getDbId(), tabletCtx.getTblId(), tabletCtx.getPartitionId(), tabletCtx.getIndexId(), tabletCtx.getTabletId(), replica.getBackendId());
        Catalog.getCurrentCatalog().getEditLog().logDeleteReplica(info);
        LOG.info("delete replica. tablet id: {}, backend id: {}. reason: {}, force: {}", (Object)tabletCtx.getTabletId(), (Object)replica.getBackendId(), (Object)reason, (Object)force);
    }

    private void sendDeleteReplicaTask(long backendId, long tabletId, int schemaHash) {
        DropReplicaTask task = new DropReplicaTask(backendId, tabletId, schemaHash);
        AgentBatchTask batchTask = new AgentBatchTask();
        batchTask.addTask(task);
        AgentTaskExecutor.submit(batchTask);
        LOG.info("send delete replica task for tablet {} in backend {}", (Object)tabletId, (Object)backendId);
    }

    private void handleReplicaClusterMigration(TabletSchedCtx tabletCtx, AgentBatchTask batchTask) throws SchedException {
        this.stat.counterReplicaMissingInClusterErr.incrementAndGet();
        this.handleReplicaMissing(tabletCtx, batchTask);
    }

    private void handleReplicaMissingForTag(TabletSchedCtx tabletCtx, AgentBatchTask batchTask) throws SchedException {
        this.stat.counterReplicaMissingForTagErr.incrementAndGet();
        this.handleReplicaMissing(tabletCtx, batchTask);
    }

    private void handleColocateMismatch(TabletSchedCtx tabletCtx, AgentBatchTask batchTask) throws SchedException {
        Preconditions.checkNotNull(tabletCtx.getColocateBackendsSet());
        this.stat.counterReplicaColocateMismatch.incrementAndGet();
        RootPathLoadStatistic destPath = this.chooseAvailableDestPath(tabletCtx, null, true);
        Preconditions.checkNotNull((Object)destPath);
        tabletCtx.setDest(destPath.getBeId(), destPath.getPathHash());
        tabletCtx.chooseSrcReplica(this.backendsWorkingSlots, -1L);
        batchTask.addTask(tabletCtx.createCloneReplicaAndTask());
    }

    private void selectTabletsForBalance() {
        if (Config.disable_balance || Config.disable_tablet_scheduler) {
            LOG.info("balance or tablet scheduler is disabled. skip selecting tablets for balance");
            return;
        }
        long numOfBalancingTablets = this.getBalanceTabletsNumber();
        if (numOfBalancingTablets > (long)Config.max_balancing_tablets) {
            LOG.info("number of balancing tablets {} exceed limit: {}, skip selecting tablets for balance", (Object)numOfBalancingTablets, (Object)Config.max_balancing_tablets);
            return;
        }
        List<TabletSchedCtx> alternativeTablets = this.rebalancer.selectAlternativeTablets();
        for (TabletSchedCtx tabletCtx : alternativeTablets) {
            this.addTablet(tabletCtx, false);
        }
    }

    private void doBalance(TabletSchedCtx tabletCtx, AgentBatchTask batchTask) throws SchedException {
        this.stat.counterBalanceSchedule.incrementAndGet();
        this.rebalancer.createBalanceTask(tabletCtx, this.backendsWorkingSlots, batchTask);
    }

    private RootPathLoadStatistic chooseAvailableDestPath(TabletSchedCtx tabletCtx, Tag tag, boolean forColocate) throws SchedException {
        PathSlot slot;
        List<Object> beStatistics;
        if (tag != null) {
            Preconditions.checkState((!forColocate ? 1 : 0) != 0);
            ClusterLoadStatistic statistic = (ClusterLoadStatistic)this.statisticMap.get((Object)tabletCtx.getCluster(), (Object)tag);
            if (statistic == null) {
                throw new SchedException(SchedException.Status.UNRECOVERABLE, "cluster does not exist");
            }
            beStatistics = statistic.getSortedBeLoadStats(null);
        } else {
            Preconditions.checkState((boolean)forColocate);
            Preconditions.checkState((tabletCtx.getColocateBackendsSet() != null ? 1 : 0) != 0);
            Set<Long> colocateBackendIds = tabletCtx.getColocateBackendsSet();
            beStatistics = Lists.newArrayList();
            Map map = this.statisticMap.row((Object)tabletCtx.getCluster());
            for (ClusterLoadStatistic clusterStatistic : map.values()) {
                for (long beId : colocateBackendIds) {
                    BackendLoadStatistic backendLoadStatistic = clusterStatistic.getBackendLoadStatistic(beId);
                    if (backendLoadStatistic == null) continue;
                    beStatistics.add(backendLoadStatistic);
                }
            }
        }
        ArrayList allFitPaths = Lists.newArrayList();
        for (BackendLoadStatistic backendLoadStatistic : beStatistics) {
            if (!backendLoadStatistic.isAvailable() || tabletCtx.containsBE(backendLoadStatistic.getBeId()) || (forColocate ? !tabletCtx.getColocateBackendsSet().contains(backendLoadStatistic.getBeId()) : !backendLoadStatistic.getTag().equals(tag))) continue;
            ArrayList resultPaths = Lists.newArrayList();
            BalanceStatus st = backendLoadStatistic.isFit(tabletCtx.getTabletSize(), tabletCtx.getStorageMedium(), resultPaths, tabletCtx.getTabletStatus() != Tablet.TabletStatus.REPLICA_RELOCATING);
            if (!st.ok() && !(st = backendLoadStatistic.isFit(tabletCtx.getTabletSize(), tabletCtx.getStorageMedium(), resultPaths, true)).ok()) {
                LOG.debug("unable to find path for supplementing tablet: {}. {}", (Object)tabletCtx, (Object)st);
                continue;
            }
            Preconditions.checkState((resultPaths.size() == 1 ? 1 : 0) != 0);
            allFitPaths.add((RootPathLoadStatistic)resultPaths.get(0));
        }
        if (allFitPaths.isEmpty()) {
            throw new SchedException(SchedException.Status.SCHEDULE_FAILED, "unable to find dest path for new replica");
        }
        for (RootPathLoadStatistic rootPathLoadStatistic : allFitPaths) {
            long pathHash;
            if (rootPathLoadStatistic.getStorageMedium() != tabletCtx.getStorageMedium() || (slot = this.backendsWorkingSlots.get(rootPathLoadStatistic.getBeId())) == null || (pathHash = slot.takeSlot(rootPathLoadStatistic.getPathHash())) == -1L) continue;
            return rootPathLoadStatistic;
        }
        for (RootPathLoadStatistic rootPathLoadStatistic : allFitPaths) {
            long pathHash;
            slot = this.backendsWorkingSlots.get(rootPathLoadStatistic.getBeId());
            if (slot == null || (pathHash = slot.takeSlot(rootPathLoadStatistic.getPathHash())) == -1L) continue;
            return rootPathLoadStatistic;
        }
        throw new SchedException(SchedException.Status.SCHEDULE_FAILED, "unable to find dest path which can be fit in");
    }

    private void dynamicAdjustPrioAndAddBackToPendingTablets(TabletSchedCtx tabletCtx, String message) {
        Preconditions.checkState((tabletCtx.getState() == TabletSchedCtx.State.PENDING ? 1 : 0) != 0);
        tabletCtx.adjustPriority(this.stat);
        this.addTablet(tabletCtx, true);
    }

    private void finalizeTabletCtx(TabletSchedCtx tabletCtx, TabletSchedCtx.State state, SchedException.Status status, String reason) {
        this.removeTabletCtx(tabletCtx, reason);
        this.releaseTabletCtx(tabletCtx, state, status == SchedException.Status.UNRECOVERABLE);
    }

    private void releaseTabletCtx(TabletSchedCtx tabletCtx, TabletSchedCtx.State state, boolean resetReplicaState) {
        tabletCtx.setState(state);
        tabletCtx.releaseResource(this);
        if (resetReplicaState) {
            tabletCtx.resetReplicaState();
        }
        tabletCtx.setFinishedTime(System.currentTimeMillis());
    }

    private synchronized void removeTabletCtx(TabletSchedCtx tabletCtx, String reason) {
        this.runningTablets.remove(tabletCtx.getTabletId());
        this.allTabletIds.remove(tabletCtx.getTabletId());
        this.schedHistory.add(tabletCtx);
        LOG.info("remove the tablet {}. because: {}", (Object)tabletCtx.getTabletId(), (Object)reason);
    }

    private synchronized List<TabletSchedCtx> getNextTabletCtxBatch() {
        TabletSchedCtx tablet;
        ArrayList list = Lists.newArrayList();
        for (int count = Math.max(50, this.getCurrentAvailableSlotNum()); count > 0 && (tablet = this.pendingTablets.poll()) != null; --count) {
            list.add(tablet);
        }
        return list;
    }

    private int getCurrentAvailableSlotNum() {
        int total = 0;
        for (PathSlot pathSlot : this.backendsWorkingSlots.values()) {
            total += pathSlot.getTotalAvailSlotNum();
        }
        return total;
    }

    public boolean finishCloneTask(CloneTask cloneTask, TFinishTaskRequest request) {
        long tabletId = cloneTask.getTabletId();
        TabletSchedCtx tabletCtx = this.takeRunningTablets(tabletId);
        if (tabletCtx == null) {
            LOG.warn("tablet info does not exist: {}", (Object)tabletId);
            return true;
        }
        Preconditions.checkState((tabletCtx.getState() == TabletSchedCtx.State.RUNNING ? 1 : 0) != 0, (Object)((Object)tabletCtx.getState()));
        try {
            tabletCtx.finishCloneTask(cloneTask, request);
        }
        catch (SchedException e) {
            tabletCtx.increaseFailedRunningCounter();
            tabletCtx.setErrMsg(e.getMessage());
            if (e.getStatus() == SchedException.Status.RUNNING_FAILED) {
                this.stat.counterCloneTaskFailed.incrementAndGet();
                this.addToRunningTablets(tabletCtx);
                return false;
            }
            if (e.getStatus() == SchedException.Status.UNRECOVERABLE) {
                this.stat.counterTabletScheduledDiscard.incrementAndGet();
                this.finalizeTabletCtx(tabletCtx, TabletSchedCtx.State.CANCELLED, e.getStatus(), e.getMessage());
                return true;
            }
            if (e.getStatus() == SchedException.Status.FINISHED) {
                this.finalizeTabletCtx(tabletCtx, TabletSchedCtx.State.CANCELLED, e.getStatus(), e.getMessage());
                return true;
            }
        }
        catch (Exception e) {
            LOG.warn("got unexpected exception when finish clone task. tablet: {}", (Object)tabletCtx.getTabletId(), (Object)e);
            this.stat.counterTabletScheduledDiscard.incrementAndGet();
            this.finalizeTabletCtx(tabletCtx, TabletSchedCtx.State.UNEXPECTED, SchedException.Status.UNRECOVERABLE, e.getMessage());
            return true;
        }
        Preconditions.checkState((tabletCtx.getState() == TabletSchedCtx.State.FINISHED ? 1 : 0) != 0);
        this.stat.counterCloneTaskSucceeded.incrementAndGet();
        this.gatherStatistics(tabletCtx);
        this.finalizeTabletCtx(tabletCtx, TabletSchedCtx.State.FINISHED, SchedException.Status.FINISHED, "finished");
        return true;
    }

    private void gatherStatistics(TabletSchedCtx tabletCtx) {
        if (tabletCtx.getCopySize() > 0L && tabletCtx.getCopyTimeMs() > 0L) {
            PathSlot pathSlot;
            if (tabletCtx.getSrcBackendId() != -1L && tabletCtx.getSrcPathHash() != -1L && (pathSlot = this.backendsWorkingSlots.get(tabletCtx.getSrcBackendId())) != null) {
                pathSlot.updateStatistic(tabletCtx.getSrcPathHash(), tabletCtx.getCopySize(), tabletCtx.getCopyTimeMs());
            }
            if (tabletCtx.getDestBackendId() != -1L && tabletCtx.getDestPathHash() != -1L && (pathSlot = this.backendsWorkingSlots.get(tabletCtx.getDestBackendId())) != null) {
                pathSlot.updateStatistic(tabletCtx.getDestPathHash(), tabletCtx.getCopySize(), tabletCtx.getCopyTimeMs());
            }
        }
        if (System.currentTimeMillis() - this.lastSlotAdjustTime < 20000L) {
            return;
        }
        this.lastSlotAdjustTime = System.currentTimeMillis();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void handleRunningTablets() {
        ArrayList timeoutTablets = Lists.newArrayList();
        TabletScheduler tabletScheduler = this;
        synchronized (tabletScheduler) {
            this.runningTablets.values().stream().filter(TabletSchedCtx::isTimeout).forEach(timeoutTablets::add);
            for (TabletSchedCtx tabletSchedCtx : timeoutTablets) {
                this.removeTabletCtx(tabletSchedCtx, "timeout");
            }
        }
        timeoutTablets.stream().forEach(t -> {
            this.releaseTabletCtx((TabletSchedCtx)t, TabletSchedCtx.State.CANCELLED, true);
            this.stat.counterCloneTaskTimeout.incrementAndGet();
        });
    }

    public List<List<String>> getPendingTabletsInfo(int limit) {
        List<TabletSchedCtx> tabletCtxs = this.getCopiedTablets(this.pendingTablets, limit);
        return this.collectTabletCtx(tabletCtxs);
    }

    public List<List<String>> getRunningTabletsInfo(int limit) {
        List<TabletSchedCtx> tabletCtxs = this.getCopiedTablets(this.runningTablets.values(), limit);
        return this.collectTabletCtx(tabletCtxs);
    }

    public List<List<String>> getHistoryTabletsInfo(int limit) {
        List<TabletSchedCtx> tabletCtxs = this.getCopiedTablets(this.schedHistory, limit);
        return this.collectTabletCtx(tabletCtxs);
    }

    private List<List<String>> collectTabletCtx(List<TabletSchedCtx> tabletCtxs) {
        ArrayList result = Lists.newArrayList();
        tabletCtxs.stream().forEach(t -> result.add(t.getBrief()));
        return result;
    }

    private synchronized List<TabletSchedCtx> getCopiedTablets(Collection<TabletSchedCtx> source, int limit) {
        ArrayList tabletCtxs = Lists.newArrayList();
        source.stream().limit(limit).forEach(t -> tabletCtxs.add(t));
        return tabletCtxs;
    }

    public synchronized int getPendingNum() {
        return this.pendingTablets.size();
    }

    public synchronized int getRunningNum() {
        return this.runningTablets.size();
    }

    public synchronized int getHistoryNum() {
        return this.schedHistory.size();
    }

    public synchronized int getTotalNum() {
        return this.allTabletIds.size();
    }

    public synchronized long getBalanceTabletsNumber() {
        return this.pendingTablets.stream().filter(t -> t.getType() == TabletSchedCtx.Type.BALANCE).count() + this.runningTablets.values().stream().filter(t -> t.getType() == TabletSchedCtx.Type.BALANCE).count();
    }

    public List<List<String>> getSlotsInfo() {
        ArrayList result = Lists.newArrayList();
        for (long beId : this.backendsWorkingSlots.keySet()) {
            PathSlot slot = this.backendsWorkingSlots.get(beId);
            result.addAll(slot.getSlotInfo(beId));
        }
        return result;
    }

    public static class Slot {
        public int total;
        public int available;
        public int balanceSlot;
        public long totalCopySize = 0L;
        public long totalCopyTimeMs = 0L;

        public Slot(int total) {
            this.total = total;
            this.available = total;
            this.balanceSlot = 2;
        }

        public void rectify() {
            if (this.total <= 0) {
                this.total = 1;
            }
            if (this.available > this.total) {
                this.available = this.total;
            }
            if (this.balanceSlot > 2) {
                this.balanceSlot = 2;
            }
        }

        public double getAvgRate() {
            if (this.totalCopyTimeMs / 1000L == 0L) {
                return 0.0;
            }
            return (double)this.totalCopySize / ((double)this.totalCopyTimeMs / 1000.0);
        }
    }

    public static class PathSlot {
        private Map<Long, Slot> pathSlots = Maps.newConcurrentMap();

        public PathSlot(List<Long> paths, int initSlotNum) {
            for (Long pathHash : paths) {
                this.pathSlots.put(pathHash, new Slot(initSlotNum));
            }
        }

        public synchronized void updatePaths(List<Long> paths) {
            this.pathSlots.entrySet().removeIf(entry -> !paths.contains(entry.getKey()));
            for (Long pathHash : paths) {
                if (this.pathSlots.containsKey(pathHash)) continue;
                this.pathSlots.put(pathHash, new Slot(Config.schedule_slot_num_per_path));
            }
        }

        public synchronized void updateSlot(List<Long> pathHashs, int delta) {
            for (Long pathHash : pathHashs) {
                Slot slot = this.pathSlots.get(pathHash);
                if (slot == null) continue;
                slot.total += delta;
                slot.rectify();
                LOG.debug("decrease path {} slots num to {}", (Object)pathHash, (Object)this.pathSlots.get((Object)pathHash).total);
            }
        }

        public synchronized void updateStatistic(long pathHash, long copySize, long copyTimeMs) {
            Slot slot = this.pathSlots.get(pathHash);
            if (slot == null) {
                return;
            }
            slot.totalCopySize += copySize;
            slot.totalCopyTimeMs += copyTimeMs;
        }

        public synchronized long takeSlot(long pathHash) throws SchedException {
            if (pathHash == -1L) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("path hash is not set.", (Throwable)new Exception());
                }
                throw new SchedException(SchedException.Status.SCHEDULE_FAILED, "path hash is not set");
            }
            Slot slot = this.pathSlots.get(pathHash);
            if (slot == null) {
                return -1L;
            }
            slot.rectify();
            if (slot.available <= 0) {
                return -1L;
            }
            --slot.available;
            return pathHash;
        }

        public synchronized void freeSlot(long pathHash) {
            Slot slot = this.pathSlots.get(pathHash);
            if (slot == null) {
                return;
            }
            ++slot.available;
            slot.rectify();
        }

        public synchronized int peekSlot(long pathHash) {
            Slot slot = this.pathSlots.get(pathHash);
            if (slot == null) {
                return -1;
            }
            slot.rectify();
            return slot.available;
        }

        public synchronized int getTotalAvailSlotNum() {
            int total = 0;
            for (Slot slot : this.pathSlots.values()) {
                total += slot.available;
            }
            return total;
        }

        public synchronized Set<Long> getAvailPathsForBalance() {
            HashSet pathHashs = Sets.newHashSet();
            for (Map.Entry<Long, Slot> entry : this.pathSlots.entrySet()) {
                if (entry.getValue().balanceSlot <= 0) continue;
                pathHashs.add(entry.getKey());
            }
            return pathHashs;
        }

        public synchronized int getAvailBalanceSlotNum() {
            int num = 0;
            for (Map.Entry<Long, Slot> entry : this.pathSlots.entrySet()) {
                num += entry.getValue().balanceSlot;
            }
            return num;
        }

        public synchronized List<List<String>> getSlotInfo(long beId) {
            ArrayList results = Lists.newArrayList();
            this.pathSlots.forEach((key, value) -> {
                value.rectify();
                ArrayList result = Lists.newArrayList();
                result.add(String.valueOf(beId));
                result.add(String.valueOf(key));
                result.add(String.valueOf(value.available));
                result.add(String.valueOf(value.total));
                result.add(String.valueOf(value.balanceSlot));
                result.add(String.valueOf(value.getAvgRate()));
                results.add(result);
            });
            return results;
        }

        public synchronized long takeBalanceSlot(long pathHash) {
            Slot slot = this.pathSlots.get(pathHash);
            if (slot == null) {
                return -1L;
            }
            if (slot.balanceSlot > 0) {
                --slot.balanceSlot;
                return pathHash;
            }
            return -1L;
        }

        public synchronized long takeAnAvailBalanceSlotFrom(Set<Long> pathHashs) {
            for (Long pathHash : pathHashs) {
                Slot slot = this.pathSlots.get(pathHash);
                if (slot == null || slot.balanceSlot <= 0) continue;
                --slot.balanceSlot;
                return pathHash;
            }
            return -1L;
        }

        public synchronized void freeBalanceSlot(long pathHash) {
            Slot slot = this.pathSlots.get(pathHash);
            if (slot == null) {
                return;
            }
            ++slot.balanceSlot;
            slot.rectify();
        }
    }

    public static enum AddResult {
        ADDED,
        ALREADY_IN,
        LIMIT_EXCEED,
        DISABLED;

    }
}

