/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.clone;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.doris.catalog.DiskInfo;
import org.apache.doris.catalog.TabletInvertedIndex;
import org.apache.doris.clone.BalanceStatus;
import org.apache.doris.clone.LoadBalanceException;
import org.apache.doris.clone.RootPathLoadStatistic;
import org.apache.doris.common.Config;
import org.apache.doris.common.util.DebugUtil;
import org.apache.doris.resource.Tag;
import org.apache.doris.system.Backend;
import org.apache.doris.system.SystemInfoService;
import org.apache.doris.thrift.TStorageMedium;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class BackendLoadStatistic {
    private static final Logger LOG = LogManager.getLogger(BackendLoadStatistic.class);
    public static final BeStatComparator HDD_COMPARATOR = new BeStatComparator(TStorageMedium.HDD);
    public static final BeStatComparator SSD_COMPARATOR = new BeStatComparator(TStorageMedium.SSD);
    public static final BeStatMixComparator MIX_COMPARATOR = new BeStatMixComparator();
    private SystemInfoService infoService;
    private TabletInvertedIndex invertedIndex;
    private long beId;
    private String clusterName;
    private boolean isAvailable;
    private Tag tag;
    private Map<TStorageMedium, Long> totalCapacityMap = Maps.newHashMap();
    private Map<TStorageMedium, Long> totalUsedCapacityMap = Maps.newHashMap();
    private Map<TStorageMedium, Long> totalReplicaNumMap = Maps.newHashMap();
    private Map<TStorageMedium, LoadScore> loadScoreMap = Maps.newHashMap();
    private Map<TStorageMedium, Classification> clazzMap = Maps.newHashMap();
    private List<RootPathLoadStatistic> pathStatistics = Lists.newArrayList();

    public BackendLoadStatistic(long beId, String clusterName, Tag tag, SystemInfoService infoService, TabletInvertedIndex invertedIndex) {
        this.beId = beId;
        this.clusterName = clusterName;
        this.tag = tag;
        this.infoService = infoService;
        this.invertedIndex = invertedIndex;
    }

    public long getBeId() {
        return this.beId;
    }

    public String getClusterName() {
        return this.clusterName;
    }

    public Tag getTag() {
        return this.tag;
    }

    public boolean isAvailable() {
        return this.isAvailable;
    }

    public long getTotalCapacityB(TStorageMedium medium) {
        return this.totalCapacityMap.getOrDefault(medium, 0L);
    }

    public long getTotalUsedCapacityB(TStorageMedium medium) {
        return this.totalUsedCapacityMap.getOrDefault(medium, 0L);
    }

    public long getReplicaNum(TStorageMedium medium) {
        return this.totalReplicaNumMap.getOrDefault(medium, 0L);
    }

    public double getLoadScore(TStorageMedium medium) {
        if (this.loadScoreMap.containsKey(medium)) {
            return this.loadScoreMap.get((Object)medium).score;
        }
        return 0.0;
    }

    public double getMixLoadScore() {
        int mediumCount = 0;
        double totalLoadScore = 0.0;
        for (TStorageMedium medium : TStorageMedium.values()) {
            if (!this.hasMedium(medium)) continue;
            ++mediumCount;
            totalLoadScore += this.getLoadScore(medium);
        }
        return totalLoadScore / (double)(mediumCount == 0 ? 1 : mediumCount);
    }

    public void setClazz(TStorageMedium medium, Classification clazz) {
        this.clazzMap.put(medium, clazz);
    }

    public Classification getClazz(TStorageMedium medium) {
        return this.clazzMap.getOrDefault(medium, Classification.INIT);
    }

    public void init() throws LoadBalanceException {
        Backend be = this.infoService.getBackend(this.beId);
        if (be == null) {
            throw new LoadBalanceException("backend " + this.beId + " does not exist");
        }
        this.isAvailable = be.isScheduleAvailable() && be.isLoadAvailable() && be.isQueryAvailable();
        ImmutableMap<String, DiskInfo> disks = be.getDisks();
        for (DiskInfo diskInfo : disks.values()) {
            TStorageMedium medium = diskInfo.getStorageMedium();
            if (diskInfo.getState() == DiskInfo.DiskState.ONLINE) {
                this.totalCapacityMap.put(medium, this.totalCapacityMap.getOrDefault(medium, 0L) + diskInfo.getTotalCapacityB());
                this.totalUsedCapacityMap.put(medium, this.totalUsedCapacityMap.getOrDefault(medium, 0L) + (diskInfo.getTotalCapacityB() - diskInfo.getAvailableCapacityB()));
            }
            RootPathLoadStatistic pathStatistic = new RootPathLoadStatistic(this.beId, diskInfo.getRootPath(), diskInfo.getPathHash(), diskInfo.getStorageMedium(), diskInfo.getTotalCapacityB(), diskInfo.getDiskUsedCapacityB(), diskInfo.getState());
            this.pathStatistics.add(pathStatistic);
        }
        this.totalReplicaNumMap = this.invertedIndex.getReplicaNumByBeIdAndStorageMedium(this.beId);
        for (TStorageMedium medium : TStorageMedium.values()) {
            if (this.hasMedium(medium)) continue;
            this.totalReplicaNumMap.put(medium, 0L);
        }
        for (TStorageMedium storageMedium : TStorageMedium.values()) {
            this.classifyPathByLoad(storageMedium);
        }
        Collections.sort(this.pathStatistics);
    }

    private void classifyPathByLoad(TStorageMedium medium) {
        long totalCapacity = 0L;
        long totalUsedCapacity = 0L;
        for (RootPathLoadStatistic pathStat : this.pathStatistics) {
            if (pathStat.getStorageMedium() != medium) continue;
            totalCapacity += pathStat.getCapacityB();
            totalUsedCapacity += pathStat.getUsedCapacityB();
        }
        double avgUsedPercent = totalCapacity == 0L ? 0.0 : (double)totalUsedCapacity / (double)totalCapacity;
        int lowCounter = 0;
        int midCounter = 0;
        int highCounter = 0;
        for (RootPathLoadStatistic pathStat : this.pathStatistics) {
            if (pathStat.getStorageMedium() != medium) continue;
            if (Math.abs(pathStat.getUsedPercent() - avgUsedPercent) / avgUsedPercent > Config.balance_load_score_threshold) {
                if (pathStat.getUsedPercent() > avgUsedPercent) {
                    pathStat.setClazz(Classification.HIGH);
                    ++highCounter;
                    continue;
                }
                if (!(pathStat.getUsedPercent() < avgUsedPercent)) continue;
                pathStat.setClazz(Classification.LOW);
                ++lowCounter;
                continue;
            }
            pathStat.setClazz(Classification.MID);
            ++midCounter;
        }
        LOG.debug("classify path by load. storage: {} avg used percent: {}. low/mid/high: {}/{}/{}", (Object)avgUsedPercent, (Object)medium, (Object)lowCounter, (Object)midCounter, (Object)highCounter);
    }

    public void calcScore(Map<TStorageMedium, Double> avgClusterUsedCapacityPercentMap, Map<TStorageMedium, Double> avgClusterReplicaNumPerBackendMap) {
        for (TStorageMedium medium : TStorageMedium.values()) {
            LoadScore loadScore = BackendLoadStatistic.calcSore(this.totalUsedCapacityMap.getOrDefault(medium, 0L), this.totalCapacityMap.getOrDefault(medium, 1L), this.totalReplicaNumMap.getOrDefault(medium, 0L), avgClusterUsedCapacityPercentMap.getOrDefault(medium, 0.0), avgClusterReplicaNumPerBackendMap.getOrDefault(medium, 0.0));
            this.loadScoreMap.put(medium, loadScore);
            LOG.debug("backend {}, medium: {}, capacity coefficient: {}, replica coefficient: {}, load score: {}", (Object)this.beId, (Object)medium, (Object)loadScore.capacityCoefficient, (Object)loadScore.replicaNumCoefficient, (Object)loadScore.score);
        }
    }

    public static LoadScore calcSore(long beUsedCapacityB, long beTotalCapacityB, long beTotalReplicaNum, double avgClusterUsedCapacityPercent, double avgClusterReplicaNumPerBackend) {
        double usedCapacityPercent = (double)beUsedCapacityB / (double)beTotalCapacityB;
        double capacityProportion = avgClusterUsedCapacityPercent <= 0.0 ? 0.0 : usedCapacityPercent / avgClusterUsedCapacityPercent;
        double replicaNumProportion = avgClusterReplicaNumPerBackend <= 0.0 ? 0.0 : (double)beTotalReplicaNum / avgClusterReplicaNumPerBackend;
        LoadScore loadScore = new LoadScore();
        loadScore.capacityCoefficient = usedCapacityPercent < 0.5 ? 0.5 : (usedCapacityPercent > Config.capacity_used_percent_high_water ? 1.0 : 2.0 * usedCapacityPercent - 0.5);
        loadScore.replicaNumCoefficient = 1.0 - loadScore.capacityCoefficient;
        loadScore.score = capacityProportion * loadScore.capacityCoefficient + replicaNumProportion * loadScore.replicaNumCoefficient;
        return loadScore;
    }

    public BalanceStatus isFit(long tabletSize, TStorageMedium medium, List<RootPathLoadStatistic> result, boolean isSupplement) {
        BalanceStatus status = new BalanceStatus(BalanceStatus.ErrCode.COMMON_ERROR);
        for (int i = 0; i < this.pathStatistics.size(); ++i) {
            RootPathLoadStatistic pathStatistic = this.pathStatistics.get(i);
            if (!isSupplement && pathStatistic.getStorageMedium() != medium) continue;
            BalanceStatus bStatus = pathStatistic.isFit(tabletSize, isSupplement);
            if (!bStatus.ok()) {
                status.addErrMsgs(bStatus.getErrMsgs());
                continue;
            }
            result.add(pathStatistic);
            return BalanceStatus.OK;
        }
        return status;
    }

    public boolean hasAvailDisk() {
        for (RootPathLoadStatistic rootPathLoadStatistic : this.pathStatistics) {
            if (rootPathLoadStatistic.getDiskState() != DiskInfo.DiskState.ONLINE) continue;
            return true;
        }
        return false;
    }

    public void getPathStatisticByClass(Set<Long> low, Set<Long> mid, Set<Long> high, TStorageMedium storageMedium) {
        for (RootPathLoadStatistic pathStat : this.pathStatistics) {
            if (pathStat.getDiskState() == DiskInfo.DiskState.OFFLINE || storageMedium != null && pathStat.getStorageMedium() != storageMedium) continue;
            if (pathStat.getClazz() == Classification.LOW) {
                low.add(pathStat.getPathHash());
                continue;
            }
            if (pathStat.getClazz() == Classification.HIGH) {
                high.add(pathStat.getPathHash());
                continue;
            }
            mid.add(pathStat.getPathHash());
        }
        LOG.debug("after adjust, backend {} path classification low/mid/high: {}/{}/{}", (Object)this.beId, (Object)low.size(), (Object)mid.size(), (Object)high.size());
    }

    public List<RootPathLoadStatistic> getPathStatistics() {
        return this.pathStatistics;
    }

    public long getAvailPathNum(TStorageMedium medium) {
        return this.pathStatistics.stream().filter(p -> p.getDiskState() == DiskInfo.DiskState.ONLINE && p.getStorageMedium() == medium).count();
    }

    public boolean hasMedium(TStorageMedium medium) {
        for (RootPathLoadStatistic rootPathLoadStatistic : this.pathStatistics) {
            if (rootPathLoadStatistic.getStorageMedium() != medium) continue;
            return true;
        }
        return false;
    }

    public String getBrief() {
        StringBuilder sb = new StringBuilder();
        sb.append(this.beId);
        for (TStorageMedium medium : TStorageMedium.values()) {
            sb.append(", ").append(medium).append(": replica: ").append(this.totalReplicaNumMap.get(medium));
            sb.append(" used: ").append(this.totalUsedCapacityMap.getOrDefault(medium, 0L));
            sb.append(" total: ").append(this.totalCapacityMap.getOrDefault(medium, 0L));
            sb.append(" score: ").append(this.loadScoreMap.getOrDefault((Object)medium, (LoadScore)LoadScore.DUMMY).score);
        }
        return sb.toString();
    }

    public List<String> getInfo(TStorageMedium medium) {
        ArrayList info = Lists.newArrayList();
        info.add(String.valueOf(this.beId));
        info.add(this.clusterName);
        info.add(String.valueOf(this.isAvailable));
        long used = this.totalUsedCapacityMap.getOrDefault(medium, 0L);
        long total = this.totalCapacityMap.getOrDefault(medium, 0L);
        info.add(String.valueOf(used));
        info.add(String.valueOf(total));
        info.add(String.valueOf(DebugUtil.DECIMAL_FORMAT_SCALE_3.format((double)(used * 100L) / (double)total)));
        info.add(String.valueOf(this.totalReplicaNumMap.getOrDefault(medium, 0L)));
        LoadScore loadScore = this.loadScoreMap.getOrDefault(medium, new LoadScore());
        info.add(String.valueOf(loadScore.capacityCoefficient));
        info.add(String.valueOf(loadScore.replicaNumCoefficient));
        info.add(String.valueOf(loadScore.score));
        info.add(this.clazzMap.getOrDefault(medium, Classification.INIT).name());
        return info;
    }

    public boolean canFitInColocate(long totalReplicaSize) {
        long totalCap;
        long afterUsedCap = this.totalUsedCapacityMap.values().stream().reduce(0L, Long::sum) + totalReplicaSize;
        double afterRatio = (double)afterUsedCap / (double)(totalCap = this.totalCapacityMap.values().stream().reduce(0L, Long::sum).longValue());
        return afterRatio < 0.9;
    }

    public static class LoadScore {
        public double replicaNumCoefficient = 0.5;
        public double capacityCoefficient = 0.5;
        public double score = 0.0;
        public static final LoadScore DUMMY = new LoadScore();
    }

    public static enum Classification {
        INIT,
        LOW,
        MID,
        HIGH;

    }

    public static class BeStatMixComparator
    implements Comparator<BackendLoadStatistic> {
        @Override
        public int compare(BackendLoadStatistic o1, BackendLoadStatistic o2) {
            Double score1 = o1.getMixLoadScore();
            Double score2 = o2.getMixLoadScore();
            return score1.compareTo(score2);
        }
    }

    public static class BeStatComparator
    implements Comparator<BackendLoadStatistic> {
        private TStorageMedium medium;

        public BeStatComparator(TStorageMedium medium) {
            this.medium = medium;
        }

        @Override
        public int compare(BackendLoadStatistic o1, BackendLoadStatistic o2) {
            double score1 = o1.getLoadScore(this.medium);
            double score2 = o2.getLoadScore(this.medium);
            return Double.compare(score1, score2);
        }
    }
}

