/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.catalog;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import com.google.gson.annotations.SerializedName;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.LongStream;
import org.apache.doris.catalog.Catalog;
import org.apache.doris.catalog.MetaObject;
import org.apache.doris.catalog.Replica;
import org.apache.doris.catalog.ReplicaAllocation;
import org.apache.doris.clone.TabletSchedCtx;
import org.apache.doris.common.Config;
import org.apache.doris.common.Pair;
import org.apache.doris.common.io.Writable;
import org.apache.doris.resource.Tag;
import org.apache.doris.system.Backend;
import org.apache.doris.system.SystemInfoService;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class Tablet
extends MetaObject
implements Writable {
    private static final Logger LOG = LogManager.getLogger(Tablet.class);
    private static final int QUERYABLE_TIMES_OF_MIN_VERSION_COUNT = 3;
    @SerializedName(value="id")
    private long id;
    @SerializedName(value="replicas")
    private List<Replica> replicas;
    @SerializedName(value="checkedVersion")
    private long checkedVersion;
    @Deprecated
    @SerializedName(value="checkedVersionHash")
    private long checkedVersionHash;
    @SerializedName(value="isConsistent")
    private boolean isConsistent;
    private long lastStatusCheckTime = -1L;

    public Tablet() {
        this(0L, new ArrayList<Replica>());
    }

    public Tablet(long tabletId) {
        this(tabletId, new ArrayList<Replica>());
    }

    public Tablet(long tabletId, List<Replica> replicas) {
        this.id = tabletId;
        this.replicas = replicas;
        if (this.replicas == null) {
            this.replicas = new ArrayList<Replica>();
        }
        this.checkedVersion = -1L;
        this.isConsistent = true;
    }

    public void setIdForRestore(long tabletId) {
        this.id = tabletId;
    }

    public long getId() {
        return this.id;
    }

    public long getCheckedVersion() {
        return this.checkedVersion;
    }

    public void setCheckedVersion(long checkedVersion) {
        this.checkedVersion = checkedVersion;
    }

    public void setIsConsistent(boolean good) {
        this.isConsistent = good;
    }

    public boolean isConsistent() {
        return this.isConsistent;
    }

    private boolean deleteRedundantReplica(long backendId, long version) {
        boolean delete = false;
        boolean hasBackend = false;
        Iterator<Replica> iterator = this.replicas.iterator();
        while (iterator.hasNext()) {
            Replica replica = iterator.next();
            if (replica.getBackendId() != backendId) continue;
            hasBackend = true;
            if (replica.getVersion() > version) continue;
            iterator.remove();
            delete = true;
        }
        return delete || !hasBackend;
    }

    public void addReplica(Replica replica, boolean isRestore) {
        if (this.deleteRedundantReplica(replica.getBackendId(), replica.getVersion())) {
            this.replicas.add(replica);
            if (!isRestore) {
                Catalog.getCurrentInvertedIndex().addReplica(this.id, replica);
            }
        }
    }

    public void addReplica(Replica replica) {
        this.addReplica(replica, false);
    }

    public List<Replica> getReplicas() {
        return this.replicas;
    }

    public Set<Long> getBackendIds() {
        HashSet beIds = Sets.newHashSet();
        for (Replica replica : this.replicas) {
            beIds.add(replica.getBackendId());
        }
        return beIds;
    }

    public List<Long> getNormalReplicaBackendIds() {
        ArrayList beIds = Lists.newArrayList();
        SystemInfoService infoService = Catalog.getCurrentSystemInfo();
        for (Replica replica : this.replicas) {
            if (replica.isBad()) continue;
            Replica.ReplicaState state = replica.getState();
            if (!infoService.checkBackendAlive(replica.getBackendId()) || !state.canLoad()) continue;
            beIds.add(replica.getBackendId());
        }
        return beIds;
    }

    public Multimap<Long, Long> getNormalReplicaBackendPathMap() {
        HashMultimap map = HashMultimap.create();
        SystemInfoService infoService = Catalog.getCurrentSystemInfo();
        for (Replica replica : this.replicas) {
            if (replica.isBad()) continue;
            Replica.ReplicaState state = replica.getState();
            if (!infoService.checkBackendLoadAvailable(replica.getBackendId()) || !state.canLoad()) continue;
            map.put((Object)replica.getBackendId(), (Object)replica.getPathHash());
        }
        return map;
    }

    public List<Replica> getQueryableReplicas(long visibleVersion, int schemaHash) {
        ArrayList allQueryableReplica = Lists.newArrayListWithCapacity((int)this.replicas.size());
        for (Replica replica2 : this.replicas) {
            Object state;
            if (replica2.isBad() || replica2.getLastFailedVersion() > 0L || !((Replica.ReplicaState)((Object)(state = replica2.getState()))).canQuery() || !replica2.checkVersionCatchUp(visibleVersion, false) || replica2.getSchemaHash() != -1 && replica2.getSchemaHash() != schemaHash) continue;
            allQueryableReplica.add(replica2);
        }
        if (Config.skip_compaction_slower_replica && allQueryableReplica.size() > 1) {
            long minVersionCount = Long.MAX_VALUE;
            for (Replica replica3 : allQueryableReplica) {
                if (replica3.getVersionCount() == -1L || replica3.getVersionCount() >= minVersionCount) continue;
                minVersionCount = replica3.getVersionCount();
            }
            long finalMinVersionCount = minVersionCount;
            return allQueryableReplica.stream().filter(replica -> replica.getVersionCount() == -1L || replica.getVersionCount() < (long)Config.min_version_count_indicate_replica_compaction_too_slow || replica.getVersionCount() < finalMinVersionCount * 3L).collect(Collectors.toList());
        }
        return allQueryableReplica;
    }

    public Replica getReplicaById(long replicaId) {
        for (Replica replica : this.replicas) {
            if (replica.getId() != replicaId) continue;
            return replica;
        }
        return null;
    }

    public Replica getReplicaByBackendId(long backendId) {
        for (Replica replica : this.replicas) {
            if (replica.getBackendId() != backendId) continue;
            return replica;
        }
        return null;
    }

    public boolean deleteReplica(Replica replica) {
        if (this.replicas.contains(replica)) {
            this.replicas.remove(replica);
            Catalog.getCurrentInvertedIndex().deleteReplica(this.id, replica.getBackendId());
            return true;
        }
        return false;
    }

    public boolean deleteReplicaByBackendId(long backendId) {
        Iterator<Replica> iterator = this.replicas.iterator();
        while (iterator.hasNext()) {
            Replica replica = iterator.next();
            if (replica.getBackendId() != backendId) continue;
            iterator.remove();
            Catalog.getCurrentInvertedIndex().deleteReplica(this.id, backendId);
            return true;
        }
        return false;
    }

    @Deprecated
    public Replica deleteReplicaById(long replicaId) {
        Iterator<Replica> iterator = this.replicas.iterator();
        while (iterator.hasNext()) {
            Replica replica = iterator.next();
            if (replica.getId() != replicaId) continue;
            LOG.info("delete replica[" + replica.getId() + "]");
            iterator.remove();
            return replica;
        }
        return null;
    }

    public void clearReplica() {
        this.replicas.clear();
    }

    public void setTabletId(long tabletId) {
        this.id = tabletId;
    }

    public static void sortReplicaByVersionDesc(List<Replica> replicas) {
        replicas.sort(Replica.VERSION_DESC_COMPARATOR);
    }

    public String toString() {
        return "tabletId=" + this.id;
    }

    @Override
    public void write(DataOutput out) throws IOException {
        super.write(out);
        out.writeLong(this.id);
        int replicaCount = this.replicas.size();
        out.writeInt(replicaCount);
        for (int i = 0; i < replicaCount; ++i) {
            this.replicas.get(i).write(out);
        }
        out.writeLong(this.checkedVersion);
        out.writeLong(this.checkedVersionHash);
        out.writeBoolean(this.isConsistent);
    }

    @Override
    public void readFields(DataInput in) throws IOException {
        super.readFields(in);
        this.id = in.readLong();
        int replicaCount = in.readInt();
        for (int i = 0; i < replicaCount; ++i) {
            Replica replica = Replica.read(in);
            if (!this.deleteRedundantReplica(replica.getBackendId(), replica.getVersion())) continue;
            this.replicas.add(replica);
        }
        this.checkedVersion = in.readLong();
        this.checkedVersionHash = in.readLong();
        this.isConsistent = in.readBoolean();
    }

    public static Tablet read(DataInput in) throws IOException {
        Tablet tablet = new Tablet();
        tablet.readFields(in);
        return tablet;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof Tablet)) {
            return false;
        }
        Tablet tablet = (Tablet)obj;
        if (this.replicas != tablet.replicas) {
            if (this.replicas.size() != tablet.replicas.size()) {
                return false;
            }
            int size = this.replicas.size();
            for (int i = 0; i < size; ++i) {
                if (tablet.replicas.contains(this.replicas.get(i))) continue;
                return false;
            }
        }
        return this.id == tablet.id;
    }

    public long getDataSize(boolean singleReplica) {
        LongStream s = this.replicas.stream().filter(r -> r.getState() == Replica.ReplicaState.NORMAL).mapToLong(Replica::getDataSize);
        return singleReplica ? Double.valueOf(s.average().orElse(0.0)).longValue() : s.sum();
    }

    public Pair<TabletStatus, TabletSchedCtx.Priority> getHealthStatusWithPriority(SystemInfoService systemInfoService, String clusterName, long visibleVersion, ReplicaAllocation replicaAlloc, List<Long> aliveBeIdsInCluster) {
        Map<Tag, Short> allocMap = replicaAlloc.getAllocMap();
        HashMap currentAllocMap = Maps.newHashMap();
        int replicationNum = replicaAlloc.getTotalReplicaNum();
        int alive = 0;
        int aliveAndVersionComplete = 0;
        int stable = 0;
        int availableInCluster = 0;
        Replica needFurtherRepairReplica = null;
        HashSet hosts = Sets.newHashSet();
        ArrayList<Long> versions = new ArrayList<Long>();
        for (Replica replica : this.replicas) {
            Backend backend = systemInfoService.getBackend(replica.getBackendId());
            if (backend == null || !backend.isAlive() || !replica.isAlive() || !hosts.add(backend.getHost()) || replica.tooSlow()) continue;
            ++alive;
            if (replica.getLastFailedVersion() > 0L || replica.getVersion() < visibleVersion) continue;
            ++aliveAndVersionComplete;
            if (!backend.isScheduleAvailable()) continue;
            ++stable;
            if (!backend.getOwnerClusterName().equals(clusterName)) continue;
            ++availableInCluster;
            if (replica.needFurtherRepair() && needFurtherRepairReplica == null) {
                needFurtherRepairReplica = replica;
            }
            versions.add(replica.getVersionCount());
            short curNum = currentAllocMap.getOrDefault(backend.getTag(), (short)0);
            currentAllocMap.put(backend.getTag(), (short)(curNum + 1));
        }
        int aliveBackendsNum = aliveBeIdsInCluster.size();
        if (alive == 0) {
            return Pair.create(TabletStatus.UNRECOVERABLE, TabletSchedCtx.Priority.VERY_HIGH);
        }
        if (alive < replicationNum && this.replicas.size() >= aliveBackendsNum && aliveBackendsNum >= replicationNum && replicationNum > 1) {
            return Pair.create(TabletStatus.FORCE_REDUNDANT, TabletSchedCtx.Priority.VERY_HIGH);
        }
        if (alive < replicationNum / 2 + 1) {
            return Pair.create(TabletStatus.REPLICA_MISSING, TabletSchedCtx.Priority.HIGH);
        }
        if (alive < replicationNum) {
            return Pair.create(TabletStatus.REPLICA_MISSING, TabletSchedCtx.Priority.NORMAL);
        }
        if (aliveAndVersionComplete == 0) {
            return Pair.create(TabletStatus.UNRECOVERABLE, TabletSchedCtx.Priority.VERY_HIGH);
        }
        if (aliveAndVersionComplete < replicationNum / 2 + 1) {
            return Pair.create(TabletStatus.VERSION_INCOMPLETE, TabletSchedCtx.Priority.HIGH);
        }
        if (aliveAndVersionComplete < replicationNum) {
            return Pair.create(TabletStatus.VERSION_INCOMPLETE, TabletSchedCtx.Priority.NORMAL);
        }
        if (aliveAndVersionComplete > replicationNum) {
            if (needFurtherRepairReplica != null) {
                return Pair.create(TabletStatus.NEED_FURTHER_REPAIR, TabletSchedCtx.Priority.HIGH);
            }
            return Pair.create(TabletStatus.REDUNDANT, TabletSchedCtx.Priority.VERY_HIGH);
        }
        if (stable < replicationNum) {
            List replicaBeIds = this.replicas.stream().map(Replica::getBackendId).collect(Collectors.toList());
            List list = aliveBeIdsInCluster.stream().filter(systemInfoService::checkBackendScheduleAvailable).collect(Collectors.toList());
            if (replicaBeIds.containsAll(list) && list.size() >= replicationNum && replicationNum > 1) {
                return Pair.create(TabletStatus.FORCE_REDUNDANT, stable < replicationNum / 2 + 1 ? TabletSchedCtx.Priority.NORMAL : TabletSchedCtx.Priority.LOW);
            }
            if (stable < replicationNum / 2 + 1) {
                return Pair.create(TabletStatus.REPLICA_RELOCATING, TabletSchedCtx.Priority.NORMAL);
            }
            if (stable < replicationNum) {
                return Pair.create(TabletStatus.REPLICA_RELOCATING, TabletSchedCtx.Priority.LOW);
            }
        }
        if (availableInCluster < replicationNum) {
            return Pair.create(TabletStatus.REPLICA_MISSING_IN_CLUSTER, TabletSchedCtx.Priority.LOW);
        }
        for (Map.Entry entry : allocMap.entrySet()) {
            if (currentAllocMap.containsKey(entry.getKey()) && (Short)currentAllocMap.get(entry.getKey()) >= (Short)entry.getValue()) continue;
            return Pair.create(TabletStatus.REPLICA_MISSING_FOR_TAG, TabletSchedCtx.Priority.NORMAL);
        }
        if (this.replicas.size() > replicationNum) {
            if (needFurtherRepairReplica != null) {
                return Pair.create(TabletStatus.NEED_FURTHER_REPAIR, TabletSchedCtx.Priority.HIGH);
            }
            return Pair.create(TabletStatus.REDUNDANT, TabletSchedCtx.Priority.VERY_HIGH);
        }
        if (Config.repair_slow_replica && versions.size() == this.replicas.size() && versions.size() > 1) {
            Collections.sort(versions);
            long delta = (Long)versions.get(versions.size() - 1) - (Long)versions.get(0);
            double ratio = (double)delta / (double)((Long)versions.get(versions.size() - 1)).longValue();
            if ((Long)versions.get(versions.size() - 1) > (long)Config.min_version_count_indicate_replica_compaction_too_slow && ratio > Config.valid_version_count_delta_ratio_between_replicas) {
                return Pair.create(TabletStatus.REPLICA_COMPACTION_TOO_SLOW, TabletSchedCtx.Priority.HIGH);
            }
        }
        return Pair.create(TabletStatus.HEALTHY, TabletSchedCtx.Priority.NORMAL);
    }

    public TabletStatus getColocateHealthStatus(long visibleVersion, ReplicaAllocation replicaAlloc, Set<Long> backendsSet) {
        Short totalReplicaNum = replicaAlloc.getTotalReplicaNum();
        Set<Long> replicaBackendIds = this.getBackendIds();
        if (!replicaBackendIds.containsAll(backendsSet)) {
            return TabletStatus.COLOCATE_MISMATCH;
        }
        for (Replica replica : this.replicas) {
            if (!backendsSet.contains(replica.getBackendId())) continue;
            if (!replica.isAlive()) {
                if (replica.isBad()) {
                    return TabletStatus.COLOCATE_REDUNDANT;
                }
                return TabletStatus.VERSION_INCOMPLETE;
            }
            if (replica.getLastFailedVersion() <= 0L && replica.getVersion() >= visibleVersion) continue;
            return TabletStatus.VERSION_INCOMPLETE;
        }
        if (this.replicas.size() > totalReplicaNum) {
            return TabletStatus.COLOCATE_REDUNDANT;
        }
        return TabletStatus.HEALTHY;
    }

    public boolean readyToBeRepaired(TabletSchedCtx.Priority priority) {
        if (priority == TabletSchedCtx.Priority.VERY_HIGH) {
            return true;
        }
        long currentTime = System.currentTimeMillis();
        if (this.lastStatusCheckTime == -1L) {
            this.lastStatusCheckTime = currentTime;
            return false;
        }
        boolean ready = false;
        switch (priority) {
            case HIGH: {
                ready = currentTime - this.lastStatusCheckTime > Config.tablet_repair_delay_factor_second * 1000L * 1L;
                break;
            }
            case NORMAL: {
                ready = currentTime - this.lastStatusCheckTime > Config.tablet_repair_delay_factor_second * 1000L * 2L;
                break;
            }
            case LOW: {
                ready = currentTime - this.lastStatusCheckTime > Config.tablet_repair_delay_factor_second * 1000L * 3L;
                break;
            }
        }
        return ready;
    }

    public void setLastStatusCheckTime(long lastStatusCheckTime) {
        this.lastStatusCheckTime = lastStatusCheckTime;
    }

    public static enum TabletStatus {
        HEALTHY,
        REPLICA_MISSING,
        VERSION_INCOMPLETE,
        REPLICA_RELOCATING,
        REDUNDANT,
        REPLICA_MISSING_IN_CLUSTER,
        REPLICA_MISSING_FOR_TAG,
        FORCE_REDUNDANT,
        COLOCATE_MISMATCH,
        COLOCATE_REDUNDANT,
        NEED_FURTHER_REPAIR,
        UNRECOVERABLE,
        REPLICA_COMPACTION_TOO_SLOW;

    }
}

