/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.catalog;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.doris.catalog.Catalog;
import org.apache.doris.catalog.DatabaseEncryptKey;
import org.apache.doris.catalog.DatabaseProperty;
import org.apache.doris.catalog.EncryptKey;
import org.apache.doris.catalog.EncryptKeySearchDesc;
import org.apache.doris.catalog.EsTable;
import org.apache.doris.catalog.Function;
import org.apache.doris.catalog.FunctionSearchDesc;
import org.apache.doris.catalog.MetaObject;
import org.apache.doris.catalog.OlapTable;
import org.apache.doris.catalog.Partition;
import org.apache.doris.catalog.Table;
import org.apache.doris.cluster.ClusterNamespace;
import org.apache.doris.common.AnalysisException;
import org.apache.doris.common.Config;
import org.apache.doris.common.DdlException;
import org.apache.doris.common.ErrorCode;
import org.apache.doris.common.MetaNotFoundException;
import org.apache.doris.common.Pair;
import org.apache.doris.common.UserException;
import org.apache.doris.common.io.Text;
import org.apache.doris.common.io.Writable;
import org.apache.doris.common.util.DebugUtil;
import org.apache.doris.persist.CreateTableInfo;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class Database
extends MetaObject
implements Writable {
    private static final Logger LOG = LogManager.getLogger(Database.class);
    private long id;
    private volatile String fullQualifiedName;
    private String clusterName;
    private ReentrantReadWriteLock rwLock;
    private Map<Long, Table> idToTable;
    private Map<String, Table> nameToTable;
    private Map<String, String> lowerCaseToTableName;
    private ConcurrentMap<String, ImmutableList<Function>> name2Function = Maps.newConcurrentMap();
    private DatabaseEncryptKey dbEncryptKey;
    private volatile long dataQuotaBytes;
    private volatile long replicaQuotaSize;
    private volatile boolean isDropped;
    private String attachDbName;
    private DbState dbState;
    private DatabaseProperty dbProperties = new DatabaseProperty();

    public Database() {
        this(0L, null);
    }

    public Database(long id, String name) {
        this.id = id;
        this.fullQualifiedName = name;
        if (this.fullQualifiedName == null) {
            this.fullQualifiedName = "";
        }
        this.rwLock = new ReentrantReadWriteLock(true);
        this.idToTable = Maps.newConcurrentMap();
        this.nameToTable = Maps.newConcurrentMap();
        this.lowerCaseToTableName = Maps.newConcurrentMap();
        this.dataQuotaBytes = Config.default_db_data_quota_bytes;
        this.replicaQuotaSize = Config.default_db_replica_quota_size;
        this.dbState = DbState.NORMAL;
        this.attachDbName = "";
        this.clusterName = "";
        this.dbEncryptKey = new DatabaseEncryptKey();
    }

    public void markDropped() {
        this.isDropped = true;
    }

    public void unmarkDropped() {
        this.isDropped = false;
    }

    public void readLock() {
        this.rwLock.readLock().lock();
    }

    public void readUnlock() {
        this.rwLock.readLock().unlock();
    }

    public void writeLock() {
        this.rwLock.writeLock().lock();
    }

    public void writeUnlock() {
        this.rwLock.writeLock().unlock();
    }

    public boolean tryWriteLock(long timeout, TimeUnit unit) {
        try {
            return this.rwLock.writeLock().tryLock(timeout, unit);
        }
        catch (InterruptedException e) {
            LOG.warn("failed to try write lock at db[" + this.id + "]", (Throwable)e);
            return false;
        }
    }

    public boolean isWriteLockHeldByCurrentThread() {
        return this.rwLock.writeLock().isHeldByCurrentThread();
    }

    public boolean writeLockIfExist() {
        if (!this.isDropped) {
            this.rwLock.writeLock().lock();
            return true;
        }
        return false;
    }

    public <E extends Exception> void writeLockOrException(E e) throws E {
        this.writeLock();
        if (this.isDropped) {
            this.writeUnlock();
            throw e;
        }
    }

    public void writeLockOrDdlException() throws DdlException {
        this.writeLockOrException(new DdlException("unknown db, dbName=" + this.fullQualifiedName));
    }

    public long getId() {
        return this.id;
    }

    public String getFullName() {
        return this.fullQualifiedName;
    }

    public void setNameWithLock(String newName) {
        this.writeLock();
        try {
            this.fullQualifiedName = newName;
        }
        finally {
            this.writeUnlock();
        }
    }

    public void setDataQuota(long newQuota) {
        Preconditions.checkArgument((newQuota >= 0L ? 1 : 0) != 0);
        LOG.info("database[{}] set quota from {} to {}", (Object)this.fullQualifiedName, (Object)this.dataQuotaBytes, (Object)newQuota);
        this.dataQuotaBytes = newQuota;
    }

    public void setReplicaQuota(long newQuota) {
        Preconditions.checkArgument((newQuota >= 0L ? 1 : 0) != 0);
        LOG.info("database[{}] set replica quota from {} to {}", (Object)this.fullQualifiedName, (Object)this.replicaQuotaSize, (Object)newQuota);
        this.replicaQuotaSize = newQuota;
    }

    public long getDataQuota() {
        return this.dataQuotaBytes;
    }

    public long getReplicaQuota() {
        return this.replicaQuotaSize;
    }

    public DatabaseProperty getDbProperties() {
        return this.dbProperties;
    }

    public void setDbProperties(DatabaseProperty dbProperties) {
        this.dbProperties = dbProperties;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long getUsedDataQuotaWithLock() {
        long usedDataQuota = 0L;
        this.readLock();
        try {
            for (Table table : this.idToTable.values()) {
                if (table.getType() != Table.TableType.OLAP) continue;
                OlapTable olapTable = (OlapTable)table;
                olapTable.readLock();
                try {
                    usedDataQuota += olapTable.getDataSize();
                }
                finally {
                    olapTable.readUnlock();
                }
            }
            long l = usedDataQuota;
            return l;
        }
        finally {
            this.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long getReplicaQuotaLeftWithLock() {
        long usedReplicaQuota = 0L;
        this.readLock();
        try {
            for (Table table : this.idToTable.values()) {
                if (table.getType() != Table.TableType.OLAP) continue;
                OlapTable olapTable = (OlapTable)table;
                olapTable.readLock();
                try {
                    usedReplicaQuota += olapTable.getReplicaCount();
                }
                finally {
                    olapTable.readUnlock();
                }
            }
            long leftReplicaQuota = this.replicaQuotaSize - usedReplicaQuota;
            long l = Math.max(leftReplicaQuota, 0L);
            return l;
        }
        finally {
            this.readUnlock();
        }
    }

    public void checkDataSizeQuota() throws DdlException {
        Pair<Double, String> quotaUnitPair = DebugUtil.getByteUint(this.dataQuotaBytes);
        String readableQuota = DebugUtil.DECIMAL_FORMAT_SCALE_3.format(quotaUnitPair.first) + " " + (String)quotaUnitPair.second;
        long usedDataQuota = this.getUsedDataQuotaWithLock();
        long leftDataQuota = Math.max(this.dataQuotaBytes - usedDataQuota, 0L);
        Pair<Double, String> leftQuotaUnitPair = DebugUtil.getByteUint(leftDataQuota);
        String readableLeftQuota = DebugUtil.DECIMAL_FORMAT_SCALE_3.format(leftQuotaUnitPair.first) + " " + (String)leftQuotaUnitPair.second;
        LOG.info("database[{}] data quota: left bytes: {} / total: {}", (Object)this.fullQualifiedName, (Object)readableLeftQuota, (Object)readableQuota);
        if (leftDataQuota <= 0L) {
            throw new DdlException("Database[" + this.fullQualifiedName + "] data size exceeds quota[" + readableQuota + "]");
        }
    }

    public void checkReplicaQuota() throws DdlException {
        long leftReplicaQuota = this.getReplicaQuotaLeftWithLock();
        LOG.info("database[{}] replica quota: left number: {} / total: {}", (Object)this.fullQualifiedName, (Object)leftReplicaQuota, (Object)this.replicaQuotaSize);
        if (leftReplicaQuota <= 0L) {
            throw new DdlException("Database[" + this.fullQualifiedName + "] replica number exceeds quota[" + this.replicaQuotaSize + "]");
        }
    }

    public void checkQuota() throws DdlException {
        this.checkDataSizeQuota();
        this.checkReplicaQuota();
    }

    private boolean isTableExist(String tableName) {
        if (Catalog.isTableNamesCaseInsensitive() && (tableName = this.lowerCaseToTableName.get(tableName.toLowerCase())) == null) {
            return false;
        }
        return this.nameToTable.containsKey(tableName);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Pair<Boolean, Boolean> createTableWithLock(Table table, boolean isReplay, boolean setIfNotExist) throws DdlException {
        boolean result = true;
        boolean isTableExist = false;
        this.writeLockOrDdlException();
        try {
            String tableName = table.getName();
            if (Catalog.isStoredTableNamesLowerCase()) {
                tableName = tableName.toLowerCase();
            }
            if (this.isTableExist(tableName)) {
                result = setIfNotExist;
                isTableExist = true;
            } else {
                this.idToTable.put(table.getId(), table);
                this.nameToTable.put(table.getName(), table);
                this.lowerCaseToTableName.put(tableName.toLowerCase(), tableName);
                if (!isReplay) {
                    CreateTableInfo info = new CreateTableInfo(this.fullQualifiedName, table);
                    Catalog.getCurrentCatalog().getEditLog().logCreateTable(info);
                }
                if (table.getType() == Table.TableType.ELASTICSEARCH) {
                    Catalog.getCurrentCatalog().getEsRepository().registerTable((EsTable)table);
                }
            }
            Pair<Boolean, Boolean> pair = Pair.create(result, isTableExist);
            return pair;
        }
        finally {
            this.writeUnlock();
        }
    }

    public boolean createTable(Table table) {
        boolean result = true;
        String tableName = table.getName();
        if (Catalog.isStoredTableNamesLowerCase()) {
            tableName = tableName.toLowerCase();
        }
        if (this.isTableExist(tableName)) {
            result = false;
        } else {
            this.idToTable.put(table.getId(), table);
            this.nameToTable.put(table.getName(), table);
            this.lowerCaseToTableName.put(tableName.toLowerCase(), tableName);
        }
        table.unmarkDropped();
        return result;
    }

    public void dropTable(String tableName) {
        Table table;
        if (Catalog.isStoredTableNamesLowerCase()) {
            tableName = tableName.toLowerCase();
        }
        if ((table = this.getTableNullable(tableName)) != null) {
            this.nameToTable.remove(tableName);
            this.idToTable.remove(table.getId());
            this.lowerCaseToTableName.remove(tableName.toLowerCase());
            table.markDropped();
        }
    }

    public List<Table> getTables() {
        return new ArrayList<Table>(this.idToTable.values());
    }

    public List<Table> getTablesOnIdOrder() {
        return this.idToTable.values().stream().sorted(Comparator.comparing(Table::getId)).collect(Collectors.toList());
    }

    public List<Table> getViews() {
        ArrayList<Table> views = new ArrayList<Table>();
        for (Table table : this.idToTable.values()) {
            if (table.getType() != Table.TableType.VIEW) continue;
            views.add(table);
        }
        return views;
    }

    public List<Table> getTablesOnIdOrderIfExist(List<Long> tableIdList) {
        ArrayList tableList = Lists.newArrayListWithCapacity((int)tableIdList.size());
        for (Long tableId : tableIdList) {
            Table table = this.idToTable.get(tableId);
            if (table == null) continue;
            tableList.add(table);
        }
        if (tableList.size() > 1) {
            return tableList.stream().sorted(Comparator.comparing(Table::getId)).collect(Collectors.toList());
        }
        return tableList;
    }

    public List<Table> getTablesOnIdOrderOrThrowException(List<Long> tableIdList) throws MetaNotFoundException {
        ArrayList tableList = Lists.newArrayListWithCapacity((int)tableIdList.size());
        for (Long tableId : tableIdList) {
            Table table = this.idToTable.get(tableId);
            if (table == null) {
                throw new MetaNotFoundException("unknown table, tableId=" + tableId);
            }
            tableList.add(table);
        }
        if (tableList.size() > 1) {
            return tableList.stream().sorted(Comparator.comparing(Table::getId)).collect(Collectors.toList());
        }
        return tableList;
    }

    public Set<String> getTableNamesWithLock() {
        this.readLock();
        try {
            HashSet<String> hashSet = new HashSet<String>(this.nameToTable.keySet());
            return hashSet;
        }
        finally {
            this.readUnlock();
        }
    }

    @Nullable
    public Table getTableNullable(String tableName) {
        if (Catalog.isStoredTableNamesLowerCase()) {
            tableName = tableName.toLowerCase();
        }
        if (Catalog.isTableNamesCaseInsensitive() && (tableName = this.lowerCaseToTableName.get(tableName.toLowerCase())) == null) {
            return null;
        }
        return this.nameToTable.get(tableName);
    }

    @Nullable
    public Table getTableNullable(long tableId) {
        return this.idToTable.get(tableId);
    }

    public Optional<Table> getTable(String tableName) {
        return Optional.ofNullable(this.getTableNullable(tableName));
    }

    public Optional<Table> getTable(long tableId) {
        return Optional.ofNullable(this.getTableNullable(tableId));
    }

    public <E extends Exception> Table getTableOrException(String tableName, java.util.function.Function<String, E> e) throws E {
        Table table = this.getTableNullable(tableName);
        if (table == null) {
            throw (Exception)e.apply(tableName);
        }
        return table;
    }

    public <E extends Exception> Table getTableOrException(long tableId, java.util.function.Function<Long, E> e) throws E {
        Table table = this.getTableNullable(tableId);
        if (table == null) {
            throw (Exception)e.apply(tableId);
        }
        return table;
    }

    public Table getTableOrMetaException(String tableName) throws MetaNotFoundException {
        return this.getTableOrException(tableName, (String t) -> new MetaNotFoundException("unknown table, tableName=" + t));
    }

    public Table getTableOrMetaException(long tableId) throws MetaNotFoundException {
        return this.getTableOrException(tableId, (Long t) -> new MetaNotFoundException("unknown table, tableId=" + t));
    }

    public <T extends Table> T getTableOrMetaException(String tableName, Table.TableType tableType) throws MetaNotFoundException {
        Table table = this.getTableOrMetaException(tableName);
        if (table.getType() != tableType) {
            throw new MetaNotFoundException("table type is not " + (Object)((Object)tableType) + ", tableName=" + tableName + ", type=" + (Object)((Object)table.getType()));
        }
        return (T)table;
    }

    public <T extends Table> T getTableOrMetaException(long tableId, Table.TableType tableType) throws MetaNotFoundException {
        Table table = this.getTableOrMetaException(tableId);
        if (table.getType() != tableType) {
            throw new MetaNotFoundException("table type is not " + (Object)((Object)tableType) + ", tableId=" + tableId + ", type=" + (Object)((Object)table.getType()));
        }
        return (T)table;
    }

    public Table getTableOrDdlException(String tableName) throws DdlException {
        return this.getTableOrException(tableName, (String t) -> new DdlException(ErrorCode.ERR_BAD_TABLE_ERROR.formatErrorMsg(t)));
    }

    public OlapTable getOlapTableOrDdlException(String tableName) throws DdlException {
        Table table = this.getTableOrDdlException(tableName);
        if (!(table instanceof OlapTable)) {
            throw new DdlException(ErrorCode.ERR_NOT_OLAP_TABLE.formatErrorMsg(tableName));
        }
        return (OlapTable)table;
    }

    public Table getTableOrDdlException(long tableId) throws DdlException {
        return this.getTableOrException(tableId, (Long t) -> new DdlException(ErrorCode.ERR_BAD_TABLE_ERROR.formatErrorMsg(t)));
    }

    public Table getTableOrAnalysisException(String tableName) throws AnalysisException {
        return this.getTableOrException(tableName, (String t) -> new AnalysisException(ErrorCode.ERR_UNKNOWN_TABLE.formatErrorMsg(t, this.fullQualifiedName)));
    }

    public OlapTable getOlapTableOrAnalysisException(String tableName) throws AnalysisException {
        Table table = this.getTableOrAnalysisException(tableName);
        if (!(table instanceof OlapTable)) {
            throw new AnalysisException(ErrorCode.ERR_NOT_OLAP_TABLE.formatErrorMsg(tableName));
        }
        return (OlapTable)table;
    }

    public Table getTableOrAnalysisException(long tableId) throws AnalysisException {
        return this.getTableOrException(tableId, (Long t) -> new AnalysisException(ErrorCode.ERR_BAD_TABLE_ERROR.formatErrorMsg(t)));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int getMaxReplicationNum() {
        short ret = 0;
        this.readLock();
        try {
            for (Table table : this.idToTable.values()) {
                if (table.getType() != Table.TableType.OLAP) continue;
                OlapTable olapTable = (OlapTable)table;
                table.readLock();
                try {
                    for (Partition partition : olapTable.getAllPartitions()) {
                        short replicationNum = olapTable.getPartitionInfo().getReplicaAllocation(partition.getId()).getTotalReplicaNum();
                        if (ret >= replicationNum) continue;
                        ret = replicationNum;
                    }
                }
                finally {
                    table.readUnlock();
                }
            }
        }
        finally {
            this.readUnlock();
        }
        return ret;
    }

    public static Database read(DataInput in) throws IOException {
        Database db = new Database();
        db.readFields(in);
        return db;
    }

    @Override
    public String getSignature(int signatureVersion) {
        StringBuilder sb = new StringBuilder(signatureVersion);
        sb.append(this.fullQualifiedName);
        String md5 = DigestUtils.md5Hex((String)sb.toString());
        LOG.debug("get signature of database {}: {}. signature string: {}", (Object)this.fullQualifiedName, (Object)md5, (Object)sb.toString());
        return md5;
    }

    @Override
    public void write(DataOutput out) throws IOException {
        super.write(out);
        out.writeLong(this.id);
        Text.writeString((DataOutput)out, (String)this.fullQualifiedName);
        int numTables = this.nameToTable.size();
        out.writeInt(numTables);
        for (Map.Entry<String, Table> entry : this.nameToTable.entrySet()) {
            entry.getValue().write(out);
        }
        out.writeLong(this.dataQuotaBytes);
        Text.writeString((DataOutput)out, (String)this.clusterName);
        Text.writeString((DataOutput)out, (String)this.dbState.name());
        Text.writeString((DataOutput)out, (String)this.attachDbName);
        out.writeInt(this.name2Function.size());
        for (Map.Entry<String, Table> entry : this.name2Function.entrySet()) {
            Text.writeString((DataOutput)out, (String)entry.getKey());
            out.writeInt(((ImmutableList)entry.getValue()).size());
            for (Function function : (ImmutableList)entry.getValue()) {
                function.write(out);
            }
        }
        this.dbEncryptKey.write(out);
        out.writeLong(this.replicaQuotaSize);
        this.dbProperties.write(out);
    }

    @Override
    public void readFields(DataInput in) throws IOException {
        super.readFields(in);
        this.id = in.readLong();
        this.fullQualifiedName = Text.readString((DataInput)in);
        int numTables = in.readInt();
        for (int i = 0; i < numTables; ++i) {
            Table table = Table.read(in);
            String tableName = table.getName();
            this.nameToTable.put(tableName, table);
            this.idToTable.put(table.getId(), table);
            this.lowerCaseToTableName.put(tableName.toLowerCase(), tableName);
        }
        this.dataQuotaBytes = in.readLong();
        this.clusterName = Text.readString((DataInput)in);
        this.dbState = DbState.valueOf(Text.readString((DataInput)in));
        this.attachDbName = Text.readString((DataInput)in);
        int numEntries = in.readInt();
        for (int i = 0; i < numEntries; ++i) {
            String name = Text.readString((DataInput)in);
            ImmutableList.Builder builder = ImmutableList.builder();
            int numFunctions = in.readInt();
            for (int j = 0; j < numFunctions; ++j) {
                builder.add((Object)Function.read(in));
            }
            this.name2Function.put(name, (ImmutableList<Function>)builder.build());
        }
        if (Catalog.getCurrentCatalogJournalVersion() >= 102) {
            this.dbEncryptKey = DatabaseEncryptKey.read(in);
        }
        this.replicaQuotaSize = in.readLong();
        if (Catalog.getCurrentCatalogJournalVersion() >= 105) {
            this.dbProperties = DatabaseProperty.read(in);
        }
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof Database)) {
            return false;
        }
        Database database = (Database)obj;
        if (this.idToTable != database.idToTable) {
            if (this.idToTable.size() != database.idToTable.size()) {
                return false;
            }
            for (Map.Entry<Long, Table> entry : this.idToTable.entrySet()) {
                long key = entry.getKey();
                if (!database.idToTable.containsKey(key)) {
                    return false;
                }
                if (entry.getValue().equals(database.idToTable.get(key))) continue;
                return false;
            }
        }
        return this.id == database.id && this.fullQualifiedName.equals(database.fullQualifiedName) && this.dataQuotaBytes == database.dataQuotaBytes;
    }

    public String getClusterName() {
        return this.clusterName;
    }

    public void setClusterName(String clusterName) {
        this.clusterName = clusterName;
    }

    public DbState getDbState() {
        return this.dbState;
    }

    public void setDbState(DbState dbState) {
        if (dbState == null) {
            return;
        }
        this.dbState = dbState;
    }

    public void setAttachDb(String name) {
        this.attachDbName = name;
    }

    public String getAttachDb() {
        return this.attachDbName;
    }

    public void setName(String name) {
        this.fullQualifiedName = name;
    }

    public synchronized void addFunction(Function function) throws UserException {
        this.addFunctionImpl(function, false);
        Catalog.getCurrentCatalog().getEditLog().logAddFunction(function);
    }

    public synchronized void replayAddFunction(Function function) {
        try {
            this.addFunctionImpl(function, true);
        }
        catch (UserException e) {
            Preconditions.checkArgument((boolean)false);
        }
    }

    private void addFunctionImpl(Function function, boolean isReplay) throws UserException {
        String functionName = function.getFunctionName().getFunction();
        List existFuncs = (List)this.name2Function.get(functionName);
        if (!isReplay) {
            if (existFuncs != null) {
                for (Function existFunc : existFuncs) {
                    if (!function.compare(existFunc, Function.CompareMode.IS_IDENTICAL)) continue;
                    throw new UserException("function already exists");
                }
            }
            long functionId = Catalog.getCurrentCatalog().getNextId();
            function.setId(functionId);
        }
        ImmutableList.Builder builder = ImmutableList.builder();
        if (existFuncs != null) {
            builder.addAll((Iterable)existFuncs);
        }
        builder.add((Object)function);
        this.name2Function.put(functionName, (ImmutableList<Function>)builder.build());
    }

    public synchronized void dropFunction(FunctionSearchDesc function) throws UserException {
        this.dropFunctionImpl(function);
        Catalog.getCurrentCatalog().getEditLog().logDropFunction(function);
    }

    public synchronized void replayDropFunction(FunctionSearchDesc functionSearchDesc) {
        try {
            this.dropFunctionImpl(functionSearchDesc);
        }
        catch (UserException e) {
            Preconditions.checkArgument((boolean)false);
        }
    }

    private void dropFunctionImpl(FunctionSearchDesc function) throws UserException {
        String functionName = function.getName().getFunction();
        List existFuncs = (List)this.name2Function.get(functionName);
        if (existFuncs == null) {
            throw new UserException("Unknown function, function=" + function.toString());
        }
        boolean isFound = false;
        ImmutableList.Builder builder = ImmutableList.builder();
        for (Function existFunc : existFuncs) {
            if (function.isIdentical(existFunc)) {
                isFound = true;
                continue;
            }
            builder.add((Object)existFunc);
        }
        if (!isFound) {
            throw new UserException("Unknown function, function=" + function.toString());
        }
        ImmutableList newFunctions = builder.build();
        if (newFunctions.isEmpty()) {
            this.name2Function.remove(functionName);
        } else {
            this.name2Function.put(functionName, (ImmutableList<Function>)newFunctions);
        }
    }

    public synchronized Function getFunction(Function desc, Function.CompareMode mode) {
        List fns = (List)this.name2Function.get(desc.getFunctionName().getFunction());
        if (fns == null) {
            return null;
        }
        return Function.getFunction(fns, desc, mode);
    }

    public synchronized Function getFunction(FunctionSearchDesc function) throws AnalysisException {
        String functionName = function.getName().getFunction();
        List existFuncs = (List)this.name2Function.get(functionName);
        if (existFuncs == null) {
            throw new AnalysisException("Unknown function, function=" + function.toString());
        }
        for (Function existFunc : existFuncs) {
            if (!function.isIdentical(existFunc)) continue;
            return existFunc;
        }
        throw new AnalysisException("Unknown function, function=" + function.toString());
    }

    public synchronized List<Function> getFunctions() {
        ArrayList functions = Lists.newArrayList();
        for (Map.Entry entry : this.name2Function.entrySet()) {
            functions.addAll((Collection)entry.getValue());
        }
        return functions;
    }

    public boolean isInfoSchemaDb() {
        return ClusterNamespace.getNameFromFullName(this.fullQualifiedName).equalsIgnoreCase("information_schema");
    }

    public synchronized void addEncryptKey(EncryptKey encryptKey) throws UserException {
        this.addEncryptKeyImpl(encryptKey, false);
        Catalog.getCurrentCatalog().getEditLog().logAddEncryptKey(encryptKey);
    }

    public synchronized void replayAddEncryptKey(EncryptKey encryptKey) {
        try {
            this.addEncryptKeyImpl(encryptKey, true);
        }
        catch (UserException e) {
            Preconditions.checkArgument((boolean)false);
        }
    }

    private void addEncryptKeyImpl(EncryptKey encryptKey, boolean isReplay) throws UserException {
        String keyName = encryptKey.getEncryptKeyName().getKeyName();
        EncryptKey existKey = (EncryptKey)this.dbEncryptKey.getName2EncryptKey().get(keyName);
        if (!isReplay && existKey != null && existKey.isIdentical(encryptKey)) {
            throw new UserException("encryptKey [" + existKey.getEncryptKeyName().toString() + "] already exists");
        }
        this.dbEncryptKey.getName2EncryptKey().put(keyName, encryptKey);
    }

    public synchronized void dropEncryptKey(EncryptKeySearchDesc encryptKeySearchDesc) throws UserException {
        this.dropEncryptKeyImpl(encryptKeySearchDesc);
        Catalog.getCurrentCatalog().getEditLog().logDropEncryptKey(encryptKeySearchDesc);
    }

    public synchronized void replayDropEncryptKey(EncryptKeySearchDesc encryptKeySearchDesc) {
        try {
            this.dropEncryptKeyImpl(encryptKeySearchDesc);
        }
        catch (UserException e) {
            Preconditions.checkArgument((boolean)false);
        }
    }

    private void dropEncryptKeyImpl(EncryptKeySearchDesc encryptKeySearchDesc) throws UserException {
        String keyName = encryptKeySearchDesc.getKeyEncryptKeyName().getKeyName();
        EncryptKey existKey = (EncryptKey)this.dbEncryptKey.getName2EncryptKey().get(keyName);
        if (existKey == null) {
            throw new UserException("Unknown encryptKey, encryptKey=" + encryptKeySearchDesc.toString());
        }
        boolean isFound = false;
        if (encryptKeySearchDesc.isIdentical(existKey)) {
            isFound = true;
        }
        if (!isFound) {
            throw new UserException("Unknown encryptKey, encryptKey=" + encryptKeySearchDesc.toString());
        }
        this.dbEncryptKey.getName2EncryptKey().remove(keyName);
    }

    public synchronized List<EncryptKey> getEncryptKeys() {
        ArrayList encryptKeys = Lists.newArrayList();
        for (Map.Entry entry : this.dbEncryptKey.getName2EncryptKey().entrySet()) {
            encryptKeys.add((EncryptKey)entry.getValue());
        }
        return encryptKeys;
    }

    public synchronized EncryptKey getEncryptKey(String keyName) {
        if (this.dbEncryptKey.getName2EncryptKey().containsKey(keyName)) {
            return (EncryptKey)this.dbEncryptKey.getName2EncryptKey().get(keyName);
        }
        return null;
    }

    public static enum DbState {
        NORMAL,
        LINK,
        MOVE;

    }
}

