/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.backup;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.time.Duration;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import org.apache.commons.collections.map.CaseInsensitiveMap;
import org.apache.commons.lang3.StringUtils;
import org.apache.doris.analysis.StorageBackend;
import org.apache.doris.backup.BlobStorage;
import org.apache.doris.backup.RemoteFile;
import org.apache.doris.backup.Status;
import org.apache.doris.common.UserException;
import org.apache.doris.common.util.S3URI;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.http.client.utils.URIBuilder;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import software.amazon.awssdk.auth.credentials.AwsBasicCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProvider;
import software.amazon.awssdk.auth.credentials.StaticCredentialsProvider;
import software.amazon.awssdk.auth.signer.AwsS3V4Signer;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkAdvancedClientOption;
import software.amazon.awssdk.core.retry.RetryPolicy;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.EqualJitterBackoffStrategy;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.S3ClientBuilder;
import software.amazon.awssdk.services.s3.S3Configuration;
import software.amazon.awssdk.services.s3.model.CopyObjectRequest;
import software.amazon.awssdk.services.s3.model.DeleteObjectRequest;
import software.amazon.awssdk.services.s3.model.DeleteObjectResponse;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.GetObjectResponse;
import software.amazon.awssdk.services.s3.model.HeadObjectRequest;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.services.s3.model.PutObjectResponse;
import software.amazon.awssdk.services.s3.model.S3Exception;

public class S3Storage
extends BlobStorage {
    public static final String S3_PROPERTIES_PREFIX = "AWS";
    public static final String S3_AK = "AWS_ACCESS_KEY";
    public static final String S3_SK = "AWS_SECRET_KEY";
    public static final String S3_ENDPOINT = "AWS_ENDPOINT";
    public static final String S3_REGION = "AWS_REGION";
    public static final String USE_PATH_STYLE = "use_path_style";
    private static final Logger LOG = LogManager.getLogger(S3Storage.class);
    private final CaseInsensitiveMap caseInsensitiveProperties = new CaseInsensitiveMap();
    private S3Client client = null;
    private boolean forceHostedStyle = false;

    public S3Storage(Map<String, String> properties) {
        this.setProperties(properties);
        this.setType(StorageBackend.StorageType.S3);
        this.setName(StorageBackend.StorageType.S3.name());
    }

    @Override
    public void setProperties(Map<String, String> properties) {
        super.setProperties(properties);
        this.caseInsensitiveProperties.putAll(properties);
        this.forceHostedStyle = !this.caseInsensitiveProperties.get((Object)S3_ENDPOINT).toString().toLowerCase().startsWith("s3") ? !this.caseInsensitiveProperties.getOrDefault((Object)USE_PATH_STYLE, (Object)"false").toString().equalsIgnoreCase("true") : false;
    }

    public static void checkS3(CaseInsensitiveMap caseInsensitiveProperties) throws UserException {
        if (!caseInsensitiveProperties.containsKey((Object)S3_REGION)) {
            throw new UserException("AWS_REGION not found.");
        }
        if (!caseInsensitiveProperties.containsKey((Object)S3_ENDPOINT)) {
            throw new UserException("AWS_ENDPOINT not found.");
        }
        if (!caseInsensitiveProperties.containsKey((Object)S3_AK)) {
            throw new UserException("AWS_ACCESS_KEY not found.");
        }
        if (!caseInsensitiveProperties.containsKey((Object)S3_SK)) {
            throw new UserException("AWS_SECRET_KEY not found.");
        }
    }

    private S3Client getClient(String bucket) throws UserException {
        if (this.client == null) {
            S3Storage.checkS3(this.caseInsensitiveProperties);
            URI tmpEndpoint = URI.create(this.caseInsensitiveProperties.get((Object)S3_ENDPOINT).toString());
            AwsBasicCredentials awsBasic = AwsBasicCredentials.create((String)this.caseInsensitiveProperties.get((Object)S3_AK).toString(), (String)this.caseInsensitiveProperties.get((Object)S3_SK).toString());
            StaticCredentialsProvider scp = StaticCredentialsProvider.create((AwsCredentials)awsBasic);
            EqualJitterBackoffStrategy backoffStrategy = EqualJitterBackoffStrategy.builder().baseDelay(Duration.ofSeconds(1L)).maxBackoffTime(Duration.ofMinutes(1L)).build();
            RetryPolicy retryPolicy = RetryPolicy.builder().numRetries(Integer.valueOf(3)).backoffStrategy((BackoffStrategy)backoffStrategy).build();
            ClientOverrideConfiguration clientConf = (ClientOverrideConfiguration)ClientOverrideConfiguration.builder().retryPolicy(retryPolicy).putAdvancedOption(SdkAdvancedClientOption.SIGNER, (Object)AwsS3V4Signer.create()).build();
            URI endpoint = StringUtils.isEmpty((CharSequence)bucket) ? tmpEndpoint : URI.create(new URIBuilder(tmpEndpoint).setHost(bucket + "." + tmpEndpoint.getHost()).toString());
            this.client = (S3Client)((S3ClientBuilder)((S3ClientBuilder)((S3ClientBuilder)((S3ClientBuilder)((S3ClientBuilder)S3Client.builder().endpointOverride(endpoint)).credentialsProvider((AwsCredentialsProvider)scp)).region(Region.of((String)this.caseInsensitiveProperties.get((Object)S3_REGION).toString()))).overrideConfiguration(clientConf)).serviceConfiguration((S3Configuration)S3Configuration.builder().chunkedEncodingEnabled(Boolean.valueOf(false)).pathStyleAccessEnabled(Boolean.valueOf(false)).build())).build();
        }
        return this.client;
    }

    @Override
    public Status downloadWithFileSize(String remoteFilePath, String localFilePath, long fileSize) {
        long start = System.currentTimeMillis();
        File localFile = new File(localFilePath);
        if (localFile.exists()) {
            try {
                Files.walk(Paths.get(localFilePath, new String[0]), FileVisitOption.FOLLOW_LINKS).sorted(Comparator.reverseOrder()).map(java.nio.file.Path::toFile).forEach(File::delete);
            }
            catch (IOException e) {
                return new Status(Status.ErrCode.COMMON_ERROR, "failed to delete exist local file: " + localFilePath);
            }
        }
        try {
            S3URI uri = S3URI.create(remoteFilePath, this.forceHostedStyle);
            GetObjectResponse response = this.getClient(uri.getVirtualBucket()).getObject((GetObjectRequest)GetObjectRequest.builder().bucket(uri.getBucket()).key(uri.getKey()).build(), localFile.toPath());
            if (localFile.length() == fileSize) {
                LOG.info("finished to download from {} to {} with size: {}. cost {} ms", (Object)remoteFilePath, (Object)localFilePath, (Object)fileSize, (Object)(System.currentTimeMillis() - start));
                return Status.OK;
            }
            return new Status(Status.ErrCode.COMMON_ERROR, response.toString());
        }
        catch (S3Exception s3Exception) {
            return new Status(Status.ErrCode.COMMON_ERROR, "get file from s3 error: " + s3Exception.awsErrorDetails().errorMessage());
        }
        catch (UserException ue) {
            LOG.warn("connect to s3 failed: ", (Throwable)ue);
            return new Status(Status.ErrCode.COMMON_ERROR, "connect to s3 failed: " + ue.getMessage());
        }
        catch (Exception e) {
            return new Status(Status.ErrCode.COMMON_ERROR, e.toString());
        }
    }

    @Override
    public Status directUpload(String content, String remoteFile) {
        try {
            S3URI uri = S3URI.create(remoteFile, this.forceHostedStyle);
            PutObjectResponse response = this.getClient(uri.getVirtualBucket()).putObject((PutObjectRequest)PutObjectRequest.builder().bucket(uri.getBucket()).key(uri.getKey()).build(), RequestBody.fromBytes((byte[])content.getBytes()));
            LOG.info("upload content success: " + response.eTag());
            return Status.OK;
        }
        catch (S3Exception e) {
            LOG.error("write content failed:", (Throwable)e);
            return new Status(Status.ErrCode.COMMON_ERROR, "write content failed: " + e.getMessage());
        }
        catch (UserException ue) {
            LOG.error("connect to s3 failed: ", (Throwable)ue);
            return new Status(Status.ErrCode.COMMON_ERROR, "connect to s3 failed: " + ue.getMessage());
        }
    }

    public Status copy(String origFilePath, String destFilePath) {
        try {
            S3URI origUri = S3URI.create(origFilePath);
            S3URI descUri = S3URI.create(destFilePath, this.forceHostedStyle);
            this.getClient(descUri.getVirtualBucket()).copyObject((CopyObjectRequest)CopyObjectRequest.builder().copySource(origUri.getBucket() + "/" + origUri.getKey()).destinationBucket(descUri.getBucket()).destinationKey(descUri.getKey()).build());
            return Status.OK;
        }
        catch (S3Exception e) {
            LOG.error("copy file failed: ", (Throwable)e);
            return new Status(Status.ErrCode.COMMON_ERROR, "copy file failed: " + e.getMessage());
        }
        catch (UserException ue) {
            LOG.error("copy to s3 failed: ", (Throwable)ue);
            return new Status(Status.ErrCode.COMMON_ERROR, "connect to s3 failed: " + ue.getMessage());
        }
    }

    @Override
    public Status upload(String localPath, String remotePath) {
        try {
            S3URI uri = S3URI.create(remotePath, this.forceHostedStyle);
            PutObjectResponse response = this.getClient(uri.getVirtualBucket()).putObject((PutObjectRequest)PutObjectRequest.builder().bucket(uri.getBucket()).key(uri.getKey()).build(), RequestBody.fromFile((File)new File(localPath)));
            LOG.info("upload file " + localPath + " success: " + response.eTag());
            return Status.OK;
        }
        catch (S3Exception e) {
            LOG.error("write file failed:", (Throwable)e);
            return new Status(Status.ErrCode.COMMON_ERROR, "write file failed: " + e.getMessage());
        }
        catch (UserException ue) {
            LOG.error("connect to s3 failed: ", (Throwable)ue);
            return new Status(Status.ErrCode.COMMON_ERROR, "connect to s3 failed: " + ue.getMessage());
        }
    }

    @Override
    public Status rename(String origFilePath, String destFilePath) {
        Status status = this.copy(origFilePath, destFilePath);
        if (status.ok()) {
            return this.delete(origFilePath);
        }
        return status;
    }

    @Override
    public Status delete(String remotePath) {
        try {
            S3URI uri = S3URI.create(remotePath, this.forceHostedStyle);
            DeleteObjectResponse response = this.getClient(uri.getVirtualBucket()).deleteObject((DeleteObjectRequest)DeleteObjectRequest.builder().bucket(uri.getBucket()).key(uri.getKey()).build());
            LOG.info("delete file " + remotePath + " success: " + response.toString());
            return Status.OK;
        }
        catch (S3Exception e) {
            LOG.error("delete file failed: ", (Throwable)e);
            if (e.statusCode() == 404) {
                return Status.OK;
            }
            return new Status(Status.ErrCode.COMMON_ERROR, "delete file failed: " + e.getMessage());
        }
        catch (UserException ue) {
            LOG.error("connect to s3 failed: ", (Throwable)ue);
            return new Status(Status.ErrCode.COMMON_ERROR, "connect to s3 failed: " + ue.getMessage());
        }
    }

    @Override
    public Status list(String remotePath, List<RemoteFile> result) {
        return this.list(remotePath, result, true);
    }

    public Status list(String remotePath, List<RemoteFile> result, boolean fileNameOnly) {
        try {
            S3Storage.checkS3(this.caseInsensitiveProperties);
            Configuration conf = new Configuration();
            String s3AK = this.caseInsensitiveProperties.get((Object)S3_AK).toString();
            String s3Sk = this.caseInsensitiveProperties.get((Object)S3_SK).toString();
            String s3Endpoint = this.caseInsensitiveProperties.get((Object)S3_ENDPOINT).toString();
            System.setProperty("com.amazonaws.services.s3.enableV4", "true");
            conf.set("fs.s3a.access.key", s3AK);
            conf.set("fs.s3a.secret.key", s3Sk);
            conf.set("fs.s3a.endpoint", s3Endpoint);
            conf.set("fs.s3.impl.disable.cache", "true");
            conf.set("fs.s3.impl", "org.apache.hadoop.fs.s3a.S3AFileSystem");
            conf.set("fs.s3a.path.style.access", this.forceHostedStyle ? "false" : "true");
            conf.set("fs.s3a.attempts.maximum", "2");
            FileSystem s3AFileSystem = FileSystem.get((URI)new URI(remotePath), (Configuration)conf);
            Path pathPattern = new Path(remotePath);
            FileStatus[] files = s3AFileSystem.globStatus(pathPattern);
            if (files == null) {
                return Status.OK;
            }
            for (FileStatus fileStatus : files) {
                RemoteFile remoteFile = new RemoteFile(fileNameOnly ? fileStatus.getPath().getName() : fileStatus.getPath().toString(), !fileStatus.isDirectory(), fileStatus.isDirectory() ? -1L : fileStatus.getLen());
                result.add(remoteFile);
            }
        }
        catch (FileNotFoundException e) {
            LOG.info("file not found: " + e.getMessage());
            return new Status(Status.ErrCode.NOT_FOUND, "file not found: " + e.getMessage());
        }
        catch (Exception e) {
            LOG.error("errors while get file status ", (Throwable)e);
            return new Status(Status.ErrCode.COMMON_ERROR, "errors while get file status " + e.getMessage());
        }
        return Status.OK;
    }

    @Override
    public Status makeDir(String remotePath) {
        if (!remotePath.endsWith("/")) {
            remotePath = remotePath + "/";
        }
        try {
            S3URI uri = S3URI.create(remotePath, this.forceHostedStyle);
            PutObjectResponse response = this.getClient(uri.getVirtualBucket()).putObject((PutObjectRequest)PutObjectRequest.builder().bucket(uri.getBucket()).key(uri.getKey()).build(), RequestBody.empty());
            LOG.info("makeDir success: " + response.eTag());
            return Status.OK;
        }
        catch (S3Exception e) {
            LOG.error("makeDir failed:", (Throwable)e);
            return new Status(Status.ErrCode.COMMON_ERROR, "makeDir failed: " + e.getMessage());
        }
        catch (UserException ue) {
            LOG.error("connect to s3 failed: ", (Throwable)ue);
            return new Status(Status.ErrCode.COMMON_ERROR, "connect to s3 failed: " + ue.getMessage());
        }
    }

    @Override
    public Status checkPathExist(String remotePath) {
        try {
            S3URI uri = S3URI.create(remotePath, this.forceHostedStyle);
            this.getClient(uri.getVirtualBucket()).headObject((HeadObjectRequest)HeadObjectRequest.builder().bucket(uri.getBucket()).key(uri.getKey()).build());
            return Status.OK;
        }
        catch (S3Exception e) {
            if (e.statusCode() == 404) {
                return new Status(Status.ErrCode.NOT_FOUND, "remote path does not exist: " + remotePath);
            }
            LOG.warn("headObject failed:", (Throwable)e);
            return new Status(Status.ErrCode.COMMON_ERROR, "headObject failed: " + e.getMessage());
        }
        catch (UserException ue) {
            LOG.warn("connect to s3 failed: ", (Throwable)ue);
            return new Status(Status.ErrCode.COMMON_ERROR, "connect to s3 failed: " + ue.getMessage());
        }
    }

    @Override
    public StorageBackend.StorageType getStorageType() {
        return StorageBackend.StorageType.S3;
    }
}

