/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.backup;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.doris.analysis.StorageBackend;
import org.apache.doris.backup.BackupHandler;
import org.apache.doris.backup.BackupJobInfo;
import org.apache.doris.backup.BackupMeta;
import org.apache.doris.backup.BlobStorage;
import org.apache.doris.backup.BrokerStorage;
import org.apache.doris.backup.RemoteFile;
import org.apache.doris.backup.S3Storage;
import org.apache.doris.backup.SnapshotInfo;
import org.apache.doris.backup.Status;
import org.apache.doris.catalog.Catalog;
import org.apache.doris.catalog.FsBroker;
import org.apache.doris.common.AnalysisException;
import org.apache.doris.common.FeConstants;
import org.apache.doris.common.Pair;
import org.apache.doris.common.io.Text;
import org.apache.doris.common.io.Writable;
import org.apache.doris.common.util.TimeUtils;
import org.apache.doris.system.Backend;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.json.simple.JSONObject;
import org.json.simple.JSONValue;

public class Repository
implements Writable {
    public static final String PREFIX_REPO = "__palo_repository_";
    public static final String PREFIX_SNAPSHOT_DIR = "__ss_";
    public static final String PREFIX_DB = "__db_";
    public static final String PREFIX_TBL = "__tbl_";
    public static final String PREFIX_PART = "__part_";
    public static final String PREFIX_IDX = "__idx_";
    public static final String PREFIX_COMMON = "__";
    public static final String PREFIX_JOB_INFO = "__info_";
    public static final String SUFFIX_TMP_FILE = "part";
    public static final String FILE_REPO_INFO = "__repo_info";
    public static final String FILE_META_INFO = "__meta";
    public static final String DIR_SNAPSHOT_CONTENT = "__ss_content";
    private static final Logger LOG = LogManager.getLogger(Repository.class);
    private static final String PATH_DELIMITER = "/";
    private static final String CHECKSUM_SEPARATOR = ".";
    private long id;
    private String name;
    private String errMsg;
    private long createTime;
    private boolean isReadOnly;
    private String location;
    private BlobStorage storage;

    private Repository() {
    }

    public Repository(long id, String name, boolean isReadOnly, String location, BlobStorage storage) {
        this.id = id;
        this.name = name;
        this.isReadOnly = isReadOnly;
        this.location = location;
        this.storage = storage;
        this.createTime = System.currentTimeMillis();
    }

    private static String jobInfoFileNameWithTimestamp(long createTime) {
        if (createTime == -1L) {
            return PREFIX_JOB_INFO;
        }
        return PREFIX_JOB_INFO + TimeUtils.longToTimeString(createTime, new SimpleDateFormat("yyyy-MM-dd-HH-mm-ss"));
    }

    private static String joinPrefix(String prefix, Object name) {
        return prefix + name;
    }

    private static String disjoinPrefix(String prefix, String nameWithPrefix) {
        return nameWithPrefix.substring(prefix.length());
    }

    private static String assembleFileNameWithSuffix(String filePath, String md5sum) {
        return filePath + CHECKSUM_SEPARATOR + md5sum;
    }

    public static Pair<String, String> decodeFileNameWithChecksum(String fileNameWithChecksum) {
        int index = fileNameWithChecksum.lastIndexOf(CHECKSUM_SEPARATOR);
        if (index == -1) {
            return null;
        }
        String fileName = fileNameWithChecksum.substring(0, index);
        String md5sum = fileNameWithChecksum.substring(index + CHECKSUM_SEPARATOR.length());
        if (md5sum.length() != 32) {
            return null;
        }
        return Pair.create(fileName, md5sum);
    }

    public static String replaceFileNameWithChecksumFileName(String origPath, String fileNameWithChecksum) {
        return origPath.substring(0, origPath.lastIndexOf(PATH_DELIMITER) + 1) + fileNameWithChecksum;
    }

    public static Repository read(DataInput in) throws IOException {
        Repository repo = new Repository();
        repo.readFields(in);
        return repo;
    }

    public long getId() {
        return this.id;
    }

    public String getName() {
        return this.name;
    }

    public boolean isReadOnly() {
        return this.isReadOnly;
    }

    public String getLocation() {
        return this.location;
    }

    public String getErrorMsg() {
        return this.errMsg;
    }

    public BlobStorage getStorage() {
        return this.storage;
    }

    public long getCreateTime() {
        return this.createTime;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Status initRepository() {
        ArrayList remoteFiles;
        String repoInfoFilePath = this.assembleRepoInfoFilePath();
        Status st = this.storage.list(repoInfoFilePath, remoteFiles = Lists.newArrayList());
        if (!st.ok()) {
            return st;
        }
        if (remoteFiles.size() == 1) {
            RemoteFile remoteFile = (RemoteFile)remoteFiles.get(0);
            if (!remoteFile.isFile()) {
                return new Status(Status.ErrCode.COMMON_ERROR, "the existing repo info is not a file");
            }
            String localFilePath = BackupHandler.BACKUP_ROOT_DIR + "/tmp_info_" + System.currentTimeMillis();
            try {
                st = this.storage.downloadWithFileSize(repoInfoFilePath, localFilePath, remoteFile.getSize());
                if (!st.ok()) {
                    Status status = st;
                    return status;
                }
                byte[] bytes = Files.readAllBytes(Paths.get(localFilePath, new String[0]));
                String json = new String(bytes, StandardCharsets.UTF_8);
                JSONObject root = (JSONObject)JSONValue.parse((String)json);
                this.name = (String)root.get((Object)"name");
                this.createTime = TimeUtils.timeStringToLong((String)root.get((Object)"create_time"));
                if (this.createTime == -1L) {
                    Status status = new Status(Status.ErrCode.COMMON_ERROR, "failed to parse create time of repository: " + root.get((Object)"create_time"));
                    return status;
                }
                Status status = Status.OK;
                return status;
            }
            catch (IOException e) {
                Status status = new Status(Status.ErrCode.COMMON_ERROR, "failed to read repo info file: " + e.getMessage());
                return status;
            }
            finally {
                File localFile = new File(localFilePath);
                localFile.delete();
            }
        }
        if (remoteFiles.size() > 1) {
            return new Status(Status.ErrCode.COMMON_ERROR, "Invalid repository dir. expected one repo info file. get more: " + remoteFiles);
        }
        JSONObject root = new JSONObject();
        root.put((Object)"name", (Object)this.name);
        root.put((Object)"create_time", (Object)TimeUtils.longToTimeString(this.createTime));
        String repoInfoContent = root.toString();
        return this.storage.directUpload(repoInfoContent, repoInfoFilePath);
    }

    public String assembleRepoInfoFilePath() {
        return Joiner.on((String)PATH_DELIMITER).join((Object)this.location, (Object)Repository.joinPrefix(PREFIX_REPO, this.name), new Object[]{FILE_REPO_INFO});
    }

    public String assembleMetaInfoFilePath(String label) {
        return Joiner.on((String)PATH_DELIMITER).join((Object)this.location, (Object)Repository.joinPrefix(PREFIX_REPO, this.name), new Object[]{Repository.joinPrefix(PREFIX_SNAPSHOT_DIR, label), FILE_META_INFO});
    }

    public String assembleJobInfoFilePath(String label, long createTime) {
        return Joiner.on((String)PATH_DELIMITER).join((Object)this.location, (Object)Repository.joinPrefix(PREFIX_REPO, this.name), new Object[]{Repository.joinPrefix(PREFIX_SNAPSHOT_DIR, label), Repository.jobInfoFileNameWithTimestamp(createTime)});
    }

    public String getRepoTabletPathBySnapshotInfo(String label, SnapshotInfo info) {
        String path = Joiner.on((String)PATH_DELIMITER).join((Object)this.location, (Object)Repository.joinPrefix(PREFIX_REPO, this.name), new Object[]{Repository.joinPrefix(PREFIX_SNAPSHOT_DIR, label), DIR_SNAPSHOT_CONTENT, Repository.joinPrefix(PREFIX_DB, info.getDbId()), Repository.joinPrefix(PREFIX_TBL, info.getTblId()), Repository.joinPrefix(PREFIX_PART, info.getPartitionId()), Repository.joinPrefix(PREFIX_IDX, info.getIndexId()), Repository.joinPrefix(PREFIX_COMMON, info.getTabletId())});
        try {
            return new URI(path).normalize().toString();
        }
        catch (URISyntaxException e) {
            LOG.warn("failed to normalize path: {}", (Object)path, (Object)e);
            return null;
        }
    }

    public String getRepoPath(String label, String childPath) {
        String path = Joiner.on((String)PATH_DELIMITER).join((Object)this.location, (Object)Repository.joinPrefix(PREFIX_REPO, this.name), new Object[]{Repository.joinPrefix(PREFIX_SNAPSHOT_DIR, label), DIR_SNAPSHOT_CONTENT, childPath});
        try {
            URI uri = new URI(path);
            return uri.normalize().toString();
        }
        catch (Exception e) {
            LOG.warn("Invalid path: " + path, (Throwable)e);
            return null;
        }
    }

    public boolean ping() {
        String path = this.location + PATH_DELIMITER + Repository.joinPrefix(PREFIX_REPO, this.name) + PATH_DELIMITER + FILE_REPO_INFO;
        try {
            URI checkUri = new URI(path);
            Status st = this.storage.checkPathExist(checkUri.normalize().toString());
            if (!st.ok()) {
                this.errMsg = TimeUtils.longToTimeString(System.currentTimeMillis()) + ": " + st.getErrMsg();
                return false;
            }
            this.errMsg = null;
            return true;
        }
        catch (URISyntaxException e) {
            this.errMsg = TimeUtils.longToTimeString(System.currentTimeMillis()) + ": Invalid path. " + path + ", error: " + e.getMessage();
            return false;
        }
    }

    public Status listSnapshots(List<String> snapshotNames) {
        ArrayList result;
        String listPath = Joiner.on((String)PATH_DELIMITER).join((Object)this.location, (Object)Repository.joinPrefix(PREFIX_REPO, this.name), new Object[]{PREFIX_SNAPSHOT_DIR}) + "*";
        Status st = this.storage.list(listPath, result = Lists.newArrayList());
        if (!st.ok()) {
            return st;
        }
        for (RemoteFile remoteFile : result) {
            if (remoteFile.isFile()) {
                LOG.debug("get snapshot path{} which is not a dir", (Object)remoteFile);
                continue;
            }
            snapshotNames.add(Repository.disjoinPrefix(PREFIX_SNAPSHOT_DIR, remoteFile.getName()));
        }
        return Status.OK;
    }

    public boolean prepareSnapshotInfo() {
        return false;
    }

    public String assembleRemoteSnapshotPath(String label, SnapshotInfo info) {
        String path = Joiner.on((String)PATH_DELIMITER).join((Object)this.location, (Object)Repository.joinPrefix(PREFIX_REPO, this.name), new Object[]{Repository.joinPrefix(PREFIX_SNAPSHOT_DIR, label), DIR_SNAPSHOT_CONTENT, Repository.joinPrefix(PREFIX_DB, info.getDbId()), Repository.joinPrefix(PREFIX_TBL, info.getTblId()), Repository.joinPrefix(PREFIX_PART, info.getPartitionId()), Repository.joinPrefix(PREFIX_IDX, info.getIndexId()), Repository.joinPrefix(PREFIX_COMMON, info.getTabletId()), Repository.joinPrefix(PREFIX_COMMON, info.getSchemaHash())});
        LOG.debug("get remote tablet snapshot path: {}", (Object)path);
        return path;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Status getSnapshotInfoFile(String label, String backupTimestamp, List<BackupJobInfo> infos) {
        String remoteInfoFilePath = this.assembleJobInfoFilePath(label, -1L) + backupTimestamp;
        File localInfoFile = new File(BackupHandler.BACKUP_ROOT_DIR + PATH_DELIMITER + "info_" + System.currentTimeMillis());
        try {
            Status st = this.download(remoteInfoFilePath, localInfoFile.getPath());
            if (!st.ok()) {
                Status status = st;
                return status;
            }
            BackupJobInfo jobInfo = BackupJobInfo.fromFile(localInfoFile.getAbsolutePath());
            infos.add(jobInfo);
        }
        catch (IOException e) {
            Status status = new Status(Status.ErrCode.COMMON_ERROR, "Failed to create job info from file: " + localInfoFile.getName() + ". msg: " + e.getMessage());
            return status;
        }
        finally {
            localInfoFile.delete();
        }
        return Status.OK;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Status getSnapshotMetaFile(String label, List<BackupMeta> backupMetas, int metaVersion) {
        String remoteMetaFilePath = this.assembleMetaInfoFilePath(label);
        File localMetaFile = new File(BackupHandler.BACKUP_ROOT_DIR + PATH_DELIMITER + "meta_" + System.currentTimeMillis());
        try {
            Status st = this.download(remoteMetaFilePath, localMetaFile.getAbsolutePath());
            if (!st.ok()) {
                Status status = st;
                return status;
            }
            BackupMeta backupMeta = BackupMeta.fromFile(localMetaFile.getAbsolutePath(), metaVersion);
            backupMetas.add(backupMeta);
        }
        catch (IOException e) {
            LOG.warn("failed to read backup meta from file", (Throwable)e);
            Status status = new Status(Status.ErrCode.COMMON_ERROR, "Failed create backup meta from file: " + localMetaFile.getAbsolutePath() + ", msg: " + e.getMessage());
            return status;
        }
        catch (IllegalArgumentException e) {
            LOG.warn("failed to set meta version", (Throwable)e);
            Status status = new Status(Status.ErrCode.COMMON_ERROR, e.getMessage());
            return status;
        }
        finally {
            localMetaFile.delete();
        }
        return Status.OK;
    }

    public Status upload(String localFilePath, String remoteFilePath) {
        File file = new File(localFilePath);
        String md5sum = null;
        try {
            md5sum = DigestUtils.md5Hex((InputStream)new FileInputStream(file));
        }
        catch (FileNotFoundException e) {
            return new Status(Status.ErrCode.NOT_FOUND, "file " + localFilePath + " does not exist");
        }
        catch (IOException e) {
            return new Status(Status.ErrCode.COMMON_ERROR, "failed to get md5sum of file: " + localFilePath);
        }
        Preconditions.checkState((!Strings.isNullOrEmpty((String)md5sum) ? 1 : 0) != 0);
        String finalRemotePath = Repository.assembleFileNameWithSuffix(remoteFilePath, md5sum);
        Status st = Status.OK;
        if (this.storage instanceof BrokerStorage) {
            String tmpRemotePath = Repository.assembleFileNameWithSuffix(remoteFilePath, SUFFIX_TMP_FILE);
            LOG.debug("get md5sum of file: {}. tmp remote path: {}. final remote path: {}", (Object)localFilePath, (Object)tmpRemotePath, (Object)finalRemotePath);
            st = this.storage.delete(tmpRemotePath);
            if (!st.ok()) {
                return st;
            }
            st = this.storage.delete(finalRemotePath);
            if (!st.ok()) {
                return st;
            }
            st = this.storage.upload(localFilePath, tmpRemotePath);
            if (!st.ok()) {
                return st;
            }
            st = this.storage.rename(tmpRemotePath, finalRemotePath);
            if (!st.ok()) {
                return st;
            }
        } else if (this.storage instanceof S3Storage) {
            LOG.debug("get md5sum of file: {}. final remote path: {}", (Object)localFilePath, (Object)finalRemotePath);
            st = this.storage.delete(finalRemotePath);
            if (!st.ok()) {
                return st;
            }
            st = this.storage.upload(localFilePath, finalRemotePath);
            if (!st.ok()) {
                return st;
            }
        }
        LOG.info("finished to upload local file {} to remote file: {}", (Object)localFilePath, (Object)finalRemotePath);
        return st;
    }

    public Status download(String remoteFilePath, String localFilePath) {
        ArrayList remoteFiles = Lists.newArrayList();
        Status status = this.storage.list(remoteFilePath + "*", remoteFiles);
        if (!status.ok()) {
            return status;
        }
        if (remoteFiles.size() != 1) {
            return new Status(Status.ErrCode.COMMON_ERROR, "Expected one file with path: " + remoteFilePath + ". get: " + remoteFiles.size());
        }
        if (!((RemoteFile)remoteFiles.get(0)).isFile()) {
            return new Status(Status.ErrCode.COMMON_ERROR, "Expected file with path: " + remoteFilePath + ". but get dir");
        }
        String remoteFilePathWithChecksum = Repository.replaceFileNameWithChecksumFileName(remoteFilePath, ((RemoteFile)remoteFiles.get(0)).getName());
        LOG.debug("get download filename with checksum: " + remoteFilePathWithChecksum);
        Pair<String, String> pair = Repository.decodeFileNameWithChecksum(remoteFilePathWithChecksum);
        if (pair == null) {
            return new Status(Status.ErrCode.COMMON_ERROR, "file name should contains checksum: " + remoteFilePathWithChecksum);
        }
        if (!remoteFilePath.endsWith((String)pair.first)) {
            return new Status(Status.ErrCode.COMMON_ERROR, "File does not exist: " + remoteFilePath);
        }
        String md5sum = (String)pair.second;
        status = this.storage.downloadWithFileSize(remoteFilePathWithChecksum, localFilePath, ((RemoteFile)remoteFiles.get(0)).getSize());
        if (!status.ok()) {
            return status;
        }
        String localMd5sum = null;
        try {
            localMd5sum = DigestUtils.md5Hex((InputStream)new FileInputStream(localFilePath));
        }
        catch (FileNotFoundException e) {
            return new Status(Status.ErrCode.NOT_FOUND, "file " + localFilePath + " does not exist");
        }
        catch (IOException e) {
            return new Status(Status.ErrCode.COMMON_ERROR, "failed to get md5sum of file: " + localFilePath);
        }
        if (!localMd5sum.equals(md5sum)) {
            return new Status(Status.ErrCode.BAD_FILE, "md5sum does not equal. local: " + localMd5sum + ", remote: " + md5sum);
        }
        return Status.OK;
    }

    public Status getBrokerAddress(Long beId, Catalog catalog, List<FsBroker> brokerAddrs) {
        Backend be = Catalog.getCurrentSystemInfo().getBackend(beId);
        if (be == null) {
            return new Status(Status.ErrCode.COMMON_ERROR, "backend " + beId + " is missing. failed to send upload snapshot task");
        }
        if (this.storage.getStorageType() != StorageBackend.StorageType.BROKER) {
            brokerAddrs.add(new FsBroker("127.0.0.1", 0));
            return Status.OK;
        }
        FsBroker brokerAddr = null;
        try {
            brokerAddr = catalog.getBrokerMgr().getBroker(((BrokerStorage)this.storage).getBrokerName(), be.getHost());
        }
        catch (AnalysisException e) {
            return new Status(Status.ErrCode.COMMON_ERROR, "failed to get address of broker " + ((BrokerStorage)this.storage).getBrokerName() + " when try to send upload snapshot task: " + e.getMessage());
        }
        if (brokerAddr == null) {
            return new Status(Status.ErrCode.COMMON_ERROR, "failed to get address of broker " + ((BrokerStorage)this.storage).getBrokerName() + " when try to send upload snapshot task");
        }
        brokerAddrs.add(brokerAddr);
        return Status.OK;
    }

    public List<String> getInfo() {
        ArrayList info = Lists.newArrayList();
        info.add(String.valueOf(this.id));
        info.add(this.name);
        info.add(TimeUtils.longToTimeString(this.createTime));
        info.add(String.valueOf(this.isReadOnly));
        info.add(this.location);
        info.add(this.storage.getType() != StorageBackend.StorageType.BROKER ? "-" : this.storage.getName());
        info.add(this.storage.getStorageType().name());
        info.add(this.errMsg == null ? FeConstants.null_string : this.errMsg);
        return info;
    }

    public List<List<String>> getSnapshotInfos(String snapshotName, String timestamp) throws AnalysisException {
        ArrayList snapshotInfos = Lists.newArrayList();
        if (Strings.isNullOrEmpty((String)snapshotName)) {
            ArrayList snapshotNames = Lists.newArrayList();
            Status status = this.listSnapshots(snapshotNames);
            if (!status.ok()) {
                throw new AnalysisException("Failed to list snapshot in repo: " + this.name + ", err: " + status.getErrMsg());
            }
            for (String ssName : snapshotNames) {
                List<String> info = this.getSnapshotInfo(ssName, null);
                snapshotInfos.add(info);
            }
        } else {
            List<String> info = this.getSnapshotInfo(snapshotName, timestamp);
            snapshotInfos.add(info);
        }
        return snapshotInfos;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<String> getSnapshotInfo(String snapshotName, String timestamp) {
        ArrayList info = Lists.newArrayList();
        if (Strings.isNullOrEmpty((String)timestamp)) {
            String infoFilePath = this.assembleJobInfoFilePath(snapshotName, -1L);
            LOG.debug("assemble infoFilePath: {}, snapshot: {}", (Object)infoFilePath, (Object)snapshotName);
            ArrayList results = Lists.newArrayList();
            Status st = this.storage.list(infoFilePath + "*", results);
            if (!st.ok()) {
                info.add(snapshotName);
                info.add(FeConstants.null_string);
                info.add("ERROR: Failed to get info: " + st.getErrMsg());
            } else {
                info.add(snapshotName);
                ArrayList tmp = Lists.newArrayList();
                for (RemoteFile file : results) {
                    Pair<String, String> pureFileName = Repository.decodeFileNameWithChecksum(file.getName());
                    if (pureFileName == null) {
                        tmp.add("Invalid: " + file.getName());
                        continue;
                    }
                    tmp.add(Repository.disjoinPrefix(PREFIX_JOB_INFO, (String)pureFileName.first));
                }
                info.add(Joiner.on((String)"\n").join((Iterable)tmp));
                info.add(tmp.isEmpty() ? "ERROR: no snapshot" : "OK");
            }
        } else {
            String localFilePath = BackupHandler.BACKUP_ROOT_DIR + PATH_DELIMITER + PREFIX_JOB_INFO + timestamp;
            try {
                String remoteInfoFilePath = this.assembleJobInfoFilePath(snapshotName, -1L) + timestamp;
                Status st = this.download(remoteInfoFilePath, localFilePath);
                if (!st.ok()) {
                    info.add(snapshotName);
                    info.add(timestamp);
                    info.add(FeConstants.null_string);
                    info.add(FeConstants.null_string);
                    info.add("Failed to get info: " + st.getErrMsg());
                } else {
                    try {
                        BackupJobInfo jobInfo = BackupJobInfo.fromFile(localFilePath);
                        info.add(snapshotName);
                        info.add(timestamp);
                        info.add(jobInfo.dbName);
                        info.add(jobInfo.getBrief());
                        info.add("OK");
                    }
                    catch (IOException e) {
                        info.add(snapshotName);
                        info.add(timestamp);
                        info.add(FeConstants.null_string);
                        info.add(FeConstants.null_string);
                        info.add("Failed to read info from local file: " + e.getMessage());
                    }
                }
            }
            finally {
                File localFile = new File(localFilePath);
                if (localFile.exists()) {
                    localFile.delete();
                }
            }
        }
        return info;
    }

    public void write(DataOutput out) throws IOException {
        out.writeLong(this.id);
        Text.writeString((DataOutput)out, (String)this.name);
        out.writeBoolean(this.isReadOnly);
        Text.writeString((DataOutput)out, (String)this.location);
        this.storage.write(out);
        out.writeLong(this.createTime);
    }

    public void readFields(DataInput in) throws IOException {
        this.id = in.readLong();
        this.name = Text.readString((DataInput)in);
        this.isReadOnly = in.readBoolean();
        this.location = Text.readString((DataInput)in);
        this.storage = BlobStorage.read(in);
        this.createTime = in.readLong();
    }
}

