/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.backup;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.commons.collections.CollectionUtils;
import org.apache.doris.analysis.AbstractBackupStmt;
import org.apache.doris.analysis.AbstractBackupTableRefClause;
import org.apache.doris.analysis.BackupStmt;
import org.apache.doris.analysis.CancelBackupStmt;
import org.apache.doris.analysis.CreateRepositoryStmt;
import org.apache.doris.analysis.DropRepositoryStmt;
import org.apache.doris.analysis.PartitionNames;
import org.apache.doris.analysis.RestoreStmt;
import org.apache.doris.analysis.StorageBackend;
import org.apache.doris.analysis.TableName;
import org.apache.doris.analysis.TableRef;
import org.apache.doris.backup.AbstractJob;
import org.apache.doris.backup.BackupJob;
import org.apache.doris.backup.BackupJobInfo;
import org.apache.doris.backup.BlobStorage;
import org.apache.doris.backup.Repository;
import org.apache.doris.backup.RepositoryMgr;
import org.apache.doris.backup.RestoreJob;
import org.apache.doris.backup.Status;
import org.apache.doris.catalog.Catalog;
import org.apache.doris.catalog.Database;
import org.apache.doris.catalog.OlapTable;
import org.apache.doris.catalog.Partition;
import org.apache.doris.catalog.Table;
import org.apache.doris.cluster.ClusterNamespace;
import org.apache.doris.common.Config;
import org.apache.doris.common.DdlException;
import org.apache.doris.common.ErrorCode;
import org.apache.doris.common.ErrorReport;
import org.apache.doris.common.Pair;
import org.apache.doris.common.io.Writable;
import org.apache.doris.common.util.MasterDaemon;
import org.apache.doris.task.DirMoveTask;
import org.apache.doris.task.DownloadTask;
import org.apache.doris.task.SnapshotTask;
import org.apache.doris.task.UploadTask;
import org.apache.doris.thrift.TFinishTaskRequest;
import org.apache.doris.thrift.TTaskType;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class BackupHandler
extends MasterDaemon
implements Writable {
    private static final Logger LOG = LogManager.getLogger(BackupHandler.class);
    public static final int SIGNATURE_VERSION = 1;
    public static final Path BACKUP_ROOT_DIR = Paths.get(Config.tmp_dir, "backup").normalize();
    public static final Path RESTORE_ROOT_DIR = Paths.get(Config.tmp_dir, "restore").normalize();
    private RepositoryMgr repoMgr = new RepositoryMgr();
    private final ReentrantLock jobLock = new ReentrantLock();
    private final Map<Long, Deque<AbstractJob>> dbIdToBackupOrRestoreJobs = new HashMap<Long, Deque<AbstractJob>>();
    private ReentrantLock seqlock = new ReentrantLock();
    private boolean isInit = false;
    private Catalog catalog;

    public BackupHandler() {
    }

    public BackupHandler(Catalog catalog) {
        super("backupHandler", 3000L);
        this.catalog = catalog;
    }

    public void setCatalog(Catalog catalog) {
        this.catalog = catalog;
    }

    @Override
    public synchronized void start() {
        Preconditions.checkNotNull((Object)this.catalog);
        super.start();
        this.repoMgr.start();
    }

    public RepositoryMgr getRepoMgr() {
        return this.repoMgr;
    }

    private boolean init() {
        File restoreDir;
        File backupDir = new File(BACKUP_ROOT_DIR.toString());
        if (!backupDir.exists()) {
            if (!backupDir.mkdirs()) {
                LOG.warn("failed to create backup dir: " + BACKUP_ROOT_DIR);
                return false;
            }
        } else if (!backupDir.isDirectory()) {
            LOG.warn("backup dir is not a directory: " + BACKUP_ROOT_DIR);
            return false;
        }
        if (!(restoreDir = new File(RESTORE_ROOT_DIR.toString())).exists()) {
            if (!restoreDir.mkdirs()) {
                LOG.warn("failed to create restore dir: " + RESTORE_ROOT_DIR);
                return false;
            }
        } else if (!restoreDir.isDirectory()) {
            LOG.warn("restore dir is not a directory: " + RESTORE_ROOT_DIR);
            return false;
        }
        this.isInit = true;
        return true;
    }

    public AbstractJob getJob(long dbId) {
        return this.getCurrentJob(dbId);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<AbstractJob> getJobs(long dbId, Predicate<String> predicate) {
        this.jobLock.lock();
        try {
            List<AbstractJob> list = ((Deque)this.dbIdToBackupOrRestoreJobs.getOrDefault(dbId, new LinkedList())).stream().filter(e -> predicate.test(e.getLabel())).collect(Collectors.toList());
            return list;
        }
        finally {
            this.jobLock.unlock();
        }
    }

    @Override
    protected void runAfterCatalogReady() {
        if (!this.isInit && !this.init()) {
            return;
        }
        for (AbstractJob job : this.getAllCurrentJobs()) {
            job.setCatalog(this.catalog);
            job.run();
        }
    }

    public void createRepository(CreateRepositoryStmt stmt) throws DdlException {
        if (!this.catalog.getBrokerMgr().containsBroker(stmt.getBrokerName()) && stmt.getStorageType() == StorageBackend.StorageType.BROKER) {
            ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "broker does not exist: " + stmt.getBrokerName());
        }
        BlobStorage storage = BlobStorage.create(stmt.getBrokerName(), stmt.getStorageType(), stmt.getProperties());
        long repoId = this.catalog.getNextId();
        Repository repo = new Repository(repoId, stmt.getName(), stmt.isReadOnly(), stmt.getLocation(), storage);
        Status st = this.repoMgr.addAndInitRepoIfNotExist(repo, false);
        if (!st.ok()) {
            ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "Failed to create repository: " + st.getErrMsg());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void dropRepository(DropRepositoryStmt stmt) throws DdlException {
        this.tryLock();
        try {
            Repository repo = this.repoMgr.getRepo(stmt.getRepoName());
            if (repo == null) {
                ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "Repository does not exist");
            }
            for (AbstractJob job : this.getAllCurrentJobs()) {
                if (job.isDone() || job.getRepoId() != repo.getId()) continue;
                ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "Backup or restore job is running on this repository. Can not drop it");
            }
            Status st = this.repoMgr.removeRepo(repo.getName(), false);
            if (!st.ok()) {
                ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "Failed to drop repository: " + st.getErrMsg());
            }
        }
        finally {
            this.seqlock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void process(AbstractBackupStmt stmt) throws DdlException {
        String repoName = stmt.getRepoName();
        Repository repository = this.repoMgr.getRepo(repoName);
        if (repository == null) {
            ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "Repository " + repoName + " does not exist");
        }
        String dbName = stmt.getDbName();
        Database db = this.catalog.getDbOrDdlException(dbName);
        this.tryLock();
        try {
            AbstractJob currentJob = this.getCurrentJob(db.getId());
            if (currentJob != null && !currentJob.isDone()) {
                ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "Can only run one backup or restore job of a database at same time");
            }
            if (stmt instanceof BackupStmt) {
                this.backup(repository, db, (BackupStmt)stmt);
            } else if (stmt instanceof RestoreStmt) {
                this.restore(repository, db, (RestoreStmt)stmt);
            }
        }
        finally {
            this.seqlock.unlock();
        }
    }

    private void tryLock() throws DdlException {
        try {
            if (!this.seqlock.tryLock(10L, TimeUnit.SECONDS)) {
                ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "Another backup or restore job is being submitted. Please wait and try again");
            }
        }
        catch (InterruptedException e) {
            ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "Got interrupted exception when try locking. Try again");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void backup(Repository repository, Database db, BackupStmt stmt) throws DdlException {
        if (repository.isReadOnly()) {
            ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "Repository " + repository.getName() + " is read only");
        }
        Set<Object> tableNames = Sets.newHashSet();
        AbstractBackupTableRefClause abstractBackupTableRefClause = stmt.getAbstractBackupTableRefClause();
        if (abstractBackupTableRefClause == null) {
            tableNames = db.getTableNamesWithLock();
        } else if (abstractBackupTableRefClause.isExclude()) {
            tableNames = db.getTableNamesWithLock();
            for (TableRef tableRef : abstractBackupTableRefClause.getTableRefList()) {
                if (tableNames.remove(tableRef.getName().getTbl())) continue;
                LOG.info("exclude table " + tableRef.getName().getTbl() + " of backup stmt is not exists in db " + db.getFullName());
            }
        }
        List<Object> tblRefs = Lists.newArrayList();
        if (abstractBackupTableRefClause != null && !abstractBackupTableRefClause.isExclude()) {
            tblRefs = abstractBackupTableRefClause.getTableRefList();
        } else {
            for (String string : tableNames) {
                TableRef tableRef = new TableRef(new TableName(db.getFullName(), string), null);
                tblRefs.add(tableRef);
            }
        }
        for (TableRef tableRef : tblRefs) {
            String tblName = tableRef.getName().getTbl();
            Table tbl = db.getTableOrDdlException(tblName);
            if (tbl.getType() == Table.TableType.VIEW || tbl.getType() == Table.TableType.ODBC) continue;
            if (tbl.getType() != Table.TableType.OLAP) {
                ErrorReport.reportDdlException(ErrorCode.ERR_NOT_OLAP_TABLE, tblName);
            }
            OlapTable olapTbl = (OlapTable)tbl;
            tbl.readLock();
            try {
                PartitionNames partitionNames;
                if (olapTbl.existTempPartitions()) {
                    ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "Do not support backup table with temp partitions");
                }
                if ((partitionNames = tableRef.getPartitionNames()) == null) continue;
                if (partitionNames.isTemp()) {
                    ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "Do not support backup temp partitions");
                }
                for (String partName : partitionNames.getPartitionNames()) {
                    Partition partition = olapTbl.getPartition(partName);
                    if (partition != null) continue;
                    ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "Unknown partition " + partName + " in table" + tblName);
                }
            }
            finally {
                tbl.readUnlock();
            }
        }
        ArrayList arrayList = Lists.newArrayList();
        Status status = repository.listSnapshots(arrayList);
        if (!status.ok()) {
            ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, status.getErrMsg());
        }
        if (arrayList.contains(stmt.getLabel())) {
            if (stmt.getType() == BackupStmt.BackupType.FULL) {
                ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "Snapshot with name '" + stmt.getLabel() + "' already exist in repository");
            } else {
                ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "Currently does not support incremental backup");
            }
        }
        BackupJob backupJob = new BackupJob(stmt.getLabel(), db.getId(), ClusterNamespace.getNameFromFullName(db.getFullName()), tblRefs, stmt.getTimeoutMs(), stmt.getContent(), this.catalog, repository.getId());
        this.catalog.getEditLog().logBackupJob(backupJob);
        this.addBackupOrRestoreJob(db.getId(), backupJob);
        LOG.info("finished to submit backup job: {}", (Object)backupJob);
    }

    private void restore(Repository repository, Database db, RestoreStmt stmt) throws DdlException {
        ArrayList infos = Lists.newArrayList();
        Status status = repository.getSnapshotInfoFile(stmt.getLabel(), stmt.getBackupTimestamp(), infos);
        if (!status.ok()) {
            ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "Failed to get info of snapshot '" + stmt.getLabel() + "' because: " + status.getErrMsg() + ". Maybe specified wrong backup timestamp");
        }
        Preconditions.checkState((infos.size() == 1 ? 1 : 0) != 0);
        BackupJobInfo jobInfo = (BackupJobInfo)infos.get(0);
        this.checkAndFilterRestoreObjsExistInSnapshot(jobInfo, stmt.getAbstractBackupTableRefClause());
        RestoreJob restoreJob = new RestoreJob(stmt.getLabel(), stmt.getBackupTimestamp(), db.getId(), db.getFullName(), jobInfo, stmt.allowLoad(), stmt.getReplicaAlloc(), stmt.getTimeoutMs(), stmt.getMetaVersion(), this.catalog, repository.getId());
        this.catalog.getEditLog().logRestoreJob(restoreJob);
        this.addBackupOrRestoreJob(db.getId(), restoreJob);
        LOG.info("finished to submit restore job: {}", (Object)restoreJob);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void addBackupOrRestoreJob(long dbId, AbstractJob job) {
        this.jobLock.lock();
        try {
            Deque jobs = this.dbIdToBackupOrRestoreJobs.computeIfAbsent(dbId, k -> Lists.newLinkedList());
            while (jobs.size() >= Config.max_backup_restore_job_num_per_db) {
                jobs.removeFirst();
            }
            AbstractJob lastJob = (AbstractJob)jobs.peekLast();
            if (lastJob != null && (lastJob.isPending() || lastJob.getJobId() == job.getJobId())) {
                jobs.removeLast();
            }
            jobs.addLast(job);
        }
        finally {
            this.jobLock.unlock();
        }
    }

    private List<AbstractJob> getAllCurrentJobs() {
        this.jobLock.lock();
        try {
            List<AbstractJob> list = this.dbIdToBackupOrRestoreJobs.values().stream().filter(CollectionUtils::isNotEmpty).map(Deque::getLast).collect(Collectors.toList());
            return list;
        }
        finally {
            this.jobLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private AbstractJob getCurrentJob(long dbId) {
        this.jobLock.lock();
        try {
            Deque<AbstractJob> jobs = this.dbIdToBackupOrRestoreJobs.getOrDefault(dbId, Lists.newLinkedList());
            AbstractJob abstractJob = jobs.isEmpty() ? null : jobs.getLast();
            return abstractJob;
        }
        finally {
            this.jobLock.unlock();
        }
    }

    private void checkAndFilterRestoreObjsExistInSnapshot(BackupJobInfo jobInfo, AbstractBackupTableRefClause backupTableRefClause) throws DdlException {
        if (backupTableRefClause == null) {
            return;
        }
        if (backupTableRefClause.isExclude()) {
            for (TableRef tblRef : backupTableRefClause.getTableRefList()) {
                String tblName = tblRef.getName().getTbl();
                Table.TableType tableType = jobInfo.getTypeByTblName(tblName);
                if (tableType == null) {
                    LOG.info("Ignore error : exclude table " + tblName + " does not exist in snapshot " + jobInfo.name);
                    continue;
                }
                if (tblRef.hasExplicitAlias()) {
                    ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "The table alias in exclude clause does not make sense");
                }
                jobInfo.removeTable(tblRef, tableType);
            }
            return;
        }
        HashSet olapTableNames = Sets.newHashSet();
        HashSet viewNames = Sets.newHashSet();
        HashSet odbcTableNames = Sets.newHashSet();
        for (TableRef tblRef : backupTableRefClause.getTableRefList()) {
            String tblName = tblRef.getName().getTbl();
            Table.TableType tableType = jobInfo.getTypeByTblName(tblName);
            if (tableType == null) {
                ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "Table " + tblName + " does not exist in snapshot " + jobInfo.name);
            }
            switch (tableType) {
                case OLAP: {
                    this.checkAndFilterRestoreOlapTableExistInSnapshot(jobInfo.backupOlapTableObjects, tblRef);
                    olapTableNames.add(tblName);
                    break;
                }
                case VIEW: {
                    viewNames.add(tblName);
                    break;
                }
                case ODBC: {
                    odbcTableNames.add(tblName);
                    break;
                }
            }
            if (!tblRef.hasExplicitAlias()) continue;
            jobInfo.setAlias(tblName, tblRef.getExplicitAlias());
        }
        jobInfo.retainOlapTables(olapTableNames);
        jobInfo.retainView(viewNames);
        jobInfo.retainOdbcTables(odbcTableNames);
    }

    public void checkAndFilterRestoreOlapTableExistInSnapshot(Map<String, BackupJobInfo.BackupOlapTableInfo> backupOlapTableInfoMap, TableRef tableRef) throws DdlException {
        String tblName = tableRef.getName().getTbl();
        BackupJobInfo.BackupOlapTableInfo tblInfo = backupOlapTableInfoMap.get(tblName);
        PartitionNames partitionNames = tableRef.getPartitionNames();
        if (partitionNames != null) {
            if (partitionNames.isTemp()) {
                ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "Do not support restoring temporary partitions");
            }
            for (String partName : partitionNames.getPartitionNames()) {
                if (tblInfo.containsPart(partName)) continue;
                ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "Partition " + partName + " of table " + tblName + " does not exist in snapshot");
            }
        }
        tblInfo.retainPartitions(partitionNames == null ? null : partitionNames.getPartitionNames());
    }

    public void cancel(CancelBackupStmt stmt) throws DdlException {
        Status status;
        String dbName = stmt.getDbName();
        Database db = this.catalog.getDbOrDdlException(dbName);
        AbstractJob job = this.getCurrentJob(db.getId());
        if (job == null || job instanceof BackupJob && stmt.isRestore() || job instanceof RestoreJob && !stmt.isRestore()) {
            ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "No " + (stmt.isRestore() ? "restore" : "backup job") + " is currently running");
        }
        if (!(status = job.cancel()).ok()) {
            ErrorReport.reportDdlException(ErrorCode.ERR_COMMON_ERROR, "Failed to cancel job: " + status.getErrMsg());
        }
        LOG.info("finished to cancel {} job: {}", (Object)(stmt.isRestore() ? "restore" : "backup"), (Object)job);
    }

    public boolean handleFinishedSnapshotTask(SnapshotTask task, TFinishTaskRequest request) {
        AbstractJob job = this.getCurrentJob(task.getDbId());
        if (job == null) {
            LOG.warn("failed to find backup or restore job for task: {}", (Object)task);
            return true;
        }
        if (job instanceof BackupJob) {
            if (task.isRestoreTask()) {
                LOG.warn("expect finding restore job, but get backup job {} for task: {}", (Object)job, (Object)task);
                return true;
            }
            return ((BackupJob)job).finishTabletSnapshotTask(task, request);
        }
        if (!task.isRestoreTask()) {
            LOG.warn("expect finding backup job, but get restore job {} for task: {}", (Object)job, (Object)task);
            return true;
        }
        return ((RestoreJob)job).finishTabletSnapshotTask(task, request);
    }

    public boolean handleFinishedSnapshotUploadTask(UploadTask task, TFinishTaskRequest request) {
        AbstractJob job = this.getCurrentJob(task.getDbId());
        if (job == null || job instanceof RestoreJob) {
            LOG.info("invalid upload task: {}, no backup job is found. db id: {}", (Object)task, (Object)task.getDbId());
            return false;
        }
        BackupJob restoreJob = (BackupJob)job;
        if (restoreJob.getJobId() != task.getJobId() || restoreJob.getState() != BackupJob.BackupJobState.UPLOADING) {
            LOG.info("invalid upload task: {}, job id: {}, job state: {}", (Object)task, (Object)restoreJob.getJobId(), (Object)restoreJob.getState().name());
            return false;
        }
        return restoreJob.finishSnapshotUploadTask(task, request);
    }

    public boolean handleDownloadSnapshotTask(DownloadTask task, TFinishTaskRequest request) {
        AbstractJob job = this.getCurrentJob(task.getDbId());
        if (!(job instanceof RestoreJob)) {
            LOG.warn("failed to find restore job for task: {}", (Object)task);
            return true;
        }
        return ((RestoreJob)job).finishTabletDownloadTask(task, request);
    }

    public boolean handleDirMoveTask(DirMoveTask task, TFinishTaskRequest request) {
        AbstractJob job = this.getCurrentJob(task.getDbId());
        if (!(job instanceof RestoreJob)) {
            LOG.warn("failed to find restore job for task: {}", (Object)task);
            return true;
        }
        return ((RestoreJob)job).finishDirMoveTask(task, request);
    }

    public void replayAddJob(AbstractJob job) {
        if (job.isCancelled()) {
            AbstractJob existingJob = this.getCurrentJob(job.getDbId());
            if (existingJob == null || existingJob.isDone()) {
                LOG.error("invalid existing job: {}. current replay job is: {}", (Object)existingJob, (Object)job);
                return;
            }
            existingJob.setCatalog(this.catalog);
            existingJob.replayCancel();
        } else if (!job.isPending()) {
            AbstractJob existingJob = this.getCurrentJob(job.getDbId());
            if (existingJob == null || existingJob.isDone()) {
                LOG.error("invalid existing job: {}. current replay job is: {}", (Object)existingJob, (Object)job);
                return;
            }
            job.replayRun();
        }
        this.addBackupOrRestoreJob(job.getDbId(), job);
    }

    public boolean report(TTaskType type, long jobId, long taskId, int finishedNum, int totalNum) {
        for (AbstractJob job : this.getAllCurrentJobs()) {
            if (job.getType() == AbstractJob.JobType.BACKUP) {
                if (job.isDone() || job.getJobId() != jobId || type != TTaskType.UPLOAD) continue;
                job.taskProgress.put(taskId, Pair.create(finishedNum, totalNum));
                return true;
            }
            if (job.getType() != AbstractJob.JobType.RESTORE || job.isDone() || job.getJobId() != jobId || type != TTaskType.DOWNLOAD) continue;
            job.taskProgress.put(taskId, Pair.create(finishedNum, totalNum));
            return true;
        }
        return false;
    }

    public static BackupHandler read(DataInput in) throws IOException {
        BackupHandler backupHandler = new BackupHandler();
        backupHandler.readFields(in);
        return backupHandler;
    }

    public void write(DataOutput out) throws IOException {
        this.repoMgr.write(out);
        List jobs = this.dbIdToBackupOrRestoreJobs.values().stream().flatMap(Collection::stream).collect(Collectors.toList());
        out.writeInt(jobs.size());
        for (AbstractJob job : jobs) {
            job.write(out);
        }
    }

    public void readFields(DataInput in) throws IOException {
        this.repoMgr = RepositoryMgr.read(in);
        int size = in.readInt();
        for (int i = 0; i < size; ++i) {
            AbstractJob job = AbstractJob.read(in);
            this.addBackupOrRestoreJob(job.getDbId(), job);
        }
    }
}

