/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.analysis;

import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.annotations.SerializedName;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.TimeZone;
import java.util.regex.Pattern;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.doris.common.AnalysisException;
import org.apache.doris.common.Pair;
import org.apache.doris.common.UserException;
import org.apache.doris.common.util.TimeUtils;
import org.apache.doris.load.routineload.LoadDataSourceType;

public class RoutineLoadDataSourceProperties {
    private static final ImmutableSet<String> DATA_SOURCE_PROPERTIES_SET = new ImmutableSet.Builder().add((Object)"kafka_broker_list").add((Object)"kafka_topic").add((Object)"kafka_partitions").add((Object)"kafka_offsets").add((Object)"kafka_default_offsets").build();
    private static final ImmutableSet<String> CONFIGURABLE_DATA_SOURCE_PROPERTIES_SET = new ImmutableSet.Builder().add((Object)"kafka_broker_list").add((Object)"kafka_topic").add((Object)"kafka_partitions").add((Object)"kafka_offsets").add((Object)"kafka_default_offsets").build();
    private Map<String, String> properties = Maps.newHashMap();
    private boolean isAlter = false;
    @SerializedName(value="type")
    private String type = "KAFKA";
    @SerializedName(value="kafkaPartitionOffsets")
    private List<Pair<Integer, Long>> kafkaPartitionOffsets = Lists.newArrayList();
    @SerializedName(value="customKafkaProperties")
    private Map<String, String> customKafkaProperties = Maps.newHashMap();
    @SerializedName(value="isOffsetsForTimes")
    private boolean isOffsetsForTimes = false;
    @SerializedName(value="kafkaBrokerList")
    private String kafkaBrokerList;
    @SerializedName(value="KafkaTopic")
    private String kafkaTopic;
    @SerializedName(value="timezone")
    private String timezone;

    public RoutineLoadDataSourceProperties() {
        this.isAlter = true;
    }

    public RoutineLoadDataSourceProperties(String type, Map<String, String> properties, boolean isAlter) {
        this.type = type.toUpperCase();
        this.properties = properties;
        this.isAlter = isAlter;
    }

    public void analyze() throws UserException {
        if (this.properties.isEmpty()) {
            if (!this.isAlter) {
                throw new AnalysisException("No data source properties");
            }
            return;
        }
        Preconditions.checkState((!Strings.isNullOrEmpty((String)this.timezone) ? 1 : 0) != 0, (Object)"timezone must be set before analyzing");
        this.checkDataSourceProperties();
    }

    public boolean hasAnalyzedProperties() {
        return !this.kafkaPartitionOffsets.isEmpty() || !this.customKafkaProperties.isEmpty();
    }

    public String getType() {
        return this.type;
    }

    public List<Pair<Integer, Long>> getKafkaPartitionOffsets() {
        return this.kafkaPartitionOffsets;
    }

    public void setKafkaPartitionOffsets(List<Pair<Integer, Long>> kafkaPartitionOffsets) {
        this.kafkaPartitionOffsets = kafkaPartitionOffsets;
    }

    public Map<String, String> getCustomKafkaProperties() {
        return this.customKafkaProperties;
    }

    public void setTimezone(String timezone) {
        this.timezone = timezone;
    }

    public String getKafkaBrokerList() {
        return this.kafkaBrokerList;
    }

    public String getKafkaTopic() {
        return this.kafkaTopic;
    }

    public boolean isOffsetsForTimes() {
        return this.isOffsetsForTimes;
    }

    private void checkDataSourceProperties() throws UserException {
        LoadDataSourceType sourceType;
        try {
            sourceType = LoadDataSourceType.valueOf(this.type);
        }
        catch (IllegalArgumentException e) {
            throw new AnalysisException("routine load job does not support this type " + this.type);
        }
        switch (sourceType) {
            case KAFKA: {
                this.checkKafkaProperties();
                break;
            }
        }
    }

    private void checkKafkaProperties() throws UserException {
        ImmutableSet<String> propertySet = this.isAlter ? CONFIGURABLE_DATA_SOURCE_PROPERTIES_SET : DATA_SOURCE_PROPERTIES_SET;
        Optional<String> optional = this.properties.keySet().stream().filter(entity -> !propertySet.contains(entity)).filter(entity -> !entity.startsWith("property.")).findFirst();
        if (optional.isPresent()) {
            throw new AnalysisException(optional.get() + " is invalid kafka property or can not be set");
        }
        this.kafkaBrokerList = Strings.nullToEmpty((String)this.properties.get("kafka_broker_list")).replaceAll(" ", "");
        if (!this.isAlter && Strings.isNullOrEmpty((String)this.kafkaBrokerList)) {
            throw new AnalysisException("kafka_broker_list is a required property");
        }
        if (!Strings.isNullOrEmpty((String)this.kafkaBrokerList)) {
            String[] kafkaBrokerList;
            for (String broker : kafkaBrokerList = this.kafkaBrokerList.split(",")) {
                if (Pattern.matches("[-A-Za-z0-9+&@#/%?=~_|!:,.;]+[-A-Za-z0-9+&@#/%=~_|]", broker)) continue;
                throw new AnalysisException("kafka_broker_list:" + broker + " not match pattern " + "[-A-Za-z0-9+&@#/%?=~_|!:,.;]+[-A-Za-z0-9+&@#/%=~_|]");
            }
        }
        this.kafkaTopic = Strings.nullToEmpty((String)this.properties.get("kafka_topic")).replaceAll(" ", "");
        if (!this.isAlter && Strings.isNullOrEmpty((String)this.kafkaTopic)) {
            throw new AnalysisException("kafka_topic is a required property");
        }
        RoutineLoadDataSourceProperties.analyzeCustomProperties(this.properties, this.customKafkaProperties);
        String kafkaPartitionsString = this.properties.get("kafka_partitions");
        if (kafkaPartitionsString != null) {
            RoutineLoadDataSourceProperties.analyzeKafkaPartitionProperty(kafkaPartitionsString, this.kafkaPartitionOffsets);
        }
        String kafkaOffsetsString = this.properties.get("kafka_offsets");
        String kafkaDefaultOffsetString = this.customKafkaProperties.get("kafka_default_offsets");
        if (kafkaOffsetsString != null && kafkaDefaultOffsetString != null) {
            throw new AnalysisException("Only one of kafka_offsets and kafka_default_offsets can be set.");
        }
        if (this.isAlter && kafkaPartitionsString != null && kafkaOffsetsString == null && kafkaDefaultOffsetString == null) {
            throw new AnalysisException("Must set offset or default offset with partition property");
        }
        if (kafkaOffsetsString != null) {
            this.isOffsetsForTimes = RoutineLoadDataSourceProperties.analyzeKafkaOffsetProperty(kafkaOffsetsString, this.kafkaPartitionOffsets, this.timezone);
        } else {
            this.isOffsetsForTimes = RoutineLoadDataSourceProperties.analyzeKafkaDefaultOffsetProperty(this.customKafkaProperties, this.timezone);
            if (!this.kafkaPartitionOffsets.isEmpty()) {
                kafkaDefaultOffsetString = this.customKafkaProperties.get("kafka_default_offsets");
                RoutineLoadDataSourceProperties.setDefaultOffsetForPartition(this.kafkaPartitionOffsets, kafkaDefaultOffsetString, this.isOffsetsForTimes);
            }
        }
    }

    private static void setDefaultOffsetForPartition(List<Pair<Integer, Long>> kafkaPartitionOffsets, String kafkaDefaultOffsetString, boolean isOffsetsForTimes) {
        if (isOffsetsForTimes) {
            for (Pair<Integer, Long> pair : kafkaPartitionOffsets) {
                pair.second = Long.valueOf(kafkaDefaultOffsetString);
            }
        } else {
            for (Pair<Integer, Long> pair : kafkaPartitionOffsets) {
                if (kafkaDefaultOffsetString.equalsIgnoreCase("OFFSET_BEGINNING")) {
                    pair.second = -2L;
                    continue;
                }
                pair.second = -1L;
            }
        }
    }

    private static boolean analyzeKafkaDefaultOffsetProperty(Map<String, String> customKafkaProperties, String timeZoneStr) throws AnalysisException {
        customKafkaProperties.putIfAbsent("kafka_default_offsets", "OFFSET_END");
        String defaultOffsetStr = customKafkaProperties.get("kafka_default_offsets");
        TimeZone timeZone = TimeUtils.getOrSystemTimeZone(timeZoneStr);
        long defaultOffset = TimeUtils.timeStringToLong(defaultOffsetStr, timeZone);
        if (defaultOffset != -1L) {
            customKafkaProperties.put("kafka_default_offsets", String.valueOf(defaultOffset));
            customKafkaProperties.put("kafka_origin_default_offsets", defaultOffsetStr);
            return true;
        }
        if (!defaultOffsetStr.equalsIgnoreCase("OFFSET_BEGINNING") && !defaultOffsetStr.equalsIgnoreCase("OFFSET_END")) {
            throw new AnalysisException("kafka_default_offsets can only be set to OFFSET_BEGINNING, OFFSET_END or date time");
        }
        return false;
    }

    private static void analyzeKafkaPartitionProperty(String kafkaPartitionsString, List<Pair<Integer, Long>> kafkaPartitionOffsets) throws AnalysisException {
        String[] kafkaPartitionsStringList;
        if ((kafkaPartitionsString = kafkaPartitionsString.replaceAll(" ", "")).isEmpty()) {
            throw new AnalysisException("kafka_partitions could not be a empty string");
        }
        for (String s : kafkaPartitionsStringList = kafkaPartitionsString.split(",")) {
            try {
                kafkaPartitionOffsets.add(Pair.create(RoutineLoadDataSourceProperties.getIntegerValueFromString(s, "kafka_partitions"), -1L));
            }
            catch (AnalysisException e) {
                throw new AnalysisException("kafka_partitions must be a number string with comma-separated");
            }
        }
    }

    private static boolean analyzeKafkaOffsetProperty(String kafkaOffsetsString, List<Pair<Integer, Long>> kafkaPartitionOffsets, String timeZoneStr) throws UserException {
        if (Strings.isNullOrEmpty((String)kafkaOffsetsString)) {
            throw new AnalysisException("kafka_offsets could not be a empty string");
        }
        List kafkaOffsetsStringList = Splitter.on((String)",").trimResults().splitToList((CharSequence)kafkaOffsetsString);
        if (kafkaOffsetsStringList.size() != kafkaPartitionOffsets.size()) {
            throw new AnalysisException("Partitions number should be equals to offsets number");
        }
        boolean foundTime = false;
        boolean foundOffset = false;
        for (String kafkaOffsetsStr : kafkaOffsetsStringList) {
            if (TimeUtils.timeStringToLong(kafkaOffsetsStr) != -1L) {
                foundTime = true;
                continue;
            }
            foundOffset = true;
        }
        if (foundTime && foundOffset) {
            throw new AnalysisException("The offset of the partition cannot be specified by the timestamp and the offset at the same time");
        }
        if (foundTime) {
            TimeZone timeZone = TimeUtils.getOrSystemTimeZone(timeZoneStr);
            for (int i = 0; i < kafkaOffsetsStringList.size(); ++i) {
                String kafkaOffsetsStr = (String)kafkaOffsetsStringList.get(i);
                long timestamp = TimeUtils.timeStringToLong(kafkaOffsetsStr, timeZone);
                Preconditions.checkState((timestamp != -1L ? 1 : 0) != 0);
                kafkaPartitionOffsets.get((int)i).second = timestamp;
            }
        } else {
            for (int i = 0; i < kafkaOffsetsStringList.size(); ++i) {
                String kafkaOffsetsStr;
                kafkaOffsetsStr = (String)kafkaOffsetsStringList.get(i);
                if (kafkaOffsetsStr.equalsIgnoreCase("OFFSET_BEGINNING")) {
                    kafkaPartitionOffsets.get((int)i).second = -2L;
                    continue;
                }
                if (kafkaOffsetsStr.equalsIgnoreCase("OFFSET_END")) {
                    kafkaPartitionOffsets.get((int)i).second = -1L;
                    continue;
                }
                if (NumberUtils.isDigits((String)kafkaOffsetsStr)) {
                    kafkaPartitionOffsets.get((int)i).second = NumberUtils.toLong((String)kafkaOffsetsStr);
                    continue;
                }
                throw new AnalysisException("kafka_offsets must be an integer or a date time");
            }
        }
        return foundTime;
    }

    private static void analyzeCustomProperties(Map<String, String> dataSourceProperties, Map<String, String> customKafkaProperties) throws AnalysisException {
        for (Map.Entry<String, String> dataSourceProperty : dataSourceProperties.entrySet()) {
            if (!dataSourceProperty.getKey().startsWith("property.")) continue;
            String propertyKey = dataSourceProperty.getKey();
            String propertyValue = dataSourceProperty.getValue();
            String[] propertyValueArr = propertyKey.split("\\.");
            if (propertyValueArr.length < 2) {
                throw new AnalysisException("kafka property value could not be a empty string");
            }
            customKafkaProperties.put(propertyKey.substring(propertyKey.indexOf(".") + 1), propertyValue);
        }
    }

    private static int getIntegerValueFromString(String valueString, String propertyName) throws AnalysisException {
        int value;
        if (valueString.isEmpty()) {
            throw new AnalysisException(propertyName + " could not be a empty string");
        }
        try {
            value = Integer.valueOf(valueString);
        }
        catch (NumberFormatException e) {
            throw new AnalysisException(propertyName + " must be a integer");
        }
        return value;
    }

    public String toString() {
        if (!this.hasAnalyzedProperties()) {
            return "empty";
        }
        StringBuilder sb = new StringBuilder();
        sb.append("type: ").append(this.type);
        sb.append(", kafka partition offsets: ").append(this.kafkaPartitionOffsets);
        sb.append(", custome properties: ").append(this.customKafkaProperties);
        return sb.toString();
    }
}

