/*
 * Decompiled with CFR 0.152.
 */
package org.apache.doris.analysis;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.collections.map.CaseInsensitiveMap;
import org.apache.doris.analysis.Analyzer;
import org.apache.doris.analysis.BrokerDesc;
import org.apache.doris.analysis.Expr;
import org.apache.doris.analysis.StorageBackend;
import org.apache.doris.backup.HDFSStorage;
import org.apache.doris.backup.S3Storage;
import org.apache.doris.catalog.PrimitiveType;
import org.apache.doris.catalog.Type;
import org.apache.doris.common.AnalysisException;
import org.apache.doris.common.Config;
import org.apache.doris.common.FeNameFormat;
import org.apache.doris.common.UserException;
import org.apache.doris.common.util.BrokerUtil;
import org.apache.doris.common.util.ParseUtil;
import org.apache.doris.common.util.PrintableMap;
import org.apache.doris.qe.ConnectContext;
import org.apache.doris.thrift.TFileFormatType;
import org.apache.doris.thrift.TResultFileSinkOptions;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class OutFileClause {
    private static final Logger LOG = LogManager.getLogger(OutFileClause.class);
    public static final List<String> RESULT_COL_NAMES = Lists.newArrayList();
    public static final List<PrimitiveType> RESULT_COL_TYPES = Lists.newArrayList();
    public static final List<String> PARQUET_REPETITION_TYPES = Lists.newArrayList();
    public static final List<String> PARQUET_DATA_TYPES = Lists.newArrayList();
    public static final String LOCAL_FILE_PREFIX = "file:///";
    private static final String S3_FILE_PREFIX = "S3://";
    private static final String HDFS_FILE_PREFIX = "hdfs://";
    private static final String HADOOP_FS_PROP_PREFIX = "dfs.";
    private static final String HADOOP_PROP_PREFIX = "hadoop.";
    private static final String BROKER_PROP_PREFIX = "broker.";
    private static final String PROP_BROKER_NAME = "broker.name";
    private static final String PROP_COLUMN_SEPARATOR = "column_separator";
    private static final String PROP_LINE_DELIMITER = "line_delimiter";
    private static final String PROP_MAX_FILE_SIZE = "max_file_size";
    private static final String PROP_SUCCESS_FILE_NAME = "success_file_name";
    private static final String PARQUET_PROP_PREFIX = "parquet.";
    private static final String SCHEMA = "schema";
    private static final long DEFAULT_MAX_FILE_SIZE_BYTES = 0x40000000L;
    private static final long MIN_FILE_SIZE_BYTES = 0x500000L;
    private static final long MAX_FILE_SIZE_BYTES = 0x80000000L;
    private String filePath;
    private String format;
    private Map<String, String> properties;
    private String columnSeparator = "\t";
    private String lineDelimiter = "\n";
    private TFileFormatType fileFormatType;
    private long maxFileSizeBytes = 0x40000000L;
    private BrokerDesc brokerDesc = null;
    private boolean isLocalOutput = false;
    private String successFileName = "";
    private List<List<String>> schema = new ArrayList<List<String>>();
    private Map<String, String> fileProperties = new HashMap<String, String>();
    private boolean isAnalyzed = false;

    public OutFileClause(String filePath, String format, Map<String, String> properties) {
        this.filePath = filePath;
        this.format = Strings.isNullOrEmpty((String)format) ? "csv" : format.toLowerCase();
        this.properties = properties;
        this.isAnalyzed = false;
    }

    public OutFileClause(OutFileClause other) {
        this.filePath = other.filePath;
        this.format = other.format;
        this.properties = other.properties == null ? null : Maps.newHashMap(other.properties);
        this.isAnalyzed = other.isAnalyzed;
    }

    public String getColumnSeparator() {
        return this.columnSeparator;
    }

    public String getLineDelimiter() {
        return this.lineDelimiter;
    }

    public TFileFormatType getFileFormatType() {
        return this.fileFormatType;
    }

    public long getMaxFileSizeBytes() {
        return this.maxFileSizeBytes;
    }

    public BrokerDesc getBrokerDesc() {
        return this.brokerDesc;
    }

    public List<List<String>> getSchema() {
        return this.schema;
    }

    public void analyze(Analyzer analyzer, List<Expr> resultExprs) throws UserException {
        if (this.isAnalyzed) {
            return;
        }
        this.analyzeFilePath();
        switch (this.format) {
            case "csv": {
                this.fileFormatType = TFileFormatType.FORMAT_CSV_PLAIN;
                break;
            }
            case "parquet": {
                this.fileFormatType = TFileFormatType.FORMAT_PARQUET;
                break;
            }
            default: {
                throw new AnalysisException("format:" + this.format + " is not supported.");
            }
        }
        this.analyzeProperties();
        if (this.brokerDesc != null && this.isLocalOutput) {
            throw new AnalysisException("No need to specify BROKER properties in OUTFILE clause for local file output");
        }
        if (this.brokerDesc == null && !this.isLocalOutput) {
            throw new AnalysisException("Must specify BROKER properties in OUTFILE clause");
        }
        this.isAnalyzed = true;
        if (this.isParquetFormat()) {
            this.analyzeForParquetFormat(resultExprs);
        }
    }

    private void analyzeForParquetFormat(List<Expr> resultExprs) throws AnalysisException {
        if (this.schema.isEmpty()) {
            this.genParquetSchema(resultExprs);
        }
        if (resultExprs.size() != this.schema.size()) {
            throw new AnalysisException("Parquet schema number does not equal to select item number");
        }
        block9: for (int i = 0; i < this.schema.size(); ++i) {
            String type = this.schema.get(i).get(1);
            Type resultType = resultExprs.get(i).getType();
            switch (resultType.getPrimitiveType()) {
                case BOOLEAN: {
                    if (type.equals("boolean")) continue block9;
                    throw new AnalysisException("project field type is BOOLEAN, should use boolean, but the type of column " + i + " is " + type);
                }
                case TINYINT: 
                case SMALLINT: 
                case INT: {
                    if (type.equals("int32")) continue block9;
                    throw new AnalysisException("project field type is TINYINT/SMALLINT/INT, should use int32, but the definition type of column " + i + " is " + type);
                }
                case BIGINT: 
                case DATE: 
                case DATETIME: {
                    if (type.equals("int64")) continue block9;
                    throw new AnalysisException("project field type is BIGINT/DATE/DATETIME, should use int64, but the definition type of column " + i + " is " + type);
                }
                case FLOAT: {
                    if (type.equals("float")) continue block9;
                    throw new AnalysisException("project field type is FLOAT, should use float, but the definition type of column " + i + " is " + type);
                }
                case DOUBLE: {
                    if (type.equals("double")) continue block9;
                    throw new AnalysisException("project field type is DOUBLE, should use double, but the definition type of column " + i + " is " + type);
                }
                case CHAR: 
                case VARCHAR: 
                case STRING: 
                case DECIMALV2: {
                    if (type.equals("byte_array")) continue block9;
                    throw new AnalysisException("project field type is CHAR/VARCHAR/STRING/DECIMAL, should use byte_array, but the definition type of column " + i + " is " + type);
                }
                case HLL: 
                case BITMAP: {
                    if (ConnectContext.get() != null && ConnectContext.get().getSessionVariable().isReturnObjectDataAsBinary()) {
                        if (type.equals("byte_array")) continue block9;
                        throw new AnalysisException("project field type is HLL/BITMAP, should use byte_array, but the definition type of column " + i + " is " + type);
                    }
                    throw new AnalysisException("Parquet format does not support column type: " + (Object)((Object)resultType.getPrimitiveType()));
                }
                default: {
                    throw new AnalysisException("Parquet format does not support column type: " + (Object)((Object)resultType.getPrimitiveType()));
                }
            }
        }
    }

    private void genParquetSchema(List<Expr> resultExprs) throws AnalysisException {
        Preconditions.checkState((boolean)this.schema.isEmpty());
        for (int i = 0; i < resultExprs.size(); ++i) {
            Expr expr = resultExprs.get(i);
            ArrayList<String> column = new ArrayList<String>();
            column.add("required");
            switch (expr.getType().getPrimitiveType()) {
                case BOOLEAN: {
                    column.add("boolean");
                    break;
                }
                case TINYINT: 
                case SMALLINT: 
                case INT: {
                    column.add("int32");
                    break;
                }
                case BIGINT: 
                case DATE: 
                case DATETIME: {
                    column.add("int64");
                    break;
                }
                case FLOAT: {
                    column.add("float");
                    break;
                }
                case DOUBLE: {
                    column.add("double");
                    break;
                }
                case CHAR: 
                case VARCHAR: 
                case STRING: 
                case DECIMALV2: {
                    column.add("byte_array");
                    break;
                }
                case HLL: 
                case BITMAP: {
                    if (ConnectContext.get() != null && ConnectContext.get().getSessionVariable().isReturnObjectDataAsBinary()) {
                        column.add("byte_array");
                    }
                }
                default: {
                    throw new AnalysisException("currently parquet do not support column type: " + (Object)((Object)expr.getType().getPrimitiveType()));
                }
            }
            column.add("col" + i);
            this.schema.add(column);
        }
    }

    private void analyzeFilePath() throws AnalysisException {
        if (Strings.isNullOrEmpty((String)this.filePath)) {
            throw new AnalysisException("Must specify file in OUTFILE clause");
        }
        if (this.filePath.startsWith(LOCAL_FILE_PREFIX)) {
            if (!Config.enable_outfile_to_local) {
                throw new AnalysisException("Exporting results to local disk is not allowed. To enable this feature, you need to add `enable_outfile_to_local=true` in fe.conf and restart FE");
            }
            this.isLocalOutput = true;
            this.filePath = this.filePath.substring(LOCAL_FILE_PREFIX.length() - 1);
        } else {
            this.isLocalOutput = false;
        }
        if (Strings.isNullOrEmpty((String)this.filePath)) {
            throw new AnalysisException("Must specify file in OUTFILE clause");
        }
    }

    private void analyzeProperties() throws UserException {
        if (this.properties == null || this.properties.isEmpty()) {
            return;
        }
        HashSet processedPropKeys = Sets.newHashSet();
        this.analyzeBrokerDesc(processedPropKeys);
        if (this.properties.containsKey(PROP_COLUMN_SEPARATOR)) {
            if (!this.isCsvFormat()) {
                throw new AnalysisException("column_separator is only for CSV format");
            }
            this.columnSeparator = this.properties.get(PROP_COLUMN_SEPARATOR);
            processedPropKeys.add(PROP_COLUMN_SEPARATOR);
        }
        if (this.properties.containsKey(PROP_LINE_DELIMITER)) {
            if (!this.isCsvFormat()) {
                throw new AnalysisException("line_delimiter is only for CSV format");
            }
            this.lineDelimiter = this.properties.get(PROP_LINE_DELIMITER);
            processedPropKeys.add(PROP_LINE_DELIMITER);
        }
        if (this.properties.containsKey(PROP_MAX_FILE_SIZE)) {
            this.maxFileSizeBytes = ParseUtil.analyzeDataVolumn(this.properties.get(PROP_MAX_FILE_SIZE));
            if (this.maxFileSizeBytes > 0x80000000L || this.maxFileSizeBytes < 0x500000L) {
                throw new AnalysisException("max file size should between 5MB and 2GB. Given: " + this.maxFileSizeBytes);
            }
            processedPropKeys.add(PROP_MAX_FILE_SIZE);
        }
        if (this.properties.containsKey(PROP_SUCCESS_FILE_NAME)) {
            this.successFileName = this.properties.get(PROP_SUCCESS_FILE_NAME);
            FeNameFormat.checkCommonName("file name", this.successFileName);
            processedPropKeys.add(PROP_SUCCESS_FILE_NAME);
        }
        if (this.fileFormatType == TFileFormatType.FORMAT_PARQUET) {
            this.getParquetProperties(processedPropKeys);
        }
        if (processedPropKeys.size() != this.properties.size()) {
            LOG.debug("{} vs {}", (Object)processedPropKeys, this.properties);
            throw new AnalysisException("Unknown properties: " + this.properties.keySet().stream().filter(k -> !processedPropKeys.contains(k)).collect(Collectors.toList()));
        }
    }

    private void analyzeBrokerDesc(Set<String> processedPropKeys) throws UserException {
        StorageBackend.StorageType storageType;
        String brokerName = this.properties.get(PROP_BROKER_NAME);
        if (this.properties.containsKey(PROP_BROKER_NAME)) {
            processedPropKeys.add(PROP_BROKER_NAME);
            storageType = StorageBackend.StorageType.BROKER;
        } else if (this.filePath.toUpperCase().startsWith(S3_FILE_PREFIX)) {
            brokerName = StorageBackend.StorageType.S3.name();
            storageType = StorageBackend.StorageType.S3;
        } else if (this.filePath.toUpperCase().startsWith(HDFS_FILE_PREFIX.toUpperCase())) {
            brokerName = StorageBackend.StorageType.HDFS.name();
            storageType = StorageBackend.StorageType.HDFS;
            this.filePath = this.filePath.substring(HDFS_FILE_PREFIX.length() - 1);
        } else {
            return;
        }
        HashMap brokerProps = Maps.newHashMap();
        for (Map.Entry<String, String> entry : this.properties.entrySet()) {
            if (entry.getKey().startsWith(BROKER_PROP_PREFIX) && !entry.getKey().equals(PROP_BROKER_NAME)) {
                brokerProps.put(entry.getKey().substring(BROKER_PROP_PREFIX.length()), entry.getValue());
                processedPropKeys.add(entry.getKey());
                continue;
            }
            if (entry.getKey().toUpperCase().startsWith("AWS")) {
                brokerProps.put(entry.getKey(), entry.getValue());
                processedPropKeys.add(entry.getKey());
                continue;
            }
            if (entry.getKey().contains(BrokerUtil.HADOOP_FS_NAME) && storageType == StorageBackend.StorageType.HDFS) {
                brokerProps.put(entry.getKey(), entry.getValue());
                processedPropKeys.add(entry.getKey());
                continue;
            }
            if (!entry.getKey().startsWith(HADOOP_FS_PROP_PREFIX) && !entry.getKey().startsWith(HADOOP_PROP_PREFIX) || storageType != StorageBackend.StorageType.HDFS) continue;
            brokerProps.put(entry.getKey(), entry.getValue());
            processedPropKeys.add(entry.getKey());
        }
        if (storageType == StorageBackend.StorageType.S3) {
            S3Storage.checkS3(new CaseInsensitiveMap((Map)brokerProps));
        } else if (storageType == StorageBackend.StorageType.HDFS) {
            HDFSStorage.checkHDFS((Map<String, String>)new CaseInsensitiveMap((Map)brokerProps));
        }
        this.brokerDesc = new BrokerDesc(brokerName, storageType, brokerProps);
    }

    private void getParquetProperties(Set<String> processedPropKeys) throws AnalysisException {
        String[] schemas;
        for (Map.Entry<String, String> entry : this.properties.entrySet()) {
            if (!entry.getKey().startsWith(PARQUET_PROP_PREFIX)) continue;
            processedPropKeys.add(entry.getKey());
            this.fileProperties.put(entry.getKey().substring(PARQUET_PROP_PREFIX.length()), entry.getValue());
        }
        String schema = this.properties.get(SCHEMA);
        if (schema == null) {
            return;
        }
        if (schema.isEmpty()) {
            throw new AnalysisException("Parquet schema property should not be empty");
        }
        schema = schema.replace(" ", "");
        schema = schema.toLowerCase();
        for (String item : schemas = schema.split(";")) {
            String[] properties = item.split(",");
            if (properties.length != 3) {
                throw new AnalysisException("must only contains repetition type/column type/column name");
            }
            if (!PARQUET_REPETITION_TYPES.contains(properties[0])) {
                throw new AnalysisException("unknown repetition type");
            }
            if (!properties[0].equalsIgnoreCase("required")) {
                throw new AnalysisException("currently only support required type");
            }
            if (!PARQUET_DATA_TYPES.contains(properties[1])) {
                throw new AnalysisException("data type is not supported:" + properties[1]);
            }
            ArrayList<String> column = new ArrayList<String>();
            column.addAll(Arrays.asList(properties));
            this.schema.add(column);
        }
        processedPropKeys.add(SCHEMA);
    }

    private boolean isCsvFormat() {
        return this.fileFormatType == TFileFormatType.FORMAT_CSV_BZ2 || this.fileFormatType == TFileFormatType.FORMAT_CSV_DEFLATE || this.fileFormatType == TFileFormatType.FORMAT_CSV_GZ || this.fileFormatType == TFileFormatType.FORMAT_CSV_LZ4FRAME || this.fileFormatType == TFileFormatType.FORMAT_CSV_LZO || this.fileFormatType == TFileFormatType.FORMAT_CSV_LZOP || this.fileFormatType == TFileFormatType.FORMAT_CSV_PLAIN;
    }

    private boolean isParquetFormat() {
        return this.fileFormatType == TFileFormatType.FORMAT_PARQUET;
    }

    public OutFileClause clone() {
        return new OutFileClause(this);
    }

    public String toSql() {
        StringBuilder sb = new StringBuilder();
        sb.append(" INTO OUTFILE '").append(this.filePath).append(" FORMAT AS ").append(this.format);
        if (this.properties != null && !this.properties.isEmpty()) {
            sb.append(" PROPERTIES(");
            sb.append(new PrintableMap<String, String>(this.properties, " = ", true, false));
            sb.append(")");
        }
        return sb.toString();
    }

    public TResultFileSinkOptions toSinkOptions() {
        TResultFileSinkOptions sinkOptions = new TResultFileSinkOptions(this.filePath, this.fileFormatType);
        if (this.isCsvFormat()) {
            sinkOptions.setColumnSeparator(this.columnSeparator);
            sinkOptions.setLineDelimiter(this.lineDelimiter);
        }
        sinkOptions.setMaxFileSizeBytes(this.maxFileSizeBytes);
        if (this.brokerDesc != null) {
            sinkOptions.setBrokerProperties(this.brokerDesc.getProperties());
        }
        if (!Strings.isNullOrEmpty((String)this.successFileName)) {
            sinkOptions.setSuccessFileName(this.successFileName);
        }
        if (this.isParquetFormat()) {
            sinkOptions.setSchema(this.schema);
            sinkOptions.setFileProperties(this.fileProperties);
        }
        return sinkOptions;
    }

    static {
        RESULT_COL_NAMES.add("FileNumber");
        RESULT_COL_NAMES.add("TotalRows");
        RESULT_COL_NAMES.add("FileSize");
        RESULT_COL_NAMES.add("URL");
        RESULT_COL_TYPES.add(PrimitiveType.INT);
        RESULT_COL_TYPES.add(PrimitiveType.BIGINT);
        RESULT_COL_TYPES.add(PrimitiveType.BIGINT);
        RESULT_COL_TYPES.add(PrimitiveType.VARCHAR);
        PARQUET_REPETITION_TYPES.add("required");
        PARQUET_REPETITION_TYPES.add("repeated");
        PARQUET_REPETITION_TYPES.add("optional");
        PARQUET_DATA_TYPES.add("boolean");
        PARQUET_DATA_TYPES.add("int32");
        PARQUET_DATA_TYPES.add("int64");
        PARQUET_DATA_TYPES.add("int96");
        PARQUET_DATA_TYPES.add("byte_array");
        PARQUET_DATA_TYPES.add("float");
        PARQUET_DATA_TYPES.add("double");
        PARQUET_DATA_TYPES.add("fixed_len_byte_array");
    }
}

